/**
 * Copyright (C) 2019 Bonitasoft S.A.
 * Bonitasoft, 32 rue Gustave Eiffel - 38000 Grenoble
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 **/
package org.bonitasoft.engine.work;

import java.time.Instant;

import javax.transaction.Status;

import org.bonitasoft.engine.transaction.BonitaTransactionSynchronization;
import org.bonitasoft.engine.transaction.UserTransactionService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WorkSynchronization implements BonitaTransactionSynchronization {

    private static final Logger LOG = LoggerFactory.getLogger(WorkSynchronization.class);

    private final WorkDescriptor work;

    private final WorkExecutorService workExecutorService;

    private final UserTransactionService transactionService;
    private final int workDelayOnMultipleXAResource;

    WorkSynchronization(final UserTransactionService transactionService, final WorkExecutorService workExecutorService,
            WorkDescriptor work, int workDelayOnMultipleXAResource) {
        this.transactionService = transactionService;
        this.workDelayOnMultipleXAResource = workDelayOnMultipleXAResource;
        this.work = work;
        this.workExecutorService = workExecutorService;
    }

    WorkDescriptor getWork() {
        return work;
    }

    @Override
    public void afterCompletion(final int transactionStatus) {
        if (Status.STATUS_COMMITTED == transactionStatus) {
            if (workDelayOnMultipleXAResource > 0) {
                transactionService.hasMultipleResources().ifPresentOrElse(
                        hasMultiple -> {
                            if (Boolean.TRUE.equals(hasMultiple)) {
                                work.mustBeExecutedAfter(Instant.now().plusMillis(workDelayOnMultipleXAResource));
                            }
                        },
                        // to be safe, if we are unable to know if there are multiple resources, we add the delay anyway.
                        () -> work.mustBeExecutedAfter(Instant.now().plusMillis(workDelayOnMultipleXAResource)));
            }
            workExecutorService.execute(work);
        } else {
            LOG.debug("Transaction completion with state {} != COMMITTED. Not triggering the work: {}",
                    transactionStatus, work);
        }
    }

}
