/**
 * Copyright (C) 2019 Bonitasoft S.A.
 * Bonitasoft, 32 rue Gustave Eiffel - 38000 Grenoble
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 **/
package org.bonitasoft.engine.api.impl;

import java.io.IOException;

import org.bonitasoft.engine.api.TenantAdministrationAPI;
import org.bonitasoft.engine.api.impl.transaction.CustomTransactions;
import org.bonitasoft.engine.business.data.BusinessDataModelRepository;
import org.bonitasoft.engine.business.data.BusinessDataRepositoryDeploymentException;
import org.bonitasoft.engine.business.data.BusinessDataRepositoryException;
import org.bonitasoft.engine.business.data.InvalidBusinessDataModelException;
import org.bonitasoft.engine.business.data.SBusinessDataRepositoryDeploymentException;
import org.bonitasoft.engine.business.data.SBusinessDataRepositoryException;
import org.bonitasoft.engine.commons.exceptions.SBonitaException;
import org.bonitasoft.engine.exception.BonitaHomeConfigurationException;
import org.bonitasoft.engine.exception.BonitaHomeNotSetException;
import org.bonitasoft.engine.exception.BonitaRuntimeException;
import org.bonitasoft.engine.exception.RetrieveException;
import org.bonitasoft.engine.exception.UpdateException;
import org.bonitasoft.engine.persistence.SBonitaReadException;
import org.bonitasoft.engine.resources.STenantResourceLight;
import org.bonitasoft.engine.resources.TenantResourcesService;
import org.bonitasoft.engine.service.ModelConvertor;
import org.bonitasoft.engine.service.PlatformServiceAccessor;
import org.bonitasoft.engine.service.TenantServiceAccessor;
import org.bonitasoft.engine.service.TenantServiceSingleton;
import org.bonitasoft.engine.service.impl.ServiceAccessorFactory;
import org.bonitasoft.engine.sessionaccessor.SessionAccessor;
import org.bonitasoft.engine.tenant.TenantResource;
import org.bonitasoft.engine.tenant.TenantResourceType;

/**
 * @author Matthieu Chaffotte
 * @author Baptiste Mesta
 */

public class TenantAdministrationAPIImpl implements TenantAdministrationAPI {

    protected PlatformServiceAccessor getPlatformAccessorNoException() {
        try {
            return ServiceAccessorFactory.getInstance().createPlatformServiceAccessor();
        } catch (final Exception e) {
            throw new BonitaRuntimeException(e);
        }
    }

    protected long getTenantId() {
        try {
            return getSessionAccessor().getTenantId();
        } catch (final Exception e) {
            throw new BonitaRuntimeException(e);
        }
    }

    @Override
    @AvailableWhenTenantIsPaused
    public boolean isPaused() {
        final long tenantId = getTenantId();
        try {
            return getPlatformAccessorNoException().getPlatformService().getTenant(tenantId).isPaused();
        } catch (final SBonitaException e) {
            throw new RetrieveException("Unable to retrieve the tenant with id " + tenantId, e);
        }
    }

    @Override
    @AvailableWhenTenantIsPaused
    @CustomTransactions
    public void pause() throws UpdateException {
        TenantServiceAccessor tenantServiceAccessor = getPlatformAccessorNoException()
                .getTenantServiceAccessor(getTenantId());
        try {
            tenantServiceAccessor.getTenantStateManager().pause();
        } catch (Exception e) {
            throw new UpdateException(e);
        }
    }

    @Override
    @AvailableWhenTenantIsPaused
    @CustomTransactions
    public void resume() throws UpdateException {
        TenantServiceAccessor tenantServiceAccessor = getPlatformAccessorNoException()
                .getTenantServiceAccessor(getTenantId());
        try {
            tenantServiceAccessor.getTenantStateManager().resume();
            tenantServiceAccessor.getUserTransactionService().executeInTransaction(() -> {
                resolveDependenciesForAllProcesses();
                return null;
            });
        } catch (Exception e) {
            throw new UpdateException(e);
        }
    }

    private void resolveDependenciesForAllProcesses() {
        final TenantServiceAccessor tenantAccessor = getTenantAccessor();
        tenantAccessor.getBusinessArchiveArtifactsManager().resolveDependenciesForAllProcesses(tenantAccessor);
    }

    @Override
    @AvailableWhenTenantIsPaused
    public TenantResource getBusinessDataModelResource() {
        return getTenantResource(TenantResourceType.BDM);
    }

    protected TenantResource getTenantResource(TenantResourceType type) {
        TenantResourcesService tenantResourcesService = getTenantAccessor().getTenantResourcesService();
        org.bonitasoft.engine.resources.TenantResourceType resourceType = org.bonitasoft.engine.resources.TenantResourceType
                .valueOf(type.name());
        try {
            STenantResourceLight tenantResource = tenantResourcesService.getSingleLightResource(resourceType);
            return ModelConvertor.toTenantResource(tenantResource);
        } catch (SBonitaReadException e) {
            return TenantResource.NONE;
        }
    }

    //visible for testing
    @Override
    @AvailableWhenTenantIsPaused
    public String getBusinessDataModelVersion() throws BusinessDataRepositoryException {
        try {
            final BusinessDataModelRepository modelRepository = getTenantAccessor().getBusinessDataModelRepository();
            return modelRepository.getInstalledBDMVersion();
        } catch (final SBusinessDataRepositoryException e) {
            throw new BusinessDataRepositoryException(e);
        }
    }

    @Override
    @AvailableWhenTenantIsPaused(onlyAvailableWhenPaused = true)
    public String installBusinessDataModel(final byte[] zip)
            throws InvalidBusinessDataModelException, BusinessDataRepositoryDeploymentException {
        final TenantServiceAccessor tenantAccessor = getTenantAccessor();
        final long userId;
        try {
            userId = getUserId();
        } catch (IllegalStateException e) {
            throw new BusinessDataRepositoryDeploymentException("Unable to determine user ID");
        }
        try {
            final BusinessDataModelRepository bdmRepository = tenantAccessor.getBusinessDataModelRepository();
            return bdmRepository.install(zip, tenantAccessor.getTenantId(), userId);
        } catch (final SBusinessDataRepositoryDeploymentException e) {
            throw new BusinessDataRepositoryDeploymentException(e);
        }
    }

    @Override
    @AvailableWhenTenantIsPaused(onlyAvailableWhenPaused = true)
    public void uninstallBusinessDataModel() throws BusinessDataRepositoryDeploymentException {
        final TenantServiceAccessor tenantAccessor = getTenantAccessor();
        try {
            final BusinessDataModelRepository bdmRepository = tenantAccessor.getBusinessDataModelRepository();
            bdmRepository.uninstall(tenantAccessor.getTenantId());
        } catch (final SBusinessDataRepositoryException sbdre) {
            throw new BusinessDataRepositoryDeploymentException(sbdre);
        }
    }

    @Override
    @AvailableWhenTenantIsPaused(onlyAvailableWhenPaused = true)
    public void cleanAndUninstallBusinessDataModel() throws BusinessDataRepositoryDeploymentException {
        final TenantServiceAccessor tenantAccessor = getTenantAccessor();
        try {
            final BusinessDataModelRepository bdmRepository = tenantAccessor.getBusinessDataModelRepository();
            bdmRepository.dropAndUninstall(tenantAccessor.getTenantId());
        } catch (final SBusinessDataRepositoryException sbdre) {
            throw new BusinessDataRepositoryDeploymentException(sbdre);
        }
    }

    @Override
    @AvailableWhenTenantIsPaused
    public byte[] getClientBDMZip() throws BusinessDataRepositoryException {
        final BusinessDataModelRepository bdmRepository = getTenantAccessor().getBusinessDataModelRepository();
        try {
            return bdmRepository.getClientBDMZip();
        } catch (final SBusinessDataRepositoryException e) {
            throw new BusinessDataRepositoryException(e);
        }
    }

    protected TenantServiceAccessor getTenantAccessor() {
        try {
            return TenantServiceSingleton.getInstance(getTenantId());
        } catch (final Exception e) {
            throw new BonitaRuntimeException(e);
        }
    }

    private SessionAccessor getSessionAccessor() throws IllegalAccessException, InstantiationException, IOException,
            ClassNotFoundException, BonitaHomeConfigurationException, BonitaHomeNotSetException {
        return ServiceAccessorFactory.getInstance().createSessionAccessor();
    }

    protected long getUserId() throws IllegalStateException {
        try {
            return getTenantAccessor().getSessionService().getSession(getSessionAccessor().getSessionId()).getUserId();
        } catch (final Exception e) {
            throw new BonitaRuntimeException(e.getMessage());
        }
    }
}
