/*
 * Decompiled with CFR 0.152.
 */
package org.bonitasoft.web.designer.controller.asset;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.List;
import java.util.UUID;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.bonitasoft.web.designer.common.repository.AssetRepository;
import org.bonitasoft.web.designer.common.repository.Repository;
import org.bonitasoft.web.designer.common.repository.exception.NotFoundException;
import org.bonitasoft.web.designer.common.repository.exception.RepositoryException;
import org.bonitasoft.web.designer.controller.importer.dependencies.AssetDependencyImporter;
import org.bonitasoft.web.designer.model.Assetable;
import org.bonitasoft.web.designer.model.asset.Asset;
import org.bonitasoft.web.designer.model.asset.AssetType;
import org.bonitasoft.web.designer.model.page.Previewable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AssetService<T extends Assetable> {
    public static final String ASSET_TYPE_IS_REQUIRED = "Asset type is required";
    public static final String ASSET_URL_IS_REQUIRED = "Asset URL is required";
    public static final String ASSET_ID_IS_REQUIRED = "Asset id is required";
    protected static final Logger logger = LoggerFactory.getLogger(AssetService.class);
    private final Repository<T> repository;
    private final AssetRepository<T> assetRepository;
    private final AssetDependencyImporter<T> assetDependencyImporter;

    public AssetService(Repository<T> repository, AssetRepository<T> assetRepository, AssetDependencyImporter<T> assetDependencyImporter) {
        this.repository = repository;
        this.assetRepository = assetRepository;
        this.assetDependencyImporter = assetDependencyImporter;
    }

    public Path findAssetPath(String id, String filename, String type) {
        return this.assetRepository.findAssetPath(id, filename, AssetType.getAsset((String)type));
    }

    private void deleteComponentAsset(T component, Predicate<Asset> assetPredicate) {
        component.getAssets().stream().filter(assetPredicate).findFirst().ifPresent(existingAsset -> {
            if (!existingAsset.isExternal()) {
                try {
                    existingAsset.setComponentId(component.getId());
                    this.assetRepository.delete(existingAsset);
                }
                catch (IOException | NotFoundException e) {
                    logger.warn("Asset to delete {} was not found", (Object)existingAsset.getName(), (Object)e);
                }
            }
            component.getAssets().remove(existingAsset);
        });
    }

    public Asset save(T component, Asset asset) {
        if (StringUtils.isEmpty((CharSequence)asset.getName())) {
            throw new IllegalArgumentException(ASSET_URL_IS_REQUIRED);
        }
        if (asset.getType() == null) {
            throw new IllegalArgumentException(ASSET_TYPE_IS_REQUIRED);
        }
        if (asset.getId() != null) {
            component.getAssets().stream().filter(element -> asset.getId().equals(element.getId())).findFirst().ifPresent(element -> {
                element.setName(asset.getName());
                element.setType(asset.getType());
                element.setActive(asset.isActive());
            });
        } else {
            asset.setId(UUID.randomUUID().toString());
            asset.setOrder(component.getNextAssetOrder());
            component.getAssets().add(asset);
        }
        this.repository.updateLastUpdateAndSave(component);
        return asset;
    }

    public Asset save(T component, Asset asset, byte[] content) {
        try {
            this.assetRepository.save(component.getId(), asset, content);
            return this.save(component, asset);
        }
        catch (IOException e) {
            throw new RepositoryException("Error while saving internal asset", (Throwable)e);
        }
    }

    public String getAssetContent(T component, Asset asset) throws IOException {
        return new String(this.getAssetBinaryContent(component, asset), StandardCharsets.UTF_8);
    }

    public byte[] getAssetBinaryContent(T component, Asset asset) throws IOException {
        return this.assetRepository.readAllBytes(component.getId(), asset);
    }

    public void duplicateAsset(Path artifactSourcePath, Path artifactTargetPath, String sourceArtifactId, String targetArtifactId) {
        this.checkArgument(StringUtils.isNotEmpty((CharSequence)sourceArtifactId), String.format("source %s id is required", this.repository.getComponentName()));
        this.checkArgument(StringUtils.isNotEmpty((CharSequence)targetArtifactId), String.format("target %s id is required", this.repository.getComponentName()));
        this.checkArgument(artifactSourcePath != null && Files.exists(artifactSourcePath, new LinkOption[0]), String.format("source %s path is required", this.repository.getComponentName()));
        this.checkArgument(artifactTargetPath != null && Files.exists(artifactTargetPath, new LinkOption[0]), String.format("target %s path is required", this.repository.getComponentName()));
        try {
            List<Asset> assets = this.assetDependencyImporter.load(this.repository.get(sourceArtifactId), artifactSourcePath);
            for (Asset asset : assets) {
                asset.setScope(null);
                asset.setComponentId(targetArtifactId);
            }
            this.assetDependencyImporter.save(assets, artifactTargetPath);
        }
        catch (IOException e) {
            throw new RepositoryException("Error on assets duplication", (Throwable)e);
        }
    }

    public void delete(T component, String assetId) {
        this.checkArgument(StringUtils.isNotEmpty((CharSequence)assetId), ASSET_ID_IS_REQUIRED);
        if (component instanceof Previewable && ((Previewable)component).getInactiveAssets().contains(assetId)) {
            ((Previewable)component).getInactiveAssets().remove(assetId);
        }
        this.deleteComponentAsset(component, asset -> assetId.equals(asset.getId()));
        this.repository.updateLastUpdateAndSave(component);
    }

    public Asset changeAssetOrderInComponent(T component, String assetId, OrderType ordering) {
        this.checkArgument(StringUtils.isNotEmpty((CharSequence)assetId), ASSET_ID_IS_REQUIRED);
        List assets = component.getAssets().stream().sorted(Asset.getComparatorByOrder()).collect(Collectors.toList());
        Asset previous = null;
        Asset actual = null;
        int i = 0;
        int size = assets.size();
        for (Asset a : assets) {
            if (actual != null) {
                if (!OrderType.INCREMENT.equals((Object)ordering)) break;
                a.setOrder(a.getOrder() - 1);
                break;
            }
            if (assetId.equals(a.getId())) {
                actual = a;
                if (OrderType.DECREMENT.equals((Object)ordering) && previous == null || OrderType.INCREMENT.equals((Object)ordering) && i == size - 1) break;
                a.setOrder(OrderType.DECREMENT.equals((Object)ordering) ? a.getOrder() - 1 : a.getOrder() + 1);
                if (previous != null && OrderType.DECREMENT.equals((Object)ordering)) {
                    previous.setOrder(previous.getOrder() + 1);
                }
            } else {
                previous = a;
            }
            ++i;
        }
        this.repository.updateLastUpdateAndSave(component);
        return actual;
    }

    public void changeAssetStateInPreviewable(T component, String assetId, boolean active) {
        this.checkArgument(StringUtils.isNotEmpty((CharSequence)assetId), ASSET_ID_IS_REQUIRED);
        if (component instanceof Previewable) {
            Previewable previewable = (Previewable)component;
            if (previewable.getInactiveAssets().contains(assetId) && active) {
                previewable.getInactiveAssets().remove(assetId);
                this.repository.updateLastUpdateAndSave(component);
            } else if (!previewable.getInactiveAssets().contains(assetId) && !active) {
                previewable.getInactiveAssets().add(assetId);
                this.repository.updateLastUpdateAndSave(component);
            }
        }
    }

    private void checkArgument(boolean expression, String errorMessage) {
        if (!expression) {
            throw new IllegalArgumentException(errorMessage);
        }
    }

    public void loadDefaultAssets(T content) {
        this.assetRepository.refreshAssets(content);
    }

    public static enum OrderType {
        INCREMENT,
        DECREMENT;

    }
}

