/*
 * Decompiled with CFR 0.152.
 */
package org.bonitasoft.web.designer.controller.asset;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Collection;
import java.util.List;
import java.util.stream.Stream;
import org.assertj.core.api.AbstractCollectionAssert;
import org.assertj.core.api.Assertions;
import org.bonitasoft.web.designer.builder.AssetBuilder;
import org.bonitasoft.web.designer.builder.PageBuilder;
import org.bonitasoft.web.designer.common.repository.AssetRepository;
import org.bonitasoft.web.designer.common.repository.Repository;
import org.bonitasoft.web.designer.common.repository.exception.RepositoryException;
import org.bonitasoft.web.designer.controller.asset.AssetService;
import org.bonitasoft.web.designer.controller.importer.dependencies.AssetDependencyImporter;
import org.bonitasoft.web.designer.model.Assetable;
import org.bonitasoft.web.designer.model.Identifiable;
import org.bonitasoft.web.designer.model.asset.Asset;
import org.bonitasoft.web.designer.model.asset.AssetType;
import org.bonitasoft.web.designer.model.page.Page;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.junit.jupiter.MockitoExtension;

@ExtendWith(value={MockitoExtension.class})
class AssetServiceTest {
    @Mock
    private Repository<Page> repository;
    @Mock
    private AssetRepository<Page> assetRepository;
    @Mock
    private AssetDependencyImporter<Page> assetDependencyImporter;
    private AssetService assetService;

    AssetServiceTest() {
    }

    @BeforeEach
    void setUp() throws Exception {
        this.assetService = new AssetService(this.repository, this.assetRepository, this.assetDependencyImporter);
    }

    @Test
    void should_return_error_when_adding_asset_with_name_null() {
        Page page = PageBuilder.aPage().withId("page-id").build();
        Asset asset = AssetBuilder.anAsset().withName(null).build();
        IllegalArgumentException exception = (IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> this.assetService.save((Assetable)page, asset));
        Assertions.assertThat((String)exception.getMessage()).isEqualTo("Asset URL is required");
    }

    @Test
    void should_return_error_when_adding_asset_with_name_empty() {
        Page page = PageBuilder.aPage().withId("page-id").build();
        Asset asset = AssetBuilder.anAsset().withName("").build();
        IllegalArgumentException exception = (IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> this.assetService.save((Assetable)page, asset));
        Assertions.assertThat((String)exception.getMessage()).isEqualTo("Asset URL is required");
    }

    @Test
    void should_return_error_when_adding_asset_with_type_invalid() {
        Page page = PageBuilder.aPage().withId("page-id").build();
        Asset asset = AssetBuilder.anAsset().withName("http://mycdn.com/myasset.js").withType(null).build();
        IllegalArgumentException exception = (IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> this.assetService.save((Assetable)page, asset));
        Assertions.assertThat((String)exception.getMessage()).isEqualTo("Asset type is required");
    }

    @Test
    void should_save_new_asset_and_populate_its_id() throws Exception {
        Page page = PageBuilder.aPage().build();
        Asset asset = this.assetService.save((Assetable)page, AssetBuilder.anAsset().withName("http://mycdn.com/myasset.js").withType(AssetType.JAVASCRIPT).build());
        ((Repository)Mockito.verify(this.repository)).updateLastUpdateAndSave((Identifiable)page);
        Assertions.assertThat((Collection)page.getAssets()).contains((Object[])new Asset[]{asset});
        Assertions.assertThat((String)asset.getId()).isNotNull();
    }

    @Test
    void should_compute_order_while_saving_a_new_asset() throws Exception {
        Page page = PageBuilder.aPage().build();
        Asset firstAsset = this.assetService.save((Assetable)page, AssetBuilder.anAsset().withName("http://mycdn.com/first.js").build());
        Asset secondtAsset = this.assetService.save((Assetable)page, AssetBuilder.anAsset().withName("http://mycdn.com/second.js").build());
        Assertions.assertThat((int)firstAsset.getOrder()).isEqualTo(1);
        Assertions.assertThat((int)secondtAsset.getOrder()).isEqualTo(2);
    }

    @Test
    void should_update_existing_local_asset() throws Exception {
        Asset existingAsset = AssetBuilder.anAsset().withId("existingAsset").withName("http://mycdn.com/myasset.js").withType(AssetType.JAVASCRIPT).active().build();
        Asset updatedAsset = AssetBuilder.anAsset().withId("existingAsset").withName("http://mycdn.com/newName.js").withType(AssetType.CSS).unactive().build();
        Page page = PageBuilder.aPage().withAsset(new Asset[]{existingAsset}).build();
        this.assetService.save((Assetable)page, updatedAsset);
        ((Repository)Mockito.verify(this.repository)).updateLastUpdateAndSave((Identifiable)page);
        Assertions.assertThat((Object)((Asset)page.getAssets().iterator().next())).isEqualTo((Object)updatedAsset);
    }

    @Test
    void should_return_error_when_error_onsave() throws Exception {
        Page page = PageBuilder.aPage().build();
        ((Repository)Mockito.doThrow(RepositoryException.class).when(this.repository)).updateLastUpdateAndSave((Identifiable)page);
        org.junit.jupiter.api.Assertions.assertThrows(RepositoryException.class, () -> this.assetService.save((Assetable)page, AssetBuilder.anAsset().withName("http://mycdn.com/myasset.js").withType(AssetType.JAVASCRIPT).build()));
    }

    @Test
    void should_not_return_error_when_adding_existing_asset_witherror_on_delete() {
        Asset asset = AssetBuilder.anAsset().withId("anAsset").build();
        Page page = PageBuilder.aPage().withAsset(new Asset[]{asset}).build();
        Asset save = this.assetService.save((Assetable)page, asset);
        Assertions.assertThat((Object)asset).isNotNull();
    }

    static Stream<Arguments> invalidArgsForDuplicate() throws Exception {
        Path tempPath = Files.createTempDirectory("test", new FileAttribute[0]);
        return Stream.of(Arguments.of((Object[])new Object[]{null, tempPath, "src-page-id", "page-id", "source page path is required"}), Arguments.of((Object[])new Object[]{tempPath, null, "src-page-id", "page-id", "target page path is required"}), Arguments.of((Object[])new Object[]{tempPath, tempPath, null, "page-id", "source page id is required"}), Arguments.of((Object[])new Object[]{tempPath, tempPath, "src-page-id", null, "target page id is required"}));
    }

    @ParameterizedTest
    @MethodSource(value={"invalidArgsForDuplicate"})
    void should_not_duplicate_asset_when_arg_invalid(Path artifactSourcePath, Path artifactTargetPath, String sourceArtifactId, String targetArtifactId, String expectedErrorMessage) {
        Mockito.when((Object)this.repository.getComponentName()).thenReturn((Object)"page");
        IllegalArgumentException exception = (IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> this.assetService.duplicateAsset(artifactSourcePath, artifactTargetPath, sourceArtifactId, targetArtifactId));
        Assertions.assertThat((String)exception.getMessage()).isEqualTo(expectedErrorMessage);
    }

    @Test
    void should_duplicate_asset() throws Exception {
        Page page = new Page();
        List<Asset> assets = List.of(AssetBuilder.anAsset().withId("UUID").withName("myfile.js").build());
        Path tempPath = Files.createTempDirectory("test", new FileAttribute[0]);
        Mockito.when((Object)((Page)this.repository.get("src-page-id"))).thenReturn((Object)page);
        Mockito.when((Object)this.assetDependencyImporter.load((Identifiable)page, tempPath)).thenReturn(assets);
        this.assetService.duplicateAsset(tempPath, tempPath, "src-page-id", "page-id");
        ((AssetDependencyImporter)Mockito.verify(this.assetDependencyImporter)).save(assets, tempPath);
    }

    @Test
    void should_return_error_when_deleting_asset_with_name_empty() {
        Page page = PageBuilder.aPage().withId("page-id").build();
        IllegalArgumentException exception = (IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> this.assetService.delete((Assetable)page, null));
        Assertions.assertThat((String)exception.getMessage()).isEqualTo("Asset id is required");
    }

    @Test
    void should_delete_existing_asset() throws Exception {
        Page page = PageBuilder.aFilledPage((String)"page-id");
        Asset asset = AssetBuilder.anAsset().withId("UIID").withName("myfile.js").withType(AssetType.JAVASCRIPT).build();
        page.getAssets().add(asset);
        this.assetService.delete((Assetable)page, "UIID");
        ((AssetRepository)Mockito.verify(this.assetRepository)).delete(asset);
    }

    @Test
    void should_remove_asset_in_inactive_list_when_delete_is_called() throws Exception {
        Page page = PageBuilder.aFilledPage((String)"page-id");
        Asset asset = AssetBuilder.anAsset().withId("UIID").withName("myfile.js").withType(AssetType.JAVASCRIPT).unactive().build();
        page.getInactiveAssets().add("UIID");
        page.getAssets().add(asset);
        Assertions.assertThat((Collection)page.getInactiveAssets()).hasSize(1);
        this.assetService.delete((Assetable)page, "UIID");
        ((AssetRepository)Mockito.verify(this.assetRepository)).delete(asset);
        Assertions.assertThat((Collection)page.getInactiveAssets()).isEmpty();
    }

    @Test
    void should_not_delete_file_for_existing_external_asset() throws Exception {
        Page page = PageBuilder.aFilledPage((String)"page-id");
        Asset asset = AssetBuilder.anAsset().withId("UIID").withName("http://mycdn.com/myasset.js").withExternal(true).withType(AssetType.JAVASCRIPT).build();
        page.getAssets().add(asset);
        this.assetService.delete((Assetable)page, "UIID");
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.assetRepository});
    }

    @Test
    void should_throw_IllegalArgument_when_sorting_asset_component_with_no_name() throws Exception {
        Page page = PageBuilder.aPage().build();
        IllegalArgumentException exception = (IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> this.assetService.changeAssetOrderInComponent((Assetable)page, null, AssetService.OrderType.DECREMENT));
        Assertions.assertThat((String)exception.getMessage()).isEqualTo("Asset id is required");
    }

    private Asset[] getSortedAssets() {
        return new Asset[]{AssetBuilder.anAsset().withId("asset1").withName("asset1").withOrder(1).build(), AssetBuilder.anAsset().withId("asset2").withName("asset2").withOrder(2).build(), AssetBuilder.anAsset().withId("asset3").withName("asset3").withOrder(3).build()};
    }

    @Test
    void should_increment_asset_order_in_component() throws Exception {
        Asset[] assets = this.getSortedAssets();
        Page page = PageBuilder.aPage().withId("page-id").withName("my-page").withAsset(assets).build();
        assets[1].setComponentId("page-id");
        Asset assetReturned = this.assetService.changeAssetOrderInComponent((Assetable)page, "asset2", AssetService.OrderType.INCREMENT);
        Assertions.assertThat((String)assetReturned.getName()).isEqualTo("asset2");
        Assertions.assertThat((int)assets[0].getOrder()).isEqualTo(1);
        Assertions.assertThat((int)assets[1].getOrder()).isEqualTo(3);
        Assertions.assertThat((int)assets[2].getOrder()).isEqualTo(2);
        ((Repository)Mockito.verify(this.repository)).updateLastUpdateAndSave((Identifiable)page);
    }

    @Test
    void should_decrement_asset_order_in_component() throws Exception {
        Asset[] assets = this.getSortedAssets();
        Page page = PageBuilder.aPage().withId("page-id").withName("my-page").withAsset(assets).build();
        assets[1].setComponentId("page-id");
        Asset assetReturned = this.assetService.changeAssetOrderInComponent((Assetable)page, "asset2", AssetService.OrderType.DECREMENT);
        Assertions.assertThat((String)assetReturned.getName()).isEqualTo("asset2");
        Assertions.assertThat((int)assets[0].getOrder()).isEqualTo(2);
        Assertions.assertThat((int)assets[1].getOrder()).isEqualTo(1);
        Assertions.assertThat((int)assets[2].getOrder()).isEqualTo(3);
        ((Repository)Mockito.verify(this.repository)).updateLastUpdateAndSave((Identifiable)page);
    }

    @Test
    void should_not_increment_asset_order_in_component_when_asset_is_the_last() throws Exception {
        Asset[] assets = this.getSortedAssets();
        Page page = PageBuilder.aPage().withId("page-id").withName("my-page").withAsset(assets).build();
        assets[2].setComponentId("page-id");
        Asset assetReturned = this.assetService.changeAssetOrderInComponent((Assetable)page, "asset3", AssetService.OrderType.INCREMENT);
        Assertions.assertThat((String)assetReturned.getName()).isEqualTo("asset3");
        Assertions.assertThat((int)assets[0].getOrder()).isEqualTo(1);
        Assertions.assertThat((int)assets[1].getOrder()).isEqualTo(2);
        Assertions.assertThat((int)assets[2].getOrder()).isEqualTo(3);
        ((Repository)Mockito.verify(this.repository)).updateLastUpdateAndSave((Identifiable)page);
    }

    @Test
    void should_decrement_asset_order_in_component_when_asset_is_the_last() throws Exception {
        Asset[] assets = this.getSortedAssets();
        Page page = PageBuilder.aPage().withId("page-id").withName("my-page").withAsset(assets).build();
        assets[2].setComponentId("page-id");
        Asset assetReturned = this.assetService.changeAssetOrderInComponent((Assetable)page, "asset3", AssetService.OrderType.DECREMENT);
        Assertions.assertThat((String)assetReturned.getName()).isEqualTo("asset3");
        Assertions.assertThat((int)assets[0].getOrder()).isEqualTo(1);
        Assertions.assertThat((int)assets[1].getOrder()).isEqualTo(3);
        Assertions.assertThat((int)assets[2].getOrder()).isEqualTo(2);
        ((Repository)Mockito.verify(this.repository)).updateLastUpdateAndSave((Identifiable)page);
    }

    @Test
    void should_not_decrement_asset_order_in_component_when_asset_is_the_first() throws Exception {
        Asset[] assets = this.getSortedAssets();
        Page page = PageBuilder.aPage().withId("page-id").withName("my-page").withAsset(assets).build();
        assets[0].setComponentId("page-id");
        Asset assetReturned = this.assetService.changeAssetOrderInComponent((Assetable)page, "asset1", AssetService.OrderType.DECREMENT);
        Assertions.assertThat((String)assetReturned.getName()).isEqualTo("asset1");
        Assertions.assertThat((int)assets[0].getOrder()).isEqualTo(1);
        Assertions.assertThat((int)assets[1].getOrder()).isEqualTo(2);
        Assertions.assertThat((int)assets[2].getOrder()).isEqualTo(3);
        ((Repository)Mockito.verify(this.repository)).updateLastUpdateAndSave((Identifiable)page);
    }

    @Test
    void should_increment_asset_order_in_component_when_asset_is_the_first() throws Exception {
        Asset[] assets = this.getSortedAssets();
        Page page = PageBuilder.aPage().withId("page-id").withName("my-page").withAsset(assets).build();
        assets[0].setComponentId("page-id");
        Asset assetReturned = this.assetService.changeAssetOrderInComponent((Assetable)page, "asset1", AssetService.OrderType.INCREMENT);
        Assertions.assertThat((String)assetReturned.getName()).isEqualTo("asset1");
        Assertions.assertThat((int)assets[0].getOrder()).isEqualTo(2);
        Assertions.assertThat((int)assets[1].getOrder()).isEqualTo(1);
        Assertions.assertThat((int)assets[2].getOrder()).isEqualTo(3);
        ((Repository)Mockito.verify(this.repository)).updateLastUpdateAndSave((Identifiable)page);
    }

    @Test
    void should_not_change_asset_state_in_previewable_when_asset_is_already_inactive() throws Exception {
        Page page = PageBuilder.aPage().withId("page-id").withName("my-page").withInactiveAsset(new String[]{"assetUIID"}).build();
        this.assetService.changeAssetStateInPreviewable((Assetable)page, "assetUIID", false);
        ((AbstractCollectionAssert)Assertions.assertThat((Collection)page.getInactiveAssets()).isNotEmpty()).contains((Object[])new String[]{"assetUIID"});
    }

    @Test
    void should_change_asset_state_in_previewable_when_asset_state_is_inactive() throws Exception {
        Page page = PageBuilder.aPage().withId("page-id").withName("my-page").build();
        this.assetService.changeAssetStateInPreviewable((Assetable)page, "assetUIID", false);
        ((AbstractCollectionAssert)Assertions.assertThat((Collection)page.getInactiveAssets()).isNotEmpty()).contains((Object[])new String[]{"assetUIID"});
        ((Repository)Mockito.verify(this.repository)).updateLastUpdateAndSave((Identifiable)page);
    }

    @Test
    void should_reactive_asset_in_previable_when_asset_is_inactive_in_previewable() throws Exception {
        Page page = PageBuilder.aPage().withId("page-id").withName("my-page").withInactiveAsset(new String[]{"assetUIID"}).build();
        this.assetService.changeAssetStateInPreviewable((Assetable)page, "assetUIID", true);
        Assertions.assertThat((Collection)page.getInactiveAssets()).isEmpty();
        ((Repository)Mockito.verify(this.repository)).updateLastUpdateAndSave((Identifiable)page);
    }

    @Test
    void should_load_default_assets() {
        Page page = PageBuilder.aPage().build();
        this.assetService.loadDefaultAssets((Assetable)page);
        ((AssetRepository)Mockito.verify(this.assetRepository)).refreshAssets((Identifiable)page);
    }

    @Test
    void should_read_asset_content() throws IOException {
        Asset myAsset = AssetBuilder.anAsset().withType(AssetType.CSS).withName("style.css").build();
        Page page = PageBuilder.aPage().withDesignerVersion("1.7.9").withAsset(new Asset[]{myAsset}).build();
        Mockito.when((Object)this.assetRepository.readAllBytes(page.getId(), myAsset)).thenReturn((Object)"myContentOfAsset".getBytes());
        String content = this.assetService.getAssetContent((Assetable)page, myAsset);
        Assertions.assertThat((String)content).isEqualTo("myContentOfAsset");
    }
}

