/*
 * Decompiled with CFR 0.152.
 */
package org.bonitasoft.web.designer.repository;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import javax.validation.Validation;
import org.assertj.core.api.Assertions;
import org.bonitasoft.web.designer.builder.PageBuilder;
import org.bonitasoft.web.designer.config.DesignerConfig;
import org.bonitasoft.web.designer.livebuild.Watcher;
import org.bonitasoft.web.designer.migration.LiveRepositoryUpdate;
import org.bonitasoft.web.designer.model.Identifiable;
import org.bonitasoft.web.designer.model.page.Page;
import org.bonitasoft.web.designer.repository.BeanValidator;
import org.bonitasoft.web.designer.repository.JsonFileBasedLoader;
import org.bonitasoft.web.designer.repository.JsonFileBasedPersister;
import org.bonitasoft.web.designer.repository.PageRepository;
import org.bonitasoft.web.designer.repository.exception.ConstraintValidationException;
import org.bonitasoft.web.designer.repository.exception.NotFoundException;
import org.bonitasoft.web.designer.repository.exception.RepositoryException;
import org.bonitasoft.web.designer.utils.rule.TemporaryFolder;
import org.joda.time.Instant;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.runners.MockitoJUnitRunner;
import org.mockito.verification.VerificationMode;

@RunWith(value=MockitoJUnitRunner.class)
public class PageRepositoryTest {
    @Rule
    public TemporaryFolder temporaryFolder = new TemporaryFolder();
    @Mock
    private LiveRepositoryUpdate<Page> liveRepositoryUpdate;
    private JsonFileBasedPersister<Page> persister;
    private JsonFileBasedLoader<Page> loader;
    private Path pagesPath;
    private PageRepository repository;

    @Before
    public void setUp() throws Exception {
        this.pagesPath = Paths.get(this.temporaryFolder.getRoot().getPath(), new String[0]);
        this.persister = (JsonFileBasedPersister)Mockito.spy((Object)new DesignerConfig().pageFileBasedPersister());
        this.loader = (JsonFileBasedLoader)Mockito.spy((Object)new DesignerConfig().pageFileBasedLoader());
        this.repository = new PageRepository(this.pagesPath, this.persister, this.loader, new BeanValidator(Validation.buildDefaultValidatorFactory().getValidator()), (Watcher)Mockito.mock(Watcher.class));
    }

    private Page addToRepository(PageBuilder page) throws Exception {
        return this.addToRepository(page.build());
    }

    private Page addToRepository(Page page) throws Exception {
        Path repo = this.temporaryFolder.newFolderPath(page.getId());
        this.persister.save(repo, (Identifiable)page);
        return page;
    }

    @Test
    public void should_get_a_page_from_a_json_file_repository() throws Exception {
        Page expectedPage = PageBuilder.aFilledPage("page-id");
        this.addToRepository(expectedPage);
        Page fetchedPage = (Page)this.repository.get(expectedPage.getId());
        Assertions.assertThat((Object)fetchedPage).isEqualTo((Object)expectedPage);
    }

    @Test(expected=NotFoundException.class)
    public void should_throw_NotFoundException_when_getting_an_inexisting_page() throws Exception {
        this.repository.get("page-id-unknown");
    }

    @Test
    public void should_get_all_page_from_repository_empty() throws Exception {
        Assertions.assertThat((List)this.repository.getAll()).isEmpty();
    }

    @Test
    public void should_get_all_page_from_repository() throws Exception {
        Page expectedPage = PageBuilder.aFilledPage("page-id");
        this.addToRepository(expectedPage);
        List fetchedPages = this.repository.getAll();
        Assertions.assertThat((List)fetchedPages).containsExactly((Object[])new Page[]{expectedPage});
    }

    @Test
    public void should_save_a_page_in_a_json_file_repository() throws Exception {
        Page page = PageBuilder.aFilledPage("page-id");
        Assertions.assertThat((File)this.pagesPath.resolve(page.getId()).resolve(page.getId() + ".json").toFile()).doesNotExist();
        this.repository.updateLastUpdateAndSave((Identifiable)page);
        Assertions.assertThat((File)this.pagesPath.resolve(page.getId()).resolve(page.getId() + ".json").toFile()).exists();
        Assertions.assertThat((Comparable)page.getLastUpdate()).isGreaterThan((Comparable)Instant.now().minus(5000L));
        Assertions.assertThat((boolean)Files.exists(Paths.get(this.repository.resolvePath(page.getId()).toString(), "assets", "css", "style.css"), new LinkOption[0])).isTrue();
    }

    @Test
    public void should_give_new_id_if_there_is_already_a_page_with_same_id() throws Exception {
        Page page = PageBuilder.aFilledPage("pageName");
        this.repository.updateLastUpdateAndSave((Identifiable)page);
        String newPageId = this.repository.getNextAvailableId("pageName");
        Assertions.assertThat((String)newPageId).isEqualTo((Object)"pageName1");
    }

    @Test
    public void should_keep_page_name_id_if_there_is_no_page_with_same_id() throws Exception {
        String newPageId = this.repository.getNextAvailableId("pageName");
        Assertions.assertThat((String)newPageId).isEqualTo((Object)"pageName");
    }

    @Test(expected=RepositoryException.class)
    public void should_throw_RepositoryException_when_error_occurs_while_saving_a_page() throws Exception {
        Page expectedPage = PageBuilder.aFilledPage("page-id");
        ((JsonFileBasedPersister)Mockito.doThrow((Throwable)new IOException()).when(this.persister)).save(this.pagesPath.resolve(expectedPage.getId()), (Identifiable)expectedPage);
        this.repository.updateLastUpdateAndSave((Identifiable)expectedPage);
    }

    @Test
    public void should_save_a_page_without_updating_last_update_date() throws Exception {
        Page page = (Page)this.repository.updateLastUpdateAndSave((Identifiable)PageBuilder.aPage().withId("page-id").withName("thePageName").build());
        Instant lastUpdate = page.getLastUpdate();
        page.setName("newName");
        this.repository.save((Identifiable)page);
        Page fetchedPage = (Page)this.repository.get(page.getId());
        Assertions.assertThat((Comparable)fetchedPage.getLastUpdate()).isEqualTo((Object)lastUpdate);
        Assertions.assertThat((String)fetchedPage.getName()).isEqualTo((Object)"newName");
    }

    @Test(expected=IllegalArgumentException.class)
    public void should_throw_IllegalArgumentException_while_saving_a_page_with_no_id_set() throws Exception {
        Page expectedPage = PageBuilder.aPage().withId(null).build();
        this.repository.updateLastUpdateAndSave((Identifiable)expectedPage);
    }

    @Test(expected=ConstraintValidationException.class)
    public void should_throw_ConstraintValidationException_while_saving_a_page_with_bad_name() throws Exception {
        Page expectedPage = PageBuilder.aPage().withId("page-id").withName("\u00e9\u00e9&\u00e9&z").build();
        this.repository.updateLastUpdateAndSave((Identifiable)expectedPage);
    }

    @Test
    public void should_save_all_page_in_a_json_file_repository() throws Exception {
        Page expectedPage = PageBuilder.aFilledPage("page-id");
        Assertions.assertThat((File)this.pagesPath.resolve(expectedPage.getId()).resolve(expectedPage.getId() + ".json").toFile()).doesNotExist();
        this.repository.saveAll(Collections.singletonList(expectedPage));
        Assertions.assertThat((File)this.pagesPath.resolve(expectedPage.getId()).resolve(expectedPage.getId() + ".json").toFile()).exists();
        Assertions.assertThat((Comparable)expectedPage.getLastUpdate()).isGreaterThan((Comparable)Instant.now().minus(5000L));
    }

    @Test
    public void should_delete_a_page_with_his_json_file_repository() throws Exception {
        Page expectedPage = PageBuilder.aFilledPage("page-id");
        this.addToRepository(expectedPage);
        Assertions.assertThat((File)this.pagesPath.resolve(expectedPage.getId()).resolve(expectedPage.getId() + ".json").toFile()).exists();
        this.repository.delete(expectedPage.getId());
        Assertions.assertThat((File)this.pagesPath.resolve(expectedPage.getId()).resolve(expectedPage.getId() + ".json").toFile()).doesNotExist();
    }

    @Test
    public void should_delete_page_metadata_when_deleting_a_page() throws Exception {
        Page expectedPage = this.addToRepository(PageBuilder.aFilledPage("page-id"));
        Assertions.assertThat((File)this.pagesPath.resolve(".metadata").resolve(expectedPage.getId() + ".json").toFile()).exists();
        this.repository.delete(expectedPage.getId());
        Assertions.assertThat((File)this.pagesPath.resolve(".metadata").resolve(expectedPage.getId() + ".json").toFile()).doesNotExist();
    }

    @Test(expected=NotFoundException.class)
    public void should_throw_NotFoundException_when_deleting_inexisting_page() throws Exception {
        this.repository.delete("foo");
    }

    @Test(expected=RepositoryException.class)
    public void should_throw_RepositoryException_when_error_occurs_on_object_included_search() throws Exception {
        Page expectedPage = PageBuilder.aFilledPage("page-id");
        ((JsonFileBasedLoader)Mockito.doThrow((Throwable)new IOException()).when(this.loader)).contains(this.pagesPath, expectedPage.getId());
        this.repository.containsObject(expectedPage.getId());
    }

    @Test(expected=RepositoryException.class)
    public void should_throw_RepositoryException_when_error_occurs_on_object_included_search_list() throws Exception {
        Page expectedPage = PageBuilder.aFilledPage("page-id");
        ((JsonFileBasedLoader)Mockito.doThrow((Throwable)new IOException()).when(this.loader)).findByObjectId(this.pagesPath, expectedPage.getId());
        this.repository.findByObjectId(expectedPage.getId());
    }

    @Test
    public void should_mark_a_page_as_favorite() throws Exception {
        Page page = this.addToRepository(PageBuilder.aPage().notFavorite());
        this.repository.markAsFavorite(page.getId());
        Page fetchedPage = (Page)this.repository.get(page.getId());
        Assertions.assertThat((boolean)fetchedPage.isFavorite()).isTrue();
    }

    @Test
    public void should_unmark_a_page_as_favorite() throws Exception {
        Page page = this.addToRepository(PageBuilder.aPage().favorite());
        this.repository.unmarkAsFavorite(page.getId());
        Page fetchedPage = (Page)this.repository.get(page.getId());
        Assertions.assertThat((boolean)fetchedPage.isFavorite()).isFalse();
    }

    @Test
    public void should_refresh_repository() throws Exception {
        Page page = this.addToRepository(PageBuilder.aPage());
        this.pagesPath.resolve(".metadata").resolve(page.getId() + ".json").toFile().delete();
        this.pagesPath.resolve(".metadata").resolve(".index.json").toFile().delete();
        this.repository.refresh(page.getId());
        Page fetchedPage = (Page)this.repository.get(page.getId());
        Assertions.assertThat((boolean)fetchedPage.isFavorite()).isFalse();
        Assertions.assertThat((File)this.pagesPath.resolve(".metadata").resolve(".index.json").toFile()).exists();
    }

    @Test
    public void should_refreshIndexing_repository() throws Exception {
        ArrayList<Page> pages = new ArrayList<Page>();
        Page page = PageBuilder.aPage().withUUID("baz-uuid").withId("page1").build();
        Page page2 = PageBuilder.aPage().withUUID("foo-uuid").withId("page2").withName("page2").build();
        pages.add(page);
        pages.add(page2);
        this.repository.refreshIndexing(pages);
        ((JsonFileBasedPersister)Mockito.verify(this.persister, (VerificationMode)Mockito.times((int)1))).refreshIndexing(this.pagesPath.resolve(".metadata"), pages);
    }
}

