/*
 * Decompiled with CFR 0.152.
 */
package org.bonitasoft.web.designer.repository;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.List;
import org.assertj.core.api.Assertions;
import org.bonitasoft.web.designer.builder.WidgetBuilder;
import org.bonitasoft.web.designer.config.DesignerConfig;
import org.bonitasoft.web.designer.model.widget.Widget;
import org.bonitasoft.web.designer.repository.WidgetFileBasedLoader;
import org.bonitasoft.web.designer.repository.exception.JsonReadException;
import org.bonitasoft.web.designer.repository.exception.NotFoundException;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

public class WidgetFileBasedLoaderTest {
    @Rule
    public TemporaryFolder temporaryFolder = new TemporaryFolder();
    private Path widgetDirectory;
    private WidgetFileBasedLoader widgetLoader;

    @Before
    public void setUp() {
        this.widgetDirectory = Paths.get(this.temporaryFolder.getRoot().getPath(), new String[0]);
        this.widgetLoader = new WidgetFileBasedLoader(new DesignerConfig().objectMapperWrapper());
    }

    private void addToDirectory(Path directory, Widget ... widgets) throws Exception {
        for (Widget widget : widgets) {
            Path widgetDir = Files.createDirectory(directory.resolve(widget.getId()), new FileAttribute[0]);
            this.writeWidgetInFile(widget, widgetDir.resolve(widget.getId() + ".json"));
        }
    }

    private void writeWidgetInFile(Widget widget, Path path) throws IOException {
        ObjectWriter writer = new ObjectMapper().writer();
        writer.writeValue(path.toFile(), (Object)widget);
    }

    @Test
    public void should_get_a_widget_by_its_id() throws Exception {
        Widget expectedWidget = WidgetBuilder.aWidget().id("input").build();
        Widget notExpectedWidget = WidgetBuilder.aWidget().id("label").build();
        this.addToDirectory(this.widgetDirectory, expectedWidget, notExpectedWidget);
        Widget widget = this.widgetLoader.get(this.widgetDirectory.resolve("input/input.json"));
        Assertions.assertThat((Object)widget).isEqualTo((Object)expectedWidget);
    }

    @Test
    public void should_get_a_widget_with_template_and_controller_by_its_id() throws Exception {
        String templateFileName = "input.tpl.html";
        String controllerFileName = "input.ctrl.js";
        Widget expectedWidget = WidgetBuilder.aWidget().id("input").template("@" + templateFileName).controller("@" + controllerFileName).build();
        this.addToDirectory(this.widgetDirectory, expectedWidget);
        String htmlContent = "<div></div>";
        Files.write(this.widgetDirectory.resolve(expectedWidget.getId()).resolve(templateFileName), htmlContent.getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        String jsContent = "function ($scope) {}";
        Files.write(this.widgetDirectory.resolve(expectedWidget.getId()).resolve(controllerFileName), jsContent.getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        Widget widget = this.widgetLoader.get(this.widgetDirectory.resolve("input/input.json"));
        Assertions.assertThat((String)widget.getTemplate()).isEqualTo((Object)htmlContent);
        Assertions.assertThat((String)widget.getController()).isEqualTo((Object)jsContent);
    }

    @Test
    public void should_load_a_widget_with_template_and_controller_by_its_id() throws Exception {
        String templateFileName = "input.tpl.html";
        String controllerFileName = "input.ctrl.js";
        Widget expectedWidget = WidgetBuilder.aWidget().id("input").template("@" + templateFileName).controller("@" + controllerFileName).build();
        this.addToDirectory(this.widgetDirectory, expectedWidget);
        String htmlContent = "<div></div>";
        Files.write(this.widgetDirectory.resolve(expectedWidget.getId()).resolve(templateFileName), htmlContent.getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        String jsContent = "function ($scope) {}";
        Files.write(this.widgetDirectory.resolve(expectedWidget.getId()).resolve(controllerFileName), jsContent.getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        Widget widget = this.widgetLoader.load(this.widgetDirectory.resolve("input/input.json"));
        Assertions.assertThat((String)widget.getTemplate()).isEqualTo((Object)htmlContent);
        Assertions.assertThat((String)widget.getController()).isEqualTo((Object)jsContent);
    }

    @Test
    public void should_retrieve_all_widgets() throws Exception {
        Widget input = WidgetBuilder.aWidget().id("input").build();
        Widget label = WidgetBuilder.aWidget().id("label").build();
        this.addToDirectory(this.widgetDirectory, input, label);
        List widgets = this.widgetLoader.getAll(this.widgetDirectory);
        Assertions.assertThat((List)widgets).containsOnly((Object[])new Widget[]{input, label});
    }

    @Test
    public void should_not_failed_when_directory_contains_an_hidden_file() throws Exception {
        Files.createDirectory(this.widgetDirectory.resolve(".DS_Store"), new FileAttribute[0]);
        this.widgetLoader.getAll(this.widgetDirectory);
    }

    @Test
    public void should_retrieve_all_custom_widgets() throws Exception {
        Widget input = WidgetBuilder.aWidget().id("pbInput").build();
        Widget custom1 = WidgetBuilder.aWidget().id("custom1").custom().build();
        Widget custom2 = WidgetBuilder.aWidget().id("custom2").custom().build();
        this.addToDirectory(this.widgetDirectory, input, custom1, custom2);
        List widgets = this.widgetLoader.loadAllCustom(this.widgetDirectory);
        Assertions.assertThat((List)widgets).containsOnly((Object[])new Widget[]{custom1, custom2});
    }

    @Test
    public void should_only_load_persisted_properties() throws Exception {
        this.addToDirectory(this.widgetDirectory, WidgetBuilder.aWidget().id("customWidget").custom().favorite().build());
        List widgets = this.widgetLoader.loadAllCustom(this.widgetDirectory);
        Assertions.assertThat((boolean)((Widget)widgets.get(0)).isFavorite()).isFalse();
    }

    @Test
    public void should_find_widget_which_use_another_widget() throws Exception {
        Widget input = WidgetBuilder.aWidget().id("input").build();
        Widget label = WidgetBuilder.aWidget().id("label").template("use <input>").build();
        this.addToDirectory(this.widgetDirectory, input, label);
        Assertions.assertThat((List)this.widgetLoader.findByObjectId(this.widgetDirectory, "input")).extracting("id").contains(new Object[]{"label"});
    }

    @Test
    public void should_find_widget_which_not_use_another_widget() throws Exception {
        Widget input = WidgetBuilder.aWidget().id("input").build();
        Widget label = WidgetBuilder.aWidget().id("label").template("use <input>").build();
        this.addToDirectory(this.widgetDirectory, input, label);
        Assertions.assertThat((List)this.widgetLoader.findByObjectId(this.widgetDirectory, "label")).isEmpty();
    }

    @Test
    public void should_load_a_single_page_in_the_import_folder() throws Exception {
        Widget input = WidgetBuilder.aWidget().id("input").build();
        this.addToDirectory(this.widgetDirectory, input);
        Widget widget = this.widgetLoader.load(this.widgetDirectory.resolve("input/input.json"));
        Assertions.assertThat((Object)widget).isEqualTo((Object)input);
    }

    @Test(expected=NotFoundException.class)
    public void should_throw_notfound_exception_when_there_are_no_pages_in_folder() throws Exception {
        this.widgetLoader.load(this.widgetDirectory.resolve("test"));
    }

    @Test(expected=JsonReadException.class)
    public void should_throw_json_read_exception_when_loaded_file_is_not_valid_json() throws Exception {
        Files.write(this.widgetDirectory.resolve("wrongjson.json"), "notJson".getBytes(), new OpenOption[0]);
        this.widgetLoader.load(this.widgetDirectory.resolve("wrongjson.json"));
    }

    @Test(expected=NotFoundException.class)
    public void should_throw_notfound_exception_when_widget_template_is_not_found() throws Exception {
        Widget input = WidgetBuilder.aWidget().id("input").build();
        input.setTemplate("@missingTemplate.tpl.html");
        this.addToDirectory(this.widgetDirectory, input);
        this.widgetLoader.load(this.widgetDirectory.resolve("input/input.json"));
    }

    @Test(expected=NotFoundException.class)
    public void should_throw_notfound_exception_when_widget_controller_is_not_found() throws Exception {
        Widget input = WidgetBuilder.aWidget().id("input").build();
        input.setController("@missingController.ctrl.js");
        this.addToDirectory(this.widgetDirectory, input);
        this.widgetLoader.load(this.widgetDirectory.resolve("input/input.json"));
    }
}

