/*
 * Decompiled with CFR 0.152.
 */
package org.bonitasoft.web.designer.controller.asset;

import com.google.common.collect.Lists;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.List;
import junitparams.JUnitParamsRunner;
import junitparams.Parameters;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.IterableAssert;
import org.bonitasoft.web.designer.builder.AssetBuilder;
import org.bonitasoft.web.designer.builder.PageBuilder;
import org.bonitasoft.web.designer.config.DesignerConfig;
import org.bonitasoft.web.designer.controller.asset.AssetService;
import org.bonitasoft.web.designer.controller.asset.MalformedJsonException;
import org.bonitasoft.web.designer.controller.importer.dependencies.AssetImporter;
import org.bonitasoft.web.designer.model.Assetable;
import org.bonitasoft.web.designer.model.Identifiable;
import org.bonitasoft.web.designer.model.asset.Asset;
import org.bonitasoft.web.designer.model.asset.AssetType;
import org.bonitasoft.web.designer.model.page.Page;
import org.bonitasoft.web.designer.repository.AssetRepository;
import org.bonitasoft.web.designer.repository.Repository;
import org.bonitasoft.web.designer.repository.exception.RepositoryException;
import org.hamcrest.CoreMatchers;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.junit.MockitoJUnit;
import org.mockito.junit.MockitoRule;
import org.springframework.mock.web.MockMultipartFile;
import org.springframework.web.multipart.MultipartFile;

@RunWith(value=JUnitParamsRunner.class)
public class AssetServiceTest {
    @Rule
    public ExpectedException expectedException = ExpectedException.none();
    @Rule
    public MockitoRule mockitoRule = MockitoJUnit.rule();
    @Mock
    private Repository<Page> repository;
    @Mock
    private AssetRepository<Page> assetRepository;
    @Mock
    private AssetImporter<Page> assetImporter;
    private AssetService assetService;

    @Before
    public void setUp() throws Exception {
        this.assetService = new AssetService(this.repository, this.assetRepository, this.assetImporter, new DesignerConfig().objectMapperWrapper());
    }

    @Test
    public void should_return_error_when_uploading_file_null() {
        this.expectedException.expect(IllegalArgumentException.class);
        this.expectedException.expectMessage(CoreMatchers.is((Object)"Part named [file] is needed to successfully import a component"));
        this.assetService.upload(null, (Assetable)PageBuilder.aPage().build(), "js");
    }

    @Test
    public void should_return_error_when_uploading_file_empty() {
        this.expectedException.expect(IllegalArgumentException.class);
        this.expectedException.expectMessage(CoreMatchers.is((Object)"Part named [file] is needed to successfully import a component"));
        MockMultipartFile file = new MockMultipartFile("file", "myfile.js", "application/js", "".getBytes());
        this.assetService.upload((MultipartFile)file, (Assetable)PageBuilder.aPage().build(), "js");
    }

    @Test
    public void should_return_error_when_uploadind_type_invalid() {
        this.expectedException.expect(IllegalArgumentException.class);
        this.expectedException.expectMessage(CoreMatchers.is((Object)"Part named [file] is needed to successfully import a component"));
        MockMultipartFile file = new MockMultipartFile("file", "myfile.js", "application/js", "".getBytes());
        this.assetService.upload((MultipartFile)file, (Assetable)PageBuilder.aPage().build(), "INVALID");
    }

    @Test
    public void should_upload_newfile_and_save_new_asset() throws Exception {
        Page page = PageBuilder.aPage().withId("aPage").build();
        byte[] fileContent = "function(){}".getBytes();
        MockMultipartFile file = new MockMultipartFile("fileName.js", "originalFileName.js", "application/javascript", fileContent);
        Asset expectedAsset = AssetBuilder.anAsset().withName("originalFileName.js").withType(AssetType.JAVASCRIPT).withOrder(1).build();
        Asset asset = this.assetService.upload((MultipartFile)file, (Assetable)page, "js");
        ((AssetRepository)Mockito.verify(this.assetRepository)).save("aPage", asset, fileContent);
        ((Repository)Mockito.verify(this.repository)).updateLastUpdateAndSave((Identifiable)page);
        Assertions.assertThat((Iterable)page.getAssets()).contains((Object[])new Asset[]{asset});
        Assertions.assertThat((String)asset.getId()).isNotNull();
        Assertions.assertThat((Object)asset).isEqualToIgnoringGivenFields((Object)expectedAsset, new String[]{"id"});
    }

    @Test
    public void should_upload_a_json_asset() throws Exception {
        Page page = PageBuilder.aPage().withId("page-id").build();
        byte[] fileContent = "{ \"some\": \"json\" }".getBytes();
        MockMultipartFile file = new MockMultipartFile("asset.json", "asset.json", "application/javascript", fileContent);
        Asset expectedAsset = AssetBuilder.anAsset().withName("asset.json").withType(AssetType.JSON).withOrder(1).build();
        Asset asset = this.assetService.upload((MultipartFile)file, (Assetable)page, "json");
        ((AssetRepository)Mockito.verify(this.assetRepository)).save("page-id", asset, fileContent);
        ((Repository)Mockito.verify(this.repository)).updateLastUpdateAndSave((Identifiable)page);
        Assertions.assertThat((Iterable)page.getAssets()).contains((Object[])new Asset[]{asset});
        Assertions.assertThat((String)asset.getId()).isNotNull();
        Assertions.assertThat((Object)asset).isEqualToIgnoringGivenFields((Object)expectedAsset, new String[]{"id"});
    }

    @Test
    public void should_return_error_when_uploading_with_error_onsave() throws Exception {
        this.expectedException.expect(RepositoryException.class);
        this.expectedException.expectMessage(CoreMatchers.is((Object)"Error while saving internal asset"));
        Page page = PageBuilder.aPage().build();
        MockMultipartFile file = new MockMultipartFile("file.js", "myfile.inv", "application/javascript", "function(){}".getBytes());
        ((Repository)Mockito.doThrow(IOException.class).when(this.repository)).updateLastUpdateAndSave((Identifiable)page);
        this.assetService.upload((MultipartFile)file, (Assetable)page, "js");
    }

    @Test
    public void should_upload_existing_file() throws Exception {
        Asset existingAsset = AssetBuilder.anAsset().withId("UIID").withName("asset.js").build();
        Page page = PageBuilder.aPage().withId("page-id").withName("my-page").withAsset(existingAsset).build();
        MockMultipartFile file = new MockMultipartFile("asset.js", "asset.js", "application/javascript", "function(){}".getBytes());
        Asset asset = this.assetService.upload((MultipartFile)file, (Assetable)page, "js");
        ((AssetRepository)Mockito.verify(this.assetRepository)).save("page-id", (Asset)page.getAssets().iterator().next(), "function(){}".getBytes());
        ((Repository)Mockito.verify(this.repository)).updateLastUpdateAndSave((Identifiable)page);
        Assertions.assertThat((String)asset.getId()).isEqualTo((Object)existingAsset.getId());
    }

    @Test(expected=MalformedJsonException.class)
    public void should_check_that_json_is_well_formed_while_uploading_a_json_asset() throws Exception {
        MockMultipartFile file = new MockMultipartFile("asset.json", "asset.json", "application/javascript", "{ not json }".getBytes());
        this.assetService.upload((MultipartFile)file, (Assetable)PageBuilder.aPage().build(), "json");
    }

    @Test
    public void should_return_error_when_adding_asset_with_name_null() {
        this.expectedException.expect(IllegalArgumentException.class);
        this.expectedException.expectMessage(CoreMatchers.is((Object)"Asset URL is required"));
        this.assetService.save((Assetable)PageBuilder.aPage().withId("page-id").build(), AssetBuilder.anAsset().withName(null).build());
    }

    @Test
    public void should_return_error_when_adding_asset_with_name_empty() {
        this.expectedException.expect(IllegalArgumentException.class);
        this.expectedException.expectMessage(CoreMatchers.is((Object)"Asset URL is required"));
        this.assetService.save((Assetable)PageBuilder.aPage().withId("page-id").build(), AssetBuilder.anAsset().withName("").build());
    }

    @Test
    public void should_return_error_when_adding_asset_with_type_invalid() {
        this.expectedException.expect(IllegalArgumentException.class);
        this.expectedException.expectMessage(CoreMatchers.is((Object)"Asset type is required"));
        this.assetService.save((Assetable)PageBuilder.aPage().withId("page-id").build(), AssetBuilder.anAsset().withName("http://mycdn.com/myasset.js").withType(null).build());
    }

    @Test
    public void should_save_new_asset_and_populate_its_id() throws Exception {
        Page page = PageBuilder.aPage().build();
        Asset asset = this.assetService.save((Assetable)page, AssetBuilder.anAsset().withName("http://mycdn.com/myasset.js").withType(AssetType.JAVASCRIPT).build());
        ((Repository)Mockito.verify(this.repository)).updateLastUpdateAndSave((Identifiable)page);
        Assertions.assertThat((Iterable)page.getAssets()).contains((Object[])new Asset[]{asset});
        Assertions.assertThat((String)asset.getId()).isNotNull();
    }

    @Test
    public void should_compute_order_while_saving_a_new_asset() throws Exception {
        Page page = PageBuilder.aPage().build();
        Asset firstAsset = this.assetService.save((Assetable)page, AssetBuilder.anAsset().withName("http://mycdn.com/first.js").build());
        Asset secondtAsset = this.assetService.save((Assetable)page, AssetBuilder.anAsset().withName("http://mycdn.com/second.js").build());
        Assertions.assertThat((int)firstAsset.getOrder()).isEqualTo(1);
        Assertions.assertThat((int)secondtAsset.getOrder()).isEqualTo(2);
    }

    @Test
    public void should_update_existing_local_asset() throws Exception {
        Asset existingAsset = AssetBuilder.anAsset().withId("existingAsset").withName("http://mycdn.com/myasset.js").withType(AssetType.JAVASCRIPT).active().build();
        Asset updatedAsset = AssetBuilder.anAsset().withId("existingAsset").withName("http://mycdn.com/newName.js").withType(AssetType.CSS).unactive().build();
        Page page = PageBuilder.aPage().withAsset(existingAsset).build();
        this.assetService.save((Assetable)page, updatedAsset);
        ((Repository)Mockito.verify(this.repository)).updateLastUpdateAndSave((Identifiable)page);
        Assertions.assertThat(page.getAssets().iterator().next()).isEqualTo((Object)updatedAsset);
    }

    @Test
    public void should_return_error_when_error_onsave() throws Exception {
        this.expectedException.expect(RepositoryException.class);
        Page page = PageBuilder.aPage().build();
        ((Repository)Mockito.doThrow(RepositoryException.class).when(this.repository)).updateLastUpdateAndSave((Identifiable)page);
        this.assetService.save((Assetable)page, AssetBuilder.anAsset().withName("http://mycdn.com/myasset.js").withType(AssetType.JAVASCRIPT).build());
    }

    @Test
    public void should_not_return_error_when_adding_existing_asset_witherror_on_delete() throws Exception {
        Asset asset = AssetBuilder.anAsset().withId("anAsset").build();
        Page page = PageBuilder.aPage().withAsset(asset).build();
        ((AssetRepository)Mockito.doThrow(IOException.class).when(this.assetRepository)).delete(asset);
        this.assetService.save((Assetable)page, asset);
    }

    protected Object[] invalidArgsForDuplicate() throws Exception {
        Path tempPath = Files.createTempDirectory("test", new FileAttribute[0]);
        return JUnitParamsRunner.$((Object[])new Object[]{JUnitParamsRunner.$((Object[])new Object[]{null, tempPath, "src-page-id", "page-id", "source page path is required"}), JUnitParamsRunner.$((Object[])new Object[]{tempPath, null, "src-page-id", "page-id", "target page path is required"}), JUnitParamsRunner.$((Object[])new Object[]{tempPath, tempPath, null, "page-id", "source page id is required"}), JUnitParamsRunner.$((Object[])new Object[]{tempPath, tempPath, "src-page-id", null, "target page id is required"})});
    }

    @Parameters(method="invalidArgsForDuplicate")
    @Test
    public void should_not_duplicate_asset_when_arg_invalid(Path artifactSourcePath, Path artifactTargetPath, String sourceArtifactId, String targetArtifactId, String expectedErrorMessage) throws Exception {
        Mockito.when((Object)this.repository.getComponentName()).thenReturn((Object)"page");
        this.expectedException.expect(IllegalArgumentException.class);
        this.expectedException.expectMessage(CoreMatchers.is((Object)expectedErrorMessage));
        this.assetService.duplicateAsset(artifactSourcePath, artifactTargetPath, sourceArtifactId, targetArtifactId);
    }

    @Test
    public void should_duplicate_asset() throws Exception {
        Page page = new Page();
        ArrayList assets = Lists.newArrayList((Object[])new Asset[]{AssetBuilder.anAsset().withId("UUID").withName("myfile.js").build()});
        Path tempPath = Files.createTempDirectory("test", new FileAttribute[0]);
        Mockito.when((Object)this.repository.get("src-page-id")).thenReturn((Object)page);
        Mockito.when((Object)this.assetImporter.load((Identifiable)page, tempPath)).thenReturn((Object)assets);
        this.assetService.duplicateAsset(tempPath, tempPath, "src-page-id", "page-id");
        ((AssetImporter)Mockito.verify(this.assetImporter)).save((List)Mockito.eq((Object)assets), (Path)Mockito.eq((Object)tempPath));
    }

    @Test
    public void should_return_error_when_deleting_asset_with_name_empty() {
        this.expectedException.expect(IllegalArgumentException.class);
        this.expectedException.expectMessage(CoreMatchers.is((Object)"Asset id is required"));
        this.assetService.delete((Assetable)PageBuilder.aPage().withId("page-id").build(), null);
    }

    @Test
    public void should_delete_existing_asset() throws Exception {
        Page page = PageBuilder.aFilledPage("page-id");
        Asset asset = AssetBuilder.anAsset().withId("UIID").withName("myfile.js").withType(AssetType.JAVASCRIPT).build();
        page.getAssets().add(asset);
        this.assetService.delete((Assetable)page, "UIID");
        ((AssetRepository)Mockito.verify(this.assetRepository)).delete(asset);
    }

    @Test
    public void should_not_delete_file_for_existing_external_asset() throws Exception {
        Page page = PageBuilder.aFilledPage("page-id");
        Asset asset = AssetBuilder.anAsset().withId("UIID").withName("http://mycdn.com/myasset.js").withExternal(true).withType(AssetType.JAVASCRIPT).build();
        page.getAssets().add(asset);
        this.assetService.delete((Assetable)page, "UIID");
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.assetRepository});
    }

    @Test
    public void should_throw_IllegalArgument_when_sorting_asset_component_with_no_name() throws Exception {
        this.expectedException.expect(IllegalArgumentException.class);
        this.expectedException.expectMessage(CoreMatchers.is((Object)"Asset id is required"));
        this.assetService.changeAssetOrderInComponent((Assetable)PageBuilder.aPage().build(), null, AssetService.OrderType.DECREMENT);
    }

    private Asset[] getSortedAssets() {
        return new Asset[]{AssetBuilder.anAsset().withId("asset1").withName("asset1").withOrder(1).build(), AssetBuilder.anAsset().withId("asset2").withName("asset2").withOrder(2).build(), AssetBuilder.anAsset().withId("asset3").withName("asset3").withOrder(3).build()};
    }

    @Test
    public void should_increment_asset_order_in_component() throws Exception {
        Asset[] assets = this.getSortedAssets();
        Page page = PageBuilder.aPage().withId("page-id").withName("my-page").withAsset(assets).build();
        Mockito.when((Object)this.repository.get("page-id")).thenReturn((Object)page);
        assets[1].setComponentId("page-id");
        Asset assetReturned = this.assetService.changeAssetOrderInComponent((Assetable)page, "asset2", AssetService.OrderType.INCREMENT);
        Assertions.assertThat((String)assetReturned.getName()).isEqualTo((Object)"asset2");
        Assertions.assertThat((int)assets[0].getOrder()).isEqualTo(1);
        Assertions.assertThat((int)assets[1].getOrder()).isEqualTo(3);
        Assertions.assertThat((int)assets[2].getOrder()).isEqualTo(2);
        ((Repository)Mockito.verify(this.repository)).updateLastUpdateAndSave((Identifiable)page);
    }

    @Test
    public void should_decrement_asset_order_in_component() throws Exception {
        Asset[] assets = this.getSortedAssets();
        Page page = PageBuilder.aPage().withId("page-id").withName("my-page").withAsset(assets).build();
        Mockito.when((Object)this.repository.get("page-id")).thenReturn((Object)page);
        assets[1].setComponentId("page-id");
        Asset assetReturned = this.assetService.changeAssetOrderInComponent((Assetable)page, "asset2", AssetService.OrderType.DECREMENT);
        Assertions.assertThat((String)assetReturned.getName()).isEqualTo((Object)"asset2");
        Assertions.assertThat((int)assets[0].getOrder()).isEqualTo(2);
        Assertions.assertThat((int)assets[1].getOrder()).isEqualTo(1);
        Assertions.assertThat((int)assets[2].getOrder()).isEqualTo(3);
        ((Repository)Mockito.verify(this.repository)).updateLastUpdateAndSave((Identifiable)page);
    }

    @Test
    public void should_not_increment_asset_order_in_component_when_asset_is_the_last() throws Exception {
        Asset[] assets = this.getSortedAssets();
        Page page = PageBuilder.aPage().withId("page-id").withName("my-page").withAsset(assets).build();
        Mockito.when((Object)this.repository.get("page-id")).thenReturn((Object)page);
        assets[2].setComponentId("page-id");
        Asset assetReturned = this.assetService.changeAssetOrderInComponent((Assetable)page, "asset3", AssetService.OrderType.INCREMENT);
        Assertions.assertThat((String)assetReturned.getName()).isEqualTo((Object)"asset3");
        Assertions.assertThat((int)assets[0].getOrder()).isEqualTo(1);
        Assertions.assertThat((int)assets[1].getOrder()).isEqualTo(2);
        Assertions.assertThat((int)assets[2].getOrder()).isEqualTo(3);
        ((Repository)Mockito.verify(this.repository)).updateLastUpdateAndSave((Identifiable)page);
    }

    @Test
    public void should_decrement_asset_order_in_component_when_asset_is_the_last() throws Exception {
        Asset[] assets = this.getSortedAssets();
        Page page = PageBuilder.aPage().withId("page-id").withName("my-page").withAsset(assets).build();
        Mockito.when((Object)this.repository.get("page-id")).thenReturn((Object)page);
        assets[2].setComponentId("page-id");
        Asset assetReturned = this.assetService.changeAssetOrderInComponent((Assetable)page, "asset3", AssetService.OrderType.DECREMENT);
        Assertions.assertThat((String)assetReturned.getName()).isEqualTo((Object)"asset3");
        Assertions.assertThat((int)assets[0].getOrder()).isEqualTo(1);
        Assertions.assertThat((int)assets[1].getOrder()).isEqualTo(3);
        Assertions.assertThat((int)assets[2].getOrder()).isEqualTo(2);
        ((Repository)Mockito.verify(this.repository)).updateLastUpdateAndSave((Identifiable)page);
    }

    @Test
    public void should_not_decrement_asset_order_in_component_when_asset_is_the_first() throws Exception {
        Asset[] assets = this.getSortedAssets();
        Page page = PageBuilder.aPage().withId("page-id").withName("my-page").withAsset(assets).build();
        Mockito.when((Object)this.repository.get("page-id")).thenReturn((Object)page);
        assets[0].setComponentId("page-id");
        Asset assetReturned = this.assetService.changeAssetOrderInComponent((Assetable)page, "asset1", AssetService.OrderType.DECREMENT);
        Assertions.assertThat((String)assetReturned.getName()).isEqualTo((Object)"asset1");
        Assertions.assertThat((int)assets[0].getOrder()).isEqualTo(1);
        Assertions.assertThat((int)assets[1].getOrder()).isEqualTo(2);
        Assertions.assertThat((int)assets[2].getOrder()).isEqualTo(3);
        ((Repository)Mockito.verify(this.repository)).updateLastUpdateAndSave((Identifiable)page);
    }

    @Test
    public void should_increment_asset_order_in_component_when_asset_is_the_first() throws Exception {
        Asset[] assets = this.getSortedAssets();
        Page page = PageBuilder.aPage().withId("page-id").withName("my-page").withAsset(assets).build();
        Mockito.when((Object)this.repository.get("page-id")).thenReturn((Object)page);
        assets[0].setComponentId("page-id");
        Asset assetReturned = this.assetService.changeAssetOrderInComponent((Assetable)page, "asset1", AssetService.OrderType.INCREMENT);
        Assertions.assertThat((String)assetReturned.getName()).isEqualTo((Object)"asset1");
        Assertions.assertThat((int)assets[0].getOrder()).isEqualTo(2);
        Assertions.assertThat((int)assets[1].getOrder()).isEqualTo(1);
        Assertions.assertThat((int)assets[2].getOrder()).isEqualTo(3);
        ((Repository)Mockito.verify(this.repository)).updateLastUpdateAndSave((Identifiable)page);
    }

    @Test
    public void should_not_change_asset_state_in_previewable_when_asset_is_already_inactive() throws Exception {
        Page page = PageBuilder.aPage().withId("page-id").withName("my-page").withInactiveAsset("assetUIID").build();
        Mockito.when((Object)this.repository.get("page-id")).thenReturn((Object)page);
        Asset assetSent = AssetBuilder.anAsset().withId("assetUIID").withComponentId("page-id").withName("myasset.js").build();
        this.assetService.changeAssetStateInPreviewable((Assetable)page, "assetUIID", false);
        ((IterableAssert)Assertions.assertThat((Iterable)page.getInactiveAssets()).isNotEmpty()).contains((Object[])new String[]{"assetUIID"});
    }

    @Test
    public void should_change_asset_state_in_previewable_when_asset_state_is_inactive() throws Exception {
        Page page = PageBuilder.aPage().withId("page-id").withName("my-page").build();
        Mockito.when((Object)this.repository.get("page-id")).thenReturn((Object)page);
        Asset assetSent = AssetBuilder.anAsset().withId("assetUIID").withComponentId("page-id").withName("myasset.js").build();
        this.assetService.changeAssetStateInPreviewable((Assetable)page, "assetUIID", false);
        ((IterableAssert)Assertions.assertThat((Iterable)page.getInactiveAssets()).isNotEmpty()).contains((Object[])new String[]{"assetUIID"});
        ((Repository)Mockito.verify(this.repository)).updateLastUpdateAndSave((Identifiable)page);
    }

    @Test
    public void should_reactive_asset_in_previable_when_asset_is_inactive_in_previewable() throws Exception {
        Page page = PageBuilder.aPage().withId("page-id").withName("my-page").withInactiveAsset("assetUIID").build();
        Mockito.when((Object)this.repository.get("page-id")).thenReturn((Object)page);
        Asset assetSent = AssetBuilder.anAsset().withId("assetUIID").withComponentId("page-id").withName("myasset.js").build();
        this.assetService.changeAssetStateInPreviewable((Assetable)page, "assetUIID", true);
        Assertions.assertThat((Iterable)page.getInactiveAssets()).isEmpty();
        ((Repository)Mockito.verify(this.repository)).updateLastUpdateAndSave((Identifiable)page);
    }

    @Test
    public void should_load_default_assets() {
        Page page = PageBuilder.aPage().build();
        this.assetService.loadDefaultAssets((Assetable)page);
        ((AssetRepository)Mockito.verify(this.assetRepository)).refreshAssets((Identifiable)page);
    }

    @Test
    public void should_read_asset_content() throws IOException {
        Asset myAsset = AssetBuilder.anAsset().withType(AssetType.CSS).withName("style.css").build();
        Page page = PageBuilder.aPage().withDesignerVersion("1.7.9").withAsset(myAsset).build();
        Mockito.when((Object)this.assetRepository.readAllBytes(page.getId(), myAsset)).thenReturn((Object)"myContentOfAsset".getBytes());
        String content = this.assetService.getAssetContent((Assetable)page, myAsset);
        Assertions.assertThat((String)content).isEqualTo((Object)"myContentOfAsset");
    }
}

