/*
 * Decompiled with CFR 0.152.
 */
package org.bonitasoft.web.designer.repository;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.List;
import org.assertj.core.api.Assertions;
import org.bonitasoft.web.designer.builder.SimpleObjectBuilder;
import org.bonitasoft.web.designer.config.DesignerConfig;
import org.bonitasoft.web.designer.model.JacksonObjectMapper;
import org.bonitasoft.web.designer.repository.JsonFileBasedLoader;
import org.bonitasoft.web.designer.repository.SimpleDesignerArtifact;
import org.bonitasoft.web.designer.repository.exception.JsonReadException;
import org.bonitasoft.web.designer.repository.exception.NotFoundException;
import org.bonitasoft.web.designer.repository.exception.RepositoryException;
import org.bonitasoft.web.designer.utils.rule.TemporaryFolder;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Matchers;
import org.mockito.Mockito;
import org.mockito.runners.MockitoJUnitRunner;

@RunWith(value=MockitoJUnitRunner.class)
public class JsonFileBasedLoaderTest {
    @Rule
    public TemporaryFolder temporaryFolder = new TemporaryFolder();
    private Path repoDirectory;
    private JacksonObjectMapper objectMapper;
    private JsonFileBasedLoader<SimpleDesignerArtifact> loader;

    @Before
    public void setUp() throws IOException {
        this.repoDirectory = this.temporaryFolder.newFolderPath("jsonrepository");
        this.objectMapper = (JacksonObjectMapper)Mockito.spy((Object)new DesignerConfig().objectMapperWrapper());
        this.loader = new JsonFileBasedLoader(this.objectMapper, SimpleDesignerArtifact.class);
    }

    private void addToRepository(SimpleDesignerArtifact ... pages) throws Exception {
        for (SimpleDesignerArtifact page : pages) {
            Path repo = this.temporaryFolder.newFolderPath("jsonrepository", page.getId());
            Files.write(repo.resolve(page.getId() + ".json"), this.objectMapper.toJson((Object)page), new OpenOption[0]);
        }
    }

    @Test
    public void should_get_an_object_from_a_json_file_and_deserialize_it() throws Exception {
        SimpleDesignerArtifact expectedObject = SimpleObjectBuilder.aFilledSimpleObject("id");
        this.addToRepository(expectedObject);
        Assertions.assertThat((Object)this.loader.get(this.repoDirectory.resolve("id/id.json"))).isEqualTo((Object)expectedObject);
    }

    @Test
    public void test_loader() throws Exception {
        SimpleDesignerArtifact a = new SimpleDesignerArtifact("id", "id", 5);
        a.setDesignerVersion("1.12.0");
        this.addToRepository(a);
        SimpleDesignerArtifact pouet = (SimpleDesignerArtifact)this.loader.get(this.repoDirectory.resolve("id/id.json"));
        Assertions.assertThat((String)pouet.getDesignerVersion()).isEqualTo((Object)"1.12.0");
        Assertions.assertThat((Object)((Object)pouet)).isEqualTo((Object)a);
    }

    @Test(expected=RepositoryException.class)
    public void should_throw_RepositoryException_when_error_occurs_while_getting_a_object() throws Exception {
        this.addToRepository(SimpleObjectBuilder.aFilledSimpleObject("foobar"));
        ((JacksonObjectMapper)Mockito.doThrow((Throwable)new IOException()).when((Object)this.objectMapper)).fromJson((byte[])Matchers.any(byte[].class), (Class)Matchers.eq(SimpleDesignerArtifact.class));
        this.loader.get(this.repoDirectory.resolve("foobar/foobar.json"));
    }

    @Test(expected=NotFoundException.class)
    public void should_throw_NotFoundException_when_while_getting_an_unexisting_object() throws Exception {
        this.loader.get(this.repoDirectory.resolve("unexisting/unexisting.json"));
    }

    @Test
    public void should_get_all_objects_from_json_files_and_deserialize_them() throws Exception {
        SimpleDesignerArtifact object1 = SimpleObjectBuilder.aFilledSimpleObject("objet1");
        SimpleDesignerArtifact object2 = SimpleObjectBuilder.aFilledSimpleObject("objet2");
        this.addToRepository(object1, object2);
        List objects = this.loader.getAll(this.repoDirectory);
        Assertions.assertThat((List)objects).containsOnly((Object[])new SimpleDesignerArtifact[]{object1, object2});
    }

    @Test(expected=RepositoryException.class)
    public void should_throw_IOException_when_error_occurs_while_getting_all_object() throws Exception {
        this.addToRepository(SimpleObjectBuilder.aFilledSimpleObject("objet1"));
        ((JacksonObjectMapper)Mockito.doThrow((Throwable)new IOException()).when((Object)this.objectMapper)).fromJson((byte[])Matchers.any(byte[].class), (Class)Matchers.eq(SimpleDesignerArtifact.class));
        this.loader.getAll(this.repoDirectory);
    }

    @Test
    public void should_fail_silently_when_an_object_is_not_found_while_getting_all_object() throws Exception {
        SimpleDesignerArtifact object1 = SimpleObjectBuilder.aFilledSimpleObject("object1");
        SimpleDesignerArtifact object2 = SimpleObjectBuilder.aFilledSimpleObject("object2");
        this.addToRepository(object1, object2);
        Files.delete(this.repoDirectory.resolve("object1/object1.json"));
        List all = this.loader.getAll(this.repoDirectory);
        Assertions.assertThat((List)all).containsOnly((Object[])new SimpleDesignerArtifact[]{object2});
    }

    @Test
    public void should_fail_silently_when_a_model_file_is_corrupted_while_getting_all_object() throws Exception {
        SimpleDesignerArtifact object1 = SimpleObjectBuilder.aFilledSimpleObject("object1");
        SimpleDesignerArtifact object2 = SimpleObjectBuilder.aFilledSimpleObject("object2");
        this.addToRepository(object1, object2);
        Files.write(this.repoDirectory.resolve("object1/object1.json"), "json corrupted".getBytes(), new OpenOption[0]);
        List all = this.loader.getAll(this.repoDirectory);
        Assertions.assertThat((List)all).containsOnly((Object[])new SimpleDesignerArtifact[]{object2});
    }

    @Test
    public void should_find_a_byte_array_in_an_another() {
        Assertions.assertThat((int)this.loader.indexOf("mon exemple complet".getBytes(), "exem".getBytes())).isEqualTo(4);
    }

    @Test
    public void should_find_a_byte_array_in_an_another_on_start_position() {
        Assertions.assertThat((int)this.loader.indexOf("mon exemple complet".getBytes(), "mon ex".getBytes())).isEqualTo(0);
    }

    @Test
    public void should_not_find_a_byte_array_in_an_another() {
        Assertions.assertThat((int)this.loader.indexOf("mon exemple complet".getBytes(), "rex".getBytes())).isEqualTo(-1);
    }

    @Test
    public void should_not_find_null_in_byte_array() {
        Assertions.assertThat((int)this.loader.indexOf("mon exemple complet".getBytes(), null)).isEqualTo(-1);
    }

    @Test
    public void should_not_find_occurence_in_byte_array_null() {
        Assertions.assertThat((int)this.loader.indexOf(null, "search".getBytes())).isEqualTo(-1);
    }

    @Test
    public void should_find_one_object_included_in_another_and_deserialize_it() throws Exception {
        SimpleDesignerArtifact object1 = SimpleObjectBuilder.aSimpleObjectBuilder().id("objet1").build();
        SimpleDesignerArtifact object2 = SimpleObjectBuilder.aSimpleObjectBuilder().id("objet2").another(object1).build();
        this.addToRepository(object1, object2);
        List objects = this.loader.findByObjectId(this.repoDirectory, "objet1");
        Assertions.assertThat((List)objects).containsOnly((Object[])new SimpleDesignerArtifact[]{object2});
    }

    @Test
    public void should_returns_an_empty_list_when_directory_does_not_exist() throws Exception {
        Assertions.assertThat((List)this.loader.findByObjectId(Paths.get("/does/not/exist", new String[0]), "objectId")).isEmpty();
    }

    @Test
    public void should_not_fail_when_searching_object_by_id_and_repo_contains_an_hidden_file() throws Exception {
        this.temporaryFolder.newFolderPath("jsonrepository", ".DS_Store");
        this.loader.findByObjectId(this.repoDirectory, "object");
    }

    @Test
    public void should_not_fail_when_searching_object_by_id_and_artifact_folder_has_no_model_file() throws Exception {
        Files.createDirectory(this.repoDirectory.resolve("artifactid"), new FileAttribute[0]);
        List objects = this.loader.findByObjectId(this.repoDirectory, "object");
        Assertions.assertThat((List)objects).isEmpty();
    }

    @Test
    public void should_not_find_an_object_even_if_id_start_the_same_as_the_one_looking_for() throws Exception {
        this.addToRepository(SimpleObjectBuilder.aSimpleObjectBuilder().id("abcd").build());
        List objects = this.loader.findByObjectId(this.repoDirectory, "abc");
        Assertions.assertThat((List)objects).isEmpty();
    }

    @Test
    public void should_not_find_object_included_in_another() throws Exception {
        SimpleDesignerArtifact object1 = SimpleObjectBuilder.aSimpleObjectBuilder().id("objet1").build();
        SimpleDesignerArtifact object2 = SimpleObjectBuilder.aSimpleObjectBuilder().id("objet2").another(object1).build();
        this.addToRepository(object1, object2);
        List objects = this.loader.findByObjectId(this.repoDirectory, "objet2");
        Assertions.assertThat((List)objects).isEmpty();
    }

    @Test(expected=IOException.class)
    public void should_throw_IOException_when_error_occurs_on_finding_object_by_id() throws Exception {
        SimpleDesignerArtifact object1 = SimpleObjectBuilder.aSimpleObjectBuilder().id("objet1").build();
        SimpleDesignerArtifact object2 = SimpleObjectBuilder.aSimpleObjectBuilder().id("objet2").another(object1).build();
        this.addToRepository(object1, object2);
        ((JacksonObjectMapper)Mockito.doThrow((Throwable)new IOException()).when((Object)this.objectMapper)).fromJson((byte[])Matchers.any(byte[].class), (Class)Matchers.eq(SimpleDesignerArtifact.class));
        this.loader.findByObjectId(this.repoDirectory, "objet1");
    }

    @Test
    public void should_find_object_included_in_another() throws Exception {
        SimpleDesignerArtifact object1 = SimpleObjectBuilder.aSimpleObjectBuilder().id("objet1").build();
        SimpleDesignerArtifact object2 = SimpleObjectBuilder.aSimpleObjectBuilder().id("objet2").another(object1).build();
        this.addToRepository(object1, object2);
        Assertions.assertThat((boolean)this.loader.contains(this.repoDirectory, "objet1")).isTrue();
    }

    @Test
    public void should_find_any_object_included_in_another() throws Exception {
        SimpleDesignerArtifact object1 = SimpleObjectBuilder.aSimpleObjectBuilder().id("objet1").build();
        SimpleDesignerArtifact object2 = SimpleObjectBuilder.aSimpleObjectBuilder().id("objet2").another(object1).build();
        this.addToRepository(object1, object2);
        Assertions.assertThat((boolean)this.loader.contains(this.repoDirectory, "object2")).isFalse();
    }

    @Test
    public void should_load_a_single_page_in_the_import_folder() throws Exception {
        SimpleDesignerArtifact object1 = SimpleObjectBuilder.aFilledSimpleObject("objet1");
        this.addToRepository(object1);
        SimpleDesignerArtifact loadedSimpleVersioned = (SimpleDesignerArtifact)this.loader.load(this.repoDirectory.resolve("objet1/objet1.json"));
        Assertions.assertThat((Object)((Object)loadedSimpleVersioned)).isEqualTo((Object)object1);
    }

    @Test(expected=NotFoundException.class)
    public void should_throw_notfound_exception_when_there_are_no_pages_in_folder() throws Exception {
        this.loader.load(this.repoDirectory.resolve("test"));
    }

    @Test(expected=JsonReadException.class)
    public void should_throw_json_read_exception_when_loaded_file_is_not_valid_json() throws Exception {
        Files.write(this.repoDirectory.resolve("wrongjson.json"), "notJson".getBytes(), new OpenOption[0]);
        this.loader.load(this.repoDirectory.resolve("wrongjson.json"));
    }

    @Test
    public void should_get_object_metadata() throws Exception {
        SimpleDesignerArtifact expectedObject = SimpleObjectBuilder.aSimpleObjectBuilder().id("id").metadata("foobar").build();
        this.addToRepository(expectedObject);
        Assertions.assertThat((String)((SimpleDesignerArtifact)this.loader.get(this.loader.resolve(this.repoDirectory, "id"))).getMetadata()).isEqualTo((Object)"foobar");
    }

    @Test
    public void should_not_load_object_metadata() throws Exception {
        SimpleDesignerArtifact expectedObject = SimpleObjectBuilder.aSimpleObjectBuilder().id("id").metadata("foobar").build();
        this.addToRepository(expectedObject);
        Assertions.assertThat((String)((SimpleDesignerArtifact)this.loader.load(this.loader.resolve(this.repoDirectory, "id"))).getMetadata()).isNull();
    }

    @Test
    public void should_get_same_object_id_as_page_name() throws Exception {
        Assertions.assertThat((String)this.loader.getNextAvailableObjectId(this.repoDirectory, "pageNameToUseAsPrefix")).isEqualTo((Object)"pageNameToUseAsPrefix");
    }

    @Test
    public void should_get_object_id_if_page_name_taken() throws Exception {
        Files.createDirectory(this.repoDirectory.resolve("pageNameToUseAsPrefix"), new FileAttribute[0]);
        Files.createDirectory(this.repoDirectory.resolve("pageNameToUseAsPrefix20"), new FileAttribute[0]);
        Files.createDirectory(this.repoDirectory.resolve("pageNameToUseAsPrefix245"), new FileAttribute[0]);
        Assertions.assertThat((String)this.loader.getNextAvailableObjectId(this.repoDirectory, "pageNameToUseAsPrefix")).isEqualTo((Object)"pageNameToUseAsPrefix246");
    }

    @Test
    public void should_get_next_available_object_id() throws Exception {
        Files.createDirectory(this.repoDirectory.resolve("pageNameToUseAsPrefix"), new FileAttribute[0]);
        Assertions.assertThat((String)this.loader.getNextAvailableObjectId(this.repoDirectory, "pageNameToUseAsPrefix")).isEqualTo((Object)"pageNameToUseAsPrefix1");
    }
}

