/*
 * Decompiled with CFR 0.152.
 */
package org.bonitasoft.web.designer.repository;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.List;
import javax.validation.Validation;
import org.assertj.core.api.Assertions;
import org.bonitasoft.web.designer.builder.FragmentBuilder;
import org.bonitasoft.web.designer.config.DesignerConfig;
import org.bonitasoft.web.designer.livebuild.Watcher;
import org.bonitasoft.web.designer.model.Identifiable;
import org.bonitasoft.web.designer.model.fragment.Fragment;
import org.bonitasoft.web.designer.repository.BeanValidator;
import org.bonitasoft.web.designer.repository.FragmentRepository;
import org.bonitasoft.web.designer.repository.JsonFileBasedLoader;
import org.bonitasoft.web.designer.repository.JsonFileBasedPersister;
import org.bonitasoft.web.designer.repository.exception.ConstraintValidationException;
import org.bonitasoft.web.designer.repository.exception.NotFoundException;
import org.bonitasoft.web.designer.repository.exception.RepositoryException;
import org.bonitasoft.web.designer.utils.rule.TemporaryFolder;
import org.joda.time.Instant;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.mockito.Mockito;

public class FragmentRepositoryTest {
    @Rule
    public TemporaryFolder temporaryFolder = new TemporaryFolder();
    private JsonFileBasedPersister<Fragment> persister;
    private JsonFileBasedLoader<Fragment> loader;
    private Path fragmentsPath;
    private FragmentRepository repository;

    @Before
    public void setUp() throws Exception {
        this.fragmentsPath = Paths.get(this.temporaryFolder.getRoot().getPath(), new String[0]);
        this.persister = (JsonFileBasedPersister)Mockito.spy((Object)new DesignerConfig().fragmentFileBasedPersister());
        this.loader = (JsonFileBasedLoader)Mockito.spy((Object)new DesignerConfig().fragmentFileBasedLoader());
        this.repository = new FragmentRepository(this.fragmentsPath, this.persister, this.loader, new BeanValidator(Validation.buildDefaultValidatorFactory().getValidator()), (Watcher)Mockito.mock(Watcher.class));
    }

    private Fragment addToRepository(Fragment fragment) throws Exception {
        Path repo = this.temporaryFolder.newFolderPath(fragment.getId());
        this.persister.save(repo, (Identifiable)fragment);
        return fragment;
    }

    private Fragment addToRepository(FragmentBuilder fragment) throws Exception {
        return this.addToRepository(fragment.build());
    }

    @Test
    public void should_get_a_fragment_from_a_json_file_repository() throws Exception {
        Fragment expectedFragment = FragmentBuilder.aFilledFragment("fragment-id");
        this.addToRepository(expectedFragment);
        Fragment fetchedFragment = (Fragment)this.repository.get(expectedFragment.getId());
        Assertions.assertThat((Object)fetchedFragment).isEqualTo((Object)expectedFragment);
    }

    @Test(expected=NotFoundException.class)
    public void should_throw_NotFoundException_when_getting_an_inexisting_fragment() throws Exception {
        this.repository.get("fragment-id-unknown");
    }

    @Test
    public void should_get_all_fragment_from_repository_empty() throws Exception {
        Assertions.assertThat((List)this.repository.getAll()).isEmpty();
    }

    @Test
    public void should_get_all_fragment_from_repository() throws Exception {
        Fragment expectedFragment = FragmentBuilder.aFilledFragment("fragment-id");
        this.addToRepository(expectedFragment);
        List fetchedFragments = this.repository.getAll();
        Assertions.assertThat((List)fetchedFragments).containsExactly((Object[])new Fragment[]{expectedFragment});
    }

    @Test
    public void should_get_all_fragment_except_itself() throws Exception {
        Fragment itself = this.addToRepository(FragmentBuilder.aFragment().id("aFragment"));
        Fragment expectedFragment = this.addToRepository(FragmentBuilder.aFragment().id("anotherFragment"));
        List fetchedFragments = this.repository.getAllNotUsingElement(itself.getId());
        Assertions.assertThat((List)fetchedFragments).containsOnly((Object[])new Fragment[]{expectedFragment});
    }

    @Test
    public void should_get_all_fragment_except_those_which_are_using_it() throws Exception {
        Fragment expectedFragment = this.addToRepository(FragmentBuilder.aFragment().id("anotherFragment").build());
        Fragment itself = this.addToRepository(FragmentBuilder.aFragment().id("aFragment"));
        this.addToRepository(FragmentBuilder.aFragment().with(itself).build());
        List fetchedFragments = this.repository.getAllNotUsingElement(itself.getId());
        Assertions.assertThat((List)fetchedFragments).containsOnly((Object[])new Fragment[]{expectedFragment});
    }

    @Test
    public void should_get_all_fragment_except_those_which_are_using_a_fragment_that_use_itself() throws Exception {
        Fragment expectedFragment = this.addToRepository(FragmentBuilder.aFragment().id("anotherFragment").build());
        Fragment itself = this.addToRepository(FragmentBuilder.aFragment().id("aFragment").build());
        Fragment container = this.addToRepository(FragmentBuilder.aFragment().with(itself));
        Fragment container2 = this.addToRepository(FragmentBuilder.aFragment().with(container));
        this.addToRepository(FragmentBuilder.aFragment().with(container2));
        List fetchedFragments = this.repository.getAllNotUsingElement(itself.getId());
        Assertions.assertThat((List)fetchedFragments).containsOnly((Object[])new Fragment[]{expectedFragment});
    }

    @Test
    public void should_save_a_fragment_in_a_json_file_repository() throws Exception {
        Fragment expectedFragment = FragmentBuilder.aFilledFragment("fragment-id");
        Assertions.assertThat((File)this.fragmentsPath.resolve(expectedFragment.getId()).resolve(expectedFragment.getId() + ".json").toFile()).doesNotExist();
        this.repository.updateLastUpdateAndSave((Identifiable)expectedFragment);
        Assertions.assertThat((File)this.fragmentsPath.resolve(expectedFragment.getId()).resolve(expectedFragment.getId() + ".json").toFile()).exists();
        Assertions.assertThat((Comparable)expectedFragment.getLastUpdate()).isGreaterThan((Comparable)Instant.now().minus(5000L));
    }

    @Test
    public void should_save_a_page_without_updating_last_update_date() throws Exception {
        Fragment fragment = (Fragment)this.repository.updateLastUpdateAndSave((Identifiable)FragmentBuilder.aFragment().id("customLabel").withName("theFragmentName").build());
        Instant lastUpdate = fragment.getLastUpdate();
        fragment.setName("newName");
        this.repository.save((Identifiable)fragment);
        Fragment fetchedFragment = (Fragment)this.repository.get(fragment.getId());
        Assertions.assertThat((Comparable)fetchedFragment.getLastUpdate()).isEqualTo((Object)lastUpdate);
        Assertions.assertThat((String)fetchedFragment.getName()).isEqualTo((Object)"newName");
    }

    @Test(expected=RepositoryException.class)
    public void should_throw_RepositoryException_when_error_occurs_while_saving_a_fragment() throws Exception {
        Fragment expectedFragment = FragmentBuilder.aFilledFragment("fragment-id");
        ((JsonFileBasedPersister)Mockito.doThrow((Throwable)new IOException()).when(this.persister)).save(this.fragmentsPath.resolve(expectedFragment.getId()), (Identifiable)expectedFragment);
        this.repository.updateLastUpdateAndSave((Identifiable)expectedFragment);
    }

    @Test(expected=IllegalArgumentException.class)
    public void should_throw_IllegalArgumentException_while_saving_a_fragment_with_no_id_set() throws Exception {
        Fragment expectedFragment = FragmentBuilder.aFragment().id(null).build();
        this.repository.updateLastUpdateAndSave((Identifiable)expectedFragment);
    }

    @Test(expected=ConstraintValidationException.class)
    public void should_throw_ConstraintValidationException_while_saving_a_fragment_with_bad_name() throws Exception {
        Fragment expectedFragment = FragmentBuilder.aFragment().id("fragment-id").withName("\u00e9\u00e9&\u00e9&z").build();
        this.repository.updateLastUpdateAndSave((Identifiable)expectedFragment);
    }

    @Test
    public void should_save_all_fragment_in_a_json_file_repository() throws Exception {
        Fragment expectedFragment = FragmentBuilder.aFilledFragment("fragment-id");
        Assertions.assertThat((File)this.fragmentsPath.resolve(expectedFragment.getId()).resolve(expectedFragment.getId() + ".json").toFile()).doesNotExist();
        this.repository.saveAll(Arrays.asList(expectedFragment));
        Assertions.assertThat((File)this.fragmentsPath.resolve(expectedFragment.getId()).resolve(expectedFragment.getId() + ".json").toFile()).exists();
        Assertions.assertThat((Comparable)expectedFragment.getLastUpdate()).isGreaterThan((Comparable)Instant.now().minus(5000L));
    }

    @Test
    public void should_not_thrown_NPE_on_save_all_fragment_when_list_null() {
        this.repository.saveAll(null);
    }

    @Test
    public void should_delete_a_fragment_with_his_json_file_repository() throws Exception {
        Fragment expectedFragment = FragmentBuilder.aFilledFragment("fragment-id");
        this.addToRepository(expectedFragment);
        Assertions.assertThat((File)this.fragmentsPath.resolve(expectedFragment.getId()).resolve(expectedFragment.getId() + ".json").toFile()).exists();
        this.repository.delete(expectedFragment.getId());
        Assertions.assertThat((File)this.fragmentsPath.resolve(expectedFragment.getId()).resolve(expectedFragment.getId() + ".json").toFile()).doesNotExist();
    }

    @Test(expected=NotFoundException.class)
    public void should_throw_NotFoundException_when_deleting_inexisting_fragment() throws Exception {
        this.repository.delete("foo");
    }

    @Test(expected=RepositoryException.class)
    public void should_throw_RepositoryException_when_error_occurs_on_object_included_search() throws Exception {
        Fragment expectedFragment = FragmentBuilder.aFilledFragment("fragment-id");
        ((JsonFileBasedLoader)Mockito.doThrow((Throwable)new IOException()).when(this.loader)).contains(this.fragmentsPath, expectedFragment.getId());
        this.repository.containsObject(expectedFragment.getId());
    }

    @Test(expected=RepositoryException.class)
    public void should_throw_RepositoryException_when_error_occurs_on_object_included_search_list() throws Exception {
        Fragment expectedFragment = FragmentBuilder.aFilledFragment("fragment-id");
        ((JsonFileBasedLoader)Mockito.doThrow((Throwable)new IOException()).when(this.loader)).findByObjectId(this.fragmentsPath, expectedFragment.getId());
        this.repository.findByObjectId(expectedFragment.getId());
    }

    @Test
    public void should_mark_a_widget_as_favorite() throws Exception {
        Fragment fragment = this.addToRepository(FragmentBuilder.aFragment().notFavorite());
        this.repository.markAsFavorite(fragment.getId());
        Fragment fetchedFragment = (Fragment)this.repository.get(fragment.getId());
        Assertions.assertThat((boolean)fetchedFragment.isFavorite()).isTrue();
    }

    @Test
    public void should_unmark_a_widget_as_favorite() throws Exception {
        Fragment fragment = this.addToRepository(FragmentBuilder.aFragment().favorite());
        this.repository.unmarkAsFavorite(fragment.getId());
        Fragment fetchedFragment = (Fragment)this.repository.get(fragment.getId());
        Assertions.assertThat((boolean)fetchedFragment.isFavorite()).isFalse();
    }

    @Test
    public void should_keep_fragment_name_id_if_there_is_no_fragment_with_same_id() throws Exception {
        String newFragmentId = this.repository.getNextAvailableId("newFragmentId");
        Assertions.assertThat((String)newFragmentId).isEqualTo((Object)"newFragmentId");
    }
}

