/*
 * Decompiled with CFR 0.152.
 */
package org.bonitasoft.web.designer.repository;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.concurrent.TimeUnit;
import org.apache.commons.io.FileUtils;
import org.assertj.core.api.AbstractStringAssert;
import org.assertj.core.api.Assertions;
import org.awaitility.Awaitility;
import org.awaitility.core.ThrowingRunnable;
import org.bonitasoft.web.designer.builder.PageBuilder;
import org.bonitasoft.web.designer.builder.SimpleObjectBuilder;
import org.bonitasoft.web.designer.config.DesignerConfig;
import org.bonitasoft.web.designer.model.Identifiable;
import org.bonitasoft.web.designer.model.JacksonObjectMapper;
import org.bonitasoft.web.designer.model.page.Page;
import org.bonitasoft.web.designer.repository.BeanValidator;
import org.bonitasoft.web.designer.repository.JsonFileBasedPersister;
import org.bonitasoft.web.designer.repository.SimpleDesignerArtifact;
import org.bonitasoft.web.designer.repository.exception.ConstraintValidationException;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Matchers;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.runners.MockitoJUnitRunner;
import org.springframework.test.util.ReflectionTestUtils;

@RunWith(value=MockitoJUnitRunner.class)
public class JsonFileBasedPersisterTest {
    private static final String DESIGNER_VERSION = "1.0.0";
    private static final String MODEL_VERSION = "2.0";
    private Path repoDirectory;
    private JacksonObjectMapper objectMapper;
    private JsonFileBasedPersister<SimpleDesignerArtifact> repository;
    @Mock
    private BeanValidator validator;

    @Before
    public void setUp() throws IOException {
        this.repoDirectory = Files.createTempDirectory("jsonrepository", new FileAttribute[0]);
        this.objectMapper = (JacksonObjectMapper)Mockito.spy((Object)new DesignerConfig().objectMapperWrapper());
        this.repository = new JsonFileBasedPersister(this.objectMapper, this.validator);
        ReflectionTestUtils.setField(this.repository, (String)"uidVersion", (Object)DESIGNER_VERSION);
        ReflectionTestUtils.setField(this.repository, (String)"modelVersion", (Object)MODEL_VERSION);
    }

    @After
    public void clean() {
        FileUtils.deleteQuietly((File)this.repoDirectory.toFile());
    }

    private SimpleDesignerArtifact getFromRepository(String id) throws IOException {
        byte[] json = Files.readAllBytes(this.repoDirectory.resolve(id + ".json"));
        return (SimpleDesignerArtifact)((Object)this.objectMapper.fromJson(json, SimpleDesignerArtifact.class));
    }

    @Test
    public void should_serialize_an_object_and_save_it_to_a_file() throws Exception {
        SimpleDesignerArtifact expectedObject = new SimpleDesignerArtifact("foo", "aName", 2);
        this.repository.save(this.repoDirectory, (Identifiable)expectedObject);
        SimpleDesignerArtifact savedObject = this.getFromRepository("foo");
        Assertions.assertThat((Object)((Object)savedObject)).isEqualTo((Object)expectedObject);
    }

    @Test
    public void should_not_set_model_version_while_saving_if_uid_version_does_not_support_model_version() throws Exception {
        SimpleDesignerArtifact expectedObject = new SimpleDesignerArtifact("foo", "aName", 2);
        this.repository.save(this.repoDirectory, (Identifiable)expectedObject);
        SimpleDesignerArtifact savedObject = this.getFromRepository("foo");
        Assertions.assertThat((String)savedObject.getModelVersion()).isEqualTo(null);
    }

    @Test
    public void should_set_model_version_while_saving_if_not_already_set() throws Exception {
        ReflectionTestUtils.setField(this.repository, (String)"uidVersion", (Object)"1.12.0");
        SimpleDesignerArtifact expectedObject = new SimpleDesignerArtifact("foo", "aName", 2);
        this.repository.save(this.repoDirectory, (Identifiable)expectedObject);
        SimpleDesignerArtifact savedObject = this.getFromRepository("foo");
        Assertions.assertThat((String)savedObject.getModelVersion()).isEqualTo((Object)MODEL_VERSION);
        ReflectionTestUtils.setField(this.repository, (String)"uidVersion", (Object)DESIGNER_VERSION);
    }

    @Test
    public void should_not_set_model_version_while_saving_if_already_set() throws Exception {
        ReflectionTestUtils.setField(this.repository, (String)"uidVersion", (Object)"1.12.0");
        SimpleDesignerArtifact expectedObject = new SimpleDesignerArtifact("foo", "aName", 2);
        expectedObject.setModelVersion("alreadySetModelVersion");
        this.repository.save(this.repoDirectory, (Identifiable)expectedObject);
        SimpleDesignerArtifact savedObject = this.getFromRepository("foo");
        Assertions.assertThat((String)savedObject.getModelVersion()).isEqualTo((Object)"alreadySetModelVersion");
        ReflectionTestUtils.setField(this.repository, (String)"uidVersion", (Object)DESIGNER_VERSION);
    }

    @Test(expected=IOException.class)
    public void should_throw_IOException_when_error_occurs_while_saving_a_object() throws Exception {
        ((JacksonObjectMapper)Mockito.doThrow((Throwable)new RuntimeException()).when((Object)this.objectMapper)).toJson(Matchers.anyObject(), (Class)Matchers.any(Class.class));
        this.repository.save(this.repoDirectory, (Identifiable)new SimpleDesignerArtifact());
    }

    @Test
    public void should_validate_beans_before_saving_them() throws Exception {
        ((BeanValidator)Mockito.doThrow(ConstraintValidationException.class).when((Object)this.validator)).validate(Matchers.any(Object.class));
        try {
            this.repository.save(this.repoDirectory, (Identifiable)new SimpleDesignerArtifact("object1", "object1", 1));
            Assertions.failBecauseExceptionWasNotThrown(ConstraintValidationException.class);
        }
        catch (ConstraintValidationException e) {
            Assertions.assertThat((File)this.repoDirectory.resolve("object1.json").toFile()).doesNotExist();
        }
    }

    @Test
    public void should_persist_metadata_in_a_seperate_file() throws Exception {
        SimpleDesignerArtifact artifact = SimpleObjectBuilder.aSimpleObjectBuilder().id("baz").metadata("foobar").build();
        this.repository.save(this.repoDirectory, (Identifiable)artifact);
        Assertions.assertThat((String)new String(Files.readAllBytes(this.repoDirectory.getParent().resolve(".metadata/baz.json")))).isEqualTo((Object)"{\"favorite\":false,\"metadata\":\"foobar\"}");
    }

    @Test
    public void should_support_parrelel_index_saves() throws Exception {
        Page page1 = PageBuilder.aPage().withUUID("baz-uuid").withName("baz").withId("baz-id").build();
        Page page2 = PageBuilder.aPage().withUUID("foo-uuid").withName("foo").withId("foo-id").build();
        JsonFileBasedPersister pageRepository = new JsonFileBasedPersister(this.objectMapper, this.validator);
        final Path metadataFolder = this.repoDirectory.resolve(".metadata");
        metadataFolder.toFile().mkdir();
        new Thread(() -> {
            try {
                pageRepository.saveInIndex(metadataFolder, (Identifiable)page2);
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }).start();
        new Thread(() -> {
            try {
                pageRepository.saveInIndex(metadataFolder, (Identifiable)page1);
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }).start();
        Awaitility.await().atMost(3L, TimeUnit.SECONDS).untilAsserted(new ThrowingRunnable(){

            public void run() throws Throwable {
                String index = new String(Files.readAllBytes(metadataFolder.resolve(".index.json")));
                ((AbstractStringAssert)Assertions.assertThat((String)index).contains(new CharSequence[]{"\"baz-uuid\":\"baz-id\""})).contains(new CharSequence[]{"\"foo-uuid\":\"foo-id\""});
            }
        });
    }

    @Test
    public void should_persist_all_artifact_id_in_index_when_refresh_indexing_is_called() throws Exception {
        ArrayList<Page> pages = new ArrayList<Page>();
        Page page = PageBuilder.aPage().withUUID("baz-uuid").withId("page1").build();
        Page page2 = PageBuilder.aPage().withUUID("foo-uuid").withId("page2").withName("page2").build();
        pages.add(page);
        pages.add(page2);
        JsonFileBasedPersister pageRepository = new JsonFileBasedPersister(this.objectMapper, this.validator);
        final Path metadataFolder = this.repoDirectory.resolve(".metadata");
        metadataFolder.toFile().mkdir();
        new Thread(() -> {
            try {
                pageRepository.refreshIndexing(metadataFolder, pages);
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }).start();
        Awaitility.await().atMost(3L, TimeUnit.SECONDS).untilAsserted(new ThrowingRunnable(){

            public void run() throws Throwable {
                String index = new String(Files.readAllBytes(metadataFolder.resolve(".index.json")));
                ((AbstractStringAssert)Assertions.assertThat((String)index).contains(new CharSequence[]{"\"baz-uuid\":\"page1\""})).contains(new CharSequence[]{"\"foo-uuid\":\"page2\""});
            }
        });
    }
}

