/*
 * Decompiled with CFR 0.152.
 */
package org.bonitasoft.web.designer.repository;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.google.common.collect.Sets;
import java.io.File;
import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import javax.validation.Validation;
import javax.validation.ValidatorFactory;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.ListAssert;
import org.bonitasoft.web.designer.builder.PropertyBuilder;
import org.bonitasoft.web.designer.builder.WidgetBuilder;
import org.bonitasoft.web.designer.config.DesignerConfig;
import org.bonitasoft.web.designer.livebuild.PathListener;
import org.bonitasoft.web.designer.livebuild.Watcher;
import org.bonitasoft.web.designer.model.Identifiable;
import org.bonitasoft.web.designer.model.JacksonObjectMapper;
import org.bonitasoft.web.designer.model.widget.Property;
import org.bonitasoft.web.designer.model.widget.Widget;
import org.bonitasoft.web.designer.repository.BeanValidator;
import org.bonitasoft.web.designer.repository.JsonFileBasedLoader;
import org.bonitasoft.web.designer.repository.JsonFileBasedPersister;
import org.bonitasoft.web.designer.repository.WidgetFileBasedLoader;
import org.bonitasoft.web.designer.repository.WidgetRepository;
import org.bonitasoft.web.designer.repository.exception.ConstraintValidationException;
import org.bonitasoft.web.designer.repository.exception.NotAllowedException;
import org.bonitasoft.web.designer.repository.exception.NotFoundException;
import org.bonitasoft.web.designer.repository.exception.RepositoryException;
import org.joda.time.Instant;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;
import org.junit.runner.RunWith;
import org.mockito.Matchers;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.runners.MockitoJUnitRunner;
import org.springframework.test.util.ReflectionTestUtils;

@RunWith(value=MockitoJUnitRunner.class)
public class WidgetRepositoryTest {
    private static final String DESIGNER_VERSION = "1.0.0";
    private static final String MODEL_VERSION = "2.0";
    private WidgetRepository widgetRepository;
    private JacksonObjectMapper objectMapper;
    private Path widgetDirectory;
    private JsonFileBasedPersister<Widget> jsonFileRepository;
    @Rule
    public TemporaryFolder temporaryFolder = new TemporaryFolder();
    @Rule
    public ExpectedException expectedException = ExpectedException.none();
    @Mock
    private Watcher watcher;

    @Before
    public void setUp() throws IOException {
        this.widgetDirectory = Paths.get(this.temporaryFolder.getRoot().getPath(), new String[0]);
        this.jsonFileRepository = new DesignerConfig().widgetFileBasedPersister();
        ValidatorFactory validatorFactory = Validation.buildDefaultValidatorFactory();
        this.objectMapper = (JacksonObjectMapper)Mockito.spy((Object)new DesignerConfig().objectMapperWrapper());
        this.widgetRepository = new WidgetRepository(this.widgetDirectory, this.jsonFileRepository, (JsonFileBasedLoader)new WidgetFileBasedLoader(this.objectMapper), new BeanValidator(validatorFactory.getValidator()), this.watcher);
        ReflectionTestUtils.setField(this.jsonFileRepository, (String)"uidVersion", (Object)DESIGNER_VERSION);
        ReflectionTestUtils.setField(this.jsonFileRepository, (String)"modelVersion", (Object)MODEL_VERSION);
    }

    @Test
    public void should_get_a_widget_by_its_id() throws Exception {
        Widget expectedWidget = WidgetBuilder.aWidget().id("input").build();
        Widget notExpectedWidget = WidgetBuilder.aWidget().id("label").build();
        this.addToRepository(expectedWidget, notExpectedWidget);
        Widget widget = (Widget)this.widgetRepository.get("input");
        Assertions.assertThat((Object)widget).isEqualTo((Object)expectedWidget);
    }

    @Test(expected=NotFoundException.class)
    public void should_throw_NotFoundException_when_trying_to_get_an_unexisting_widget() throws Exception {
        this.widgetRepository.get("notExistingWidget");
    }

    @Test(expected=RepositoryException.class)
    public void should_throw_RepositoryException_when_error_occurs_when_getting_a_widget() throws Exception {
        ((JacksonObjectMapper)Mockito.doThrow((Throwable)new IOException()).when((Object)this.objectMapper)).fromJson((byte[])Matchers.any(byte[].class), (Class)Matchers.eq(Widget.class));
        this.addToRepository(WidgetBuilder.aWidget().id("input").build());
        this.widgetRepository.get("input");
    }

    @Test
    public void should_retrieve_all_widgets() throws Exception {
        Widget input = WidgetBuilder.aWidget().id("input").build();
        Widget label = WidgetBuilder.aWidget().id("label").build();
        this.addToRepository(input, label);
        List widgets = this.widgetRepository.getAll();
        Assertions.assertThat((List)widgets).containsOnly((Object[])new Widget[]{input, label});
    }

    @Test(expected=RepositoryException.class)
    public void should_throw_RepositoryException_if_error_occurs_while_getting_all_widgets() throws Exception {
        ((JacksonObjectMapper)Mockito.doThrow((Throwable)new IOException()).when((Object)this.objectMapper)).fromJson((byte[])Matchers.any(byte[].class), (Class)Matchers.eq(Widget.class));
        this.addToRepository(WidgetBuilder.aWidget().id("input").build());
        this.widgetRepository.getAll();
    }

    @Test
    public void should_save_a_custom_widget() throws Exception {
        Widget customLabel = WidgetBuilder.aWidget().custom().id("customLabel").build();
        Files.createDirectories(this.widgetDirectory.resolve("customLabel"), new FileAttribute[0]);
        this.widgetRepository.updateLastUpdateAndSave((Identifiable)customLabel);
        Assertions.assertThat((File)this.jsonFile(customLabel)).exists();
        Assertions.assertThat((Comparable)customLabel.getLastUpdate()).isGreaterThan((Comparable)Instant.now().minus(5000L));
    }

    @Test
    public void should_save_a_page_without_updating_last_update_date() throws Exception {
        Widget widget = (Widget)this.widgetRepository.updateLastUpdateAndSave((Identifiable)WidgetBuilder.aWidget().id("customLabel").name("theWidgetName").build());
        Instant lastUpdate = widget.getLastUpdate();
        widget.setName("newName");
        this.widgetRepository.save((Identifiable)widget);
        Widget fetchedWidget = (Widget)this.widgetRepository.get(widget.getId());
        Assertions.assertThat((Comparable)fetchedWidget.getLastUpdate()).isEqualTo((Object)lastUpdate);
        Assertions.assertThat((String)fetchedWidget.getName()).isEqualTo((Object)"newName");
    }

    @Test
    public void should_save_a_list_of_custom_widgets() throws Exception {
        Widget customLabel = WidgetBuilder.aWidget().custom().id("customLabel").build();
        Widget customInput = WidgetBuilder.aWidget().custom().id("customInput").build();
        Files.createDirectories(this.widgetDirectory.resolve("customLabel"), new FileAttribute[0]);
        this.widgetRepository.saveAll(Arrays.asList(customInput, customLabel));
        Assertions.assertThat((File)this.jsonFile(customLabel)).exists();
        Assertions.assertThat((File)this.jsonFile(customInput)).exists();
    }

    @Test
    public void should_set_model_version_while_saving_if_not_already_set() throws Exception {
        Widget customLabel = WidgetBuilder.aWidget().custom().id("customLabel").designerVersion("1.12.0").build();
        Files.createDirectories(this.widgetDirectory.resolve("customLabel"), new FileAttribute[0]);
        this.widgetRepository.updateLastUpdateAndSave((Identifiable)customLabel);
        Assertions.assertThat((String)customLabel.getModelVersion()).isEqualTo((Object)MODEL_VERSION);
    }

    @Test
    public void should_not_set_model_version_while_saving_if_already_set() throws Exception {
        Widget customLabel = WidgetBuilder.aWidget().custom().id("customLabel").build();
        customLabel.setModelVersion("alreadySetModelVersion");
        Files.createDirectories(this.widgetDirectory.resolve("customLabel"), new FileAttribute[0]);
        this.widgetRepository.updateLastUpdateAndSave((Identifiable)customLabel);
        Assertions.assertThat((String)customLabel.getModelVersion()).isEqualTo((Object)"alreadySetModelVersion");
    }

    @Test(expected=IllegalArgumentException.class)
    public void should_throw_IllegalArgumentException_while_saving_a_custom_widget_with_no_id_set() throws Exception {
        Widget aWidget = WidgetBuilder.aWidget().id(null).custom().build();
        this.widgetRepository.updateLastUpdateAndSave((Identifiable)aWidget);
    }

    @Test(expected=ConstraintValidationException.class)
    public void should_not_allow_to_save_a_widget_without_name() throws Exception {
        Widget widget = WidgetBuilder.aWidget().name(" ").build();
        this.widgetRepository.updateLastUpdateAndSave((Identifiable)widget);
    }

    @Test(expected=ConstraintValidationException.class)
    public void should_not_allow_to_save_a_widget_with_name_containing_non_alphanumeric_chars() throws Exception {
        Widget widget = WidgetBuilder.aWidget().name("h\u00e9llo").build();
        this.widgetRepository.create(widget);
    }

    @Test(expected=ConstraintValidationException.class)
    public void should_not_allow_to_save_as_widget_with_an_invalid_property() throws Exception {
        Widget widget = WidgetBuilder.aWidget().property(PropertyBuilder.aProperty().name("ze invalid name")).custom().build();
        this.widgetRepository.create(widget);
    }

    @Test(expected=ConstraintValidationException.class)
    public void should_allow_to_save_a_custom_widget_with_name_containing_space() throws Exception {
        Widget widget = WidgetBuilder.aWidget().name("hello world").custom().build();
        this.widgetRepository.create(widget);
    }

    @Test
    public void should_allow_to_save_a_widget_with_name_containing_space() throws Exception {
        Widget widget = WidgetBuilder.aWidget().name("hello world").build();
        Files.createDirectories(this.widgetDirectory.resolve("anId"), new FileAttribute[0]);
        this.widgetRepository.updateLastUpdateAndSave((Identifiable)widget);
    }

    @Test
    public void should_delete_a_custom_widget() throws Exception {
        Widget customLabel = WidgetBuilder.aWidget().id("customLabel").build();
        customLabel.setController("$scope.hello = 'Hello'");
        customLabel.setTemplate("<div>{{ hello + 'there'}}</div>");
        customLabel.setCustom(true);
        Files.createDirectories(this.widgetDirectory.resolve("customLabel"), new FileAttribute[0]);
        this.widgetRepository.updateLastUpdateAndSave((Identifiable)customLabel);
        Files.write(this.widgetDirectory.resolve("customLabel").resolve("customLabel.js"), this.objectMapper.toJson((Object)""), new OpenOption[0]);
        this.widgetRepository.delete("customLabel");
        Assertions.assertThat((File)this.jsonFile(customLabel)).doesNotExist();
        Assertions.assertThat((File)this.jsFile(customLabel)).doesNotExist();
    }

    @Test(expected=NotAllowedException.class)
    public void should_not_allow_to_delete_a_pb_widget() throws Exception {
        Widget pbLabel = WidgetBuilder.aWidget().id("pbLabel").build();
        pbLabel.setCustom(false);
        this.addToRepository(pbLabel);
        this.widgetRepository.delete("pbLabel");
    }

    @Test(expected=ConstraintValidationException.class)
    public void should_not_allow_to_create_a_widget_without_name() throws Exception {
        Widget widget = WidgetBuilder.aWidget().name(" ").build();
        this.widgetRepository.create(widget);
    }

    @Test(expected=ConstraintValidationException.class)
    public void should_not_allow_to_create_a_widget_with_name_containing_non_alphanumeric_chars() throws Exception {
        Widget widget = WidgetBuilder.aWidget().name("h\u00e9llo").build();
        this.widgetRepository.create(widget);
    }

    @Test(expected=ConstraintValidationException.class)
    public void should_not_allow_to_create_a_custom_widget_with_name_containing_non_space() throws Exception {
        Widget widget = WidgetBuilder.aWidget().name("hello world").custom().build();
        this.widgetRepository.create(widget);
    }

    @Test(expected=ConstraintValidationException.class)
    public void should_allow_create_a_widget_with_name_containing_space_for_normal_widget() throws Exception {
        Widget widget = WidgetBuilder.aWidget().name("hello world").build();
        this.widgetRepository.create(widget);
    }

    @Test
    public void should_create_a_widget_and_set_his_id() throws Exception {
        Widget expectedWidget = WidgetBuilder.aWidget().name("aName").build();
        Widget createdWidget = this.widgetRepository.create(expectedWidget);
        expectedWidget.setId("customAName");
        expectedWidget.setCustom(true);
        Assertions.assertThat((Object)expectedWidget).isEqualTo((Object)createdWidget);
        Assertions.assertThat((File)this.jsonFile(createdWidget)).exists();
    }

    @Test(expected=NotAllowedException.class)
    public void should_not_allow_to_create_a_widget_with_an_already_existing_name() throws Exception {
        Widget widget = WidgetBuilder.aWidget().name("existingName").id("customExistingName").build();
        this.addToRepository(widget);
        this.widgetRepository.create(WidgetBuilder.aWidget().name("existingName").build());
    }

    @Test
    public void should_verify_that_a_widget_exists_in_the_repository() throws Exception {
        Files.write(this.temporaryFolder.newFolder("pbInput").toPath().resolve("pbInput.json"), "contents".getBytes(), new OpenOption[0]);
        Assertions.assertThat((boolean)this.widgetRepository.exists("pbInput")).isTrue();
        Assertions.assertThat((boolean)this.widgetRepository.exists("pbLabel")).isFalse();
    }

    @Test
    public void should_save_a_new_property() throws Exception {
        Widget aWidget = this.addToRepository(WidgetBuilder.aWidget().custom().build());
        Property expectedProperty = PropertyBuilder.aProperty().build();
        this.widgetRepository.addProperty(aWidget.getId(), expectedProperty);
        Widget widget = this.getFromRepository(aWidget.getId());
        Assertions.assertThat((List)widget.getProperties()).contains((Object[])new Property[]{expectedProperty});
    }

    @Test(expected=NotAllowedException.class)
    public void should_not_allow_to_save_a_new_property_when_property_with_same_name_already_exists() throws Exception {
        Property alreadyAddedProperty = PropertyBuilder.aProperty().build();
        Widget aWidget = this.addToRepository(WidgetBuilder.aWidget().custom().property(alreadyAddedProperty).build());
        this.widgetRepository.addProperty(aWidget.getId(), alreadyAddedProperty);
    }

    @Test
    public void should_update_an_existing_property() throws Exception {
        Property initialParam = PropertyBuilder.aProperty().name("propertyName").label("propertyLabel").build();
        Property updatedParam = PropertyBuilder.aProperty().name("newName").label("newLablel").build();
        Widget aWidget = this.addToRepository(WidgetBuilder.aWidget().custom().property(initialParam).build());
        this.widgetRepository.updateProperty(aWidget.getId(), initialParam.getName(), updatedParam);
        Widget widget = this.getFromRepository(aWidget.getId());
        Assertions.assertThat((List)widget.getProperties()).contains((Object[])new Property[]{updatedParam});
        Assertions.assertThat((List)widget.getProperties()).doesNotContain((Object[])new Property[]{initialParam});
    }

    @Test(expected=NotFoundException.class)
    public void should_fail_when_trying_to_update_a_not_existing_property() throws Exception {
        Widget expectedWidget = this.addToRepository(WidgetBuilder.aWidget().custom().build());
        this.widgetRepository.updateProperty(expectedWidget.getId(), "notExistingProperty", new Property());
    }

    @Test
    public void should_delete_a_widget_property() throws Exception {
        Property aProperty = PropertyBuilder.aProperty().name("aParam").build();
        Widget aWidget = this.addToRepository(WidgetBuilder.aWidget().property(aProperty).property(PropertyBuilder.aProperty().name("anotherParam")).build());
        List properties = this.widgetRepository.deleteProperty(aWidget.getId(), "aParam");
        Widget widget = this.getFromRepository(aWidget.getId());
        Assertions.assertThat((List)widget.getProperties()).doesNotContain((Object[])new Property[]{aProperty});
        Assertions.assertThat((List)properties).containsOnlyElementsOf((Iterable)widget.getProperties());
    }

    @Test(expected=NotFoundException.class)
    public void should_fail_when_trying_to_delete_a_property_on_an_unknown_widget() throws Exception {
        this.widgetRepository.deleteProperty("unknownWidget", "aParam");
    }

    @Test(expected=NotFoundException.class)
    public void should_fail_when_trying_to_delete_an_unknown_property() throws Exception {
        Widget aWidget = this.addToRepository(WidgetBuilder.aWidget().build());
        this.widgetRepository.deleteProperty(aWidget.getId(), "unknownPrameter");
    }

    @Test
    public void should_find_widget_which_use_another_widget() throws Exception {
        Widget input = WidgetBuilder.aWidget().id("input").build();
        Widget label = WidgetBuilder.aWidget().id("label").template("use <input>").build();
        this.addToRepository(input, label);
        Assertions.assertThat((List)this.widgetRepository.findByObjectId("input")).extracting("id").containsExactly(new Object[]{"label"});
    }

    @Test
    public void should_find_widget_which_not_use_another_widget() throws Exception {
        Widget input = WidgetBuilder.aWidget().id("input").build();
        Widget label = WidgetBuilder.aWidget().id("label").template("use <input>").build();
        this.addToRepository(input, label);
        Assertions.assertThat((List)this.widgetRepository.findByObjectId("label")).isEmpty();
    }

    @Test
    public void should_find_widget_by_id() throws Exception {
        Widget input = WidgetBuilder.aWidget().id("input").build();
        Widget label = WidgetBuilder.aWidget().id("label").build();
        this.addToRepository(input, label);
        ((ListAssert)Assertions.assertThat((List)this.widgetRepository.getByIds((Set)Sets.newHashSet((Object[])new String[]{"input", "label"}))).hasSize(2)).extracting("id").containsOnly(new Object[]{"input", "label"});
    }

    @Test
    public void should_walk_widget_repository() throws Exception {
        File file = this.temporaryFolder.newFile("file");
        final ArrayList visitedPaths = new ArrayList();
        this.widgetRepository.walk((FileVisitor)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                visitedPaths.add(file);
                return super.visitFile(file, attrs);
            }
        });
        Assertions.assertThat(visitedPaths).containsExactly((Object[])new Path[]{file.toPath()});
    }

    @Test
    public void should_watch_widget_repository() throws Exception {
        PathListener pathListener = path -> System.out.println(path);
        this.widgetRepository.watch(pathListener);
        ((Watcher)Mockito.verify((Object)this.watcher)).watch(this.widgetDirectory, pathListener);
    }

    @Test
    public void should_mark_a_widget_as_favorite() throws Exception {
        Widget widget = this.addToRepository(WidgetBuilder.aWidget().notFavorite());
        this.widgetRepository.markAsFavorite(widget.getId());
        Widget fetchedWidget = this.getFromRepository(widget.getId());
        Assertions.assertThat((boolean)fetchedWidget.isFavorite()).isTrue();
    }

    @Test
    public void should_unmark_a_widget_as_favorite() throws Exception {
        Widget widget = this.addToRepository(WidgetBuilder.aWidget().favorite());
        this.widgetRepository.unmarkAsFavorite(widget.getId());
        Widget fetchedWidget = this.getFromRepository(widget.getId());
        Assertions.assertThat((boolean)fetchedWidget.isFavorite()).isFalse();
    }

    private void addToRepository(Widget ... widgets) throws Exception {
        for (Widget widget : widgets) {
            this.addToRepository(widget);
        }
    }

    private Widget addToRepository(WidgetBuilder widget) throws Exception {
        return this.addToRepository(widget.build());
    }

    private Widget addToRepository(Widget widget) throws Exception {
        Path widgetDir = Files.createDirectory(this.widgetDirectory.resolve(widget.getId()), new FileAttribute[0]);
        this.writeWidgetMetadataInFile(widget, widgetDir.resolve(widget.getId() + ".json"));
        return this.getFromRepository(widget.getId());
    }

    private Widget getFromRepository(String widgetId) {
        return (Widget)this.widgetRepository.get(widgetId);
    }

    private void writeWidgetMetadataInFile(Widget widget, Path path) throws IOException {
        ObjectWriter writer = new ObjectMapper().writer();
        writer.writeValue(path.toFile(), (Object)widget);
    }

    private File jsonFile(Widget widget) {
        return this.jsonFile(widget.getId());
    }

    private File jsonFile(String widgetId) {
        return this.widgetDirectory.resolve(widgetId).resolve(widgetId + ".json").toFile();
    }

    private File jsFile(Widget widget) {
        return this.widgetDirectory.resolve(widget.getId()).resolve(widget.getId() + ".js").toFile();
    }
}

