/*
 * Decompiled with CFR 0.152.
 */
package org.bonitasoft.web.designer.repository;

import java.io.File;
import java.net.URI;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.List;
import java.util.NoSuchElementException;
import org.assertj.core.api.AbstractByteArrayAssert;
import org.assertj.core.api.Assertions;
import org.bonitasoft.web.designer.builder.AssetBuilder;
import org.bonitasoft.web.designer.builder.PageBuilder;
import org.bonitasoft.web.designer.model.Identifiable;
import org.bonitasoft.web.designer.model.asset.Asset;
import org.bonitasoft.web.designer.model.asset.AssetType;
import org.bonitasoft.web.designer.model.page.Page;
import org.bonitasoft.web.designer.repository.AssetRepository;
import org.bonitasoft.web.designer.repository.BeanValidator;
import org.bonitasoft.web.designer.repository.PageRepository;
import org.bonitasoft.web.designer.repository.exception.ConstraintValidationException;
import org.bonitasoft.web.designer.repository.exception.NotAllowedException;
import org.bonitasoft.web.designer.repository.exception.NotFoundException;
import org.bonitasoft.web.designer.utils.rule.TemporaryFolder;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.runner.RunWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.runners.MockitoJUnitRunner;

@RunWith(value=MockitoJUnitRunner.class)
public class AssetRepositoryTest {
    @Rule
    public TemporaryFolder temporaryFolder = new TemporaryFolder();
    @Rule
    public ExpectedException exception = ExpectedException.none();
    private Path pagesPath;
    @Mock
    private BeanValidator validator;
    @Mock
    private PageRepository pageRepository;
    @InjectMocks
    private AssetRepository<Page> assetRepository;

    @Before
    public void setUp() throws Exception {
        this.pagesPath = Paths.get(this.temporaryFolder.getRoot().getPath(), new String[0]);
    }

    @Test
    public void should_resolveAssetPath() {
        Page page = PageBuilder.aPage().withId("page-id").build();
        Asset asset = AssetBuilder.aFilledAsset(page);
        Mockito.when((Object)this.pageRepository.resolvePathFolder("page-id")).thenReturn((Object)this.pagesPath);
        Path path = this.assetRepository.resolveAssetPath(asset);
        Assertions.assertThat((URI)path.toUri()).isEqualTo((Object)this.pagesPath.resolve("assets").resolve("js").resolve(asset.getName()).toUri());
    }

    @Test
    public void should_not_resolveAssetPath_when_asset_invalid() {
        Page page = PageBuilder.aPage().withId("page-id").build();
        Asset asset = AssetBuilder.aFilledAsset(page);
        this.exception.expect(ConstraintValidationException.class);
        ((BeanValidator)Mockito.doThrow(ConstraintValidationException.class).when((Object)this.validator)).validate((Object)asset);
        this.assetRepository.resolveAssetPath(asset);
    }

    @Test
    public void should_save_asset() throws Exception {
        Page page = PageBuilder.aPage().withId("page-id").build();
        Asset asset = AssetBuilder.aFilledAsset(page);
        Path fileExpected = this.pagesPath.resolve("assets").resolve("js").resolve(asset.getName());
        Assertions.assertThat((File)fileExpected.toFile()).doesNotExist();
        Mockito.when((Object)this.pageRepository.resolvePathFolder("page-id")).thenReturn((Object)this.pagesPath);
        Mockito.when((Object)this.pageRepository.get(asset.getComponentId())).thenReturn((Object)page);
        this.assetRepository.save(asset, "My example with special characters r\u00e9\u00e8@# \ntest".getBytes(Charset.forName("UTF-8")));
        Assertions.assertThat((File)fileExpected.toFile()).exists();
        Assertions.assertThat((String)com.google.common.io.Files.readFirstLine((File)fileExpected.toFile(), (Charset)Charset.forName("UTF-8"))).isEqualTo((Object)"My example with special characters r\u00e9\u00e8@# ");
    }

    @Test(expected=NullPointerException.class)
    public void should_throw_NullPointerException_when_deleting_asset_componentId_null() throws Exception {
        this.assetRepository.delete(new Asset());
    }

    @Test
    public void should_delete_asset() throws Exception {
        Page page = PageBuilder.aPage().withId("page-id").build();
        Asset asset = AssetBuilder.aFilledAsset(page);
        Path fileExpected = this.pagesPath.resolve("assets").resolve("js").resolve(asset.getName());
        Files.createDirectories(this.pagesPath.resolve("assets").resolve("js"), new FileAttribute[0]);
        this.temporaryFolder.newFilePath("assets/js/" + asset.getName());
        Assertions.assertThat((File)fileExpected.toFile()).exists();
        Mockito.when((Object)this.pageRepository.resolvePathFolder("page-id")).thenReturn((Object)this.pagesPath);
        Mockito.when((Object)this.pageRepository.get(asset.getComponentId())).thenReturn((Object)page);
        this.assetRepository.delete(asset);
        Assertions.assertThat((File)fileExpected.toFile()).doesNotExist();
    }

    @Test(expected=NotFoundException.class)
    public void should_throw_NotFoundException_when_deleting_inexisting_page() throws Exception {
        Page page = PageBuilder.aPage().withId("page-id").build();
        Asset asset = AssetBuilder.aFilledAsset(page);
        Mockito.when((Object)this.pageRepository.resolvePathFolder("page-id")).thenReturn((Object)this.pagesPath);
        Mockito.when((Object)this.pageRepository.get(asset.getComponentId())).thenReturn((Object)page);
        this.assetRepository.delete(asset);
    }

    @Test
    public void should_readAllBytes_asset() throws Exception {
        Page page = PageBuilder.aPage().withId("page-id").build();
        Asset asset = AssetBuilder.aFilledAsset(page);
        Path fileExpected = this.pagesPath.resolve("assets").resolve("js").resolve(asset.getName());
        Files.createDirectories(this.pagesPath.resolve("assets").resolve("js"), new FileAttribute[0]);
        this.temporaryFolder.newFilePath("assets/js/" + asset.getName());
        Assertions.assertThat((File)fileExpected.toFile()).exists();
        Mockito.when((Object)this.pageRepository.resolvePathFolder("page-id")).thenReturn((Object)this.pagesPath);
        Mockito.when((Object)this.pageRepository.get(asset.getComponentId())).thenReturn((Object)page);
        ((AbstractByteArrayAssert)Assertions.assertThat((byte[])this.assetRepository.readAllBytes(asset)).isNotNull()).isEmpty();
    }

    @Test(expected=NullPointerException.class)
    public void should_throw_NullPointerException_when_reading_asset_with_component_id_null() throws Exception {
        this.assetRepository.readAllBytes(new Asset());
    }

    @Test(expected=NotFoundException.class)
    public void should_throw_NotFoundException_when_reading_inexisting_page() throws Exception {
        Page page = PageBuilder.aPage().withId("page-id").build();
        Asset asset = AssetBuilder.aFilledAsset(page);
        Mockito.when((Object)this.pageRepository.resolvePathFolder("page-id")).thenReturn((Object)this.pagesPath);
        Mockito.when((Object)this.pageRepository.get(asset.getComponentId())).thenReturn((Object)page);
        this.assetRepository.readAllBytes(asset);
    }

    @Test
    public void should_find_asset_path_used_by_a_component() throws Exception {
        Page page = PageBuilder.aPage().withId("page-id").build();
        Asset asset = AssetBuilder.aFilledAsset(page);
        System.out.println(asset.getName());
        this.pagesPath.resolve("assets").resolve("js").resolve(asset.getName());
        Files.createDirectories(this.pagesPath.resolve("assets").resolve("js"), new FileAttribute[0]);
        this.temporaryFolder.newFilePath("assets/js/" + asset.getName());
        Mockito.when((Object)this.pageRepository.resolvePathFolder("page-id")).thenReturn((Object)this.pagesPath);
        Mockito.when((Object)this.pageRepository.get("page-id")).thenReturn((Object)PageBuilder.aPage().withAsset(asset).build());
        Assertions.assertThat((File)this.assetRepository.findAssetPath("page-id", "myasset.js", AssetType.JAVASCRIPT).toFile()).exists();
    }

    @Test
    public void should_throw_NotAllowedException_when_find_external_asset() throws Exception {
        this.exception.expect(NotAllowedException.class);
        this.exception.expectMessage("We can't load an external asset. Use the link http://mycdnserver.myasset.js");
        Page page = PageBuilder.aPage().withId("page-id").build();
        Asset asset = AssetBuilder.aFilledAsset(page);
        asset.setName("http://mycdnserver.myasset.js");
        asset.setExternal(true);
        Mockito.when((Object)this.pageRepository.get("page-id")).thenReturn((Object)PageBuilder.aPage().withAsset(asset).build());
        this.assetRepository.findAssetPath("page-id", "http://mycdnserver.myasset.js", AssetType.JAVASCRIPT);
    }

    @Test
    public void should_throw_NullPointerException_when_find_asset_with_filename_null() throws Exception {
        this.exception.expect(NullPointerException.class);
        this.exception.expectMessage("Filename is required");
        this.assetRepository.findAssetPath("page-id", null, AssetType.JAVASCRIPT);
    }

    @Test
    public void should_throw_NullPointerException_when_find_asset_path_with_type_null() throws Exception {
        this.exception.expect(NullPointerException.class);
        this.exception.expectMessage("Asset type is required");
        this.assetRepository.findAssetPath("page-id", "myfile.js", null);
    }

    @Test(expected=NoSuchElementException.class)
    public void should_throw_NoSuchElementException_when_finding_inexistant_asset() throws Exception {
        Page page = PageBuilder.aPage().withId("page-id").build();
        Asset asset = AssetBuilder.aFilledAsset(page);
        this.pagesPath.resolve(asset.getName());
        this.temporaryFolder.newFilePath(asset.getName());
        Mockito.when((Object)this.pageRepository.resolvePathFolder("page-id")).thenReturn((Object)this.pagesPath);
        Mockito.when((Object)this.pageRepository.get("page-id")).thenReturn((Object)PageBuilder.aPage().withAsset(asset).build());
        this.assetRepository.findAssetPath("page-id", "inexistant.js", AssetType.JAVASCRIPT);
    }

    @Test
    public void should_findAssetInPath_asset() throws Exception {
        Page page = PageBuilder.aPage().withId("page-id").build();
        Files.write(this.pagesPath.resolve("file1.css"), "<style>.maclass1{}</style>".getBytes(), new OpenOption[0]);
        Files.write(this.pagesPath.resolve("file2.css"), "<style>.maclass2{}</style>".getBytes(), new OpenOption[0]);
        List assets = this.assetRepository.findAssetInPath((Identifiable)page, AssetType.CSS, this.pagesPath);
        Assertions.assertThat((List)assets).hasSize(2);
        Assertions.assertThat((List)assets).extracting("name").contains(new Object[]{"file1.css", "file2.css"});
    }

    @Test
    public void should_findAssetInPath_asset_when_noone_is_present() throws Exception {
        Page page = PageBuilder.aPage().withId("page-id").build();
        List assets = this.assetRepository.findAssetInPath((Identifiable)page, AssetType.CSS, this.pagesPath);
        Assertions.assertThat((List)assets).isEmpty();
    }

    @Test
    public void should_refresh_component_assets_from_disk() throws Exception {
        Page page = PageBuilder.aPage().withId("page-id").withAsset(AssetBuilder.anAsset().withName("existing-asset.js")).build();
        this.temporaryFolder.newFolder(new String[]{"page-id", "assets", "css"});
        Files.write(this.pagesPath.resolve("page-id/assets/css/file1.css"), "<style>.maclass1{}</style>".getBytes(), new OpenOption[0]);
        Files.write(this.pagesPath.resolve("page-id/assets/css/file2.css"), "<style>.maclass2{}</style>".getBytes(), new OpenOption[0]);
        Mockito.when((Object)this.pageRepository.resolvePath("page-id")).thenReturn((Object)this.pagesPath.resolve("page-id"));
        this.assetRepository.refreshAssets((Identifiable)page);
        Assertions.assertThat((Iterable)page.getAssets()).hasSize(3);
        Assertions.assertThat((Iterable)page.getAssets()).extracting("name").contains(new Object[]{"file1.css", "file2.css", "existing-asset.js"});
        ((PageRepository)Mockito.verify((Object)this.pageRepository)).updateLastUpdateAndSave((Identifiable)page);
    }
}

