/*
 * Copyright (c) 2011-2015, Peter Abeles. All Rights Reserved.
 *
 * This file is part of BoofCV (http://boofcv.org).
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package boofcv.factory.shape;

import boofcv.alg.shapes.edge.PolygonEdgeIntensity;
import boofcv.alg.shapes.polygon.BinaryPolygonDetector;
import boofcv.alg.shapes.polyline.MinimizeEnergyPrune;
import boofcv.struct.Configuration;

/**
 * Configuration for {@link BinaryPolygonDetector} for use in {@link FactoryShapeDetector}.
 *
 * @author Peter Abeles
 */
public class ConfigPolygonDetector implements Configuration {

	/**
	 * Minimum number of sides
	 */
	public int minimumSides;

	/**
	 * Maximum number of sides
	 */
	public int maximumSides;

	/**
	 * If false then polygons which touch the image border are pruned
	 */
	public boolean canTouchBorder = false;

	/**
	 * A line is split if a point along the contour between the two end points has a distance from the line
	 * which is greater than this fraction of the line's length
	 */
	public double contour2Poly_splitFraction = 0.10;

	/**
	 * Number of split and merge iterations when converting contour into polygon
	 */
	public int contour2Poly_iterations = 10;

	/**
	 * The minimum allowed length of a side as a fraction of the total contour length
	 */
	public double contour2Poly_minimumSideFraction = 0.025;

	/**
	 * Magic number used to penalize a polygon when a new corner is added
	 *
	 * @see MinimizeEnergyPrune
	 */
	public double splitPenalty = 2.0;

	/**
	 * <p>
	 * The minimum allowed edge intensity for a shape.  Used to remove false positives generated by noise, which
	 * is especially common when using a local threshold during binarization.
	 * </p>
	 *
	 * <p>Set to zero to disable.</p>
	 *
	 * @see PolygonEdgeIntensity
	 */
	public double minimumEdgeIntensity = 6.0;

	/**
	 * Specifies the minimum allowed contour length as a fraction of the input image's width.  Smaller numbers
	 * mean smaller objects are allowed.
	 */
	public double minContourImageWidthFraction = 0.05;

	/**
	 * Will the found polygons be in clockwise order?
	 */
	public boolean clockwise = true;

	/**
	 * Does it require that the found polygons be convex?
	 */
	public boolean convex = true;

	/**
	 * Configuration for sub-pixel refinement of line.  If null then this step is skipped.  Defaults to
	 * a line refinement algorithm.
	 *
	 * @see ConfigRefinePolygonLineToImage
	 * @see ConfigRefinePolygonCornersToImage
	 */
	public Configuration refine = new ConfigRefinePolygonLineToImage();

	/**
	 * Specifies the number of sides in the polygon and uses default settings for everything else
	 */
	public ConfigPolygonDetector( int minimumSides , int maximumSides) {
		this.minimumSides = minimumSides;
		this.maximumSides = maximumSides;
	}

	public ConfigPolygonDetector(boolean clockwise, int minimumSides , int maximumSides) {
		this.minimumSides = minimumSides;
		this.maximumSides = maximumSides;

		this.clockwise = clockwise;
	}

	@Override
	public void checkValidity() {

	}

	@Override
	public String toString() {
		return "ConfigPolygonDetector{" +
				"minimumSides=" + minimumSides +
				", maximumSides=" + maximumSides +
				", contour2Poly_splitFraction=" + contour2Poly_splitFraction +
				", contour2Poly_iterations=" + contour2Poly_iterations +
				", contour2Poly_minimumSplitFraction=" + contour2Poly_minimumSideFraction +
				", splitPenalty=" + splitPenalty +
				", minimumEdgeIntensity=" + minimumEdgeIntensity +
				", minContourImageWidthFraction=" + minContourImageWidthFraction +
				", clockwise=" + clockwise +
				", convex=" + convex +
				", refine=" + refine +
				'}';
	}
}
