/*
 * Copyright (c) 2011-2016, Peter Abeles. All Rights Reserved.
 *
 * This file is part of BoofCV (http://boofcv.org).
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package boofcv.alg.misc;

import boofcv.alg.InputSanityCheck;
import boofcv.struct.image.*;

import javax.annotation.Generated;

/**
 * Functions which perform basic arithmetic (e.g. addition, subtraction, multiplication, or division) on a pixel by pixel basis.
 *
 * <p>DO NOT MODIFY: Generated by boofcv.alg.misc.GeneratePixelMath.</p>
 *
 * @author Peter Abeles
 */
@Generated("boofcv.alg.misc.GeneratePixelMath")
public class PixelMath {

	/**
	 * Sets each pixel in the output image to be the absolute value of the input image.
	 * Both the input and output image can be the same instance.
	 * 
	 * @param input The input image. Not modified.
	 * @param output Where the absolute value image is written to. Modified.
	 */
	public static void abs( ImageSInt8 input , ImageSInt8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		abs(input.data,input.startIndex,input.stride,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Sets each pixel in the output image to be the absolute value of the input image.
	 * Both the input and output image can be the same instance.
	 * 
	 * @param input The input image. Not modified.
	 * @param output Where the absolute value image is written to. Modified.
	 */
	public static void abs( InterleavedS8 input , InterleavedS8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		abs(input.data,input.startIndex,input.stride,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void abs( byte[] input , int inputStart , int inputStride ,
							   byte[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (byte)Math.abs(input[indexSrc]);
			}
		}
	}

	/**
	 * Sets each pixel in the output image to be the absolute value of the input image.
	 * Both the input and output image can be the same instance.
	 * 
	 * @param input The input image. Not modified.
	 * @param output Where the absolute value image is written to. Modified.
	 */
	public static void abs( ImageSInt16 input , ImageSInt16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		abs(input.data,input.startIndex,input.stride,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Sets each pixel in the output image to be the absolute value of the input image.
	 * Both the input and output image can be the same instance.
	 * 
	 * @param input The input image. Not modified.
	 * @param output Where the absolute value image is written to. Modified.
	 */
	public static void abs( InterleavedS16 input , InterleavedS16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		abs(input.data,input.startIndex,input.stride,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void abs( short[] input , int inputStart , int inputStride ,
							   short[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (short)Math.abs(input[indexSrc]);
			}
		}
	}

	/**
	 * Sets each pixel in the output image to be the absolute value of the input image.
	 * Both the input and output image can be the same instance.
	 * 
	 * @param input The input image. Not modified.
	 * @param output Where the absolute value image is written to. Modified.
	 */
	public static void abs( ImageSInt32 input , ImageSInt32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		abs(input.data,input.startIndex,input.stride,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Sets each pixel in the output image to be the absolute value of the input image.
	 * Both the input and output image can be the same instance.
	 * 
	 * @param input The input image. Not modified.
	 * @param output Where the absolute value image is written to. Modified.
	 */
	public static void abs( InterleavedS32 input , InterleavedS32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		abs(input.data,input.startIndex,input.stride,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void abs( int[] input , int inputStart , int inputStride ,
							   int[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = Math.abs(input[indexSrc]);
			}
		}
	}

	/**
	 * Sets each pixel in the output image to be the absolute value of the input image.
	 * Both the input and output image can be the same instance.
	 * 
	 * @param input The input image. Not modified.
	 * @param output Where the absolute value image is written to. Modified.
	 */
	public static void abs( ImageSInt64 input , ImageSInt64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		abs(input.data,input.startIndex,input.stride,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Sets each pixel in the output image to be the absolute value of the input image.
	 * Both the input and output image can be the same instance.
	 * 
	 * @param input The input image. Not modified.
	 * @param output Where the absolute value image is written to. Modified.
	 */
	public static void abs( InterleavedS64 input , InterleavedS64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		abs(input.data,input.startIndex,input.stride,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void abs( long[] input , int inputStart , int inputStride ,
							   long[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = Math.abs(input[indexSrc]);
			}
		}
	}

	/**
	 * Sets each pixel in the output image to be the absolute value of the input image.
	 * Both the input and output image can be the same instance.
	 * 
	 * @param input The input image. Not modified.
	 * @param output Where the absolute value image is written to. Modified.
	 */
	public static void abs( ImageFloat32 input , ImageFloat32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		abs(input.data,input.startIndex,input.stride,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Sets each pixel in the output image to be the absolute value of the input image.
	 * Both the input and output image can be the same instance.
	 * 
	 * @param input The input image. Not modified.
	 * @param output Where the absolute value image is written to. Modified.
	 */
	public static void abs( InterleavedF32 input , InterleavedF32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		abs(input.data,input.startIndex,input.stride,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void abs( float[] input , int inputStart , int inputStride ,
							   float[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = Math.abs(input[indexSrc]);
			}
		}
	}

	/**
	 * Sets each pixel in the output image to be the absolute value of the input image.
	 * Both the input and output image can be the same instance.
	 * 
	 * @param input The input image. Not modified.
	 * @param output Where the absolute value image is written to. Modified.
	 */
	public static void abs( ImageFloat64 input , ImageFloat64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		abs(input.data,input.startIndex,input.stride,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Sets each pixel in the output image to be the absolute value of the input image.
	 * Both the input and output image can be the same instance.
	 * 
	 * @param input The input image. Not modified.
	 * @param output Where the absolute value image is written to. Modified.
	 */
	public static void abs( InterleavedF64 input , InterleavedF64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		abs(input.data,input.startIndex,input.stride,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void abs( double[] input , int inputStart , int inputStride ,
							   double[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = Math.abs(input[indexSrc]);
			}
		}
	}

	/**
	 * Changes the sign of every pixel in the image: output[x,y] = -input[x,y]
	 *
	 * @param input The input image. Not modified.
	 * @param output Where the inverted image is written to. Modified.
	 */
	public static void invert( ImageSInt8 input , ImageSInt8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		invert(input.data,input.startIndex,input.stride,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Changes the sign of every pixel in the image: output[x,y] = -input[x,y]
	 *
	 * @param input The input image. Not modified.
	 * @param output Where the inverted image is written to. Modified.
	 */
	public static void invert( InterleavedS8 input , InterleavedS8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		invert(input.data,input.startIndex,input.stride,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void invert( byte[] input , int inputStart , int inputStride ,
							   byte[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (byte)-input[indexSrc];
			}
		}
	}

	/**
	 * Changes the sign of every pixel in the image: output[x,y] = -input[x,y]
	 *
	 * @param input The input image. Not modified.
	 * @param output Where the inverted image is written to. Modified.
	 */
	public static void invert( ImageSInt16 input , ImageSInt16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		invert(input.data,input.startIndex,input.stride,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Changes the sign of every pixel in the image: output[x,y] = -input[x,y]
	 *
	 * @param input The input image. Not modified.
	 * @param output Where the inverted image is written to. Modified.
	 */
	public static void invert( InterleavedS16 input , InterleavedS16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		invert(input.data,input.startIndex,input.stride,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void invert( short[] input , int inputStart , int inputStride ,
							   short[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (short)-input[indexSrc];
			}
		}
	}

	/**
	 * Changes the sign of every pixel in the image: output[x,y] = -input[x,y]
	 *
	 * @param input The input image. Not modified.
	 * @param output Where the inverted image is written to. Modified.
	 */
	public static void invert( ImageSInt32 input , ImageSInt32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		invert(input.data,input.startIndex,input.stride,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Changes the sign of every pixel in the image: output[x,y] = -input[x,y]
	 *
	 * @param input The input image. Not modified.
	 * @param output Where the inverted image is written to. Modified.
	 */
	public static void invert( InterleavedS32 input , InterleavedS32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		invert(input.data,input.startIndex,input.stride,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void invert( int[] input , int inputStart , int inputStride ,
							   int[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = -input[indexSrc];
			}
		}
	}

	/**
	 * Changes the sign of every pixel in the image: output[x,y] = -input[x,y]
	 *
	 * @param input The input image. Not modified.
	 * @param output Where the inverted image is written to. Modified.
	 */
	public static void invert( ImageSInt64 input , ImageSInt64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		invert(input.data,input.startIndex,input.stride,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Changes the sign of every pixel in the image: output[x,y] = -input[x,y]
	 *
	 * @param input The input image. Not modified.
	 * @param output Where the inverted image is written to. Modified.
	 */
	public static void invert( InterleavedS64 input , InterleavedS64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		invert(input.data,input.startIndex,input.stride,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void invert( long[] input , int inputStart , int inputStride ,
							   long[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = -input[indexSrc];
			}
		}
	}

	/**
	 * Changes the sign of every pixel in the image: output[x,y] = -input[x,y]
	 *
	 * @param input The input image. Not modified.
	 * @param output Where the inverted image is written to. Modified.
	 */
	public static void invert( ImageFloat32 input , ImageFloat32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		invert(input.data,input.startIndex,input.stride,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Changes the sign of every pixel in the image: output[x,y] = -input[x,y]
	 *
	 * @param input The input image. Not modified.
	 * @param output Where the inverted image is written to. Modified.
	 */
	public static void invert( InterleavedF32 input , InterleavedF32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		invert(input.data,input.startIndex,input.stride,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void invert( float[] input , int inputStart , int inputStride ,
							   float[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = -input[indexSrc];
			}
		}
	}

	/**
	 * Changes the sign of every pixel in the image: output[x,y] = -input[x,y]
	 *
	 * @param input The input image. Not modified.
	 * @param output Where the inverted image is written to. Modified.
	 */
	public static void invert( ImageFloat64 input , ImageFloat64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		invert(input.data,input.startIndex,input.stride,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Changes the sign of every pixel in the image: output[x,y] = -input[x,y]
	 *
	 * @param input The input image. Not modified.
	 * @param output Where the inverted image is written to. Modified.
	 */
	public static void invert( InterleavedF64 input , InterleavedF64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		invert(input.data,input.startIndex,input.stride,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void invert( double[] input , int inputStart , int inputStride ,
							   double[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = -input[indexSrc];
			}
		}
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( ImageUInt8 input , double value , ImageUInt8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		multiplyU_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( InterleavedU8 input , double value , InterleavedU8 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		multiplyU_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void multiplyU_A( byte[] input , int inputStart , int inputStride , 
							   double value ,
							   byte[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (byte)Math.round((input[indexSrc] & 0xFF) * value);
			}
		}
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( ImageSInt8 input , double value , ImageSInt8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		multiply_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( InterleavedS8 input , double value , InterleavedS8 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		multiply_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void multiply_A( byte[] input , int inputStart , int inputStride , 
							   double value ,
							   byte[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (byte)Math.round((input[indexSrc] ) * value);
			}
		}
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( ImageUInt16 input , double value , ImageUInt16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		multiplyU_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( InterleavedU16 input , double value , InterleavedU16 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		multiplyU_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void multiplyU_A( short[] input , int inputStart , int inputStride , 
							   double value ,
							   short[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (short)Math.round((input[indexSrc] & 0xFFFF) * value);
			}
		}
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( ImageSInt16 input , double value , ImageSInt16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		multiply_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( InterleavedS16 input , double value , InterleavedS16 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		multiply_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void multiply_A( short[] input , int inputStart , int inputStride , 
							   double value ,
							   short[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (short)Math.round((input[indexSrc] ) * value);
			}
		}
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( ImageSInt32 input , double value , ImageSInt32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		multiply_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( InterleavedS32 input , double value , InterleavedS32 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		multiply_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void multiply_A( int[] input , int inputStart , int inputStride , 
							   double value ,
							   int[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (int)Math.round((input[indexSrc] ) * value);
			}
		}
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( ImageSInt64 input , double value , ImageSInt64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		multiply_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( InterleavedS64 input , double value , InterleavedS64 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		multiply_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void multiply_A( long[] input , int inputStart , int inputStride , 
							   double value ,
							   long[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (long)Math.round((input[indexSrc] ) * value);
			}
		}
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( ImageFloat32 input , float value , ImageFloat32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		multiply_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( InterleavedF32 input , float value , InterleavedF32 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		multiply_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void multiply_A( float[] input , int inputStart , int inputStride , 
							   float value ,
							   float[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = ((input[indexSrc] ) * value);
			}
		}
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( ImageFloat64 input , double value , ImageFloat64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		multiply_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( InterleavedF64 input , double value , InterleavedF64 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		multiply_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void multiply_A( double[] input , int inputStart , int inputStride , 
							   double value ,
							   double[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = ((input[indexSrc] ) * value);
			}
		}
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( ImageUInt8 input , double value , int lower , int upper , ImageUInt8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		multiplyU_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( InterleavedU8 input , double value , int lower , int upper , InterleavedU8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		multiplyU_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void multiplyU_A( byte[] input , int inputStart , int inputStride , 
							   double value , int lower , int upper ,
							   byte[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				int val = (int)Math.round((input[indexSrc] & 0xFF) * value);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = (byte)val;
			}
		}
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( ImageSInt8 input , double value , int lower , int upper , ImageSInt8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		multiply_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( InterleavedS8 input , double value , int lower , int upper , InterleavedS8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		multiply_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void multiply_A( byte[] input , int inputStart , int inputStride , 
							   double value , int lower , int upper ,
							   byte[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				int val = (int)Math.round((input[indexSrc] ) * value);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = (byte)val;
			}
		}
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( ImageUInt16 input , double value , int lower , int upper , ImageUInt16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		multiplyU_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( InterleavedU16 input , double value , int lower , int upper , InterleavedU16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		multiplyU_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void multiplyU_A( short[] input , int inputStart , int inputStride , 
							   double value , int lower , int upper ,
							   short[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				int val = (int)Math.round((input[indexSrc] & 0xFFFF) * value);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = (short)val;
			}
		}
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( ImageSInt16 input , double value , int lower , int upper , ImageSInt16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		multiply_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( InterleavedS16 input , double value , int lower , int upper , InterleavedS16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		multiply_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void multiply_A( short[] input , int inputStart , int inputStride , 
							   double value , int lower , int upper ,
							   short[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				int val = (int)Math.round((input[indexSrc] ) * value);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = (short)val;
			}
		}
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( ImageSInt32 input , double value , int lower , int upper , ImageSInt32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		multiply_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( InterleavedS32 input , double value , int lower , int upper , InterleavedS32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		multiply_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void multiply_A( int[] input , int inputStart , int inputStride , 
							   double value , int lower , int upper ,
							   int[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				int val = (int)Math.round((input[indexSrc] ) * value);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = val;
			}
		}
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( ImageSInt64 input , double value , long lower , long upper , ImageSInt64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		multiply_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( InterleavedS64 input , double value , long lower , long upper , InterleavedS64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		multiply_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void multiply_A( long[] input , int inputStart , int inputStride , 
							   double value , long lower , long upper ,
							   long[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				long val = (long)Math.round((input[indexSrc] ) * value);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = val;
			}
		}
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( ImageFloat32 input , float value , float lower , float upper , ImageFloat32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		multiply_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( InterleavedF32 input , float value , float lower , float upper , InterleavedF32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		multiply_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void multiply_A( float[] input , int inputStart , int inputStride , 
							   float value , float lower , float upper ,
							   float[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				float val = ((input[indexSrc] ) * value);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = val;
			}
		}
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( ImageFloat64 input , double value , double lower , double upper , ImageFloat64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		multiply_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Multiply each element by a scalar value. Both input and output images can
	 * be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What each element is multiplied by.
	 * @param output The output image. Modified.
	 */
	public static void multiply( InterleavedF64 input , double value , double lower , double upper , InterleavedF64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		multiply_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void multiply_A( double[] input , int inputStart , int inputStride , 
							   double value , double lower , double upper ,
							   double[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				double val = ((input[indexSrc] ) * value);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = val;
			}
		}
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( ImageUInt8 input , double denominator , ImageUInt8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		divideU_A(input.data,input.startIndex,input.stride,denominator , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( InterleavedU8 input , double denominator , InterleavedU8 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		divideU_A(input.data,input.startIndex,input.stride,denominator , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void divideU_A( byte[] input , int inputStart , int inputStride , 
							   double denominator ,
							   byte[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (byte)Math.round((input[indexSrc] & 0xFF) / denominator);
			}
		}
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( ImageSInt8 input , double denominator , ImageSInt8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		divide_A(input.data,input.startIndex,input.stride,denominator , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( InterleavedS8 input , double denominator , InterleavedS8 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		divide_A(input.data,input.startIndex,input.stride,denominator , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void divide_A( byte[] input , int inputStart , int inputStride , 
							   double denominator ,
							   byte[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (byte)Math.round((input[indexSrc] ) / denominator);
			}
		}
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( ImageUInt16 input , double denominator , ImageUInt16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		divideU_A(input.data,input.startIndex,input.stride,denominator , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( InterleavedU16 input , double denominator , InterleavedU16 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		divideU_A(input.data,input.startIndex,input.stride,denominator , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void divideU_A( short[] input , int inputStart , int inputStride , 
							   double denominator ,
							   short[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (short)Math.round((input[indexSrc] & 0xFFFF) / denominator);
			}
		}
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( ImageSInt16 input , double denominator , ImageSInt16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		divide_A(input.data,input.startIndex,input.stride,denominator , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( InterleavedS16 input , double denominator , InterleavedS16 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		divide_A(input.data,input.startIndex,input.stride,denominator , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void divide_A( short[] input , int inputStart , int inputStride , 
							   double denominator ,
							   short[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (short)Math.round((input[indexSrc] ) / denominator);
			}
		}
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( ImageSInt32 input , double denominator , ImageSInt32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		divide_A(input.data,input.startIndex,input.stride,denominator , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( InterleavedS32 input , double denominator , InterleavedS32 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		divide_A(input.data,input.startIndex,input.stride,denominator , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void divide_A( int[] input , int inputStart , int inputStride , 
							   double denominator ,
							   int[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (int)Math.round((input[indexSrc] ) / denominator);
			}
		}
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( ImageSInt64 input , double denominator , ImageSInt64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		divide_A(input.data,input.startIndex,input.stride,denominator , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( InterleavedS64 input , double denominator , InterleavedS64 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		divide_A(input.data,input.startIndex,input.stride,denominator , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void divide_A( long[] input , int inputStart , int inputStride , 
							   double denominator ,
							   long[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (long)Math.round((input[indexSrc] ) / denominator);
			}
		}
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( ImageFloat32 input , float denominator , ImageFloat32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		divide_A(input.data,input.startIndex,input.stride,denominator , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( InterleavedF32 input , float denominator , InterleavedF32 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		divide_A(input.data,input.startIndex,input.stride,denominator , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void divide_A( float[] input , int inputStart , int inputStride , 
							   float denominator ,
							   float[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = ((input[indexSrc] ) / denominator);
			}
		}
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( ImageFloat64 input , double denominator , ImageFloat64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		divide_A(input.data,input.startIndex,input.stride,denominator , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( InterleavedF64 input , double denominator , InterleavedF64 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		divide_A(input.data,input.startIndex,input.stride,denominator , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void divide_A( double[] input , int inputStart , int inputStride , 
							   double denominator ,
							   double[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = ((input[indexSrc] ) / denominator);
			}
		}
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( ImageUInt8 input , double denominator , int lower , int upper , ImageUInt8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		divideU_A(input.data,input.startIndex,input.stride,denominator, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( InterleavedU8 input , double denominator , int lower , int upper , InterleavedU8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		divideU_A(input.data,input.startIndex,input.stride,denominator, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void divideU_A( byte[] input , int inputStart , int inputStride , 
							   double denominator , int lower , int upper ,
							   byte[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				int val = (int)Math.round((input[indexSrc] & 0xFF) / denominator);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = (byte)val;
			}
		}
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( ImageSInt8 input , double denominator , int lower , int upper , ImageSInt8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		divide_A(input.data,input.startIndex,input.stride,denominator, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( InterleavedS8 input , double denominator , int lower , int upper , InterleavedS8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		divide_A(input.data,input.startIndex,input.stride,denominator, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void divide_A( byte[] input , int inputStart , int inputStride , 
							   double denominator , int lower , int upper ,
							   byte[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				int val = (int)Math.round((input[indexSrc] ) / denominator);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = (byte)val;
			}
		}
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( ImageUInt16 input , double denominator , int lower , int upper , ImageUInt16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		divideU_A(input.data,input.startIndex,input.stride,denominator, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( InterleavedU16 input , double denominator , int lower , int upper , InterleavedU16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		divideU_A(input.data,input.startIndex,input.stride,denominator, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void divideU_A( short[] input , int inputStart , int inputStride , 
							   double denominator , int lower , int upper ,
							   short[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				int val = (int)Math.round((input[indexSrc] & 0xFFFF) / denominator);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = (short)val;
			}
		}
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( ImageSInt16 input , double denominator , int lower , int upper , ImageSInt16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		divide_A(input.data,input.startIndex,input.stride,denominator, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( InterleavedS16 input , double denominator , int lower , int upper , InterleavedS16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		divide_A(input.data,input.startIndex,input.stride,denominator, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void divide_A( short[] input , int inputStart , int inputStride , 
							   double denominator , int lower , int upper ,
							   short[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				int val = (int)Math.round((input[indexSrc] ) / denominator);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = (short)val;
			}
		}
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( ImageSInt32 input , double denominator , int lower , int upper , ImageSInt32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		divide_A(input.data,input.startIndex,input.stride,denominator, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( InterleavedS32 input , double denominator , int lower , int upper , InterleavedS32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		divide_A(input.data,input.startIndex,input.stride,denominator, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void divide_A( int[] input , int inputStart , int inputStride , 
							   double denominator , int lower , int upper ,
							   int[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				int val = (int)Math.round((input[indexSrc] ) / denominator);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = val;
			}
		}
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( ImageSInt64 input , double denominator , long lower , long upper , ImageSInt64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		divide_A(input.data,input.startIndex,input.stride,denominator, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( InterleavedS64 input , double denominator , long lower , long upper , InterleavedS64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		divide_A(input.data,input.startIndex,input.stride,denominator, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void divide_A( long[] input , int inputStart , int inputStride , 
							   double denominator , long lower , long upper ,
							   long[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				long val = (long)Math.round((input[indexSrc] ) / denominator);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = val;
			}
		}
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( ImageFloat32 input , float denominator , float lower , float upper , ImageFloat32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		divide_A(input.data,input.startIndex,input.stride,denominator, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( InterleavedF32 input , float denominator , float lower , float upper , InterleavedF32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		divide_A(input.data,input.startIndex,input.stride,denominator, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void divide_A( float[] input , int inputStart , int inputStride , 
							   float denominator , float lower , float upper ,
							   float[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				float val = ((input[indexSrc] ) / denominator);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = val;
			}
		}
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( ImageFloat64 input , double denominator , double lower , double upper , ImageFloat64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		divide_A(input.data,input.startIndex,input.stride,denominator, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Divide each element by a scalar value. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param denominator What each element is divided by.
	 * @param output The output image. Modified.
	 */
	public static void divide( InterleavedF64 input , double denominator , double lower , double upper , InterleavedF64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		divide_A(input.data,input.startIndex,input.stride,denominator, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void divide_A( double[] input , int inputStart , int inputStride , 
							   double denominator , double lower , double upper ,
							   double[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				double val = ((input[indexSrc] ) / denominator);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = val;
			}
		}
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( ImageUInt8 input , int value , ImageUInt8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		plusU_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( InterleavedU8 input , int value , InterleavedU8 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		plusU_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void plusU_A( byte[] input , int inputStart , int inputStride , 
							   int value ,
							   byte[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (byte)((input[indexSrc] & 0xFF) + value);
			}
		}
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( ImageSInt8 input , int value , ImageSInt8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		plus_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( InterleavedS8 input , int value , InterleavedS8 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		plus_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void plus_A( byte[] input , int inputStart , int inputStride , 
							   int value ,
							   byte[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (byte)((input[indexSrc] ) + value);
			}
		}
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( ImageUInt16 input , int value , ImageUInt16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		plusU_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( InterleavedU16 input , int value , InterleavedU16 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		plusU_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void plusU_A( short[] input , int inputStart , int inputStride , 
							   int value ,
							   short[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (short)((input[indexSrc] & 0xFFFF) + value);
			}
		}
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( ImageSInt16 input , int value , ImageSInt16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		plus_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( InterleavedS16 input , int value , InterleavedS16 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		plus_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void plus_A( short[] input , int inputStart , int inputStride , 
							   int value ,
							   short[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (short)((input[indexSrc] ) + value);
			}
		}
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( ImageSInt32 input , int value , ImageSInt32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		plus_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( InterleavedS32 input , int value , InterleavedS32 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		plus_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void plus_A( int[] input , int inputStart , int inputStride , 
							   int value ,
							   int[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = ((input[indexSrc] ) + value);
			}
		}
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( ImageSInt64 input , long value , ImageSInt64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		plus_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( InterleavedS64 input , long value , InterleavedS64 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		plus_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void plus_A( long[] input , int inputStart , int inputStride , 
							   long value ,
							   long[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = ((input[indexSrc] ) + value);
			}
		}
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( ImageFloat32 input , float value , ImageFloat32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		plus_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( InterleavedF32 input , float value , InterleavedF32 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		plus_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void plus_A( float[] input , int inputStart , int inputStride , 
							   float value ,
							   float[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = ((input[indexSrc] ) + value);
			}
		}
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( ImageFloat64 input , double value , ImageFloat64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		plus_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( InterleavedF64 input , double value , InterleavedF64 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		plus_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void plus_A( double[] input , int inputStart , int inputStride , 
							   double value ,
							   double[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = ((input[indexSrc] ) + value);
			}
		}
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( ImageUInt8 input , int value , int lower , int upper , ImageUInt8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		plusU_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( InterleavedU8 input , int value , int lower , int upper , InterleavedU8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		plusU_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void plusU_A( byte[] input , int inputStart , int inputStride , 
							   int value , int lower , int upper ,
							   byte[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				int val = ((input[indexSrc] & 0xFF) + value);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = (byte)val;
			}
		}
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( ImageSInt8 input , int value , int lower , int upper , ImageSInt8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		plus_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( InterleavedS8 input , int value , int lower , int upper , InterleavedS8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		plus_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void plus_A( byte[] input , int inputStart , int inputStride , 
							   int value , int lower , int upper ,
							   byte[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				int val = ((input[indexSrc] ) + value);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = (byte)val;
			}
		}
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( ImageUInt16 input , int value , int lower , int upper , ImageUInt16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		plusU_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( InterleavedU16 input , int value , int lower , int upper , InterleavedU16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		plusU_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void plusU_A( short[] input , int inputStart , int inputStride , 
							   int value , int lower , int upper ,
							   short[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				int val = ((input[indexSrc] & 0xFFFF) + value);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = (short)val;
			}
		}
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( ImageSInt16 input , int value , int lower , int upper , ImageSInt16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		plus_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( InterleavedS16 input , int value , int lower , int upper , InterleavedS16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		plus_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void plus_A( short[] input , int inputStart , int inputStride , 
							   int value , int lower , int upper ,
							   short[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				int val = ((input[indexSrc] ) + value);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = (short)val;
			}
		}
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( ImageSInt32 input , int value , int lower , int upper , ImageSInt32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		plus_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( InterleavedS32 input , int value , int lower , int upper , InterleavedS32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		plus_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void plus_A( int[] input , int inputStart , int inputStride , 
							   int value , int lower , int upper ,
							   int[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				int val = ((input[indexSrc] ) + value);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = val;
			}
		}
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( ImageSInt64 input , long value , long lower , long upper , ImageSInt64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		plus_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( InterleavedS64 input , long value , long lower , long upper , InterleavedS64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		plus_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void plus_A( long[] input , int inputStart , int inputStride , 
							   long value , long lower , long upper ,
							   long[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				long val = ((input[indexSrc] ) + value);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = val;
			}
		}
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( ImageFloat32 input , float value , float lower , float upper , ImageFloat32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		plus_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( InterleavedF32 input , float value , float lower , float upper , InterleavedF32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		plus_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void plus_A( float[] input , int inputStart , int inputStride , 
							   float value , float lower , float upper ,
							   float[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				float val = ((input[indexSrc] ) + value);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = val;
			}
		}
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( ImageFloat64 input , double value , double lower , double upper , ImageFloat64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		plus_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Adds a scalar value to each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is added to each element.
	 * @param output The output image. Modified.
	 */
	public static void plus( InterleavedF64 input , double value , double lower , double upper , InterleavedF64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		plus_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void plus_A( double[] input , int inputStart , int inputStride , 
							   double value , double lower , double upper ,
							   double[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				double val = ((input[indexSrc] ) + value);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = val;
			}
		}
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( ImageUInt8 input , int value , ImageUInt8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minusU_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( InterleavedU8 input , int value , InterleavedU8 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		minusU_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minusU_A( byte[] input , int inputStart , int inputStride , 
							   int value ,
							   byte[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (byte)((input[indexSrc] & 0xFF) - value);
			}
		}
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( ImageSInt8 input , int value , ImageSInt8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minus_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( InterleavedS8 input , int value , InterleavedS8 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		minus_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minus_A( byte[] input , int inputStart , int inputStride , 
							   int value ,
							   byte[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (byte)((input[indexSrc] ) - value);
			}
		}
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( ImageUInt16 input , int value , ImageUInt16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minusU_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( InterleavedU16 input , int value , InterleavedU16 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		minusU_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minusU_A( short[] input , int inputStart , int inputStride , 
							   int value ,
							   short[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (short)((input[indexSrc] & 0xFFFF) - value);
			}
		}
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( ImageSInt16 input , int value , ImageSInt16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minus_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( InterleavedS16 input , int value , InterleavedS16 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		minus_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minus_A( short[] input , int inputStart , int inputStride , 
							   int value ,
							   short[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (short)((input[indexSrc] ) - value);
			}
		}
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( ImageSInt32 input , int value , ImageSInt32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minus_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( InterleavedS32 input , int value , InterleavedS32 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		minus_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minus_A( int[] input , int inputStart , int inputStride , 
							   int value ,
							   int[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = ((input[indexSrc] ) - value);
			}
		}
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( ImageSInt64 input , long value , ImageSInt64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minus_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( InterleavedS64 input , long value , InterleavedS64 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		minus_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minus_A( long[] input , int inputStart , int inputStride , 
							   long value ,
							   long[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = ((input[indexSrc] ) - value);
			}
		}
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( ImageFloat32 input , float value , ImageFloat32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minus_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( InterleavedF32 input , float value , InterleavedF32 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		minus_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minus_A( float[] input , int inputStart , int inputStride , 
							   float value ,
							   float[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = ((input[indexSrc] ) - value);
			}
		}
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( ImageFloat64 input , double value , ImageFloat64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minus_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( InterleavedF64 input , double value , InterleavedF64 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		minus_A(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minus_A( double[] input , int inputStart , int inputStride , 
							   double value ,
							   double[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = ((input[indexSrc] ) - value);
			}
		}
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( ImageUInt8 input , int value , int lower , int upper , ImageUInt8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minusU_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( InterleavedU8 input , int value , int lower , int upper , InterleavedU8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		minusU_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minusU_A( byte[] input , int inputStart , int inputStride , 
							   int value , int lower , int upper ,
							   byte[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				int val = ((input[indexSrc] & 0xFF) - value);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = (byte)val;
			}
		}
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( ImageSInt8 input , int value , int lower , int upper , ImageSInt8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minus_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( InterleavedS8 input , int value , int lower , int upper , InterleavedS8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		minus_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minus_A( byte[] input , int inputStart , int inputStride , 
							   int value , int lower , int upper ,
							   byte[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				int val = ((input[indexSrc] ) - value);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = (byte)val;
			}
		}
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( ImageUInt16 input , int value , int lower , int upper , ImageUInt16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minusU_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( InterleavedU16 input , int value , int lower , int upper , InterleavedU16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		minusU_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minusU_A( short[] input , int inputStart , int inputStride , 
							   int value , int lower , int upper ,
							   short[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				int val = ((input[indexSrc] & 0xFFFF) - value);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = (short)val;
			}
		}
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( ImageSInt16 input , int value , int lower , int upper , ImageSInt16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minus_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( InterleavedS16 input , int value , int lower , int upper , InterleavedS16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		minus_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minus_A( short[] input , int inputStart , int inputStride , 
							   int value , int lower , int upper ,
							   short[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				int val = ((input[indexSrc] ) - value);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = (short)val;
			}
		}
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( ImageSInt32 input , int value , int lower , int upper , ImageSInt32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minus_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( InterleavedS32 input , int value , int lower , int upper , InterleavedS32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		minus_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minus_A( int[] input , int inputStart , int inputStride , 
							   int value , int lower , int upper ,
							   int[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				int val = ((input[indexSrc] ) - value);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = val;
			}
		}
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( ImageSInt64 input , long value , long lower , long upper , ImageSInt64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minus_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( InterleavedS64 input , long value , long lower , long upper , InterleavedS64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		minus_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minus_A( long[] input , int inputStart , int inputStride , 
							   long value , long lower , long upper ,
							   long[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				long val = ((input[indexSrc] ) - value);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = val;
			}
		}
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( ImageFloat32 input , float value , float lower , float upper , ImageFloat32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minus_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( InterleavedF32 input , float value , float lower , float upper , InterleavedF32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		minus_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minus_A( float[] input , int inputStart , int inputStride , 
							   float value , float lower , float upper ,
							   float[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				float val = ((input[indexSrc] ) - value);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = val;
			}
		}
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( ImageFloat64 input , double value , double lower , double upper , ImageFloat64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minus_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts a scalar value from each element. Both input and output images can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param value What is subtracted from each element.
	 * @param output The output image. Modified.
	 */
	public static void minus( InterleavedF64 input , double value , double lower , double upper , InterleavedF64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		minus_A(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minus_A( double[] input , int inputStart , int inputStride , 
							   double value , double lower , double upper ,
							   double[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				double val = ((input[indexSrc] ) - value);
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = val;
			}
		}
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( int value , ImageUInt8 input , ImageUInt8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minusU_B(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( int value , InterleavedU8 input , InterleavedU8 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		minusU_B(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minusU_B( byte[] input , int inputStart , int inputStride , 
							   int value ,
							   byte[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (byte)(value - (input[indexSrc] & 0xFF));
			}
		}
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( int value , ImageSInt8 input , ImageSInt8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minus_B(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( int value , InterleavedS8 input , InterleavedS8 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		minus_B(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minus_B( byte[] input , int inputStart , int inputStride , 
							   int value ,
							   byte[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (byte)(value - (input[indexSrc] ));
			}
		}
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( int value , ImageUInt16 input , ImageUInt16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minusU_B(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( int value , InterleavedU16 input , InterleavedU16 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		minusU_B(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minusU_B( short[] input , int inputStart , int inputStride , 
							   int value ,
							   short[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (short)(value - (input[indexSrc] & 0xFFFF));
			}
		}
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( int value , ImageSInt16 input , ImageSInt16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minus_B(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( int value , InterleavedS16 input , InterleavedS16 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		minus_B(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minus_B( short[] input , int inputStart , int inputStride , 
							   int value ,
							   short[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (short)(value - (input[indexSrc] ));
			}
		}
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( int value , ImageSInt32 input , ImageSInt32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minus_B(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( int value , InterleavedS32 input , InterleavedS32 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		minus_B(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minus_B( int[] input , int inputStart , int inputStride , 
							   int value ,
							   int[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (value - (input[indexSrc] ));
			}
		}
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( long value , ImageSInt64 input , ImageSInt64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minus_B(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( long value , InterleavedS64 input , InterleavedS64 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		minus_B(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minus_B( long[] input , int inputStart , int inputStride , 
							   long value ,
							   long[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (value - (input[indexSrc] ));
			}
		}
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( float value , ImageFloat32 input , ImageFloat32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minus_B(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( float value , InterleavedF32 input , InterleavedF32 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		minus_B(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minus_B( float[] input , int inputStart , int inputStride , 
							   float value ,
							   float[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (value - (input[indexSrc] ));
			}
		}
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( double value , ImageFloat64 input , ImageFloat64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minus_B(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( double value , InterleavedF64 input , InterleavedF64 output ) {

		InputSanityCheck.checkSameShapeB(input,output);

		int columns = input.width*input.numBands;
		minus_B(input.data,input.startIndex,input.stride,value , 
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minus_B( double[] input , int inputStart , int inputStride , 
							   double value ,
							   double[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output[indexDst] = (value - (input[indexSrc] ));
			}
		}
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( int value , ImageUInt8 input , int lower , int upper , ImageUInt8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minusU_B(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( int value , InterleavedU8 input , int lower , int upper , InterleavedU8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		minusU_B(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minusU_B( byte[] input , int inputStart , int inputStride , 
							   int value , int lower , int upper ,
							   byte[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				int val = (value - (input[indexSrc] & 0xFF));
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = (byte)val;
			}
		}
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( int value , ImageSInt8 input , int lower , int upper , ImageSInt8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minus_B(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( int value , InterleavedS8 input , int lower , int upper , InterleavedS8 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		minus_B(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minus_B( byte[] input , int inputStart , int inputStride , 
							   int value , int lower , int upper ,
							   byte[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				int val = (value - (input[indexSrc] ));
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = (byte)val;
			}
		}
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( int value , ImageUInt16 input , int lower , int upper , ImageUInt16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minusU_B(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( int value , InterleavedU16 input , int lower , int upper , InterleavedU16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		minusU_B(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minusU_B( short[] input , int inputStart , int inputStride , 
							   int value , int lower , int upper ,
							   short[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				int val = (value - (input[indexSrc] & 0xFFFF));
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = (short)val;
			}
		}
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( int value , ImageSInt16 input , int lower , int upper , ImageSInt16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minus_B(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( int value , InterleavedS16 input , int lower , int upper , InterleavedS16 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		minus_B(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minus_B( short[] input , int inputStart , int inputStride , 
							   int value , int lower , int upper ,
							   short[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				int val = (value - (input[indexSrc] ));
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = (short)val;
			}
		}
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( int value , ImageSInt32 input , int lower , int upper , ImageSInt32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minus_B(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( int value , InterleavedS32 input , int lower , int upper , InterleavedS32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		minus_B(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minus_B( int[] input , int inputStart , int inputStride , 
							   int value , int lower , int upper ,
							   int[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				int val = (value - (input[indexSrc] ));
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = val;
			}
		}
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( long value , ImageSInt64 input , long lower , long upper , ImageSInt64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minus_B(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( long value , InterleavedS64 input , long lower , long upper , InterleavedS64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		minus_B(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minus_B( long[] input , int inputStart , int inputStride , 
							   long value , long lower , long upper ,
							   long[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				long val = (value - (input[indexSrc] ));
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = val;
			}
		}
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( float value , ImageFloat32 input , float lower , float upper , ImageFloat32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minus_B(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( float value , InterleavedF32 input , float lower , float upper , InterleavedF32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		minus_B(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minus_B( float[] input , int inputStart , int inputStride , 
							   float value , float lower , float upper ,
							   float[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				float val = (value - (input[indexSrc] ));
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = val;
			}
		}
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( double value , ImageFloat64 input , double lower , double upper , ImageFloat64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width;
		minus_B(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	/**
	 * Subtracts each element's value from a scalar. Both input and output images can be the same instance.
	 *
	 * @param value Scalar value
	 * @param input The input image. Not modified.
	 * @param output The output image. Modified.
	 */
	public static void minus( double value , InterleavedF64 input , double lower , double upper , InterleavedF64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		int columns = input.width*input.numBands;
		minus_B(input.data,input.startIndex,input.stride,value, lower, upper ,
				output.data,output.startIndex,output.stride,
				input.height,columns);
	}

	private static void minus_B( double[] input , int inputStart , int inputStride , 
							   double value , double lower , double upper ,
							   double[] output , int outputStart , int outputStride ,
							   int rows , int cols )
	{
		for( int y = 0; y < rows; y++ ) {
			int indexSrc = inputStart + y*inputStride;
			int indexDst = outputStart + y*outputStride;
			int end = indexSrc + cols;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				double val = (value - (input[indexSrc] ));
				if( val < lower ) val = lower;
				if( val > upper ) val = upper;
				output[indexDst] = val;
			}
		}
	}

	/**
	 * Bounds image pixels to be between these two values
	 * 
	 * @param img Image
	 * @param min minimum value.
	 * @param max maximum value.
	 */
	public static void boundImage( ImageUInt8 img , int min , int max ) {
		final int h = img.getHeight();
		final int w = img.getWidth();

		byte[] data = img.data;

		for (int y = 0; y < h; y++) {
			int index = img.getStartIndex() + y * img.getStride();
			int indexEnd = index+w;
			// for(int x = 0; x < w; x++ ) {
			for (; index < indexEnd; index++) {
				int value = data[index]& 0xFF;
				if( value < min )
					data[index] = (byte)min;
				else if( value > max )
					data[index] = (byte)max;
			}
		}
	}

	/**
	 * <p>
	 * Computes the absolute value of the difference between each pixel in the two images.<br>
	 * d(x,y) = |img1(x,y) - img2(x,y)|
	 * </p>
	 * @param imgA Input image. Not modified.
	 * @param imgB Input image. Not modified.
	 * @param diff Absolute value of difference image. Modified.
	 */
	public static void diffAbs( ImageUInt8 imgA , ImageUInt8 imgB , ImageUInt8 diff ) {
		InputSanityCheck.checkSameShape(imgA,imgB,diff);
		
		final int h = imgA.getHeight();
		final int w = imgA.getWidth();

		for (int y = 0; y < h; y++) {
			int indexA = imgA.getStartIndex() + y * imgA.getStride();
			int indexB = imgB.getStartIndex() + y * imgB.getStride();
			int indexDiff = diff.getStartIndex() + y * diff.getStride();
			
			int indexEnd = indexA+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexA < indexEnd; indexA++, indexB++, indexDiff++ ) {
				diff.data[indexDiff] = (byte)Math.abs((imgA.data[indexA] & 0xFF) - (imgB.data[indexB] & 0xFF));
			}
		}
	}

	/**
	 * Computes the average for each pixel across all bands in the {@link MultiSpectral} image.
	 * 
	 * @param input MultiSpectral image
	 * @param output Gray scale image containing average pixel values
	 */
	public static void averageBand( MultiSpectral<ImageUInt8> input , ImageUInt8 output ) {
		final int h = input.getHeight();
		final int w = input.getWidth();

		ImageUInt8[] bands = input.bands;
		
		for (int y = 0; y < h; y++) {
			int indexInput = input.getStartIndex() + y * input.getStride();
			int indexOutput = output.getStartIndex() + y * output.getStride();

			int indexEnd = indexInput+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexInput < indexEnd; indexInput++, indexOutput++ ) {
				int total = 0;
				for( int i = 0; i < bands.length; i++ ) {
					total += bands[i].data[ indexInput ]& 0xFF;
				}
				output.data[indexOutput] = (byte)(total / bands.length);
			}
		}
	}

	/**
	 * Bounds image pixels to be between these two values
	 * 
	 * @param img Image
	 * @param min minimum value.
	 * @param max maximum value.
	 */
	public static void boundImage( ImageSInt8 img , int min , int max ) {
		final int h = img.getHeight();
		final int w = img.getWidth();

		byte[] data = img.data;

		for (int y = 0; y < h; y++) {
			int index = img.getStartIndex() + y * img.getStride();
			int indexEnd = index+w;
			// for(int x = 0; x < w; x++ ) {
			for (; index < indexEnd; index++) {
				int value = data[index];
				if( value < min )
					data[index] = (byte)min;
				else if( value > max )
					data[index] = (byte)max;
			}
		}
	}

	/**
	 * <p>
	 * Computes the absolute value of the difference between each pixel in the two images.<br>
	 * d(x,y) = |img1(x,y) - img2(x,y)|
	 * </p>
	 * @param imgA Input image. Not modified.
	 * @param imgB Input image. Not modified.
	 * @param diff Absolute value of difference image. Modified.
	 */
	public static void diffAbs( ImageSInt8 imgA , ImageSInt8 imgB , ImageSInt8 diff ) {
		InputSanityCheck.checkSameShape(imgA,imgB,diff);
		
		final int h = imgA.getHeight();
		final int w = imgA.getWidth();

		for (int y = 0; y < h; y++) {
			int indexA = imgA.getStartIndex() + y * imgA.getStride();
			int indexB = imgB.getStartIndex() + y * imgB.getStride();
			int indexDiff = diff.getStartIndex() + y * diff.getStride();
			
			int indexEnd = indexA+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexA < indexEnd; indexA++, indexB++, indexDiff++ ) {
				diff.data[indexDiff] = (byte)Math.abs((imgA.data[indexA] ) - (imgB.data[indexB] ));
			}
		}
	}

	/**
	 * Computes the average for each pixel across all bands in the {@link MultiSpectral} image.
	 * 
	 * @param input MultiSpectral image
	 * @param output Gray scale image containing average pixel values
	 */
	public static void averageBand( MultiSpectral<ImageSInt8> input , ImageSInt8 output ) {
		final int h = input.getHeight();
		final int w = input.getWidth();

		ImageSInt8[] bands = input.bands;
		
		for (int y = 0; y < h; y++) {
			int indexInput = input.getStartIndex() + y * input.getStride();
			int indexOutput = output.getStartIndex() + y * output.getStride();

			int indexEnd = indexInput+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexInput < indexEnd; indexInput++, indexOutput++ ) {
				int total = 0;
				for( int i = 0; i < bands.length; i++ ) {
					total += bands[i].data[ indexInput ];
				}
				output.data[indexOutput] = (byte)(total / bands.length);
			}
		}
	}

	/**
	 * Bounds image pixels to be between these two values
	 * 
	 * @param img Image
	 * @param min minimum value.
	 * @param max maximum value.
	 */
	public static void boundImage( ImageUInt16 img , int min , int max ) {
		final int h = img.getHeight();
		final int w = img.getWidth();

		short[] data = img.data;

		for (int y = 0; y < h; y++) {
			int index = img.getStartIndex() + y * img.getStride();
			int indexEnd = index+w;
			// for(int x = 0; x < w; x++ ) {
			for (; index < indexEnd; index++) {
				int value = data[index]& 0xFFFF;
				if( value < min )
					data[index] = (short)min;
				else if( value > max )
					data[index] = (short)max;
			}
		}
	}

	/**
	 * <p>
	 * Computes the absolute value of the difference between each pixel in the two images.<br>
	 * d(x,y) = |img1(x,y) - img2(x,y)|
	 * </p>
	 * @param imgA Input image. Not modified.
	 * @param imgB Input image. Not modified.
	 * @param diff Absolute value of difference image. Modified.
	 */
	public static void diffAbs( ImageUInt16 imgA , ImageUInt16 imgB , ImageUInt16 diff ) {
		InputSanityCheck.checkSameShape(imgA,imgB,diff);
		
		final int h = imgA.getHeight();
		final int w = imgA.getWidth();

		for (int y = 0; y < h; y++) {
			int indexA = imgA.getStartIndex() + y * imgA.getStride();
			int indexB = imgB.getStartIndex() + y * imgB.getStride();
			int indexDiff = diff.getStartIndex() + y * diff.getStride();
			
			int indexEnd = indexA+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexA < indexEnd; indexA++, indexB++, indexDiff++ ) {
				diff.data[indexDiff] = (short)Math.abs((imgA.data[indexA] & 0xFFFF) - (imgB.data[indexB] & 0xFFFF));
			}
		}
	}

	/**
	 * Computes the average for each pixel across all bands in the {@link MultiSpectral} image.
	 * 
	 * @param input MultiSpectral image
	 * @param output Gray scale image containing average pixel values
	 */
	public static void averageBand( MultiSpectral<ImageUInt16> input , ImageUInt16 output ) {
		final int h = input.getHeight();
		final int w = input.getWidth();

		ImageUInt16[] bands = input.bands;
		
		for (int y = 0; y < h; y++) {
			int indexInput = input.getStartIndex() + y * input.getStride();
			int indexOutput = output.getStartIndex() + y * output.getStride();

			int indexEnd = indexInput+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexInput < indexEnd; indexInput++, indexOutput++ ) {
				int total = 0;
				for( int i = 0; i < bands.length; i++ ) {
					total += bands[i].data[ indexInput ]& 0xFFFF;
				}
				output.data[indexOutput] = (short)(total / bands.length);
			}
		}
	}

	/**
	 * Bounds image pixels to be between these two values
	 * 
	 * @param img Image
	 * @param min minimum value.
	 * @param max maximum value.
	 */
	public static void boundImage( ImageSInt16 img , int min , int max ) {
		final int h = img.getHeight();
		final int w = img.getWidth();

		short[] data = img.data;

		for (int y = 0; y < h; y++) {
			int index = img.getStartIndex() + y * img.getStride();
			int indexEnd = index+w;
			// for(int x = 0; x < w; x++ ) {
			for (; index < indexEnd; index++) {
				int value = data[index];
				if( value < min )
					data[index] = (short)min;
				else if( value > max )
					data[index] = (short)max;
			}
		}
	}

	/**
	 * <p>
	 * Computes the absolute value of the difference between each pixel in the two images.<br>
	 * d(x,y) = |img1(x,y) - img2(x,y)|
	 * </p>
	 * @param imgA Input image. Not modified.
	 * @param imgB Input image. Not modified.
	 * @param diff Absolute value of difference image. Modified.
	 */
	public static void diffAbs( ImageSInt16 imgA , ImageSInt16 imgB , ImageSInt16 diff ) {
		InputSanityCheck.checkSameShape(imgA,imgB,diff);
		
		final int h = imgA.getHeight();
		final int w = imgA.getWidth();

		for (int y = 0; y < h; y++) {
			int indexA = imgA.getStartIndex() + y * imgA.getStride();
			int indexB = imgB.getStartIndex() + y * imgB.getStride();
			int indexDiff = diff.getStartIndex() + y * diff.getStride();
			
			int indexEnd = indexA+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexA < indexEnd; indexA++, indexB++, indexDiff++ ) {
				diff.data[indexDiff] = (short)Math.abs((imgA.data[indexA] ) - (imgB.data[indexB] ));
			}
		}
	}

	/**
	 * Computes the average for each pixel across all bands in the {@link MultiSpectral} image.
	 * 
	 * @param input MultiSpectral image
	 * @param output Gray scale image containing average pixel values
	 */
	public static void averageBand( MultiSpectral<ImageSInt16> input , ImageSInt16 output ) {
		final int h = input.getHeight();
		final int w = input.getWidth();

		ImageSInt16[] bands = input.bands;
		
		for (int y = 0; y < h; y++) {
			int indexInput = input.getStartIndex() + y * input.getStride();
			int indexOutput = output.getStartIndex() + y * output.getStride();

			int indexEnd = indexInput+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexInput < indexEnd; indexInput++, indexOutput++ ) {
				int total = 0;
				for( int i = 0; i < bands.length; i++ ) {
					total += bands[i].data[ indexInput ];
				}
				output.data[indexOutput] = (short)(total / bands.length);
			}
		}
	}

	/**
	 * Bounds image pixels to be between these two values
	 * 
	 * @param img Image
	 * @param min minimum value.
	 * @param max maximum value.
	 */
	public static void boundImage( ImageSInt32 img , int min , int max ) {
		final int h = img.getHeight();
		final int w = img.getWidth();

		int[] data = img.data;

		for (int y = 0; y < h; y++) {
			int index = img.getStartIndex() + y * img.getStride();
			int indexEnd = index+w;
			// for(int x = 0; x < w; x++ ) {
			for (; index < indexEnd; index++) {
				int value = data[index];
				if( value < min )
					data[index] = min;
				else if( value > max )
					data[index] = max;
			}
		}
	}

	/**
	 * <p>
	 * Computes the absolute value of the difference between each pixel in the two images.<br>
	 * d(x,y) = |img1(x,y) - img2(x,y)|
	 * </p>
	 * @param imgA Input image. Not modified.
	 * @param imgB Input image. Not modified.
	 * @param diff Absolute value of difference image. Modified.
	 */
	public static void diffAbs( ImageSInt32 imgA , ImageSInt32 imgB , ImageSInt32 diff ) {
		InputSanityCheck.checkSameShape(imgA,imgB,diff);
		
		final int h = imgA.getHeight();
		final int w = imgA.getWidth();

		for (int y = 0; y < h; y++) {
			int indexA = imgA.getStartIndex() + y * imgA.getStride();
			int indexB = imgB.getStartIndex() + y * imgB.getStride();
			int indexDiff = diff.getStartIndex() + y * diff.getStride();
			
			int indexEnd = indexA+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexA < indexEnd; indexA++, indexB++, indexDiff++ ) {
				diff.data[indexDiff] = (int)Math.abs((imgA.data[indexA] ) - (imgB.data[indexB] ));
			}
		}
	}

	/**
	 * Computes the average for each pixel across all bands in the {@link MultiSpectral} image.
	 * 
	 * @param input MultiSpectral image
	 * @param output Gray scale image containing average pixel values
	 */
	public static void averageBand( MultiSpectral<ImageSInt32> input , ImageSInt32 output ) {
		final int h = input.getHeight();
		final int w = input.getWidth();

		ImageSInt32[] bands = input.bands;
		
		for (int y = 0; y < h; y++) {
			int indexInput = input.getStartIndex() + y * input.getStride();
			int indexOutput = output.getStartIndex() + y * output.getStride();

			int indexEnd = indexInput+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexInput < indexEnd; indexInput++, indexOutput++ ) {
				int total = 0;
				for( int i = 0; i < bands.length; i++ ) {
					total += bands[i].data[ indexInput ];
				}
				output.data[indexOutput] = (total / bands.length);
			}
		}
	}

	/**
	 * Bounds image pixels to be between these two values
	 * 
	 * @param img Image
	 * @param min minimum value.
	 * @param max maximum value.
	 */
	public static void boundImage( ImageSInt64 img , long min , long max ) {
		final int h = img.getHeight();
		final int w = img.getWidth();

		long[] data = img.data;

		for (int y = 0; y < h; y++) {
			int index = img.getStartIndex() + y * img.getStride();
			int indexEnd = index+w;
			// for(int x = 0; x < w; x++ ) {
			for (; index < indexEnd; index++) {
				long value = data[index];
				if( value < min )
					data[index] = min;
				else if( value > max )
					data[index] = max;
			}
		}
	}

	/**
	 * <p>
	 * Computes the absolute value of the difference between each pixel in the two images.<br>
	 * d(x,y) = |img1(x,y) - img2(x,y)|
	 * </p>
	 * @param imgA Input image. Not modified.
	 * @param imgB Input image. Not modified.
	 * @param diff Absolute value of difference image. Modified.
	 */
	public static void diffAbs( ImageSInt64 imgA , ImageSInt64 imgB , ImageSInt64 diff ) {
		InputSanityCheck.checkSameShape(imgA,imgB,diff);
		
		final int h = imgA.getHeight();
		final int w = imgA.getWidth();

		for (int y = 0; y < h; y++) {
			int indexA = imgA.getStartIndex() + y * imgA.getStride();
			int indexB = imgB.getStartIndex() + y * imgB.getStride();
			int indexDiff = diff.getStartIndex() + y * diff.getStride();
			
			int indexEnd = indexA+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexA < indexEnd; indexA++, indexB++, indexDiff++ ) {
				diff.data[indexDiff] = (long)Math.abs((imgA.data[indexA] ) - (imgB.data[indexB] ));
			}
		}
	}

	/**
	 * Computes the average for each pixel across all bands in the {@link MultiSpectral} image.
	 * 
	 * @param input MultiSpectral image
	 * @param output Gray scale image containing average pixel values
	 */
	public static void averageBand( MultiSpectral<ImageSInt64> input , ImageSInt64 output ) {
		final int h = input.getHeight();
		final int w = input.getWidth();

		ImageSInt64[] bands = input.bands;
		
		for (int y = 0; y < h; y++) {
			int indexInput = input.getStartIndex() + y * input.getStride();
			int indexOutput = output.getStartIndex() + y * output.getStride();

			int indexEnd = indexInput+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexInput < indexEnd; indexInput++, indexOutput++ ) {
				long total = 0;
				for( int i = 0; i < bands.length; i++ ) {
					total += bands[i].data[ indexInput ];
				}
				output.data[indexOutput] = (total / bands.length);
			}
		}
	}

	/**
	 * Bounds image pixels to be between these two values
	 * 
	 * @param img Image
	 * @param min minimum value.
	 * @param max maximum value.
	 */
	public static void boundImage( ImageFloat32 img , float min , float max ) {
		final int h = img.getHeight();
		final int w = img.getWidth();

		float[] data = img.data;

		for (int y = 0; y < h; y++) {
			int index = img.getStartIndex() + y * img.getStride();
			int indexEnd = index+w;
			// for(int x = 0; x < w; x++ ) {
			for (; index < indexEnd; index++) {
				float value = data[index];
				if( value < min )
					data[index] = min;
				else if( value > max )
					data[index] = max;
			}
		}
	}

	/**
	 * <p>
	 * Computes the absolute value of the difference between each pixel in the two images.<br>
	 * d(x,y) = |img1(x,y) - img2(x,y)|
	 * </p>
	 * @param imgA Input image. Not modified.
	 * @param imgB Input image. Not modified.
	 * @param diff Absolute value of difference image. Modified.
	 */
	public static void diffAbs( ImageFloat32 imgA , ImageFloat32 imgB , ImageFloat32 diff ) {
		InputSanityCheck.checkSameShape(imgA,imgB,diff);
		
		final int h = imgA.getHeight();
		final int w = imgA.getWidth();

		for (int y = 0; y < h; y++) {
			int indexA = imgA.getStartIndex() + y * imgA.getStride();
			int indexB = imgB.getStartIndex() + y * imgB.getStride();
			int indexDiff = diff.getStartIndex() + y * diff.getStride();
			
			int indexEnd = indexA+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexA < indexEnd; indexA++, indexB++, indexDiff++ ) {
				diff.data[indexDiff] = Math.abs((imgA.data[indexA] ) - (imgB.data[indexB] ));
			}
		}
	}

	/**
	 * Computes the average for each pixel across all bands in the {@link MultiSpectral} image.
	 * 
	 * @param input MultiSpectral image
	 * @param output Gray scale image containing average pixel values
	 */
	public static void averageBand( MultiSpectral<ImageFloat32> input , ImageFloat32 output ) {
		final int h = input.getHeight();
		final int w = input.getWidth();

		ImageFloat32[] bands = input.bands;
		
		for (int y = 0; y < h; y++) {
			int indexInput = input.getStartIndex() + y * input.getStride();
			int indexOutput = output.getStartIndex() + y * output.getStride();

			int indexEnd = indexInput+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexInput < indexEnd; indexInput++, indexOutput++ ) {
				float total = 0;
				for( int i = 0; i < bands.length; i++ ) {
					total += bands[i].data[ indexInput ];
				}
				output.data[indexOutput] = (total / bands.length);
			}
		}
	}

	/**
	 * Bounds image pixels to be between these two values
	 * 
	 * @param img Image
	 * @param min minimum value.
	 * @param max maximum value.
	 */
	public static void boundImage( ImageFloat64 img , double min , double max ) {
		final int h = img.getHeight();
		final int w = img.getWidth();

		double[] data = img.data;

		for (int y = 0; y < h; y++) {
			int index = img.getStartIndex() + y * img.getStride();
			int indexEnd = index+w;
			// for(int x = 0; x < w; x++ ) {
			for (; index < indexEnd; index++) {
				double value = data[index];
				if( value < min )
					data[index] = min;
				else if( value > max )
					data[index] = max;
			}
		}
	}

	/**
	 * <p>
	 * Computes the absolute value of the difference between each pixel in the two images.<br>
	 * d(x,y) = |img1(x,y) - img2(x,y)|
	 * </p>
	 * @param imgA Input image. Not modified.
	 * @param imgB Input image. Not modified.
	 * @param diff Absolute value of difference image. Modified.
	 */
	public static void diffAbs( ImageFloat64 imgA , ImageFloat64 imgB , ImageFloat64 diff ) {
		InputSanityCheck.checkSameShape(imgA,imgB,diff);
		
		final int h = imgA.getHeight();
		final int w = imgA.getWidth();

		for (int y = 0; y < h; y++) {
			int indexA = imgA.getStartIndex() + y * imgA.getStride();
			int indexB = imgB.getStartIndex() + y * imgB.getStride();
			int indexDiff = diff.getStartIndex() + y * diff.getStride();
			
			int indexEnd = indexA+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexA < indexEnd; indexA++, indexB++, indexDiff++ ) {
				diff.data[indexDiff] = Math.abs((imgA.data[indexA] ) - (imgB.data[indexB] ));
			}
		}
	}

	/**
	 * Computes the average for each pixel across all bands in the {@link MultiSpectral} image.
	 * 
	 * @param input MultiSpectral image
	 * @param output Gray scale image containing average pixel values
	 */
	public static void averageBand( MultiSpectral<ImageFloat64> input , ImageFloat64 output ) {
		final int h = input.getHeight();
		final int w = input.getWidth();

		ImageFloat64[] bands = input.bands;
		
		for (int y = 0; y < h; y++) {
			int indexInput = input.getStartIndex() + y * input.getStride();
			int indexOutput = output.getStartIndex() + y * output.getStride();

			int indexEnd = indexInput+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexInput < indexEnd; indexInput++, indexOutput++ ) {
				double total = 0;
				for( int i = 0; i < bands.length; i++ ) {
					total += bands[i].data[ indexInput ];
				}
				output.data[indexOutput] = (total / bands.length);
			}
		}
	}

	/**
	 * <p>
	 * Performs pixel-wise addition<br>
	 * output(x,y) = imgA(x,y) + imgB(x,y)
	 * </p>
	 * @param imgA Input image. Not modified.
	 * @param imgB Input image. Not modified.
	 * @param output Output image. Modified.
	 */
	public static void add( ImageUInt8 imgA , ImageUInt8 imgB , ImageUInt16 output ) {
		InputSanityCheck.checkSameShape(imgA,imgB,output);
		
		final int h = imgA.getHeight();
		final int w = imgA.getWidth();

		for (int y = 0; y < h; y++) {
			int indexA = imgA.getStartIndex() + y * imgA.getStride();
			int indexB = imgB.getStartIndex() + y * imgB.getStride();
			int indexOut = output.getStartIndex() + y * output.getStride();
			
			int indexEnd = indexA+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexA < indexEnd; indexA++, indexB++, indexOut++ ) {
				output.data[indexOut] = (short)((imgA.data[indexA] & 0xFF) + (imgB.data[indexB] & 0xFF));
			}
		}
	}

	/**
	 * <p>
	 * Performs pixel-wise subtraction.<br>
	 * output(x,y) = imgA(x,y) - imgB(x,y)
	 * </p>
	 * @param imgA Input image. Not modified.
	 * @param imgB Input image. Not modified.
	 * @param output Output image. Modified.
	 */
	public static void subtract( ImageUInt8 imgA , ImageUInt8 imgB , ImageInt16 output ) {
		InputSanityCheck.checkSameShape(imgA,imgB,output);
		
		final int h = imgA.getHeight();
		final int w = imgA.getWidth();

		for (int y = 0; y < h; y++) {
			int indexA = imgA.getStartIndex() + y * imgA.getStride();
			int indexB = imgB.getStartIndex() + y * imgB.getStride();
			int indexOut = output.getStartIndex() + y * output.getStride();
			
			int indexEnd = indexA+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexA < indexEnd; indexA++, indexB++, indexOut++ ) {
				output.data[indexOut] = (short)((imgA.data[indexA] & 0xFF) - (imgB.data[indexB] & 0xFF));
			}
		}
	}

	/**
	 * <p>
	 * Performs pixel-wise addition<br>
	 * output(x,y) = imgA(x,y) + imgB(x,y)
	 * </p>
	 * @param imgA Input image. Not modified.
	 * @param imgB Input image. Not modified.
	 * @param output Output image. Modified.
	 */
	public static void add( ImageSInt8 imgA , ImageSInt8 imgB , ImageSInt16 output ) {
		InputSanityCheck.checkSameShape(imgA,imgB,output);
		
		final int h = imgA.getHeight();
		final int w = imgA.getWidth();

		for (int y = 0; y < h; y++) {
			int indexA = imgA.getStartIndex() + y * imgA.getStride();
			int indexB = imgB.getStartIndex() + y * imgB.getStride();
			int indexOut = output.getStartIndex() + y * output.getStride();
			
			int indexEnd = indexA+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexA < indexEnd; indexA++, indexB++, indexOut++ ) {
				output.data[indexOut] = (short)((imgA.data[indexA] ) + (imgB.data[indexB] ));
			}
		}
	}

	/**
	 * <p>
	 * Performs pixel-wise subtraction.<br>
	 * output(x,y) = imgA(x,y) - imgB(x,y)
	 * </p>
	 * @param imgA Input image. Not modified.
	 * @param imgB Input image. Not modified.
	 * @param output Output image. Modified.
	 */
	public static void subtract( ImageSInt8 imgA , ImageSInt8 imgB , ImageSInt16 output ) {
		InputSanityCheck.checkSameShape(imgA,imgB,output);
		
		final int h = imgA.getHeight();
		final int w = imgA.getWidth();

		for (int y = 0; y < h; y++) {
			int indexA = imgA.getStartIndex() + y * imgA.getStride();
			int indexB = imgB.getStartIndex() + y * imgB.getStride();
			int indexOut = output.getStartIndex() + y * output.getStride();
			
			int indexEnd = indexA+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexA < indexEnd; indexA++, indexB++, indexOut++ ) {
				output.data[indexOut] = (short)((imgA.data[indexA] ) - (imgB.data[indexB] ));
			}
		}
	}

	/**
	 * <p>
	 * Performs pixel-wise addition<br>
	 * output(x,y) = imgA(x,y) + imgB(x,y)
	 * </p>
	 * @param imgA Input image. Not modified.
	 * @param imgB Input image. Not modified.
	 * @param output Output image. Modified.
	 */
	public static void add( ImageUInt16 imgA , ImageUInt16 imgB , ImageSInt32 output ) {
		InputSanityCheck.checkSameShape(imgA,imgB,output);
		
		final int h = imgA.getHeight();
		final int w = imgA.getWidth();

		for (int y = 0; y < h; y++) {
			int indexA = imgA.getStartIndex() + y * imgA.getStride();
			int indexB = imgB.getStartIndex() + y * imgB.getStride();
			int indexOut = output.getStartIndex() + y * output.getStride();
			
			int indexEnd = indexA+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexA < indexEnd; indexA++, indexB++, indexOut++ ) {
				output.data[indexOut] = (int)((imgA.data[indexA] & 0xFFFF) + (imgB.data[indexB] & 0xFFFF));
			}
		}
	}

	/**
	 * <p>
	 * Performs pixel-wise subtraction.<br>
	 * output(x,y) = imgA(x,y) - imgB(x,y)
	 * </p>
	 * @param imgA Input image. Not modified.
	 * @param imgB Input image. Not modified.
	 * @param output Output image. Modified.
	 */
	public static void subtract( ImageUInt16 imgA , ImageUInt16 imgB , ImageSInt32 output ) {
		InputSanityCheck.checkSameShape(imgA,imgB,output);
		
		final int h = imgA.getHeight();
		final int w = imgA.getWidth();

		for (int y = 0; y < h; y++) {
			int indexA = imgA.getStartIndex() + y * imgA.getStride();
			int indexB = imgB.getStartIndex() + y * imgB.getStride();
			int indexOut = output.getStartIndex() + y * output.getStride();
			
			int indexEnd = indexA+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexA < indexEnd; indexA++, indexB++, indexOut++ ) {
				output.data[indexOut] = (int)((imgA.data[indexA] & 0xFFFF) - (imgB.data[indexB] & 0xFFFF));
			}
		}
	}

	/**
	 * <p>
	 * Performs pixel-wise addition<br>
	 * output(x,y) = imgA(x,y) + imgB(x,y)
	 * </p>
	 * @param imgA Input image. Not modified.
	 * @param imgB Input image. Not modified.
	 * @param output Output image. Modified.
	 */
	public static void add( ImageSInt16 imgA , ImageSInt16 imgB , ImageSInt32 output ) {
		InputSanityCheck.checkSameShape(imgA,imgB,output);
		
		final int h = imgA.getHeight();
		final int w = imgA.getWidth();

		for (int y = 0; y < h; y++) {
			int indexA = imgA.getStartIndex() + y * imgA.getStride();
			int indexB = imgB.getStartIndex() + y * imgB.getStride();
			int indexOut = output.getStartIndex() + y * output.getStride();
			
			int indexEnd = indexA+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexA < indexEnd; indexA++, indexB++, indexOut++ ) {
				output.data[indexOut] = (int)((imgA.data[indexA] ) + (imgB.data[indexB] ));
			}
		}
	}

	/**
	 * <p>
	 * Performs pixel-wise subtraction.<br>
	 * output(x,y) = imgA(x,y) - imgB(x,y)
	 * </p>
	 * @param imgA Input image. Not modified.
	 * @param imgB Input image. Not modified.
	 * @param output Output image. Modified.
	 */
	public static void subtract( ImageSInt16 imgA , ImageSInt16 imgB , ImageSInt32 output ) {
		InputSanityCheck.checkSameShape(imgA,imgB,output);
		
		final int h = imgA.getHeight();
		final int w = imgA.getWidth();

		for (int y = 0; y < h; y++) {
			int indexA = imgA.getStartIndex() + y * imgA.getStride();
			int indexB = imgB.getStartIndex() + y * imgB.getStride();
			int indexOut = output.getStartIndex() + y * output.getStride();
			
			int indexEnd = indexA+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexA < indexEnd; indexA++, indexB++, indexOut++ ) {
				output.data[indexOut] = (int)((imgA.data[indexA] ) - (imgB.data[indexB] ));
			}
		}
	}

	/**
	 * <p>
	 * Performs pixel-wise addition<br>
	 * output(x,y) = imgA(x,y) + imgB(x,y)
	 * </p>
	 * @param imgA Input image. Not modified.
	 * @param imgB Input image. Not modified.
	 * @param output Output image. Modified.
	 */
	public static void add( ImageSInt32 imgA , ImageSInt32 imgB , ImageSInt32 output ) {
		InputSanityCheck.checkSameShape(imgA,imgB,output);
		
		final int h = imgA.getHeight();
		final int w = imgA.getWidth();

		for (int y = 0; y < h; y++) {
			int indexA = imgA.getStartIndex() + y * imgA.getStride();
			int indexB = imgB.getStartIndex() + y * imgB.getStride();
			int indexOut = output.getStartIndex() + y * output.getStride();
			
			int indexEnd = indexA+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexA < indexEnd; indexA++, indexB++, indexOut++ ) {
				output.data[indexOut] = (int)((imgA.data[indexA] ) + (imgB.data[indexB] ));
			}
		}
	}

	/**
	 * <p>
	 * Performs pixel-wise subtraction.<br>
	 * output(x,y) = imgA(x,y) - imgB(x,y)
	 * </p>
	 * @param imgA Input image. Not modified.
	 * @param imgB Input image. Not modified.
	 * @param output Output image. Modified.
	 */
	public static void subtract( ImageSInt32 imgA , ImageSInt32 imgB , ImageSInt32 output ) {
		InputSanityCheck.checkSameShape(imgA,imgB,output);
		
		final int h = imgA.getHeight();
		final int w = imgA.getWidth();

		for (int y = 0; y < h; y++) {
			int indexA = imgA.getStartIndex() + y * imgA.getStride();
			int indexB = imgB.getStartIndex() + y * imgB.getStride();
			int indexOut = output.getStartIndex() + y * output.getStride();
			
			int indexEnd = indexA+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexA < indexEnd; indexA++, indexB++, indexOut++ ) {
				output.data[indexOut] = (int)((imgA.data[indexA] ) - (imgB.data[indexB] ));
			}
		}
	}

	/**
	 * <p>
	 * Performs pixel-wise addition<br>
	 * output(x,y) = imgA(x,y) + imgB(x,y)
	 * </p>
	 * @param imgA Input image. Not modified.
	 * @param imgB Input image. Not modified.
	 * @param output Output image. Modified.
	 */
	public static void add( ImageSInt64 imgA , ImageSInt64 imgB , ImageSInt64 output ) {
		InputSanityCheck.checkSameShape(imgA,imgB,output);
		
		final int h = imgA.getHeight();
		final int w = imgA.getWidth();

		for (int y = 0; y < h; y++) {
			int indexA = imgA.getStartIndex() + y * imgA.getStride();
			int indexB = imgB.getStartIndex() + y * imgB.getStride();
			int indexOut = output.getStartIndex() + y * output.getStride();
			
			int indexEnd = indexA+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexA < indexEnd; indexA++, indexB++, indexOut++ ) {
				output.data[indexOut] = (long)((imgA.data[indexA] ) + (imgB.data[indexB] ));
			}
		}
	}

	/**
	 * <p>
	 * Performs pixel-wise subtraction.<br>
	 * output(x,y) = imgA(x,y) - imgB(x,y)
	 * </p>
	 * @param imgA Input image. Not modified.
	 * @param imgB Input image. Not modified.
	 * @param output Output image. Modified.
	 */
	public static void subtract( ImageSInt64 imgA , ImageSInt64 imgB , ImageSInt64 output ) {
		InputSanityCheck.checkSameShape(imgA,imgB,output);
		
		final int h = imgA.getHeight();
		final int w = imgA.getWidth();

		for (int y = 0; y < h; y++) {
			int indexA = imgA.getStartIndex() + y * imgA.getStride();
			int indexB = imgB.getStartIndex() + y * imgB.getStride();
			int indexOut = output.getStartIndex() + y * output.getStride();
			
			int indexEnd = indexA+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexA < indexEnd; indexA++, indexB++, indexOut++ ) {
				output.data[indexOut] = (long)((imgA.data[indexA] ) - (imgB.data[indexB] ));
			}
		}
	}

	/**
	 * <p>
	 * Performs pixel-wise addition<br>
	 * output(x,y) = imgA(x,y) + imgB(x,y)
	 * </p>
	 * @param imgA Input image. Not modified.
	 * @param imgB Input image. Not modified.
	 * @param output Output image. Modified.
	 */
	public static void add( ImageFloat32 imgA , ImageFloat32 imgB , ImageFloat32 output ) {
		InputSanityCheck.checkSameShape(imgA,imgB,output);
		
		final int h = imgA.getHeight();
		final int w = imgA.getWidth();

		for (int y = 0; y < h; y++) {
			int indexA = imgA.getStartIndex() + y * imgA.getStride();
			int indexB = imgB.getStartIndex() + y * imgB.getStride();
			int indexOut = output.getStartIndex() + y * output.getStride();
			
			int indexEnd = indexA+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexA < indexEnd; indexA++, indexB++, indexOut++ ) {
				output.data[indexOut] = ((imgA.data[indexA] ) + (imgB.data[indexB] ));
			}
		}
	}

	/**
	 * <p>
	 * Performs pixel-wise subtraction.<br>
	 * output(x,y) = imgA(x,y) - imgB(x,y)
	 * </p>
	 * @param imgA Input image. Not modified.
	 * @param imgB Input image. Not modified.
	 * @param output Output image. Modified.
	 */
	public static void subtract( ImageFloat32 imgA , ImageFloat32 imgB , ImageFloat32 output ) {
		InputSanityCheck.checkSameShape(imgA,imgB,output);
		
		final int h = imgA.getHeight();
		final int w = imgA.getWidth();

		for (int y = 0; y < h; y++) {
			int indexA = imgA.getStartIndex() + y * imgA.getStride();
			int indexB = imgB.getStartIndex() + y * imgB.getStride();
			int indexOut = output.getStartIndex() + y * output.getStride();
			
			int indexEnd = indexA+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexA < indexEnd; indexA++, indexB++, indexOut++ ) {
				output.data[indexOut] = ((imgA.data[indexA] ) - (imgB.data[indexB] ));
			}
		}
	}

	/**
	 * <p>
	 * Performs pixel-wise multiplication<br>
	 * output(x,y) = imgA(x,y) * imgB(x,y)
	 * </p>
	 * @param imgA Input image. Not modified.
	 * @param imgB Input image. Not modified.
	 * @param output Output image. Modified.
	 */
	public static void multiply( ImageFloat32 imgA , ImageFloat32 imgB , ImageFloat32 output ) {
		InputSanityCheck.checkSameShape(imgA,imgB,output);
		
		final int h = imgA.getHeight();
		final int w = imgA.getWidth();

		for (int y = 0; y < h; y++) {
			int indexA = imgA.getStartIndex() + y * imgA.getStride();
			int indexB = imgB.getStartIndex() + y * imgB.getStride();
			int indexOut = output.getStartIndex() + y * output.getStride();
			
			int indexEnd = indexA+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexA < indexEnd; indexA++, indexB++, indexOut++ ) {
				output.data[indexOut] = ((imgA.data[indexA] ) * (imgB.data[indexB] ));
			}
		}
	}

	/**
	 * <p>
	 * Performs pixel-wise division<br>
	 * output(x,y) = imgA(x,y) / imgB(x,y)
	 * </p>
	 * @param imgA Input image. Not modified.
	 * @param imgB Input image. Not modified.
	 * @param output Output image. Modified.
	 */
	public static void divide( ImageFloat32 imgA , ImageFloat32 imgB , ImageFloat32 output ) {
		InputSanityCheck.checkSameShape(imgA,imgB,output);
		
		final int h = imgA.getHeight();
		final int w = imgA.getWidth();

		for (int y = 0; y < h; y++) {
			int indexA = imgA.getStartIndex() + y * imgA.getStride();
			int indexB = imgB.getStartIndex() + y * imgB.getStride();
			int indexOut = output.getStartIndex() + y * output.getStride();
			
			int indexEnd = indexA+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexA < indexEnd; indexA++, indexB++, indexOut++ ) {
				output.data[indexOut] = ((imgA.data[indexA] ) / (imgB.data[indexB] ));
			}
		}
	}

	/**
	 * Sets each pixel in the output image to log( 1 + input(x,y)) of the input image.
	 * Both the input and output image can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param output Where the log image is written to. Modified.
	 */
	public static void log( ImageFloat32 input , ImageFloat32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		for( int y = 0; y < input.height; y++ ) {
			int indexSrc = input.startIndex + y* input.stride;
			int indexDst = output.startIndex + y* output.stride;
			int end = indexSrc + input.width;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output.data[indexDst] = (float)Math.log(1 + input.data[indexSrc]);
			}
		}
	}

	/**
	 * Raises each pixel in the input image to the power of two. Both the input and output image can be the 
	 * same instance.	 *
	 * @param input The input image. Not modified.
	 * @param output Where the pow2 image is written to. Modified.
	 */
	public static void pow2( ImageFloat32 input , ImageFloat32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		for( int y = 0; y < input.height; y++ ) {
			int indexSrc = input.startIndex + y* input.stride;
			int indexDst = output.startIndex + y* output.stride;
			int end = indexSrc + input.width;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				float v = input.data[indexSrc];
				output.data[indexDst] = v*v;
			}
		}
	}

	/**
	 * Computes the square root of each pixel in the input image. Both the input and output image can be the
	 * same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param output Where the sqrt() image is written to. Modified.
	 */
	public static void sqrt( ImageFloat32 input , ImageFloat32 output ) {

		InputSanityCheck.checkSameShape(input,output);

		for( int y = 0; y < input.height; y++ ) {
			int indexSrc = input.startIndex + y* input.stride;
			int indexDst = output.startIndex + y* output.stride;
			int end = indexSrc + input.width;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output.data[indexDst] = (float)Math.sqrt(input.data[indexSrc]);
			}
		}
	}

	/**
	 * <p>
	 * Performs pixel-wise addition<br>
	 * output(x,y) = imgA(x,y) + imgB(x,y)
	 * </p>
	 * @param imgA Input image. Not modified.
	 * @param imgB Input image. Not modified.
	 * @param output Output image. Modified.
	 */
	public static void add( ImageFloat64 imgA , ImageFloat64 imgB , ImageFloat64 output ) {
		InputSanityCheck.checkSameShape(imgA,imgB,output);
		
		final int h = imgA.getHeight();
		final int w = imgA.getWidth();

		for (int y = 0; y < h; y++) {
			int indexA = imgA.getStartIndex() + y * imgA.getStride();
			int indexB = imgB.getStartIndex() + y * imgB.getStride();
			int indexOut = output.getStartIndex() + y * output.getStride();
			
			int indexEnd = indexA+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexA < indexEnd; indexA++, indexB++, indexOut++ ) {
				output.data[indexOut] = ((imgA.data[indexA] ) + (imgB.data[indexB] ));
			}
		}
	}

	/**
	 * <p>
	 * Performs pixel-wise subtraction.<br>
	 * output(x,y) = imgA(x,y) - imgB(x,y)
	 * </p>
	 * @param imgA Input image. Not modified.
	 * @param imgB Input image. Not modified.
	 * @param output Output image. Modified.
	 */
	public static void subtract( ImageFloat64 imgA , ImageFloat64 imgB , ImageFloat64 output ) {
		InputSanityCheck.checkSameShape(imgA,imgB,output);
		
		final int h = imgA.getHeight();
		final int w = imgA.getWidth();

		for (int y = 0; y < h; y++) {
			int indexA = imgA.getStartIndex() + y * imgA.getStride();
			int indexB = imgB.getStartIndex() + y * imgB.getStride();
			int indexOut = output.getStartIndex() + y * output.getStride();
			
			int indexEnd = indexA+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexA < indexEnd; indexA++, indexB++, indexOut++ ) {
				output.data[indexOut] = ((imgA.data[indexA] ) - (imgB.data[indexB] ));
			}
		}
	}

	/**
	 * <p>
	 * Performs pixel-wise multiplication<br>
	 * output(x,y) = imgA(x,y) * imgB(x,y)
	 * </p>
	 * @param imgA Input image. Not modified.
	 * @param imgB Input image. Not modified.
	 * @param output Output image. Modified.
	 */
	public static void multiply( ImageFloat64 imgA , ImageFloat64 imgB , ImageFloat64 output ) {
		InputSanityCheck.checkSameShape(imgA,imgB,output);
		
		final int h = imgA.getHeight();
		final int w = imgA.getWidth();

		for (int y = 0; y < h; y++) {
			int indexA = imgA.getStartIndex() + y * imgA.getStride();
			int indexB = imgB.getStartIndex() + y * imgB.getStride();
			int indexOut = output.getStartIndex() + y * output.getStride();
			
			int indexEnd = indexA+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexA < indexEnd; indexA++, indexB++, indexOut++ ) {
				output.data[indexOut] = ((imgA.data[indexA] ) * (imgB.data[indexB] ));
			}
		}
	}

	/**
	 * <p>
	 * Performs pixel-wise division<br>
	 * output(x,y) = imgA(x,y) / imgB(x,y)
	 * </p>
	 * @param imgA Input image. Not modified.
	 * @param imgB Input image. Not modified.
	 * @param output Output image. Modified.
	 */
	public static void divide( ImageFloat64 imgA , ImageFloat64 imgB , ImageFloat64 output ) {
		InputSanityCheck.checkSameShape(imgA,imgB,output);
		
		final int h = imgA.getHeight();
		final int w = imgA.getWidth();

		for (int y = 0; y < h; y++) {
			int indexA = imgA.getStartIndex() + y * imgA.getStride();
			int indexB = imgB.getStartIndex() + y * imgB.getStride();
			int indexOut = output.getStartIndex() + y * output.getStride();
			
			int indexEnd = indexA+w;
			// for(int x = 0; x < w; x++ ) {
			for (; indexA < indexEnd; indexA++, indexB++, indexOut++ ) {
				output.data[indexOut] = ((imgA.data[indexA] ) / (imgB.data[indexB] ));
			}
		}
	}

	/**
	 * Sets each pixel in the output image to log( 1 + input(x,y)) of the input image.
	 * Both the input and output image can be the same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param output Where the log image is written to. Modified.
	 */
	public static void log( ImageFloat64 input , ImageFloat64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		for( int y = 0; y < input.height; y++ ) {
			int indexSrc = input.startIndex + y* input.stride;
			int indexDst = output.startIndex + y* output.stride;
			int end = indexSrc + input.width;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output.data[indexDst] = Math.log(1 + input.data[indexSrc]);
			}
		}
	}

	/**
	 * Raises each pixel in the input image to the power of two. Both the input and output image can be the 
	 * same instance.	 *
	 * @param input The input image. Not modified.
	 * @param output Where the pow2 image is written to. Modified.
	 */
	public static void pow2( ImageFloat64 input , ImageFloat64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		for( int y = 0; y < input.height; y++ ) {
			int indexSrc = input.startIndex + y* input.stride;
			int indexDst = output.startIndex + y* output.stride;
			int end = indexSrc + input.width;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				double v = input.data[indexSrc];
				output.data[indexDst] = v*v;
			}
		}
	}

	/**
	 * Computes the square root of each pixel in the input image. Both the input and output image can be the
	 * same instance.
	 *
	 * @param input The input image. Not modified.
	 * @param output Where the sqrt() image is written to. Modified.
	 */
	public static void sqrt( ImageFloat64 input , ImageFloat64 output ) {

		InputSanityCheck.checkSameShape(input,output);

		for( int y = 0; y < input.height; y++ ) {
			int indexSrc = input.startIndex + y* input.stride;
			int indexDst = output.startIndex + y* output.stride;
			int end = indexSrc + input.width;

			for( ; indexSrc < end; indexSrc++ , indexDst++) {
				output.data[indexDst] = Math.sqrt(input.data[indexSrc]);
			}
		}
	}

}
