/*
 * Copyright (c) 2011-2016, Peter Abeles. All Rights Reserved.
 *
 * This file is part of BoofCV (http://boofcv.org).
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package boofcv.core.image;

import boofcv.alg.InputSanityCheck;
import boofcv.core.image.impl.ConvertInterleavedToSingle;
import boofcv.core.image.impl.ImplConvertImage;
import boofcv.core.image.impl.ImplConvertMsToSingle;
import boofcv.struct.image.*;

import javax.annotation.Generated;

/**
 * <p>
 * Functions for converting between different image types. Pixel values are converted by typecasting.
 * When converting between signed and unsigned types, care should be taken to avoid numerical overflow.
 * </p>
 *
 * <p>
 * DO NOT MODIFY: This class was automatically generated by GenerateConvertImage
 * </p>
 *
 * @author Peter Abeles
 */
@Generated("boofcv.core.image.GenerateConvertImage")
public class ConvertImage {

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageUInt8} into a {@link boofcv.struct.image.ImageSInt8}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt8 convert(ImageUInt8 input, ImageSInt8 output) {
		if (output == null) {
			output = new ImageSInt8(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedU8} into a {@link boofcv.struct.image.InterleavedS8}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS8 convert(InterleavedU8 input, InterleavedS8 output) {
		if (output == null) {
			output = new InterleavedS8(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageUInt8} into a {@link boofcv.struct.image.ImageUInt16}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageUInt16 convert(ImageUInt8 input, ImageUInt16 output) {
		if (output == null) {
			output = new ImageUInt16(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedU8} into a {@link boofcv.struct.image.InterleavedU16}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedU16 convert(InterleavedU8 input, InterleavedU16 output) {
		if (output == null) {
			output = new InterleavedU16(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageUInt8} into a {@link boofcv.struct.image.ImageSInt16}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt16 convert(ImageUInt8 input, ImageSInt16 output) {
		if (output == null) {
			output = new ImageSInt16(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedU8} into a {@link boofcv.struct.image.InterleavedS16}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS16 convert(InterleavedU8 input, InterleavedS16 output) {
		if (output == null) {
			output = new InterleavedS16(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageUInt8} into a {@link boofcv.struct.image.ImageSInt32}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt32 convert(ImageUInt8 input, ImageSInt32 output) {
		if (output == null) {
			output = new ImageSInt32(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedU8} into a {@link boofcv.struct.image.InterleavedS32}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS32 convert(InterleavedU8 input, InterleavedS32 output) {
		if (output == null) {
			output = new InterleavedS32(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageUInt8} into a {@link boofcv.struct.image.ImageSInt64}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt64 convert(ImageUInt8 input, ImageSInt64 output) {
		if (output == null) {
			output = new ImageSInt64(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedU8} into a {@link boofcv.struct.image.InterleavedS64}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS64 convert(InterleavedU8 input, InterleavedS64 output) {
		if (output == null) {
			output = new InterleavedS64(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageUInt8} into a {@link boofcv.struct.image.ImageFloat32}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageFloat32 convert(ImageUInt8 input, ImageFloat32 output) {
		if (output == null) {
			output = new ImageFloat32(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedU8} into a {@link boofcv.struct.image.InterleavedF32}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedF32 convert(InterleavedU8 input, InterleavedF32 output) {
		if (output == null) {
			output = new InterleavedF32(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageUInt8} into a {@link boofcv.struct.image.ImageFloat64}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageFloat64 convert(ImageUInt8 input, ImageFloat64 output) {
		if (output == null) {
			output = new ImageFloat64(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedU8} into a {@link boofcv.struct.image.InterleavedF64}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedF64 convert(InterleavedU8 input, InterleavedF64 output) {
		if (output == null) {
			output = new InterleavedF64(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * Converts a {@link MultiSpectral} into a {@link ImageSingleBand} by computing the average value of each pixel
	 * across all the bands.
	 * 
	 * @param input Input MultiSpectral image that is being converted. Not modified.
	 * @param output (Optional) The single band output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageUInt8 average( MultiSpectral<ImageUInt8> input , ImageUInt8 output ) {
		if (output == null) {
			output = new ImageUInt8(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertMsToSingle.average(input, output);

		return output;
	}

	/**
	 * Converts a {@link MultiSpectral} into the equivalent {@link InterleavedU8}
	 *
	 * @param input (Input) MultiSpectral image that is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedU8 convert( MultiSpectral<ImageUInt8> input , InterleavedU8 output ) {
		if (output == null) {
			output = new InterleavedU8(input.width, input.height,input.getNumBands());
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input,output);

		return output;
	}

	/**
	 * Converts a {@link InterleavedU8} into a {@link ImageUInt8} by computing the average value of each pixel
	 * across all the bands.
	 * 
	 * @param input (Input) The ImageInterleaved that is being converted. Not modified.
	 * @param output (Optional) The single band output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageUInt8 average( InterleavedU8 input , ImageUInt8 output ) {
		if (output == null) {
			output = new ImageUInt8(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ConvertInterleavedToSingle.average(input, output);

		return output;
	}

	/**
	 * Converts a {@link InterleavedU8} into the equivalent {@link MultiSpectral}
	 * 
	 * @param input (Input) ImageInterleaved that is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static MultiSpectral<ImageUInt8> convert( InterleavedU8 input , MultiSpectral<ImageUInt8> output ) {
		if (output == null) {
			output = new MultiSpectral<ImageUInt8>(ImageUInt8.class,input.width, input.height,input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input,output);

		return output;
	}

	/**
	 * Converts pixel values in the input image into an integer values from 0 to numValues. 
	 * @param input Input image
	 * @param min minimum input pixel value, inclusive
	 * @param max maximum input pixel value, inclusive
	 * @param numValues Number of possible pixel values in output image
	 * @param output (Optional) Storage for the output image.  Can be null.
	 * @return The converted output image.
	 */
	public static ImageUInt8 convert(ImageUInt8 input , int min , int max , int numValues , ImageUInt8 output )
	{
		if (output == null) {
			output = new ImageUInt8(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}
		if( numValues < 0 || numValues > 256 )
			throw new IllegalArgumentException("0 <= numValues <= 256");

		numValues -= 1;
		int range = max-min;

		for (int y = 0; y < input.height; y++) {
			int indexIn = input.startIndex + y*input.stride;
			int indexOut = output.startIndex + y*output.stride;

			for (int x = 0; x < input.width; x++) {
				int value = (int)(numValues*((input.data[indexIn++]& 0xFF)-min)/range );
				output.data[indexOut++] = (byte)value;
			}
		}
	return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageSInt8} into a {@link boofcv.struct.image.ImageUInt8}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageUInt8 convert(ImageSInt8 input, ImageUInt8 output) {
		if (output == null) {
			output = new ImageUInt8(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedS8} into a {@link boofcv.struct.image.InterleavedU8}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedU8 convert(InterleavedS8 input, InterleavedU8 output) {
		if (output == null) {
			output = new InterleavedU8(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageSInt8} into a {@link boofcv.struct.image.ImageUInt16}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageUInt16 convert(ImageSInt8 input, ImageUInt16 output) {
		if (output == null) {
			output = new ImageUInt16(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedS8} into a {@link boofcv.struct.image.InterleavedU16}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedU16 convert(InterleavedS8 input, InterleavedU16 output) {
		if (output == null) {
			output = new InterleavedU16(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageSInt8} into a {@link boofcv.struct.image.ImageSInt16}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt16 convert(ImageSInt8 input, ImageSInt16 output) {
		if (output == null) {
			output = new ImageSInt16(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedS8} into a {@link boofcv.struct.image.InterleavedS16}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS16 convert(InterleavedS8 input, InterleavedS16 output) {
		if (output == null) {
			output = new InterleavedS16(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageSInt8} into a {@link boofcv.struct.image.ImageSInt32}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt32 convert(ImageSInt8 input, ImageSInt32 output) {
		if (output == null) {
			output = new ImageSInt32(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedS8} into a {@link boofcv.struct.image.InterleavedS32}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS32 convert(InterleavedS8 input, InterleavedS32 output) {
		if (output == null) {
			output = new InterleavedS32(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageSInt8} into a {@link boofcv.struct.image.ImageSInt64}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt64 convert(ImageSInt8 input, ImageSInt64 output) {
		if (output == null) {
			output = new ImageSInt64(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedS8} into a {@link boofcv.struct.image.InterleavedS64}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS64 convert(InterleavedS8 input, InterleavedS64 output) {
		if (output == null) {
			output = new InterleavedS64(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageSInt8} into a {@link boofcv.struct.image.ImageFloat32}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageFloat32 convert(ImageSInt8 input, ImageFloat32 output) {
		if (output == null) {
			output = new ImageFloat32(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedS8} into a {@link boofcv.struct.image.InterleavedF32}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedF32 convert(InterleavedS8 input, InterleavedF32 output) {
		if (output == null) {
			output = new InterleavedF32(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageSInt8} into a {@link boofcv.struct.image.ImageFloat64}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageFloat64 convert(ImageSInt8 input, ImageFloat64 output) {
		if (output == null) {
			output = new ImageFloat64(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedS8} into a {@link boofcv.struct.image.InterleavedF64}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedF64 convert(InterleavedS8 input, InterleavedF64 output) {
		if (output == null) {
			output = new InterleavedF64(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * Converts a {@link MultiSpectral} into a {@link ImageSingleBand} by computing the average value of each pixel
	 * across all the bands.
	 * 
	 * @param input Input MultiSpectral image that is being converted. Not modified.
	 * @param output (Optional) The single band output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt8 average( MultiSpectral<ImageSInt8> input , ImageSInt8 output ) {
		if (output == null) {
			output = new ImageSInt8(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertMsToSingle.average(input, output);

		return output;
	}

	/**
	 * Converts a {@link MultiSpectral} into the equivalent {@link InterleavedS8}
	 *
	 * @param input (Input) MultiSpectral image that is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS8 convert( MultiSpectral<ImageSInt8> input , InterleavedS8 output ) {
		if (output == null) {
			output = new InterleavedS8(input.width, input.height,input.getNumBands());
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input,output);

		return output;
	}

	/**
	 * Converts a {@link InterleavedS8} into a {@link ImageSInt8} by computing the average value of each pixel
	 * across all the bands.
	 * 
	 * @param input (Input) The ImageInterleaved that is being converted. Not modified.
	 * @param output (Optional) The single band output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt8 average( InterleavedS8 input , ImageSInt8 output ) {
		if (output == null) {
			output = new ImageSInt8(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ConvertInterleavedToSingle.average(input, output);

		return output;
	}

	/**
	 * Converts a {@link InterleavedS8} into the equivalent {@link MultiSpectral}
	 * 
	 * @param input (Input) ImageInterleaved that is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static MultiSpectral<ImageSInt8> convert( InterleavedS8 input , MultiSpectral<ImageSInt8> output ) {
		if (output == null) {
			output = new MultiSpectral<ImageSInt8>(ImageSInt8.class,input.width, input.height,input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input,output);

		return output;
	}

	/**
	 * Converts pixel values in the input image into an integer values from 0 to numValues. 
	 * @param input Input image
	 * @param min minimum input pixel value, inclusive
	 * @param max maximum input pixel value, inclusive
	 * @param numValues Number of possible pixel values in output image
	 * @param output (Optional) Storage for the output image.  Can be null.
	 * @return The converted output image.
	 */
	public static ImageUInt8 convert(ImageSInt8 input , int min , int max , int numValues , ImageUInt8 output )
	{
		if (output == null) {
			output = new ImageUInt8(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}
		if( numValues < 0 || numValues > 256 )
			throw new IllegalArgumentException("0 <= numValues <= 256");

		numValues -= 1;
		int range = max-min;

		for (int y = 0; y < input.height; y++) {
			int indexIn = input.startIndex + y*input.stride;
			int indexOut = output.startIndex + y*output.stride;

			for (int x = 0; x < input.width; x++) {
				int value = (int)(numValues*((input.data[indexIn++])-min)/range );
				output.data[indexOut++] = (byte)value;
			}
		}
	return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageUInt16} into a {@link boofcv.struct.image.ImageUInt8}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageUInt8 convert(ImageUInt16 input, ImageUInt8 output) {
		if (output == null) {
			output = new ImageUInt8(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedU16} into a {@link boofcv.struct.image.InterleavedU8}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedU8 convert(InterleavedU16 input, InterleavedU8 output) {
		if (output == null) {
			output = new InterleavedU8(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageUInt16} into a {@link boofcv.struct.image.ImageSInt8}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt8 convert(ImageUInt16 input, ImageSInt8 output) {
		if (output == null) {
			output = new ImageSInt8(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedU16} into a {@link boofcv.struct.image.InterleavedS8}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS8 convert(InterleavedU16 input, InterleavedS8 output) {
		if (output == null) {
			output = new InterleavedS8(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageUInt16} into a {@link boofcv.struct.image.ImageSInt16}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt16 convert(ImageUInt16 input, ImageSInt16 output) {
		if (output == null) {
			output = new ImageSInt16(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedU16} into a {@link boofcv.struct.image.InterleavedS16}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS16 convert(InterleavedU16 input, InterleavedS16 output) {
		if (output == null) {
			output = new InterleavedS16(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageUInt16} into a {@link boofcv.struct.image.ImageSInt32}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt32 convert(ImageUInt16 input, ImageSInt32 output) {
		if (output == null) {
			output = new ImageSInt32(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedU16} into a {@link boofcv.struct.image.InterleavedS32}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS32 convert(InterleavedU16 input, InterleavedS32 output) {
		if (output == null) {
			output = new InterleavedS32(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageUInt16} into a {@link boofcv.struct.image.ImageSInt64}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt64 convert(ImageUInt16 input, ImageSInt64 output) {
		if (output == null) {
			output = new ImageSInt64(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedU16} into a {@link boofcv.struct.image.InterleavedS64}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS64 convert(InterleavedU16 input, InterleavedS64 output) {
		if (output == null) {
			output = new InterleavedS64(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageUInt16} into a {@link boofcv.struct.image.ImageFloat32}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageFloat32 convert(ImageUInt16 input, ImageFloat32 output) {
		if (output == null) {
			output = new ImageFloat32(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedU16} into a {@link boofcv.struct.image.InterleavedF32}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedF32 convert(InterleavedU16 input, InterleavedF32 output) {
		if (output == null) {
			output = new InterleavedF32(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageUInt16} into a {@link boofcv.struct.image.ImageFloat64}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageFloat64 convert(ImageUInt16 input, ImageFloat64 output) {
		if (output == null) {
			output = new ImageFloat64(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedU16} into a {@link boofcv.struct.image.InterleavedF64}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedF64 convert(InterleavedU16 input, InterleavedF64 output) {
		if (output == null) {
			output = new InterleavedF64(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * Converts a {@link MultiSpectral} into a {@link ImageSingleBand} by computing the average value of each pixel
	 * across all the bands.
	 * 
	 * @param input Input MultiSpectral image that is being converted. Not modified.
	 * @param output (Optional) The single band output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageUInt16 average( MultiSpectral<ImageUInt16> input , ImageUInt16 output ) {
		if (output == null) {
			output = new ImageUInt16(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertMsToSingle.average(input, output);

		return output;
	}

	/**
	 * Converts a {@link MultiSpectral} into the equivalent {@link InterleavedU16}
	 *
	 * @param input (Input) MultiSpectral image that is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedU16 convert( MultiSpectral<ImageUInt16> input , InterleavedU16 output ) {
		if (output == null) {
			output = new InterleavedU16(input.width, input.height,input.getNumBands());
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input,output);

		return output;
	}

	/**
	 * Converts a {@link InterleavedU16} into a {@link ImageUInt16} by computing the average value of each pixel
	 * across all the bands.
	 * 
	 * @param input (Input) The ImageInterleaved that is being converted. Not modified.
	 * @param output (Optional) The single band output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageUInt16 average( InterleavedU16 input , ImageUInt16 output ) {
		if (output == null) {
			output = new ImageUInt16(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ConvertInterleavedToSingle.average(input, output);

		return output;
	}

	/**
	 * Converts a {@link InterleavedU16} into the equivalent {@link MultiSpectral}
	 * 
	 * @param input (Input) ImageInterleaved that is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static MultiSpectral<ImageUInt16> convert( InterleavedU16 input , MultiSpectral<ImageUInt16> output ) {
		if (output == null) {
			output = new MultiSpectral<ImageUInt16>(ImageUInt16.class,input.width, input.height,input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input,output);

		return output;
	}

	/**
	 * Converts pixel values in the input image into an integer values from 0 to numValues. 
	 * @param input Input image
	 * @param min minimum input pixel value, inclusive
	 * @param max maximum input pixel value, inclusive
	 * @param numValues Number of possible pixel values in output image
	 * @param output (Optional) Storage for the output image.  Can be null.
	 * @return The converted output image.
	 */
	public static ImageUInt8 convert(ImageUInt16 input , int min , int max , int numValues , ImageUInt8 output )
	{
		if (output == null) {
			output = new ImageUInt8(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}
		if( numValues < 0 || numValues > 256 )
			throw new IllegalArgumentException("0 <= numValues <= 256");

		numValues -= 1;
		int range = max-min;

		for (int y = 0; y < input.height; y++) {
			int indexIn = input.startIndex + y*input.stride;
			int indexOut = output.startIndex + y*output.stride;

			for (int x = 0; x < input.width; x++) {
				int value = (int)(numValues*((input.data[indexIn++]& 0xFFFF)-min)/range );
				output.data[indexOut++] = (byte)value;
			}
		}
	return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageSInt16} into a {@link boofcv.struct.image.ImageUInt8}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageUInt8 convert(ImageSInt16 input, ImageUInt8 output) {
		if (output == null) {
			output = new ImageUInt8(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedS16} into a {@link boofcv.struct.image.InterleavedU8}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedU8 convert(InterleavedS16 input, InterleavedU8 output) {
		if (output == null) {
			output = new InterleavedU8(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageSInt16} into a {@link boofcv.struct.image.ImageSInt8}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt8 convert(ImageSInt16 input, ImageSInt8 output) {
		if (output == null) {
			output = new ImageSInt8(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedS16} into a {@link boofcv.struct.image.InterleavedS8}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS8 convert(InterleavedS16 input, InterleavedS8 output) {
		if (output == null) {
			output = new InterleavedS8(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageSInt16} into a {@link boofcv.struct.image.ImageUInt16}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageUInt16 convert(ImageSInt16 input, ImageUInt16 output) {
		if (output == null) {
			output = new ImageUInt16(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedS16} into a {@link boofcv.struct.image.InterleavedU16}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedU16 convert(InterleavedS16 input, InterleavedU16 output) {
		if (output == null) {
			output = new InterleavedU16(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageSInt16} into a {@link boofcv.struct.image.ImageSInt32}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt32 convert(ImageSInt16 input, ImageSInt32 output) {
		if (output == null) {
			output = new ImageSInt32(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedS16} into a {@link boofcv.struct.image.InterleavedS32}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS32 convert(InterleavedS16 input, InterleavedS32 output) {
		if (output == null) {
			output = new InterleavedS32(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageSInt16} into a {@link boofcv.struct.image.ImageSInt64}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt64 convert(ImageSInt16 input, ImageSInt64 output) {
		if (output == null) {
			output = new ImageSInt64(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedS16} into a {@link boofcv.struct.image.InterleavedS64}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS64 convert(InterleavedS16 input, InterleavedS64 output) {
		if (output == null) {
			output = new InterleavedS64(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageSInt16} into a {@link boofcv.struct.image.ImageFloat32}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageFloat32 convert(ImageSInt16 input, ImageFloat32 output) {
		if (output == null) {
			output = new ImageFloat32(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedS16} into a {@link boofcv.struct.image.InterleavedF32}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedF32 convert(InterleavedS16 input, InterleavedF32 output) {
		if (output == null) {
			output = new InterleavedF32(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageSInt16} into a {@link boofcv.struct.image.ImageFloat64}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageFloat64 convert(ImageSInt16 input, ImageFloat64 output) {
		if (output == null) {
			output = new ImageFloat64(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedS16} into a {@link boofcv.struct.image.InterleavedF64}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedF64 convert(InterleavedS16 input, InterleavedF64 output) {
		if (output == null) {
			output = new InterleavedF64(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * Converts a {@link MultiSpectral} into a {@link ImageSingleBand} by computing the average value of each pixel
	 * across all the bands.
	 * 
	 * @param input Input MultiSpectral image that is being converted. Not modified.
	 * @param output (Optional) The single band output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt16 average( MultiSpectral<ImageSInt16> input , ImageSInt16 output ) {
		if (output == null) {
			output = new ImageSInt16(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertMsToSingle.average(input, output);

		return output;
	}

	/**
	 * Converts a {@link MultiSpectral} into the equivalent {@link InterleavedS16}
	 *
	 * @param input (Input) MultiSpectral image that is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS16 convert( MultiSpectral<ImageSInt16> input , InterleavedS16 output ) {
		if (output == null) {
			output = new InterleavedS16(input.width, input.height,input.getNumBands());
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input,output);

		return output;
	}

	/**
	 * Converts a {@link InterleavedS16} into a {@link ImageSInt16} by computing the average value of each pixel
	 * across all the bands.
	 * 
	 * @param input (Input) The ImageInterleaved that is being converted. Not modified.
	 * @param output (Optional) The single band output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt16 average( InterleavedS16 input , ImageSInt16 output ) {
		if (output == null) {
			output = new ImageSInt16(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ConvertInterleavedToSingle.average(input, output);

		return output;
	}

	/**
	 * Converts a {@link InterleavedS16} into the equivalent {@link MultiSpectral}
	 * 
	 * @param input (Input) ImageInterleaved that is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static MultiSpectral<ImageSInt16> convert( InterleavedS16 input , MultiSpectral<ImageSInt16> output ) {
		if (output == null) {
			output = new MultiSpectral<ImageSInt16>(ImageSInt16.class,input.width, input.height,input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input,output);

		return output;
	}

	/**
	 * Converts pixel values in the input image into an integer values from 0 to numValues. 
	 * @param input Input image
	 * @param min minimum input pixel value, inclusive
	 * @param max maximum input pixel value, inclusive
	 * @param numValues Number of possible pixel values in output image
	 * @param output (Optional) Storage for the output image.  Can be null.
	 * @return The converted output image.
	 */
	public static ImageUInt8 convert(ImageSInt16 input , int min , int max , int numValues , ImageUInt8 output )
	{
		if (output == null) {
			output = new ImageUInt8(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}
		if( numValues < 0 || numValues > 256 )
			throw new IllegalArgumentException("0 <= numValues <= 256");

		numValues -= 1;
		int range = max-min;

		for (int y = 0; y < input.height; y++) {
			int indexIn = input.startIndex + y*input.stride;
			int indexOut = output.startIndex + y*output.stride;

			for (int x = 0; x < input.width; x++) {
				int value = (int)(numValues*((input.data[indexIn++])-min)/range );
				output.data[indexOut++] = (byte)value;
			}
		}
	return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageSInt32} into a {@link boofcv.struct.image.ImageUInt8}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageUInt8 convert(ImageSInt32 input, ImageUInt8 output) {
		if (output == null) {
			output = new ImageUInt8(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedS32} into a {@link boofcv.struct.image.InterleavedU8}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedU8 convert(InterleavedS32 input, InterleavedU8 output) {
		if (output == null) {
			output = new InterleavedU8(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageSInt32} into a {@link boofcv.struct.image.ImageSInt8}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt8 convert(ImageSInt32 input, ImageSInt8 output) {
		if (output == null) {
			output = new ImageSInt8(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedS32} into a {@link boofcv.struct.image.InterleavedS8}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS8 convert(InterleavedS32 input, InterleavedS8 output) {
		if (output == null) {
			output = new InterleavedS8(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageSInt32} into a {@link boofcv.struct.image.ImageUInt16}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageUInt16 convert(ImageSInt32 input, ImageUInt16 output) {
		if (output == null) {
			output = new ImageUInt16(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedS32} into a {@link boofcv.struct.image.InterleavedU16}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedU16 convert(InterleavedS32 input, InterleavedU16 output) {
		if (output == null) {
			output = new InterleavedU16(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageSInt32} into a {@link boofcv.struct.image.ImageSInt16}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt16 convert(ImageSInt32 input, ImageSInt16 output) {
		if (output == null) {
			output = new ImageSInt16(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedS32} into a {@link boofcv.struct.image.InterleavedS16}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS16 convert(InterleavedS32 input, InterleavedS16 output) {
		if (output == null) {
			output = new InterleavedS16(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageSInt32} into a {@link boofcv.struct.image.ImageSInt64}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt64 convert(ImageSInt32 input, ImageSInt64 output) {
		if (output == null) {
			output = new ImageSInt64(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedS32} into a {@link boofcv.struct.image.InterleavedS64}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS64 convert(InterleavedS32 input, InterleavedS64 output) {
		if (output == null) {
			output = new InterleavedS64(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageSInt32} into a {@link boofcv.struct.image.ImageFloat32}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageFloat32 convert(ImageSInt32 input, ImageFloat32 output) {
		if (output == null) {
			output = new ImageFloat32(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedS32} into a {@link boofcv.struct.image.InterleavedF32}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedF32 convert(InterleavedS32 input, InterleavedF32 output) {
		if (output == null) {
			output = new InterleavedF32(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageSInt32} into a {@link boofcv.struct.image.ImageFloat64}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageFloat64 convert(ImageSInt32 input, ImageFloat64 output) {
		if (output == null) {
			output = new ImageFloat64(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedS32} into a {@link boofcv.struct.image.InterleavedF64}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedF64 convert(InterleavedS32 input, InterleavedF64 output) {
		if (output == null) {
			output = new InterleavedF64(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * Converts a {@link MultiSpectral} into a {@link ImageSingleBand} by computing the average value of each pixel
	 * across all the bands.
	 * 
	 * @param input Input MultiSpectral image that is being converted. Not modified.
	 * @param output (Optional) The single band output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt32 average( MultiSpectral<ImageSInt32> input , ImageSInt32 output ) {
		if (output == null) {
			output = new ImageSInt32(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertMsToSingle.average(input, output);

		return output;
	}

	/**
	 * Converts a {@link MultiSpectral} into the equivalent {@link InterleavedS32}
	 *
	 * @param input (Input) MultiSpectral image that is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS32 convert( MultiSpectral<ImageSInt32> input , InterleavedS32 output ) {
		if (output == null) {
			output = new InterleavedS32(input.width, input.height,input.getNumBands());
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input,output);

		return output;
	}

	/**
	 * Converts a {@link InterleavedS32} into a {@link ImageSInt32} by computing the average value of each pixel
	 * across all the bands.
	 * 
	 * @param input (Input) The ImageInterleaved that is being converted. Not modified.
	 * @param output (Optional) The single band output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt32 average( InterleavedS32 input , ImageSInt32 output ) {
		if (output == null) {
			output = new ImageSInt32(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ConvertInterleavedToSingle.average(input, output);

		return output;
	}

	/**
	 * Converts a {@link InterleavedS32} into the equivalent {@link MultiSpectral}
	 * 
	 * @param input (Input) ImageInterleaved that is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static MultiSpectral<ImageSInt32> convert( InterleavedS32 input , MultiSpectral<ImageSInt32> output ) {
		if (output == null) {
			output = new MultiSpectral<ImageSInt32>(ImageSInt32.class,input.width, input.height,input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input,output);

		return output;
	}

	/**
	 * Converts pixel values in the input image into an integer values from 0 to numValues. 
	 * @param input Input image
	 * @param min minimum input pixel value, inclusive
	 * @param max maximum input pixel value, inclusive
	 * @param numValues Number of possible pixel values in output image
	 * @param output (Optional) Storage for the output image.  Can be null.
	 * @return The converted output image.
	 */
	public static ImageUInt8 convert(ImageSInt32 input , int min , int max , int numValues , ImageUInt8 output )
	{
		if (output == null) {
			output = new ImageUInt8(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}
		if( numValues < 0 || numValues > 256 )
			throw new IllegalArgumentException("0 <= numValues <= 256");

		numValues -= 1;
		int range = max-min;

		for (int y = 0; y < input.height; y++) {
			int indexIn = input.startIndex + y*input.stride;
			int indexOut = output.startIndex + y*output.stride;

			for (int x = 0; x < input.width; x++) {
				int value = (int)(numValues*((input.data[indexIn++])-min)/range );
				output.data[indexOut++] = (byte)value;
			}
		}
	return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageSInt64} into a {@link boofcv.struct.image.ImageUInt8}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageUInt8 convert(ImageSInt64 input, ImageUInt8 output) {
		if (output == null) {
			output = new ImageUInt8(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedS64} into a {@link boofcv.struct.image.InterleavedU8}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedU8 convert(InterleavedS64 input, InterleavedU8 output) {
		if (output == null) {
			output = new InterleavedU8(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageSInt64} into a {@link boofcv.struct.image.ImageSInt8}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt8 convert(ImageSInt64 input, ImageSInt8 output) {
		if (output == null) {
			output = new ImageSInt8(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedS64} into a {@link boofcv.struct.image.InterleavedS8}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS8 convert(InterleavedS64 input, InterleavedS8 output) {
		if (output == null) {
			output = new InterleavedS8(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageSInt64} into a {@link boofcv.struct.image.ImageUInt16}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageUInt16 convert(ImageSInt64 input, ImageUInt16 output) {
		if (output == null) {
			output = new ImageUInt16(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedS64} into a {@link boofcv.struct.image.InterleavedU16}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedU16 convert(InterleavedS64 input, InterleavedU16 output) {
		if (output == null) {
			output = new InterleavedU16(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageSInt64} into a {@link boofcv.struct.image.ImageSInt16}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt16 convert(ImageSInt64 input, ImageSInt16 output) {
		if (output == null) {
			output = new ImageSInt16(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedS64} into a {@link boofcv.struct.image.InterleavedS16}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS16 convert(InterleavedS64 input, InterleavedS16 output) {
		if (output == null) {
			output = new InterleavedS16(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageSInt64} into a {@link boofcv.struct.image.ImageSInt32}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt32 convert(ImageSInt64 input, ImageSInt32 output) {
		if (output == null) {
			output = new ImageSInt32(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedS64} into a {@link boofcv.struct.image.InterleavedS32}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS32 convert(InterleavedS64 input, InterleavedS32 output) {
		if (output == null) {
			output = new InterleavedS32(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageSInt64} into a {@link boofcv.struct.image.ImageFloat32}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageFloat32 convert(ImageSInt64 input, ImageFloat32 output) {
		if (output == null) {
			output = new ImageFloat32(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedS64} into a {@link boofcv.struct.image.InterleavedF32}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedF32 convert(InterleavedS64 input, InterleavedF32 output) {
		if (output == null) {
			output = new InterleavedF32(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageSInt64} into a {@link boofcv.struct.image.ImageFloat64}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageFloat64 convert(ImageSInt64 input, ImageFloat64 output) {
		if (output == null) {
			output = new ImageFloat64(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedS64} into a {@link boofcv.struct.image.InterleavedF64}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedF64 convert(InterleavedS64 input, InterleavedF64 output) {
		if (output == null) {
			output = new InterleavedF64(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * Converts a {@link MultiSpectral} into a {@link ImageSingleBand} by computing the average value of each pixel
	 * across all the bands.
	 * 
	 * @param input Input MultiSpectral image that is being converted. Not modified.
	 * @param output (Optional) The single band output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt64 average( MultiSpectral<ImageSInt64> input , ImageSInt64 output ) {
		if (output == null) {
			output = new ImageSInt64(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertMsToSingle.average(input, output);

		return output;
	}

	/**
	 * Converts a {@link MultiSpectral} into the equivalent {@link InterleavedS64}
	 *
	 * @param input (Input) MultiSpectral image that is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS64 convert( MultiSpectral<ImageSInt64> input , InterleavedS64 output ) {
		if (output == null) {
			output = new InterleavedS64(input.width, input.height,input.getNumBands());
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input,output);

		return output;
	}

	/**
	 * Converts a {@link InterleavedS64} into a {@link ImageSInt64} by computing the average value of each pixel
	 * across all the bands.
	 * 
	 * @param input (Input) The ImageInterleaved that is being converted. Not modified.
	 * @param output (Optional) The single band output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt64 average( InterleavedS64 input , ImageSInt64 output ) {
		if (output == null) {
			output = new ImageSInt64(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ConvertInterleavedToSingle.average(input, output);

		return output;
	}

	/**
	 * Converts a {@link InterleavedS64} into the equivalent {@link MultiSpectral}
	 * 
	 * @param input (Input) ImageInterleaved that is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static MultiSpectral<ImageSInt64> convert( InterleavedS64 input , MultiSpectral<ImageSInt64> output ) {
		if (output == null) {
			output = new MultiSpectral<ImageSInt64>(ImageSInt64.class,input.width, input.height,input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input,output);

		return output;
	}

	/**
	 * Converts pixel values in the input image into an integer values from 0 to numValues. 
	 * @param input Input image
	 * @param min minimum input pixel value, inclusive
	 * @param max maximum input pixel value, inclusive
	 * @param numValues Number of possible pixel values in output image
	 * @param output (Optional) Storage for the output image.  Can be null.
	 * @return The converted output image.
	 */
	public static ImageUInt8 convert(ImageSInt64 input , long min , long max , int numValues , ImageUInt8 output )
	{
		if (output == null) {
			output = new ImageUInt8(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}
		if( numValues < 0 || numValues > 256 )
			throw new IllegalArgumentException("0 <= numValues <= 256");

		numValues -= 1;
		long range = max-min;

		for (int y = 0; y < input.height; y++) {
			int indexIn = input.startIndex + y*input.stride;
			int indexOut = output.startIndex + y*output.stride;

			for (int x = 0; x < input.width; x++) {
				int value = (int)(numValues*((input.data[indexIn++])-min)/range );
				output.data[indexOut++] = (byte)value;
			}
		}
	return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageFloat32} into a {@link boofcv.struct.image.ImageUInt8}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageUInt8 convert(ImageFloat32 input, ImageUInt8 output) {
		if (output == null) {
			output = new ImageUInt8(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedF32} into a {@link boofcv.struct.image.InterleavedU8}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedU8 convert(InterleavedF32 input, InterleavedU8 output) {
		if (output == null) {
			output = new InterleavedU8(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageFloat32} into a {@link boofcv.struct.image.ImageSInt8}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt8 convert(ImageFloat32 input, ImageSInt8 output) {
		if (output == null) {
			output = new ImageSInt8(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedF32} into a {@link boofcv.struct.image.InterleavedS8}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS8 convert(InterleavedF32 input, InterleavedS8 output) {
		if (output == null) {
			output = new InterleavedS8(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageFloat32} into a {@link boofcv.struct.image.ImageUInt16}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageUInt16 convert(ImageFloat32 input, ImageUInt16 output) {
		if (output == null) {
			output = new ImageUInt16(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedF32} into a {@link boofcv.struct.image.InterleavedU16}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedU16 convert(InterleavedF32 input, InterleavedU16 output) {
		if (output == null) {
			output = new InterleavedU16(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageFloat32} into a {@link boofcv.struct.image.ImageSInt16}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt16 convert(ImageFloat32 input, ImageSInt16 output) {
		if (output == null) {
			output = new ImageSInt16(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedF32} into a {@link boofcv.struct.image.InterleavedS16}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS16 convert(InterleavedF32 input, InterleavedS16 output) {
		if (output == null) {
			output = new InterleavedS16(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageFloat32} into a {@link boofcv.struct.image.ImageSInt32}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt32 convert(ImageFloat32 input, ImageSInt32 output) {
		if (output == null) {
			output = new ImageSInt32(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedF32} into a {@link boofcv.struct.image.InterleavedS32}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS32 convert(InterleavedF32 input, InterleavedS32 output) {
		if (output == null) {
			output = new InterleavedS32(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageFloat32} into a {@link boofcv.struct.image.ImageSInt64}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt64 convert(ImageFloat32 input, ImageSInt64 output) {
		if (output == null) {
			output = new ImageSInt64(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedF32} into a {@link boofcv.struct.image.InterleavedS64}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS64 convert(InterleavedF32 input, InterleavedS64 output) {
		if (output == null) {
			output = new InterleavedS64(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageFloat32} into a {@link boofcv.struct.image.ImageFloat64}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageFloat64 convert(ImageFloat32 input, ImageFloat64 output) {
		if (output == null) {
			output = new ImageFloat64(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedF32} into a {@link boofcv.struct.image.InterleavedF64}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedF64 convert(InterleavedF32 input, InterleavedF64 output) {
		if (output == null) {
			output = new InterleavedF64(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * Converts a {@link MultiSpectral} into a {@link ImageSingleBand} by computing the average value of each pixel
	 * across all the bands.
	 * 
	 * @param input Input MultiSpectral image that is being converted. Not modified.
	 * @param output (Optional) The single band output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageFloat32 average( MultiSpectral<ImageFloat32> input , ImageFloat32 output ) {
		if (output == null) {
			output = new ImageFloat32(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertMsToSingle.average(input, output);

		return output;
	}

	/**
	 * Converts a {@link MultiSpectral} into the equivalent {@link InterleavedF32}
	 *
	 * @param input (Input) MultiSpectral image that is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedF32 convert( MultiSpectral<ImageFloat32> input , InterleavedF32 output ) {
		if (output == null) {
			output = new InterleavedF32(input.width, input.height,input.getNumBands());
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input,output);

		return output;
	}

	/**
	 * Converts a {@link InterleavedF32} into a {@link ImageFloat32} by computing the average value of each pixel
	 * across all the bands.
	 * 
	 * @param input (Input) The ImageInterleaved that is being converted. Not modified.
	 * @param output (Optional) The single band output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageFloat32 average( InterleavedF32 input , ImageFloat32 output ) {
		if (output == null) {
			output = new ImageFloat32(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ConvertInterleavedToSingle.average(input, output);

		return output;
	}

	/**
	 * Converts a {@link InterleavedF32} into the equivalent {@link MultiSpectral}
	 * 
	 * @param input (Input) ImageInterleaved that is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static MultiSpectral<ImageFloat32> convert( InterleavedF32 input , MultiSpectral<ImageFloat32> output ) {
		if (output == null) {
			output = new MultiSpectral<ImageFloat32>(ImageFloat32.class,input.width, input.height,input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input,output);

		return output;
	}

	/**
	 * Converts pixel values in the input image into an integer values from 0 to numValues. 
	 * @param input Input image
	 * @param min minimum input pixel value, inclusive
	 * @param max maximum input pixel value, inclusive
	 * @param numValues Number of possible pixel values in output image
	 * @param output (Optional) Storage for the output image.  Can be null.
	 * @return The converted output image.
	 */
	public static ImageUInt8 convert(ImageFloat32 input , float min , float max , int numValues , ImageUInt8 output )
	{
		if (output == null) {
			output = new ImageUInt8(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}
		if( numValues < 0 || numValues > 256 )
			throw new IllegalArgumentException("0 <= numValues <= 256");

		numValues -= 1;
		float range = max-min;

		for (int y = 0; y < input.height; y++) {
			int indexIn = input.startIndex + y*input.stride;
			int indexOut = output.startIndex + y*output.stride;

			for (int x = 0; x < input.width; x++) {
				int value = (int)(numValues*((input.data[indexIn++])-min)/range + 0.5f);
				output.data[indexOut++] = (byte)value;
			}
		}
	return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageFloat64} into a {@link boofcv.struct.image.ImageUInt8}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageUInt8 convert(ImageFloat64 input, ImageUInt8 output) {
		if (output == null) {
			output = new ImageUInt8(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedF64} into a {@link boofcv.struct.image.InterleavedU8}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedU8 convert(InterleavedF64 input, InterleavedU8 output) {
		if (output == null) {
			output = new InterleavedU8(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageFloat64} into a {@link boofcv.struct.image.ImageSInt8}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt8 convert(ImageFloat64 input, ImageSInt8 output) {
		if (output == null) {
			output = new ImageSInt8(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedF64} into a {@link boofcv.struct.image.InterleavedS8}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS8 convert(InterleavedF64 input, InterleavedS8 output) {
		if (output == null) {
			output = new InterleavedS8(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageFloat64} into a {@link boofcv.struct.image.ImageUInt16}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageUInt16 convert(ImageFloat64 input, ImageUInt16 output) {
		if (output == null) {
			output = new ImageUInt16(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedF64} into a {@link boofcv.struct.image.InterleavedU16}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedU16 convert(InterleavedF64 input, InterleavedU16 output) {
		if (output == null) {
			output = new InterleavedU16(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageFloat64} into a {@link boofcv.struct.image.ImageSInt16}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt16 convert(ImageFloat64 input, ImageSInt16 output) {
		if (output == null) {
			output = new ImageSInt16(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedF64} into a {@link boofcv.struct.image.InterleavedS16}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS16 convert(InterleavedF64 input, InterleavedS16 output) {
		if (output == null) {
			output = new InterleavedS16(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageFloat64} into a {@link boofcv.struct.image.ImageSInt32}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt32 convert(ImageFloat64 input, ImageSInt32 output) {
		if (output == null) {
			output = new ImageSInt32(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedF64} into a {@link boofcv.struct.image.InterleavedS32}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS32 convert(InterleavedF64 input, InterleavedS32 output) {
		if (output == null) {
			output = new InterleavedS32(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageFloat64} into a {@link boofcv.struct.image.ImageSInt64}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageSInt64 convert(ImageFloat64 input, ImageSInt64 output) {
		if (output == null) {
			output = new ImageSInt64(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedF64} into a {@link boofcv.struct.image.InterleavedS64}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedS64 convert(InterleavedF64 input, InterleavedS64 output) {
		if (output == null) {
			output = new InterleavedS64(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.ImageFloat64} into a {@link boofcv.struct.image.ImageFloat32}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageFloat32 convert(ImageFloat64 input, ImageFloat32 output) {
		if (output == null) {
			output = new ImageFloat32(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * <p>
	 * Converts an {@link boofcv.struct.image.InterleavedF64} into a {@link boofcv.struct.image.InterleavedF32}.
	 * </p>
	 *
	 * @param input Input image which is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedF32 convert(InterleavedF64 input, InterleavedF32 output) {
		if (output == null) {
			output = new InterleavedF32(input.width, input.height, input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input, output);

		return output;
	}

	/**
	 * Converts a {@link MultiSpectral} into a {@link ImageSingleBand} by computing the average value of each pixel
	 * across all the bands.
	 * 
	 * @param input Input MultiSpectral image that is being converted. Not modified.
	 * @param output (Optional) The single band output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageFloat64 average( MultiSpectral<ImageFloat64> input , ImageFloat64 output ) {
		if (output == null) {
			output = new ImageFloat64(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertMsToSingle.average(input, output);

		return output;
	}

	/**
	 * Converts a {@link MultiSpectral} into the equivalent {@link InterleavedF64}
	 *
	 * @param input (Input) MultiSpectral image that is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static InterleavedF64 convert( MultiSpectral<ImageFloat64> input , InterleavedF64 output ) {
		if (output == null) {
			output = new InterleavedF64(input.width, input.height,input.getNumBands());
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input,output);

		return output;
	}

	/**
	 * Converts a {@link InterleavedF64} into a {@link ImageFloat64} by computing the average value of each pixel
	 * across all the bands.
	 * 
	 * @param input (Input) The ImageInterleaved that is being converted. Not modified.
	 * @param output (Optional) The single band output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static ImageFloat64 average( InterleavedF64 input , ImageFloat64 output ) {
		if (output == null) {
			output = new ImageFloat64(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ConvertInterleavedToSingle.average(input, output);

		return output;
	}

	/**
	 * Converts a {@link InterleavedF64} into the equivalent {@link MultiSpectral}
	 * 
	 * @param input (Input) ImageInterleaved that is being converted. Not modified.
	 * @param output (Optional) The output image.  If null a new image is created. Modified.
	 * @return Converted image.
	 */
	public static MultiSpectral<ImageFloat64> convert( InterleavedF64 input , MultiSpectral<ImageFloat64> output ) {
		if (output == null) {
			output = new MultiSpectral<ImageFloat64>(ImageFloat64.class,input.width, input.height,input.numBands);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}

		ImplConvertImage.convert(input,output);

		return output;
	}

	/**
	 * Converts pixel values in the input image into an integer values from 0 to numValues. 
	 * @param input Input image
	 * @param min minimum input pixel value, inclusive
	 * @param max maximum input pixel value, inclusive
	 * @param numValues Number of possible pixel values in output image
	 * @param output (Optional) Storage for the output image.  Can be null.
	 * @return The converted output image.
	 */
	public static ImageUInt8 convert(ImageFloat64 input , double min , double max , int numValues , ImageUInt8 output )
	{
		if (output == null) {
			output = new ImageUInt8(input.width, input.height);
		} else {
			InputSanityCheck.checkSameShape(input, output);
		}
		if( numValues < 0 || numValues > 256 )
			throw new IllegalArgumentException("0 <= numValues <= 256");

		numValues -= 1;
		double range = max-min;

		for (int y = 0; y < input.height; y++) {
			int indexIn = input.startIndex + y*input.stride;
			int indexOut = output.startIndex + y*output.stride;

			for (int x = 0; x < input.width; x++) {
				int value = (int)(numValues*((input.data[indexIn++])-min)/range + 0.5);
				output.data[indexOut++] = (byte)value;
			}
		}
	return output;
	}


}
