package org.bsc.markdown;

import lombok.NonNull;
import org.bsc.confluence.FileExtension;
import org.bsc.confluence.model.Site;

import java.net.MalformedURLException;
import java.net.URL;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Optional;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static java.lang.String.format;
import static java.lang.String.valueOf;
import static java.util.Optional.ofNullable;

public class MarkdownVisitorHelper {

    /**
     *
     * @param uri
     * @return
     */
    private static Optional<String> getFileName(String uri ) {

        try {
            final java.net.URI uriObject = java.net.URI.create(uri);

            final String scheme = uriObject.getScheme();
            if (scheme != null) {

                switch (scheme.toLowerCase()) {
                    case "classpath":
                        return Optional.empty();
                    case "http":
                    case "https":
                        return ofNullable(uri);
                }
            }

            final Path path = Paths.get(uriObject.getPath());

            return ofNullable(path.getFileName().toString());

        }
        catch( Throwable e ) {
            return Optional.empty();
        }

    }

    public static boolean isURL( String url ) {
        try {
            new URL(url);
        } catch (MalformedURLException e) {
            // ignore exception;
            return false;
        }
        return true;
    }

    private static Pattern patternUri = Pattern.compile("(?:(\\$\\{.+\\})\\^)?(.+)");

    /**
     *
     * @param url
     * @return
     */
    public static  String processImageUrl( String url, MarkdownParserContext context ) {

        if( isURL(url) ) {
            return url;
        }

        final Matcher m = patternUri.matcher(url);

        if( !m.matches() ) {
            return url;
        }

        if( m.group(1) != null ) { // the uri contains explictly a macro : ${ ... }
            return url;
        }

        return getFileName(m.group(2))
                .map( fileName -> (context.isLinkPrefixEnabled()) ? "${page.title}^".concat(fileName) : fileName )
                .orElse(url);


    }

    public static String processLinkUrl( String url, MarkdownParserContext parseContext ) {

        // GUARD CONDITION
        if( !parseContext.getPage().isPresent() || isURL(url) || !FileExtension.MARKDOWN.isExentionOf(url) )
            return url;

        final Predicate<Site.Page> comparePath = ( p ) -> {

            final Path parentPath = Paths.get(parseContext.getPage().get().getUri()).getParent();

            final Path relativePath = parentPath.relativize( Paths.get(p.getUri()));

            final boolean result =  relativePath.equals( Paths.get(url) );

            return result;
        };


        return parseContext.getSite()
                .flatMap( site -> site.getHome().findPage( comparePath ) )
                .map( page -> parseContext.getPagePrefixToApply()
                        .map( prefixToApply -> prefixToApply.concat(" - "))
                        .filter( prefixToApply -> !url.startsWith(prefixToApply) ) // check prefix already applied
                        .filter( prefixToApply -> !page.getName().startsWith(prefixToApply) ) // check prefix already applied
                        .map( prefixToApply -> prefixToApply.concat( page.getName() ) )
                        .orElse( page.getName() ) )
                .orElse(url)
                ;

    }

//    public enum SkipEscapeMarkdownText {
//
//        TOC( "^\\{[Tt][Oo][Cc](([:]\\w+=\\w+)([|].+)*)?\\}$" ),
//        CHILDREN( "^\\{[Cc]hildren(([:]\\w+=\\w+)([|].+)*)?\\}$" )
//        ;
//
//        private final Pattern patternToSkip;
//
//        public boolean matches( String text ) {
//            return this.patternToSkip.matcher(text).matches();
//        }
//
//        SkipEscapeMarkdownText( String patternToSkip ) {
//            this.patternToSkip = Pattern.compile(patternToSkip);
//        }
//
//    }

    /**
     * [Match multiline text using regular expression](https://stackoverflow.com/a/3652392/521197)
     */
    private static Pattern isConfluenceMacroPattern = Pattern.compile( "^[\\s]*\\{([\\w-]+)(([:][\\w-]+(=(.+))?)([|].+)*)?\\}[\\s]*$", Pattern.DOTALL  );
    private static Pattern confluenceMacroWithContentPattern = Pattern.compile("^\\s*(\\{.+\\})(.+)(\\{.+\\})\\s*$", Pattern.DOTALL );
    private static Pattern isConfluenceVariablePattern = Pattern.compile( "^[\\s]*\\$\\{([\\w-\\.]+)\\}[\\s]*$" );

    /**
     *
     * @param text
     * @return
     */
    public static boolean isConfluenceMacroOrVariable( String text ) {
        // GUARD
        if( text == null || text.isEmpty() ) return false;
        return isConfluenceMacroPattern.matcher(text).matches() ||
            isConfluenceVariablePattern.matcher(text).matches();
    }
    /**
     *
     * @param text
     * @return
     */
    public static boolean isConfluenceMacro( String text ) {
        // GUARD
        if( text == null || text.isEmpty() ) return false;
        return  isConfluenceMacroPattern.matcher(text).matches();
    }

    public static boolean isConfluenceVariable( String text ) {
        // GUARD
        if( text == null || text.isEmpty() ) return false;
        return isConfluenceVariablePattern.matcher(text).matches();
    }

    /**
     *
     * @param text
     * @return
     */
    public static Matcher parseConfluenceMacro( @NonNull String text ) {
        return confluenceMacroWithContentPattern.matcher(text);
    }

    public enum EscapeTextEnum {
        LeftCurlyBrace( "(\\\\)?(\\{)" ),
        RightCurlyBrace("(\\\\)?(\\})"),
        LeftSquareBrace("(\\\\)?(\\[)"),
        RightSquareBrace("(\\\\)?(])");

        private final  String pattern;
        EscapeTextEnum( String pattern) {
            this.pattern = pattern;
        }

        public String replaceAll( String value)   {
            if( value == null || value.isEmpty() ) return value;

            final Matcher m = Pattern.compile(pattern).matcher(value);

            boolean result = m.find();
            if (result) {
                final StringBuffer sb = new StringBuffer();
                do {
                    m.appendReplacement(sb, " $2");
                    sb.setCharAt( sb.length() - 2, '\\');
                    result = m.find();
                } while (result);
                m.appendTail(sb);
                return sb.toString();
            }
            return value;
        }
    }

    /**
     * Escapes special Markdown characters in a string.
     *
     * @param text text The text to escape
     * @param firstEscape
     * @param nextEscapes
     * @return The escaped text
     */
    public static String escapeMarkdownText(String text, EscapeTextEnum firstEscape, EscapeTextEnum... nextEscapes ) {
        String result = firstEscape.replaceAll(text);
        for (EscapeTextEnum escapeTextEnum : nextEscapes) {
            result = escapeTextEnum.replaceAll(result);
        }
        return result;
    }

}
