// Copyright 2017 Google Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#ifndef CPU_FEATURES_INCLUDE_CPUINFO_ARM_H_
#define CPU_FEATURES_INCLUDE_CPUINFO_ARM_H_

#include <stdint.h>  // uint32_t
#include "cpu_features_macros.h"

CPU_FEATURES_START_CPP_NAMESPACE

typedef struct {
  int vfp : 1;       // Vector Floating Point.
  int iwmmxt : 1;    // Intel Wireless MMX Technology.
  int neon : 1;      // Advanced SIMD.
  int vfpv3 : 1;     // VFP version 3
  int vfpv3d16 : 1;  // VFP version 3 with 16 D-registers
  int vfpv4 : 1;     // VFP version 4 with fast context switching
  int idiva : 1;     // SDIV and UDIV hardware division in ARM mode.
  int idivt : 1;     // SDIV and UDIV hardware division in Thumb mode.
  int aes : 1;       // Hardware-accelerated Advanced Encryption Standard.
  int pmull : 1;     // Polynomial multiply long.
  int sha1 : 1;      // Hardware-accelerated SHA1.
  int sha2 : 1;      // Hardware-accelerated SHA2-256.
  int crc32 : 1;     // Hardware-accelerated CRC-32.

  // Make sure to update ArmFeaturesEnum below if you add a field here.
} ArmFeatures;

typedef struct {
  ArmFeatures features;
  int implementer;
  int architecture;
  int variant;
  int part;
  int revision;
} ArmInfo;

// TODO(user): Add macros to know which features are present at compile
// time.

ArmInfo GetArmInfo(void);

// Compute CpuId from ArmInfo.
uint32_t GetArmCpuId(const ArmInfo* const info);

////////////////////////////////////////////////////////////////////////////////
// Introspection functions

typedef enum {
  ARM_VFP,
  ARM_IWMMXT,
  ARM_NEON,
  ARM_VFPV3,
  ARM_VFPV3D16,
  ARM_VFPV4,
  ARM_IDIVA,
  ARM_IDIVT,
  ARM_AES,
  ARM_PMULL,
  ARM_SHA1,
  ARM_SHA2,
  ARM_CRC32,
  ARM_LAST_,
} ArmFeaturesEnum;

int GetArmFeaturesEnumValue(const ArmFeatures* features, ArmFeaturesEnum value);

const char* GetArmFeaturesEnumName(ArmFeaturesEnum);

#if !defined(CPU_FEATURES_ARCH_ARM)
ArmInfo GetArmInfo(void) { return ArmInfo(); }
uint32_t GetArmCpuId(const ArmInfo* const info) { return 0; }
int GetArmFeaturesEnumValue(const ArmFeatures* features, ArmFeaturesEnum value) { return 0; }
const char* GetArmFeaturesEnumName(ArmFeaturesEnum) { return NULL; }
#endif

CPU_FEATURES_END_CPP_NAMESPACE

#endif  // CPU_FEATURES_INCLUDE_CPUINFO_ARM_H_
