package org.cafienne.infrastructure.akkahttp.authentication

import com.typesafe.scalalogging.LazyLogging
import org.cafienne.actormodel.identity.{PlatformUser, UserIdentity}
import org.cafienne.actormodel.response.ActorLastModified
import org.cafienne.cmmn.repository.file.SimpleLRUCache
import org.cafienne.infrastructure.Cafienne
import org.cafienne.querydb.materializer.tenant.TenantReader
import org.cafienne.querydb.query.{TenantQueriesImpl, UserQueries}
import org.cafienne.querydb.record.TenantRecord

import scala.concurrent.{ExecutionContext, Future}

class IdentityCache(implicit val ec: ExecutionContext) extends IdentityProvider with LazyLogging {
  val userQueries: UserQueries = new TenantQueriesImpl

  // TODO: this should be a most recently used cache
  // TODO: check for multithreading issues now that event materializer can clear.
  private val cache = new SimpleLRUCache[String, PlatformUser](Cafienne.config.api.security.identityCacheSize)
  private val tenantCache = new SimpleLRUCache[String, TenantRecord](Cafienne.config.api.security.identityCacheSize)

  override def getPlatformUser(user: UserIdentity, tlm: Option[String]): Future[PlatformUser] = {
    tlm match {
      case Some(s) =>
        // Wait for the TenantReader to be informed about the tenant-last-modified timestamp
        for {
          p <- TenantReader.lastModifiedRegistration.waitFor(new ActorLastModified(s)).future
          u <- executeUserQuery(user)
        } yield (p, u)._2
      // Nothing to wait for, just continue and execute the query straight on
      case None => executeUserQuery(user)
    }
  }

  private def cacheUser(user: PlatformUser) = {
    cache.put(user.id, user)
    user
  }

  private def executeUserQuery(user: UserIdentity): Future[PlatformUser] = {
    cache.get(user.id) match {
      case user: PlatformUser => Future(user)
      case null => userQueries.getPlatformUser(user.id).map(cacheUser)
    }
  }

  override def getTenant(tenantId: String): Future[TenantRecord] = {
    tenantCache.get(tenantId) match {
      case tenant: TenantRecord => Future(tenant)
      case null => userQueries.getTenant(tenantId).map(tenant => {
        tenantCache.put(tenantId, tenant)
        tenant
      })
    }
  }

  override def clear(userId: String): Unit = {
    // NOTE: We can also extend this to update the cache information, instead of removing keys.
    cache.remove(userId)
  }

}
