/*
 * Copyright (C) 2013 The Calrissian Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.calrissian.accumulorecipes.lastn.iterator;

import org.apache.accumulo.core.client.IteratorSetting;
import org.apache.accumulo.core.data.ByteSequence;
import org.apache.accumulo.core.data.Key;
import org.apache.accumulo.core.data.Range;
import org.apache.accumulo.core.data.Value;
import org.apache.accumulo.core.iterators.IteratorEnvironment;
import org.apache.accumulo.core.iterators.SortedKeyValueIterator;
import org.apache.accumulo.core.iterators.WrappingIterator;
import org.apache.hadoop.io.Text;
import org.calrissian.accumulorecipes.commons.hadoop.EventWritable;
import org.calrissian.mango.domain.Tuple;
import org.calrissian.mango.domain.event.BaseEvent;
import org.calrissian.mango.domain.event.Event;
import org.calrissian.mango.io.Serializables;
import org.calrissian.mango.types.TypeRegistry;

import java.io.IOException;
import java.util.*;

import static com.google.common.base.Preconditions.checkNotNull;
import static org.calrissian.accumulorecipes.commons.support.Constants.NULL_BYTE;
import static org.calrissian.accumulorecipes.commons.support.Constants.END_BYTE;
import static org.calrissian.accumulorecipes.commons.support.WritableUtils2.serialize;
import static org.calrissian.accumulorecipes.commons.support.tuple.Metadata.Visiblity.setVisibility;

/**
 * An iterator to return StoreEntry objects serialized to JSON so that grouping can be done server side instead of
 * client side.
 */
public class EntryIterator extends WrappingIterator {

    public static final String TYPE_REGISTRY = "typeRegistry";

    private TypeRegistry<String> typeRegistry;
    private SortedKeyValueIterator<Key, Value> sourceItr;
    private EventWritable writable;

    public void init(SortedKeyValueIterator<Key, Value> source, java.util.Map<String, String> options,
                     IteratorEnvironment env) throws IOException {

        super.init(source, options, env);
        sourceItr = source.deepCopy(env);
        this.typeRegistry = getTypeRegistry(options);
        this.writable = new EventWritable();
    }

    public static void setTypeRegistry(IteratorSetting setting, TypeRegistry<String> registry) {
        checkNotNull(registry);
        try {
            setting.addOption(TYPE_REGISTRY, new String(Serializables.toBase64(registry)));
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private TypeRegistry<String> getTypeRegistry(Map<String,String> options) {
        if(options.containsKey(TYPE_REGISTRY)) {
            try {
                return Serializables.fromBase64(options.get(TYPE_REGISTRY).getBytes());
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        }

        else
            throw new RuntimeException("Error: Type registry was not configured on iterator.");
    }

    /**
     * For each index row in the lastN store, grab the associated getTuples (further down in the tablet) and construct
     * the entry to be returned.
     *
     * @return
     */
    @Override
    public Value getTopValue() {

        if (hasTop()) {

            Key topKey = getTopKey();
            Value topVal = super.getTopValue();
            String entryId = new String(topVal.get());

            Key startRangeKey = new Key(topKey.getRow(), new Text(END_BYTE + entryId));
            Key stopRangeKey = new Key(topKey.getRow(), new Text(END_BYTE + entryId + END_BYTE));

            Range range = new Range(startRangeKey, stopRangeKey);

            long timestamp = 0;

            try {
                sourceItr.seek(range, Collections.<ByteSequence>emptyList(), false);

                Collection<Tuple> tuples = new ArrayList<Tuple>();
                while (sourceItr.hasTop()) {

                    Key nextKey = sourceItr.getTopKey();
                    sourceItr.next();

                    if (!nextKey.getColumnFamily().toString().endsWith(entryId)) {
                        break;
                    }

                    String[] keyValueDatatype = nextKey.getColumnQualifier().toString().split(NULL_BYTE);

                    if (keyValueDatatype.length == 3) {

                        String vis = nextKey.getColumnVisibility().toString();

                        Tuple tuple = new Tuple(
                                keyValueDatatype[0],
                                typeRegistry.decode(keyValueDatatype[2], keyValueDatatype[1]),
                                setVisibility(new HashMap<String, Object>(1), vis)
                        );


                        tuples.add(tuple);
                        timestamp = nextKey.getTimestamp();
                    }
                }

                Event entry = new BaseEvent(entryId, timestamp);
                writable.set(entry);

                if (tuples.size() > 0)
                    entry.putAll(tuples);

                return new Value(serialize(writable));

            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        }

        return new Value("".getBytes());
    }


}
