/* ******************************************************************** */
/*                                                                      */
/*  Scenario                                                            */
/*                                                                      */
/*  Store a scenario. It is a list of order to execute                  */
/* ******************************************************************** */
package org.camunda.automator.definition;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import org.camunda.automator.configuration.ConfigurationBpmEngine;
import org.camunda.automator.engine.AutomatorException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;

/**
 * the Scenario Head group a scenario definition
 */
public class Scenario {
  static Logger logger = LoggerFactory.getLogger(Scenario.class);

  private final List<ScenarioExecution> executions = new ArrayList<>();
  private final List<ScenarioDeployment> deployments = new ArrayList<>();
  private final List<ScenarioStep> flows = new ArrayList<>();
  private ScenarioWarmingUp warmingUp;
  private ScenarioFlowControl flowControl;

  private String name;
  private String version;
  private String processName;
  private String processId;

  /**
   * Server to run the scenario
   */
  private String serverName;

  private String serverType;

  /**
   * This value is fulfill only if the scenario was read from a file
   */
  private String scenarioFile = null;

  public static Scenario createFromJson(String jsonContent) {
    GsonBuilder builder = new GsonBuilder();
    builder.setPrettyPrinting();

    Gson gson = builder.create();
    Scenario scnHead = gson.fromJson(jsonContent, Scenario.class);
    if (scnHead == null) {
      logger.error("Scenario: Can't build scenario from content [{}]", jsonContent);
      return null;
    }
    scnHead.afterUnSerialize();
    return scnHead;
  }

  public static Scenario createFromFile(File scenarioFile) throws AutomatorException {
    try {

      Scenario scenario = createFromInputStream(new FileInputStream(scenarioFile), scenarioFile.getAbsolutePath());
      scenario.scenarioFile = scenarioFile.getAbsolutePath();
      return scenario;

    } catch (FileNotFoundException e) {
      throw new AutomatorException("Can't access file [" + scenarioFile.getAbsolutePath() + "] " + e.getMessage());
    } catch (AutomatorException e) {
      throw e;
    }
  }

  /**
   * Load the scenario from a File
   *
   * @param scenarioInput InputStream to read
   * @return the scenario
   * @throws AutomatorException if file cannot be read or it's not a Json file
   */
  public static Scenario createFromInputStream(InputStream scenarioInput, String origin) throws AutomatorException {
    logger.info("Load Scenario [{}] from InputStream", origin);
    try (BufferedReader br = new BufferedReader(new InputStreamReader(scenarioInput))) {
      StringBuilder jsonContent = new StringBuilder();
      String st;
      while ((st = br.readLine()) != null)
        jsonContent.append(st);

      Scenario scnHead = createFromJson(jsonContent.toString());
      if (scnHead == null) {
        throw new AutomatorException("Scenario: can't load from JSON [" + jsonContent + "] ");

      }
      return scnHead;
    } catch (IOException e) {
      logger.error("CreateScenarioFromInputString: origin[{}] error {} : {} ", origin, e.getMessage(), e.toString());
      throw new AutomatorException("Can't load content from [" + origin + "] " + e.getMessage());
    }

  }

  /**
   * Add a new execution
   *
   * @return the scenario itself
   */
  public Scenario addExecution(ScenarioExecution scnExecution) {
    executions.add(scnExecution);
    return this;
  }

  public List<ScenarioExecution> getExecutions() {
    return executions;
  }

  public List<ScenarioStep> getFlows() {
    return flows;
  }

  public ScenarioWarmingUp getWarmingUp() {
    return warmingUp;
  }

  public ScenarioFlowControl getFlowControl() {
    return flowControl;
  }

  public List<ScenarioDeployment> getDeployments() {
    return deployments;
  }

  public String getName() {
    return name;
  }

  public Scenario setName(String name) {
    this.name = name;
    return this;
  }

  public String getVersion() {
    return version;
  }

  public String getProcessName() {
    return processName;
  }

  public String getProcessId() {
    return processId;
  }

  public Scenario setProcessId(String processId) {
    this.processId = processId;
    return this;
  }

  public File getScenarioFile() {
    try {
      return new File(scenarioFile);
    } catch (Exception e) {
      logger.error("Can't access file [{}] ", scenarioFile);
      return null;
    }
  }

  public String getServerName() {
    if (serverName == null || serverName.isEmpty())
      return null;
    return serverName;
  }

  public ConfigurationBpmEngine.CamundaEngine getServerType() {
    try {
      return ConfigurationBpmEngine.CamundaEngine.valueOf(serverType.toUpperCase());
    } catch (Exception e) {
      return null;
    }

  }

  private void afterUnSerialize() {
    // Attention, now we have to manually set the tree relation
    for (ScenarioExecution scnExecution : getExecutions()) {
      scnExecution.afterUnSerialize(this);
    }
  }

}
