package org.camunda.community.bpmndt.strategy;

import org.camunda.community.bpmndt.api.TestCaseInstanceElement.UserTaskElement;
import org.camunda.community.bpmndt.model.BpmnElement;
import org.camunda.community.bpmndt.model.BpmnElementType;
import org.camunda.community.bpmndt.model.BpmnEventSupport;

import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.TypeName;

import io.camunda.zeebe.model.bpmn.instance.BoundaryEvent;
import io.camunda.zeebe.model.bpmn.instance.zeebe.ZeebeAssignmentDefinition;
import io.camunda.zeebe.model.bpmn.instance.zeebe.ZeebeTaskSchedule;

public class UserTaskStrategy extends DefaultHandlerStrategy {

  public UserTaskStrategy(BpmnElement element) {
    super(element);
  }

  @Override
  public TypeName getHandlerType() {
    return USER_TASK;
  }

  @Override
  public void initHandler(MethodSpec.Builder methodBuilder) {
    super.initHandler(methodBuilder);

    if (!element.hasNext()) {
      return;
    }

    var next = element.getNext();
    if (!next.getType().isBoundaryEvent()) {
      return;
    }

    if (next.getType() == BpmnElementType.ERROR_BOUNDARY) {
      var event = next.getFlowNode(BoundaryEvent.class);
      var eventSupport = new BpmnEventSupport(event);

      var errorCode = eventSupport.getErrorCode();
      methodBuilder.addStatement("$L.throwBpmnError($S, $S)", literal, errorCode, "error message, generated by BPMNDT");
    } else if (next.getType() != BpmnElementType.ESCALATION_BOUNDARY) {
      methodBuilder.addStatement("$L.waitForBoundaryEvent()", literal);
    }
  }

  @Override
  public void initHandlerElement(MethodSpec.Builder methodBuilder) {
    methodBuilder.addCode("\n// $L: $L\n", element.getTypeName(), element.getId());
    methodBuilder.addStatement("$T $LElement = new $T()", UserTaskElement.class, literal, UserTaskElement.class);
    methodBuilder.addStatement("$LElement.id = $S", literal, element.getId());

    var extensionElements = element.getFlowNode().getExtensionElements();
    if (extensionElements == null) {
      return;
    }

    var assignmentDefinition = (ZeebeAssignmentDefinition) extensionElements.getUniqueChildElementByType(ZeebeAssignmentDefinition.class);
    if (assignmentDefinition != null) {
      if (assignmentDefinition.getAssignee() != null) {
        methodBuilder.addStatement("$LElement.assignee = $S", literal, assignmentDefinition.getAssignee());
      }
      if (assignmentDefinition.getCandidateGroups() != null) {
        methodBuilder.addStatement("$LElement.candidateGroups = $S", literal, assignmentDefinition.getCandidateGroups());
      }
      if (assignmentDefinition.getCandidateUsers() != null) {
        methodBuilder.addStatement("$LElement.candidateUsers = $S", literal, assignmentDefinition.getCandidateUsers());
      }
    }

    var taskSchedule = (ZeebeTaskSchedule) extensionElements.getUniqueChildElementByType(ZeebeTaskSchedule.class);
    if (taskSchedule != null) {
      if (taskSchedule.getDueDate() != null) {
        methodBuilder.addStatement("$LElement.dueDate = $S", literal, taskSchedule.getDueDate());
      }
      if (taskSchedule.getFollowUpDate() != null) {
        methodBuilder.addStatement("$LElement.followUpDate = $S", literal, taskSchedule.getFollowUpDate());
      }
    }
  }
}
