/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.web.util;

import java.net.IDN;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.function.Consumer;
import java.util.function.IntPredicate;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.lang.Nullable;
import org.springframework.util.Assert;
import org.springframework.web.util.InvalidUrlException;
import org.springframework.web.util.UriUtils;

final class WhatWgUrlParser {
    public static final UrlRecord EMPTY_RECORD = new UrlRecord();
    private static final Log logger = LogFactory.getLog(WhatWgUrlParser.class);
    private static final int EOF = -1;
    private static final int MAX_PORT = 65535;
    private final StringBuilder input;
    @Nullable
    private final UrlRecord base;
    @Nullable
    private Charset encoding;
    @Nullable
    private final Consumer<String> validationErrorHandler;
    private int pointer;
    private final StringBuilder buffer;
    @Nullable
    private State state;
    @Nullable
    private State stateOverride;
    private boolean atSignSeen;
    private boolean passwordTokenSeen;
    private boolean insideBrackets;
    private int openCurlyBracketCount;
    private boolean stopMainLoop = false;

    private WhatWgUrlParser(String input, @Nullable UrlRecord base, @Nullable Charset encoding, @Nullable Consumer<String> validationErrorHandler) {
        this.input = new StringBuilder(input);
        this.base = base;
        this.encoding = encoding;
        this.validationErrorHandler = validationErrorHandler;
        this.buffer = new StringBuilder(this.input.length() / 2);
    }

    public static UrlRecord parse(String input, @Nullable UrlRecord base, @Nullable Charset encoding, @Nullable Consumer<String> validationErrorHandler) throws InvalidUrlException {
        Assert.notNull((Object)input, "Input must not be null");
        WhatWgUrlParser parser = new WhatWgUrlParser(input, base, encoding, validationErrorHandler);
        return parser.basicUrlParser(null, null);
    }

    private UrlRecord basicUrlParser(@Nullable UrlRecord url, @Nullable State stateOverride) {
        if (url == null) {
            url = new UrlRecord();
            this.sanitizeInput(true);
        } else {
            this.sanitizeInput(false);
        }
        this.state = stateOverride != null ? stateOverride : State.SCHEME_START;
        this.stateOverride = stateOverride;
        while (!this.stopMainLoop && this.pointer <= this.input.length()) {
            int c = this.pointer < this.input.length() ? this.input.codePointAt(this.pointer) : -1;
            if (logger.isTraceEnabled()) {
                logger.trace("current: " + (c != -1 ? Character.toString(c) : "EOF") + " ptr: " + this.pointer + " Buffer: " + String.valueOf(this.buffer) + " State: " + String.valueOf((Object)this.state));
            }
            this.state.handle(c, url, this);
            ++this.pointer;
        }
        return url;
    }

    void sanitizeInput(boolean removeC0ControlOrSpace) {
        int c;
        int i;
        boolean strip = true;
        for (i = 0; i < this.input.length(); ++i) {
            boolean isTabOrNL;
            c = this.input.codePointAt(i);
            boolean isSpaceOrC0 = c == 32 || WhatWgUrlParser.isC0Control(c);
            boolean bl = isTabOrNL = c == 9 || WhatWgUrlParser.isNewline(c);
            if (strip && isSpaceOrC0 || isTabOrNL) {
                if (this.validate()) {
                    this.validationError("Code point \"" + c + "\" is not a URL unit.");
                }
                if (removeC0ControlOrSpace && isSpaceOrC0) {
                    this.input.deleteCharAt(i);
                } else if (isTabOrNL) {
                    this.input.deleteCharAt(i);
                }
                --i;
                continue;
            }
            strip = false;
        }
        if (removeC0ControlOrSpace) {
            for (i = this.input.length() - 1; i >= 0 && ((c = this.input.codePointAt(i)) == 32 || WhatWgUrlParser.isC0Control(c)); --i) {
                if (this.validate()) {
                    this.validationError("Code point \"" + c + "\" is not a URL unit.");
                }
                this.input.deleteCharAt(i);
            }
        }
    }

    private void setState(State newState) {
        if (logger.isTraceEnabled()) {
            String c = this.pointer < this.input.length() ? Character.toString(this.input.codePointAt(this.pointer)) : "EOF";
            logger.trace("Changing state from " + String.valueOf((Object)this.state) + " to " + String.valueOf((Object)newState) + " (cur: " + c + ")");
        }
        this.state = newState;
        this.openCurlyBracketCount = this.buffer.toString().equals("{") ? this.openCurlyBracketCount : 0;
    }

    private boolean processCurlyBrackets(int c) {
        if (c == 123) {
            ++this.openCurlyBracketCount;
            return true;
        }
        if (c == 125) {
            if (this.openCurlyBracketCount > 0) {
                --this.openCurlyBracketCount;
                return true;
            }
            return false;
        }
        return this.openCurlyBracketCount > 0 && c != -1;
    }

    private static LinkedList<String> strictSplit(String input, int delimiter) {
        int position = 0;
        LinkedList<String> tokens = new LinkedList<String>();
        int delIdx = input.indexOf(delimiter, position);
        String token = delIdx != -1 ? input.substring(position, delIdx) : input.substring(position);
        position = delIdx;
        tokens.add(token);
        while (position != -1) {
            Assert.state(input.codePointAt(position) == delimiter, "Codepoint is not a delimiter");
            delIdx = input.indexOf(delimiter, ++position);
            token = delIdx != -1 ? input.substring(position, delIdx) : input.substring(position);
            position = delIdx;
            tokens.add(token);
        }
        return tokens;
    }

    private static String domainToAscii(String domain, boolean beStrict) {
        if (!beStrict && WhatWgUrlParser.containsOnlyAscii(domain)) {
            int dotIdx = domain.indexOf(46);
            boolean onlyLowerCase = true;
            while (dotIdx != -1) {
                if (domain.length() - dotIdx > 4) {
                    int ch0 = domain.codePointAt(dotIdx + 1);
                    int ch1 = domain.codePointAt(dotIdx + 2);
                    int ch2 = domain.codePointAt(dotIdx + 3);
                    int ch3 = domain.codePointAt(dotIdx + 4);
                    if (!(ch0 != 120 && ch0 != 88 || ch1 != 110 && ch1 != 78 || ch2 != 45 || ch3 != 95)) {
                        onlyLowerCase = false;
                        break;
                    }
                }
                dotIdx = domain.indexOf(46, dotIdx + 1);
            }
            if (onlyLowerCase) {
                return domain.toLowerCase(Locale.ENGLISH);
            }
        }
        int flag = 0;
        if (beStrict) {
            flag |= 2;
        }
        try {
            return IDN.toASCII(domain, flag);
        }
        catch (IllegalArgumentException ex) {
            throw new InvalidUrlException("Could not convert \"" + domain + "\" to ASCII: " + ex.getMessage(), ex);
        }
    }

    private boolean validate() {
        return this.validationErrorHandler != null;
    }

    private void validationError(@Nullable String additionalInfo) {
        if (this.validationErrorHandler != null) {
            StringBuilder message = new StringBuilder("URL validation error for URL [");
            message.append((CharSequence)this.input);
            message.append("]@");
            message.append(this.pointer);
            if (additionalInfo != null) {
                message.append(". ");
                message.append(additionalInfo);
            }
            this.validationErrorHandler.accept(message.toString());
        }
    }

    private void failure(@Nullable String additionalInfo) {
        StringBuilder message = new StringBuilder("URL parsing failure for URL [");
        message.append((CharSequence)this.input);
        message.append("] @ ");
        message.append(this.pointer);
        if (additionalInfo != null) {
            message.append(". ");
            message.append(additionalInfo);
        }
        throw new InvalidUrlException(message.toString());
    }

    private static boolean c0ControlPercentEncodeSet(int ch) {
        return WhatWgUrlParser.isC0Control(ch) || Integer.compareUnsigned(ch, 126) > 0;
    }

    private static boolean fragmentPercentEncodeSet(int ch) {
        return WhatWgUrlParser.c0ControlPercentEncodeSet(ch) || ch == 32 || ch == 34 || ch == 60 || ch == 62 || ch == 96;
    }

    private static boolean queryPercentEncodeSet(int ch) {
        return WhatWgUrlParser.c0ControlPercentEncodeSet(ch) || ch == 32 || ch == 34 || ch == 35 || ch == 60 || ch == 62;
    }

    private static boolean specialQueryPercentEncodeSet(int ch) {
        return WhatWgUrlParser.queryPercentEncodeSet(ch) || ch == 39;
    }

    private static boolean pathPercentEncodeSet(int ch) {
        return WhatWgUrlParser.queryPercentEncodeSet(ch) || ch == 63 || ch == 96 || ch == 123 || ch == 125;
    }

    private static boolean userinfoPercentEncodeSet(int ch) {
        return WhatWgUrlParser.pathPercentEncodeSet(ch) || ch == 47 || ch == 58 || ch == 59 || ch == 61 || ch == 64 || Integer.compareUnsigned(ch, 91) >= 0 && Integer.compareUnsigned(ch, 94) <= 0 || ch == 124;
    }

    private static boolean isC0Control(int ch) {
        return ch >= 0 && ch <= 31;
    }

    private static boolean isNewline(int ch) {
        return ch == 13 || ch == 10;
    }

    private static boolean isAsciiAlpha(int ch) {
        return ch >= 65 && ch <= 90 || ch >= 97 && ch <= 122;
    }

    private static boolean containsOnlyAsciiDigits(CharSequence string) {
        for (int i = 0; i < string.length(); ++i) {
            int ch = WhatWgUrlParser.codePointAt(string, i);
            if (WhatWgUrlParser.isAsciiDigit(ch)) continue;
            return false;
        }
        return true;
    }

    private static boolean containsOnlyAscii(String string) {
        for (int i = 0; i < string.length(); ++i) {
            int ch = string.codePointAt(i);
            if (WhatWgUrlParser.isAsciiCodePoint(ch)) continue;
            return false;
        }
        return true;
    }

    private static boolean isAsciiCodePoint(int ch) {
        return Integer.compareUnsigned(ch, 0) >= 0 && Integer.compareUnsigned(ch, 127) <= 0;
    }

    private static boolean isAsciiDigit(int ch) {
        return ch >= 48 && ch <= 57;
    }

    private static boolean isAsciiAlphaNumeric(int ch) {
        return WhatWgUrlParser.isAsciiAlpha(ch) || WhatWgUrlParser.isAsciiDigit(ch);
    }

    private static boolean isAsciiHexDigit(int ch) {
        return WhatWgUrlParser.isAsciiDigit(ch) || ch >= 65 && ch <= 70 || ch >= 97 && ch <= 102;
    }

    private static boolean isForbiddenDomain(int ch) {
        return WhatWgUrlParser.isForbiddenHost(ch) || WhatWgUrlParser.isC0Control(ch) || ch == 37 || ch == 127;
    }

    private static boolean isForbiddenHost(int ch) {
        return ch == 0 || ch == 9 || WhatWgUrlParser.isNewline(ch) || ch == 32 || ch == 35 || ch == 47 || ch == 58 || ch == 60 || ch == 62 || ch == 63 || ch == 64 || ch == 91 || ch == 92 || ch == 93 || ch == 94 || ch == 124;
    }

    private static boolean isNonCharacter(int ch) {
        return ch >= 64976 && ch <= 65007 || ch == 65534 || ch == 65535 || ch == 131070 || ch == 131071 || ch == 196606 || ch == 196607 || ch == 262142 || ch == 262143 || ch == 327678 || ch == 327679 || ch == 393214 || ch == 393215 || ch == 458750 || ch == 458751 || ch == 524286 || ch == 524287 || ch == 589822 || ch == 589823 || ch == 655358 || ch == 655359 || ch == 720894 || ch == 720895 || ch == 786430 || ch == 786431 || ch == 851966 || ch == 851967 || ch == 917502 || ch == 917503 || ch == 983038 || ch == 983039 || ch == 1048574 || ch == 1048575 || ch == 1114110 || ch == 0x10FFFF;
    }

    private static boolean isUrlCodePoint(int ch) {
        return WhatWgUrlParser.isAsciiAlphaNumeric(ch) || ch == 33 || ch == 36 || ch == 38 || ch == 39 || ch == 40 || ch == 41 || ch == 42 || ch == 43 || ch == 44 || ch == 45 || ch == 46 || ch == 47 || ch == 58 || ch == 59 || ch == 61 || ch == 63 || ch == 64 || ch == 95 || ch == 126 || ch >= 160 && ch <= 1114109 && !Character.isSurrogate((char)ch) && !WhatWgUrlParser.isNonCharacter(ch);
    }

    private static boolean isSpecialScheme(String scheme) {
        return "ftp".equals(scheme) || "file".equals(scheme) || "http".equals(scheme) || "https".equals(scheme) || "ws".equals(scheme) || "wss".equals(scheme);
    }

    private static int defaultPort(@Nullable String scheme) {
        if (scheme != null) {
            return switch (scheme) {
                case "ftp" -> 21;
                case "http", "ws" -> 80;
                case "https", "wss" -> 443;
                default -> -1;
            };
        }
        return -1;
    }

    private void append(String s) {
        this.buffer.append(s);
    }

    private void append(char ch) {
        this.buffer.append(ch);
    }

    private void append(int ch) {
        this.buffer.appendCodePoint(ch);
    }

    private void prepend(String s) {
        this.buffer.insert(0, s);
    }

    private void emptyBuffer() {
        this.buffer.setLength(0);
    }

    private int remaining(int deltaPos) {
        int pos = this.pointer + deltaPos + 1;
        return pos < this.input.length() ? this.input.codePointAt(pos) : -1;
    }

    private static String percentDecode(String input) {
        try {
            return UriUtils.decode(input, StandardCharsets.UTF_8);
        }
        catch (IllegalArgumentException ex) {
            throw new InvalidUrlException("Could not decode \"" + input + "\": " + ex.getMessage(), ex);
        }
    }

    @Nullable
    private String percentEncode(int c, IntPredicate percentEncodeSet) {
        if (this.encoding == null) {
            return null;
        }
        return this.percentEncode(Character.toString(c), percentEncodeSet);
    }

    private String percentEncode(String input, IntPredicate percentEncodeSet) {
        if (this.encoding == null) {
            return input;
        }
        byte[] bytes = input.getBytes(this.encoding);
        boolean original = true;
        for (byte b : bytes) {
            if (!percentEncodeSet.test(b)) continue;
            original = false;
            break;
        }
        if (original) {
            return input;
        }
        StringBuilder output = new StringBuilder();
        for (byte b : bytes) {
            if (!percentEncodeSet.test(b)) {
                output.append((char)b);
                continue;
            }
            output.append('%');
            char hex1 = Character.toUpperCase(Character.forDigit(b >> 4 & 0xF, 16));
            char hex2 = Character.toUpperCase(Character.forDigit(b & 0xF, 16));
            output.append(hex1);
            output.append(hex2);
        }
        return output.toString();
    }

    private static boolean isSingleDotPathSegment(StringBuilder b) {
        int len = b.length();
        switch (len) {
            case 1: {
                int ch0 = b.codePointAt(0);
                return ch0 == 46;
            }
            case 2: {
                int ch0 = b.codePointAt(0);
                int ch1 = b.codePointAt(1);
                return ch0 == 47 && ch1 == 46;
            }
            case 3: {
                int ch0 = b.codePointAt(0);
                int ch1 = b.codePointAt(1);
                int ch2 = b.codePointAt(2);
                return ch0 == 37 && ch1 == 50 && (ch2 == 101 || ch2 == 69);
            }
            case 4: {
                int ch0 = b.codePointAt(0);
                int ch1 = b.codePointAt(1);
                int ch2 = b.codePointAt(2);
                int ch3 = b.codePointAt(3);
                return ch0 == 47 && ch1 == 37 && ch2 == 50 && (ch3 == 101 || ch3 == 69);
            }
        }
        return false;
    }

    private static boolean isDoubleDotPathSegment(StringBuilder b) {
        int len = b.length();
        switch (len) {
            case 2: {
                int ch0 = b.codePointAt(0);
                int ch1 = b.codePointAt(1);
                return ch0 == 46 && ch1 == 46;
            }
            case 3: {
                int ch0 = b.codePointAt(0);
                int ch1 = b.codePointAt(1);
                int ch2 = b.codePointAt(2);
                return ch0 == 47 && ch1 == 46 && ch2 == 46;
            }
            case 4: {
                int ch0 = b.codePointAt(0);
                int ch1 = b.codePointAt(1);
                int ch2 = b.codePointAt(2);
                int ch3 = b.codePointAt(3);
                return ch0 == 46 && ch1 == 37 && ch2 == 50 && (ch3 == 101 || ch3 == 69) || ch0 == 37 && ch1 == 50 && (ch2 == 101 || ch2 == 69) && ch3 == 46;
            }
            case 5: {
                int ch0 = b.codePointAt(0);
                int ch1 = b.codePointAt(1);
                int ch2 = b.codePointAt(2);
                int ch3 = b.codePointAt(3);
                int ch4 = b.codePointAt(4);
                return ch0 == 47 && (ch1 == 46 && ch2 == 37 && ch3 == 50 && (ch4 == 101 || ch4 == 69) || ch1 == 37 && ch2 == 50 && (ch3 == 101 || ch3 == 69) && ch4 == 46);
            }
            case 6: {
                int ch0 = b.codePointAt(0);
                int ch1 = b.codePointAt(1);
                int ch2 = b.codePointAt(2);
                int ch3 = b.codePointAt(3);
                int ch4 = b.codePointAt(4);
                int ch5 = b.codePointAt(5);
                return !(ch0 != 37 || ch1 != 50 || ch2 != 101 && ch2 != 69 || ch3 != 37 || ch4 != 50 || ch5 != 101 && ch5 != 69);
            }
            case 7: {
                int ch0 = b.codePointAt(0);
                int ch1 = b.codePointAt(1);
                int ch2 = b.codePointAt(2);
                int ch3 = b.codePointAt(3);
                int ch4 = b.codePointAt(4);
                int ch5 = b.codePointAt(5);
                int ch6 = b.codePointAt(6);
                return !(ch0 != 47 || ch1 != 37 || ch2 != 50 || ch3 != 101 && ch3 != 69 || ch4 != 37 || ch5 != 50 || ch6 != 101 && ch6 != 69);
            }
        }
        return false;
    }

    private static boolean isWindowsDriveLetter(CharSequence input, boolean normalized) {
        if (input.length() != 2) {
            return false;
        }
        return WhatWgUrlParser.isWindowsDriveLetterInternal(input, normalized);
    }

    private static boolean startsWithWindowsDriveLetter(String input) {
        int len = input.length();
        if (len < 2) {
            return false;
        }
        if (!WhatWgUrlParser.isWindowsDriveLetterInternal(input, false)) {
            return false;
        }
        if (len == 2) {
            return true;
        }
        int ch2 = input.codePointAt(2);
        return ch2 == 47 || ch2 == 92 || ch2 == 63 || ch2 == 35;
    }

    private static boolean isWindowsDriveLetterInternal(CharSequence s, boolean normalized) {
        int ch0 = WhatWgUrlParser.codePointAt(s, 0);
        if (!WhatWgUrlParser.isAsciiAlpha(ch0)) {
            return false;
        }
        int ch1 = WhatWgUrlParser.codePointAt(s, 1);
        if (normalized) {
            return ch1 == 58;
        }
        return ch1 == 58 || ch1 == 124;
    }

    private static int codePointAt(CharSequence s, int index) {
        if (s instanceof String) {
            String string = (String)s;
            return string.codePointAt(index);
        }
        if (s instanceof StringBuilder) {
            StringBuilder builder = (StringBuilder)s;
            return builder.codePointAt(index);
        }
        throw new IllegalStateException();
    }

    static final class UrlRecord {
        private String scheme = "";
        @Nullable
        private StringBuilder username = null;
        @Nullable
        private StringBuilder password = null;
        @Nullable
        private Host host = null;
        @Nullable
        private Port port = null;
        private Path path = new PathSegments();
        @Nullable
        private StringBuilder query = null;
        @Nullable
        private StringBuilder fragment = null;

        public boolean isSpecial() {
            return WhatWgUrlParser.isSpecialScheme(this.scheme);
        }

        public boolean includesCredentials() {
            return this.username != null && !this.username.isEmpty() || this.password != null && !this.password.isEmpty();
        }

        public boolean hasOpaquePath() {
            return this.path().isOpaque();
        }

        public String scheme() {
            return this.scheme;
        }

        public String protocol() {
            return this.scheme() + ":";
        }

        public String username() {
            return this.username != null ? this.username.toString() : "";
        }

        void appendToUsername(int codePoint) {
            if (this.username == null) {
                this.username = new StringBuilder(2);
            }
            this.username.appendCodePoint(codePoint);
        }

        public void appendToUsername(String s) {
            if (this.username == null) {
                this.username = new StringBuilder(s);
            } else {
                this.username.append(s);
            }
        }

        public String password() {
            return this.password != null ? this.password.toString() : "";
        }

        void appendToPassword(int codePoint) {
            if (this.password == null) {
                this.password = new StringBuilder(2);
            }
            this.password.appendCodePoint(codePoint);
        }

        void appendToPassword(String s) {
            if (this.password == null) {
                this.password = new StringBuilder(s);
            } else {
                this.password.append(s);
            }
        }

        @Nullable
        public String userInfo() {
            if (!this.includesCredentials()) {
                return null;
            }
            StringBuilder userInfo = new StringBuilder(this.username());
            if (!this.password().isEmpty()) {
                userInfo.append(':');
                userInfo.append(this.password());
            }
            return userInfo.toString();
        }

        @Nullable
        public Host host() {
            return this.host;
        }

        public String hostString() {
            if (this.host() == null) {
                return "";
            }
            StringBuilder builder = new StringBuilder(this.hostname());
            Port port = this.port();
            if (port != null) {
                builder.append(':');
                builder.append(port);
            }
            return builder.toString();
        }

        public String hostname() {
            Host host = this.host();
            return host != null ? host.toString() : "";
        }

        @Nullable
        public Port port() {
            return this.port;
        }

        public String portString() {
            return this.port() != null ? this.port().toString() : "";
        }

        public Path path() {
            return this.path;
        }

        public String pathname() {
            return this.path().name();
        }

        public void shortenPath() {
            this.path.shorten(this.scheme);
        }

        @Nullable
        public String query() {
            return this.query != null ? this.query.toString() : null;
        }

        public String search() {
            String query = this.query();
            if (query == null) {
                return "";
            }
            return "?" + query;
        }

        @Nullable
        public String fragment() {
            return this.fragment != null ? this.fragment.toString() : null;
        }

        public String hash() {
            String fragment = this.fragment();
            return fragment != null && !fragment.isEmpty() ? "#" + fragment : "";
        }

        public String href() {
            String fragment;
            PathSegments pathSegments;
            Object port;
            StringBuilder output = new StringBuilder(this.scheme());
            output.append(':');
            Host host = this.host();
            if (host != null) {
                output.append("//");
                if (this.includesCredentials()) {
                    output.append(this.username());
                    String password = this.password();
                    if (!password.isEmpty()) {
                        output.append(':');
                        output.append(password);
                    }
                    output.append('@');
                }
                output.append(this.hostname());
                port = this.port();
                if (port != null) {
                    output.append(':');
                    output.append(this.port());
                }
            } else if (!this.hasOpaquePath() && (port = this.path()) instanceof PathSegments && (pathSegments = (PathSegments)port).size() > 1 && pathSegments.get(0).isEmpty()) {
                output.append("/.");
            }
            output.append(this.pathname());
            String query = this.query();
            if (query != null) {
                output.append('?');
                output.append(query);
            }
            if ((fragment = this.fragment()) != null) {
                output.append('#');
                output.append(fragment);
            }
            return output.toString();
        }

        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (obj == null || obj.getClass() != this.getClass()) {
                return false;
            }
            UrlRecord that = (UrlRecord)obj;
            return Objects.equals(this.scheme(), that.scheme()) && Objects.equals(this.username(), that.username()) && Objects.equals(this.password(), that.password()) && Objects.equals(this.host(), that.host()) && Objects.equals(this.port(), that.port()) && Objects.equals(this.path(), that.path()) && Objects.equals(this.query(), that.query()) && Objects.equals(this.fragment(), that.fragment());
        }

        public int hashCode() {
            return Objects.hash(this.scheme, this.username, this.password, this.host, this.port, this.path, this.query, this.fragment);
        }

        public String toString() {
            return "UrlRecord[scheme=" + this.scheme + ", username=" + String.valueOf(this.username) + ", password=" + String.valueOf(this.password) + ", host=" + String.valueOf(this.host) + ", port=" + String.valueOf(this.port) + ", path=" + String.valueOf(this.path) + ", query=" + String.valueOf(this.query) + ", fragment=" + String.valueOf(this.fragment) + "]";
        }
    }

    /*
     * Uses 'sealed' constructs - enablewith --sealed true
     */
    private static enum State {
        SCHEME_START{

            @Override
            public void handle(int c, UrlRecord url, WhatWgUrlParser p) {
                if (WhatWgUrlParser.isAsciiAlpha(c)) {
                    p.append(p.openCurlyBracketCount == 0 ? (int)Character.toLowerCase((char)c) : c);
                    p.setState(SCHEME);
                } else if (c == 123) {
                    ++p.openCurlyBracketCount;
                    p.append(c);
                    p.setState(SCHEME);
                } else if (p.stateOverride == null) {
                    p.setState(NO_SCHEME);
                    --p.pointer;
                } else {
                    p.failure(null);
                }
            }
        }
        ,
        SCHEME{

            @Override
            public void handle(int c, UrlRecord url, WhatWgUrlParser p) {
                if (WhatWgUrlParser.isAsciiAlphaNumeric(c) || c == 43 || c == 45 || c == 46) {
                    p.append(p.openCurlyBracketCount == 0 ? (int)Character.toLowerCase((char)c) : c);
                } else if (c == 58) {
                    IntPort intPort;
                    Port port;
                    if (p.stateOverride != null) {
                        boolean urlSpecialScheme = url.isSpecial();
                        String bufferString = p.buffer.toString();
                        boolean bufferSpecialScheme = WhatWgUrlParser.isSpecialScheme(bufferString);
                        if (urlSpecialScheme && !bufferSpecialScheme) {
                            return;
                        }
                        if (!urlSpecialScheme && bufferSpecialScheme) {
                            return;
                        }
                        if ((url.includesCredentials() || url.port() != null) && "file".equals(bufferString)) {
                            return;
                        }
                        if ("file".equals(url.scheme()) && (url.host() == null || url.host() == EmptyHost.INSTANCE)) {
                            return;
                        }
                    }
                    url.scheme = p.buffer.toString();
                    if (p.stateOverride != null && (port = url.port) instanceof IntPort && (intPort = (IntPort)port).value() == WhatWgUrlParser.defaultPort(url.scheme)) {
                        url.port = null;
                        p.stopMainLoop = true;
                        return;
                    }
                    p.emptyBuffer();
                    if (url.scheme.equals("file")) {
                        if (p.validate() && (p.remaining(0) != 47 || p.remaining(1) != 47)) {
                            p.validationError("\"file\" scheme not followed by \"//\".");
                        }
                        p.setState(FILE);
                    } else if (url.isSpecial() && p.base != null && p.base.scheme().equals(url.scheme)) {
                        Assert.state(!p.base.path().isOpaque(), "Opaque path not expected");
                        p.setState(SPECIAL_RELATIVE_OR_AUTHORITY);
                    } else if (url.isSpecial()) {
                        p.setState(SPECIAL_AUTHORITY_SLASHES);
                    } else if (p.remaining(0) == 47) {
                        p.setState(PATH_OR_AUTHORITY);
                        ++p.pointer;
                    } else {
                        url.path = new PathSegment("");
                        p.setState(OPAQUE_PATH);
                    }
                } else if (p.processCurlyBrackets(c)) {
                    p.append(c);
                } else if (p.stateOverride == null) {
                    p.emptyBuffer();
                    p.setState(NO_SCHEME);
                    p.pointer = -1;
                } else {
                    p.failure(null);
                }
            }
        }
        ,
        NO_SCHEME{

            @Override
            public void handle(int c, UrlRecord url, WhatWgUrlParser p) {
                if (p.base == null || p.base.path().isOpaque() && c != 35) {
                    p.failure("The input is missing a scheme, because it does not begin with an ASCII alpha \"" + (c != -1 ? Character.toString(c) : "") + "\", and no base URL was provided.");
                } else if (p.base.path().isOpaque() && c == 35) {
                    url.scheme = p.base.scheme();
                    url.path = p.base.path();
                    url.query = p.base.query;
                    url.fragment = new StringBuilder();
                    p.setState(FRAGMENT);
                } else if (!"file".equals(p.base.scheme())) {
                    p.setState(RELATIVE);
                    --p.pointer;
                } else {
                    p.setState(FILE);
                    --p.pointer;
                }
            }
        }
        ,
        SPECIAL_RELATIVE_OR_AUTHORITY{

            @Override
            public void handle(int c, UrlRecord url, WhatWgUrlParser p) {
                if (c == 47 && p.remaining(0) == 47) {
                    p.setState(SPECIAL_AUTHORITY_IGNORE_SLASHES);
                    ++p.pointer;
                } else {
                    if (p.validate()) {
                        p.validationError("The input\u2019s scheme is not followed by \"//\".");
                    }
                    p.setState(RELATIVE);
                    --p.pointer;
                }
            }
        }
        ,
        PATH_OR_AUTHORITY{

            @Override
            public void handle(int c, UrlRecord url, WhatWgUrlParser p) {
                if (c == 47) {
                    p.setState(AUTHORITY);
                } else {
                    p.setState(PATH);
                    --p.pointer;
                }
            }
        }
        ,
        RELATIVE{

            @Override
            public void handle(int c, UrlRecord url, WhatWgUrlParser p) {
                Assert.state(p.base != null && !"file".equals(p.base.scheme()), "Base scheme not provided or supported");
                url.scheme = p.base.scheme;
                if (c == 47) {
                    p.append('/');
                    p.setState(RELATIVE_SLASH);
                } else if (url.isSpecial() && c == 92) {
                    if (p.validate()) {
                        p.validationError("URL uses \\ instead of /.");
                    }
                    p.append('/');
                    p.setState(RELATIVE_SLASH);
                } else {
                    url.username = p.base.username != null ? new StringBuilder(p.base.username) : null;
                    url.password = p.base.password != null ? new StringBuilder(p.base.password) : null;
                    url.host = p.base.host();
                    url.port = p.base.port();
                    url.path = p.base.path().clone();
                    url.query = p.base.query;
                    if (c == 63) {
                        url.query = new StringBuilder();
                        p.setState(QUERY);
                    } else if (c == 35) {
                        url.fragment = new StringBuilder();
                        p.setState(FRAGMENT);
                    } else if (c != -1) {
                        url.query = null;
                        url.shortenPath();
                        p.setState(PATH);
                        --p.pointer;
                    }
                }
            }
        }
        ,
        RELATIVE_SLASH{

            @Override
            public void handle(int c, UrlRecord url, WhatWgUrlParser p) {
                if (url.isSpecial() && (c == 47 || c == 92)) {
                    if (p.validate() && c == 92) {
                        p.validationError("URL uses \\ instead of /.");
                    }
                    p.setState(SPECIAL_AUTHORITY_IGNORE_SLASHES);
                } else if (c == 47) {
                    p.emptyBuffer();
                    p.setState(AUTHORITY);
                } else {
                    Assert.state(p.base != null, "No base URL available");
                    url.username = p.base.username != null ? new StringBuilder(p.base.username) : null;
                    url.password = p.base.password != null ? new StringBuilder(p.base.password) : null;
                    url.host = p.base.host();
                    url.port = p.base.port();
                    p.setState(PATH);
                    --p.pointer;
                }
            }
        }
        ,
        SPECIAL_AUTHORITY_SLASHES{

            @Override
            public void handle(int c, UrlRecord url, WhatWgUrlParser p) {
                if (c == 47 && p.remaining(0) == 47) {
                    p.setState(SPECIAL_AUTHORITY_IGNORE_SLASHES);
                    ++p.pointer;
                } else {
                    if (p.validate()) {
                        p.validationError("Scheme \"" + url.scheme + "\" not followed by \"//\".");
                    }
                    p.setState(SPECIAL_AUTHORITY_IGNORE_SLASHES);
                    --p.pointer;
                }
            }
        }
        ,
        SPECIAL_AUTHORITY_IGNORE_SLASHES{

            @Override
            public void handle(int c, UrlRecord url, WhatWgUrlParser p) {
                if (c != 47 && c != 92) {
                    p.setState(AUTHORITY);
                    --p.pointer;
                } else if (p.validate()) {
                    p.validationError("Scheme \"" + url.scheme + "\" not followed by \"//\".");
                }
            }
        }
        ,
        AUTHORITY{

            @Override
            public void handle(int c, UrlRecord url, WhatWgUrlParser p) {
                if (c == 64) {
                    if (p.validate()) {
                        p.validationError("Invalid credentials");
                    }
                    if (p.atSignSeen) {
                        p.prepend("%40");
                    }
                    p.atSignSeen = true;
                    int bufferLen = p.buffer.length();
                    for (int i = 0; i < bufferLen; ++i) {
                        int codePoint = p.buffer.codePointAt(i);
                        if (codePoint == 58 && !p.passwordTokenSeen) {
                            p.passwordTokenSeen = true;
                            continue;
                        }
                        String encodedCodePoints = p.percentEncode(codePoint, WhatWgUrlParser::userinfoPercentEncodeSet);
                        if (p.passwordTokenSeen) {
                            if (encodedCodePoints != null) {
                                url.appendToPassword(encodedCodePoints);
                                continue;
                            }
                            url.appendToPassword(codePoint);
                            continue;
                        }
                        if (encodedCodePoints != null) {
                            url.appendToUsername(encodedCodePoints);
                            continue;
                        }
                        url.appendToUsername(codePoint);
                    }
                    p.emptyBuffer();
                } else if (c == -1 || c == 47 || c == 63 || c == 35 || url.isSpecial() && c == 92) {
                    if (p.atSignSeen && p.buffer.isEmpty()) {
                        p.failure("Missing host.");
                    }
                    p.pointer -= p.buffer.length() + 1;
                    p.emptyBuffer();
                    p.setState(HOST);
                } else {
                    p.append(c);
                }
            }
        }
        ,
        HOST{

            @Override
            public void handle(int c, UrlRecord url, WhatWgUrlParser p) {
                if (p.stateOverride != null && "file".equals(url.scheme())) {
                    --p.pointer;
                    p.setState(FILE_HOST);
                } else if (c == 58 && !p.insideBrackets) {
                    if (p.buffer.isEmpty()) {
                        p.failure("Missing host.");
                    }
                    if (p.stateOverride == HOST) {
                        p.stopMainLoop = true;
                        return;
                    }
                    url.host = Host.parse(p.buffer.toString(), !url.isSpecial(), p);
                    p.emptyBuffer();
                    p.setState(PORT);
                } else if (c == -1 || c == 47 || c == 63 || c == 35 || url.isSpecial() && c == 92) {
                    --p.pointer;
                    if (url.isSpecial() && p.buffer.isEmpty()) {
                        p.failure("The input has a special scheme, but does not contain a host.");
                    } else if (p.stateOverride != null && p.buffer.isEmpty() && (url.includesCredentials() || url.port() != null)) {
                        p.stopMainLoop = true;
                        return;
                    }
                    url.host = !p.buffer.isEmpty() ? Host.parse(p.buffer.toString(), !url.isSpecial(), p) : EmptyHost.INSTANCE;
                    p.emptyBuffer();
                    p.setState(PATH_START);
                    if (p.stateOverride != null) {
                        p.stopMainLoop = true;
                    }
                } else {
                    if (c == 91) {
                        p.insideBrackets = true;
                    } else if (c == 93) {
                        p.insideBrackets = false;
                    }
                    p.append(c);
                }
            }
        }
        ,
        PORT{

            @Override
            public void handle(int c, UrlRecord url, WhatWgUrlParser p) {
                if (WhatWgUrlParser.isAsciiDigit(c)) {
                    p.append(c);
                } else if (c == -1 || c == 47 || c == 63 || c == 35 || url.isSpecial() && c == 92 || p.stateOverride != null) {
                    if (!p.buffer.isEmpty()) {
                        block14: {
                            if (WhatWgUrlParser.containsOnlyAsciiDigits(p.buffer)) {
                                try {
                                    int defaultPort;
                                    int port = Integer.parseInt(p.buffer, 0, p.buffer.length(), 10);
                                    if (port > 65535) {
                                        p.failure("Port \"" + port + "\" is out of range");
                                    }
                                    if ((defaultPort = WhatWgUrlParser.defaultPort(url.scheme)) != -1 && port == defaultPort) {
                                        url.port = null;
                                        break block14;
                                    }
                                    url.port = new IntPort(port);
                                }
                                catch (NumberFormatException ex) {
                                    p.failure(ex.getMessage());
                                }
                            } else {
                                url.port = new StringPort(p.buffer.toString());
                            }
                        }
                        p.emptyBuffer();
                    }
                    if (p.stateOverride != null) {
                        p.stopMainLoop = true;
                        return;
                    }
                    p.setState(PATH_START);
                    --p.pointer;
                } else if (p.processCurlyBrackets(c)) {
                    p.append(c);
                } else {
                    p.failure("Invalid port: \"" + Character.toString(c) + "\"");
                }
            }
        }
        ,
        FILE{

            @Override
            public void handle(int c, UrlRecord url, WhatWgUrlParser p) {
                url.scheme = "file";
                url.host = EmptyHost.INSTANCE;
                if (c == 47 || c == 92) {
                    if (p.validate() && c == 92) {
                        p.validationError("URL uses \\ instead of /.");
                    }
                    p.setState(FILE_SLASH);
                } else if (p.base != null && p.base.scheme().equals("file")) {
                    url.host = p.base.host;
                    url.path = p.base.path().clone();
                    url.query = p.base.query;
                    if (c == 63) {
                        url.query = new StringBuilder();
                        p.setState(QUERY);
                    } else if (c == 35) {
                        url.fragment = new StringBuilder();
                        p.setState(FRAGMENT);
                    } else if (c != -1) {
                        url.query = null;
                        String substring = p.input.substring(p.pointer);
                        if (!WhatWgUrlParser.startsWithWindowsDriveLetter(substring)) {
                            url.shortenPath();
                        } else {
                            if (p.validate()) {
                                p.validationError("The input is a relative-URL string that starts with a Windows drive letter and the base URL\u2019s scheme is \"file\".");
                            }
                            url.path = new PathSegments();
                        }
                        p.setState(PATH);
                        --p.pointer;
                    }
                } else {
                    p.setState(PATH);
                    --p.pointer;
                }
            }
        }
        ,
        FILE_SLASH{

            @Override
            public void handle(int c, UrlRecord url, WhatWgUrlParser p) {
                if (c == 47 || c == 92) {
                    if (p.validate() && c == 92) {
                        p.validationError("URL uses \\ instead of /.");
                    }
                    p.setState(FILE_HOST);
                } else {
                    if (p.base != null && p.base.scheme.equals("file")) {
                        PathSegments basePath;
                        Path path;
                        url.host = p.base.host;
                        String substring = p.input.substring(p.pointer);
                        if (!WhatWgUrlParser.startsWithWindowsDriveLetter(substring) && (path = p.base.path) instanceof PathSegments && !(basePath = (PathSegments)path).isEmpty() && WhatWgUrlParser.isWindowsDriveLetter(basePath.get(0), true)) {
                            url.path.append(basePath.get(0));
                        }
                    }
                    p.setState(PATH);
                    --p.pointer;
                }
            }
        }
        ,
        FILE_HOST{

            @Override
            public void handle(int c, UrlRecord url, WhatWgUrlParser p) {
                if (c == -1 || c == 47 || c == 92 || c == 63 || c == 35) {
                    --p.pointer;
                    if (p.stateOverride == null && WhatWgUrlParser.isWindowsDriveLetter(p.buffer, false)) {
                        p.validationError("A file: URL\u2019s host is a Windows drive letter.");
                        p.setState(PATH);
                    } else if (p.buffer.isEmpty()) {
                        url.host = EmptyHost.INSTANCE;
                        if (p.stateOverride != null) {
                            p.stopMainLoop = true;
                            return;
                        }
                        p.setState(PATH_START);
                    } else {
                        Domain domain;
                        Host host = Host.parse(p.buffer.toString(), !url.isSpecial(), p);
                        if (host instanceof Domain && (domain = (Domain)host).domain().equals("localhost")) {
                            host = EmptyHost.INSTANCE;
                        }
                        url.host = host;
                        if (p.stateOverride != null) {
                            p.stopMainLoop = true;
                            return;
                        }
                        p.emptyBuffer();
                        p.setState(PATH_START);
                    }
                } else {
                    p.append(c);
                }
            }
        }
        ,
        PATH_START{

            @Override
            public void handle(int c, UrlRecord url, WhatWgUrlParser p) {
                if (url.isSpecial()) {
                    if (p.validate() && c == 92) {
                        p.validationError("URL uses \"\\\" instead of \"/\"");
                    }
                    p.setState(PATH);
                    if (c != 47 && c != 92) {
                        --p.pointer;
                    } else {
                        p.append('/');
                    }
                } else if (p.stateOverride == null && c == 63) {
                    url.query = new StringBuilder();
                    p.setState(QUERY);
                } else if (p.stateOverride == null && c == 35) {
                    url.fragment = new StringBuilder();
                    p.setState(FRAGMENT);
                } else if (c != -1) {
                    p.setState(PATH);
                    if (c != 47) {
                        --p.pointer;
                    } else {
                        p.append('/');
                    }
                } else if (p.stateOverride != null && url.host() == null) {
                    url.path().append("");
                }
            }
        }
        ,
        PATH{

            @Override
            public void handle(int c, UrlRecord url, WhatWgUrlParser p) {
                if (c == -1 || c == 47 || url.isSpecial() && c == 92 || p.stateOverride == null && (c == 63 || c == 35)) {
                    if (p.validate() && url.isSpecial() && c == 92) {
                        p.validationError("URL uses \"\\\" instead of \"/\"");
                    }
                    if (WhatWgUrlParser.isDoubleDotPathSegment(p.buffer)) {
                        url.shortenPath();
                        if (!(c == 47 || url.isSpecial() && c == 92)) {
                            url.path.append("");
                        }
                    } else {
                        boolean singlePathSegment = WhatWgUrlParser.isSingleDotPathSegment(p.buffer);
                        if (singlePathSegment && c != 47 && (!url.isSpecial() || c != 92)) {
                            url.path.append("");
                        } else if (!singlePathSegment) {
                            if ("file".equals(url.scheme) && url.path.isEmpty() && WhatWgUrlParser.isWindowsDriveLetter(p.buffer, false)) {
                                p.buffer.setCharAt(1, ':');
                            }
                            url.path.append(p.buffer.toString());
                        }
                    }
                    p.emptyBuffer();
                    if (c == 47 || url.isSpecial() && c == 92) {
                        p.append('/');
                    }
                    if (c == 63) {
                        url.query = new StringBuilder();
                        p.setState(QUERY);
                    }
                    if (c == 35) {
                        url.fragment = new StringBuilder();
                        p.setState(FRAGMENT);
                    }
                } else {
                    String encoded;
                    if (p.validate()) {
                        if (!WhatWgUrlParser.isUrlCodePoint(c) && c != 37) {
                            p.validationError("Invalid URL Unit: \"" + (char)c + "\"");
                        } else if (!(c != 37 || p.pointer < p.input.length() - 2 && WhatWgUrlParser.isAsciiHexDigit(p.input.codePointAt(p.pointer + 1)) && WhatWgUrlParser.isAsciiHexDigit(p.input.codePointAt(p.pointer + 2)))) {
                            p.validationError("Invalid URL Unit: \"" + (char)c + "\"");
                        }
                    }
                    if ((encoded = p.percentEncode(c, WhatWgUrlParser::pathPercentEncodeSet)) != null) {
                        p.append(encoded);
                    } else {
                        p.append(c);
                    }
                }
            }
        }
        ,
        OPAQUE_PATH{

            @Override
            public void handle(int c, UrlRecord url, WhatWgUrlParser p) {
                if (c == 63) {
                    url.query = new StringBuilder();
                    p.setState(QUERY);
                } else if (c == 35) {
                    url.fragment = new StringBuilder();
                    p.setState(FRAGMENT);
                } else {
                    if (p.validate()) {
                        if (c != -1 && !WhatWgUrlParser.isUrlCodePoint(c) && c != 37) {
                            p.validationError("Invalid URL Unit: \"" + (char)c + "\"");
                        } else if (!(c != 37 || p.pointer < p.input.length() - 2 && WhatWgUrlParser.isAsciiHexDigit(p.input.codePointAt(p.pointer + 1)) && WhatWgUrlParser.isAsciiHexDigit(p.input.codePointAt(p.pointer + 2)))) {
                            p.validationError("Invalid URL Unit: \"" + (char)c + "\"");
                        }
                    }
                    if (c != -1) {
                        String encoded = p.percentEncode(c, WhatWgUrlParser::c0ControlPercentEncodeSet);
                        if (encoded != null) {
                            url.path.append(encoded);
                        } else {
                            url.path.append(c);
                        }
                    }
                }
            }
        }
        ,
        QUERY{

            @Override
            public void handle(int c, UrlRecord url, WhatWgUrlParser p) {
                if (p.encoding != null && !StandardCharsets.UTF_8.equals(p.encoding) && (!url.isSpecial() || "ws".equals(url.scheme) || "wss".equals(url.scheme))) {
                    p.encoding = StandardCharsets.UTF_8;
                }
                if (p.stateOverride == null && c == 35 || c == -1) {
                    IntPredicate queryPercentEncodeSet = url.isSpecial() ? WhatWgUrlParser::specialQueryPercentEncodeSet : WhatWgUrlParser::queryPercentEncodeSet;
                    String encoded = p.percentEncode(p.buffer.toString(), queryPercentEncodeSet);
                    Assert.state(url.query != null, "Url's query should not be null");
                    url.query.append(encoded);
                    p.emptyBuffer();
                    if (c == 35) {
                        url.fragment = new StringBuilder();
                        p.setState(FRAGMENT);
                    }
                } else {
                    if (p.validate()) {
                        if (!WhatWgUrlParser.isUrlCodePoint(c) && c != 37) {
                            p.validationError("Invalid URL Unit: \"" + (char)c + "\"");
                        } else if (!(c != 37 || p.pointer < p.input.length() - 2 && WhatWgUrlParser.isAsciiHexDigit(p.input.codePointAt(p.pointer + 1)) && WhatWgUrlParser.isAsciiHexDigit(p.input.codePointAt(p.pointer + 2)))) {
                            p.validationError("Invalid URL Unit: \"" + (char)c + "\"");
                        }
                    }
                    p.append(c);
                }
            }
        }
        ,
        FRAGMENT{

            @Override
            public void handle(int c, UrlRecord url, WhatWgUrlParser p) {
                if (c != -1) {
                    if (p.validate()) {
                        if (!WhatWgUrlParser.isUrlCodePoint(c) && c != 37) {
                            p.validationError("Invalid URL Unit: \"" + (char)c + "\"");
                        } else if (!(c != 37 || p.pointer < p.input.length() - 2 && WhatWgUrlParser.isAsciiHexDigit(p.input.codePointAt(p.pointer + 1)) && WhatWgUrlParser.isAsciiHexDigit(p.input.codePointAt(p.pointer + 2)))) {
                            p.validationError("Invalid URL Unit: \"" + (char)c + "\"");
                        }
                    }
                    String encoded = p.percentEncode(c, WhatWgUrlParser::fragmentPercentEncodeSet);
                    Assert.state(url.fragment != null, "Url's fragment should not be null");
                    if (encoded != null) {
                        url.fragment.append(encoded);
                    } else {
                        url.fragment.appendCodePoint(c);
                    }
                }
            }
        };


        public abstract void handle(int var1, UrlRecord var2, WhatWgUrlParser var3);
    }

    private static final class ParseIpv4NumberFailure
    implements ParseIpv4NumberResult {
        public static final ParseIpv4NumberFailure INSTANCE = new ParseIpv4NumberFailure();

        private ParseIpv4NumberFailure() {
        }
    }

    private record ParseIpv4NumberSuccess(int number, boolean validationError) implements ParseIpv4NumberResult
    {
    }

    /*
     * Uses 'sealed' constructs - enablewith --sealed true
     */
    private static interface ParseIpv4NumberResult {
    }

    static final class PathSegments
    implements Path {
        private final List<PathSegment> segments;

        public PathSegments() {
            this.segments = new ArrayList<PathSegment>();
        }

        public PathSegments(List<PathSegment> segments) {
            this.segments = new ArrayList<PathSegment>(segments);
        }

        @Override
        public void append(int codePoint) {
            this.segments.add(new PathSegment(codePoint));
        }

        @Override
        public void append(String segment) {
            this.segments.add(new PathSegment(segment));
        }

        public int size() {
            return this.segments.size();
        }

        public String get(int i) {
            return this.segments.get(i).segment();
        }

        @Override
        public boolean isEmpty() {
            return this.segments.isEmpty();
        }

        @Override
        public void shorten(String scheme) {
            int size = this.size();
            if ("file".equals(scheme) && size == 1 && WhatWgUrlParser.isWindowsDriveLetter(this.get(0), true)) {
                return;
            }
            if (!this.isEmpty()) {
                this.segments.remove(size - 1);
            }
        }

        @Override
        public boolean isOpaque() {
            return false;
        }

        @Override
        public Path clone() {
            return new PathSegments(this.segments);
        }

        @Override
        public String name() {
            StringBuilder output = new StringBuilder();
            for (PathSegment segment : this.segments) {
                output.append('/');
                output.append(segment.name());
            }
            return output.toString();
        }

        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (o instanceof PathSegments) {
                PathSegments other = (PathSegments)o;
                return this.segments.equals(other.segments);
            }
            return false;
        }

        public int hashCode() {
            return this.segments.hashCode();
        }

        public String toString() {
            StringBuilder output = new StringBuilder();
            for (PathSegment segment : this.segments) {
                output.append(segment);
            }
            return output.toString();
        }
    }

    static final class PathSegment
    implements Path {
        @Nullable
        private StringBuilder builder = null;
        @Nullable
        String segment;

        PathSegment(String segment) {
            this.segment = segment;
        }

        PathSegment(int codePoint) {
            this.append(codePoint);
        }

        public String segment() {
            String result = this.segment;
            if (result == null) {
                Assert.state(this.builder != null, "String nor StringBuilder available");
                this.segment = result = this.builder.toString();
            }
            return result;
        }

        @Override
        public void append(int codePoint) {
            this.segment = null;
            if (this.builder == null) {
                this.builder = new StringBuilder(2);
            }
            this.builder.appendCodePoint(codePoint);
        }

        @Override
        public void append(String s) {
            this.segment = null;
            if (this.builder == null) {
                this.builder = new StringBuilder(s);
            } else {
                this.builder.append(s);
            }
        }

        @Override
        public String name() {
            String name = this.segment();
            if (name.startsWith("/")) {
                name = name.substring(1);
            }
            return name;
        }

        @Override
        public boolean isEmpty() {
            if (this.segment != null) {
                return this.segment.isEmpty();
            }
            Assert.state(this.builder != null, "String nor StringBuilder available");
            return this.builder.isEmpty();
        }

        @Override
        public void shorten(String scheme) {
            throw new IllegalStateException("Opaque path not expected");
        }

        @Override
        public boolean isOpaque() {
            return true;
        }

        @Override
        public Path clone() {
            return new PathSegment(this.segment());
        }

        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (o instanceof PathSegment) {
                PathSegment other = (PathSegment)o;
                return this.segment().equals(other.segment());
            }
            return false;
        }

        public int hashCode() {
            return this.segment().hashCode();
        }

        public String toString() {
            return this.segment();
        }
    }

    /*
     * Uses 'sealed' constructs - enablewith --sealed true
     */
    static interface Path {
        public void append(int var1);

        public void append(String var1);

        public boolean isEmpty();

        public void shorten(String var1);

        public boolean isOpaque();

        public Path clone();

        public String name();
    }

    static final class IntPort
    implements Port {
        private final int port;

        public IntPort(int port) {
            this.port = port;
        }

        public int value() {
            return this.port;
        }

        public String toString() {
            return Integer.toString(this.port);
        }
    }

    static final class StringPort
    implements Port {
        private final String port;

        public StringPort(String port) {
            this.port = port;
        }

        public String value() {
            return this.port;
        }

        public String toString() {
            return this.port;
        }
    }

    /*
     * Uses 'sealed' constructs - enablewith --sealed true
     */
    static interface Port {
    }

    static final class Ipv6Address
    implements IpAddress {
        private final int[] pieces;
        private final String string;

        private Ipv6Address(int[] pieces) {
            Assert.state(pieces.length == 8, "Invalid amount of IPv6 pieces");
            this.pieces = pieces;
            this.string = Ipv6Address.serialize(pieces);
        }

        public static Ipv6Address parse(String input) {
            int c;
            int[] address = new int[8];
            int pieceIndex = 0;
            Integer compress = null;
            int pointer = 0;
            int inputLength = input.length();
            int n = c = inputLength > 0 ? input.codePointAt(0) : -1;
            if (c == 58) {
                if (inputLength > 1 && input.codePointAt(1) != 58) {
                    throw new InvalidUrlException("IPv6 address begins with improper compression.");
                }
                pointer += 2;
                compress = ++pieceIndex;
            }
            int n2 = c = pointer < inputLength ? input.codePointAt(pointer) : -1;
            while (c != -1) {
                int length;
                if (pieceIndex == 8) {
                    throw new InvalidUrlException("IPv6 address contains more than 8 pieces.");
                }
                if (c == 58) {
                    if (compress != null) {
                        throw new InvalidUrlException("IPv6 address is compressed in more than one spot.");
                    }
                    compress = ++pieceIndex;
                    c = ++pointer < inputLength ? input.codePointAt(pointer) : -1;
                    continue;
                }
                int value = 0;
                for (length = 0; length < 4 && WhatWgUrlParser.isAsciiHexDigit(c); ++length) {
                    int cHex = Character.digit(c, 16);
                    value = value * 16 + cHex;
                    c = ++pointer < inputLength ? input.codePointAt(pointer) : -1;
                }
                if (c == 46) {
                    if (length == 0) {
                        throw new InvalidUrlException("IPv6 address with IPv4 address syntax: IPv4 part is empty.");
                    }
                    pointer -= length;
                    if (pieceIndex > 6) {
                        throw new InvalidUrlException("IPv6 address with IPv4 address syntax: IPv6 address has more than 6 pieces.");
                    }
                    int numbersSeen = 0;
                    int n3 = c = pointer < inputLength ? input.codePointAt(pointer) : -1;
                    while (c != -1) {
                        Integer ipv4Piece = null;
                        if (numbersSeen > 0) {
                            if (c == 46 && numbersSeen < 4) {
                                c = ++pointer < inputLength ? input.codePointAt(pointer) : -1;
                            } else {
                                throw new InvalidUrlException("IPv6 address with IPv4 address syntax: IPv4 part is empty or contains a non-ASCII digit.");
                            }
                        }
                        if (!WhatWgUrlParser.isAsciiDigit(c)) {
                            throw new InvalidUrlException("IPv6 address with IPv4 address syntax: IPv4 part contains a non-ASCII digit.");
                        }
                        while (WhatWgUrlParser.isAsciiDigit(c)) {
                            int number = Character.digit(c, 10);
                            if (ipv4Piece == null) {
                                ipv4Piece = number;
                            } else {
                                if (ipv4Piece == 0) {
                                    throw new InvalidUrlException("IPv6 address with IPv4 address syntax: IPv4 part contains a non-ASCII digit.");
                                }
                                ipv4Piece = ipv4Piece * 10 + number;
                            }
                            if (ipv4Piece > 255) {
                                throw new InvalidUrlException("IPv6 address with IPv4 address syntax: IPv4 part exceeds 255.");
                            }
                            c = ++pointer < inputLength ? input.codePointAt(pointer) : -1;
                        }
                        address[pieceIndex] = address[pieceIndex] * 256 + (ipv4Piece != null ? ipv4Piece : 0);
                        if (++numbersSeen == 2 || numbersSeen == 4) {
                            ++pieceIndex;
                        }
                        c = pointer < inputLength ? input.codePointAt(pointer) : -1;
                    }
                    if (numbersSeen == 4) break;
                    throw new InvalidUrlException("IPv6 address with IPv4 address syntax: IPv4 address contains too few parts.");
                }
                if (c == 58) {
                    int n4 = c = ++pointer < inputLength ? input.codePointAt(pointer) : -1;
                    if (c == -1) {
                        throw new InvalidUrlException("IPv6 address unexpectedly ends.");
                    }
                } else if (c != -1) {
                    throw new InvalidUrlException("IPv6 address contains \"" + Character.toString(c) + "\", which is neither an ASCII hex digit nor a ':'.");
                }
                address[pieceIndex] = value;
                ++pieceIndex;
            }
            if (compress != null) {
                int swaps = pieceIndex - compress;
                for (pieceIndex = 7; pieceIndex != 0 && swaps > 0; --pieceIndex, --swaps) {
                    int tmp = address[pieceIndex];
                    address[pieceIndex] = address[compress + swaps - 1];
                    address[compress.intValue() + swaps - 1] = tmp;
                }
            } else if (pieceIndex != 8) {
                throw new InvalidUrlException("An uncompressed IPv6 address contains fewer than 8 pieces.");
            }
            return new Ipv6Address(address);
        }

        private static String serialize(int[] address) {
            StringBuilder output = new StringBuilder();
            int compress = Ipv6Address.longestSequenceOf0Pieces(address);
            boolean ignore0 = false;
            for (int pieceIndex = 0; pieceIndex <= 7; ++pieceIndex) {
                if (ignore0 && address[pieceIndex] == 0) continue;
                if (ignore0) {
                    ignore0 = false;
                }
                if (compress == pieceIndex) {
                    String separator = pieceIndex == 0 ? "::" : ":";
                    output.append(separator);
                    ignore0 = true;
                    continue;
                }
                output.append(Integer.toHexString(address[pieceIndex]));
                if (pieceIndex == 7) continue;
                output.append(':');
            }
            return output.toString();
        }

        private static int longestSequenceOf0Pieces(int[] pieces) {
            int longestStart = -1;
            int longestLength = -1;
            int start = -1;
            for (int i = 0; i < pieces.length + 1; ++i) {
                if (i < pieces.length && pieces[i] == 0) {
                    if (start >= 0) continue;
                    start = i;
                    continue;
                }
                if (start < 0) continue;
                int length = i - start;
                if (length > longestLength) {
                    longestStart = start;
                    longestLength = length;
                }
                start = -1;
            }
            if (longestLength > 1) {
                return longestStart;
            }
            return -1;
        }

        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (obj instanceof Ipv6Address) {
                Ipv6Address other = (Ipv6Address)obj;
                return Arrays.equals(this.pieces, other.pieces);
            }
            return false;
        }

        public int hashCode() {
            return Arrays.hashCode(this.pieces);
        }

        public String toString() {
            return this.string;
        }
    }

    static final class Ipv4Address
    implements IpAddress {
        private final int address;
        private final String string;

        Ipv4Address(int address) {
            this.address = address;
            this.string = Ipv4Address.serialize(address);
        }

        private static String serialize(int address) {
            StringBuilder output = new StringBuilder();
            int n = address;
            for (int i = 1; i <= 4; ++i) {
                output.insert(0, Integer.toUnsignedString(Integer.remainderUnsigned(n, 256)));
                if (i != 4) {
                    output.insert(0, '.');
                }
                n = Math.floorDiv(n, 256);
            }
            return output.toString();
        }

        public static Ipv4Address parse(String input, WhatWgUrlParser p) {
            int partsSize;
            LinkedList<String> parts = WhatWgUrlParser.strictSplit(input, 46);
            if (((String)parts.get((partsSize = parts.size()) - 1)).isEmpty()) {
                p.validationError("IPv4 address ends with \".\"");
                if (partsSize > 1) {
                    parts.remove(partsSize - 1);
                    --partsSize;
                }
            }
            if (partsSize > 4) {
                throw new InvalidUrlException("IPv4 address does not consist of exactly 4 parts.");
            }
            ArrayList<Integer> numbers = new ArrayList<Integer>(partsSize);
            for (int i = 0; i < partsSize; ++i) {
                String part = (String)parts.get(i);
                ParseIpv4NumberResult result = Ipv4Address.parseIpv4Number(part);
                if (result == ParseIpv4NumberFailure.INSTANCE) {
                    p.failure("An IPv4 address part is not numeric.");
                    continue;
                }
                ParseIpv4NumberSuccess success = (ParseIpv4NumberSuccess)result;
                if (p.validate() && success.validationError()) {
                    p.validationError("The IPv4 address contains numbers expressed using hexadecimal or octal digits.");
                }
                numbers.add(success.number());
            }
            Iterator iterator = numbers.iterator();
            while (iterator.hasNext()) {
                Integer number = (Integer)iterator.next();
                if (p.validate() && number > 255) {
                    p.validationError("An IPv4 address part exceeds 255.");
                }
                if (iterator.hasNext()) {
                    if (number <= 255) continue;
                    throw new InvalidUrlException("An IPv4 address part exceeds 255.");
                }
                double limit = Math.pow(256.0, 5 - numbers.size());
                if (!((double)number.intValue() >= limit)) continue;
                throw new InvalidUrlException("IPv4 address part " + number + " exceeds " + limit + ".'");
            }
            int ipv4 = (Integer)numbers.get(numbers.size() - 1);
            numbers.remove(numbers.size() - 1);
            int counter = 0;
            for (Integer n : numbers) {
                int increment = n * (int)Math.pow(256.0, 3 - counter);
                ipv4 += increment;
                ++counter;
            }
            return new Ipv4Address(ipv4);
        }

        private static ParseIpv4NumberResult parseIpv4Number(String input) {
            if (input.isEmpty()) {
                return ParseIpv4NumberFailure.INSTANCE;
            }
            boolean validationError = false;
            int r = 10;
            int len = input.length();
            if (len >= 2) {
                int ch0 = input.codePointAt(0);
                int ch1 = input.codePointAt(1);
                if (ch0 == 48 && (ch1 == 88 || ch1 == 120)) {
                    validationError = true;
                    input = input.substring(2);
                    r = 16;
                } else if (ch0 == 48) {
                    validationError = true;
                    input = input.substring(1);
                    r = 8;
                }
            }
            if (input.isEmpty()) {
                return new ParseIpv4NumberSuccess(0, true);
            }
            for (int i = 0; i < input.length(); ++i) {
                int c = input.codePointAt(i);
                int digit = Character.digit(c, r);
                if (digit != -1) continue;
                return ParseIpv4NumberFailure.INSTANCE;
            }
            try {
                int output = Integer.parseInt(input, r);
                return new ParseIpv4NumberSuccess(output, validationError);
            }
            catch (NumberFormatException ex) {
                return ParseIpv4NumberFailure.INSTANCE;
            }
        }

        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (o instanceof Ipv4Address) {
                Ipv4Address other = (Ipv4Address)o;
                return this.address == other.address;
            }
            return false;
        }

        public int hashCode() {
            return this.address;
        }

        public String toString() {
            return this.string;
        }
    }

    /*
     * Uses 'sealed' constructs - enablewith --sealed true
     */
    static interface IpAddress {
    }

    static final class EmptyHost
    implements Host {
        static final EmptyHost INSTANCE = new EmptyHost();

        private EmptyHost() {
        }

        public boolean equals(Object obj) {
            return obj == this || obj != null && obj.getClass() == this.getClass();
        }

        public int hashCode() {
            return 1;
        }

        public String toString() {
            return "";
        }
    }

    static final class OpaqueHost
    implements Host {
        private final String host;

        private OpaqueHost(String host) {
            this.host = host;
        }

        public static OpaqueHost parse(String input, WhatWgUrlParser p) {
            for (int i = 0; i < input.length(); ++i) {
                int ch = input.codePointAt(i);
                if (WhatWgUrlParser.isForbiddenHost(ch)) {
                    throw new InvalidUrlException("An opaque host contains a forbidden host code point.");
                }
                if (p.validate() && !WhatWgUrlParser.isUrlCodePoint(ch) && ch != 37) {
                    p.validationError("Code point \"" + ch + "\" is not a URL unit.");
                }
                if (!p.validate() || ch != 37 || input.length() - i >= 2 && WhatWgUrlParser.isAsciiDigit(input.codePointAt(i + 1)) && WhatWgUrlParser.isAsciiDigit(input.codePointAt(i + 2))) continue;
                p.validationError("Code point \"" + ch + "\" is not a URL unit.");
            }
            String encoded = p.percentEncode(input, WhatWgUrlParser::c0ControlPercentEncodeSet);
            return new OpaqueHost(encoded);
        }

        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (obj instanceof OpaqueHost) {
                OpaqueHost other = (OpaqueHost)obj;
                return this.host.equals(other.host);
            }
            return false;
        }

        public int hashCode() {
            return this.host.hashCode();
        }

        public String toString() {
            return this.host;
        }
    }

    static final class IpAddressHost
    implements Host {
        private final IpAddress address;
        private final String addressString;

        IpAddressHost(IpAddress address) {
            this.address = address;
            this.addressString = address instanceof Ipv6Address ? "[" + String.valueOf(address) + "]" : address.toString();
        }

        public IpAddress address() {
            return this.address;
        }

        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (obj instanceof IpAddressHost) {
                IpAddressHost other = (IpAddressHost)obj;
                return this.address.equals(other.address);
            }
            return false;
        }

        public int hashCode() {
            return this.address.hashCode();
        }

        public String toString() {
            return this.addressString;
        }
    }

    static final class Domain
    implements Host {
        private final String domain;

        Domain(String domain) {
            this.domain = domain;
        }

        public String domain() {
            return this.domain;
        }

        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (o instanceof Domain) {
                Domain other = (Domain)o;
                return this.domain.equals(other.domain);
            }
            return false;
        }

        public int hashCode() {
            return this.domain.hashCode();
        }

        public String toString() {
            return this.domain;
        }
    }

    /*
     * Uses 'sealed' constructs - enablewith --sealed true
     */
    static interface Host {
        public static Host parse(String input, boolean isOpaque, WhatWgUrlParser p) {
            if (!input.isEmpty() && input.codePointAt(0) == 91) {
                int last = input.length() - 1;
                if (input.codePointAt(last) != 93) {
                    throw new InvalidUrlException("IPv6 address is missing the closing \"]\").");
                }
                String ipv6Host = input.substring(1, last);
                return new IpAddressHost(Ipv6Address.parse(ipv6Host));
            }
            if (isOpaque) {
                return OpaqueHost.parse(input, p);
            }
            Assert.state(!input.isEmpty(), "Input should not be empty");
            String domain = WhatWgUrlParser.percentDecode(input);
            String asciiDomain = WhatWgUrlParser.domainToAscii(domain, false);
            for (int i = 0; i < asciiDomain.length(); ++i) {
                int ch = asciiDomain.codePointAt(i);
                if (!WhatWgUrlParser.isForbiddenDomain(ch)) continue;
                throw new InvalidUrlException("Invalid character \"" + ch + "\" in domain \"" + input + "\"");
            }
            if (Host.endsInNumber(asciiDomain)) {
                Ipv4Address address = Ipv4Address.parse(asciiDomain, p);
                return new IpAddressHost(address);
            }
            return new Domain(asciiDomain);
        }

        private static boolean endsInNumber(String input) {
            String last;
            LinkedList<String> parts = WhatWgUrlParser.strictSplit(input, 46);
            if (parts.isEmpty()) {
                return false;
            }
            if (parts.getLast().isEmpty()) {
                if (parts.size() == 1) {
                    return false;
                }
                parts.removeLast();
            }
            if (!(last = parts.getLast()).isEmpty() && WhatWgUrlParser.containsOnlyAsciiDigits(last)) {
                return true;
            }
            ParseIpv4NumberResult result = Ipv4Address.parseIpv4Number(last);
            return result != ParseIpv4NumberFailure.INSTANCE;
        }
    }
}

