(function (global, factory) {
  typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
  typeof define === 'function' && define.amd ? define(['exports'], factory) :
  (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.FormViewer = {}));
})(this, (function (exports) { 'use strict';

  function createCommonjsModule(fn, module) {
  	return module = { exports: {} }, fn(module, module.exports), module.exports;
  }

  var hat_1 = createCommonjsModule(function (module) {
  var hat = module.exports = function (bits, base) {
      if (!base) base = 16;
      if (bits === undefined) bits = 128;
      if (bits <= 0) return '0';
      
      var digits = Math.log(Math.pow(2, bits)) / Math.log(base);
      for (var i = 2; digits === Infinity; i *= 2) {
          digits = Math.log(Math.pow(2, bits / i)) / Math.log(base) * i;
      }
      
      var rem = digits - Math.floor(digits);
      
      var res = '';
      
      for (var i = 0; i < Math.floor(digits); i++) {
          var x = Math.floor(Math.random() * base).toString(base);
          res = x + res;
      }
      
      if (rem) {
          var b = Math.pow(base, rem);
          var x = Math.floor(Math.random() * b).toString(base);
          res = x + res;
      }
      
      var parsed = parseInt(res, base);
      if (parsed !== Infinity && parsed >= Math.pow(2, bits)) {
          return hat(bits, base)
      }
      else return res;
  };

  hat.rack = function (bits, base, expandBy) {
      var fn = function (data) {
          var iters = 0;
          do {
              if (iters ++ > 10) {
                  if (expandBy) bits += expandBy;
                  else throw new Error('too many ID collisions, use more bits')
              }
              
              var id = hat(bits, base);
          } while (Object.hasOwnProperty.call(hats, id));
          
          hats[id] = data;
          return id;
      };
      var hats = fn.hats = {};
      
      fn.get = function (id) {
          return fn.hats[id];
      };
      
      fn.set = function (id, value) {
          fn.hats[id] = value;
          return fn;
      };
      
      fn.bits = bits || 128;
      fn.base = base || 16;
      return fn;
  };
  });

  /**
   * Create a new id generator / cache instance.
   *
   * You may optionally provide a seed that is used internally.
   *
   * @param {Seed} seed
   */

  function Ids(seed) {
    if (!(this instanceof Ids)) {
      return new Ids(seed);
    }

    seed = seed || [128, 36, 1];
    this._seed = seed.length ? hat_1.rack(seed[0], seed[1], seed[2]) : seed;
  }
  /**
   * Generate a next id.
   *
   * @param {Object} [element] element to bind the id to
   *
   * @return {String} id
   */

  Ids.prototype.next = function (element) {
    return this._seed(element || true);
  };
  /**
   * Generate a next id with a given prefix.
   *
   * @param {Object} [element] element to bind the id to
   *
   * @return {String} id
   */


  Ids.prototype.nextPrefixed = function (prefix, element) {
    var id;

    do {
      id = prefix + this.next(true);
    } while (this.assigned(id)); // claim {prefix}{random}


    this.claim(id, element); // return

    return id;
  };
  /**
   * Manually claim an existing id.
   *
   * @param {String} id
   * @param {String} [element] element the id is claimed by
   */


  Ids.prototype.claim = function (id, element) {
    this._seed.set(id, element || true);
  };
  /**
   * Returns true if the given id has already been assigned.
   *
   * @param  {String} id
   * @return {Boolean}
   */


  Ids.prototype.assigned = function (id) {
    return this._seed.get(id) || false;
  };
  /**
   * Unclaim an id.
   *
   * @param  {String} id the id to unclaim
   */


  Ids.prototype.unclaim = function (id) {
    delete this._seed.hats[id];
  };
  /**
   * Clear all claimed ids.
   */


  Ids.prototype.clear = function () {
    var hats = this._seed.hats,
        id;

    for (id in hats) {
      this.unclaim(id);
    }
  };

  /**
   * Flatten array, one level deep.
   *
   * @param {Array<?>} arr
   *
   * @return {Array<?>}
   */
  function flatten$3(arr) {
    return Array.prototype.concat.apply([], arr);
  }

  const nativeToString = Object.prototype.toString;
  const nativeHasOwnProperty = Object.prototype.hasOwnProperty;

  function isUndefined$1(obj) {
    return obj === undefined;
  }

  function isDefined(obj) {
    return obj !== undefined;
  }

  function isNil(obj) {
    return obj == null;
  }

  function isArray$3(obj) {
    return nativeToString.call(obj) === '[object Array]';
  }

  function isObject(obj) {
    return nativeToString.call(obj) === '[object Object]';
  }

  function isNumber$3(obj) {
    return nativeToString.call(obj) === '[object Number]';
  }

  function isFunction(obj) {
    const tag = nativeToString.call(obj);

    return (
      tag === '[object Function]' ||
      tag === '[object AsyncFunction]' ||
      tag === '[object GeneratorFunction]' ||
      tag === '[object AsyncGeneratorFunction]' ||
      tag === '[object Proxy]'
    );
  }

  function isString$3(obj) {
    return nativeToString.call(obj) === '[object String]';
  }

  /**
   * Return true, if target owns a property with the given key.
   *
   * @param {Object} target
   * @param {String} key
   *
   * @return {Boolean}
   */
  function has(target, key) {
    return nativeHasOwnProperty.call(target, key);
  }


  /**
   * Find element index in collection.
   *
   * @param  {Array|Object} collection
   * @param  {Function} matcher
   *
   * @return {Object}
   */
  function findIndex(collection, matcher) {

    matcher = toMatcher(matcher);

    let idx = isArray$3(collection) ? -1 : undefined;

    forEach(collection, function(val, key) {
      if (matcher(val, key)) {
        idx = key;

        return false;
      }
    });

    return idx;
  }


  /**
   * Iterate over collection; returning something
   * (non-undefined) will stop iteration.
   *
   * @param  {Array|Object} collection
   * @param  {Function} iterator
   *
   * @return {Object} return result that stopped the iteration
   */
  function forEach(collection, iterator) {

    let val,
        result;

    if (isUndefined$1(collection)) {
      return;
    }

    const convertKey = isArray$3(collection) ? toNum : identity;

    for (let key in collection) {

      if (has(collection, key)) {
        val = collection[key];

        result = iterator(val, convertKey(key));

        if (result === false) {
          return val;
        }
      }
    }
  }


  /**
   * Group collection members by attribute.
   *
   * @param  {Object|Array} collection
   * @param  {Function} extractor
   *
   * @return {Object} map with { attrValue => [ a, b, c ] }
   */
  function groupBy(collection, extractor, grouped = {}) {

    extractor = toExtractor(extractor);

    forEach(collection, function(val) {
      let discriminator = extractor(val) || '_';

      let group = grouped[discriminator];

      if (!group) {
        group = grouped[discriminator] = [];
      }

      group.push(val);
    });

    return grouped;
  }


  function toExtractor(extractor) {
    return isFunction(extractor) ? extractor : (e) => {
      return e[extractor];
    };
  }


  function toMatcher(matcher) {
    return isFunction(matcher) ? matcher : (e) => {
      return e === matcher;
    };
  }


  function identity(arg) {
    return arg;
  }

  function toNum(arg) {
    return Number(arg);
  }

  /**
   * Bind function against target <this>.
   *
   * @param  {Function} fn
   * @param  {Object}   target
   *
   * @return {Function} bound function
   */
  function bind(fn, target) {
    return fn.bind(target);
  }

  /**
   * Convenience wrapper for `Object.assign`.
   *
   * @param {Object} target
   * @param {...Object} others
   *
   * @return {Object} the target
   */
  function assign(target, ...others) {
    return Object.assign(target, ...others);
  }

  /**
   * Sets a nested property of a given object to the specified value.
   *
   * This mutates the object and returns it.
   *
   * @param {Object} target The target of the set operation.
   * @param {(string|number)[]} path The path to the nested value.
   * @param {any} value The value to set.
   */
  function set(target, path, value) {

    let currentTarget = target;

    forEach(path, function(key, idx) {

      if (typeof key !== 'number' && typeof key !== 'string') {
        throw new Error('illegal key type: ' + typeof key + '. Key should be of type number or string.');
      }

      if (key === 'constructor') {
        throw new Error('illegal key: constructor');
      }

      if (key === '__proto__') {
        throw new Error('illegal key: __proto__');
      }

      let nextKey = path[idx + 1];
      let nextTarget = currentTarget[key];

      if (isDefined(nextKey) && isNil(nextTarget)) {
        nextTarget = currentTarget[key] = isNaN(+nextKey) ? {} : [];
      }

      if (isUndefined$1(nextKey)) {
        if (isUndefined$1(value)) {
          delete currentTarget[key];
        } else {
          currentTarget[key] = value;
        }
      } else {
        currentTarget = nextTarget;
      }
    });

    return target;
  }


  /**
   * Gets a nested property of a given object.
   *
   * @param {Object} target The target of the get operation.
   * @param {(string|number)[]} path The path to the nested value.
   * @param {any} [defaultValue] The value to return if no value exists.
   */
  function get(target, path, defaultValue) {

    let currentTarget = target;

    forEach(path, function(key) {

      // accessing nil property yields <undefined>
      if (isNil(currentTarget)) {
        currentTarget = undefined;

        return false;
      }

      currentTarget = currentTarget[key];
    });

    return isUndefined$1(currentTarget) ? defaultValue : currentTarget;
  }

  // these aren't really private, but nor are they really useful to document

  /**
   * @private
   */
  class LuxonError extends Error {}

  /**
   * @private
   */
  class InvalidDateTimeError extends LuxonError {
    constructor(reason) {
      super(`Invalid DateTime: ${reason.toMessage()}`);
    }
  }

  /**
   * @private
   */
  class InvalidIntervalError extends LuxonError {
    constructor(reason) {
      super(`Invalid Interval: ${reason.toMessage()}`);
    }
  }

  /**
   * @private
   */
  class InvalidDurationError extends LuxonError {
    constructor(reason) {
      super(`Invalid Duration: ${reason.toMessage()}`);
    }
  }

  /**
   * @private
   */
  class ConflictingSpecificationError extends LuxonError {}

  /**
   * @private
   */
  class InvalidUnitError extends LuxonError {
    constructor(unit) {
      super(`Invalid unit ${unit}`);
    }
  }

  /**
   * @private
   */
  class InvalidArgumentError extends LuxonError {}

  /**
   * @private
   */
  class ZoneIsAbstractError extends LuxonError {
    constructor() {
      super("Zone is an abstract class");
    }
  }

  /**
   * @private
   */

  const n$2 = "numeric",
    s$2 = "short",
    l$3 = "long";

  const DATE_SHORT = {
    year: n$2,
    month: n$2,
    day: n$2,
  };

  const DATE_MED = {
    year: n$2,
    month: s$2,
    day: n$2,
  };

  const DATE_MED_WITH_WEEKDAY = {
    year: n$2,
    month: s$2,
    day: n$2,
    weekday: s$2,
  };

  const DATE_FULL = {
    year: n$2,
    month: l$3,
    day: n$2,
  };

  const DATE_HUGE = {
    year: n$2,
    month: l$3,
    day: n$2,
    weekday: l$3,
  };

  const TIME_SIMPLE = {
    hour: n$2,
    minute: n$2,
  };

  const TIME_WITH_SECONDS = {
    hour: n$2,
    minute: n$2,
    second: n$2,
  };

  const TIME_WITH_SHORT_OFFSET = {
    hour: n$2,
    minute: n$2,
    second: n$2,
    timeZoneName: s$2,
  };

  const TIME_WITH_LONG_OFFSET = {
    hour: n$2,
    minute: n$2,
    second: n$2,
    timeZoneName: l$3,
  };

  const TIME_24_SIMPLE = {
    hour: n$2,
    minute: n$2,
    hourCycle: "h23",
  };

  const TIME_24_WITH_SECONDS = {
    hour: n$2,
    minute: n$2,
    second: n$2,
    hourCycle: "h23",
  };

  const TIME_24_WITH_SHORT_OFFSET = {
    hour: n$2,
    minute: n$2,
    second: n$2,
    hourCycle: "h23",
    timeZoneName: s$2,
  };

  const TIME_24_WITH_LONG_OFFSET = {
    hour: n$2,
    minute: n$2,
    second: n$2,
    hourCycle: "h23",
    timeZoneName: l$3,
  };

  const DATETIME_SHORT = {
    year: n$2,
    month: n$2,
    day: n$2,
    hour: n$2,
    minute: n$2,
  };

  const DATETIME_SHORT_WITH_SECONDS = {
    year: n$2,
    month: n$2,
    day: n$2,
    hour: n$2,
    minute: n$2,
    second: n$2,
  };

  const DATETIME_MED = {
    year: n$2,
    month: s$2,
    day: n$2,
    hour: n$2,
    minute: n$2,
  };

  const DATETIME_MED_WITH_SECONDS = {
    year: n$2,
    month: s$2,
    day: n$2,
    hour: n$2,
    minute: n$2,
    second: n$2,
  };

  const DATETIME_MED_WITH_WEEKDAY = {
    year: n$2,
    month: s$2,
    day: n$2,
    weekday: s$2,
    hour: n$2,
    minute: n$2,
  };

  const DATETIME_FULL = {
    year: n$2,
    month: l$3,
    day: n$2,
    hour: n$2,
    minute: n$2,
    timeZoneName: s$2,
  };

  const DATETIME_FULL_WITH_SECONDS = {
    year: n$2,
    month: l$3,
    day: n$2,
    hour: n$2,
    minute: n$2,
    second: n$2,
    timeZoneName: s$2,
  };

  const DATETIME_HUGE = {
    year: n$2,
    month: l$3,
    day: n$2,
    weekday: l$3,
    hour: n$2,
    minute: n$2,
    timeZoneName: l$3,
  };

  const DATETIME_HUGE_WITH_SECONDS = {
    year: n$2,
    month: l$3,
    day: n$2,
    weekday: l$3,
    hour: n$2,
    minute: n$2,
    second: n$2,
    timeZoneName: l$3,
  };

  /**
   * @interface
   */
  class Zone {
    /**
     * The type of zone
     * @abstract
     * @type {string}
     */
    get type() {
      throw new ZoneIsAbstractError();
    }

    /**
     * The name of this zone.
     * @abstract
     * @type {string}
     */
    get name() {
      throw new ZoneIsAbstractError();
    }

    get ianaName() {
      return this.name;
    }

    /**
     * Returns whether the offset is known to be fixed for the whole year.
     * @abstract
     * @type {boolean}
     */
    get isUniversal() {
      throw new ZoneIsAbstractError();
    }

    /**
     * Returns the offset's common name (such as EST) at the specified timestamp
     * @abstract
     * @param {number} ts - Epoch milliseconds for which to get the name
     * @param {Object} opts - Options to affect the format
     * @param {string} opts.format - What style of offset to return. Accepts 'long' or 'short'.
     * @param {string} opts.locale - What locale to return the offset name in.
     * @return {string}
     */
    offsetName(ts, opts) {
      throw new ZoneIsAbstractError();
    }

    /**
     * Returns the offset's value as a string
     * @abstract
     * @param {number} ts - Epoch milliseconds for which to get the offset
     * @param {string} format - What style of offset to return.
     *                          Accepts 'narrow', 'short', or 'techie'. Returning '+6', '+06:00', or '+0600' respectively
     * @return {string}
     */
    formatOffset(ts, format) {
      throw new ZoneIsAbstractError();
    }

    /**
     * Return the offset in minutes for this zone at the specified timestamp.
     * @abstract
     * @param {number} ts - Epoch milliseconds for which to compute the offset
     * @return {number}
     */
    offset(ts) {
      throw new ZoneIsAbstractError();
    }

    /**
     * Return whether this Zone is equal to another zone
     * @abstract
     * @param {Zone} otherZone - the zone to compare
     * @return {boolean}
     */
    equals(otherZone) {
      throw new ZoneIsAbstractError();
    }

    /**
     * Return whether this Zone is valid.
     * @abstract
     * @type {boolean}
     */
    get isValid() {
      throw new ZoneIsAbstractError();
    }
  }

  let singleton$1 = null;

  /**
   * Represents the local zone for this JavaScript environment.
   * @implements {Zone}
   */
  class SystemZone extends Zone {
    /**
     * Get a singleton instance of the local zone
     * @return {SystemZone}
     */
    static get instance() {
      if (singleton$1 === null) {
        singleton$1 = new SystemZone();
      }
      return singleton$1;
    }

    /** @override **/
    get type() {
      return "system";
    }

    /** @override **/
    get name() {
      return new Intl.DateTimeFormat().resolvedOptions().timeZone;
    }

    /** @override **/
    get isUniversal() {
      return false;
    }

    /** @override **/
    offsetName(ts, { format, locale }) {
      return parseZoneInfo(ts, format, locale);
    }

    /** @override **/
    formatOffset(ts, format) {
      return formatOffset(this.offset(ts), format);
    }

    /** @override **/
    offset(ts) {
      return -new Date(ts).getTimezoneOffset();
    }

    /** @override **/
    equals(otherZone) {
      return otherZone.type === "system";
    }

    /** @override **/
    get isValid() {
      return true;
    }
  }

  let dtfCache = {};
  function makeDTF(zone) {
    if (!dtfCache[zone]) {
      dtfCache[zone] = new Intl.DateTimeFormat("en-US", {
        hour12: false,
        timeZone: zone,
        year: "numeric",
        month: "2-digit",
        day: "2-digit",
        hour: "2-digit",
        minute: "2-digit",
        second: "2-digit",
        era: "short",
      });
    }
    return dtfCache[zone];
  }

  const typeToPos = {
    year: 0,
    month: 1,
    day: 2,
    era: 3,
    hour: 4,
    minute: 5,
    second: 6,
  };

  function hackyOffset(dtf, date) {
    const formatted = dtf.format(date).replace(/\u200E/g, ""),
      parsed = /(\d+)\/(\d+)\/(\d+) (AD|BC),? (\d+):(\d+):(\d+)/.exec(formatted),
      [, fMonth, fDay, fYear, fadOrBc, fHour, fMinute, fSecond] = parsed;
    return [fYear, fMonth, fDay, fadOrBc, fHour, fMinute, fSecond];
  }

  function partsOffset(dtf, date) {
    const formatted = dtf.formatToParts(date);
    const filled = [];
    for (let i = 0; i < formatted.length; i++) {
      const { type, value } = formatted[i];
      const pos = typeToPos[type];

      if (type === "era") {
        filled[pos] = value;
      } else if (!isUndefined(pos)) {
        filled[pos] = parseInt(value, 10);
      }
    }
    return filled;
  }

  let ianaZoneCache = {};
  /**
   * A zone identified by an IANA identifier, like America/New_York
   * @implements {Zone}
   */
  class IANAZone extends Zone {
    /**
     * @param {string} name - Zone name
     * @return {IANAZone}
     */
    static create(name) {
      if (!ianaZoneCache[name]) {
        ianaZoneCache[name] = new IANAZone(name);
      }
      return ianaZoneCache[name];
    }

    /**
     * Reset local caches. Should only be necessary in testing scenarios.
     * @return {void}
     */
    static resetCache() {
      ianaZoneCache = {};
      dtfCache = {};
    }

    /**
     * Returns whether the provided string is a valid specifier. This only checks the string's format, not that the specifier identifies a known zone; see isValidZone for that.
     * @param {string} s - The string to check validity on
     * @example IANAZone.isValidSpecifier("America/New_York") //=> true
     * @example IANAZone.isValidSpecifier("Sport~~blorp") //=> false
     * @deprecated This method returns false for some valid IANA names. Use isValidZone instead.
     * @return {boolean}
     */
    static isValidSpecifier(s) {
      return this.isValidZone(s);
    }

    /**
     * Returns whether the provided string identifies a real zone
     * @param {string} zone - The string to check
     * @example IANAZone.isValidZone("America/New_York") //=> true
     * @example IANAZone.isValidZone("Fantasia/Castle") //=> false
     * @example IANAZone.isValidZone("Sport~~blorp") //=> false
     * @return {boolean}
     */
    static isValidZone(zone) {
      if (!zone) {
        return false;
      }
      try {
        new Intl.DateTimeFormat("en-US", { timeZone: zone }).format();
        return true;
      } catch (e) {
        return false;
      }
    }

    constructor(name) {
      super();
      /** @private **/
      this.zoneName = name;
      /** @private **/
      this.valid = IANAZone.isValidZone(name);
    }

    /** @override **/
    get type() {
      return "iana";
    }

    /** @override **/
    get name() {
      return this.zoneName;
    }

    /** @override **/
    get isUniversal() {
      return false;
    }

    /** @override **/
    offsetName(ts, { format, locale }) {
      return parseZoneInfo(ts, format, locale, this.name);
    }

    /** @override **/
    formatOffset(ts, format) {
      return formatOffset(this.offset(ts), format);
    }

    /** @override **/
    offset(ts) {
      const date = new Date(ts);

      if (isNaN(date)) return NaN;

      const dtf = makeDTF(this.name);
      let [year, month, day, adOrBc, hour, minute, second] = dtf.formatToParts
        ? partsOffset(dtf, date)
        : hackyOffset(dtf, date);

      if (adOrBc === "BC") {
        year = -Math.abs(year) + 1;
      }

      // because we're using hour12 and https://bugs.chromium.org/p/chromium/issues/detail?id=1025564&can=2&q=%2224%3A00%22%20datetimeformat
      const adjustedHour = hour === 24 ? 0 : hour;

      const asUTC = objToLocalTS({
        year,
        month,
        day,
        hour: adjustedHour,
        minute,
        second,
        millisecond: 0,
      });

      let asTS = +date;
      const over = asTS % 1000;
      asTS -= over >= 0 ? over : 1000 + over;
      return (asUTC - asTS) / (60 * 1000);
    }

    /** @override **/
    equals(otherZone) {
      return otherZone.type === "iana" && otherZone.name === this.name;
    }

    /** @override **/
    get isValid() {
      return this.valid;
    }
  }

  // todo - remap caching

  let intlLFCache = {};
  function getCachedLF(locString, opts = {}) {
    const key = JSON.stringify([locString, opts]);
    let dtf = intlLFCache[key];
    if (!dtf) {
      dtf = new Intl.ListFormat(locString, opts);
      intlLFCache[key] = dtf;
    }
    return dtf;
  }

  let intlDTCache = {};
  function getCachedDTF(locString, opts = {}) {
    const key = JSON.stringify([locString, opts]);
    let dtf = intlDTCache[key];
    if (!dtf) {
      dtf = new Intl.DateTimeFormat(locString, opts);
      intlDTCache[key] = dtf;
    }
    return dtf;
  }

  let intlNumCache = {};
  function getCachedINF(locString, opts = {}) {
    const key = JSON.stringify([locString, opts]);
    let inf = intlNumCache[key];
    if (!inf) {
      inf = new Intl.NumberFormat(locString, opts);
      intlNumCache[key] = inf;
    }
    return inf;
  }

  let intlRelCache = {};
  function getCachedRTF(locString, opts = {}) {
    const { base, ...cacheKeyOpts } = opts; // exclude `base` from the options
    const key = JSON.stringify([locString, cacheKeyOpts]);
    let inf = intlRelCache[key];
    if (!inf) {
      inf = new Intl.RelativeTimeFormat(locString, opts);
      intlRelCache[key] = inf;
    }
    return inf;
  }

  let sysLocaleCache = null;
  function systemLocale() {
    if (sysLocaleCache) {
      return sysLocaleCache;
    } else {
      sysLocaleCache = new Intl.DateTimeFormat().resolvedOptions().locale;
      return sysLocaleCache;
    }
  }

  function parseLocaleString(localeStr) {
    // I really want to avoid writing a BCP 47 parser
    // see, e.g. https://github.com/wooorm/bcp-47
    // Instead, we'll do this:

    // a) if the string has no -u extensions, just leave it alone
    // b) if it does, use Intl to resolve everything
    // c) if Intl fails, try again without the -u

    // private subtags and unicode subtags have ordering requirements,
    // and we're not properly parsing this, so just strip out the
    // private ones if they exist.
    const xIndex = localeStr.indexOf("-x-");
    if (xIndex !== -1) {
      localeStr = localeStr.substring(0, xIndex);
    }

    const uIndex = localeStr.indexOf("-u-");
    if (uIndex === -1) {
      return [localeStr];
    } else {
      let options;
      let selectedStr;
      try {
        options = getCachedDTF(localeStr).resolvedOptions();
        selectedStr = localeStr;
      } catch (e) {
        const smaller = localeStr.substring(0, uIndex);
        options = getCachedDTF(smaller).resolvedOptions();
        selectedStr = smaller;
      }

      const { numberingSystem, calendar } = options;
      return [selectedStr, numberingSystem, calendar];
    }
  }

  function intlConfigString(localeStr, numberingSystem, outputCalendar) {
    if (outputCalendar || numberingSystem) {
      if (!localeStr.includes("-u-")) {
        localeStr += "-u";
      }

      if (outputCalendar) {
        localeStr += `-ca-${outputCalendar}`;
      }

      if (numberingSystem) {
        localeStr += `-nu-${numberingSystem}`;
      }
      return localeStr;
    } else {
      return localeStr;
    }
  }

  function mapMonths(f) {
    const ms = [];
    for (let i = 1; i <= 12; i++) {
      const dt = DateTime.utc(2016, i, 1);
      ms.push(f(dt));
    }
    return ms;
  }

  function mapWeekdays(f) {
    const ms = [];
    for (let i = 1; i <= 7; i++) {
      const dt = DateTime.utc(2016, 11, 13 + i);
      ms.push(f(dt));
    }
    return ms;
  }

  function listStuff(loc, length, defaultOK, englishFn, intlFn) {
    const mode = loc.listingMode(defaultOK);

    if (mode === "error") {
      return null;
    } else if (mode === "en") {
      return englishFn(length);
    } else {
      return intlFn(length);
    }
  }

  function supportsFastNumbers(loc) {
    if (loc.numberingSystem && loc.numberingSystem !== "latn") {
      return false;
    } else {
      return (
        loc.numberingSystem === "latn" ||
        !loc.locale ||
        loc.locale.startsWith("en") ||
        new Intl.DateTimeFormat(loc.intl).resolvedOptions().numberingSystem === "latn"
      );
    }
  }

  /**
   * @private
   */

  class PolyNumberFormatter {
    constructor(intl, forceSimple, opts) {
      this.padTo = opts.padTo || 0;
      this.floor = opts.floor || false;

      const { padTo, floor, ...otherOpts } = opts;

      if (!forceSimple || Object.keys(otherOpts).length > 0) {
        const intlOpts = { useGrouping: false, ...opts };
        if (opts.padTo > 0) intlOpts.minimumIntegerDigits = opts.padTo;
        this.inf = getCachedINF(intl, intlOpts);
      }
    }

    format(i) {
      if (this.inf) {
        const fixed = this.floor ? Math.floor(i) : i;
        return this.inf.format(fixed);
      } else {
        // to match the browser's numberformatter defaults
        const fixed = this.floor ? Math.floor(i) : roundTo(i, 3);
        return padStart(fixed, this.padTo);
      }
    }
  }

  /**
   * @private
   */

  class PolyDateFormatter {
    constructor(dt, intl, opts) {
      this.opts = opts;

      let z = undefined;
      if (dt.zone.isUniversal) {
        // UTC-8 or Etc/UTC-8 are not part of tzdata, only Etc/GMT+8 and the like.
        // That is why fixed-offset TZ is set to that unless it is:
        // 1. Representing offset 0 when UTC is used to maintain previous behavior and does not become GMT.
        // 2. Unsupported by the browser:
        //    - some do not support Etc/
        //    - < Etc/GMT-14, > Etc/GMT+12, and 30-minute or 45-minute offsets are not part of tzdata
        const gmtOffset = -1 * (dt.offset / 60);
        const offsetZ = gmtOffset >= 0 ? `Etc/GMT+${gmtOffset}` : `Etc/GMT${gmtOffset}`;
        if (dt.offset !== 0 && IANAZone.create(offsetZ).valid) {
          z = offsetZ;
          this.dt = dt;
        } else {
          // Not all fixed-offset zones like Etc/+4:30 are present in tzdata.
          // So we have to make do. Two cases:
          // 1. The format options tell us to show the zone. We can't do that, so the best
          // we can do is format the date in UTC.
          // 2. The format options don't tell us to show the zone. Then we can adjust them
          // the time and tell the formatter to show it to us in UTC, so that the time is right
          // and the bad zone doesn't show up.
          z = "UTC";
          if (opts.timeZoneName) {
            this.dt = dt;
          } else {
            this.dt = dt.offset === 0 ? dt : DateTime.fromMillis(dt.ts + dt.offset * 60 * 1000);
          }
        }
      } else if (dt.zone.type === "system") {
        this.dt = dt;
      } else {
        this.dt = dt;
        z = dt.zone.name;
      }

      const intlOpts = { ...this.opts };
      intlOpts.timeZone = intlOpts.timeZone || z;
      this.dtf = getCachedDTF(intl, intlOpts);
    }

    format() {
      return this.dtf.format(this.dt.toJSDate());
    }

    formatToParts() {
      return this.dtf.formatToParts(this.dt.toJSDate());
    }

    resolvedOptions() {
      return this.dtf.resolvedOptions();
    }
  }

  /**
   * @private
   */
  class PolyRelFormatter {
    constructor(intl, isEnglish, opts) {
      this.opts = { style: "long", ...opts };
      if (!isEnglish && hasRelative()) {
        this.rtf = getCachedRTF(intl, opts);
      }
    }

    format(count, unit) {
      if (this.rtf) {
        return this.rtf.format(count, unit);
      } else {
        return formatRelativeTime(unit, count, this.opts.numeric, this.opts.style !== "long");
      }
    }

    formatToParts(count, unit) {
      if (this.rtf) {
        return this.rtf.formatToParts(count, unit);
      } else {
        return [];
      }
    }
  }

  /**
   * @private
   */

  class Locale {
    static fromOpts(opts) {
      return Locale.create(opts.locale, opts.numberingSystem, opts.outputCalendar, opts.defaultToEN);
    }

    static create(locale, numberingSystem, outputCalendar, defaultToEN = false) {
      const specifiedLocale = locale || Settings.defaultLocale;
      // the system locale is useful for human readable strings but annoying for parsing/formatting known formats
      const localeR = specifiedLocale || (defaultToEN ? "en-US" : systemLocale());
      const numberingSystemR = numberingSystem || Settings.defaultNumberingSystem;
      const outputCalendarR = outputCalendar || Settings.defaultOutputCalendar;
      return new Locale(localeR, numberingSystemR, outputCalendarR, specifiedLocale);
    }

    static resetCache() {
      sysLocaleCache = null;
      intlDTCache = {};
      intlNumCache = {};
      intlRelCache = {};
    }

    static fromObject({ locale, numberingSystem, outputCalendar } = {}) {
      return Locale.create(locale, numberingSystem, outputCalendar);
    }

    constructor(locale, numbering, outputCalendar, specifiedLocale) {
      const [parsedLocale, parsedNumberingSystem, parsedOutputCalendar] = parseLocaleString(locale);

      this.locale = parsedLocale;
      this.numberingSystem = numbering || parsedNumberingSystem || null;
      this.outputCalendar = outputCalendar || parsedOutputCalendar || null;
      this.intl = intlConfigString(this.locale, this.numberingSystem, this.outputCalendar);

      this.weekdaysCache = { format: {}, standalone: {} };
      this.monthsCache = { format: {}, standalone: {} };
      this.meridiemCache = null;
      this.eraCache = {};

      this.specifiedLocale = specifiedLocale;
      this.fastNumbersCached = null;
    }

    get fastNumbers() {
      if (this.fastNumbersCached == null) {
        this.fastNumbersCached = supportsFastNumbers(this);
      }

      return this.fastNumbersCached;
    }

    listingMode() {
      const isActuallyEn = this.isEnglish();
      const hasNoWeirdness =
        (this.numberingSystem === null || this.numberingSystem === "latn") &&
        (this.outputCalendar === null || this.outputCalendar === "gregory");
      return isActuallyEn && hasNoWeirdness ? "en" : "intl";
    }

    clone(alts) {
      if (!alts || Object.getOwnPropertyNames(alts).length === 0) {
        return this;
      } else {
        return Locale.create(
          alts.locale || this.specifiedLocale,
          alts.numberingSystem || this.numberingSystem,
          alts.outputCalendar || this.outputCalendar,
          alts.defaultToEN || false
        );
      }
    }

    redefaultToEN(alts = {}) {
      return this.clone({ ...alts, defaultToEN: true });
    }

    redefaultToSystem(alts = {}) {
      return this.clone({ ...alts, defaultToEN: false });
    }

    months(length, format = false, defaultOK = true) {
      return listStuff(this, length, defaultOK, months, () => {
        const intl = format ? { month: length, day: "numeric" } : { month: length },
          formatStr = format ? "format" : "standalone";
        if (!this.monthsCache[formatStr][length]) {
          this.monthsCache[formatStr][length] = mapMonths((dt) => this.extract(dt, intl, "month"));
        }
        return this.monthsCache[formatStr][length];
      });
    }

    weekdays(length, format = false, defaultOK = true) {
      return listStuff(this, length, defaultOK, weekdays, () => {
        const intl = format
            ? { weekday: length, year: "numeric", month: "long", day: "numeric" }
            : { weekday: length },
          formatStr = format ? "format" : "standalone";
        if (!this.weekdaysCache[formatStr][length]) {
          this.weekdaysCache[formatStr][length] = mapWeekdays((dt) =>
            this.extract(dt, intl, "weekday")
          );
        }
        return this.weekdaysCache[formatStr][length];
      });
    }

    meridiems(defaultOK = true) {
      return listStuff(
        this,
        undefined,
        defaultOK,
        () => meridiems,
        () => {
          // In theory there could be aribitrary day periods. We're gonna assume there are exactly two
          // for AM and PM. This is probably wrong, but it's makes parsing way easier.
          if (!this.meridiemCache) {
            const intl = { hour: "numeric", hourCycle: "h12" };
            this.meridiemCache = [DateTime.utc(2016, 11, 13, 9), DateTime.utc(2016, 11, 13, 19)].map(
              (dt) => this.extract(dt, intl, "dayperiod")
            );
          }

          return this.meridiemCache;
        }
      );
    }

    eras(length, defaultOK = true) {
      return listStuff(this, length, defaultOK, eras, () => {
        const intl = { era: length };

        // This is problematic. Different calendars are going to define eras totally differently. What I need is the minimum set of dates
        // to definitely enumerate them.
        if (!this.eraCache[length]) {
          this.eraCache[length] = [DateTime.utc(-40, 1, 1), DateTime.utc(2017, 1, 1)].map((dt) =>
            this.extract(dt, intl, "era")
          );
        }

        return this.eraCache[length];
      });
    }

    extract(dt, intlOpts, field) {
      const df = this.dtFormatter(dt, intlOpts),
        results = df.formatToParts(),
        matching = results.find((m) => m.type.toLowerCase() === field);
      return matching ? matching.value : null;
    }

    numberFormatter(opts = {}) {
      // this forcesimple option is never used (the only caller short-circuits on it, but it seems safer to leave)
      // (in contrast, the rest of the condition is used heavily)
      return new PolyNumberFormatter(this.intl, opts.forceSimple || this.fastNumbers, opts);
    }

    dtFormatter(dt, intlOpts = {}) {
      return new PolyDateFormatter(dt, this.intl, intlOpts);
    }

    relFormatter(opts = {}) {
      return new PolyRelFormatter(this.intl, this.isEnglish(), opts);
    }

    listFormatter(opts = {}) {
      return getCachedLF(this.intl, opts);
    }

    isEnglish() {
      return (
        this.locale === "en" ||
        this.locale.toLowerCase() === "en-us" ||
        new Intl.DateTimeFormat(this.intl).resolvedOptions().locale.startsWith("en-us")
      );
    }

    equals(other) {
      return (
        this.locale === other.locale &&
        this.numberingSystem === other.numberingSystem &&
        this.outputCalendar === other.outputCalendar
      );
    }
  }

  let singleton = null;

  /**
   * A zone with a fixed offset (meaning no DST)
   * @implements {Zone}
   */
  class FixedOffsetZone extends Zone {
    /**
     * Get a singleton instance of UTC
     * @return {FixedOffsetZone}
     */
    static get utcInstance() {
      if (singleton === null) {
        singleton = new FixedOffsetZone(0);
      }
      return singleton;
    }

    /**
     * Get an instance with a specified offset
     * @param {number} offset - The offset in minutes
     * @return {FixedOffsetZone}
     */
    static instance(offset) {
      return offset === 0 ? FixedOffsetZone.utcInstance : new FixedOffsetZone(offset);
    }

    /**
     * Get an instance of FixedOffsetZone from a UTC offset string, like "UTC+6"
     * @param {string} s - The offset string to parse
     * @example FixedOffsetZone.parseSpecifier("UTC+6")
     * @example FixedOffsetZone.parseSpecifier("UTC+06")
     * @example FixedOffsetZone.parseSpecifier("UTC-6:00")
     * @return {FixedOffsetZone}
     */
    static parseSpecifier(s) {
      if (s) {
        const r = s.match(/^utc(?:([+-]\d{1,2})(?::(\d{2}))?)?$/i);
        if (r) {
          return new FixedOffsetZone(signedOffset(r[1], r[2]));
        }
      }
      return null;
    }

    constructor(offset) {
      super();
      /** @private **/
      this.fixed = offset;
    }

    /** @override **/
    get type() {
      return "fixed";
    }

    /** @override **/
    get name() {
      return this.fixed === 0 ? "UTC" : `UTC${formatOffset(this.fixed, "narrow")}`;
    }

    get ianaName() {
      if (this.fixed === 0) {
        return "Etc/UTC";
      } else {
        return `Etc/GMT${formatOffset(-this.fixed, "narrow")}`;
      }
    }

    /** @override **/
    offsetName() {
      return this.name;
    }

    /** @override **/
    formatOffset(ts, format) {
      return formatOffset(this.fixed, format);
    }

    /** @override **/
    get isUniversal() {
      return true;
    }

    /** @override **/
    offset() {
      return this.fixed;
    }

    /** @override **/
    equals(otherZone) {
      return otherZone.type === "fixed" && otherZone.fixed === this.fixed;
    }

    /** @override **/
    get isValid() {
      return true;
    }
  }

  /**
   * A zone that failed to parse. You should never need to instantiate this.
   * @implements {Zone}
   */
  class InvalidZone extends Zone {
    constructor(zoneName) {
      super();
      /**  @private */
      this.zoneName = zoneName;
    }

    /** @override **/
    get type() {
      return "invalid";
    }

    /** @override **/
    get name() {
      return this.zoneName;
    }

    /** @override **/
    get isUniversal() {
      return false;
    }

    /** @override **/
    offsetName() {
      return null;
    }

    /** @override **/
    formatOffset() {
      return "";
    }

    /** @override **/
    offset() {
      return NaN;
    }

    /** @override **/
    equals() {
      return false;
    }

    /** @override **/
    get isValid() {
      return false;
    }
  }

  /**
   * @private
   */

  function normalizeZone(input, defaultZone) {
    if (isUndefined(input) || input === null) {
      return defaultZone;
    } else if (input instanceof Zone) {
      return input;
    } else if (isString$2(input)) {
      const lowered = input.toLowerCase();
      if (lowered === "default") return defaultZone;
      else if (lowered === "local" || lowered === "system") return SystemZone.instance;
      else if (lowered === "utc" || lowered === "gmt") return FixedOffsetZone.utcInstance;
      else return FixedOffsetZone.parseSpecifier(lowered) || IANAZone.create(input);
    } else if (isNumber$2(input)) {
      return FixedOffsetZone.instance(input);
    } else if (typeof input === "object" && input.offset && typeof input.offset === "number") {
      // This is dumb, but the instanceof check above doesn't seem to really work
      // so we're duck checking it
      return input;
    } else {
      return new InvalidZone(input);
    }
  }

  let now = () => Date.now(),
    defaultZone = "system",
    defaultLocale = null,
    defaultNumberingSystem = null,
    defaultOutputCalendar = null,
    twoDigitCutoffYear = 60,
    throwOnInvalid;

  /**
   * Settings contains static getters and setters that control Luxon's overall behavior. Luxon is a simple library with few options, but the ones it does have live here.
   */
  class Settings {
    /**
     * Get the callback for returning the current timestamp.
     * @type {function}
     */
    static get now() {
      return now;
    }

    /**
     * Set the callback for returning the current timestamp.
     * The function should return a number, which will be interpreted as an Epoch millisecond count
     * @type {function}
     * @example Settings.now = () => Date.now() + 3000 // pretend it is 3 seconds in the future
     * @example Settings.now = () => 0 // always pretend it's Jan 1, 1970 at midnight in UTC time
     */
    static set now(n) {
      now = n;
    }

    /**
     * Set the default time zone to create DateTimes in. Does not affect existing instances.
     * Use the value "system" to reset this value to the system's time zone.
     * @type {string}
     */
    static set defaultZone(zone) {
      defaultZone = zone;
    }

    /**
     * Get the default time zone object currently used to create DateTimes. Does not affect existing instances.
     * The default value is the system's time zone (the one set on the machine that runs this code).
     * @type {Zone}
     */
    static get defaultZone() {
      return normalizeZone(defaultZone, SystemZone.instance);
    }

    /**
     * Get the default locale to create DateTimes with. Does not affect existing instances.
     * @type {string}
     */
    static get defaultLocale() {
      return defaultLocale;
    }

    /**
     * Set the default locale to create DateTimes with. Does not affect existing instances.
     * @type {string}
     */
    static set defaultLocale(locale) {
      defaultLocale = locale;
    }

    /**
     * Get the default numbering system to create DateTimes with. Does not affect existing instances.
     * @type {string}
     */
    static get defaultNumberingSystem() {
      return defaultNumberingSystem;
    }

    /**
     * Set the default numbering system to create DateTimes with. Does not affect existing instances.
     * @type {string}
     */
    static set defaultNumberingSystem(numberingSystem) {
      defaultNumberingSystem = numberingSystem;
    }

    /**
     * Get the default output calendar to create DateTimes with. Does not affect existing instances.
     * @type {string}
     */
    static get defaultOutputCalendar() {
      return defaultOutputCalendar;
    }

    /**
     * Set the default output calendar to create DateTimes with. Does not affect existing instances.
     * @type {string}
     */
    static set defaultOutputCalendar(outputCalendar) {
      defaultOutputCalendar = outputCalendar;
    }

    /**
     * Get the cutoff year after which a string encoding a year as two digits is interpreted to occur in the current century.
     * @type {number}
     */
    static get twoDigitCutoffYear() {
      return twoDigitCutoffYear;
    }

    /**
     * Set the cutoff year after which a string encoding a year as two digits is interpreted to occur in the current century.
     * @type {number}
     * @example Settings.twoDigitCutoffYear = 0 // cut-off year is 0, so all 'yy' are interpretted as current century
     * @example Settings.twoDigitCutoffYear = 50 // '49' -> 1949; '50' -> 2050
     * @example Settings.twoDigitCutoffYear = 1950 // interpretted as 50
     * @example Settings.twoDigitCutoffYear = 2050 // ALSO interpretted as 50
     */
    static set twoDigitCutoffYear(cutoffYear) {
      twoDigitCutoffYear = cutoffYear % 100;
    }

    /**
     * Get whether Luxon will throw when it encounters invalid DateTimes, Durations, or Intervals
     * @type {boolean}
     */
    static get throwOnInvalid() {
      return throwOnInvalid;
    }

    /**
     * Set whether Luxon will throw when it encounters invalid DateTimes, Durations, or Intervals
     * @type {boolean}
     */
    static set throwOnInvalid(t) {
      throwOnInvalid = t;
    }

    /**
     * Reset Luxon's global caches. Should only be necessary in testing scenarios.
     * @return {void}
     */
    static resetCaches() {
      Locale.resetCache();
      IANAZone.resetCache();
    }
  }

  /*
    This is just a junk drawer, containing anything used across multiple classes.
    Because Luxon is small(ish), this should stay small and we won't worry about splitting
    it up into, say, parsingUtil.js and basicUtil.js and so on. But they are divided up by feature area.
  */

  /**
   * @private
   */

  // TYPES

  function isUndefined(o) {
    return typeof o === "undefined";
  }

  function isNumber$2(o) {
    return typeof o === "number";
  }

  function isInteger(o) {
    return typeof o === "number" && o % 1 === 0;
  }

  function isString$2(o) {
    return typeof o === "string";
  }

  function isDate(o) {
    return Object.prototype.toString.call(o) === "[object Date]";
  }

  // CAPABILITIES

  function hasRelative() {
    try {
      return typeof Intl !== "undefined" && !!Intl.RelativeTimeFormat;
    } catch (e) {
      return false;
    }
  }

  // OBJECTS AND ARRAYS

  function maybeArray(thing) {
    return Array.isArray(thing) ? thing : [thing];
  }

  function bestBy(arr, by, compare) {
    if (arr.length === 0) {
      return undefined;
    }
    return arr.reduce((best, next) => {
      const pair = [by(next), next];
      if (!best) {
        return pair;
      } else if (compare(best[0], pair[0]) === best[0]) {
        return best;
      } else {
        return pair;
      }
    }, null)[1];
  }

  function pick(obj, keys) {
    return keys.reduce((a, k) => {
      a[k] = obj[k];
      return a;
    }, {});
  }

  function hasOwnProperty(obj, prop) {
    return Object.prototype.hasOwnProperty.call(obj, prop);
  }

  // NUMBERS AND STRINGS

  function integerBetween(thing, bottom, top) {
    return isInteger(thing) && thing >= bottom && thing <= top;
  }

  // x % n but takes the sign of n instead of x
  function floorMod(x, n) {
    return x - n * Math.floor(x / n);
  }

  function padStart(input, n = 2) {
    const isNeg = input < 0;
    let padded;
    if (isNeg) {
      padded = "-" + ("" + -input).padStart(n, "0");
    } else {
      padded = ("" + input).padStart(n, "0");
    }
    return padded;
  }

  function parseInteger(string) {
    if (isUndefined(string) || string === null || string === "") {
      return undefined;
    } else {
      return parseInt(string, 10);
    }
  }

  function parseFloating(string) {
    if (isUndefined(string) || string === null || string === "") {
      return undefined;
    } else {
      return parseFloat(string);
    }
  }

  function parseMillis(fraction) {
    // Return undefined (instead of 0) in these cases, where fraction is not set
    if (isUndefined(fraction) || fraction === null || fraction === "") {
      return undefined;
    } else {
      const f = parseFloat("0." + fraction) * 1000;
      return Math.floor(f);
    }
  }

  function roundTo(number, digits, towardZero = false) {
    const factor = 10 ** digits,
      rounder = towardZero ? Math.trunc : Math.round;
    return rounder(number * factor) / factor;
  }

  // DATE BASICS

  function isLeapYear(year) {
    return year % 4 === 0 && (year % 100 !== 0 || year % 400 === 0);
  }

  function daysInYear(year) {
    return isLeapYear(year) ? 366 : 365;
  }

  function daysInMonth(year, month) {
    const modMonth = floorMod(month - 1, 12) + 1,
      modYear = year + (month - modMonth) / 12;

    if (modMonth === 2) {
      return isLeapYear(modYear) ? 29 : 28;
    } else {
      return [31, null, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31][modMonth - 1];
    }
  }

  // covert a calendar object to a local timestamp (epoch, but with the offset baked in)
  function objToLocalTS(obj) {
    let d = Date.UTC(
      obj.year,
      obj.month - 1,
      obj.day,
      obj.hour,
      obj.minute,
      obj.second,
      obj.millisecond
    );

    // for legacy reasons, years between 0 and 99 are interpreted as 19XX; revert that
    if (obj.year < 100 && obj.year >= 0) {
      d = new Date(d);
      d.setUTCFullYear(d.getUTCFullYear() - 1900);
    }
    return +d;
  }

  function weeksInWeekYear(weekYear) {
    const p1 =
        (weekYear +
          Math.floor(weekYear / 4) -
          Math.floor(weekYear / 100) +
          Math.floor(weekYear / 400)) %
        7,
      last = weekYear - 1,
      p2 = (last + Math.floor(last / 4) - Math.floor(last / 100) + Math.floor(last / 400)) % 7;
    return p1 === 4 || p2 === 3 ? 53 : 52;
  }

  function untruncateYear(year) {
    if (year > 99) {
      return year;
    } else return year > Settings.twoDigitCutoffYear ? 1900 + year : 2000 + year;
  }

  // PARSING

  function parseZoneInfo(ts, offsetFormat, locale, timeZone = null) {
    const date = new Date(ts),
      intlOpts = {
        hourCycle: "h23",
        year: "numeric",
        month: "2-digit",
        day: "2-digit",
        hour: "2-digit",
        minute: "2-digit",
      };

    if (timeZone) {
      intlOpts.timeZone = timeZone;
    }

    const modified = { timeZoneName: offsetFormat, ...intlOpts };

    const parsed = new Intl.DateTimeFormat(locale, modified)
      .formatToParts(date)
      .find((m) => m.type.toLowerCase() === "timezonename");
    return parsed ? parsed.value : null;
  }

  // signedOffset('-5', '30') -> -330
  function signedOffset(offHourStr, offMinuteStr) {
    let offHour = parseInt(offHourStr, 10);

    // don't || this because we want to preserve -0
    if (Number.isNaN(offHour)) {
      offHour = 0;
    }

    const offMin = parseInt(offMinuteStr, 10) || 0,
      offMinSigned = offHour < 0 || Object.is(offHour, -0) ? -offMin : offMin;
    return offHour * 60 + offMinSigned;
  }

  // COERCION

  function asNumber(value) {
    const numericValue = Number(value);
    if (typeof value === "boolean" || value === "" || Number.isNaN(numericValue))
      throw new InvalidArgumentError(`Invalid unit value ${value}`);
    return numericValue;
  }

  function normalizeObject(obj, normalizer) {
    const normalized = {};
    for (const u in obj) {
      if (hasOwnProperty(obj, u)) {
        const v = obj[u];
        if (v === undefined || v === null) continue;
        normalized[normalizer(u)] = asNumber(v);
      }
    }
    return normalized;
  }

  function formatOffset(offset, format) {
    const hours = Math.trunc(Math.abs(offset / 60)),
      minutes = Math.trunc(Math.abs(offset % 60)),
      sign = offset >= 0 ? "+" : "-";

    switch (format) {
      case "short":
        return `${sign}${padStart(hours, 2)}:${padStart(minutes, 2)}`;
      case "narrow":
        return `${sign}${hours}${minutes > 0 ? `:${minutes}` : ""}`;
      case "techie":
        return `${sign}${padStart(hours, 2)}${padStart(minutes, 2)}`;
      default:
        throw new RangeError(`Value format ${format} is out of range for property format`);
    }
  }

  function timeObject(obj) {
    return pick(obj, ["hour", "minute", "second", "millisecond"]);
  }

  /**
   * @private
   */

  const monthsLong = [
    "January",
    "February",
    "March",
    "April",
    "May",
    "June",
    "July",
    "August",
    "September",
    "October",
    "November",
    "December",
  ];

  const monthsShort = [
    "Jan",
    "Feb",
    "Mar",
    "Apr",
    "May",
    "Jun",
    "Jul",
    "Aug",
    "Sep",
    "Oct",
    "Nov",
    "Dec",
  ];

  const monthsNarrow = ["J", "F", "M", "A", "M", "J", "J", "A", "S", "O", "N", "D"];

  function months(length) {
    switch (length) {
      case "narrow":
        return [...monthsNarrow];
      case "short":
        return [...monthsShort];
      case "long":
        return [...monthsLong];
      case "numeric":
        return ["1", "2", "3", "4", "5", "6", "7", "8", "9", "10", "11", "12"];
      case "2-digit":
        return ["01", "02", "03", "04", "05", "06", "07", "08", "09", "10", "11", "12"];
      default:
        return null;
    }
  }

  const weekdaysLong = [
    "Monday",
    "Tuesday",
    "Wednesday",
    "Thursday",
    "Friday",
    "Saturday",
    "Sunday",
  ];

  const weekdaysShort = ["Mon", "Tue", "Wed", "Thu", "Fri", "Sat", "Sun"];

  const weekdaysNarrow = ["M", "T", "W", "T", "F", "S", "S"];

  function weekdays(length) {
    switch (length) {
      case "narrow":
        return [...weekdaysNarrow];
      case "short":
        return [...weekdaysShort];
      case "long":
        return [...weekdaysLong];
      case "numeric":
        return ["1", "2", "3", "4", "5", "6", "7"];
      default:
        return null;
    }
  }

  const meridiems = ["AM", "PM"];

  const erasLong = ["Before Christ", "Anno Domini"];

  const erasShort = ["BC", "AD"];

  const erasNarrow = ["B", "A"];

  function eras(length) {
    switch (length) {
      case "narrow":
        return [...erasNarrow];
      case "short":
        return [...erasShort];
      case "long":
        return [...erasLong];
      default:
        return null;
    }
  }

  function meridiemForDateTime(dt) {
    return meridiems[dt.hour < 12 ? 0 : 1];
  }

  function weekdayForDateTime(dt, length) {
    return weekdays(length)[dt.weekday - 1];
  }

  function monthForDateTime(dt, length) {
    return months(length)[dt.month - 1];
  }

  function eraForDateTime(dt, length) {
    return eras(length)[dt.year < 0 ? 0 : 1];
  }

  function formatRelativeTime(unit, count, numeric = "always", narrow = false) {
    const units = {
      years: ["year", "yr."],
      quarters: ["quarter", "qtr."],
      months: ["month", "mo."],
      weeks: ["week", "wk."],
      days: ["day", "day", "days"],
      hours: ["hour", "hr."],
      minutes: ["minute", "min."],
      seconds: ["second", "sec."],
    };

    const lastable = ["hours", "minutes", "seconds"].indexOf(unit) === -1;

    if (numeric === "auto" && lastable) {
      const isDay = unit === "days";
      switch (count) {
        case 1:
          return isDay ? "tomorrow" : `next ${units[unit][0]}`;
        case -1:
          return isDay ? "yesterday" : `last ${units[unit][0]}`;
        case 0:
          return isDay ? "today" : `this ${units[unit][0]}`;
      }
    }

    const isInPast = Object.is(count, -0) || count < 0,
      fmtValue = Math.abs(count),
      singular = fmtValue === 1,
      lilUnits = units[unit],
      fmtUnit = narrow
        ? singular
          ? lilUnits[1]
          : lilUnits[2] || lilUnits[1]
        : singular
        ? units[unit][0]
        : unit;
    return isInPast ? `${fmtValue} ${fmtUnit} ago` : `in ${fmtValue} ${fmtUnit}`;
  }

  function stringifyTokens(splits, tokenToString) {
    let s = "";
    for (const token of splits) {
      if (token.literal) {
        s += token.val;
      } else {
        s += tokenToString(token.val);
      }
    }
    return s;
  }

  const macroTokenToFormatOpts = {
    D: DATE_SHORT,
    DD: DATE_MED,
    DDD: DATE_FULL,
    DDDD: DATE_HUGE,
    t: TIME_SIMPLE,
    tt: TIME_WITH_SECONDS,
    ttt: TIME_WITH_SHORT_OFFSET,
    tttt: TIME_WITH_LONG_OFFSET,
    T: TIME_24_SIMPLE,
    TT: TIME_24_WITH_SECONDS,
    TTT: TIME_24_WITH_SHORT_OFFSET,
    TTTT: TIME_24_WITH_LONG_OFFSET,
    f: DATETIME_SHORT,
    ff: DATETIME_MED,
    fff: DATETIME_FULL,
    ffff: DATETIME_HUGE,
    F: DATETIME_SHORT_WITH_SECONDS,
    FF: DATETIME_MED_WITH_SECONDS,
    FFF: DATETIME_FULL_WITH_SECONDS,
    FFFF: DATETIME_HUGE_WITH_SECONDS,
  };

  /**
   * @private
   */

  class Formatter {
    static create(locale, opts = {}) {
      return new Formatter(locale, opts);
    }

    static parseFormat(fmt) {
      let current = null,
        currentFull = "",
        bracketed = false;
      const splits = [];
      for (let i = 0; i < fmt.length; i++) {
        const c = fmt.charAt(i);
        if (c === "'") {
          if (currentFull.length > 0) {
            splits.push({ literal: bracketed, val: currentFull });
          }
          current = null;
          currentFull = "";
          bracketed = !bracketed;
        } else if (bracketed) {
          currentFull += c;
        } else if (c === current) {
          currentFull += c;
        } else {
          if (currentFull.length > 0) {
            splits.push({ literal: false, val: currentFull });
          }
          currentFull = c;
          current = c;
        }
      }

      if (currentFull.length > 0) {
        splits.push({ literal: bracketed, val: currentFull });
      }

      return splits;
    }

    static macroTokenToFormatOpts(token) {
      return macroTokenToFormatOpts[token];
    }

    constructor(locale, formatOpts) {
      this.opts = formatOpts;
      this.loc = locale;
      this.systemLoc = null;
    }

    formatWithSystemDefault(dt, opts) {
      if (this.systemLoc === null) {
        this.systemLoc = this.loc.redefaultToSystem();
      }
      const df = this.systemLoc.dtFormatter(dt, { ...this.opts, ...opts });
      return df.format();
    }

    formatDateTime(dt, opts = {}) {
      const df = this.loc.dtFormatter(dt, { ...this.opts, ...opts });
      return df.format();
    }

    formatDateTimeParts(dt, opts = {}) {
      const df = this.loc.dtFormatter(dt, { ...this.opts, ...opts });
      return df.formatToParts();
    }

    formatInterval(interval, opts = {}) {
      const df = this.loc.dtFormatter(interval.start, { ...this.opts, ...opts });
      return df.dtf.formatRange(interval.start.toJSDate(), interval.end.toJSDate());
    }

    resolvedOptions(dt, opts = {}) {
      const df = this.loc.dtFormatter(dt, { ...this.opts, ...opts });
      return df.resolvedOptions();
    }

    num(n, p = 0) {
      // we get some perf out of doing this here, annoyingly
      if (this.opts.forceSimple) {
        return padStart(n, p);
      }

      const opts = { ...this.opts };

      if (p > 0) {
        opts.padTo = p;
      }

      return this.loc.numberFormatter(opts).format(n);
    }

    formatDateTimeFromString(dt, fmt) {
      const knownEnglish = this.loc.listingMode() === "en",
        useDateTimeFormatter = this.loc.outputCalendar && this.loc.outputCalendar !== "gregory",
        string = (opts, extract) => this.loc.extract(dt, opts, extract),
        formatOffset = (opts) => {
          if (dt.isOffsetFixed && dt.offset === 0 && opts.allowZ) {
            return "Z";
          }

          return dt.isValid ? dt.zone.formatOffset(dt.ts, opts.format) : "";
        },
        meridiem = () =>
          knownEnglish
            ? meridiemForDateTime(dt)
            : string({ hour: "numeric", hourCycle: "h12" }, "dayperiod"),
        month = (length, standalone) =>
          knownEnglish
            ? monthForDateTime(dt, length)
            : string(standalone ? { month: length } : { month: length, day: "numeric" }, "month"),
        weekday = (length, standalone) =>
          knownEnglish
            ? weekdayForDateTime(dt, length)
            : string(
                standalone ? { weekday: length } : { weekday: length, month: "long", day: "numeric" },
                "weekday"
              ),
        maybeMacro = (token) => {
          const formatOpts = Formatter.macroTokenToFormatOpts(token);
          if (formatOpts) {
            return this.formatWithSystemDefault(dt, formatOpts);
          } else {
            return token;
          }
        },
        era = (length) =>
          knownEnglish ? eraForDateTime(dt, length) : string({ era: length }, "era"),
        tokenToString = (token) => {
          // Where possible: http://cldr.unicode.org/translation/date-time-1/date-time#TOC-Standalone-vs.-Format-Styles
          switch (token) {
            // ms
            case "S":
              return this.num(dt.millisecond);
            case "u":
            // falls through
            case "SSS":
              return this.num(dt.millisecond, 3);
            // seconds
            case "s":
              return this.num(dt.second);
            case "ss":
              return this.num(dt.second, 2);
            // fractional seconds
            case "uu":
              return this.num(Math.floor(dt.millisecond / 10), 2);
            case "uuu":
              return this.num(Math.floor(dt.millisecond / 100));
            // minutes
            case "m":
              return this.num(dt.minute);
            case "mm":
              return this.num(dt.minute, 2);
            // hours
            case "h":
              return this.num(dt.hour % 12 === 0 ? 12 : dt.hour % 12);
            case "hh":
              return this.num(dt.hour % 12 === 0 ? 12 : dt.hour % 12, 2);
            case "H":
              return this.num(dt.hour);
            case "HH":
              return this.num(dt.hour, 2);
            // offset
            case "Z":
              // like +6
              return formatOffset({ format: "narrow", allowZ: this.opts.allowZ });
            case "ZZ":
              // like +06:00
              return formatOffset({ format: "short", allowZ: this.opts.allowZ });
            case "ZZZ":
              // like +0600
              return formatOffset({ format: "techie", allowZ: this.opts.allowZ });
            case "ZZZZ":
              // like EST
              return dt.zone.offsetName(dt.ts, { format: "short", locale: this.loc.locale });
            case "ZZZZZ":
              // like Eastern Standard Time
              return dt.zone.offsetName(dt.ts, { format: "long", locale: this.loc.locale });
            // zone
            case "z":
              // like America/New_York
              return dt.zoneName;
            // meridiems
            case "a":
              return meridiem();
            // dates
            case "d":
              return useDateTimeFormatter ? string({ day: "numeric" }, "day") : this.num(dt.day);
            case "dd":
              return useDateTimeFormatter ? string({ day: "2-digit" }, "day") : this.num(dt.day, 2);
            // weekdays - standalone
            case "c":
              // like 1
              return this.num(dt.weekday);
            case "ccc":
              // like 'Tues'
              return weekday("short", true);
            case "cccc":
              // like 'Tuesday'
              return weekday("long", true);
            case "ccccc":
              // like 'T'
              return weekday("narrow", true);
            // weekdays - format
            case "E":
              // like 1
              return this.num(dt.weekday);
            case "EEE":
              // like 'Tues'
              return weekday("short", false);
            case "EEEE":
              // like 'Tuesday'
              return weekday("long", false);
            case "EEEEE":
              // like 'T'
              return weekday("narrow", false);
            // months - standalone
            case "L":
              // like 1
              return useDateTimeFormatter
                ? string({ month: "numeric", day: "numeric" }, "month")
                : this.num(dt.month);
            case "LL":
              // like 01, doesn't seem to work
              return useDateTimeFormatter
                ? string({ month: "2-digit", day: "numeric" }, "month")
                : this.num(dt.month, 2);
            case "LLL":
              // like Jan
              return month("short", true);
            case "LLLL":
              // like January
              return month("long", true);
            case "LLLLL":
              // like J
              return month("narrow", true);
            // months - format
            case "M":
              // like 1
              return useDateTimeFormatter
                ? string({ month: "numeric" }, "month")
                : this.num(dt.month);
            case "MM":
              // like 01
              return useDateTimeFormatter
                ? string({ month: "2-digit" }, "month")
                : this.num(dt.month, 2);
            case "MMM":
              // like Jan
              return month("short", false);
            case "MMMM":
              // like January
              return month("long", false);
            case "MMMMM":
              // like J
              return month("narrow", false);
            // years
            case "y":
              // like 2014
              return useDateTimeFormatter ? string({ year: "numeric" }, "year") : this.num(dt.year);
            case "yy":
              // like 14
              return useDateTimeFormatter
                ? string({ year: "2-digit" }, "year")
                : this.num(dt.year.toString().slice(-2), 2);
            case "yyyy":
              // like 0012
              return useDateTimeFormatter
                ? string({ year: "numeric" }, "year")
                : this.num(dt.year, 4);
            case "yyyyyy":
              // like 000012
              return useDateTimeFormatter
                ? string({ year: "numeric" }, "year")
                : this.num(dt.year, 6);
            // eras
            case "G":
              // like AD
              return era("short");
            case "GG":
              // like Anno Domini
              return era("long");
            case "GGGGG":
              return era("narrow");
            case "kk":
              return this.num(dt.weekYear.toString().slice(-2), 2);
            case "kkkk":
              return this.num(dt.weekYear, 4);
            case "W":
              return this.num(dt.weekNumber);
            case "WW":
              return this.num(dt.weekNumber, 2);
            case "o":
              return this.num(dt.ordinal);
            case "ooo":
              return this.num(dt.ordinal, 3);
            case "q":
              // like 1
              return this.num(dt.quarter);
            case "qq":
              // like 01
              return this.num(dt.quarter, 2);
            case "X":
              return this.num(Math.floor(dt.ts / 1000));
            case "x":
              return this.num(dt.ts);
            default:
              return maybeMacro(token);
          }
        };

      return stringifyTokens(Formatter.parseFormat(fmt), tokenToString);
    }

    formatDurationFromString(dur, fmt) {
      const tokenToField = (token) => {
          switch (token[0]) {
            case "S":
              return "millisecond";
            case "s":
              return "second";
            case "m":
              return "minute";
            case "h":
              return "hour";
            case "d":
              return "day";
            case "w":
              return "week";
            case "M":
              return "month";
            case "y":
              return "year";
            default:
              return null;
          }
        },
        tokenToString = (lildur) => (token) => {
          const mapped = tokenToField(token);
          if (mapped) {
            return this.num(lildur.get(mapped), token.length);
          } else {
            return token;
          }
        },
        tokens = Formatter.parseFormat(fmt),
        realTokens = tokens.reduce(
          (found, { literal, val }) => (literal ? found : found.concat(val)),
          []
        ),
        collapsed = dur.shiftTo(...realTokens.map(tokenToField).filter((t) => t));
      return stringifyTokens(tokens, tokenToString(collapsed));
    }
  }

  class Invalid {
    constructor(reason, explanation) {
      this.reason = reason;
      this.explanation = explanation;
    }

    toMessage() {
      if (this.explanation) {
        return `${this.reason}: ${this.explanation}`;
      } else {
        return this.reason;
      }
    }
  }

  /*
   * This file handles parsing for well-specified formats. Here's how it works:
   * Two things go into parsing: a regex to match with and an extractor to take apart the groups in the match.
   * An extractor is just a function that takes a regex match array and returns a { year: ..., month: ... } object
   * parse() does the work of executing the regex and applying the extractor. It takes multiple regex/extractor pairs to try in sequence.
   * Extractors can take a "cursor" representing the offset in the match to look at. This makes it easy to combine extractors.
   * combineExtractors() does the work of combining them, keeping track of the cursor through multiple extractions.
   * Some extractions are super dumb and simpleParse and fromStrings help DRY them.
   */

  const ianaRegex = /[A-Za-z_+-]{1,256}(?::?\/[A-Za-z0-9_+-]{1,256}(?:\/[A-Za-z0-9_+-]{1,256})?)?/;

  function combineRegexes(...regexes) {
    const full = regexes.reduce((f, r) => f + r.source, "");
    return RegExp(`^${full}$`);
  }

  function combineExtractors(...extractors) {
    return (m) =>
      extractors
        .reduce(
          ([mergedVals, mergedZone, cursor], ex) => {
            const [val, zone, next] = ex(m, cursor);
            return [{ ...mergedVals, ...val }, zone || mergedZone, next];
          },
          [{}, null, 1]
        )
        .slice(0, 2);
  }

  function parse$1(s, ...patterns) {
    if (s == null) {
      return [null, null];
    }

    for (const [regex, extractor] of patterns) {
      const m = regex.exec(s);
      if (m) {
        return extractor(m);
      }
    }
    return [null, null];
  }

  function simpleParse(...keys) {
    return (match, cursor) => {
      const ret = {};
      let i;

      for (i = 0; i < keys.length; i++) {
        ret[keys[i]] = parseInteger(match[cursor + i]);
      }
      return [ret, null, cursor + i];
    };
  }

  // ISO and SQL parsing
  const offsetRegex = /(?:(Z)|([+-]\d\d)(?::?(\d\d))?)/;
  const isoExtendedZone = `(?:${offsetRegex.source}?(?:\\[(${ianaRegex.source})\\])?)?`;
  const isoTimeBaseRegex = /(\d\d)(?::?(\d\d)(?::?(\d\d)(?:[.,](\d{1,30}))?)?)?/;
  const isoTimeRegex = RegExp(`${isoTimeBaseRegex.source}${isoExtendedZone}`);
  const isoTimeExtensionRegex = RegExp(`(?:T${isoTimeRegex.source})?`);
  const isoYmdRegex = /([+-]\d{6}|\d{4})(?:-?(\d\d)(?:-?(\d\d))?)?/;
  const isoWeekRegex = /(\d{4})-?W(\d\d)(?:-?(\d))?/;
  const isoOrdinalRegex = /(\d{4})-?(\d{3})/;
  const extractISOWeekData = simpleParse("weekYear", "weekNumber", "weekDay");
  const extractISOOrdinalData = simpleParse("year", "ordinal");
  const sqlYmdRegex = /(\d{4})-(\d\d)-(\d\d)/; // dumbed-down version of the ISO one
  const sqlTimeRegex = RegExp(
    `${isoTimeBaseRegex.source} ?(?:${offsetRegex.source}|(${ianaRegex.source}))?`
  );
  const sqlTimeExtensionRegex = RegExp(`(?: ${sqlTimeRegex.source})?`);

  function int$1(match, pos, fallback) {
    const m = match[pos];
    return isUndefined(m) ? fallback : parseInteger(m);
  }

  function extractISOYmd(match, cursor) {
    const item = {
      year: int$1(match, cursor),
      month: int$1(match, cursor + 1, 1),
      day: int$1(match, cursor + 2, 1),
    };

    return [item, null, cursor + 3];
  }

  function extractISOTime(match, cursor) {
    const item = {
      hours: int$1(match, cursor, 0),
      minutes: int$1(match, cursor + 1, 0),
      seconds: int$1(match, cursor + 2, 0),
      milliseconds: parseMillis(match[cursor + 3]),
    };

    return [item, null, cursor + 4];
  }

  function extractISOOffset(match, cursor) {
    const local = !match[cursor] && !match[cursor + 1],
      fullOffset = signedOffset(match[cursor + 1], match[cursor + 2]),
      zone = local ? null : FixedOffsetZone.instance(fullOffset);
    return [{}, zone, cursor + 3];
  }

  function extractIANAZone(match, cursor) {
    const zone = match[cursor] ? IANAZone.create(match[cursor]) : null;
    return [{}, zone, cursor + 1];
  }

  // ISO time parsing

  const isoTimeOnly = RegExp(`^T?${isoTimeBaseRegex.source}$`);

  // ISO duration parsing

  const isoDuration =
    /^-?P(?:(?:(-?\d{1,20}(?:\.\d{1,20})?)Y)?(?:(-?\d{1,20}(?:\.\d{1,20})?)M)?(?:(-?\d{1,20}(?:\.\d{1,20})?)W)?(?:(-?\d{1,20}(?:\.\d{1,20})?)D)?(?:T(?:(-?\d{1,20}(?:\.\d{1,20})?)H)?(?:(-?\d{1,20}(?:\.\d{1,20})?)M)?(?:(-?\d{1,20})(?:[.,](-?\d{1,20}))?S)?)?)$/;

  function extractISODuration(match) {
    const [s, yearStr, monthStr, weekStr, dayStr, hourStr, minuteStr, secondStr, millisecondsStr] =
      match;

    const hasNegativePrefix = s[0] === "-";
    const negativeSeconds = secondStr && secondStr[0] === "-";

    const maybeNegate = (num, force = false) =>
      num !== undefined && (force || (num && hasNegativePrefix)) ? -num : num;

    return [
      {
        years: maybeNegate(parseFloating(yearStr)),
        months: maybeNegate(parseFloating(monthStr)),
        weeks: maybeNegate(parseFloating(weekStr)),
        days: maybeNegate(parseFloating(dayStr)),
        hours: maybeNegate(parseFloating(hourStr)),
        minutes: maybeNegate(parseFloating(minuteStr)),
        seconds: maybeNegate(parseFloating(secondStr), secondStr === "-0"),
        milliseconds: maybeNegate(parseMillis(millisecondsStr), negativeSeconds),
      },
    ];
  }

  // These are a little braindead. EDT *should* tell us that we're in, say, America/New_York
  // and not just that we're in -240 *right now*. But since I don't think these are used that often
  // I'm just going to ignore that
  const obsOffsets = {
    GMT: 0,
    EDT: -4 * 60,
    EST: -5 * 60,
    CDT: -5 * 60,
    CST: -6 * 60,
    MDT: -6 * 60,
    MST: -7 * 60,
    PDT: -7 * 60,
    PST: -8 * 60,
  };

  function fromStrings(weekdayStr, yearStr, monthStr, dayStr, hourStr, minuteStr, secondStr) {
    const result = {
      year: yearStr.length === 2 ? untruncateYear(parseInteger(yearStr)) : parseInteger(yearStr),
      month: monthsShort.indexOf(monthStr) + 1,
      day: parseInteger(dayStr),
      hour: parseInteger(hourStr),
      minute: parseInteger(minuteStr),
    };

    if (secondStr) result.second = parseInteger(secondStr);
    if (weekdayStr) {
      result.weekday =
        weekdayStr.length > 3
          ? weekdaysLong.indexOf(weekdayStr) + 1
          : weekdaysShort.indexOf(weekdayStr) + 1;
    }

    return result;
  }

  // RFC 2822/5322
  const rfc2822 =
    /^(?:(Mon|Tue|Wed|Thu|Fri|Sat|Sun),\s)?(\d{1,2})\s(Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec)\s(\d{2,4})\s(\d\d):(\d\d)(?::(\d\d))?\s(?:(UT|GMT|[ECMP][SD]T)|([Zz])|(?:([+-]\d\d)(\d\d)))$/;

  function extractRFC2822(match) {
    const [
        ,
        weekdayStr,
        dayStr,
        monthStr,
        yearStr,
        hourStr,
        minuteStr,
        secondStr,
        obsOffset,
        milOffset,
        offHourStr,
        offMinuteStr,
      ] = match,
      result = fromStrings(weekdayStr, yearStr, monthStr, dayStr, hourStr, minuteStr, secondStr);

    let offset;
    if (obsOffset) {
      offset = obsOffsets[obsOffset];
    } else if (milOffset) {
      offset = 0;
    } else {
      offset = signedOffset(offHourStr, offMinuteStr);
    }

    return [result, new FixedOffsetZone(offset)];
  }

  function preprocessRFC2822(s) {
    // Remove comments and folding whitespace and replace multiple-spaces with a single space
    return s
      .replace(/\([^()]*\)|[\n\t]/g, " ")
      .replace(/(\s\s+)/g, " ")
      .trim();
  }

  // http date

  const rfc1123 =
      /^(Mon|Tue|Wed|Thu|Fri|Sat|Sun), (\d\d) (Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec) (\d{4}) (\d\d):(\d\d):(\d\d) GMT$/,
    rfc850 =
      /^(Monday|Tuesday|Wednesday|Thursday|Friday|Saturday|Sunday), (\d\d)-(Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec)-(\d\d) (\d\d):(\d\d):(\d\d) GMT$/,
    ascii =
      /^(Mon|Tue|Wed|Thu|Fri|Sat|Sun) (Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec) ( \d|\d\d) (\d\d):(\d\d):(\d\d) (\d{4})$/;

  function extractRFC1123Or850(match) {
    const [, weekdayStr, dayStr, monthStr, yearStr, hourStr, minuteStr, secondStr] = match,
      result = fromStrings(weekdayStr, yearStr, monthStr, dayStr, hourStr, minuteStr, secondStr);
    return [result, FixedOffsetZone.utcInstance];
  }

  function extractASCII(match) {
    const [, weekdayStr, monthStr, dayStr, hourStr, minuteStr, secondStr, yearStr] = match,
      result = fromStrings(weekdayStr, yearStr, monthStr, dayStr, hourStr, minuteStr, secondStr);
    return [result, FixedOffsetZone.utcInstance];
  }

  const isoYmdWithTimeExtensionRegex = combineRegexes(isoYmdRegex, isoTimeExtensionRegex);
  const isoWeekWithTimeExtensionRegex = combineRegexes(isoWeekRegex, isoTimeExtensionRegex);
  const isoOrdinalWithTimeExtensionRegex = combineRegexes(isoOrdinalRegex, isoTimeExtensionRegex);
  const isoTimeCombinedRegex = combineRegexes(isoTimeRegex);

  const extractISOYmdTimeAndOffset = combineExtractors(
    extractISOYmd,
    extractISOTime,
    extractISOOffset,
    extractIANAZone
  );
  const extractISOWeekTimeAndOffset = combineExtractors(
    extractISOWeekData,
    extractISOTime,
    extractISOOffset,
    extractIANAZone
  );
  const extractISOOrdinalDateAndTime = combineExtractors(
    extractISOOrdinalData,
    extractISOTime,
    extractISOOffset,
    extractIANAZone
  );
  const extractISOTimeAndOffset = combineExtractors(
    extractISOTime,
    extractISOOffset,
    extractIANAZone
  );

  /*
   * @private
   */

  function parseISODate(s) {
    return parse$1(
      s,
      [isoYmdWithTimeExtensionRegex, extractISOYmdTimeAndOffset],
      [isoWeekWithTimeExtensionRegex, extractISOWeekTimeAndOffset],
      [isoOrdinalWithTimeExtensionRegex, extractISOOrdinalDateAndTime],
      [isoTimeCombinedRegex, extractISOTimeAndOffset]
    );
  }

  function parseRFC2822Date(s) {
    return parse$1(preprocessRFC2822(s), [rfc2822, extractRFC2822]);
  }

  function parseHTTPDate(s) {
    return parse$1(
      s,
      [rfc1123, extractRFC1123Or850],
      [rfc850, extractRFC1123Or850],
      [ascii, extractASCII]
    );
  }

  function parseISODuration(s) {
    return parse$1(s, [isoDuration, extractISODuration]);
  }

  const extractISOTimeOnly = combineExtractors(extractISOTime);

  function parseISOTimeOnly(s) {
    return parse$1(s, [isoTimeOnly, extractISOTimeOnly]);
  }

  const sqlYmdWithTimeExtensionRegex = combineRegexes(sqlYmdRegex, sqlTimeExtensionRegex);
  const sqlTimeCombinedRegex = combineRegexes(sqlTimeRegex);

  const extractISOTimeOffsetAndIANAZone = combineExtractors(
    extractISOTime,
    extractISOOffset,
    extractIANAZone
  );

  function parseSQL(s) {
    return parse$1(
      s,
      [sqlYmdWithTimeExtensionRegex, extractISOYmdTimeAndOffset],
      [sqlTimeCombinedRegex, extractISOTimeOffsetAndIANAZone]
    );
  }

  const INVALID$3 = "Invalid Duration";

  // unit conversion constants
  const lowOrderMatrix = {
      weeks: {
        days: 7,
        hours: 7 * 24,
        minutes: 7 * 24 * 60,
        seconds: 7 * 24 * 60 * 60,
        milliseconds: 7 * 24 * 60 * 60 * 1000,
      },
      days: {
        hours: 24,
        minutes: 24 * 60,
        seconds: 24 * 60 * 60,
        milliseconds: 24 * 60 * 60 * 1000,
      },
      hours: { minutes: 60, seconds: 60 * 60, milliseconds: 60 * 60 * 1000 },
      minutes: { seconds: 60, milliseconds: 60 * 1000 },
      seconds: { milliseconds: 1000 },
    },
    casualMatrix = {
      years: {
        quarters: 4,
        months: 12,
        weeks: 52,
        days: 365,
        hours: 365 * 24,
        minutes: 365 * 24 * 60,
        seconds: 365 * 24 * 60 * 60,
        milliseconds: 365 * 24 * 60 * 60 * 1000,
      },
      quarters: {
        months: 3,
        weeks: 13,
        days: 91,
        hours: 91 * 24,
        minutes: 91 * 24 * 60,
        seconds: 91 * 24 * 60 * 60,
        milliseconds: 91 * 24 * 60 * 60 * 1000,
      },
      months: {
        weeks: 4,
        days: 30,
        hours: 30 * 24,
        minutes: 30 * 24 * 60,
        seconds: 30 * 24 * 60 * 60,
        milliseconds: 30 * 24 * 60 * 60 * 1000,
      },

      ...lowOrderMatrix,
    },
    daysInYearAccurate = 146097.0 / 400,
    daysInMonthAccurate = 146097.0 / 4800,
    accurateMatrix = {
      years: {
        quarters: 4,
        months: 12,
        weeks: daysInYearAccurate / 7,
        days: daysInYearAccurate,
        hours: daysInYearAccurate * 24,
        minutes: daysInYearAccurate * 24 * 60,
        seconds: daysInYearAccurate * 24 * 60 * 60,
        milliseconds: daysInYearAccurate * 24 * 60 * 60 * 1000,
      },
      quarters: {
        months: 3,
        weeks: daysInYearAccurate / 28,
        days: daysInYearAccurate / 4,
        hours: (daysInYearAccurate * 24) / 4,
        minutes: (daysInYearAccurate * 24 * 60) / 4,
        seconds: (daysInYearAccurate * 24 * 60 * 60) / 4,
        milliseconds: (daysInYearAccurate * 24 * 60 * 60 * 1000) / 4,
      },
      months: {
        weeks: daysInMonthAccurate / 7,
        days: daysInMonthAccurate,
        hours: daysInMonthAccurate * 24,
        minutes: daysInMonthAccurate * 24 * 60,
        seconds: daysInMonthAccurate * 24 * 60 * 60,
        milliseconds: daysInMonthAccurate * 24 * 60 * 60 * 1000,
      },
      ...lowOrderMatrix,
    };

  // units ordered by size
  const orderedUnits$1 = [
    "years",
    "quarters",
    "months",
    "weeks",
    "days",
    "hours",
    "minutes",
    "seconds",
    "milliseconds",
  ];

  const reverseUnits = orderedUnits$1.slice(0).reverse();

  // clone really means "create another instance just like this one, but with these changes"
  function clone$2(dur, alts, clear = false) {
    // deep merge for vals
    const conf = {
      values: clear ? alts.values : { ...dur.values, ...(alts.values || {}) },
      loc: dur.loc.clone(alts.loc),
      conversionAccuracy: alts.conversionAccuracy || dur.conversionAccuracy,
      matrix: alts.matrix || dur.matrix,
    };
    return new Duration(conf);
  }

  function antiTrunc(n) {
    return n < 0 ? Math.floor(n) : Math.ceil(n);
  }

  // NB: mutates parameters
  function convert(matrix, fromMap, fromUnit, toMap, toUnit) {
    const conv = matrix[toUnit][fromUnit],
      raw = fromMap[fromUnit] / conv,
      sameSign = Math.sign(raw) === Math.sign(toMap[toUnit]),
      // ok, so this is wild, but see the matrix in the tests
      added =
        !sameSign && toMap[toUnit] !== 0 && Math.abs(raw) <= 1 ? antiTrunc(raw) : Math.trunc(raw);
    toMap[toUnit] += added;
    fromMap[fromUnit] -= added * conv;
  }

  // NB: mutates parameters
  function normalizeValues(matrix, vals) {
    reverseUnits.reduce((previous, current) => {
      if (!isUndefined(vals[current])) {
        if (previous) {
          convert(matrix, vals, previous, vals, current);
        }
        return current;
      } else {
        return previous;
      }
    }, null);
  }

  // Remove all properties with a value of 0 from an object
  function removeZeroes(vals) {
    const newVals = {};
    for (const [key, value] of Object.entries(vals)) {
      if (value !== 0) {
        newVals[key] = value;
      }
    }
    return newVals;
  }

  /**
   * A Duration object represents a period of time, like "2 months" or "1 day, 1 hour". Conceptually, it's just a map of units to their quantities, accompanied by some additional configuration and methods for creating, parsing, interrogating, transforming, and formatting them. They can be used on their own or in conjunction with other Luxon types; for example, you can use {@link DateTime#plus} to add a Duration object to a DateTime, producing another DateTime.
   *
   * Here is a brief overview of commonly used methods and getters in Duration:
   *
   * * **Creation** To create a Duration, use {@link Duration.fromMillis}, {@link Duration.fromObject}, or {@link Duration.fromISO}.
   * * **Unit values** See the {@link Duration#years}, {@link Duration#months}, {@link Duration#weeks}, {@link Duration#days}, {@link Duration#hours}, {@link Duration#minutes}, {@link Duration#seconds}, {@link Duration#milliseconds} accessors.
   * * **Configuration** See  {@link Duration#locale} and {@link Duration#numberingSystem} accessors.
   * * **Transformation** To create new Durations out of old ones use {@link Duration#plus}, {@link Duration#minus}, {@link Duration#normalize}, {@link Duration#set}, {@link Duration#reconfigure}, {@link Duration#shiftTo}, and {@link Duration#negate}.
   * * **Output** To convert the Duration into other representations, see {@link Duration#as}, {@link Duration#toISO}, {@link Duration#toFormat}, and {@link Duration#toJSON}
   *
   * There's are more methods documented below. In addition, for more information on subtler topics like internationalization and validity, see the external documentation.
   */
  class Duration {
    /**
     * @private
     */
    constructor(config) {
      const accurate = config.conversionAccuracy === "longterm" || false;
      let matrix = accurate ? accurateMatrix : casualMatrix;

      if (config.matrix) {
        matrix = config.matrix;
      }

      /**
       * @access private
       */
      this.values = config.values;
      /**
       * @access private
       */
      this.loc = config.loc || Locale.create();
      /**
       * @access private
       */
      this.conversionAccuracy = accurate ? "longterm" : "casual";
      /**
       * @access private
       */
      this.invalid = config.invalid || null;
      /**
       * @access private
       */
      this.matrix = matrix;
      /**
       * @access private
       */
      this.isLuxonDuration = true;
    }

    /**
     * Create Duration from a number of milliseconds.
     * @param {number} count of milliseconds
     * @param {Object} opts - options for parsing
     * @param {string} [opts.locale='en-US'] - the locale to use
     * @param {string} opts.numberingSystem - the numbering system to use
     * @param {string} [opts.conversionAccuracy='casual'] - the conversion system to use
     * @return {Duration}
     */
    static fromMillis(count, opts) {
      return Duration.fromObject({ milliseconds: count }, opts);
    }

    /**
     * Create a Duration from a JavaScript object with keys like 'years' and 'hours'.
     * If this object is empty then a zero milliseconds duration is returned.
     * @param {Object} obj - the object to create the DateTime from
     * @param {number} obj.years
     * @param {number} obj.quarters
     * @param {number} obj.months
     * @param {number} obj.weeks
     * @param {number} obj.days
     * @param {number} obj.hours
     * @param {number} obj.minutes
     * @param {number} obj.seconds
     * @param {number} obj.milliseconds
     * @param {Object} [opts=[]] - options for creating this Duration
     * @param {string} [opts.locale='en-US'] - the locale to use
     * @param {string} opts.numberingSystem - the numbering system to use
     * @param {string} [opts.conversionAccuracy='casual'] - the preset conversion system to use
     * @param {string} [opts.matrix=Object] - the custom conversion system to use
     * @return {Duration}
     */
    static fromObject(obj, opts = {}) {
      if (obj == null || typeof obj !== "object") {
        throw new InvalidArgumentError(
          `Duration.fromObject: argument expected to be an object, got ${
          obj === null ? "null" : typeof obj
        }`
        );
      }

      return new Duration({
        values: normalizeObject(obj, Duration.normalizeUnit),
        loc: Locale.fromObject(opts),
        conversionAccuracy: opts.conversionAccuracy,
        matrix: opts.matrix,
      });
    }

    /**
     * Create a Duration from DurationLike.
     *
     * @param {Object | number | Duration} durationLike
     * One of:
     * - object with keys like 'years' and 'hours'.
     * - number representing milliseconds
     * - Duration instance
     * @return {Duration}
     */
    static fromDurationLike(durationLike) {
      if (isNumber$2(durationLike)) {
        return Duration.fromMillis(durationLike);
      } else if (Duration.isDuration(durationLike)) {
        return durationLike;
      } else if (typeof durationLike === "object") {
        return Duration.fromObject(durationLike);
      } else {
        throw new InvalidArgumentError(
          `Unknown duration argument ${durationLike} of type ${typeof durationLike}`
        );
      }
    }

    /**
     * Create a Duration from an ISO 8601 duration string.
     * @param {string} text - text to parse
     * @param {Object} opts - options for parsing
     * @param {string} [opts.locale='en-US'] - the locale to use
     * @param {string} opts.numberingSystem - the numbering system to use
     * @param {string} [opts.conversionAccuracy='casual'] - the preset conversion system to use
     * @param {string} [opts.matrix=Object] - the preset conversion system to use
     * @see https://en.wikipedia.org/wiki/ISO_8601#Durations
     * @example Duration.fromISO('P3Y6M1W4DT12H30M5S').toObject() //=> { years: 3, months: 6, weeks: 1, days: 4, hours: 12, minutes: 30, seconds: 5 }
     * @example Duration.fromISO('PT23H').toObject() //=> { hours: 23 }
     * @example Duration.fromISO('P5Y3M').toObject() //=> { years: 5, months: 3 }
     * @return {Duration}
     */
    static fromISO(text, opts) {
      const [parsed] = parseISODuration(text);
      if (parsed) {
        return Duration.fromObject(parsed, opts);
      } else {
        return Duration.invalid("unparsable", `the input "${text}" can't be parsed as ISO 8601`);
      }
    }

    /**
     * Create a Duration from an ISO 8601 time string.
     * @param {string} text - text to parse
     * @param {Object} opts - options for parsing
     * @param {string} [opts.locale='en-US'] - the locale to use
     * @param {string} opts.numberingSystem - the numbering system to use
     * @param {string} [opts.conversionAccuracy='casual'] - the preset conversion system to use
     * @param {string} [opts.matrix=Object] - the conversion system to use
     * @see https://en.wikipedia.org/wiki/ISO_8601#Times
     * @example Duration.fromISOTime('11:22:33.444').toObject() //=> { hours: 11, minutes: 22, seconds: 33, milliseconds: 444 }
     * @example Duration.fromISOTime('11:00').toObject() //=> { hours: 11, minutes: 0, seconds: 0 }
     * @example Duration.fromISOTime('T11:00').toObject() //=> { hours: 11, minutes: 0, seconds: 0 }
     * @example Duration.fromISOTime('1100').toObject() //=> { hours: 11, minutes: 0, seconds: 0 }
     * @example Duration.fromISOTime('T1100').toObject() //=> { hours: 11, minutes: 0, seconds: 0 }
     * @return {Duration}
     */
    static fromISOTime(text, opts) {
      const [parsed] = parseISOTimeOnly(text);
      if (parsed) {
        return Duration.fromObject(parsed, opts);
      } else {
        return Duration.invalid("unparsable", `the input "${text}" can't be parsed as ISO 8601`);
      }
    }

    /**
     * Create an invalid Duration.
     * @param {string} reason - simple string of why this datetime is invalid. Should not contain parameters or anything else data-dependent
     * @param {string} [explanation=null] - longer explanation, may include parameters and other useful debugging information
     * @return {Duration}
     */
    static invalid(reason, explanation = null) {
      if (!reason) {
        throw new InvalidArgumentError("need to specify a reason the Duration is invalid");
      }

      const invalid = reason instanceof Invalid ? reason : new Invalid(reason, explanation);

      if (Settings.throwOnInvalid) {
        throw new InvalidDurationError(invalid);
      } else {
        return new Duration({ invalid });
      }
    }

    /**
     * @private
     */
    static normalizeUnit(unit) {
      const normalized = {
        year: "years",
        years: "years",
        quarter: "quarters",
        quarters: "quarters",
        month: "months",
        months: "months",
        week: "weeks",
        weeks: "weeks",
        day: "days",
        days: "days",
        hour: "hours",
        hours: "hours",
        minute: "minutes",
        minutes: "minutes",
        second: "seconds",
        seconds: "seconds",
        millisecond: "milliseconds",
        milliseconds: "milliseconds",
      }[unit ? unit.toLowerCase() : unit];

      if (!normalized) throw new InvalidUnitError(unit);

      return normalized;
    }

    /**
     * Check if an object is a Duration. Works across context boundaries
     * @param {object} o
     * @return {boolean}
     */
    static isDuration(o) {
      return (o && o.isLuxonDuration) || false;
    }

    /**
     * Get  the locale of a Duration, such 'en-GB'
     * @type {string}
     */
    get locale() {
      return this.isValid ? this.loc.locale : null;
    }

    /**
     * Get the numbering system of a Duration, such 'beng'. The numbering system is used when formatting the Duration
     *
     * @type {string}
     */
    get numberingSystem() {
      return this.isValid ? this.loc.numberingSystem : null;
    }

    /**
     * Returns a string representation of this Duration formatted according to the specified format string. You may use these tokens:
     * * `S` for milliseconds
     * * `s` for seconds
     * * `m` for minutes
     * * `h` for hours
     * * `d` for days
     * * `w` for weeks
     * * `M` for months
     * * `y` for years
     * Notes:
     * * Add padding by repeating the token, e.g. "yy" pads the years to two digits, "hhhh" pads the hours out to four digits
     * * Tokens can be escaped by wrapping with single quotes.
     * * The duration will be converted to the set of units in the format string using {@link Duration#shiftTo} and the Durations's conversion accuracy setting.
     * @param {string} fmt - the format string
     * @param {Object} opts - options
     * @param {boolean} [opts.floor=true] - floor numerical values
     * @example Duration.fromObject({ years: 1, days: 6, seconds: 2 }).toFormat("y d s") //=> "1 6 2"
     * @example Duration.fromObject({ years: 1, days: 6, seconds: 2 }).toFormat("yy dd sss") //=> "01 06 002"
     * @example Duration.fromObject({ years: 1, days: 6, seconds: 2 }).toFormat("M S") //=> "12 518402000"
     * @return {string}
     */
    toFormat(fmt, opts = {}) {
      // reverse-compat since 1.2; we always round down now, never up, and we do it by default
      const fmtOpts = {
        ...opts,
        floor: opts.round !== false && opts.floor !== false,
      };
      return this.isValid
        ? Formatter.create(this.loc, fmtOpts).formatDurationFromString(this, fmt)
        : INVALID$3;
    }

    /**
     * Returns a string representation of a Duration with all units included.
     * To modify its behavior use the `listStyle` and any Intl.NumberFormat option, though `unitDisplay` is especially relevant.
     * @see https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Intl/NumberFormat
     * @param opts - On option object to override the formatting. Accepts the same keys as the options parameter of the native `Int.NumberFormat` constructor, as well as `listStyle`.
     * @example
     * ```js
     * var dur = Duration.fromObject({ days: 1, hours: 5, minutes: 6 })
     * dur.toHuman() //=> '1 day, 5 hours, 6 minutes'
     * dur.toHuman({ listStyle: "long" }) //=> '1 day, 5 hours, and 6 minutes'
     * dur.toHuman({ unitDisplay: "short" }) //=> '1 day, 5 hr, 6 min'
     * ```
     */
    toHuman(opts = {}) {
      const l = orderedUnits$1
        .map((unit) => {
          const val = this.values[unit];
          if (isUndefined(val)) {
            return null;
          }
          return this.loc
            .numberFormatter({ style: "unit", unitDisplay: "long", ...opts, unit: unit.slice(0, -1) })
            .format(val);
        })
        .filter((n) => n);

      return this.loc
        .listFormatter({ type: "conjunction", style: opts.listStyle || "narrow", ...opts })
        .format(l);
    }

    /**
     * Returns a JavaScript object with this Duration's values.
     * @example Duration.fromObject({ years: 1, days: 6, seconds: 2 }).toObject() //=> { years: 1, days: 6, seconds: 2 }
     * @return {Object}
     */
    toObject() {
      if (!this.isValid) return {};
      return { ...this.values };
    }

    /**
     * Returns an ISO 8601-compliant string representation of this Duration.
     * @see https://en.wikipedia.org/wiki/ISO_8601#Durations
     * @example Duration.fromObject({ years: 3, seconds: 45 }).toISO() //=> 'P3YT45S'
     * @example Duration.fromObject({ months: 4, seconds: 45 }).toISO() //=> 'P4MT45S'
     * @example Duration.fromObject({ months: 5 }).toISO() //=> 'P5M'
     * @example Duration.fromObject({ minutes: 5 }).toISO() //=> 'PT5M'
     * @example Duration.fromObject({ milliseconds: 6 }).toISO() //=> 'PT0.006S'
     * @return {string}
     */
    toISO() {
      // we could use the formatter, but this is an easier way to get the minimum string
      if (!this.isValid) return null;

      let s = "P";
      if (this.years !== 0) s += this.years + "Y";
      if (this.months !== 0 || this.quarters !== 0) s += this.months + this.quarters * 3 + "M";
      if (this.weeks !== 0) s += this.weeks + "W";
      if (this.days !== 0) s += this.days + "D";
      if (this.hours !== 0 || this.minutes !== 0 || this.seconds !== 0 || this.milliseconds !== 0)
        s += "T";
      if (this.hours !== 0) s += this.hours + "H";
      if (this.minutes !== 0) s += this.minutes + "M";
      if (this.seconds !== 0 || this.milliseconds !== 0)
        // this will handle "floating point madness" by removing extra decimal places
        // https://stackoverflow.com/questions/588004/is-floating-point-math-broken
        s += roundTo(this.seconds + this.milliseconds / 1000, 3) + "S";
      if (s === "P") s += "T0S";
      return s;
    }

    /**
     * Returns an ISO 8601-compliant string representation of this Duration, formatted as a time of day.
     * Note that this will return null if the duration is invalid, negative, or equal to or greater than 24 hours.
     * @see https://en.wikipedia.org/wiki/ISO_8601#Times
     * @param {Object} opts - options
     * @param {boolean} [opts.suppressMilliseconds=false] - exclude milliseconds from the format if they're 0
     * @param {boolean} [opts.suppressSeconds=false] - exclude seconds from the format if they're 0
     * @param {boolean} [opts.includePrefix=false] - include the `T` prefix
     * @param {string} [opts.format='extended'] - choose between the basic and extended format
     * @example Duration.fromObject({ hours: 11 }).toISOTime() //=> '11:00:00.000'
     * @example Duration.fromObject({ hours: 11 }).toISOTime({ suppressMilliseconds: true }) //=> '11:00:00'
     * @example Duration.fromObject({ hours: 11 }).toISOTime({ suppressSeconds: true }) //=> '11:00'
     * @example Duration.fromObject({ hours: 11 }).toISOTime({ includePrefix: true }) //=> 'T11:00:00.000'
     * @example Duration.fromObject({ hours: 11 }).toISOTime({ format: 'basic' }) //=> '110000.000'
     * @return {string}
     */
    toISOTime(opts = {}) {
      if (!this.isValid) return null;

      const millis = this.toMillis();
      if (millis < 0 || millis >= 86400000) return null;

      opts = {
        suppressMilliseconds: false,
        suppressSeconds: false,
        includePrefix: false,
        format: "extended",
        ...opts,
      };

      const value = this.shiftTo("hours", "minutes", "seconds", "milliseconds");

      let fmt = opts.format === "basic" ? "hhmm" : "hh:mm";

      if (!opts.suppressSeconds || value.seconds !== 0 || value.milliseconds !== 0) {
        fmt += opts.format === "basic" ? "ss" : ":ss";
        if (!opts.suppressMilliseconds || value.milliseconds !== 0) {
          fmt += ".SSS";
        }
      }

      let str = value.toFormat(fmt);

      if (opts.includePrefix) {
        str = "T" + str;
      }

      return str;
    }

    /**
     * Returns an ISO 8601 representation of this Duration appropriate for use in JSON.
     * @return {string}
     */
    toJSON() {
      return this.toISO();
    }

    /**
     * Returns an ISO 8601 representation of this Duration appropriate for use in debugging.
     * @return {string}
     */
    toString() {
      return this.toISO();
    }

    /**
     * Returns an milliseconds value of this Duration.
     * @return {number}
     */
    toMillis() {
      return this.as("milliseconds");
    }

    /**
     * Returns an milliseconds value of this Duration. Alias of {@link toMillis}
     * @return {number}
     */
    valueOf() {
      return this.toMillis();
    }

    /**
     * Make this Duration longer by the specified amount. Return a newly-constructed Duration.
     * @param {Duration|Object|number} duration - The amount to add. Either a Luxon Duration, a number of milliseconds, the object argument to Duration.fromObject()
     * @return {Duration}
     */
    plus(duration) {
      if (!this.isValid) return this;

      const dur = Duration.fromDurationLike(duration),
        result = {};

      for (const k of orderedUnits$1) {
        if (hasOwnProperty(dur.values, k) || hasOwnProperty(this.values, k)) {
          result[k] = dur.get(k) + this.get(k);
        }
      }

      return clone$2(this, { values: result }, true);
    }

    /**
     * Make this Duration shorter by the specified amount. Return a newly-constructed Duration.
     * @param {Duration|Object|number} duration - The amount to subtract. Either a Luxon Duration, a number of milliseconds, the object argument to Duration.fromObject()
     * @return {Duration}
     */
    minus(duration) {
      if (!this.isValid) return this;

      const dur = Duration.fromDurationLike(duration);
      return this.plus(dur.negate());
    }

    /**
     * Scale this Duration by the specified amount. Return a newly-constructed Duration.
     * @param {function} fn - The function to apply to each unit. Arity is 1 or 2: the value of the unit and, optionally, the unit name. Must return a number.
     * @example Duration.fromObject({ hours: 1, minutes: 30 }).mapUnits(x => x * 2) //=> { hours: 2, minutes: 60 }
     * @example Duration.fromObject({ hours: 1, minutes: 30 }).mapUnits((x, u) => u === "hours" ? x * 2 : x) //=> { hours: 2, minutes: 30 }
     * @return {Duration}
     */
    mapUnits(fn) {
      if (!this.isValid) return this;
      const result = {};
      for (const k of Object.keys(this.values)) {
        result[k] = asNumber(fn(this.values[k], k));
      }
      return clone$2(this, { values: result }, true);
    }

    /**
     * Get the value of unit.
     * @param {string} unit - a unit such as 'minute' or 'day'
     * @example Duration.fromObject({years: 2, days: 3}).get('years') //=> 2
     * @example Duration.fromObject({years: 2, days: 3}).get('months') //=> 0
     * @example Duration.fromObject({years: 2, days: 3}).get('days') //=> 3
     * @return {number}
     */
    get(unit) {
      return this[Duration.normalizeUnit(unit)];
    }

    /**
     * "Set" the values of specified units. Return a newly-constructed Duration.
     * @param {Object} values - a mapping of units to numbers
     * @example dur.set({ years: 2017 })
     * @example dur.set({ hours: 8, minutes: 30 })
     * @return {Duration}
     */
    set(values) {
      if (!this.isValid) return this;

      const mixed = { ...this.values, ...normalizeObject(values, Duration.normalizeUnit) };
      return clone$2(this, { values: mixed });
    }

    /**
     * "Set" the locale and/or numberingSystem.  Returns a newly-constructed Duration.
     * @example dur.reconfigure({ locale: 'en-GB' })
     * @return {Duration}
     */
    reconfigure({ locale, numberingSystem, conversionAccuracy, matrix } = {}) {
      const loc = this.loc.clone({ locale, numberingSystem });
      const opts = { loc, matrix, conversionAccuracy };
      return clone$2(this, opts);
    }

    /**
     * Return the length of the duration in the specified unit.
     * @param {string} unit - a unit such as 'minutes' or 'days'
     * @example Duration.fromObject({years: 1}).as('days') //=> 365
     * @example Duration.fromObject({years: 1}).as('months') //=> 12
     * @example Duration.fromObject({hours: 60}).as('days') //=> 2.5
     * @return {number}
     */
    as(unit) {
      return this.isValid ? this.shiftTo(unit).get(unit) : NaN;
    }

    /**
     * Reduce this Duration to its canonical representation in its current units.
     * @example Duration.fromObject({ years: 2, days: 5000 }).normalize().toObject() //=> { years: 15, days: 255 }
     * @example Duration.fromObject({ hours: 12, minutes: -45 }).normalize().toObject() //=> { hours: 11, minutes: 15 }
     * @return {Duration}
     */
    normalize() {
      if (!this.isValid) return this;
      const vals = this.toObject();
      normalizeValues(this.matrix, vals);
      return clone$2(this, { values: vals }, true);
    }

    /**
     * Rescale units to its largest representation
     * @example Duration.fromObject({ milliseconds: 90000 }).rescale().toObject() //=> { minutes: 1, seconds: 30 }
     * @return {Duration}
     */
    rescale() {
      if (!this.isValid) return this;
      const vals = removeZeroes(this.normalize().shiftToAll().toObject());
      return clone$2(this, { values: vals }, true);
    }

    /**
     * Convert this Duration into its representation in a different set of units.
     * @example Duration.fromObject({ hours: 1, seconds: 30 }).shiftTo('minutes', 'milliseconds').toObject() //=> { minutes: 60, milliseconds: 30000 }
     * @return {Duration}
     */
    shiftTo(...units) {
      if (!this.isValid) return this;

      if (units.length === 0) {
        return this;
      }

      units = units.map((u) => Duration.normalizeUnit(u));

      const built = {},
        accumulated = {},
        vals = this.toObject();
      let lastUnit;

      for (const k of orderedUnits$1) {
        if (units.indexOf(k) >= 0) {
          lastUnit = k;

          let own = 0;

          // anything we haven't boiled down yet should get boiled to this unit
          for (const ak in accumulated) {
            own += this.matrix[ak][k] * accumulated[ak];
            accumulated[ak] = 0;
          }

          // plus anything that's already in this unit
          if (isNumber$2(vals[k])) {
            own += vals[k];
          }

          const i = Math.trunc(own);
          built[k] = i;
          accumulated[k] = (own * 1000 - i * 1000) / 1000;

          // plus anything further down the chain that should be rolled up in to this
          for (const down in vals) {
            if (orderedUnits$1.indexOf(down) > orderedUnits$1.indexOf(k)) {
              convert(this.matrix, vals, down, built, k);
            }
          }
          // otherwise, keep it in the wings to boil it later
        } else if (isNumber$2(vals[k])) {
          accumulated[k] = vals[k];
        }
      }

      // anything leftover becomes the decimal for the last unit
      // lastUnit must be defined since units is not empty
      for (const key in accumulated) {
        if (accumulated[key] !== 0) {
          built[lastUnit] +=
            key === lastUnit ? accumulated[key] : accumulated[key] / this.matrix[lastUnit][key];
        }
      }

      return clone$2(this, { values: built }, true).normalize();
    }

    /**
     * Shift this Duration to all available units.
     * Same as shiftTo("years", "months", "weeks", "days", "hours", "minutes", "seconds", "milliseconds")
     * @return {Duration}
     */
    shiftToAll() {
      if (!this.isValid) return this;
      return this.shiftTo(
        "years",
        "months",
        "weeks",
        "days",
        "hours",
        "minutes",
        "seconds",
        "milliseconds"
      );
    }

    /**
     * Return the negative of this Duration.
     * @example Duration.fromObject({ hours: 1, seconds: 30 }).negate().toObject() //=> { hours: -1, seconds: -30 }
     * @return {Duration}
     */
    negate() {
      if (!this.isValid) return this;
      const negated = {};
      for (const k of Object.keys(this.values)) {
        negated[k] = this.values[k] === 0 ? 0 : -this.values[k];
      }
      return clone$2(this, { values: negated }, true);
    }

    /**
     * Get the years.
     * @type {number}
     */
    get years() {
      return this.isValid ? this.values.years || 0 : NaN;
    }

    /**
     * Get the quarters.
     * @type {number}
     */
    get quarters() {
      return this.isValid ? this.values.quarters || 0 : NaN;
    }

    /**
     * Get the months.
     * @type {number}
     */
    get months() {
      return this.isValid ? this.values.months || 0 : NaN;
    }

    /**
     * Get the weeks
     * @type {number}
     */
    get weeks() {
      return this.isValid ? this.values.weeks || 0 : NaN;
    }

    /**
     * Get the days.
     * @type {number}
     */
    get days() {
      return this.isValid ? this.values.days || 0 : NaN;
    }

    /**
     * Get the hours.
     * @type {number}
     */
    get hours() {
      return this.isValid ? this.values.hours || 0 : NaN;
    }

    /**
     * Get the minutes.
     * @type {number}
     */
    get minutes() {
      return this.isValid ? this.values.minutes || 0 : NaN;
    }

    /**
     * Get the seconds.
     * @return {number}
     */
    get seconds() {
      return this.isValid ? this.values.seconds || 0 : NaN;
    }

    /**
     * Get the milliseconds.
     * @return {number}
     */
    get milliseconds() {
      return this.isValid ? this.values.milliseconds || 0 : NaN;
    }

    /**
     * Returns whether the Duration is invalid. Invalid durations are returned by diff operations
     * on invalid DateTimes or Intervals.
     * @return {boolean}
     */
    get isValid() {
      return this.invalid === null;
    }

    /**
     * Returns an error code if this Duration became invalid, or null if the Duration is valid
     * @return {string}
     */
    get invalidReason() {
      return this.invalid ? this.invalid.reason : null;
    }

    /**
     * Returns an explanation of why this Duration became invalid, or null if the Duration is valid
     * @type {string}
     */
    get invalidExplanation() {
      return this.invalid ? this.invalid.explanation : null;
    }

    /**
     * Equality check
     * Two Durations are equal iff they have the same units and the same values for each unit.
     * @param {Duration} other
     * @return {boolean}
     */
    equals(other) {
      if (!this.isValid || !other.isValid) {
        return false;
      }

      if (!this.loc.equals(other.loc)) {
        return false;
      }

      function eq(v1, v2) {
        // Consider 0 and undefined as equal
        if (v1 === undefined || v1 === 0) return v2 === undefined || v2 === 0;
        return v1 === v2;
      }

      for (const u of orderedUnits$1) {
        if (!eq(this.values[u], other.values[u])) {
          return false;
        }
      }
      return true;
    }
  }

  const INVALID$2 = "Invalid Interval";

  // checks if the start is equal to or before the end
  function validateStartEnd(start, end) {
    if (!start || !start.isValid) {
      return Interval.invalid("missing or invalid start");
    } else if (!end || !end.isValid) {
      return Interval.invalid("missing or invalid end");
    } else if (end < start) {
      return Interval.invalid(
        "end before start",
        `The end of an interval must be after its start, but you had start=${start.toISO()} and end=${end.toISO()}`
      );
    } else {
      return null;
    }
  }

  /**
   * An Interval object represents a half-open interval of time, where each endpoint is a {@link DateTime}. Conceptually, it's a container for those two endpoints, accompanied by methods for creating, parsing, interrogating, comparing, transforming, and formatting them.
   *
   * Here is a brief overview of the most commonly used methods and getters in Interval:
   *
   * * **Creation** To create an Interval, use {@link Interval.fromDateTimes}, {@link Interval.after}, {@link Interval.before}, or {@link Interval.fromISO}.
   * * **Accessors** Use {@link Interval#start} and {@link Interval#end} to get the start and end.
   * * **Interrogation** To analyze the Interval, use {@link Interval#count}, {@link Interval#length}, {@link Interval#hasSame}, {@link Interval#contains}, {@link Interval#isAfter}, or {@link Interval#isBefore}.
   * * **Transformation** To create other Intervals out of this one, use {@link Interval#set}, {@link Interval#splitAt}, {@link Interval#splitBy}, {@link Interval#divideEqually}, {@link Interval.merge}, {@link Interval.xor}, {@link Interval#union}, {@link Interval#intersection}, or {@link Interval#difference}.
   * * **Comparison** To compare this Interval to another one, use {@link Interval#equals}, {@link Interval#overlaps}, {@link Interval#abutsStart}, {@link Interval#abutsEnd}, {@link Interval#engulfs}
   * * **Output** To convert the Interval into other representations, see {@link Interval#toString}, {@link Interval#toLocaleString}, {@link Interval#toISO}, {@link Interval#toISODate}, {@link Interval#toISOTime}, {@link Interval#toFormat}, and {@link Interval#toDuration}.
   */
  class Interval {
    /**
     * @private
     */
    constructor(config) {
      /**
       * @access private
       */
      this.s = config.start;
      /**
       * @access private
       */
      this.e = config.end;
      /**
       * @access private
       */
      this.invalid = config.invalid || null;
      /**
       * @access private
       */
      this.isLuxonInterval = true;
    }

    /**
     * Create an invalid Interval.
     * @param {string} reason - simple string of why this Interval is invalid. Should not contain parameters or anything else data-dependent
     * @param {string} [explanation=null] - longer explanation, may include parameters and other useful debugging information
     * @return {Interval}
     */
    static invalid(reason, explanation = null) {
      if (!reason) {
        throw new InvalidArgumentError("need to specify a reason the Interval is invalid");
      }

      const invalid = reason instanceof Invalid ? reason : new Invalid(reason, explanation);

      if (Settings.throwOnInvalid) {
        throw new InvalidIntervalError(invalid);
      } else {
        return new Interval({ invalid });
      }
    }

    /**
     * Create an Interval from a start DateTime and an end DateTime. Inclusive of the start but not the end.
     * @param {DateTime|Date|Object} start
     * @param {DateTime|Date|Object} end
     * @return {Interval}
     */
    static fromDateTimes(start, end) {
      const builtStart = friendlyDateTime(start),
        builtEnd = friendlyDateTime(end);

      const validateError = validateStartEnd(builtStart, builtEnd);

      if (validateError == null) {
        return new Interval({
          start: builtStart,
          end: builtEnd,
        });
      } else {
        return validateError;
      }
    }

    /**
     * Create an Interval from a start DateTime and a Duration to extend to.
     * @param {DateTime|Date|Object} start
     * @param {Duration|Object|number} duration - the length of the Interval.
     * @return {Interval}
     */
    static after(start, duration) {
      const dur = Duration.fromDurationLike(duration),
        dt = friendlyDateTime(start);
      return Interval.fromDateTimes(dt, dt.plus(dur));
    }

    /**
     * Create an Interval from an end DateTime and a Duration to extend backwards to.
     * @param {DateTime|Date|Object} end
     * @param {Duration|Object|number} duration - the length of the Interval.
     * @return {Interval}
     */
    static before(end, duration) {
      const dur = Duration.fromDurationLike(duration),
        dt = friendlyDateTime(end);
      return Interval.fromDateTimes(dt.minus(dur), dt);
    }

    /**
     * Create an Interval from an ISO 8601 string.
     * Accepts `<start>/<end>`, `<start>/<duration>`, and `<duration>/<end>` formats.
     * @param {string} text - the ISO string to parse
     * @param {Object} [opts] - options to pass {@link DateTime#fromISO} and optionally {@link Duration#fromISO}
     * @see https://en.wikipedia.org/wiki/ISO_8601#Time_intervals
     * @return {Interval}
     */
    static fromISO(text, opts) {
      const [s, e] = (text || "").split("/", 2);
      if (s && e) {
        let start, startIsValid;
        try {
          start = DateTime.fromISO(s, opts);
          startIsValid = start.isValid;
        } catch (e) {
          startIsValid = false;
        }

        let end, endIsValid;
        try {
          end = DateTime.fromISO(e, opts);
          endIsValid = end.isValid;
        } catch (e) {
          endIsValid = false;
        }

        if (startIsValid && endIsValid) {
          return Interval.fromDateTimes(start, end);
        }

        if (startIsValid) {
          const dur = Duration.fromISO(e, opts);
          if (dur.isValid) {
            return Interval.after(start, dur);
          }
        } else if (endIsValid) {
          const dur = Duration.fromISO(s, opts);
          if (dur.isValid) {
            return Interval.before(end, dur);
          }
        }
      }
      return Interval.invalid("unparsable", `the input "${text}" can't be parsed as ISO 8601`);
    }

    /**
     * Check if an object is an Interval. Works across context boundaries
     * @param {object} o
     * @return {boolean}
     */
    static isInterval(o) {
      return (o && o.isLuxonInterval) || false;
    }

    /**
     * Returns the start of the Interval
     * @type {DateTime}
     */
    get start() {
      return this.isValid ? this.s : null;
    }

    /**
     * Returns the end of the Interval
     * @type {DateTime}
     */
    get end() {
      return this.isValid ? this.e : null;
    }

    /**
     * Returns whether this Interval's end is at least its start, meaning that the Interval isn't 'backwards'.
     * @type {boolean}
     */
    get isValid() {
      return this.invalidReason === null;
    }

    /**
     * Returns an error code if this Interval is invalid, or null if the Interval is valid
     * @type {string}
     */
    get invalidReason() {
      return this.invalid ? this.invalid.reason : null;
    }

    /**
     * Returns an explanation of why this Interval became invalid, or null if the Interval is valid
     * @type {string}
     */
    get invalidExplanation() {
      return this.invalid ? this.invalid.explanation : null;
    }

    /**
     * Returns the length of the Interval in the specified unit.
     * @param {string} unit - the unit (such as 'hours' or 'days') to return the length in.
     * @return {number}
     */
    length(unit = "milliseconds") {
      return this.isValid ? this.toDuration(...[unit]).get(unit) : NaN;
    }

    /**
     * Returns the count of minutes, hours, days, months, or years included in the Interval, even in part.
     * Unlike {@link Interval#length} this counts sections of the calendar, not periods of time, e.g. specifying 'day'
     * asks 'what dates are included in this interval?', not 'how many days long is this interval?'
     * @param {string} [unit='milliseconds'] - the unit of time to count.
     * @return {number}
     */
    count(unit = "milliseconds") {
      if (!this.isValid) return NaN;
      const start = this.start.startOf(unit),
        end = this.end.startOf(unit);
      return Math.floor(end.diff(start, unit).get(unit)) + 1;
    }

    /**
     * Returns whether this Interval's start and end are both in the same unit of time
     * @param {string} unit - the unit of time to check sameness on
     * @return {boolean}
     */
    hasSame(unit) {
      return this.isValid ? this.isEmpty() || this.e.minus(1).hasSame(this.s, unit) : false;
    }

    /**
     * Return whether this Interval has the same start and end DateTimes.
     * @return {boolean}
     */
    isEmpty() {
      return this.s.valueOf() === this.e.valueOf();
    }

    /**
     * Return whether this Interval's start is after the specified DateTime.
     * @param {DateTime} dateTime
     * @return {boolean}
     */
    isAfter(dateTime) {
      if (!this.isValid) return false;
      return this.s > dateTime;
    }

    /**
     * Return whether this Interval's end is before the specified DateTime.
     * @param {DateTime} dateTime
     * @return {boolean}
     */
    isBefore(dateTime) {
      if (!this.isValid) return false;
      return this.e <= dateTime;
    }

    /**
     * Return whether this Interval contains the specified DateTime.
     * @param {DateTime} dateTime
     * @return {boolean}
     */
    contains(dateTime) {
      if (!this.isValid) return false;
      return this.s <= dateTime && this.e > dateTime;
    }

    /**
     * "Sets" the start and/or end dates. Returns a newly-constructed Interval.
     * @param {Object} values - the values to set
     * @param {DateTime} values.start - the starting DateTime
     * @param {DateTime} values.end - the ending DateTime
     * @return {Interval}
     */
    set({ start, end } = {}) {
      if (!this.isValid) return this;
      return Interval.fromDateTimes(start || this.s, end || this.e);
    }

    /**
     * Split this Interval at each of the specified DateTimes
     * @param {...DateTime} dateTimes - the unit of time to count.
     * @return {Array}
     */
    splitAt(...dateTimes) {
      if (!this.isValid) return [];
      const sorted = dateTimes
          .map(friendlyDateTime)
          .filter((d) => this.contains(d))
          .sort(),
        results = [];
      let { s } = this,
        i = 0;

      while (s < this.e) {
        const added = sorted[i] || this.e,
          next = +added > +this.e ? this.e : added;
        results.push(Interval.fromDateTimes(s, next));
        s = next;
        i += 1;
      }

      return results;
    }

    /**
     * Split this Interval into smaller Intervals, each of the specified length.
     * Left over time is grouped into a smaller interval
     * @param {Duration|Object|number} duration - The length of each resulting interval.
     * @return {Array}
     */
    splitBy(duration) {
      const dur = Duration.fromDurationLike(duration);

      if (!this.isValid || !dur.isValid || dur.as("milliseconds") === 0) {
        return [];
      }

      let { s } = this,
        idx = 1,
        next;

      const results = [];
      while (s < this.e) {
        const added = this.start.plus(dur.mapUnits((x) => x * idx));
        next = +added > +this.e ? this.e : added;
        results.push(Interval.fromDateTimes(s, next));
        s = next;
        idx += 1;
      }

      return results;
    }

    /**
     * Split this Interval into the specified number of smaller intervals.
     * @param {number} numberOfParts - The number of Intervals to divide the Interval into.
     * @return {Array}
     */
    divideEqually(numberOfParts) {
      if (!this.isValid) return [];
      return this.splitBy(this.length() / numberOfParts).slice(0, numberOfParts);
    }

    /**
     * Return whether this Interval overlaps with the specified Interval
     * @param {Interval} other
     * @return {boolean}
     */
    overlaps(other) {
      return this.e > other.s && this.s < other.e;
    }

    /**
     * Return whether this Interval's end is adjacent to the specified Interval's start.
     * @param {Interval} other
     * @return {boolean}
     */
    abutsStart(other) {
      if (!this.isValid) return false;
      return +this.e === +other.s;
    }

    /**
     * Return whether this Interval's start is adjacent to the specified Interval's end.
     * @param {Interval} other
     * @return {boolean}
     */
    abutsEnd(other) {
      if (!this.isValid) return false;
      return +other.e === +this.s;
    }

    /**
     * Return whether this Interval engulfs the start and end of the specified Interval.
     * @param {Interval} other
     * @return {boolean}
     */
    engulfs(other) {
      if (!this.isValid) return false;
      return this.s <= other.s && this.e >= other.e;
    }

    /**
     * Return whether this Interval has the same start and end as the specified Interval.
     * @param {Interval} other
     * @return {boolean}
     */
    equals(other) {
      if (!this.isValid || !other.isValid) {
        return false;
      }

      return this.s.equals(other.s) && this.e.equals(other.e);
    }

    /**
     * Return an Interval representing the intersection of this Interval and the specified Interval.
     * Specifically, the resulting Interval has the maximum start time and the minimum end time of the two Intervals.
     * Returns null if the intersection is empty, meaning, the intervals don't intersect.
     * @param {Interval} other
     * @return {Interval}
     */
    intersection(other) {
      if (!this.isValid) return this;
      const s = this.s > other.s ? this.s : other.s,
        e = this.e < other.e ? this.e : other.e;

      if (s >= e) {
        return null;
      } else {
        return Interval.fromDateTimes(s, e);
      }
    }

    /**
     * Return an Interval representing the union of this Interval and the specified Interval.
     * Specifically, the resulting Interval has the minimum start time and the maximum end time of the two Intervals.
     * @param {Interval} other
     * @return {Interval}
     */
    union(other) {
      if (!this.isValid) return this;
      const s = this.s < other.s ? this.s : other.s,
        e = this.e > other.e ? this.e : other.e;
      return Interval.fromDateTimes(s, e);
    }

    /**
     * Merge an array of Intervals into a equivalent minimal set of Intervals.
     * Combines overlapping and adjacent Intervals.
     * @param {Array} intervals
     * @return {Array}
     */
    static merge(intervals) {
      const [found, final] = intervals
        .sort((a, b) => a.s - b.s)
        .reduce(
          ([sofar, current], item) => {
            if (!current) {
              return [sofar, item];
            } else if (current.overlaps(item) || current.abutsStart(item)) {
              return [sofar, current.union(item)];
            } else {
              return [sofar.concat([current]), item];
            }
          },
          [[], null]
        );
      if (final) {
        found.push(final);
      }
      return found;
    }

    /**
     * Return an array of Intervals representing the spans of time that only appear in one of the specified Intervals.
     * @param {Array} intervals
     * @return {Array}
     */
    static xor(intervals) {
      let start = null,
        currentCount = 0;
      const results = [],
        ends = intervals.map((i) => [
          { time: i.s, type: "s" },
          { time: i.e, type: "e" },
        ]),
        flattened = Array.prototype.concat(...ends),
        arr = flattened.sort((a, b) => a.time - b.time);

      for (const i of arr) {
        currentCount += i.type === "s" ? 1 : -1;

        if (currentCount === 1) {
          start = i.time;
        } else {
          if (start && +start !== +i.time) {
            results.push(Interval.fromDateTimes(start, i.time));
          }

          start = null;
        }
      }

      return Interval.merge(results);
    }

    /**
     * Return an Interval representing the span of time in this Interval that doesn't overlap with any of the specified Intervals.
     * @param {...Interval} intervals
     * @return {Array}
     */
    difference(...intervals) {
      return Interval.xor([this].concat(intervals))
        .map((i) => this.intersection(i))
        .filter((i) => i && !i.isEmpty());
    }

    /**
     * Returns a string representation of this Interval appropriate for debugging.
     * @return {string}
     */
    toString() {
      if (!this.isValid) return INVALID$2;
      return `[${this.s.toISO()} – ${this.e.toISO()})`;
    }

    /**
     * Returns a localized string representing this Interval. Accepts the same options as the
     * Intl.DateTimeFormat constructor and any presets defined by Luxon, such as
     * {@link DateTime.DATE_FULL} or {@link DateTime.TIME_SIMPLE}. The exact behavior of this method
     * is browser-specific, but in general it will return an appropriate representation of the
     * Interval in the assigned locale. Defaults to the system's locale if no locale has been
     * specified.
     * @see https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/DateTimeFormat
     * @param {Object} [formatOpts=DateTime.DATE_SHORT] - Either a DateTime preset or
     * Intl.DateTimeFormat constructor options.
     * @param {Object} opts - Options to override the configuration of the start DateTime.
     * @example Interval.fromISO('2022-11-07T09:00Z/2022-11-08T09:00Z').toLocaleString(); //=> 11/7/2022 – 11/8/2022
     * @example Interval.fromISO('2022-11-07T09:00Z/2022-11-08T09:00Z').toLocaleString(DateTime.DATE_FULL); //=> November 7 – 8, 2022
     * @example Interval.fromISO('2022-11-07T09:00Z/2022-11-08T09:00Z').toLocaleString(DateTime.DATE_FULL, { locale: 'fr-FR' }); //=> 7–8 novembre 2022
     * @example Interval.fromISO('2022-11-07T17:00Z/2022-11-07T19:00Z').toLocaleString(DateTime.TIME_SIMPLE); //=> 6:00 – 8:00 PM
     * @example Interval.fromISO('2022-11-07T17:00Z/2022-11-07T19:00Z').toLocaleString({ weekday: 'short', month: 'short', day: '2-digit', hour: '2-digit', minute: '2-digit' }); //=> Mon, Nov 07, 6:00 – 8:00 p
     * @return {string}
     */
    toLocaleString(formatOpts = DATE_SHORT, opts = {}) {
      return this.isValid
        ? Formatter.create(this.s.loc.clone(opts), formatOpts).formatInterval(this)
        : INVALID$2;
    }

    /**
     * Returns an ISO 8601-compliant string representation of this Interval.
     * @see https://en.wikipedia.org/wiki/ISO_8601#Time_intervals
     * @param {Object} opts - The same options as {@link DateTime#toISO}
     * @return {string}
     */
    toISO(opts) {
      if (!this.isValid) return INVALID$2;
      return `${this.s.toISO(opts)}/${this.e.toISO(opts)}`;
    }

    /**
     * Returns an ISO 8601-compliant string representation of date of this Interval.
     * The time components are ignored.
     * @see https://en.wikipedia.org/wiki/ISO_8601#Time_intervals
     * @return {string}
     */
    toISODate() {
      if (!this.isValid) return INVALID$2;
      return `${this.s.toISODate()}/${this.e.toISODate()}`;
    }

    /**
     * Returns an ISO 8601-compliant string representation of time of this Interval.
     * The date components are ignored.
     * @see https://en.wikipedia.org/wiki/ISO_8601#Time_intervals
     * @param {Object} opts - The same options as {@link DateTime#toISO}
     * @return {string}
     */
    toISOTime(opts) {
      if (!this.isValid) return INVALID$2;
      return `${this.s.toISOTime(opts)}/${this.e.toISOTime(opts)}`;
    }

    /**
     * Returns a string representation of this Interval formatted according to the specified format
     * string. **You may not want this.** See {@link Interval#toLocaleString} for a more flexible
     * formatting tool.
     * @param {string} dateFormat - The format string. This string formats the start and end time.
     * See {@link DateTime#toFormat} for details.
     * @param {Object} opts - Options.
     * @param {string} [opts.separator =  ' – '] - A separator to place between the start and end
     * representations.
     * @return {string}
     */
    toFormat(dateFormat, { separator = " – " } = {}) {
      if (!this.isValid) return INVALID$2;
      return `${this.s.toFormat(dateFormat)}${separator}${this.e.toFormat(dateFormat)}`;
    }

    /**
     * Return a Duration representing the time spanned by this interval.
     * @param {string|string[]} [unit=['milliseconds']] - the unit or units (such as 'hours' or 'days') to include in the duration.
     * @param {Object} opts - options that affect the creation of the Duration
     * @param {string} [opts.conversionAccuracy='casual'] - the conversion system to use
     * @example Interval.fromDateTimes(dt1, dt2).toDuration().toObject() //=> { milliseconds: 88489257 }
     * @example Interval.fromDateTimes(dt1, dt2).toDuration('days').toObject() //=> { days: 1.0241812152777778 }
     * @example Interval.fromDateTimes(dt1, dt2).toDuration(['hours', 'minutes']).toObject() //=> { hours: 24, minutes: 34.82095 }
     * @example Interval.fromDateTimes(dt1, dt2).toDuration(['hours', 'minutes', 'seconds']).toObject() //=> { hours: 24, minutes: 34, seconds: 49.257 }
     * @example Interval.fromDateTimes(dt1, dt2).toDuration('seconds').toObject() //=> { seconds: 88489.257 }
     * @return {Duration}
     */
    toDuration(unit, opts) {
      if (!this.isValid) {
        return Duration.invalid(this.invalidReason);
      }
      return this.e.diff(this.s, unit, opts);
    }

    /**
     * Run mapFn on the interval start and end, returning a new Interval from the resulting DateTimes
     * @param {function} mapFn
     * @return {Interval}
     * @example Interval.fromDateTimes(dt1, dt2).mapEndpoints(endpoint => endpoint.toUTC())
     * @example Interval.fromDateTimes(dt1, dt2).mapEndpoints(endpoint => endpoint.plus({ hours: 2 }))
     */
    mapEndpoints(mapFn) {
      return Interval.fromDateTimes(mapFn(this.s), mapFn(this.e));
    }
  }

  /**
   * The Info class contains static methods for retrieving general time and date related data. For example, it has methods for finding out if a time zone has a DST, for listing the months in any supported locale, and for discovering which of Luxon features are available in the current environment.
   */
  class Info {
    /**
     * Return whether the specified zone contains a DST.
     * @param {string|Zone} [zone='local'] - Zone to check. Defaults to the environment's local zone.
     * @return {boolean}
     */
    static hasDST(zone = Settings.defaultZone) {
      const proto = DateTime.now().setZone(zone).set({ month: 12 });

      return !zone.isUniversal && proto.offset !== proto.set({ month: 6 }).offset;
    }

    /**
     * Return whether the specified zone is a valid IANA specifier.
     * @param {string} zone - Zone to check
     * @return {boolean}
     */
    static isValidIANAZone(zone) {
      return IANAZone.isValidZone(zone);
    }

    /**
     * Converts the input into a {@link Zone} instance.
     *
     * * If `input` is already a Zone instance, it is returned unchanged.
     * * If `input` is a string containing a valid time zone name, a Zone instance
     *   with that name is returned.
     * * If `input` is a string that doesn't refer to a known time zone, a Zone
     *   instance with {@link Zone#isValid} == false is returned.
     * * If `input is a number, a Zone instance with the specified fixed offset
     *   in minutes is returned.
     * * If `input` is `null` or `undefined`, the default zone is returned.
     * @param {string|Zone|number} [input] - the value to be converted
     * @return {Zone}
     */
    static normalizeZone(input) {
      return normalizeZone(input, Settings.defaultZone);
    }

    /**
     * Return an array of standalone month names.
     * @see https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/DateTimeFormat
     * @param {string} [length='long'] - the length of the month representation, such as "numeric", "2-digit", "narrow", "short", "long"
     * @param {Object} opts - options
     * @param {string} [opts.locale] - the locale code
     * @param {string} [opts.numberingSystem=null] - the numbering system
     * @param {string} [opts.locObj=null] - an existing locale object to use
     * @param {string} [opts.outputCalendar='gregory'] - the calendar
     * @example Info.months()[0] //=> 'January'
     * @example Info.months('short')[0] //=> 'Jan'
     * @example Info.months('numeric')[0] //=> '1'
     * @example Info.months('short', { locale: 'fr-CA' } )[0] //=> 'janv.'
     * @example Info.months('numeric', { locale: 'ar' })[0] //=> '١'
     * @example Info.months('long', { outputCalendar: 'islamic' })[0] //=> 'Rabiʻ I'
     * @return {Array}
     */
    static months(
      length = "long",
      { locale = null, numberingSystem = null, locObj = null, outputCalendar = "gregory" } = {}
    ) {
      return (locObj || Locale.create(locale, numberingSystem, outputCalendar)).months(length);
    }

    /**
     * Return an array of format month names.
     * Format months differ from standalone months in that they're meant to appear next to the day of the month. In some languages, that
     * changes the string.
     * See {@link Info#months}
     * @param {string} [length='long'] - the length of the month representation, such as "numeric", "2-digit", "narrow", "short", "long"
     * @param {Object} opts - options
     * @param {string} [opts.locale] - the locale code
     * @param {string} [opts.numberingSystem=null] - the numbering system
     * @param {string} [opts.locObj=null] - an existing locale object to use
     * @param {string} [opts.outputCalendar='gregory'] - the calendar
     * @return {Array}
     */
    static monthsFormat(
      length = "long",
      { locale = null, numberingSystem = null, locObj = null, outputCalendar = "gregory" } = {}
    ) {
      return (locObj || Locale.create(locale, numberingSystem, outputCalendar)).months(length, true);
    }

    /**
     * Return an array of standalone week names.
     * @see https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/DateTimeFormat
     * @param {string} [length='long'] - the length of the weekday representation, such as "narrow", "short", "long".
     * @param {Object} opts - options
     * @param {string} [opts.locale] - the locale code
     * @param {string} [opts.numberingSystem=null] - the numbering system
     * @param {string} [opts.locObj=null] - an existing locale object to use
     * @example Info.weekdays()[0] //=> 'Monday'
     * @example Info.weekdays('short')[0] //=> 'Mon'
     * @example Info.weekdays('short', { locale: 'fr-CA' })[0] //=> 'lun.'
     * @example Info.weekdays('short', { locale: 'ar' })[0] //=> 'الاثنين'
     * @return {Array}
     */
    static weekdays(length = "long", { locale = null, numberingSystem = null, locObj = null } = {}) {
      return (locObj || Locale.create(locale, numberingSystem, null)).weekdays(length);
    }

    /**
     * Return an array of format week names.
     * Format weekdays differ from standalone weekdays in that they're meant to appear next to more date information. In some languages, that
     * changes the string.
     * See {@link Info#weekdays}
     * @param {string} [length='long'] - the length of the month representation, such as "narrow", "short", "long".
     * @param {Object} opts - options
     * @param {string} [opts.locale=null] - the locale code
     * @param {string} [opts.numberingSystem=null] - the numbering system
     * @param {string} [opts.locObj=null] - an existing locale object to use
     * @return {Array}
     */
    static weekdaysFormat(
      length = "long",
      { locale = null, numberingSystem = null, locObj = null } = {}
    ) {
      return (locObj || Locale.create(locale, numberingSystem, null)).weekdays(length, true);
    }

    /**
     * Return an array of meridiems.
     * @param {Object} opts - options
     * @param {string} [opts.locale] - the locale code
     * @example Info.meridiems() //=> [ 'AM', 'PM' ]
     * @example Info.meridiems({ locale: 'my' }) //=> [ 'နံနက်', 'ညနေ' ]
     * @return {Array}
     */
    static meridiems({ locale = null } = {}) {
      return Locale.create(locale).meridiems();
    }

    /**
     * Return an array of eras, such as ['BC', 'AD']. The locale can be specified, but the calendar system is always Gregorian.
     * @param {string} [length='short'] - the length of the era representation, such as "short" or "long".
     * @param {Object} opts - options
     * @param {string} [opts.locale] - the locale code
     * @example Info.eras() //=> [ 'BC', 'AD' ]
     * @example Info.eras('long') //=> [ 'Before Christ', 'Anno Domini' ]
     * @example Info.eras('long', { locale: 'fr' }) //=> [ 'avant Jésus-Christ', 'après Jésus-Christ' ]
     * @return {Array}
     */
    static eras(length = "short", { locale = null } = {}) {
      return Locale.create(locale, null, "gregory").eras(length);
    }

    /**
     * Return the set of available features in this environment.
     * Some features of Luxon are not available in all environments. For example, on older browsers, relative time formatting support is not available. Use this function to figure out if that's the case.
     * Keys:
     * * `relative`: whether this environment supports relative time formatting
     * @example Info.features() //=> { relative: false }
     * @return {Object}
     */
    static features() {
      return { relative: hasRelative() };
    }
  }

  function dayDiff(earlier, later) {
    const utcDayStart = (dt) => dt.toUTC(0, { keepLocalTime: true }).startOf("day").valueOf(),
      ms = utcDayStart(later) - utcDayStart(earlier);
    return Math.floor(Duration.fromMillis(ms).as("days"));
  }

  function highOrderDiffs(cursor, later, units) {
    const differs = [
      ["years", (a, b) => b.year - a.year],
      ["quarters", (a, b) => b.quarter - a.quarter + (b.year - a.year) * 4],
      ["months", (a, b) => b.month - a.month + (b.year - a.year) * 12],
      [
        "weeks",
        (a, b) => {
          const days = dayDiff(a, b);
          return (days - (days % 7)) / 7;
        },
      ],
      ["days", dayDiff],
    ];

    const results = {};
    const earlier = cursor;
    let lowestOrder, highWater;

    for (const [unit, differ] of differs) {
      if (units.indexOf(unit) >= 0) {
        lowestOrder = unit;

        results[unit] = differ(cursor, later);
        highWater = earlier.plus(results);

        if (highWater > later) {
          results[unit]--;
          cursor = earlier.plus(results);
        } else {
          cursor = highWater;
        }
      }
    }

    return [cursor, results, highWater, lowestOrder];
  }

  function diff (earlier, later, units, opts) {
    let [cursor, results, highWater, lowestOrder] = highOrderDiffs(earlier, later, units);

    const remainingMillis = later - cursor;

    const lowerOrderUnits = units.filter(
      (u) => ["hours", "minutes", "seconds", "milliseconds"].indexOf(u) >= 0
    );

    if (lowerOrderUnits.length === 0) {
      if (highWater < later) {
        highWater = cursor.plus({ [lowestOrder]: 1 });
      }

      if (highWater !== cursor) {
        results[lowestOrder] = (results[lowestOrder] || 0) + remainingMillis / (highWater - cursor);
      }
    }

    const duration = Duration.fromObject(results, opts);

    if (lowerOrderUnits.length > 0) {
      return Duration.fromMillis(remainingMillis, opts)
        .shiftTo(...lowerOrderUnits)
        .plus(duration);
    } else {
      return duration;
    }
  }

  const numberingSystems = {
    arab: "[\u0660-\u0669]",
    arabext: "[\u06F0-\u06F9]",
    bali: "[\u1B50-\u1B59]",
    beng: "[\u09E6-\u09EF]",
    deva: "[\u0966-\u096F]",
    fullwide: "[\uFF10-\uFF19]",
    gujr: "[\u0AE6-\u0AEF]",
    hanidec: "[〇|一|二|三|四|五|六|七|八|九]",
    khmr: "[\u17E0-\u17E9]",
    knda: "[\u0CE6-\u0CEF]",
    laoo: "[\u0ED0-\u0ED9]",
    limb: "[\u1946-\u194F]",
    mlym: "[\u0D66-\u0D6F]",
    mong: "[\u1810-\u1819]",
    mymr: "[\u1040-\u1049]",
    orya: "[\u0B66-\u0B6F]",
    tamldec: "[\u0BE6-\u0BEF]",
    telu: "[\u0C66-\u0C6F]",
    thai: "[\u0E50-\u0E59]",
    tibt: "[\u0F20-\u0F29]",
    latn: "\\d",
  };

  const numberingSystemsUTF16 = {
    arab: [1632, 1641],
    arabext: [1776, 1785],
    bali: [6992, 7001],
    beng: [2534, 2543],
    deva: [2406, 2415],
    fullwide: [65296, 65303],
    gujr: [2790, 2799],
    khmr: [6112, 6121],
    knda: [3302, 3311],
    laoo: [3792, 3801],
    limb: [6470, 6479],
    mlym: [3430, 3439],
    mong: [6160, 6169],
    mymr: [4160, 4169],
    orya: [2918, 2927],
    tamldec: [3046, 3055],
    telu: [3174, 3183],
    thai: [3664, 3673],
    tibt: [3872, 3881],
  };

  const hanidecChars = numberingSystems.hanidec.replace(/[\[|\]]/g, "").split("");

  function parseDigits(str) {
    let value = parseInt(str, 10);
    if (isNaN(value)) {
      value = "";
      for (let i = 0; i < str.length; i++) {
        const code = str.charCodeAt(i);

        if (str[i].search(numberingSystems.hanidec) !== -1) {
          value += hanidecChars.indexOf(str[i]);
        } else {
          for (const key in numberingSystemsUTF16) {
            const [min, max] = numberingSystemsUTF16[key];
            if (code >= min && code <= max) {
              value += code - min;
            }
          }
        }
      }
      return parseInt(value, 10);
    } else {
      return value;
    }
  }

  function digitRegex({ numberingSystem }, append = "") {
    return new RegExp(`${numberingSystems[numberingSystem || "latn"]}${append}`);
  }

  const MISSING_FTP = "missing Intl.DateTimeFormat.formatToParts support";

  function intUnit(regex, post = (i) => i) {
    return { regex, deser: ([s]) => post(parseDigits(s)) };
  }

  const NBSP = String.fromCharCode(160);
  const spaceOrNBSP = `[ ${NBSP}]`;
  const spaceOrNBSPRegExp = new RegExp(spaceOrNBSP, "g");

  function fixListRegex(s) {
    // make dots optional and also make them literal
    // make space and non breakable space characters interchangeable
    return s.replace(/\./g, "\\.?").replace(spaceOrNBSPRegExp, spaceOrNBSP);
  }

  function stripInsensitivities(s) {
    return s
      .replace(/\./g, "") // ignore dots that were made optional
      .replace(spaceOrNBSPRegExp, " ") // interchange space and nbsp
      .toLowerCase();
  }

  function oneOf(strings, startIndex) {
    if (strings === null) {
      return null;
    } else {
      return {
        regex: RegExp(strings.map(fixListRegex).join("|")),
        deser: ([s]) =>
          strings.findIndex((i) => stripInsensitivities(s) === stripInsensitivities(i)) + startIndex,
      };
    }
  }

  function offset(regex, groups) {
    return { regex, deser: ([, h, m]) => signedOffset(h, m), groups };
  }

  function simple(regex) {
    return { regex, deser: ([s]) => s };
  }

  function escapeToken(value) {
    return value.replace(/[\-\[\]{}()*+?.,\\\^$|#\s]/g, "\\$&");
  }

  function unitForToken(token, loc) {
    const one = digitRegex(loc),
      two = digitRegex(loc, "{2}"),
      three = digitRegex(loc, "{3}"),
      four = digitRegex(loc, "{4}"),
      six = digitRegex(loc, "{6}"),
      oneOrTwo = digitRegex(loc, "{1,2}"),
      oneToThree = digitRegex(loc, "{1,3}"),
      oneToSix = digitRegex(loc, "{1,6}"),
      oneToNine = digitRegex(loc, "{1,9}"),
      twoToFour = digitRegex(loc, "{2,4}"),
      fourToSix = digitRegex(loc, "{4,6}"),
      literal = (t) => ({ regex: RegExp(escapeToken(t.val)), deser: ([s]) => s, literal: true }),
      unitate = (t) => {
        if (token.literal) {
          return literal(t);
        }
        switch (t.val) {
          // era
          case "G":
            return oneOf(loc.eras("short", false), 0);
          case "GG":
            return oneOf(loc.eras("long", false), 0);
          // years
          case "y":
            return intUnit(oneToSix);
          case "yy":
            return intUnit(twoToFour, untruncateYear);
          case "yyyy":
            return intUnit(four);
          case "yyyyy":
            return intUnit(fourToSix);
          case "yyyyyy":
            return intUnit(six);
          // months
          case "M":
            return intUnit(oneOrTwo);
          case "MM":
            return intUnit(two);
          case "MMM":
            return oneOf(loc.months("short", true, false), 1);
          case "MMMM":
            return oneOf(loc.months("long", true, false), 1);
          case "L":
            return intUnit(oneOrTwo);
          case "LL":
            return intUnit(two);
          case "LLL":
            return oneOf(loc.months("short", false, false), 1);
          case "LLLL":
            return oneOf(loc.months("long", false, false), 1);
          // dates
          case "d":
            return intUnit(oneOrTwo);
          case "dd":
            return intUnit(two);
          // ordinals
          case "o":
            return intUnit(oneToThree);
          case "ooo":
            return intUnit(three);
          // time
          case "HH":
            return intUnit(two);
          case "H":
            return intUnit(oneOrTwo);
          case "hh":
            return intUnit(two);
          case "h":
            return intUnit(oneOrTwo);
          case "mm":
            return intUnit(two);
          case "m":
            return intUnit(oneOrTwo);
          case "q":
            return intUnit(oneOrTwo);
          case "qq":
            return intUnit(two);
          case "s":
            return intUnit(oneOrTwo);
          case "ss":
            return intUnit(two);
          case "S":
            return intUnit(oneToThree);
          case "SSS":
            return intUnit(three);
          case "u":
            return simple(oneToNine);
          case "uu":
            return simple(oneOrTwo);
          case "uuu":
            return intUnit(one);
          // meridiem
          case "a":
            return oneOf(loc.meridiems(), 0);
          // weekYear (k)
          case "kkkk":
            return intUnit(four);
          case "kk":
            return intUnit(twoToFour, untruncateYear);
          // weekNumber (W)
          case "W":
            return intUnit(oneOrTwo);
          case "WW":
            return intUnit(two);
          // weekdays
          case "E":
          case "c":
            return intUnit(one);
          case "EEE":
            return oneOf(loc.weekdays("short", false, false), 1);
          case "EEEE":
            return oneOf(loc.weekdays("long", false, false), 1);
          case "ccc":
            return oneOf(loc.weekdays("short", true, false), 1);
          case "cccc":
            return oneOf(loc.weekdays("long", true, false), 1);
          // offset/zone
          case "Z":
          case "ZZ":
            return offset(new RegExp(`([+-]${oneOrTwo.source})(?::(${two.source}))?`), 2);
          case "ZZZ":
            return offset(new RegExp(`([+-]${oneOrTwo.source})(${two.source})?`), 2);
          // we don't support ZZZZ (PST) or ZZZZZ (Pacific Standard Time) in parsing
          // because we don't have any way to figure out what they are
          case "z":
            return simple(/[a-z_+-/]{1,256}?/i);
          default:
            return literal(t);
        }
      };

    const unit = unitate(token) || {
      invalidReason: MISSING_FTP,
    };

    unit.token = token;

    return unit;
  }

  const partTypeStyleToTokenVal = {
    year: {
      "2-digit": "yy",
      numeric: "yyyyy",
    },
    month: {
      numeric: "M",
      "2-digit": "MM",
      short: "MMM",
      long: "MMMM",
    },
    day: {
      numeric: "d",
      "2-digit": "dd",
    },
    weekday: {
      short: "EEE",
      long: "EEEE",
    },
    dayperiod: "a",
    dayPeriod: "a",
    hour: {
      numeric: "h",
      "2-digit": "hh",
    },
    minute: {
      numeric: "m",
      "2-digit": "mm",
    },
    second: {
      numeric: "s",
      "2-digit": "ss",
    },
    timeZoneName: {
      long: "ZZZZZ",
      short: "ZZZ",
    },
  };

  function tokenForPart(part, formatOpts) {
    const { type, value } = part;

    if (type === "literal") {
      return {
        literal: true,
        val: value,
      };
    }

    const style = formatOpts[type];

    let val = partTypeStyleToTokenVal[type];
    if (typeof val === "object") {
      val = val[style];
    }

    if (val) {
      return {
        literal: false,
        val,
      };
    }

    return undefined;
  }

  function buildRegex(units) {
    const re = units.map((u) => u.regex).reduce((f, r) => `${f}(${r.source})`, "");
    return [`^${re}$`, units];
  }

  function match(input, regex, handlers) {
    const matches = input.match(regex);

    if (matches) {
      const all = {};
      let matchIndex = 1;
      for (const i in handlers) {
        if (hasOwnProperty(handlers, i)) {
          const h = handlers[i],
            groups = h.groups ? h.groups + 1 : 1;
          if (!h.literal && h.token) {
            all[h.token.val[0]] = h.deser(matches.slice(matchIndex, matchIndex + groups));
          }
          matchIndex += groups;
        }
      }
      return [matches, all];
    } else {
      return [matches, {}];
    }
  }

  function dateTimeFromMatches(matches) {
    const toField = (token) => {
      switch (token) {
        case "S":
          return "millisecond";
        case "s":
          return "second";
        case "m":
          return "minute";
        case "h":
        case "H":
          return "hour";
        case "d":
          return "day";
        case "o":
          return "ordinal";
        case "L":
        case "M":
          return "month";
        case "y":
          return "year";
        case "E":
        case "c":
          return "weekday";
        case "W":
          return "weekNumber";
        case "k":
          return "weekYear";
        case "q":
          return "quarter";
        default:
          return null;
      }
    };

    let zone = null;
    let specificOffset;
    if (!isUndefined(matches.z)) {
      zone = IANAZone.create(matches.z);
    }

    if (!isUndefined(matches.Z)) {
      if (!zone) {
        zone = new FixedOffsetZone(matches.Z);
      }
      specificOffset = matches.Z;
    }

    if (!isUndefined(matches.q)) {
      matches.M = (matches.q - 1) * 3 + 1;
    }

    if (!isUndefined(matches.h)) {
      if (matches.h < 12 && matches.a === 1) {
        matches.h += 12;
      } else if (matches.h === 12 && matches.a === 0) {
        matches.h = 0;
      }
    }

    if (matches.G === 0 && matches.y) {
      matches.y = -matches.y;
    }

    if (!isUndefined(matches.u)) {
      matches.S = parseMillis(matches.u);
    }

    const vals = Object.keys(matches).reduce((r, k) => {
      const f = toField(k);
      if (f) {
        r[f] = matches[k];
      }

      return r;
    }, {});

    return [vals, zone, specificOffset];
  }

  let dummyDateTimeCache = null;

  function getDummyDateTime() {
    if (!dummyDateTimeCache) {
      dummyDateTimeCache = DateTime.fromMillis(1555555555555);
    }

    return dummyDateTimeCache;
  }

  function maybeExpandMacroToken(token, locale) {
    if (token.literal) {
      return token;
    }

    const formatOpts = Formatter.macroTokenToFormatOpts(token.val);
    const tokens = formatOptsToTokens(formatOpts, locale);

    if (tokens == null || tokens.includes(undefined)) {
      return token;
    }

    return tokens;
  }

  function expandMacroTokens(tokens, locale) {
    return Array.prototype.concat(...tokens.map((t) => maybeExpandMacroToken(t, locale)));
  }

  /**
   * @private
   */

  function explainFromTokens(locale, input, format) {
    const tokens = expandMacroTokens(Formatter.parseFormat(format), locale),
      units = tokens.map((t) => unitForToken(t, locale)),
      disqualifyingUnit = units.find((t) => t.invalidReason);

    if (disqualifyingUnit) {
      return { input, tokens, invalidReason: disqualifyingUnit.invalidReason };
    } else {
      const [regexString, handlers] = buildRegex(units),
        regex = RegExp(regexString, "i"),
        [rawMatches, matches] = match(input, regex, handlers),
        [result, zone, specificOffset] = matches
          ? dateTimeFromMatches(matches)
          : [null, null, undefined];
      if (hasOwnProperty(matches, "a") && hasOwnProperty(matches, "H")) {
        throw new ConflictingSpecificationError(
          "Can't include meridiem when specifying 24-hour format"
        );
      }
      return { input, tokens, regex, rawMatches, matches, result, zone, specificOffset };
    }
  }

  function parseFromTokens(locale, input, format) {
    const { result, zone, specificOffset, invalidReason } = explainFromTokens(locale, input, format);
    return [result, zone, specificOffset, invalidReason];
  }

  function formatOptsToTokens(formatOpts, locale) {
    if (!formatOpts) {
      return null;
    }

    const formatter = Formatter.create(locale, formatOpts);
    const parts = formatter.formatDateTimeParts(getDummyDateTime());
    return parts.map((p) => tokenForPart(p, formatOpts));
  }

  const nonLeapLadder = [0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334],
    leapLadder = [0, 31, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335];

  function unitOutOfRange(unit, value) {
    return new Invalid(
      "unit out of range",
      `you specified ${value} (of type ${typeof value}) as a ${unit}, which is invalid`
    );
  }

  function dayOfWeek(year, month, day) {
    const d = new Date(Date.UTC(year, month - 1, day));

    if (year < 100 && year >= 0) {
      d.setUTCFullYear(d.getUTCFullYear() - 1900);
    }

    const js = d.getUTCDay();

    return js === 0 ? 7 : js;
  }

  function computeOrdinal(year, month, day) {
    return day + (isLeapYear(year) ? leapLadder : nonLeapLadder)[month - 1];
  }

  function uncomputeOrdinal(year, ordinal) {
    const table = isLeapYear(year) ? leapLadder : nonLeapLadder,
      month0 = table.findIndex((i) => i < ordinal),
      day = ordinal - table[month0];
    return { month: month0 + 1, day };
  }

  /**
   * @private
   */

  function gregorianToWeek(gregObj) {
    const { year, month, day } = gregObj,
      ordinal = computeOrdinal(year, month, day),
      weekday = dayOfWeek(year, month, day);

    let weekNumber = Math.floor((ordinal - weekday + 10) / 7),
      weekYear;

    if (weekNumber < 1) {
      weekYear = year - 1;
      weekNumber = weeksInWeekYear(weekYear);
    } else if (weekNumber > weeksInWeekYear(year)) {
      weekYear = year + 1;
      weekNumber = 1;
    } else {
      weekYear = year;
    }

    return { weekYear, weekNumber, weekday, ...timeObject(gregObj) };
  }

  function weekToGregorian(weekData) {
    const { weekYear, weekNumber, weekday } = weekData,
      weekdayOfJan4 = dayOfWeek(weekYear, 1, 4),
      yearInDays = daysInYear(weekYear);

    let ordinal = weekNumber * 7 + weekday - weekdayOfJan4 - 3,
      year;

    if (ordinal < 1) {
      year = weekYear - 1;
      ordinal += daysInYear(year);
    } else if (ordinal > yearInDays) {
      year = weekYear + 1;
      ordinal -= daysInYear(weekYear);
    } else {
      year = weekYear;
    }

    const { month, day } = uncomputeOrdinal(year, ordinal);
    return { year, month, day, ...timeObject(weekData) };
  }

  function gregorianToOrdinal(gregData) {
    const { year, month, day } = gregData;
    const ordinal = computeOrdinal(year, month, day);
    return { year, ordinal, ...timeObject(gregData) };
  }

  function ordinalToGregorian(ordinalData) {
    const { year, ordinal } = ordinalData;
    const { month, day } = uncomputeOrdinal(year, ordinal);
    return { year, month, day, ...timeObject(ordinalData) };
  }

  function hasInvalidWeekData(obj) {
    const validYear = isInteger(obj.weekYear),
      validWeek = integerBetween(obj.weekNumber, 1, weeksInWeekYear(obj.weekYear)),
      validWeekday = integerBetween(obj.weekday, 1, 7);

    if (!validYear) {
      return unitOutOfRange("weekYear", obj.weekYear);
    } else if (!validWeek) {
      return unitOutOfRange("week", obj.week);
    } else if (!validWeekday) {
      return unitOutOfRange("weekday", obj.weekday);
    } else return false;
  }

  function hasInvalidOrdinalData(obj) {
    const validYear = isInteger(obj.year),
      validOrdinal = integerBetween(obj.ordinal, 1, daysInYear(obj.year));

    if (!validYear) {
      return unitOutOfRange("year", obj.year);
    } else if (!validOrdinal) {
      return unitOutOfRange("ordinal", obj.ordinal);
    } else return false;
  }

  function hasInvalidGregorianData(obj) {
    const validYear = isInteger(obj.year),
      validMonth = integerBetween(obj.month, 1, 12),
      validDay = integerBetween(obj.day, 1, daysInMonth(obj.year, obj.month));

    if (!validYear) {
      return unitOutOfRange("year", obj.year);
    } else if (!validMonth) {
      return unitOutOfRange("month", obj.month);
    } else if (!validDay) {
      return unitOutOfRange("day", obj.day);
    } else return false;
  }

  function hasInvalidTimeData(obj) {
    const { hour, minute, second, millisecond } = obj;
    const validHour =
        integerBetween(hour, 0, 23) ||
        (hour === 24 && minute === 0 && second === 0 && millisecond === 0),
      validMinute = integerBetween(minute, 0, 59),
      validSecond = integerBetween(second, 0, 59),
      validMillisecond = integerBetween(millisecond, 0, 999);

    if (!validHour) {
      return unitOutOfRange("hour", hour);
    } else if (!validMinute) {
      return unitOutOfRange("minute", minute);
    } else if (!validSecond) {
      return unitOutOfRange("second", second);
    } else if (!validMillisecond) {
      return unitOutOfRange("millisecond", millisecond);
    } else return false;
  }

  const INVALID$1 = "Invalid DateTime";
  const MAX_DATE = 8.64e15;

  function unsupportedZone(zone) {
    return new Invalid("unsupported zone", `the zone "${zone.name}" is not supported`);
  }

  // we cache week data on the DT object and this intermediates the cache
  function possiblyCachedWeekData(dt) {
    if (dt.weekData === null) {
      dt.weekData = gregorianToWeek(dt.c);
    }
    return dt.weekData;
  }

  // clone really means, "make a new object with these modifications". all "setters" really use this
  // to create a new object while only changing some of the properties
  function clone$1(inst, alts) {
    const current = {
      ts: inst.ts,
      zone: inst.zone,
      c: inst.c,
      o: inst.o,
      loc: inst.loc,
      invalid: inst.invalid,
    };
    return new DateTime({ ...current, ...alts, old: current });
  }

  // find the right offset a given local time. The o input is our guess, which determines which
  // offset we'll pick in ambiguous cases (e.g. there are two 3 AMs b/c Fallback DST)
  function fixOffset(localTS, o, tz) {
    // Our UTC time is just a guess because our offset is just a guess
    let utcGuess = localTS - o * 60 * 1000;

    // Test whether the zone matches the offset for this ts
    const o2 = tz.offset(utcGuess);

    // If so, offset didn't change and we're done
    if (o === o2) {
      return [utcGuess, o];
    }

    // If not, change the ts by the difference in the offset
    utcGuess -= (o2 - o) * 60 * 1000;

    // If that gives us the local time we want, we're done
    const o3 = tz.offset(utcGuess);
    if (o2 === o3) {
      return [utcGuess, o2];
    }

    // If it's different, we're in a hole time. The offset has changed, but the we don't adjust the time
    return [localTS - Math.min(o2, o3) * 60 * 1000, Math.max(o2, o3)];
  }

  // convert an epoch timestamp into a calendar object with the given offset
  function tsToObj(ts, offset) {
    ts += offset * 60 * 1000;

    const d = new Date(ts);

    return {
      year: d.getUTCFullYear(),
      month: d.getUTCMonth() + 1,
      day: d.getUTCDate(),
      hour: d.getUTCHours(),
      minute: d.getUTCMinutes(),
      second: d.getUTCSeconds(),
      millisecond: d.getUTCMilliseconds(),
    };
  }

  // convert a calendar object to a epoch timestamp
  function objToTS(obj, offset, zone) {
    return fixOffset(objToLocalTS(obj), offset, zone);
  }

  // create a new DT instance by adding a duration, adjusting for DSTs
  function adjustTime(inst, dur) {
    const oPre = inst.o,
      year = inst.c.year + Math.trunc(dur.years),
      month = inst.c.month + Math.trunc(dur.months) + Math.trunc(dur.quarters) * 3,
      c = {
        ...inst.c,
        year,
        month,
        day:
          Math.min(inst.c.day, daysInMonth(year, month)) +
          Math.trunc(dur.days) +
          Math.trunc(dur.weeks) * 7,
      },
      millisToAdd = Duration.fromObject({
        years: dur.years - Math.trunc(dur.years),
        quarters: dur.quarters - Math.trunc(dur.quarters),
        months: dur.months - Math.trunc(dur.months),
        weeks: dur.weeks - Math.trunc(dur.weeks),
        days: dur.days - Math.trunc(dur.days),
        hours: dur.hours,
        minutes: dur.minutes,
        seconds: dur.seconds,
        milliseconds: dur.milliseconds,
      }).as("milliseconds"),
      localTS = objToLocalTS(c);

    let [ts, o] = fixOffset(localTS, oPre, inst.zone);

    if (millisToAdd !== 0) {
      ts += millisToAdd;
      // that could have changed the offset by going over a DST, but we want to keep the ts the same
      o = inst.zone.offset(ts);
    }

    return { ts, o };
  }

  // helper useful in turning the results of parsing into real dates
  // by handling the zone options
  function parseDataToDateTime(parsed, parsedZone, opts, format, text, specificOffset) {
    const { setZone, zone } = opts;
    if (parsed && Object.keys(parsed).length !== 0) {
      const interpretationZone = parsedZone || zone,
        inst = DateTime.fromObject(parsed, {
          ...opts,
          zone: interpretationZone,
          specificOffset,
        });
      return setZone ? inst : inst.setZone(zone);
    } else {
      return DateTime.invalid(
        new Invalid("unparsable", `the input "${text}" can't be parsed as ${format}`)
      );
    }
  }

  // if you want to output a technical format (e.g. RFC 2822), this helper
  // helps handle the details
  function toTechFormat(dt, format, allowZ = true) {
    return dt.isValid
      ? Formatter.create(Locale.create("en-US"), {
          allowZ,
          forceSimple: true,
        }).formatDateTimeFromString(dt, format)
      : null;
  }

  function toISODate(o, extended) {
    const longFormat = o.c.year > 9999 || o.c.year < 0;
    let c = "";
    if (longFormat && o.c.year >= 0) c += "+";
    c += padStart(o.c.year, longFormat ? 6 : 4);

    if (extended) {
      c += "-";
      c += padStart(o.c.month);
      c += "-";
      c += padStart(o.c.day);
    } else {
      c += padStart(o.c.month);
      c += padStart(o.c.day);
    }
    return c;
  }

  function toISOTime(
    o,
    extended,
    suppressSeconds,
    suppressMilliseconds,
    includeOffset,
    extendedZone
  ) {
    let c = padStart(o.c.hour);
    if (extended) {
      c += ":";
      c += padStart(o.c.minute);
      if (o.c.second !== 0 || !suppressSeconds) {
        c += ":";
      }
    } else {
      c += padStart(o.c.minute);
    }

    if (o.c.second !== 0 || !suppressSeconds) {
      c += padStart(o.c.second);

      if (o.c.millisecond !== 0 || !suppressMilliseconds) {
        c += ".";
        c += padStart(o.c.millisecond, 3);
      }
    }

    if (includeOffset) {
      if (o.isOffsetFixed && o.offset === 0 && !extendedZone) {
        c += "Z";
      } else if (o.o < 0) {
        c += "-";
        c += padStart(Math.trunc(-o.o / 60));
        c += ":";
        c += padStart(Math.trunc(-o.o % 60));
      } else {
        c += "+";
        c += padStart(Math.trunc(o.o / 60));
        c += ":";
        c += padStart(Math.trunc(o.o % 60));
      }
    }

    if (extendedZone) {
      c += "[" + o.zone.ianaName + "]";
    }
    return c;
  }

  // defaults for unspecified units in the supported calendars
  const defaultUnitValues = {
      month: 1,
      day: 1,
      hour: 0,
      minute: 0,
      second: 0,
      millisecond: 0,
    },
    defaultWeekUnitValues = {
      weekNumber: 1,
      weekday: 1,
      hour: 0,
      minute: 0,
      second: 0,
      millisecond: 0,
    },
    defaultOrdinalUnitValues = {
      ordinal: 1,
      hour: 0,
      minute: 0,
      second: 0,
      millisecond: 0,
    };

  // Units in the supported calendars, sorted by bigness
  const orderedUnits = ["year", "month", "day", "hour", "minute", "second", "millisecond"],
    orderedWeekUnits = [
      "weekYear",
      "weekNumber",
      "weekday",
      "hour",
      "minute",
      "second",
      "millisecond",
    ],
    orderedOrdinalUnits = ["year", "ordinal", "hour", "minute", "second", "millisecond"];

  // standardize case and plurality in units
  function normalizeUnit(unit) {
    const normalized = {
      year: "year",
      years: "year",
      month: "month",
      months: "month",
      day: "day",
      days: "day",
      hour: "hour",
      hours: "hour",
      minute: "minute",
      minutes: "minute",
      quarter: "quarter",
      quarters: "quarter",
      second: "second",
      seconds: "second",
      millisecond: "millisecond",
      milliseconds: "millisecond",
      weekday: "weekday",
      weekdays: "weekday",
      weeknumber: "weekNumber",
      weeksnumber: "weekNumber",
      weeknumbers: "weekNumber",
      weekyear: "weekYear",
      weekyears: "weekYear",
      ordinal: "ordinal",
    }[unit.toLowerCase()];

    if (!normalized) throw new InvalidUnitError(unit);

    return normalized;
  }

  // this is a dumbed down version of fromObject() that runs about 60% faster
  // but doesn't do any validation, makes a bunch of assumptions about what units
  // are present, and so on.
  function quickDT(obj, opts) {
    const zone = normalizeZone(opts.zone, Settings.defaultZone),
      loc = Locale.fromObject(opts),
      tsNow = Settings.now();

    let ts, o;

    // assume we have the higher-order units
    if (!isUndefined(obj.year)) {
      for (const u of orderedUnits) {
        if (isUndefined(obj[u])) {
          obj[u] = defaultUnitValues[u];
        }
      }

      const invalid = hasInvalidGregorianData(obj) || hasInvalidTimeData(obj);
      if (invalid) {
        return DateTime.invalid(invalid);
      }

      const offsetProvis = zone.offset(tsNow);
      [ts, o] = objToTS(obj, offsetProvis, zone);
    } else {
      ts = tsNow;
    }

    return new DateTime({ ts, zone, loc, o });
  }

  function diffRelative(start, end, opts) {
    const round = isUndefined(opts.round) ? true : opts.round,
      format = (c, unit) => {
        c = roundTo(c, round || opts.calendary ? 0 : 2, true);
        const formatter = end.loc.clone(opts).relFormatter(opts);
        return formatter.format(c, unit);
      },
      differ = (unit) => {
        if (opts.calendary) {
          if (!end.hasSame(start, unit)) {
            return end.startOf(unit).diff(start.startOf(unit), unit).get(unit);
          } else return 0;
        } else {
          return end.diff(start, unit).get(unit);
        }
      };

    if (opts.unit) {
      return format(differ(opts.unit), opts.unit);
    }

    for (const unit of opts.units) {
      const count = differ(unit);
      if (Math.abs(count) >= 1) {
        return format(count, unit);
      }
    }
    return format(start > end ? -0 : 0, opts.units[opts.units.length - 1]);
  }

  function lastOpts(argList) {
    let opts = {},
      args;
    if (argList.length > 0 && typeof argList[argList.length - 1] === "object") {
      opts = argList[argList.length - 1];
      args = Array.from(argList).slice(0, argList.length - 1);
    } else {
      args = Array.from(argList);
    }
    return [opts, args];
  }

  /**
   * A DateTime is an immutable data structure representing a specific date and time and accompanying methods. It contains class and instance methods for creating, parsing, interrogating, transforming, and formatting them.
   *
   * A DateTime comprises of:
   * * A timestamp. Each DateTime instance refers to a specific millisecond of the Unix epoch.
   * * A time zone. Each instance is considered in the context of a specific zone (by default the local system's zone).
   * * Configuration properties that effect how output strings are formatted, such as `locale`, `numberingSystem`, and `outputCalendar`.
   *
   * Here is a brief overview of the most commonly used functionality it provides:
   *
   * * **Creation**: To create a DateTime from its components, use one of its factory class methods: {@link DateTime.local}, {@link DateTime.utc}, and (most flexibly) {@link DateTime.fromObject}. To create one from a standard string format, use {@link DateTime.fromISO}, {@link DateTime.fromHTTP}, and {@link DateTime.fromRFC2822}. To create one from a custom string format, use {@link DateTime.fromFormat}. To create one from a native JS date, use {@link DateTime.fromJSDate}.
   * * **Gregorian calendar and time**: To examine the Gregorian properties of a DateTime individually (i.e as opposed to collectively through {@link DateTime#toObject}), use the {@link DateTime#year}, {@link DateTime#month},
   * {@link DateTime#day}, {@link DateTime#hour}, {@link DateTime#minute}, {@link DateTime#second}, {@link DateTime#millisecond} accessors.
   * * **Week calendar**: For ISO week calendar attributes, see the {@link DateTime#weekYear}, {@link DateTime#weekNumber}, and {@link DateTime#weekday} accessors.
   * * **Configuration** See the {@link DateTime#locale} and {@link DateTime#numberingSystem} accessors.
   * * **Transformation**: To transform the DateTime into other DateTimes, use {@link DateTime#set}, {@link DateTime#reconfigure}, {@link DateTime#setZone}, {@link DateTime#setLocale}, {@link DateTime.plus}, {@link DateTime#minus}, {@link DateTime#endOf}, {@link DateTime#startOf}, {@link DateTime#toUTC}, and {@link DateTime#toLocal}.
   * * **Output**: To convert the DateTime to other representations, use the {@link DateTime#toRelative}, {@link DateTime#toRelativeCalendar}, {@link DateTime#toJSON}, {@link DateTime#toISO}, {@link DateTime#toHTTP}, {@link DateTime#toObject}, {@link DateTime#toRFC2822}, {@link DateTime#toString}, {@link DateTime#toLocaleString}, {@link DateTime#toFormat}, {@link DateTime#toMillis} and {@link DateTime#toJSDate}.
   *
   * There's plenty others documented below. In addition, for more information on subtler topics like internationalization, time zones, alternative calendars, validity, and so on, see the external documentation.
   */
  class DateTime {
    /**
     * @access private
     */
    constructor(config) {
      const zone = config.zone || Settings.defaultZone;

      let invalid =
        config.invalid ||
        (Number.isNaN(config.ts) ? new Invalid("invalid input") : null) ||
        (!zone.isValid ? unsupportedZone(zone) : null);
      /**
       * @access private
       */
      this.ts = isUndefined(config.ts) ? Settings.now() : config.ts;

      let c = null,
        o = null;
      if (!invalid) {
        const unchanged = config.old && config.old.ts === this.ts && config.old.zone.equals(zone);

        if (unchanged) {
          [c, o] = [config.old.c, config.old.o];
        } else {
          const ot = zone.offset(this.ts);
          c = tsToObj(this.ts, ot);
          invalid = Number.isNaN(c.year) ? new Invalid("invalid input") : null;
          c = invalid ? null : c;
          o = invalid ? null : ot;
        }
      }

      /**
       * @access private
       */
      this._zone = zone;
      /**
       * @access private
       */
      this.loc = config.loc || Locale.create();
      /**
       * @access private
       */
      this.invalid = invalid;
      /**
       * @access private
       */
      this.weekData = null;
      /**
       * @access private
       */
      this.c = c;
      /**
       * @access private
       */
      this.o = o;
      /**
       * @access private
       */
      this.isLuxonDateTime = true;
    }

    // CONSTRUCT

    /**
     * Create a DateTime for the current instant, in the system's time zone.
     *
     * Use Settings to override these default values if needed.
     * @example DateTime.now().toISO() //~> now in the ISO format
     * @return {DateTime}
     */
    static now() {
      return new DateTime({});
    }

    /**
     * Create a local DateTime
     * @param {number} [year] - The calendar year. If omitted (as in, call `local()` with no arguments), the current time will be used
     * @param {number} [month=1] - The month, 1-indexed
     * @param {number} [day=1] - The day of the month, 1-indexed
     * @param {number} [hour=0] - The hour of the day, in 24-hour time
     * @param {number} [minute=0] - The minute of the hour, meaning a number between 0 and 59
     * @param {number} [second=0] - The second of the minute, meaning a number between 0 and 59
     * @param {number} [millisecond=0] - The millisecond of the second, meaning a number between 0 and 999
     * @example DateTime.local()                                  //~> now
     * @example DateTime.local({ zone: "America/New_York" })      //~> now, in US east coast time
     * @example DateTime.local(2017)                              //~> 2017-01-01T00:00:00
     * @example DateTime.local(2017, 3)                           //~> 2017-03-01T00:00:00
     * @example DateTime.local(2017, 3, 12, { locale: "fr" })     //~> 2017-03-12T00:00:00, with a French locale
     * @example DateTime.local(2017, 3, 12, 5)                    //~> 2017-03-12T05:00:00
     * @example DateTime.local(2017, 3, 12, 5, { zone: "utc" })   //~> 2017-03-12T05:00:00, in UTC
     * @example DateTime.local(2017, 3, 12, 5, 45)                //~> 2017-03-12T05:45:00
     * @example DateTime.local(2017, 3, 12, 5, 45, 10)            //~> 2017-03-12T05:45:10
     * @example DateTime.local(2017, 3, 12, 5, 45, 10, 765)       //~> 2017-03-12T05:45:10.765
     * @return {DateTime}
     */
    static local() {
      const [opts, args] = lastOpts(arguments),
        [year, month, day, hour, minute, second, millisecond] = args;
      return quickDT({ year, month, day, hour, minute, second, millisecond }, opts);
    }

    /**
     * Create a DateTime in UTC
     * @param {number} [year] - The calendar year. If omitted (as in, call `utc()` with no arguments), the current time will be used
     * @param {number} [month=1] - The month, 1-indexed
     * @param {number} [day=1] - The day of the month
     * @param {number} [hour=0] - The hour of the day, in 24-hour time
     * @param {number} [minute=0] - The minute of the hour, meaning a number between 0 and 59
     * @param {number} [second=0] - The second of the minute, meaning a number between 0 and 59
     * @param {number} [millisecond=0] - The millisecond of the second, meaning a number between 0 and 999
     * @param {Object} options - configuration options for the DateTime
     * @param {string} [options.locale] - a locale to set on the resulting DateTime instance
     * @param {string} [options.outputCalendar] - the output calendar to set on the resulting DateTime instance
     * @param {string} [options.numberingSystem] - the numbering system to set on the resulting DateTime instance
     * @example DateTime.utc()                                              //~> now
     * @example DateTime.utc(2017)                                          //~> 2017-01-01T00:00:00Z
     * @example DateTime.utc(2017, 3)                                       //~> 2017-03-01T00:00:00Z
     * @example DateTime.utc(2017, 3, 12)                                   //~> 2017-03-12T00:00:00Z
     * @example DateTime.utc(2017, 3, 12, 5)                                //~> 2017-03-12T05:00:00Z
     * @example DateTime.utc(2017, 3, 12, 5, 45)                            //~> 2017-03-12T05:45:00Z
     * @example DateTime.utc(2017, 3, 12, 5, 45, { locale: "fr" })          //~> 2017-03-12T05:45:00Z with a French locale
     * @example DateTime.utc(2017, 3, 12, 5, 45, 10)                        //~> 2017-03-12T05:45:10Z
     * @example DateTime.utc(2017, 3, 12, 5, 45, 10, 765, { locale: "fr" }) //~> 2017-03-12T05:45:10.765Z with a French locale
     * @return {DateTime}
     */
    static utc() {
      const [opts, args] = lastOpts(arguments),
        [year, month, day, hour, minute, second, millisecond] = args;

      opts.zone = FixedOffsetZone.utcInstance;
      return quickDT({ year, month, day, hour, minute, second, millisecond }, opts);
    }

    /**
     * Create a DateTime from a JavaScript Date object. Uses the default zone.
     * @param {Date} date - a JavaScript Date object
     * @param {Object} options - configuration options for the DateTime
     * @param {string|Zone} [options.zone='local'] - the zone to place the DateTime into
     * @return {DateTime}
     */
    static fromJSDate(date, options = {}) {
      const ts = isDate(date) ? date.valueOf() : NaN;
      if (Number.isNaN(ts)) {
        return DateTime.invalid("invalid input");
      }

      const zoneToUse = normalizeZone(options.zone, Settings.defaultZone);
      if (!zoneToUse.isValid) {
        return DateTime.invalid(unsupportedZone(zoneToUse));
      }

      return new DateTime({
        ts: ts,
        zone: zoneToUse,
        loc: Locale.fromObject(options),
      });
    }

    /**
     * Create a DateTime from a number of milliseconds since the epoch (meaning since 1 January 1970 00:00:00 UTC). Uses the default zone.
     * @param {number} milliseconds - a number of milliseconds since 1970 UTC
     * @param {Object} options - configuration options for the DateTime
     * @param {string|Zone} [options.zone='local'] - the zone to place the DateTime into
     * @param {string} [options.locale] - a locale to set on the resulting DateTime instance
     * @param {string} options.outputCalendar - the output calendar to set on the resulting DateTime instance
     * @param {string} options.numberingSystem - the numbering system to set on the resulting DateTime instance
     * @return {DateTime}
     */
    static fromMillis(milliseconds, options = {}) {
      if (!isNumber$2(milliseconds)) {
        throw new InvalidArgumentError(
          `fromMillis requires a numerical input, but received a ${typeof milliseconds} with value ${milliseconds}`
        );
      } else if (milliseconds < -MAX_DATE || milliseconds > MAX_DATE) {
        // this isn't perfect because because we can still end up out of range because of additional shifting, but it's a start
        return DateTime.invalid("Timestamp out of range");
      } else {
        return new DateTime({
          ts: milliseconds,
          zone: normalizeZone(options.zone, Settings.defaultZone),
          loc: Locale.fromObject(options),
        });
      }
    }

    /**
     * Create a DateTime from a number of seconds since the epoch (meaning since 1 January 1970 00:00:00 UTC). Uses the default zone.
     * @param {number} seconds - a number of seconds since 1970 UTC
     * @param {Object} options - configuration options for the DateTime
     * @param {string|Zone} [options.zone='local'] - the zone to place the DateTime into
     * @param {string} [options.locale] - a locale to set on the resulting DateTime instance
     * @param {string} options.outputCalendar - the output calendar to set on the resulting DateTime instance
     * @param {string} options.numberingSystem - the numbering system to set on the resulting DateTime instance
     * @return {DateTime}
     */
    static fromSeconds(seconds, options = {}) {
      if (!isNumber$2(seconds)) {
        throw new InvalidArgumentError("fromSeconds requires a numerical input");
      } else {
        return new DateTime({
          ts: seconds * 1000,
          zone: normalizeZone(options.zone, Settings.defaultZone),
          loc: Locale.fromObject(options),
        });
      }
    }

    /**
     * Create a DateTime from a JavaScript object with keys like 'year' and 'hour' with reasonable defaults.
     * @param {Object} obj - the object to create the DateTime from
     * @param {number} obj.year - a year, such as 1987
     * @param {number} obj.month - a month, 1-12
     * @param {number} obj.day - a day of the month, 1-31, depending on the month
     * @param {number} obj.ordinal - day of the year, 1-365 or 366
     * @param {number} obj.weekYear - an ISO week year
     * @param {number} obj.weekNumber - an ISO week number, between 1 and 52 or 53, depending on the year
     * @param {number} obj.weekday - an ISO weekday, 1-7, where 1 is Monday and 7 is Sunday
     * @param {number} obj.hour - hour of the day, 0-23
     * @param {number} obj.minute - minute of the hour, 0-59
     * @param {number} obj.second - second of the minute, 0-59
     * @param {number} obj.millisecond - millisecond of the second, 0-999
     * @param {Object} opts - options for creating this DateTime
     * @param {string|Zone} [opts.zone='local'] - interpret the numbers in the context of a particular zone. Can take any value taken as the first argument to setZone()
     * @param {string} [opts.locale='system's locale'] - a locale to set on the resulting DateTime instance
     * @param {string} opts.outputCalendar - the output calendar to set on the resulting DateTime instance
     * @param {string} opts.numberingSystem - the numbering system to set on the resulting DateTime instance
     * @example DateTime.fromObject({ year: 1982, month: 5, day: 25}).toISODate() //=> '1982-05-25'
     * @example DateTime.fromObject({ year: 1982 }).toISODate() //=> '1982-01-01'
     * @example DateTime.fromObject({ hour: 10, minute: 26, second: 6 }) //~> today at 10:26:06
     * @example DateTime.fromObject({ hour: 10, minute: 26, second: 6 }, { zone: 'utc' }),
     * @example DateTime.fromObject({ hour: 10, minute: 26, second: 6 }, { zone: 'local' })
     * @example DateTime.fromObject({ hour: 10, minute: 26, second: 6 }, { zone: 'America/New_York' })
     * @example DateTime.fromObject({ weekYear: 2016, weekNumber: 2, weekday: 3 }).toISODate() //=> '2016-01-13'
     * @return {DateTime}
     */
    static fromObject(obj, opts = {}) {
      obj = obj || {};
      const zoneToUse = normalizeZone(opts.zone, Settings.defaultZone);
      if (!zoneToUse.isValid) {
        return DateTime.invalid(unsupportedZone(zoneToUse));
      }

      const tsNow = Settings.now(),
        offsetProvis = !isUndefined(opts.specificOffset)
          ? opts.specificOffset
          : zoneToUse.offset(tsNow),
        normalized = normalizeObject(obj, normalizeUnit),
        containsOrdinal = !isUndefined(normalized.ordinal),
        containsGregorYear = !isUndefined(normalized.year),
        containsGregorMD = !isUndefined(normalized.month) || !isUndefined(normalized.day),
        containsGregor = containsGregorYear || containsGregorMD,
        definiteWeekDef = normalized.weekYear || normalized.weekNumber,
        loc = Locale.fromObject(opts);

      // cases:
      // just a weekday -> this week's instance of that weekday, no worries
      // (gregorian data or ordinal) + (weekYear or weekNumber) -> error
      // (gregorian month or day) + ordinal -> error
      // otherwise just use weeks or ordinals or gregorian, depending on what's specified

      if ((containsGregor || containsOrdinal) && definiteWeekDef) {
        throw new ConflictingSpecificationError(
          "Can't mix weekYear/weekNumber units with year/month/day or ordinals"
        );
      }

      if (containsGregorMD && containsOrdinal) {
        throw new ConflictingSpecificationError("Can't mix ordinal dates with month/day");
      }

      const useWeekData = definiteWeekDef || (normalized.weekday && !containsGregor);

      // configure ourselves to deal with gregorian dates or week stuff
      let units,
        defaultValues,
        objNow = tsToObj(tsNow, offsetProvis);
      if (useWeekData) {
        units = orderedWeekUnits;
        defaultValues = defaultWeekUnitValues;
        objNow = gregorianToWeek(objNow);
      } else if (containsOrdinal) {
        units = orderedOrdinalUnits;
        defaultValues = defaultOrdinalUnitValues;
        objNow = gregorianToOrdinal(objNow);
      } else {
        units = orderedUnits;
        defaultValues = defaultUnitValues;
      }

      // set default values for missing stuff
      let foundFirst = false;
      for (const u of units) {
        const v = normalized[u];
        if (!isUndefined(v)) {
          foundFirst = true;
        } else if (foundFirst) {
          normalized[u] = defaultValues[u];
        } else {
          normalized[u] = objNow[u];
        }
      }

      // make sure the values we have are in range
      const higherOrderInvalid = useWeekData
          ? hasInvalidWeekData(normalized)
          : containsOrdinal
          ? hasInvalidOrdinalData(normalized)
          : hasInvalidGregorianData(normalized),
        invalid = higherOrderInvalid || hasInvalidTimeData(normalized);

      if (invalid) {
        return DateTime.invalid(invalid);
      }

      // compute the actual time
      const gregorian = useWeekData
          ? weekToGregorian(normalized)
          : containsOrdinal
          ? ordinalToGregorian(normalized)
          : normalized,
        [tsFinal, offsetFinal] = objToTS(gregorian, offsetProvis, zoneToUse),
        inst = new DateTime({
          ts: tsFinal,
          zone: zoneToUse,
          o: offsetFinal,
          loc,
        });

      // gregorian data + weekday serves only to validate
      if (normalized.weekday && containsGregor && obj.weekday !== inst.weekday) {
        return DateTime.invalid(
          "mismatched weekday",
          `you can't specify both a weekday of ${normalized.weekday} and a date of ${inst.toISO()}`
        );
      }

      return inst;
    }

    /**
     * Create a DateTime from an ISO 8601 string
     * @param {string} text - the ISO string
     * @param {Object} opts - options to affect the creation
     * @param {string|Zone} [opts.zone='local'] - use this zone if no offset is specified in the input string itself. Will also convert the time to this zone
     * @param {boolean} [opts.setZone=false] - override the zone with a fixed-offset zone specified in the string itself, if it specifies one
     * @param {string} [opts.locale='system's locale'] - a locale to set on the resulting DateTime instance
     * @param {string} [opts.outputCalendar] - the output calendar to set on the resulting DateTime instance
     * @param {string} [opts.numberingSystem] - the numbering system to set on the resulting DateTime instance
     * @example DateTime.fromISO('2016-05-25T09:08:34.123')
     * @example DateTime.fromISO('2016-05-25T09:08:34.123+06:00')
     * @example DateTime.fromISO('2016-05-25T09:08:34.123+06:00', {setZone: true})
     * @example DateTime.fromISO('2016-05-25T09:08:34.123', {zone: 'utc'})
     * @example DateTime.fromISO('2016-W05-4')
     * @return {DateTime}
     */
    static fromISO(text, opts = {}) {
      const [vals, parsedZone] = parseISODate(text);
      return parseDataToDateTime(vals, parsedZone, opts, "ISO 8601", text);
    }

    /**
     * Create a DateTime from an RFC 2822 string
     * @param {string} text - the RFC 2822 string
     * @param {Object} opts - options to affect the creation
     * @param {string|Zone} [opts.zone='local'] - convert the time to this zone. Since the offset is always specified in the string itself, this has no effect on the interpretation of string, merely the zone the resulting DateTime is expressed in.
     * @param {boolean} [opts.setZone=false] - override the zone with a fixed-offset zone specified in the string itself, if it specifies one
     * @param {string} [opts.locale='system's locale'] - a locale to set on the resulting DateTime instance
     * @param {string} opts.outputCalendar - the output calendar to set on the resulting DateTime instance
     * @param {string} opts.numberingSystem - the numbering system to set on the resulting DateTime instance
     * @example DateTime.fromRFC2822('25 Nov 2016 13:23:12 GMT')
     * @example DateTime.fromRFC2822('Fri, 25 Nov 2016 13:23:12 +0600')
     * @example DateTime.fromRFC2822('25 Nov 2016 13:23 Z')
     * @return {DateTime}
     */
    static fromRFC2822(text, opts = {}) {
      const [vals, parsedZone] = parseRFC2822Date(text);
      return parseDataToDateTime(vals, parsedZone, opts, "RFC 2822", text);
    }

    /**
     * Create a DateTime from an HTTP header date
     * @see https://www.w3.org/Protocols/rfc2616/rfc2616-sec3.html#sec3.3.1
     * @param {string} text - the HTTP header date
     * @param {Object} opts - options to affect the creation
     * @param {string|Zone} [opts.zone='local'] - convert the time to this zone. Since HTTP dates are always in UTC, this has no effect on the interpretation of string, merely the zone the resulting DateTime is expressed in.
     * @param {boolean} [opts.setZone=false] - override the zone with the fixed-offset zone specified in the string. For HTTP dates, this is always UTC, so this option is equivalent to setting the `zone` option to 'utc', but this option is included for consistency with similar methods.
     * @param {string} [opts.locale='system's locale'] - a locale to set on the resulting DateTime instance
     * @param {string} opts.outputCalendar - the output calendar to set on the resulting DateTime instance
     * @param {string} opts.numberingSystem - the numbering system to set on the resulting DateTime instance
     * @example DateTime.fromHTTP('Sun, 06 Nov 1994 08:49:37 GMT')
     * @example DateTime.fromHTTP('Sunday, 06-Nov-94 08:49:37 GMT')
     * @example DateTime.fromHTTP('Sun Nov  6 08:49:37 1994')
     * @return {DateTime}
     */
    static fromHTTP(text, opts = {}) {
      const [vals, parsedZone] = parseHTTPDate(text);
      return parseDataToDateTime(vals, parsedZone, opts, "HTTP", opts);
    }

    /**
     * Create a DateTime from an input string and format string.
     * Defaults to en-US if no locale has been specified, regardless of the system's locale. For a table of tokens and their interpretations, see [here](https://moment.github.io/luxon/#/parsing?id=table-of-tokens).
     * @param {string} text - the string to parse
     * @param {string} fmt - the format the string is expected to be in (see the link below for the formats)
     * @param {Object} opts - options to affect the creation
     * @param {string|Zone} [opts.zone='local'] - use this zone if no offset is specified in the input string itself. Will also convert the DateTime to this zone
     * @param {boolean} [opts.setZone=false] - override the zone with a zone specified in the string itself, if it specifies one
     * @param {string} [opts.locale='en-US'] - a locale string to use when parsing. Will also set the DateTime to this locale
     * @param {string} opts.numberingSystem - the numbering system to use when parsing. Will also set the resulting DateTime to this numbering system
     * @param {string} opts.outputCalendar - the output calendar to set on the resulting DateTime instance
     * @return {DateTime}
     */
    static fromFormat(text, fmt, opts = {}) {
      if (isUndefined(text) || isUndefined(fmt)) {
        throw new InvalidArgumentError("fromFormat requires an input string and a format");
      }

      const { locale = null, numberingSystem = null } = opts,
        localeToUse = Locale.fromOpts({
          locale,
          numberingSystem,
          defaultToEN: true,
        }),
        [vals, parsedZone, specificOffset, invalid] = parseFromTokens(localeToUse, text, fmt);
      if (invalid) {
        return DateTime.invalid(invalid);
      } else {
        return parseDataToDateTime(vals, parsedZone, opts, `format ${fmt}`, text, specificOffset);
      }
    }

    /**
     * @deprecated use fromFormat instead
     */
    static fromString(text, fmt, opts = {}) {
      return DateTime.fromFormat(text, fmt, opts);
    }

    /**
     * Create a DateTime from a SQL date, time, or datetime
     * Defaults to en-US if no locale has been specified, regardless of the system's locale
     * @param {string} text - the string to parse
     * @param {Object} opts - options to affect the creation
     * @param {string|Zone} [opts.zone='local'] - use this zone if no offset is specified in the input string itself. Will also convert the DateTime to this zone
     * @param {boolean} [opts.setZone=false] - override the zone with a zone specified in the string itself, if it specifies one
     * @param {string} [opts.locale='en-US'] - a locale string to use when parsing. Will also set the DateTime to this locale
     * @param {string} opts.numberingSystem - the numbering system to use when parsing. Will also set the resulting DateTime to this numbering system
     * @param {string} opts.outputCalendar - the output calendar to set on the resulting DateTime instance
     * @example DateTime.fromSQL('2017-05-15')
     * @example DateTime.fromSQL('2017-05-15 09:12:34')
     * @example DateTime.fromSQL('2017-05-15 09:12:34.342')
     * @example DateTime.fromSQL('2017-05-15 09:12:34.342+06:00')
     * @example DateTime.fromSQL('2017-05-15 09:12:34.342 America/Los_Angeles')
     * @example DateTime.fromSQL('2017-05-15 09:12:34.342 America/Los_Angeles', { setZone: true })
     * @example DateTime.fromSQL('2017-05-15 09:12:34.342', { zone: 'America/Los_Angeles' })
     * @example DateTime.fromSQL('09:12:34.342')
     * @return {DateTime}
     */
    static fromSQL(text, opts = {}) {
      const [vals, parsedZone] = parseSQL(text);
      return parseDataToDateTime(vals, parsedZone, opts, "SQL", text);
    }

    /**
     * Create an invalid DateTime.
     * @param {DateTime} reason - simple string of why this DateTime is invalid. Should not contain parameters or anything else data-dependent
     * @param {string} [explanation=null] - longer explanation, may include parameters and other useful debugging information
     * @return {DateTime}
     */
    static invalid(reason, explanation = null) {
      if (!reason) {
        throw new InvalidArgumentError("need to specify a reason the DateTime is invalid");
      }

      const invalid = reason instanceof Invalid ? reason : new Invalid(reason, explanation);

      if (Settings.throwOnInvalid) {
        throw new InvalidDateTimeError(invalid);
      } else {
        return new DateTime({ invalid });
      }
    }

    /**
     * Check if an object is an instance of DateTime. Works across context boundaries
     * @param {object} o
     * @return {boolean}
     */
    static isDateTime(o) {
      return (o && o.isLuxonDateTime) || false;
    }

    /**
     * Produce the format string for a set of options
     * @param formatOpts
     * @param localeOpts
     * @returns {string}
     */
    static parseFormatForOpts(formatOpts, localeOpts = {}) {
      const tokenList = formatOptsToTokens(formatOpts, Locale.fromObject(localeOpts));
      return !tokenList ? null : tokenList.map((t) => (t ? t.val : null)).join("");
    }

    /**
     * Produce the the fully expanded format token for the locale
     * Does NOT quote characters, so quoted tokens will not round trip correctly
     * @param fmt
     * @param localeOpts
     * @returns {string}
     */
    static expandFormat(fmt, localeOpts = {}) {
      const expanded = expandMacroTokens(Formatter.parseFormat(fmt), Locale.fromObject(localeOpts));
      return expanded.map((t) => t.val).join("");
    }

    // INFO

    /**
     * Get the value of unit.
     * @param {string} unit - a unit such as 'minute' or 'day'
     * @example DateTime.local(2017, 7, 4).get('month'); //=> 7
     * @example DateTime.local(2017, 7, 4).get('day'); //=> 4
     * @return {number}
     */
    get(unit) {
      return this[unit];
    }

    /**
     * Returns whether the DateTime is valid. Invalid DateTimes occur when:
     * * The DateTime was created from invalid calendar information, such as the 13th month or February 30
     * * The DateTime was created by an operation on another invalid date
     * @type {boolean}
     */
    get isValid() {
      return this.invalid === null;
    }

    /**
     * Returns an error code if this DateTime is invalid, or null if the DateTime is valid
     * @type {string}
     */
    get invalidReason() {
      return this.invalid ? this.invalid.reason : null;
    }

    /**
     * Returns an explanation of why this DateTime became invalid, or null if the DateTime is valid
     * @type {string}
     */
    get invalidExplanation() {
      return this.invalid ? this.invalid.explanation : null;
    }

    /**
     * Get the locale of a DateTime, such 'en-GB'. The locale is used when formatting the DateTime
     *
     * @type {string}
     */
    get locale() {
      return this.isValid ? this.loc.locale : null;
    }

    /**
     * Get the numbering system of a DateTime, such 'beng'. The numbering system is used when formatting the DateTime
     *
     * @type {string}
     */
    get numberingSystem() {
      return this.isValid ? this.loc.numberingSystem : null;
    }

    /**
     * Get the output calendar of a DateTime, such 'islamic'. The output calendar is used when formatting the DateTime
     *
     * @type {string}
     */
    get outputCalendar() {
      return this.isValid ? this.loc.outputCalendar : null;
    }

    /**
     * Get the time zone associated with this DateTime.
     * @type {Zone}
     */
    get zone() {
      return this._zone;
    }

    /**
     * Get the name of the time zone.
     * @type {string}
     */
    get zoneName() {
      return this.isValid ? this.zone.name : null;
    }

    /**
     * Get the year
     * @example DateTime.local(2017, 5, 25).year //=> 2017
     * @type {number}
     */
    get year() {
      return this.isValid ? this.c.year : NaN;
    }

    /**
     * Get the quarter
     * @example DateTime.local(2017, 5, 25).quarter //=> 2
     * @type {number}
     */
    get quarter() {
      return this.isValid ? Math.ceil(this.c.month / 3) : NaN;
    }

    /**
     * Get the month (1-12).
     * @example DateTime.local(2017, 5, 25).month //=> 5
     * @type {number}
     */
    get month() {
      return this.isValid ? this.c.month : NaN;
    }

    /**
     * Get the day of the month (1-30ish).
     * @example DateTime.local(2017, 5, 25).day //=> 25
     * @type {number}
     */
    get day() {
      return this.isValid ? this.c.day : NaN;
    }

    /**
     * Get the hour of the day (0-23).
     * @example DateTime.local(2017, 5, 25, 9).hour //=> 9
     * @type {number}
     */
    get hour() {
      return this.isValid ? this.c.hour : NaN;
    }

    /**
     * Get the minute of the hour (0-59).
     * @example DateTime.local(2017, 5, 25, 9, 30).minute //=> 30
     * @type {number}
     */
    get minute() {
      return this.isValid ? this.c.minute : NaN;
    }

    /**
     * Get the second of the minute (0-59).
     * @example DateTime.local(2017, 5, 25, 9, 30, 52).second //=> 52
     * @type {number}
     */
    get second() {
      return this.isValid ? this.c.second : NaN;
    }

    /**
     * Get the millisecond of the second (0-999).
     * @example DateTime.local(2017, 5, 25, 9, 30, 52, 654).millisecond //=> 654
     * @type {number}
     */
    get millisecond() {
      return this.isValid ? this.c.millisecond : NaN;
    }

    /**
     * Get the week year
     * @see https://en.wikipedia.org/wiki/ISO_week_date
     * @example DateTime.local(2014, 12, 31).weekYear //=> 2015
     * @type {number}
     */
    get weekYear() {
      return this.isValid ? possiblyCachedWeekData(this).weekYear : NaN;
    }

    /**
     * Get the week number of the week year (1-52ish).
     * @see https://en.wikipedia.org/wiki/ISO_week_date
     * @example DateTime.local(2017, 5, 25).weekNumber //=> 21
     * @type {number}
     */
    get weekNumber() {
      return this.isValid ? possiblyCachedWeekData(this).weekNumber : NaN;
    }

    /**
     * Get the day of the week.
     * 1 is Monday and 7 is Sunday
     * @see https://en.wikipedia.org/wiki/ISO_week_date
     * @example DateTime.local(2014, 11, 31).weekday //=> 4
     * @type {number}
     */
    get weekday() {
      return this.isValid ? possiblyCachedWeekData(this).weekday : NaN;
    }

    /**
     * Get the ordinal (meaning the day of the year)
     * @example DateTime.local(2017, 5, 25).ordinal //=> 145
     * @type {number|DateTime}
     */
    get ordinal() {
      return this.isValid ? gregorianToOrdinal(this.c).ordinal : NaN;
    }

    /**
     * Get the human readable short month name, such as 'Oct'.
     * Defaults to the system's locale if no locale has been specified
     * @example DateTime.local(2017, 10, 30).monthShort //=> Oct
     * @type {string}
     */
    get monthShort() {
      return this.isValid ? Info.months("short", { locObj: this.loc })[this.month - 1] : null;
    }

    /**
     * Get the human readable long month name, such as 'October'.
     * Defaults to the system's locale if no locale has been specified
     * @example DateTime.local(2017, 10, 30).monthLong //=> October
     * @type {string}
     */
    get monthLong() {
      return this.isValid ? Info.months("long", { locObj: this.loc })[this.month - 1] : null;
    }

    /**
     * Get the human readable short weekday, such as 'Mon'.
     * Defaults to the system's locale if no locale has been specified
     * @example DateTime.local(2017, 10, 30).weekdayShort //=> Mon
     * @type {string}
     */
    get weekdayShort() {
      return this.isValid ? Info.weekdays("short", { locObj: this.loc })[this.weekday - 1] : null;
    }

    /**
     * Get the human readable long weekday, such as 'Monday'.
     * Defaults to the system's locale if no locale has been specified
     * @example DateTime.local(2017, 10, 30).weekdayLong //=> Monday
     * @type {string}
     */
    get weekdayLong() {
      return this.isValid ? Info.weekdays("long", { locObj: this.loc })[this.weekday - 1] : null;
    }

    /**
     * Get the UTC offset of this DateTime in minutes
     * @example DateTime.now().offset //=> -240
     * @example DateTime.utc().offset //=> 0
     * @type {number}
     */
    get offset() {
      return this.isValid ? +this.o : NaN;
    }

    /**
     * Get the short human name for the zone's current offset, for example "EST" or "EDT".
     * Defaults to the system's locale if no locale has been specified
     * @type {string}
     */
    get offsetNameShort() {
      if (this.isValid) {
        return this.zone.offsetName(this.ts, {
          format: "short",
          locale: this.locale,
        });
      } else {
        return null;
      }
    }

    /**
     * Get the long human name for the zone's current offset, for example "Eastern Standard Time" or "Eastern Daylight Time".
     * Defaults to the system's locale if no locale has been specified
     * @type {string}
     */
    get offsetNameLong() {
      if (this.isValid) {
        return this.zone.offsetName(this.ts, {
          format: "long",
          locale: this.locale,
        });
      } else {
        return null;
      }
    }

    /**
     * Get whether this zone's offset ever changes, as in a DST.
     * @type {boolean}
     */
    get isOffsetFixed() {
      return this.isValid ? this.zone.isUniversal : null;
    }

    /**
     * Get whether the DateTime is in a DST.
     * @type {boolean}
     */
    get isInDST() {
      if (this.isOffsetFixed) {
        return false;
      } else {
        return (
          this.offset > this.set({ month: 1, day: 1 }).offset ||
          this.offset > this.set({ month: 5 }).offset
        );
      }
    }

    /**
     * Returns true if this DateTime is in a leap year, false otherwise
     * @example DateTime.local(2016).isInLeapYear //=> true
     * @example DateTime.local(2013).isInLeapYear //=> false
     * @type {boolean}
     */
    get isInLeapYear() {
      return isLeapYear(this.year);
    }

    /**
     * Returns the number of days in this DateTime's month
     * @example DateTime.local(2016, 2).daysInMonth //=> 29
     * @example DateTime.local(2016, 3).daysInMonth //=> 31
     * @type {number}
     */
    get daysInMonth() {
      return daysInMonth(this.year, this.month);
    }

    /**
     * Returns the number of days in this DateTime's year
     * @example DateTime.local(2016).daysInYear //=> 366
     * @example DateTime.local(2013).daysInYear //=> 365
     * @type {number}
     */
    get daysInYear() {
      return this.isValid ? daysInYear(this.year) : NaN;
    }

    /**
     * Returns the number of weeks in this DateTime's year
     * @see https://en.wikipedia.org/wiki/ISO_week_date
     * @example DateTime.local(2004).weeksInWeekYear //=> 53
     * @example DateTime.local(2013).weeksInWeekYear //=> 52
     * @type {number}
     */
    get weeksInWeekYear() {
      return this.isValid ? weeksInWeekYear(this.weekYear) : NaN;
    }

    /**
     * Returns the resolved Intl options for this DateTime.
     * This is useful in understanding the behavior of formatting methods
     * @param {Object} opts - the same options as toLocaleString
     * @return {Object}
     */
    resolvedLocaleOptions(opts = {}) {
      const { locale, numberingSystem, calendar } = Formatter.create(
        this.loc.clone(opts),
        opts
      ).resolvedOptions(this);
      return { locale, numberingSystem, outputCalendar: calendar };
    }

    // TRANSFORM

    /**
     * "Set" the DateTime's zone to UTC. Returns a newly-constructed DateTime.
     *
     * Equivalent to {@link DateTime#setZone}('utc')
     * @param {number} [offset=0] - optionally, an offset from UTC in minutes
     * @param {Object} [opts={}] - options to pass to `setZone()`
     * @return {DateTime}
     */
    toUTC(offset = 0, opts = {}) {
      return this.setZone(FixedOffsetZone.instance(offset), opts);
    }

    /**
     * "Set" the DateTime's zone to the host's local zone. Returns a newly-constructed DateTime.
     *
     * Equivalent to `setZone('local')`
     * @return {DateTime}
     */
    toLocal() {
      return this.setZone(Settings.defaultZone);
    }

    /**
     * "Set" the DateTime's zone to specified zone. Returns a newly-constructed DateTime.
     *
     * By default, the setter keeps the underlying time the same (as in, the same timestamp), but the new instance will report different local times and consider DSTs when making computations, as with {@link DateTime#plus}. You may wish to use {@link DateTime#toLocal} and {@link DateTime#toUTC} which provide simple convenience wrappers for commonly used zones.
     * @param {string|Zone} [zone='local'] - a zone identifier. As a string, that can be any IANA zone supported by the host environment, or a fixed-offset name of the form 'UTC+3', or the strings 'local' or 'utc'. You may also supply an instance of a {@link DateTime#Zone} class.
     * @param {Object} opts - options
     * @param {boolean} [opts.keepLocalTime=false] - If true, adjust the underlying time so that the local time stays the same, but in the target zone. You should rarely need this.
     * @return {DateTime}
     */
    setZone(zone, { keepLocalTime = false, keepCalendarTime = false } = {}) {
      zone = normalizeZone(zone, Settings.defaultZone);
      if (zone.equals(this.zone)) {
        return this;
      } else if (!zone.isValid) {
        return DateTime.invalid(unsupportedZone(zone));
      } else {
        let newTS = this.ts;
        if (keepLocalTime || keepCalendarTime) {
          const offsetGuess = zone.offset(this.ts);
          const asObj = this.toObject();
          [newTS] = objToTS(asObj, offsetGuess, zone);
        }
        return clone$1(this, { ts: newTS, zone });
      }
    }

    /**
     * "Set" the locale, numberingSystem, or outputCalendar. Returns a newly-constructed DateTime.
     * @param {Object} properties - the properties to set
     * @example DateTime.local(2017, 5, 25).reconfigure({ locale: 'en-GB' })
     * @return {DateTime}
     */
    reconfigure({ locale, numberingSystem, outputCalendar } = {}) {
      const loc = this.loc.clone({ locale, numberingSystem, outputCalendar });
      return clone$1(this, { loc });
    }

    /**
     * "Set" the locale. Returns a newly-constructed DateTime.
     * Just a convenient alias for reconfigure({ locale })
     * @example DateTime.local(2017, 5, 25).setLocale('en-GB')
     * @return {DateTime}
     */
    setLocale(locale) {
      return this.reconfigure({ locale });
    }

    /**
     * "Set" the values of specified units. Returns a newly-constructed DateTime.
     * You can only set units with this method; for "setting" metadata, see {@link DateTime#reconfigure} and {@link DateTime#setZone}.
     * @param {Object} values - a mapping of units to numbers
     * @example dt.set({ year: 2017 })
     * @example dt.set({ hour: 8, minute: 30 })
     * @example dt.set({ weekday: 5 })
     * @example dt.set({ year: 2005, ordinal: 234 })
     * @return {DateTime}
     */
    set(values) {
      if (!this.isValid) return this;

      const normalized = normalizeObject(values, normalizeUnit),
        settingWeekStuff =
          !isUndefined(normalized.weekYear) ||
          !isUndefined(normalized.weekNumber) ||
          !isUndefined(normalized.weekday),
        containsOrdinal = !isUndefined(normalized.ordinal),
        containsGregorYear = !isUndefined(normalized.year),
        containsGregorMD = !isUndefined(normalized.month) || !isUndefined(normalized.day),
        containsGregor = containsGregorYear || containsGregorMD,
        definiteWeekDef = normalized.weekYear || normalized.weekNumber;

      if ((containsGregor || containsOrdinal) && definiteWeekDef) {
        throw new ConflictingSpecificationError(
          "Can't mix weekYear/weekNumber units with year/month/day or ordinals"
        );
      }

      if (containsGregorMD && containsOrdinal) {
        throw new ConflictingSpecificationError("Can't mix ordinal dates with month/day");
      }

      let mixed;
      if (settingWeekStuff) {
        mixed = weekToGregorian({ ...gregorianToWeek(this.c), ...normalized });
      } else if (!isUndefined(normalized.ordinal)) {
        mixed = ordinalToGregorian({ ...gregorianToOrdinal(this.c), ...normalized });
      } else {
        mixed = { ...this.toObject(), ...normalized };

        // if we didn't set the day but we ended up on an overflow date,
        // use the last day of the right month
        if (isUndefined(normalized.day)) {
          mixed.day = Math.min(daysInMonth(mixed.year, mixed.month), mixed.day);
        }
      }

      const [ts, o] = objToTS(mixed, this.o, this.zone);
      return clone$1(this, { ts, o });
    }

    /**
     * Add a period of time to this DateTime and return the resulting DateTime
     *
     * Adding hours, minutes, seconds, or milliseconds increases the timestamp by the right number of milliseconds. Adding days, months, or years shifts the calendar, accounting for DSTs and leap years along the way. Thus, `dt.plus({ hours: 24 })` may result in a different time than `dt.plus({ days: 1 })` if there's a DST shift in between.
     * @param {Duration|Object|number} duration - The amount to add. Either a Luxon Duration, a number of milliseconds, the object argument to Duration.fromObject()
     * @example DateTime.now().plus(123) //~> in 123 milliseconds
     * @example DateTime.now().plus({ minutes: 15 }) //~> in 15 minutes
     * @example DateTime.now().plus({ days: 1 }) //~> this time tomorrow
     * @example DateTime.now().plus({ days: -1 }) //~> this time yesterday
     * @example DateTime.now().plus({ hours: 3, minutes: 13 }) //~> in 3 hr, 13 min
     * @example DateTime.now().plus(Duration.fromObject({ hours: 3, minutes: 13 })) //~> in 3 hr, 13 min
     * @return {DateTime}
     */
    plus(duration) {
      if (!this.isValid) return this;
      const dur = Duration.fromDurationLike(duration);
      return clone$1(this, adjustTime(this, dur));
    }

    /**
     * Subtract a period of time to this DateTime and return the resulting DateTime
     * See {@link DateTime#plus}
     * @param {Duration|Object|number} duration - The amount to subtract. Either a Luxon Duration, a number of milliseconds, the object argument to Duration.fromObject()
     @return {DateTime}
     */
    minus(duration) {
      if (!this.isValid) return this;
      const dur = Duration.fromDurationLike(duration).negate();
      return clone$1(this, adjustTime(this, dur));
    }

    /**
     * "Set" this DateTime to the beginning of a unit of time.
     * @param {string} unit - The unit to go to the beginning of. Can be 'year', 'quarter', 'month', 'week', 'day', 'hour', 'minute', 'second', or 'millisecond'.
     * @example DateTime.local(2014, 3, 3).startOf('month').toISODate(); //=> '2014-03-01'
     * @example DateTime.local(2014, 3, 3).startOf('year').toISODate(); //=> '2014-01-01'
     * @example DateTime.local(2014, 3, 3).startOf('week').toISODate(); //=> '2014-03-03', weeks always start on Mondays
     * @example DateTime.local(2014, 3, 3, 5, 30).startOf('day').toISOTime(); //=> '00:00.000-05:00'
     * @example DateTime.local(2014, 3, 3, 5, 30).startOf('hour').toISOTime(); //=> '05:00:00.000-05:00'
     * @return {DateTime}
     */
    startOf(unit) {
      if (!this.isValid) return this;
      const o = {},
        normalizedUnit = Duration.normalizeUnit(unit);
      switch (normalizedUnit) {
        case "years":
          o.month = 1;
        // falls through
        case "quarters":
        case "months":
          o.day = 1;
        // falls through
        case "weeks":
        case "days":
          o.hour = 0;
        // falls through
        case "hours":
          o.minute = 0;
        // falls through
        case "minutes":
          o.second = 0;
        // falls through
        case "seconds":
          o.millisecond = 0;
          break;
        // no default, invalid units throw in normalizeUnit()
      }

      if (normalizedUnit === "weeks") {
        o.weekday = 1;
      }

      if (normalizedUnit === "quarters") {
        const q = Math.ceil(this.month / 3);
        o.month = (q - 1) * 3 + 1;
      }

      return this.set(o);
    }

    /**
     * "Set" this DateTime to the end (meaning the last millisecond) of a unit of time
     * @param {string} unit - The unit to go to the end of. Can be 'year', 'quarter', 'month', 'week', 'day', 'hour', 'minute', 'second', or 'millisecond'.
     * @example DateTime.local(2014, 3, 3).endOf('month').toISO(); //=> '2014-03-31T23:59:59.999-05:00'
     * @example DateTime.local(2014, 3, 3).endOf('year').toISO(); //=> '2014-12-31T23:59:59.999-05:00'
     * @example DateTime.local(2014, 3, 3).endOf('week').toISO(); // => '2014-03-09T23:59:59.999-05:00', weeks start on Mondays
     * @example DateTime.local(2014, 3, 3, 5, 30).endOf('day').toISO(); //=> '2014-03-03T23:59:59.999-05:00'
     * @example DateTime.local(2014, 3, 3, 5, 30).endOf('hour').toISO(); //=> '2014-03-03T05:59:59.999-05:00'
     * @return {DateTime}
     */
    endOf(unit) {
      return this.isValid
        ? this.plus({ [unit]: 1 })
            .startOf(unit)
            .minus(1)
        : this;
    }

    // OUTPUT

    /**
     * Returns a string representation of this DateTime formatted according to the specified format string.
     * **You may not want this.** See {@link DateTime#toLocaleString} for a more flexible formatting tool. For a table of tokens and their interpretations, see [here](https://moment.github.io/luxon/#/formatting?id=table-of-tokens).
     * Defaults to en-US if no locale has been specified, regardless of the system's locale.
     * @param {string} fmt - the format string
     * @param {Object} opts - opts to override the configuration options on this DateTime
     * @example DateTime.now().toFormat('yyyy LLL dd') //=> '2017 Apr 22'
     * @example DateTime.now().setLocale('fr').toFormat('yyyy LLL dd') //=> '2017 avr. 22'
     * @example DateTime.now().toFormat('yyyy LLL dd', { locale: "fr" }) //=> '2017 avr. 22'
     * @example DateTime.now().toFormat("HH 'hours and' mm 'minutes'") //=> '20 hours and 55 minutes'
     * @return {string}
     */
    toFormat(fmt, opts = {}) {
      return this.isValid
        ? Formatter.create(this.loc.redefaultToEN(opts)).formatDateTimeFromString(this, fmt)
        : INVALID$1;
    }

    /**
     * Returns a localized string representing this date. Accepts the same options as the Intl.DateTimeFormat constructor and any presets defined by Luxon, such as `DateTime.DATE_FULL` or `DateTime.TIME_SIMPLE`.
     * The exact behavior of this method is browser-specific, but in general it will return an appropriate representation
     * of the DateTime in the assigned locale.
     * Defaults to the system's locale if no locale has been specified
     * @see https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/DateTimeFormat
     * @param formatOpts {Object} - Intl.DateTimeFormat constructor options and configuration options
     * @param {Object} opts - opts to override the configuration options on this DateTime
     * @example DateTime.now().toLocaleString(); //=> 4/20/2017
     * @example DateTime.now().setLocale('en-gb').toLocaleString(); //=> '20/04/2017'
     * @example DateTime.now().toLocaleString(DateTime.DATE_FULL); //=> 'April 20, 2017'
     * @example DateTime.now().toLocaleString(DateTime.DATE_FULL, { locale: 'fr' }); //=> '28 août 2022'
     * @example DateTime.now().toLocaleString(DateTime.TIME_SIMPLE); //=> '11:32 AM'
     * @example DateTime.now().toLocaleString(DateTime.DATETIME_SHORT); //=> '4/20/2017, 11:32 AM'
     * @example DateTime.now().toLocaleString({ weekday: 'long', month: 'long', day: '2-digit' }); //=> 'Thursday, April 20'
     * @example DateTime.now().toLocaleString({ weekday: 'short', month: 'short', day: '2-digit', hour: '2-digit', minute: '2-digit' }); //=> 'Thu, Apr 20, 11:27 AM'
     * @example DateTime.now().toLocaleString({ hour: '2-digit', minute: '2-digit', hourCycle: 'h23' }); //=> '11:32'
     * @return {string}
     */
    toLocaleString(formatOpts = DATE_SHORT, opts = {}) {
      return this.isValid
        ? Formatter.create(this.loc.clone(opts), formatOpts).formatDateTime(this)
        : INVALID$1;
    }

    /**
     * Returns an array of format "parts", meaning individual tokens along with metadata. This is allows callers to post-process individual sections of the formatted output.
     * Defaults to the system's locale if no locale has been specified
     * @see https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/DateTimeFormat/formatToParts
     * @param opts {Object} - Intl.DateTimeFormat constructor options, same as `toLocaleString`.
     * @example DateTime.now().toLocaleParts(); //=> [
     *                                   //=>   { type: 'day', value: '25' },
     *                                   //=>   { type: 'literal', value: '/' },
     *                                   //=>   { type: 'month', value: '05' },
     *                                   //=>   { type: 'literal', value: '/' },
     *                                   //=>   { type: 'year', value: '1982' }
     *                                   //=> ]
     */
    toLocaleParts(opts = {}) {
      return this.isValid
        ? Formatter.create(this.loc.clone(opts), opts).formatDateTimeParts(this)
        : [];
    }

    /**
     * Returns an ISO 8601-compliant string representation of this DateTime
     * @param {Object} opts - options
     * @param {boolean} [opts.suppressMilliseconds=false] - exclude milliseconds from the format if they're 0
     * @param {boolean} [opts.suppressSeconds=false] - exclude seconds from the format if they're 0
     * @param {boolean} [opts.includeOffset=true] - include the offset, such as 'Z' or '-04:00'
     * @param {boolean} [opts.extendedZone=false] - add the time zone format extension
     * @param {string} [opts.format='extended'] - choose between the basic and extended format
     * @example DateTime.utc(1983, 5, 25).toISO() //=> '1982-05-25T00:00:00.000Z'
     * @example DateTime.now().toISO() //=> '2017-04-22T20:47:05.335-04:00'
     * @example DateTime.now().toISO({ includeOffset: false }) //=> '2017-04-22T20:47:05.335'
     * @example DateTime.now().toISO({ format: 'basic' }) //=> '20170422T204705.335-0400'
     * @return {string}
     */
    toISO({
      format = "extended",
      suppressSeconds = false,
      suppressMilliseconds = false,
      includeOffset = true,
      extendedZone = false,
    } = {}) {
      if (!this.isValid) {
        return null;
      }

      const ext = format === "extended";

      let c = toISODate(this, ext);
      c += "T";
      c += toISOTime(this, ext, suppressSeconds, suppressMilliseconds, includeOffset, extendedZone);
      return c;
    }

    /**
     * Returns an ISO 8601-compliant string representation of this DateTime's date component
     * @param {Object} opts - options
     * @param {string} [opts.format='extended'] - choose between the basic and extended format
     * @example DateTime.utc(1982, 5, 25).toISODate() //=> '1982-05-25'
     * @example DateTime.utc(1982, 5, 25).toISODate({ format: 'basic' }) //=> '19820525'
     * @return {string}
     */
    toISODate({ format = "extended" } = {}) {
      if (!this.isValid) {
        return null;
      }

      return toISODate(this, format === "extended");
    }

    /**
     * Returns an ISO 8601-compliant string representation of this DateTime's week date
     * @example DateTime.utc(1982, 5, 25).toISOWeekDate() //=> '1982-W21-2'
     * @return {string}
     */
    toISOWeekDate() {
      return toTechFormat(this, "kkkk-'W'WW-c");
    }

    /**
     * Returns an ISO 8601-compliant string representation of this DateTime's time component
     * @param {Object} opts - options
     * @param {boolean} [opts.suppressMilliseconds=false] - exclude milliseconds from the format if they're 0
     * @param {boolean} [opts.suppressSeconds=false] - exclude seconds from the format if they're 0
     * @param {boolean} [opts.includeOffset=true] - include the offset, such as 'Z' or '-04:00'
     * @param {boolean} [opts.extendedZone=true] - add the time zone format extension
     * @param {boolean} [opts.includePrefix=false] - include the `T` prefix
     * @param {string} [opts.format='extended'] - choose between the basic and extended format
     * @example DateTime.utc().set({ hour: 7, minute: 34 }).toISOTime() //=> '07:34:19.361Z'
     * @example DateTime.utc().set({ hour: 7, minute: 34, seconds: 0, milliseconds: 0 }).toISOTime({ suppressSeconds: true }) //=> '07:34Z'
     * @example DateTime.utc().set({ hour: 7, minute: 34 }).toISOTime({ format: 'basic' }) //=> '073419.361Z'
     * @example DateTime.utc().set({ hour: 7, minute: 34 }).toISOTime({ includePrefix: true }) //=> 'T07:34:19.361Z'
     * @return {string}
     */
    toISOTime({
      suppressMilliseconds = false,
      suppressSeconds = false,
      includeOffset = true,
      includePrefix = false,
      extendedZone = false,
      format = "extended",
    } = {}) {
      if (!this.isValid) {
        return null;
      }

      let c = includePrefix ? "T" : "";
      return (
        c +
        toISOTime(
          this,
          format === "extended",
          suppressSeconds,
          suppressMilliseconds,
          includeOffset,
          extendedZone
        )
      );
    }

    /**
     * Returns an RFC 2822-compatible string representation of this DateTime
     * @example DateTime.utc(2014, 7, 13).toRFC2822() //=> 'Sun, 13 Jul 2014 00:00:00 +0000'
     * @example DateTime.local(2014, 7, 13).toRFC2822() //=> 'Sun, 13 Jul 2014 00:00:00 -0400'
     * @return {string}
     */
    toRFC2822() {
      return toTechFormat(this, "EEE, dd LLL yyyy HH:mm:ss ZZZ", false);
    }

    /**
     * Returns a string representation of this DateTime appropriate for use in HTTP headers. The output is always expressed in GMT.
     * Specifically, the string conforms to RFC 1123.
     * @see https://www.w3.org/Protocols/rfc2616/rfc2616-sec3.html#sec3.3.1
     * @example DateTime.utc(2014, 7, 13).toHTTP() //=> 'Sun, 13 Jul 2014 00:00:00 GMT'
     * @example DateTime.utc(2014, 7, 13, 19).toHTTP() //=> 'Sun, 13 Jul 2014 19:00:00 GMT'
     * @return {string}
     */
    toHTTP() {
      return toTechFormat(this.toUTC(), "EEE, dd LLL yyyy HH:mm:ss 'GMT'");
    }

    /**
     * Returns a string representation of this DateTime appropriate for use in SQL Date
     * @example DateTime.utc(2014, 7, 13).toSQLDate() //=> '2014-07-13'
     * @return {string}
     */
    toSQLDate() {
      if (!this.isValid) {
        return null;
      }
      return toISODate(this, true);
    }

    /**
     * Returns a string representation of this DateTime appropriate for use in SQL Time
     * @param {Object} opts - options
     * @param {boolean} [opts.includeZone=false] - include the zone, such as 'America/New_York'. Overrides includeOffset.
     * @param {boolean} [opts.includeOffset=true] - include the offset, such as 'Z' or '-04:00'
     * @param {boolean} [opts.includeOffsetSpace=true] - include the space between the time and the offset, such as '05:15:16.345 -04:00'
     * @example DateTime.utc().toSQL() //=> '05:15:16.345'
     * @example DateTime.now().toSQL() //=> '05:15:16.345 -04:00'
     * @example DateTime.now().toSQL({ includeOffset: false }) //=> '05:15:16.345'
     * @example DateTime.now().toSQL({ includeZone: false }) //=> '05:15:16.345 America/New_York'
     * @return {string}
     */
    toSQLTime({ includeOffset = true, includeZone = false, includeOffsetSpace = true } = {}) {
      let fmt = "HH:mm:ss.SSS";

      if (includeZone || includeOffset) {
        if (includeOffsetSpace) {
          fmt += " ";
        }
        if (includeZone) {
          fmt += "z";
        } else if (includeOffset) {
          fmt += "ZZ";
        }
      }

      return toTechFormat(this, fmt, true);
    }

    /**
     * Returns a string representation of this DateTime appropriate for use in SQL DateTime
     * @param {Object} opts - options
     * @param {boolean} [opts.includeZone=false] - include the zone, such as 'America/New_York'. Overrides includeOffset.
     * @param {boolean} [opts.includeOffset=true] - include the offset, such as 'Z' or '-04:00'
     * @param {boolean} [opts.includeOffsetSpace=true] - include the space between the time and the offset, such as '05:15:16.345 -04:00'
     * @example DateTime.utc(2014, 7, 13).toSQL() //=> '2014-07-13 00:00:00.000 Z'
     * @example DateTime.local(2014, 7, 13).toSQL() //=> '2014-07-13 00:00:00.000 -04:00'
     * @example DateTime.local(2014, 7, 13).toSQL({ includeOffset: false }) //=> '2014-07-13 00:00:00.000'
     * @example DateTime.local(2014, 7, 13).toSQL({ includeZone: true }) //=> '2014-07-13 00:00:00.000 America/New_York'
     * @return {string}
     */
    toSQL(opts = {}) {
      if (!this.isValid) {
        return null;
      }

      return `${this.toSQLDate()} ${this.toSQLTime(opts)}`;
    }

    /**
     * Returns a string representation of this DateTime appropriate for debugging
     * @return {string}
     */
    toString() {
      return this.isValid ? this.toISO() : INVALID$1;
    }

    /**
     * Returns the epoch milliseconds of this DateTime. Alias of {@link DateTime#toMillis}
     * @return {number}
     */
    valueOf() {
      return this.toMillis();
    }

    /**
     * Returns the epoch milliseconds of this DateTime.
     * @return {number}
     */
    toMillis() {
      return this.isValid ? this.ts : NaN;
    }

    /**
     * Returns the epoch seconds of this DateTime.
     * @return {number}
     */
    toSeconds() {
      return this.isValid ? this.ts / 1000 : NaN;
    }

    /**
     * Returns the epoch seconds (as a whole number) of this DateTime.
     * @return {number}
     */
    toUnixInteger() {
      return this.isValid ? Math.floor(this.ts / 1000) : NaN;
    }

    /**
     * Returns an ISO 8601 representation of this DateTime appropriate for use in JSON.
     * @return {string}
     */
    toJSON() {
      return this.toISO();
    }

    /**
     * Returns a BSON serializable equivalent to this DateTime.
     * @return {Date}
     */
    toBSON() {
      return this.toJSDate();
    }

    /**
     * Returns a JavaScript object with this DateTime's year, month, day, and so on.
     * @param opts - options for generating the object
     * @param {boolean} [opts.includeConfig=false] - include configuration attributes in the output
     * @example DateTime.now().toObject() //=> { year: 2017, month: 4, day: 22, hour: 20, minute: 49, second: 42, millisecond: 268 }
     * @return {Object}
     */
    toObject(opts = {}) {
      if (!this.isValid) return {};

      const base = { ...this.c };

      if (opts.includeConfig) {
        base.outputCalendar = this.outputCalendar;
        base.numberingSystem = this.loc.numberingSystem;
        base.locale = this.loc.locale;
      }
      return base;
    }

    /**
     * Returns a JavaScript Date equivalent to this DateTime.
     * @return {Date}
     */
    toJSDate() {
      return new Date(this.isValid ? this.ts : NaN);
    }

    // COMPARE

    /**
     * Return the difference between two DateTimes as a Duration.
     * @param {DateTime} otherDateTime - the DateTime to compare this one to
     * @param {string|string[]} [unit=['milliseconds']] - the unit or array of units (such as 'hours' or 'days') to include in the duration.
     * @param {Object} opts - options that affect the creation of the Duration
     * @param {string} [opts.conversionAccuracy='casual'] - the conversion system to use
     * @example
     * var i1 = DateTime.fromISO('1982-05-25T09:45'),
     *     i2 = DateTime.fromISO('1983-10-14T10:30');
     * i2.diff(i1).toObject() //=> { milliseconds: 43807500000 }
     * i2.diff(i1, 'hours').toObject() //=> { hours: 12168.75 }
     * i2.diff(i1, ['months', 'days']).toObject() //=> { months: 16, days: 19.03125 }
     * i2.diff(i1, ['months', 'days', 'hours']).toObject() //=> { months: 16, days: 19, hours: 0.75 }
     * @return {Duration}
     */
    diff(otherDateTime, unit = "milliseconds", opts = {}) {
      if (!this.isValid || !otherDateTime.isValid) {
        return Duration.invalid("created by diffing an invalid DateTime");
      }

      const durOpts = { locale: this.locale, numberingSystem: this.numberingSystem, ...opts };

      const units = maybeArray(unit).map(Duration.normalizeUnit),
        otherIsLater = otherDateTime.valueOf() > this.valueOf(),
        earlier = otherIsLater ? this : otherDateTime,
        later = otherIsLater ? otherDateTime : this,
        diffed = diff(earlier, later, units, durOpts);

      return otherIsLater ? diffed.negate() : diffed;
    }

    /**
     * Return the difference between this DateTime and right now.
     * See {@link DateTime#diff}
     * @param {string|string[]} [unit=['milliseconds']] - the unit or units units (such as 'hours' or 'days') to include in the duration
     * @param {Object} opts - options that affect the creation of the Duration
     * @param {string} [opts.conversionAccuracy='casual'] - the conversion system to use
     * @return {Duration}
     */
    diffNow(unit = "milliseconds", opts = {}) {
      return this.diff(DateTime.now(), unit, opts);
    }

    /**
     * Return an Interval spanning between this DateTime and another DateTime
     * @param {DateTime} otherDateTime - the other end point of the Interval
     * @return {Interval}
     */
    until(otherDateTime) {
      return this.isValid ? Interval.fromDateTimes(this, otherDateTime) : this;
    }

    /**
     * Return whether this DateTime is in the same unit of time as another DateTime.
     * Higher-order units must also be identical for this function to return `true`.
     * Note that time zones are **ignored** in this comparison, which compares the **local** calendar time. Use {@link DateTime#setZone} to convert one of the dates if needed.
     * @param {DateTime} otherDateTime - the other DateTime
     * @param {string} unit - the unit of time to check sameness on
     * @example DateTime.now().hasSame(otherDT, 'day'); //~> true if otherDT is in the same current calendar day
     * @return {boolean}
     */
    hasSame(otherDateTime, unit) {
      if (!this.isValid) return false;

      const inputMs = otherDateTime.valueOf();
      const adjustedToZone = this.setZone(otherDateTime.zone, { keepLocalTime: true });
      return adjustedToZone.startOf(unit) <= inputMs && inputMs <= adjustedToZone.endOf(unit);
    }

    /**
     * Equality check
     * Two DateTimes are equal if and only if they represent the same millisecond, have the same zone and location, and are both valid.
     * To compare just the millisecond values, use `+dt1 === +dt2`.
     * @param {DateTime} other - the other DateTime
     * @return {boolean}
     */
    equals(other) {
      return (
        this.isValid &&
        other.isValid &&
        this.valueOf() === other.valueOf() &&
        this.zone.equals(other.zone) &&
        this.loc.equals(other.loc)
      );
    }

    /**
     * Returns a string representation of a this time relative to now, such as "in two days". Can only internationalize if your
     * platform supports Intl.RelativeTimeFormat. Rounds down by default.
     * @param {Object} options - options that affect the output
     * @param {DateTime} [options.base=DateTime.now()] - the DateTime to use as the basis to which this time is compared. Defaults to now.
     * @param {string} [options.style="long"] - the style of units, must be "long", "short", or "narrow"
     * @param {string|string[]} options.unit - use a specific unit or array of units; if omitted, or an array, the method will pick the best unit. Use an array or one of "years", "quarters", "months", "weeks", "days", "hours", "minutes", or "seconds"
     * @param {boolean} [options.round=true] - whether to round the numbers in the output.
     * @param {number} [options.padding=0] - padding in milliseconds. This allows you to round up the result if it fits inside the threshold. Don't use in combination with {round: false} because the decimal output will include the padding.
     * @param {string} options.locale - override the locale of this DateTime
     * @param {string} options.numberingSystem - override the numberingSystem of this DateTime. The Intl system may choose not to honor this
     * @example DateTime.now().plus({ days: 1 }).toRelative() //=> "in 1 day"
     * @example DateTime.now().setLocale("es").toRelative({ days: 1 }) //=> "dentro de 1 día"
     * @example DateTime.now().plus({ days: 1 }).toRelative({ locale: "fr" }) //=> "dans 23 heures"
     * @example DateTime.now().minus({ days: 2 }).toRelative() //=> "2 days ago"
     * @example DateTime.now().minus({ days: 2 }).toRelative({ unit: "hours" }) //=> "48 hours ago"
     * @example DateTime.now().minus({ hours: 36 }).toRelative({ round: false }) //=> "1.5 days ago"
     */
    toRelative(options = {}) {
      if (!this.isValid) return null;
      const base = options.base || DateTime.fromObject({}, { zone: this.zone }),
        padding = options.padding ? (this < base ? -options.padding : options.padding) : 0;
      let units = ["years", "months", "days", "hours", "minutes", "seconds"];
      let unit = options.unit;
      if (Array.isArray(options.unit)) {
        units = options.unit;
        unit = undefined;
      }
      return diffRelative(base, this.plus(padding), {
        ...options,
        numeric: "always",
        units,
        unit,
      });
    }

    /**
     * Returns a string representation of this date relative to today, such as "yesterday" or "next month".
     * Only internationalizes on platforms that supports Intl.RelativeTimeFormat.
     * @param {Object} options - options that affect the output
     * @param {DateTime} [options.base=DateTime.now()] - the DateTime to use as the basis to which this time is compared. Defaults to now.
     * @param {string} options.locale - override the locale of this DateTime
     * @param {string} options.unit - use a specific unit; if omitted, the method will pick the unit. Use one of "years", "quarters", "months", "weeks", or "days"
     * @param {string} options.numberingSystem - override the numberingSystem of this DateTime. The Intl system may choose not to honor this
     * @example DateTime.now().plus({ days: 1 }).toRelativeCalendar() //=> "tomorrow"
     * @example DateTime.now().setLocale("es").plus({ days: 1 }).toRelative() //=> ""mañana"
     * @example DateTime.now().plus({ days: 1 }).toRelativeCalendar({ locale: "fr" }) //=> "demain"
     * @example DateTime.now().minus({ days: 2 }).toRelativeCalendar() //=> "2 days ago"
     */
    toRelativeCalendar(options = {}) {
      if (!this.isValid) return null;

      return diffRelative(options.base || DateTime.fromObject({}, { zone: this.zone }), this, {
        ...options,
        numeric: "auto",
        units: ["years", "months", "days"],
        calendary: true,
      });
    }

    /**
     * Return the min of several date times
     * @param {...DateTime} dateTimes - the DateTimes from which to choose the minimum
     * @return {DateTime} the min DateTime, or undefined if called with no argument
     */
    static min(...dateTimes) {
      if (!dateTimes.every(DateTime.isDateTime)) {
        throw new InvalidArgumentError("min requires all arguments be DateTimes");
      }
      return bestBy(dateTimes, (i) => i.valueOf(), Math.min);
    }

    /**
     * Return the max of several date times
     * @param {...DateTime} dateTimes - the DateTimes from which to choose the maximum
     * @return {DateTime} the max DateTime, or undefined if called with no argument
     */
    static max(...dateTimes) {
      if (!dateTimes.every(DateTime.isDateTime)) {
        throw new InvalidArgumentError("max requires all arguments be DateTimes");
      }
      return bestBy(dateTimes, (i) => i.valueOf(), Math.max);
    }

    // MISC

    /**
     * Explain how a string would be parsed by fromFormat()
     * @param {string} text - the string to parse
     * @param {string} fmt - the format the string is expected to be in (see description)
     * @param {Object} options - options taken by fromFormat()
     * @return {Object}
     */
    static fromFormatExplain(text, fmt, options = {}) {
      const { locale = null, numberingSystem = null } = options,
        localeToUse = Locale.fromOpts({
          locale,
          numberingSystem,
          defaultToEN: true,
        });
      return explainFromTokens(localeToUse, text, fmt);
    }

    /**
     * @deprecated use fromFormatExplain instead
     */
    static fromStringExplain(text, fmt, options = {}) {
      return DateTime.fromFormatExplain(text, fmt, options);
    }

    // FORMAT PRESETS

    /**
     * {@link DateTime#toLocaleString} format like 10/14/1983
     * @type {Object}
     */
    static get DATE_SHORT() {
      return DATE_SHORT;
    }

    /**
     * {@link DateTime#toLocaleString} format like 'Oct 14, 1983'
     * @type {Object}
     */
    static get DATE_MED() {
      return DATE_MED;
    }

    /**
     * {@link DateTime#toLocaleString} format like 'Fri, Oct 14, 1983'
     * @type {Object}
     */
    static get DATE_MED_WITH_WEEKDAY() {
      return DATE_MED_WITH_WEEKDAY;
    }

    /**
     * {@link DateTime#toLocaleString} format like 'October 14, 1983'
     * @type {Object}
     */
    static get DATE_FULL() {
      return DATE_FULL;
    }

    /**
     * {@link DateTime#toLocaleString} format like 'Tuesday, October 14, 1983'
     * @type {Object}
     */
    static get DATE_HUGE() {
      return DATE_HUGE;
    }

    /**
     * {@link DateTime#toLocaleString} format like '09:30 AM'. Only 12-hour if the locale is.
     * @type {Object}
     */
    static get TIME_SIMPLE() {
      return TIME_SIMPLE;
    }

    /**
     * {@link DateTime#toLocaleString} format like '09:30:23 AM'. Only 12-hour if the locale is.
     * @type {Object}
     */
    static get TIME_WITH_SECONDS() {
      return TIME_WITH_SECONDS;
    }

    /**
     * {@link DateTime#toLocaleString} format like '09:30:23 AM EDT'. Only 12-hour if the locale is.
     * @type {Object}
     */
    static get TIME_WITH_SHORT_OFFSET() {
      return TIME_WITH_SHORT_OFFSET;
    }

    /**
     * {@link DateTime#toLocaleString} format like '09:30:23 AM Eastern Daylight Time'. Only 12-hour if the locale is.
     * @type {Object}
     */
    static get TIME_WITH_LONG_OFFSET() {
      return TIME_WITH_LONG_OFFSET;
    }

    /**
     * {@link DateTime#toLocaleString} format like '09:30', always 24-hour.
     * @type {Object}
     */
    static get TIME_24_SIMPLE() {
      return TIME_24_SIMPLE;
    }

    /**
     * {@link DateTime#toLocaleString} format like '09:30:23', always 24-hour.
     * @type {Object}
     */
    static get TIME_24_WITH_SECONDS() {
      return TIME_24_WITH_SECONDS;
    }

    /**
     * {@link DateTime#toLocaleString} format like '09:30:23 EDT', always 24-hour.
     * @type {Object}
     */
    static get TIME_24_WITH_SHORT_OFFSET() {
      return TIME_24_WITH_SHORT_OFFSET;
    }

    /**
     * {@link DateTime#toLocaleString} format like '09:30:23 Eastern Daylight Time', always 24-hour.
     * @type {Object}
     */
    static get TIME_24_WITH_LONG_OFFSET() {
      return TIME_24_WITH_LONG_OFFSET;
    }

    /**
     * {@link DateTime#toLocaleString} format like '10/14/1983, 9:30 AM'. Only 12-hour if the locale is.
     * @type {Object}
     */
    static get DATETIME_SHORT() {
      return DATETIME_SHORT;
    }

    /**
     * {@link DateTime#toLocaleString} format like '10/14/1983, 9:30:33 AM'. Only 12-hour if the locale is.
     * @type {Object}
     */
    static get DATETIME_SHORT_WITH_SECONDS() {
      return DATETIME_SHORT_WITH_SECONDS;
    }

    /**
     * {@link DateTime#toLocaleString} format like 'Oct 14, 1983, 9:30 AM'. Only 12-hour if the locale is.
     * @type {Object}
     */
    static get DATETIME_MED() {
      return DATETIME_MED;
    }

    /**
     * {@link DateTime#toLocaleString} format like 'Oct 14, 1983, 9:30:33 AM'. Only 12-hour if the locale is.
     * @type {Object}
     */
    static get DATETIME_MED_WITH_SECONDS() {
      return DATETIME_MED_WITH_SECONDS;
    }

    /**
     * {@link DateTime#toLocaleString} format like 'Fri, 14 Oct 1983, 9:30 AM'. Only 12-hour if the locale is.
     * @type {Object}
     */
    static get DATETIME_MED_WITH_WEEKDAY() {
      return DATETIME_MED_WITH_WEEKDAY;
    }

    /**
     * {@link DateTime#toLocaleString} format like 'October 14, 1983, 9:30 AM EDT'. Only 12-hour if the locale is.
     * @type {Object}
     */
    static get DATETIME_FULL() {
      return DATETIME_FULL;
    }

    /**
     * {@link DateTime#toLocaleString} format like 'October 14, 1983, 9:30:33 AM EDT'. Only 12-hour if the locale is.
     * @type {Object}
     */
    static get DATETIME_FULL_WITH_SECONDS() {
      return DATETIME_FULL_WITH_SECONDS;
    }

    /**
     * {@link DateTime#toLocaleString} format like 'Friday, October 14, 1983, 9:30 AM Eastern Daylight Time'. Only 12-hour if the locale is.
     * @type {Object}
     */
    static get DATETIME_HUGE() {
      return DATETIME_HUGE;
    }

    /**
     * {@link DateTime#toLocaleString} format like 'Friday, October 14, 1983, 9:30:33 AM Eastern Daylight Time'. Only 12-hour if the locale is.
     * @type {Object}
     */
    static get DATETIME_HUGE_WITH_SECONDS() {
      return DATETIME_HUGE_WITH_SECONDS;
    }
  }

  /**
   * @private
   */
  function friendlyDateTime(dateTimeish) {
    if (DateTime.isDateTime(dateTimeish)) {
      return dateTimeish;
    } else if (dateTimeish && dateTimeish.valueOf && isNumber$2(dateTimeish.valueOf())) {
      return DateTime.fromJSDate(dateTimeish);
    } else if (dateTimeish && typeof dateTimeish === "object") {
      return DateTime.fromObject(dateTimeish);
    } else {
      throw new InvalidArgumentError(
        `Unknown datetime argument: ${dateTimeish}, of type ${typeof dateTimeish}`
      );
    }
  }

  // FIXME profile adding a per-Tree TreeNode cache, validating it by
  // parent pointer
  /// The default maximum length of a `TreeBuffer` node.
  const DefaultBufferLength = 1024;
  let nextPropID = 0;
  let Range$3 = class Range {
      constructor(from, to) {
          this.from = from;
          this.to = to;
      }
  };
  /// Each [node type](#common.NodeType) or [individual tree](#common.Tree)
  /// can have metadata associated with it in props. Instances of this
  /// class represent prop names.
  class NodeProp {
      /// Create a new node prop type.
      constructor(config = {}) {
          this.id = nextPropID++;
          this.perNode = !!config.perNode;
          this.deserialize = config.deserialize || (() => {
              throw new Error("This node type doesn't define a deserialize function");
          });
      }
      /// This is meant to be used with
      /// [`NodeSet.extend`](#common.NodeSet.extend) or
      /// [`LRParser.configure`](#lr.ParserConfig.props) to compute
      /// prop values for each node type in the set. Takes a [match
      /// object](#common.NodeType^match) or function that returns undefined
      /// if the node type doesn't get this prop, and the prop's value if
      /// it does.
      add(match) {
          if (this.perNode)
              throw new RangeError("Can't add per-node props to node types");
          if (typeof match != "function")
              match = NodeType.match(match);
          return (type) => {
              let result = match(type);
              return result === undefined ? null : [this, result];
          };
      }
  }
  /// Prop that is used to describe matching delimiters. For opening
  /// delimiters, this holds an array of node names (written as a
  /// space-separated string when declaring this prop in a grammar)
  /// for the node types of closing delimiters that match it.
  NodeProp.closedBy = new NodeProp({ deserialize: str => str.split(" ") });
  /// The inverse of [`closedBy`](#common.NodeProp^closedBy). This is
  /// attached to closing delimiters, holding an array of node names
  /// of types of matching opening delimiters.
  NodeProp.openedBy = new NodeProp({ deserialize: str => str.split(" ") });
  /// Used to assign node types to groups (for example, all node
  /// types that represent an expression could be tagged with an
  /// `"Expression"` group).
  NodeProp.group = new NodeProp({ deserialize: str => str.split(" ") });
  /// The hash of the [context](#lr.ContextTracker.constructor)
  /// that the node was parsed in, if any. Used to limit reuse of
  /// contextual nodes.
  NodeProp.contextHash = new NodeProp({ perNode: true });
  /// The distance beyond the end of the node that the tokenizer
  /// looked ahead for any of the tokens inside the node. (The LR
  /// parser only stores this when it is larger than 25, for
  /// efficiency reasons.)
  NodeProp.lookAhead = new NodeProp({ perNode: true });
  /// This per-node prop is used to replace a given node, or part of a
  /// node, with another tree. This is useful to include trees from
  /// different languages in mixed-language parsers.
  NodeProp.mounted = new NodeProp({ perNode: true });
  const noProps = Object.create(null);
  /// Each node in a syntax tree has a node type associated with it.
  class NodeType {
      /// @internal
      constructor(
      /// The name of the node type. Not necessarily unique, but if the
      /// grammar was written properly, different node types with the
      /// same name within a node set should play the same semantic
      /// role.
      name, 
      /// @internal
      props, 
      /// The id of this node in its set. Corresponds to the term ids
      /// used in the parser.
      id, 
      /// @internal
      flags = 0) {
          this.name = name;
          this.props = props;
          this.id = id;
          this.flags = flags;
      }
      /// Define a node type.
      static define(spec) {
          let props = spec.props && spec.props.length ? Object.create(null) : noProps;
          let flags = (spec.top ? 1 /* Top */ : 0) | (spec.skipped ? 2 /* Skipped */ : 0) |
              (spec.error ? 4 /* Error */ : 0) | (spec.name == null ? 8 /* Anonymous */ : 0);
          let type = new NodeType(spec.name || "", props, spec.id, flags);
          if (spec.props)
              for (let src of spec.props) {
                  if (!Array.isArray(src))
                      src = src(type);
                  if (src) {
                      if (src[0].perNode)
                          throw new RangeError("Can't store a per-node prop on a node type");
                      props[src[0].id] = src[1];
                  }
              }
          return type;
      }
      /// Retrieves a node prop for this type. Will return `undefined` if
      /// the prop isn't present on this node.
      prop(prop) { return this.props[prop.id]; }
      /// True when this is the top node of a grammar.
      get isTop() { return (this.flags & 1 /* Top */) > 0; }
      /// True when this node is produced by a skip rule.
      get isSkipped() { return (this.flags & 2 /* Skipped */) > 0; }
      /// Indicates whether this is an error node.
      get isError() { return (this.flags & 4 /* Error */) > 0; }
      /// When true, this node type doesn't correspond to a user-declared
      /// named node, for example because it is used to cache repetition.
      get isAnonymous() { return (this.flags & 8 /* Anonymous */) > 0; }
      /// Returns true when this node's name or one of its
      /// [groups](#common.NodeProp^group) matches the given string.
      is(name) {
          if (typeof name == 'string') {
              if (this.name == name)
                  return true;
              let group = this.prop(NodeProp.group);
              return group ? group.indexOf(name) > -1 : false;
          }
          return this.id == name;
      }
      /// Create a function from node types to arbitrary values by
      /// specifying an object whose property names are node or
      /// [group](#common.NodeProp^group) names. Often useful with
      /// [`NodeProp.add`](#common.NodeProp.add). You can put multiple
      /// names, separated by spaces, in a single property name to map
      /// multiple node names to a single value.
      static match(map) {
          let direct = Object.create(null);
          for (let prop in map)
              for (let name of prop.split(" "))
                  direct[name] = map[prop];
          return (node) => {
              for (let groups = node.prop(NodeProp.group), i = -1; i < (groups ? groups.length : 0); i++) {
                  let found = direct[i < 0 ? node.name : groups[i]];
                  if (found)
                      return found;
              }
          };
      }
  }
  /// An empty dummy node type to use when no actual type is available.
  NodeType.none = new NodeType("", Object.create(null), 0, 8 /* Anonymous */);
  /// A node set holds a collection of node types. It is used to
  /// compactly represent trees by storing their type ids, rather than a
  /// full pointer to the type object, in a numeric array. Each parser
  /// [has](#lr.LRParser.nodeSet) a node set, and [tree
  /// buffers](#common.TreeBuffer) can only store collections of nodes
  /// from the same set. A set can have a maximum of 2**16 (65536) node
  /// types in it, so that the ids fit into 16-bit typed array slots.
  class NodeSet {
      /// Create a set with the given types. The `id` property of each
      /// type should correspond to its position within the array.
      constructor(
      /// The node types in this set, by id.
      types) {
          this.types = types;
          for (let i = 0; i < types.length; i++)
              if (types[i].id != i)
                  throw new RangeError("Node type ids should correspond to array positions when creating a node set");
      }
      /// Create a copy of this set with some node properties added. The
      /// arguments to this method can be created with
      /// [`NodeProp.add`](#common.NodeProp.add).
      extend(...props) {
          let newTypes = [];
          for (let type of this.types) {
              let newProps = null;
              for (let source of props) {
                  let add = source(type);
                  if (add) {
                      if (!newProps)
                          newProps = Object.assign({}, type.props);
                      newProps[add[0].id] = add[1];
                  }
              }
              newTypes.push(newProps ? new NodeType(type.name, newProps, type.id, type.flags) : type);
          }
          return new NodeSet(newTypes);
      }
  }
  const CachedNode = new WeakMap(), CachedInnerNode = new WeakMap();
  /// Options that control iteration. Can be combined with the `|`
  /// operator to enable multiple ones.
  var IterMode;
  (function (IterMode) {
      /// When enabled, iteration will only visit [`Tree`](#common.Tree)
      /// objects, not nodes packed into
      /// [`TreeBuffer`](#common.TreeBuffer)s.
      IterMode[IterMode["ExcludeBuffers"] = 1] = "ExcludeBuffers";
      /// Enable this to make iteration include anonymous nodes (such as
      /// the nodes that wrap repeated grammar constructs into a balanced
      /// tree).
      IterMode[IterMode["IncludeAnonymous"] = 2] = "IncludeAnonymous";
      /// By default, regular [mounted](#common.NodeProp^mounted) nodes
      /// replace their base node in iteration. Enable this to ignore them
      /// instead.
      IterMode[IterMode["IgnoreMounts"] = 4] = "IgnoreMounts";
      /// This option only applies in
      /// [`enter`](#common.SyntaxNode.enter)-style methods. It tells the
      /// library to not enter mounted overlays if one covers the given
      /// position.
      IterMode[IterMode["IgnoreOverlays"] = 8] = "IgnoreOverlays";
  })(IterMode || (IterMode = {}));
  /// A piece of syntax tree. There are two ways to approach these
  /// trees: the way they are actually stored in memory, and the
  /// convenient way.
  ///
  /// Syntax trees are stored as a tree of `Tree` and `TreeBuffer`
  /// objects. By packing detail information into `TreeBuffer` leaf
  /// nodes, the representation is made a lot more memory-efficient.
  ///
  /// However, when you want to actually work with tree nodes, this
  /// representation is very awkward, so most client code will want to
  /// use the [`TreeCursor`](#common.TreeCursor) or
  /// [`SyntaxNode`](#common.SyntaxNode) interface instead, which provides
  /// a view on some part of this data structure, and can be used to
  /// move around to adjacent nodes.
  class Tree {
      /// Construct a new tree. See also [`Tree.build`](#common.Tree^build).
      constructor(
      /// The type of the top node.
      type, 
      /// This node's child nodes.
      children, 
      /// The positions (offsets relative to the start of this tree) of
      /// the children.
      positions, 
      /// The total length of this tree
      length, 
      /// Per-node [node props](#common.NodeProp) to associate with this node.
      props) {
          this.type = type;
          this.children = children;
          this.positions = positions;
          this.length = length;
          /// @internal
          this.props = null;
          if (props && props.length) {
              this.props = Object.create(null);
              for (let [prop, value] of props)
                  this.props[typeof prop == "number" ? prop : prop.id] = value;
          }
      }
      /// @internal
      toString() {
          let mounted = this.prop(NodeProp.mounted);
          if (mounted && !mounted.overlay)
              return mounted.tree.toString();
          let children = "";
          for (let ch of this.children) {
              let str = ch.toString();
              if (str) {
                  if (children)
                      children += ",";
                  children += str;
              }
          }
          return !this.type.name ? children :
              (/\W/.test(this.type.name) && !this.type.isError ? JSON.stringify(this.type.name) : this.type.name) +
                  (children.length ? "(" + children + ")" : "");
      }
      /// Get a [tree cursor](#common.TreeCursor) positioned at the top of
      /// the tree. Mode can be used to [control](#common.IterMode) which
      /// nodes the cursor visits.
      cursor(mode = 0) {
          return new TreeCursor(this.topNode, mode);
      }
      /// Get a [tree cursor](#common.TreeCursor) pointing into this tree
      /// at the given position and side (see
      /// [`moveTo`](#common.TreeCursor.moveTo).
      cursorAt(pos, side = 0, mode = 0) {
          let scope = CachedNode.get(this) || this.topNode;
          let cursor = new TreeCursor(scope);
          cursor.moveTo(pos, side);
          CachedNode.set(this, cursor._tree);
          return cursor;
      }
      /// Get a [syntax node](#common.SyntaxNode) object for the top of the
      /// tree.
      get topNode() {
          return new TreeNode(this, 0, 0, null);
      }
      /// Get the [syntax node](#common.SyntaxNode) at the given position.
      /// If `side` is -1, this will move into nodes that end at the
      /// position. If 1, it'll move into nodes that start at the
      /// position. With 0, it'll only enter nodes that cover the position
      /// from both sides.
      ///
      /// Note that this will not enter
      /// [overlays](#common.MountedTree.overlay), and you often want
      /// [`resolveInner`](#common.Tree.resolveInner) instead.
      resolve(pos, side = 0) {
          let node = resolveNode(CachedNode.get(this) || this.topNode, pos, side, false);
          CachedNode.set(this, node);
          return node;
      }
      /// Like [`resolve`](#common.Tree.resolve), but will enter
      /// [overlaid](#common.MountedTree.overlay) nodes, producing a syntax node
      /// pointing into the innermost overlaid tree at the given position
      /// (with parent links going through all parent structure, including
      /// the host trees).
      resolveInner(pos, side = 0) {
          let node = resolveNode(CachedInnerNode.get(this) || this.topNode, pos, side, true);
          CachedInnerNode.set(this, node);
          return node;
      }
      /// Iterate over the tree and its children, calling `enter` for any
      /// node that touches the `from`/`to` region (if given) before
      /// running over such a node's children, and `leave` (if given) when
      /// leaving the node. When `enter` returns `false`, that node will
      /// not have its children iterated over (or `leave` called).
      iterate(spec) {
          let { enter, leave, from = 0, to = this.length } = spec;
          for (let c = this.cursor((spec.mode || 0) | IterMode.IncludeAnonymous);;) {
              let entered = false;
              if (c.from <= to && c.to >= from && (c.type.isAnonymous || enter(c) !== false)) {
                  if (c.firstChild())
                      continue;
                  entered = true;
              }
              for (;;) {
                  if (entered && leave && !c.type.isAnonymous)
                      leave(c);
                  if (c.nextSibling())
                      break;
                  if (!c.parent())
                      return;
                  entered = true;
              }
          }
      }
      /// Get the value of the given [node prop](#common.NodeProp) for this
      /// node. Works with both per-node and per-type props.
      prop(prop) {
          return !prop.perNode ? this.type.prop(prop) : this.props ? this.props[prop.id] : undefined;
      }
      /// Returns the node's [per-node props](#common.NodeProp.perNode) in a
      /// format that can be passed to the [`Tree`](#common.Tree)
      /// constructor.
      get propValues() {
          let result = [];
          if (this.props)
              for (let id in this.props)
                  result.push([+id, this.props[id]]);
          return result;
      }
      /// Balance the direct children of this tree, producing a copy of
      /// which may have children grouped into subtrees with type
      /// [`NodeType.none`](#common.NodeType^none).
      balance(config = {}) {
          return this.children.length <= 8 /* BranchFactor */ ? this :
              balanceRange(NodeType.none, this.children, this.positions, 0, this.children.length, 0, this.length, (children, positions, length) => new Tree(this.type, children, positions, length, this.propValues), config.makeTree || ((children, positions, length) => new Tree(NodeType.none, children, positions, length)));
      }
      /// Build a tree from a postfix-ordered buffer of node information,
      /// or a cursor over such a buffer.
      static build(data) { return buildTree(data); }
  }
  /// The empty tree
  Tree.empty = new Tree(NodeType.none, [], [], 0);
  class FlatBufferCursor {
      constructor(buffer, index) {
          this.buffer = buffer;
          this.index = index;
      }
      get id() { return this.buffer[this.index - 4]; }
      get start() { return this.buffer[this.index - 3]; }
      get end() { return this.buffer[this.index - 2]; }
      get size() { return this.buffer[this.index - 1]; }
      get pos() { return this.index; }
      next() { this.index -= 4; }
      fork() { return new FlatBufferCursor(this.buffer, this.index); }
  }
  /// Tree buffers contain (type, start, end, endIndex) quads for each
  /// node. In such a buffer, nodes are stored in prefix order (parents
  /// before children, with the endIndex of the parent indicating which
  /// children belong to it).
  class TreeBuffer {
      /// Create a tree buffer.
      constructor(
      /// The buffer's content.
      buffer, 
      /// The total length of the group of nodes in the buffer.
      length, 
      /// The node set used in this buffer.
      set) {
          this.buffer = buffer;
          this.length = length;
          this.set = set;
      }
      /// @internal
      get type() { return NodeType.none; }
      /// @internal
      toString() {
          let result = [];
          for (let index = 0; index < this.buffer.length;) {
              result.push(this.childString(index));
              index = this.buffer[index + 3];
          }
          return result.join(",");
      }
      /// @internal
      childString(index) {
          let id = this.buffer[index], endIndex = this.buffer[index + 3];
          let type = this.set.types[id], result = type.name;
          if (/\W/.test(result) && !type.isError)
              result = JSON.stringify(result);
          index += 4;
          if (endIndex == index)
              return result;
          let children = [];
          while (index < endIndex) {
              children.push(this.childString(index));
              index = this.buffer[index + 3];
          }
          return result + "(" + children.join(",") + ")";
      }
      /// @internal
      findChild(startIndex, endIndex, dir, pos, side) {
          let { buffer } = this, pick = -1;
          for (let i = startIndex; i != endIndex; i = buffer[i + 3]) {
              if (checkSide(side, pos, buffer[i + 1], buffer[i + 2])) {
                  pick = i;
                  if (dir > 0)
                      break;
              }
          }
          return pick;
      }
      /// @internal
      slice(startI, endI, from, to) {
          let b = this.buffer;
          let copy = new Uint16Array(endI - startI);
          for (let i = startI, j = 0; i < endI;) {
              copy[j++] = b[i++];
              copy[j++] = b[i++] - from;
              copy[j++] = b[i++] - from;
              copy[j++] = b[i++] - startI;
          }
          return new TreeBuffer(copy, to - from, this.set);
      }
  }
  function checkSide(side, pos, from, to) {
      switch (side) {
          case -2 /* Before */: return from < pos;
          case -1 /* AtOrBefore */: return to >= pos && from < pos;
          case 0 /* Around */: return from < pos && to > pos;
          case 1 /* AtOrAfter */: return from <= pos && to > pos;
          case 2 /* After */: return to > pos;
          case 4 /* DontCare */: return true;
      }
  }
  function enterUnfinishedNodesBefore(node, pos) {
      let scan = node.childBefore(pos);
      while (scan) {
          let last = scan.lastChild;
          if (!last || last.to != scan.to)
              break;
          if (last.type.isError && last.from == last.to) {
              node = scan;
              scan = last.prevSibling;
          }
          else {
              scan = last;
          }
      }
      return node;
  }
  function resolveNode(node, pos, side, overlays) {
      var _a;
      // Move up to a node that actually holds the position, if possible
      while (node.from == node.to ||
          (side < 1 ? node.from >= pos : node.from > pos) ||
          (side > -1 ? node.to <= pos : node.to < pos)) {
          let parent = !overlays && node instanceof TreeNode && node.index < 0 ? null : node.parent;
          if (!parent)
              return node;
          node = parent;
      }
      let mode = overlays ? 0 : IterMode.IgnoreOverlays;
      // Must go up out of overlays when those do not overlap with pos
      if (overlays)
          for (let scan = node, parent = scan.parent; parent; scan = parent, parent = scan.parent) {
              if (scan instanceof TreeNode && scan.index < 0 && ((_a = parent.enter(pos, side, mode)) === null || _a === void 0 ? void 0 : _a.from) != scan.from)
                  node = parent;
          }
      for (;;) {
          let inner = node.enter(pos, side, mode);
          if (!inner)
              return node;
          node = inner;
      }
  }
  class TreeNode {
      constructor(_tree, from, 
      // Index in parent node, set to -1 if the node is not a direct child of _parent.node (overlay)
      index, _parent) {
          this._tree = _tree;
          this.from = from;
          this.index = index;
          this._parent = _parent;
      }
      get type() { return this._tree.type; }
      get name() { return this._tree.type.name; }
      get to() { return this.from + this._tree.length; }
      nextChild(i, dir, pos, side, mode = 0) {
          for (let parent = this;;) {
              for (let { children, positions } = parent._tree, e = dir > 0 ? children.length : -1; i != e; i += dir) {
                  let next = children[i], start = positions[i] + parent.from;
                  if (!checkSide(side, pos, start, start + next.length))
                      continue;
                  if (next instanceof TreeBuffer) {
                      if (mode & IterMode.ExcludeBuffers)
                          continue;
                      let index = next.findChild(0, next.buffer.length, dir, pos - start, side);
                      if (index > -1)
                          return new BufferNode(new BufferContext(parent, next, i, start), null, index);
                  }
                  else if ((mode & IterMode.IncludeAnonymous) || (!next.type.isAnonymous || hasChild(next))) {
                      let mounted;
                      if (!(mode & IterMode.IgnoreMounts) &&
                          next.props && (mounted = next.prop(NodeProp.mounted)) && !mounted.overlay)
                          return new TreeNode(mounted.tree, start, i, parent);
                      let inner = new TreeNode(next, start, i, parent);
                      return (mode & IterMode.IncludeAnonymous) || !inner.type.isAnonymous ? inner
                          : inner.nextChild(dir < 0 ? next.children.length - 1 : 0, dir, pos, side);
                  }
              }
              if ((mode & IterMode.IncludeAnonymous) || !parent.type.isAnonymous)
                  return null;
              if (parent.index >= 0)
                  i = parent.index + dir;
              else
                  i = dir < 0 ? -1 : parent._parent._tree.children.length;
              parent = parent._parent;
              if (!parent)
                  return null;
          }
      }
      get firstChild() { return this.nextChild(0, 1, 0, 4 /* DontCare */); }
      get lastChild() { return this.nextChild(this._tree.children.length - 1, -1, 0, 4 /* DontCare */); }
      childAfter(pos) { return this.nextChild(0, 1, pos, 2 /* After */); }
      childBefore(pos) { return this.nextChild(this._tree.children.length - 1, -1, pos, -2 /* Before */); }
      enter(pos, side, mode = 0) {
          let mounted;
          if (!(mode & IterMode.IgnoreOverlays) && (mounted = this._tree.prop(NodeProp.mounted)) && mounted.overlay) {
              let rPos = pos - this.from;
              for (let { from, to } of mounted.overlay) {
                  if ((side > 0 ? from <= rPos : from < rPos) &&
                      (side < 0 ? to >= rPos : to > rPos))
                      return new TreeNode(mounted.tree, mounted.overlay[0].from + this.from, -1, this);
              }
          }
          return this.nextChild(0, 1, pos, side, mode);
      }
      nextSignificantParent() {
          let val = this;
          while (val.type.isAnonymous && val._parent)
              val = val._parent;
          return val;
      }
      get parent() {
          return this._parent ? this._parent.nextSignificantParent() : null;
      }
      get nextSibling() {
          return this._parent && this.index >= 0 ? this._parent.nextChild(this.index + 1, 1, 0, 4 /* DontCare */) : null;
      }
      get prevSibling() {
          return this._parent && this.index >= 0 ? this._parent.nextChild(this.index - 1, -1, 0, 4 /* DontCare */) : null;
      }
      cursor(mode = 0) { return new TreeCursor(this, mode); }
      get tree() { return this._tree; }
      toTree() { return this._tree; }
      resolve(pos, side = 0) {
          return resolveNode(this, pos, side, false);
      }
      resolveInner(pos, side = 0) {
          return resolveNode(this, pos, side, true);
      }
      enterUnfinishedNodesBefore(pos) { return enterUnfinishedNodesBefore(this, pos); }
      getChild(type, before = null, after = null) {
          let r = getChildren(this, type, before, after);
          return r.length ? r[0] : null;
      }
      getChildren(type, before = null, after = null) {
          return getChildren(this, type, before, after);
      }
      /// @internal
      toString() { return this._tree.toString(); }
      get node() { return this; }
      matchContext(context) { return matchNodeContext(this, context); }
  }
  function getChildren(node, type, before, after) {
      let cur = node.cursor(), result = [];
      if (!cur.firstChild())
          return result;
      if (before != null)
          while (!cur.type.is(before))
              if (!cur.nextSibling())
                  return result;
      for (;;) {
          if (after != null && cur.type.is(after))
              return result;
          if (cur.type.is(type))
              result.push(cur.node);
          if (!cur.nextSibling())
              return after == null ? result : [];
      }
  }
  function matchNodeContext(node, context, i = context.length - 1) {
      for (let p = node.parent; i >= 0; p = p.parent) {
          if (!p)
              return false;
          if (!p.type.isAnonymous) {
              if (context[i] && context[i] != p.name)
                  return false;
              i--;
          }
      }
      return true;
  }
  class BufferContext {
      constructor(parent, buffer, index, start) {
          this.parent = parent;
          this.buffer = buffer;
          this.index = index;
          this.start = start;
      }
  }
  class BufferNode {
      constructor(context, _parent, index) {
          this.context = context;
          this._parent = _parent;
          this.index = index;
          this.type = context.buffer.set.types[context.buffer.buffer[index]];
      }
      get name() { return this.type.name; }
      get from() { return this.context.start + this.context.buffer.buffer[this.index + 1]; }
      get to() { return this.context.start + this.context.buffer.buffer[this.index + 2]; }
      child(dir, pos, side) {
          let { buffer } = this.context;
          let index = buffer.findChild(this.index + 4, buffer.buffer[this.index + 3], dir, pos - this.context.start, side);
          return index < 0 ? null : new BufferNode(this.context, this, index);
      }
      get firstChild() { return this.child(1, 0, 4 /* DontCare */); }
      get lastChild() { return this.child(-1, 0, 4 /* DontCare */); }
      childAfter(pos) { return this.child(1, pos, 2 /* After */); }
      childBefore(pos) { return this.child(-1, pos, -2 /* Before */); }
      enter(pos, side, mode = 0) {
          if (mode & IterMode.ExcludeBuffers)
              return null;
          let { buffer } = this.context;
          let index = buffer.findChild(this.index + 4, buffer.buffer[this.index + 3], side > 0 ? 1 : -1, pos - this.context.start, side);
          return index < 0 ? null : new BufferNode(this.context, this, index);
      }
      get parent() {
          return this._parent || this.context.parent.nextSignificantParent();
      }
      externalSibling(dir) {
          return this._parent ? null : this.context.parent.nextChild(this.context.index + dir, dir, 0, 4 /* DontCare */);
      }
      get nextSibling() {
          let { buffer } = this.context;
          let after = buffer.buffer[this.index + 3];
          if (after < (this._parent ? buffer.buffer[this._parent.index + 3] : buffer.buffer.length))
              return new BufferNode(this.context, this._parent, after);
          return this.externalSibling(1);
      }
      get prevSibling() {
          let { buffer } = this.context;
          let parentStart = this._parent ? this._parent.index + 4 : 0;
          if (this.index == parentStart)
              return this.externalSibling(-1);
          return new BufferNode(this.context, this._parent, buffer.findChild(parentStart, this.index, -1, 0, 4 /* DontCare */));
      }
      cursor(mode = 0) { return new TreeCursor(this, mode); }
      get tree() { return null; }
      toTree() {
          let children = [], positions = [];
          let { buffer } = this.context;
          let startI = this.index + 4, endI = buffer.buffer[this.index + 3];
          if (endI > startI) {
              let from = buffer.buffer[this.index + 1], to = buffer.buffer[this.index + 2];
              children.push(buffer.slice(startI, endI, from, to));
              positions.push(0);
          }
          return new Tree(this.type, children, positions, this.to - this.from);
      }
      resolve(pos, side = 0) {
          return resolveNode(this, pos, side, false);
      }
      resolveInner(pos, side = 0) {
          return resolveNode(this, pos, side, true);
      }
      enterUnfinishedNodesBefore(pos) { return enterUnfinishedNodesBefore(this, pos); }
      /// @internal
      toString() { return this.context.buffer.childString(this.index); }
      getChild(type, before = null, after = null) {
          let r = getChildren(this, type, before, after);
          return r.length ? r[0] : null;
      }
      getChildren(type, before = null, after = null) {
          return getChildren(this, type, before, after);
      }
      get node() { return this; }
      matchContext(context) { return matchNodeContext(this, context); }
  }
  /// A tree cursor object focuses on a given node in a syntax tree, and
  /// allows you to move to adjacent nodes.
  class TreeCursor {
      /// @internal
      constructor(node, 
      /// @internal
      mode = 0) {
          this.mode = mode;
          /// @internal
          this.buffer = null;
          this.stack = [];
          /// @internal
          this.index = 0;
          this.bufferNode = null;
          if (node instanceof TreeNode) {
              this.yieldNode(node);
          }
          else {
              this._tree = node.context.parent;
              this.buffer = node.context;
              for (let n = node._parent; n; n = n._parent)
                  this.stack.unshift(n.index);
              this.bufferNode = node;
              this.yieldBuf(node.index);
          }
      }
      /// Shorthand for `.type.name`.
      get name() { return this.type.name; }
      yieldNode(node) {
          if (!node)
              return false;
          this._tree = node;
          this.type = node.type;
          this.from = node.from;
          this.to = node.to;
          return true;
      }
      yieldBuf(index, type) {
          this.index = index;
          let { start, buffer } = this.buffer;
          this.type = type || buffer.set.types[buffer.buffer[index]];
          this.from = start + buffer.buffer[index + 1];
          this.to = start + buffer.buffer[index + 2];
          return true;
      }
      yield(node) {
          if (!node)
              return false;
          if (node instanceof TreeNode) {
              this.buffer = null;
              return this.yieldNode(node);
          }
          this.buffer = node.context;
          return this.yieldBuf(node.index, node.type);
      }
      /// @internal
      toString() {
          return this.buffer ? this.buffer.buffer.childString(this.index) : this._tree.toString();
      }
      /// @internal
      enterChild(dir, pos, side) {
          if (!this.buffer)
              return this.yield(this._tree.nextChild(dir < 0 ? this._tree._tree.children.length - 1 : 0, dir, pos, side, this.mode));
          let { buffer } = this.buffer;
          let index = buffer.findChild(this.index + 4, buffer.buffer[this.index + 3], dir, pos - this.buffer.start, side);
          if (index < 0)
              return false;
          this.stack.push(this.index);
          return this.yieldBuf(index);
      }
      /// Move the cursor to this node's first child. When this returns
      /// false, the node has no child, and the cursor has not been moved.
      firstChild() { return this.enterChild(1, 0, 4 /* DontCare */); }
      /// Move the cursor to this node's last child.
      lastChild() { return this.enterChild(-1, 0, 4 /* DontCare */); }
      /// Move the cursor to the first child that ends after `pos`.
      childAfter(pos) { return this.enterChild(1, pos, 2 /* After */); }
      /// Move to the last child that starts before `pos`.
      childBefore(pos) { return this.enterChild(-1, pos, -2 /* Before */); }
      /// Move the cursor to the child around `pos`. If side is -1 the
      /// child may end at that position, when 1 it may start there. This
      /// will also enter [overlaid](#common.MountedTree.overlay)
      /// [mounted](#common.NodeProp^mounted) trees unless `overlays` is
      /// set to false.
      enter(pos, side, mode = this.mode) {
          if (!this.buffer)
              return this.yield(this._tree.enter(pos, side, mode));
          return mode & IterMode.ExcludeBuffers ? false : this.enterChild(1, pos, side);
      }
      /// Move to the node's parent node, if this isn't the top node.
      parent() {
          if (!this.buffer)
              return this.yieldNode((this.mode & IterMode.IncludeAnonymous) ? this._tree._parent : this._tree.parent);
          if (this.stack.length)
              return this.yieldBuf(this.stack.pop());
          let parent = (this.mode & IterMode.IncludeAnonymous) ? this.buffer.parent : this.buffer.parent.nextSignificantParent();
          this.buffer = null;
          return this.yieldNode(parent);
      }
      /// @internal
      sibling(dir) {
          if (!this.buffer)
              return !this._tree._parent ? false
                  : this.yield(this._tree.index < 0 ? null
                      : this._tree._parent.nextChild(this._tree.index + dir, dir, 0, 4 /* DontCare */, this.mode));
          let { buffer } = this.buffer, d = this.stack.length - 1;
          if (dir < 0) {
              let parentStart = d < 0 ? 0 : this.stack[d] + 4;
              if (this.index != parentStart)
                  return this.yieldBuf(buffer.findChild(parentStart, this.index, -1, 0, 4 /* DontCare */));
          }
          else {
              let after = buffer.buffer[this.index + 3];
              if (after < (d < 0 ? buffer.buffer.length : buffer.buffer[this.stack[d] + 3]))
                  return this.yieldBuf(after);
          }
          return d < 0 ? this.yield(this.buffer.parent.nextChild(this.buffer.index + dir, dir, 0, 4 /* DontCare */, this.mode)) : false;
      }
      /// Move to this node's next sibling, if any.
      nextSibling() { return this.sibling(1); }
      /// Move to this node's previous sibling, if any.
      prevSibling() { return this.sibling(-1); }
      atLastNode(dir) {
          let index, parent, { buffer } = this;
          if (buffer) {
              if (dir > 0) {
                  if (this.index < buffer.buffer.buffer.length)
                      return false;
              }
              else {
                  for (let i = 0; i < this.index; i++)
                      if (buffer.buffer.buffer[i + 3] < this.index)
                          return false;
              }
              ({ index, parent } = buffer);
          }
          else {
              ({ index, _parent: parent } = this._tree);
          }
          for (; parent; { index, _parent: parent } = parent) {
              if (index > -1)
                  for (let i = index + dir, e = dir < 0 ? -1 : parent._tree.children.length; i != e; i += dir) {
                      let child = parent._tree.children[i];
                      if ((this.mode & IterMode.IncludeAnonymous) ||
                          child instanceof TreeBuffer ||
                          !child.type.isAnonymous ||
                          hasChild(child))
                          return false;
                  }
          }
          return true;
      }
      move(dir, enter) {
          if (enter && this.enterChild(dir, 0, 4 /* DontCare */))
              return true;
          for (;;) {
              if (this.sibling(dir))
                  return true;
              if (this.atLastNode(dir) || !this.parent())
                  return false;
          }
      }
      /// Move to the next node in a
      /// [pre-order](https://en.wikipedia.org/wiki/Tree_traversal#Pre-order_(NLR))
      /// traversal, going from a node to its first child or, if the
      /// current node is empty or `enter` is false, its next sibling or
      /// the next sibling of the first parent node that has one.
      next(enter = true) { return this.move(1, enter); }
      /// Move to the next node in a last-to-first pre-order traveral. A
      /// node is followed by its last child or, if it has none, its
      /// previous sibling or the previous sibling of the first parent
      /// node that has one.
      prev(enter = true) { return this.move(-1, enter); }
      /// Move the cursor to the innermost node that covers `pos`. If
      /// `side` is -1, it will enter nodes that end at `pos`. If it is 1,
      /// it will enter nodes that start at `pos`.
      moveTo(pos, side = 0) {
          // Move up to a node that actually holds the position, if possible
          while (this.from == this.to ||
              (side < 1 ? this.from >= pos : this.from > pos) ||
              (side > -1 ? this.to <= pos : this.to < pos))
              if (!this.parent())
                  break;
          // Then scan down into child nodes as far as possible
          while (this.enterChild(1, pos, side)) { }
          return this;
      }
      /// Get a [syntax node](#common.SyntaxNode) at the cursor's current
      /// position.
      get node() {
          if (!this.buffer)
              return this._tree;
          let cache = this.bufferNode, result = null, depth = 0;
          if (cache && cache.context == this.buffer) {
              scan: for (let index = this.index, d = this.stack.length; d >= 0;) {
                  for (let c = cache; c; c = c._parent)
                      if (c.index == index) {
                          if (index == this.index)
                              return c;
                          result = c;
                          depth = d + 1;
                          break scan;
                      }
                  index = this.stack[--d];
              }
          }
          for (let i = depth; i < this.stack.length; i++)
              result = new BufferNode(this.buffer, result, this.stack[i]);
          return this.bufferNode = new BufferNode(this.buffer, result, this.index);
      }
      /// Get the [tree](#common.Tree) that represents the current node, if
      /// any. Will return null when the node is in a [tree
      /// buffer](#common.TreeBuffer).
      get tree() {
          return this.buffer ? null : this._tree._tree;
      }
      /// Iterate over the current node and all its descendants, calling
      /// `enter` when entering a node and `leave`, if given, when leaving
      /// one. When `enter` returns `false`, any children of that node are
      /// skipped, and `leave` isn't called for it.
      iterate(enter, leave) {
          for (let depth = 0;;) {
              let mustLeave = false;
              if (this.type.isAnonymous || enter(this) !== false) {
                  if (this.firstChild()) {
                      depth++;
                      continue;
                  }
                  if (!this.type.isAnonymous)
                      mustLeave = true;
              }
              for (;;) {
                  if (mustLeave && leave)
                      leave(this);
                  mustLeave = this.type.isAnonymous;
                  if (this.nextSibling())
                      break;
                  if (!depth)
                      return;
                  this.parent();
                  depth--;
                  mustLeave = true;
              }
          }
      }
      /// Test whether the current node matches a given context—a sequence
      /// of direct parent node names. Empty strings in the context array
      /// are treated as wildcards.
      matchContext(context) {
          if (!this.buffer)
              return matchNodeContext(this.node, context);
          let { buffer } = this.buffer, { types } = buffer.set;
          for (let i = context.length - 1, d = this.stack.length - 1; i >= 0; d--) {
              if (d < 0)
                  return matchNodeContext(this.node, context, i);
              let type = types[buffer.buffer[this.stack[d]]];
              if (!type.isAnonymous) {
                  if (context[i] && context[i] != type.name)
                      return false;
                  i--;
              }
          }
          return true;
      }
  }
  function hasChild(tree) {
      return tree.children.some(ch => ch instanceof TreeBuffer || !ch.type.isAnonymous || hasChild(ch));
  }
  function buildTree(data) {
      var _a;
      let { buffer, nodeSet, maxBufferLength = DefaultBufferLength, reused = [], minRepeatType = nodeSet.types.length } = data;
      let cursor = Array.isArray(buffer) ? new FlatBufferCursor(buffer, buffer.length) : buffer;
      let types = nodeSet.types;
      let contextHash = 0, lookAhead = 0;
      function takeNode(parentStart, minPos, children, positions, inRepeat) {
          let { id, start, end, size } = cursor;
          let lookAheadAtStart = lookAhead;
          while (size < 0) {
              cursor.next();
              if (size == -1 /* Reuse */) {
                  let node = reused[id];
                  children.push(node);
                  positions.push(start - parentStart);
                  return;
              }
              else if (size == -3 /* ContextChange */) { // Context change
                  contextHash = id;
                  return;
              }
              else if (size == -4 /* LookAhead */) {
                  lookAhead = id;
                  return;
              }
              else {
                  throw new RangeError(`Unrecognized record size: ${size}`);
              }
          }
          let type = types[id], node, buffer;
          let startPos = start - parentStart;
          if (end - start <= maxBufferLength && (buffer = findBufferSize(cursor.pos - minPos, inRepeat))) {
              // Small enough for a buffer, and no reused nodes inside
              let data = new Uint16Array(buffer.size - buffer.skip);
              let endPos = cursor.pos - buffer.size, index = data.length;
              while (cursor.pos > endPos)
                  index = copyToBuffer(buffer.start, data, index);
              node = new TreeBuffer(data, end - buffer.start, nodeSet);
              startPos = buffer.start - parentStart;
          }
          else { // Make it a node
              let endPos = cursor.pos - size;
              cursor.next();
              let localChildren = [], localPositions = [];
              let localInRepeat = id >= minRepeatType ? id : -1;
              let lastGroup = 0, lastEnd = end;
              while (cursor.pos > endPos) {
                  if (localInRepeat >= 0 && cursor.id == localInRepeat && cursor.size >= 0) {
                      if (cursor.end <= lastEnd - maxBufferLength) {
                          makeRepeatLeaf(localChildren, localPositions, start, lastGroup, cursor.end, lastEnd, localInRepeat, lookAheadAtStart);
                          lastGroup = localChildren.length;
                          lastEnd = cursor.end;
                      }
                      cursor.next();
                  }
                  else {
                      takeNode(start, endPos, localChildren, localPositions, localInRepeat);
                  }
              }
              if (localInRepeat >= 0 && lastGroup > 0 && lastGroup < localChildren.length)
                  makeRepeatLeaf(localChildren, localPositions, start, lastGroup, start, lastEnd, localInRepeat, lookAheadAtStart);
              localChildren.reverse();
              localPositions.reverse();
              if (localInRepeat > -1 && lastGroup > 0) {
                  let make = makeBalanced(type);
                  node = balanceRange(type, localChildren, localPositions, 0, localChildren.length, 0, end - start, make, make);
              }
              else {
                  node = makeTree(type, localChildren, localPositions, end - start, lookAheadAtStart - end);
              }
          }
          children.push(node);
          positions.push(startPos);
      }
      function makeBalanced(type) {
          return (children, positions, length) => {
              let lookAhead = 0, lastI = children.length - 1, last, lookAheadProp;
              if (lastI >= 0 && (last = children[lastI]) instanceof Tree) {
                  if (!lastI && last.type == type && last.length == length)
                      return last;
                  if (lookAheadProp = last.prop(NodeProp.lookAhead))
                      lookAhead = positions[lastI] + last.length + lookAheadProp;
              }
              return makeTree(type, children, positions, length, lookAhead);
          };
      }
      function makeRepeatLeaf(children, positions, base, i, from, to, type, lookAhead) {
          let localChildren = [], localPositions = [];
          while (children.length > i) {
              localChildren.push(children.pop());
              localPositions.push(positions.pop() + base - from);
          }
          children.push(makeTree(nodeSet.types[type], localChildren, localPositions, to - from, lookAhead - to));
          positions.push(from - base);
      }
      function makeTree(type, children, positions, length, lookAhead = 0, props) {
          if (contextHash) {
              let pair = [NodeProp.contextHash, contextHash];
              props = props ? [pair].concat(props) : [pair];
          }
          if (lookAhead > 25) {
              let pair = [NodeProp.lookAhead, lookAhead];
              props = props ? [pair].concat(props) : [pair];
          }
          return new Tree(type, children, positions, length, props);
      }
      function findBufferSize(maxSize, inRepeat) {
          // Scan through the buffer to find previous siblings that fit
          // together in a TreeBuffer, and don't contain any reused nodes
          // (which can't be stored in a buffer).
          // If `inRepeat` is > -1, ignore node boundaries of that type for
          // nesting, but make sure the end falls either at the start
          // (`maxSize`) or before such a node.
          let fork = cursor.fork();
          let size = 0, start = 0, skip = 0, minStart = fork.end - maxBufferLength;
          let result = { size: 0, start: 0, skip: 0 };
          scan: for (let minPos = fork.pos - maxSize; fork.pos > minPos;) {
              let nodeSize = fork.size;
              // Pretend nested repeat nodes of the same type don't exist
              if (fork.id == inRepeat && nodeSize >= 0) {
                  // Except that we store the current state as a valid return
                  // value.
                  result.size = size;
                  result.start = start;
                  result.skip = skip;
                  skip += 4;
                  size += 4;
                  fork.next();
                  continue;
              }
              let startPos = fork.pos - nodeSize;
              if (nodeSize < 0 || startPos < minPos || fork.start < minStart)
                  break;
              let localSkipped = fork.id >= minRepeatType ? 4 : 0;
              let nodeStart = fork.start;
              fork.next();
              while (fork.pos > startPos) {
                  if (fork.size < 0) {
                      if (fork.size == -3 /* ContextChange */)
                          localSkipped += 4;
                      else
                          break scan;
                  }
                  else if (fork.id >= minRepeatType) {
                      localSkipped += 4;
                  }
                  fork.next();
              }
              start = nodeStart;
              size += nodeSize;
              skip += localSkipped;
          }
          if (inRepeat < 0 || size == maxSize) {
              result.size = size;
              result.start = start;
              result.skip = skip;
          }
          return result.size > 4 ? result : undefined;
      }
      function copyToBuffer(bufferStart, buffer, index) {
          let { id, start, end, size } = cursor;
          cursor.next();
          if (size >= 0 && id < minRepeatType) {
              let startIndex = index;
              if (size > 4) {
                  let endPos = cursor.pos - (size - 4);
                  while (cursor.pos > endPos)
                      index = copyToBuffer(bufferStart, buffer, index);
              }
              buffer[--index] = startIndex;
              buffer[--index] = end - bufferStart;
              buffer[--index] = start - bufferStart;
              buffer[--index] = id;
          }
          else if (size == -3 /* ContextChange */) {
              contextHash = id;
          }
          else if (size == -4 /* LookAhead */) {
              lookAhead = id;
          }
          return index;
      }
      let children = [], positions = [];
      while (cursor.pos > 0)
          takeNode(data.start || 0, data.bufferStart || 0, children, positions, -1);
      let length = (_a = data.length) !== null && _a !== void 0 ? _a : (children.length ? positions[0] + children[0].length : 0);
      return new Tree(types[data.topID], children.reverse(), positions.reverse(), length);
  }
  const nodeSizeCache = new WeakMap;
  function nodeSize(balanceType, node) {
      if (!balanceType.isAnonymous || node instanceof TreeBuffer || node.type != balanceType)
          return 1;
      let size = nodeSizeCache.get(node);
      if (size == null) {
          size = 1;
          for (let child of node.children) {
              if (child.type != balanceType || !(child instanceof Tree)) {
                  size = 1;
                  break;
              }
              size += nodeSize(balanceType, child);
          }
          nodeSizeCache.set(node, size);
      }
      return size;
  }
  function balanceRange(
  // The type the balanced tree's inner nodes.
  balanceType, 
  // The direct children and their positions
  children, positions, 
  // The index range in children/positions to use
  from, to, 
  // The start position of the nodes, relative to their parent.
  start, 
  // Length of the outer node
  length, 
  // Function to build the top node of the balanced tree
  mkTop, 
  // Function to build internal nodes for the balanced tree
  mkTree) {
      let total = 0;
      for (let i = from; i < to; i++)
          total += nodeSize(balanceType, children[i]);
      let maxChild = Math.ceil((total * 1.5) / 8 /* BranchFactor */);
      let localChildren = [], localPositions = [];
      function divide(children, positions, from, to, offset) {
          for (let i = from; i < to;) {
              let groupFrom = i, groupStart = positions[i], groupSize = nodeSize(balanceType, children[i]);
              i++;
              for (; i < to; i++) {
                  let nextSize = nodeSize(balanceType, children[i]);
                  if (groupSize + nextSize >= maxChild)
                      break;
                  groupSize += nextSize;
              }
              if (i == groupFrom + 1) {
                  if (groupSize > maxChild) {
                      let only = children[groupFrom]; // Only trees can have a size > 1
                      divide(only.children, only.positions, 0, only.children.length, positions[groupFrom] + offset);
                      continue;
                  }
                  localChildren.push(children[groupFrom]);
              }
              else {
                  let length = positions[i - 1] + children[i - 1].length - groupStart;
                  localChildren.push(balanceRange(balanceType, children, positions, groupFrom, i, groupStart, length, null, mkTree));
              }
              localPositions.push(groupStart + offset - start);
          }
      }
      divide(children, positions, from, to, 0);
      return (mkTop || mkTree)(localChildren, localPositions, length);
  }

  /// Tree fragments are used during [incremental
  /// parsing](#common.Parser.startParse) to track parts of old trees
  /// that can be reused in a new parse. An array of fragments is used
  /// to track regions of an old tree whose nodes might be reused in new
  /// parses. Use the static
  /// [`applyChanges`](#common.TreeFragment^applyChanges) method to
  /// update fragments for document changes.
  class TreeFragment {
      /// Construct a tree fragment. You'll usually want to use
      /// [`addTree`](#common.TreeFragment^addTree) and
      /// [`applyChanges`](#common.TreeFragment^applyChanges) instead of
      /// calling this directly.
      constructor(
      /// The start of the unchanged range pointed to by this fragment.
      /// This refers to an offset in the _updated_ document (as opposed
      /// to the original tree).
      from, 
      /// The end of the unchanged range.
      to, 
      /// The tree that this fragment is based on.
      tree, 
      /// The offset between the fragment's tree and the document that
      /// this fragment can be used against. Add this when going from
      /// document to tree positions, subtract it to go from tree to
      /// document positions.
      offset, openStart = false, openEnd = false) {
          this.from = from;
          this.to = to;
          this.tree = tree;
          this.offset = offset;
          this.open = (openStart ? 1 /* Start */ : 0) | (openEnd ? 2 /* End */ : 0);
      }
      /// Whether the start of the fragment represents the start of a
      /// parse, or the end of a change. (In the second case, it may not
      /// be safe to reuse some nodes at the start, depending on the
      /// parsing algorithm.)
      get openStart() { return (this.open & 1 /* Start */) > 0; }
      /// Whether the end of the fragment represents the end of a
      /// full-document parse, or the start of a change.
      get openEnd() { return (this.open & 2 /* End */) > 0; }
      /// Create a set of fragments from a freshly parsed tree, or update
      /// an existing set of fragments by replacing the ones that overlap
      /// with a tree with content from the new tree. When `partial` is
      /// true, the parse is treated as incomplete, and the resulting
      /// fragment has [`openEnd`](#common.TreeFragment.openEnd) set to
      /// true.
      static addTree(tree, fragments = [], partial = false) {
          let result = [new TreeFragment(0, tree.length, tree, 0, false, partial)];
          for (let f of fragments)
              if (f.to > tree.length)
                  result.push(f);
          return result;
      }
      /// Apply a set of edits to an array of fragments, removing or
      /// splitting fragments as necessary to remove edited ranges, and
      /// adjusting offsets for fragments that moved.
      static applyChanges(fragments, changes, minGap = 128) {
          if (!changes.length)
              return fragments;
          let result = [];
          let fI = 1, nextF = fragments.length ? fragments[0] : null;
          for (let cI = 0, pos = 0, off = 0;; cI++) {
              let nextC = cI < changes.length ? changes[cI] : null;
              let nextPos = nextC ? nextC.fromA : 1e9;
              if (nextPos - pos >= minGap)
                  while (nextF && nextF.from < nextPos) {
                      let cut = nextF;
                      if (pos >= cut.from || nextPos <= cut.to || off) {
                          let fFrom = Math.max(cut.from, pos) - off, fTo = Math.min(cut.to, nextPos) - off;
                          cut = fFrom >= fTo ? null : new TreeFragment(fFrom, fTo, cut.tree, cut.offset + off, cI > 0, !!nextC);
                      }
                      if (cut)
                          result.push(cut);
                      if (nextF.to > nextPos)
                          break;
                      nextF = fI < fragments.length ? fragments[fI++] : null;
                  }
              if (!nextC)
                  break;
              pos = nextC.toA;
              off = nextC.toA - nextC.toB;
          }
          return result;
      }
  }
  /// A superclass that parsers should extend.
  class Parser {
      /// Start a parse, returning a [partial parse](#common.PartialParse)
      /// object. [`fragments`](#common.TreeFragment) can be passed in to
      /// make the parse incremental.
      ///
      /// By default, the entire input is parsed. You can pass `ranges`,
      /// which should be a sorted array of non-empty, non-overlapping
      /// ranges, to parse only those ranges. The tree returned in that
      /// case will start at `ranges[0].from`.
      startParse(input, fragments, ranges) {
          if (typeof input == "string")
              input = new StringInput(input);
          ranges = !ranges ? [new Range$3(0, input.length)] : ranges.length ? ranges.map(r => new Range$3(r.from, r.to)) : [new Range$3(0, 0)];
          return this.createParse(input, fragments || [], ranges);
      }
      /// Run a full parse, returning the resulting tree.
      parse(input, fragments, ranges) {
          let parse = this.startParse(input, fragments, ranges);
          for (;;) {
              let done = parse.advance();
              if (done)
                  return done;
          }
      }
  }
  class StringInput {
      constructor(string) {
          this.string = string;
      }
      get length() { return this.string.length; }
      chunk(from) { return this.string.slice(from); }
      get lineChunks() { return false; }
      read(from, to) { return this.string.slice(from, to); }
  }
  new NodeProp({ perNode: true });

  /// A parse stack. These are used internally by the parser to track
  /// parsing progress. They also provide some properties and methods
  /// that external code such as a tokenizer can use to get information
  /// about the parse state.
  class Stack {
      /// @internal
      constructor(
      /// The parse that this stack is part of @internal
      p, 
      /// Holds state, input pos, buffer index triplets for all but the
      /// top state @internal
      stack, 
      /// The current parse state @internal
      state, 
      // The position at which the next reduce should take place. This
      // can be less than `this.pos` when skipped expressions have been
      // added to the stack (which should be moved outside of the next
      // reduction)
      /// @internal
      reducePos, 
      /// The input position up to which this stack has parsed.
      pos, 
      /// The dynamic score of the stack, including dynamic precedence
      /// and error-recovery penalties
      /// @internal
      score, 
      // The output buffer. Holds (type, start, end, size) quads
      // representing nodes created by the parser, where `size` is
      // amount of buffer array entries covered by this node.
      /// @internal
      buffer, 
      // The base offset of the buffer. When stacks are split, the split
      // instance shared the buffer history with its parent up to
      // `bufferBase`, which is the absolute offset (including the
      // offset of previous splits) into the buffer at which this stack
      // starts writing.
      /// @internal
      bufferBase, 
      /// @internal
      curContext, 
      /// @internal
      lookAhead = 0, 
      // A parent stack from which this was split off, if any. This is
      // set up so that it always points to a stack that has some
      // additional buffer content, never to a stack with an equal
      // `bufferBase`.
      /// @internal
      parent) {
          this.p = p;
          this.stack = stack;
          this.state = state;
          this.reducePos = reducePos;
          this.pos = pos;
          this.score = score;
          this.buffer = buffer;
          this.bufferBase = bufferBase;
          this.curContext = curContext;
          this.lookAhead = lookAhead;
          this.parent = parent;
      }
      /// @internal
      toString() {
          return `[${this.stack.filter((_, i) => i % 3 == 0).concat(this.state)}]@${this.pos}${this.score ? "!" + this.score : ""}`;
      }
      // Start an empty stack
      /// @internal
      static start(p, state, pos = 0) {
          let cx = p.parser.context;
          return new Stack(p, [], state, pos, pos, 0, [], 0, cx ? new StackContext(cx, cx.start) : null, 0, null);
      }
      /// The stack's current [context](#lr.ContextTracker) value, if
      /// any. Its type will depend on the context tracker's type
      /// parameter, or it will be `null` if there is no context
      /// tracker.
      get context() { return this.curContext ? this.curContext.context : null; }
      // Push a state onto the stack, tracking its start position as well
      // as the buffer base at that point.
      /// @internal
      pushState(state, start) {
          this.stack.push(this.state, start, this.bufferBase + this.buffer.length);
          this.state = state;
      }
      // Apply a reduce action
      /// @internal
      reduce(action) {
          let depth = action >> 19 /* ReduceDepthShift */, type = action & 65535 /* ValueMask */;
          let { parser } = this.p;
          let dPrec = parser.dynamicPrecedence(type);
          if (dPrec)
              this.score += dPrec;
          if (depth == 0) {
              this.pushState(parser.getGoto(this.state, type, true), this.reducePos);
              // Zero-depth reductions are a special case—they add stuff to
              // the stack without popping anything off.
              if (type < parser.minRepeatTerm)
                  this.storeNode(type, this.reducePos, this.reducePos, 4, true);
              this.reduceContext(type, this.reducePos);
              return;
          }
          // Find the base index into `this.stack`, content after which will
          // be dropped. Note that with `StayFlag` reductions we need to
          // consume two extra frames (the dummy parent node for the skipped
          // expression and the state that we'll be staying in, which should
          // be moved to `this.state`).
          let base = this.stack.length - ((depth - 1) * 3) - (action & 262144 /* StayFlag */ ? 6 : 0);
          let start = this.stack[base - 2];
          let bufferBase = this.stack[base - 1], count = this.bufferBase + this.buffer.length - bufferBase;
          // Store normal terms or `R -> R R` repeat reductions
          if (type < parser.minRepeatTerm || (action & 131072 /* RepeatFlag */)) {
              let pos = parser.stateFlag(this.state, 1 /* Skipped */) ? this.pos : this.reducePos;
              this.storeNode(type, start, pos, count + 4, true);
          }
          if (action & 262144 /* StayFlag */) {
              this.state = this.stack[base];
          }
          else {
              let baseStateID = this.stack[base - 3];
              this.state = parser.getGoto(baseStateID, type, true);
          }
          while (this.stack.length > base)
              this.stack.pop();
          this.reduceContext(type, start);
      }
      // Shift a value into the buffer
      /// @internal
      storeNode(term, start, end, size = 4, isReduce = false) {
          if (term == 0 /* Err */ &&
              (!this.stack.length || this.stack[this.stack.length - 1] < this.buffer.length + this.bufferBase)) {
              // Try to omit/merge adjacent error nodes
              let cur = this, top = this.buffer.length;
              if (top == 0 && cur.parent) {
                  top = cur.bufferBase - cur.parent.bufferBase;
                  cur = cur.parent;
              }
              if (top > 0 && cur.buffer[top - 4] == 0 /* Err */ && cur.buffer[top - 1] > -1) {
                  if (start == end)
                      return;
                  if (cur.buffer[top - 2] >= start) {
                      cur.buffer[top - 2] = end;
                      return;
                  }
              }
          }
          if (!isReduce || this.pos == end) { // Simple case, just append
              this.buffer.push(term, start, end, size);
          }
          else { // There may be skipped nodes that have to be moved forward
              let index = this.buffer.length;
              if (index > 0 && this.buffer[index - 4] != 0 /* Err */)
                  while (index > 0 && this.buffer[index - 2] > end) {
                      // Move this record forward
                      this.buffer[index] = this.buffer[index - 4];
                      this.buffer[index + 1] = this.buffer[index - 3];
                      this.buffer[index + 2] = this.buffer[index - 2];
                      this.buffer[index + 3] = this.buffer[index - 1];
                      index -= 4;
                      if (size > 4)
                          size -= 4;
                  }
              this.buffer[index] = term;
              this.buffer[index + 1] = start;
              this.buffer[index + 2] = end;
              this.buffer[index + 3] = size;
          }
      }
      // Apply a shift action
      /// @internal
      shift(action, next, nextEnd) {
          let start = this.pos;
          if (action & 131072 /* GotoFlag */) {
              this.pushState(action & 65535 /* ValueMask */, this.pos);
          }
          else if ((action & 262144 /* StayFlag */) == 0) { // Regular shift
              let nextState = action, { parser } = this.p;
              if (nextEnd > this.pos || next <= parser.maxNode) {
                  this.pos = nextEnd;
                  if (!parser.stateFlag(nextState, 1 /* Skipped */))
                      this.reducePos = nextEnd;
              }
              this.pushState(nextState, start);
              this.shiftContext(next, start);
              if (next <= parser.maxNode)
                  this.buffer.push(next, start, nextEnd, 4);
          }
          else { // Shift-and-stay, which means this is a skipped token
              this.pos = nextEnd;
              this.shiftContext(next, start);
              if (next <= this.p.parser.maxNode)
                  this.buffer.push(next, start, nextEnd, 4);
          }
      }
      // Apply an action
      /// @internal
      apply(action, next, nextEnd) {
          if (action & 65536 /* ReduceFlag */)
              this.reduce(action);
          else
              this.shift(action, next, nextEnd);
      }
      // Add a prebuilt (reused) node into the buffer.
      /// @internal
      useNode(value, next) {
          let index = this.p.reused.length - 1;
          if (index < 0 || this.p.reused[index] != value) {
              this.p.reused.push(value);
              index++;
          }
          let start = this.pos;
          this.reducePos = this.pos = start + value.length;
          this.pushState(next, start);
          this.buffer.push(index, start, this.reducePos, -1 /* size == -1 means this is a reused value */);
          if (this.curContext)
              this.updateContext(this.curContext.tracker.reuse(this.curContext.context, value, this, this.p.stream.reset(this.pos - value.length)));
      }
      // Split the stack. Due to the buffer sharing and the fact
      // that `this.stack` tends to stay quite shallow, this isn't very
      // expensive.
      /// @internal
      split() {
          let parent = this;
          let off = parent.buffer.length;
          // Because the top of the buffer (after this.pos) may be mutated
          // to reorder reductions and skipped tokens, and shared buffers
          // should be immutable, this copies any outstanding skipped tokens
          // to the new buffer, and puts the base pointer before them.
          while (off > 0 && parent.buffer[off - 2] > parent.reducePos)
              off -= 4;
          let buffer = parent.buffer.slice(off), base = parent.bufferBase + off;
          // Make sure parent points to an actual parent with content, if there is such a parent.
          while (parent && base == parent.bufferBase)
              parent = parent.parent;
          return new Stack(this.p, this.stack.slice(), this.state, this.reducePos, this.pos, this.score, buffer, base, this.curContext, this.lookAhead, parent);
      }
      // Try to recover from an error by 'deleting' (ignoring) one token.
      /// @internal
      recoverByDelete(next, nextEnd) {
          let isNode = next <= this.p.parser.maxNode;
          if (isNode)
              this.storeNode(next, this.pos, nextEnd, 4);
          this.storeNode(0 /* Err */, this.pos, nextEnd, isNode ? 8 : 4);
          this.pos = this.reducePos = nextEnd;
          this.score -= 190 /* Delete */;
      }
      /// Check if the given term would be able to be shifted (optionally
      /// after some reductions) on this stack. This can be useful for
      /// external tokenizers that want to make sure they only provide a
      /// given token when it applies.
      canShift(term) {
          for (let sim = new SimulatedStack(this);;) {
              let action = this.p.parser.stateSlot(sim.state, 4 /* DefaultReduce */) || this.p.parser.hasAction(sim.state, term);
              if (action == 0)
                  return false;
              if ((action & 65536 /* ReduceFlag */) == 0)
                  return true;
              sim.reduce(action);
          }
      }
      // Apply up to Recover.MaxNext recovery actions that conceptually
      // inserts some missing token or rule.
      /// @internal
      recoverByInsert(next) {
          if (this.stack.length >= 300 /* MaxInsertStackDepth */)
              return [];
          let nextStates = this.p.parser.nextStates(this.state);
          if (nextStates.length > 4 /* MaxNext */ << 1 || this.stack.length >= 120 /* DampenInsertStackDepth */) {
              let best = [];
              for (let i = 0, s; i < nextStates.length; i += 2) {
                  if ((s = nextStates[i + 1]) != this.state && this.p.parser.hasAction(s, next))
                      best.push(nextStates[i], s);
              }
              if (this.stack.length < 120 /* DampenInsertStackDepth */)
                  for (let i = 0; best.length < 4 /* MaxNext */ << 1 && i < nextStates.length; i += 2) {
                      let s = nextStates[i + 1];
                      if (!best.some((v, i) => (i & 1) && v == s))
                          best.push(nextStates[i], s);
                  }
              nextStates = best;
          }
          let result = [];
          for (let i = 0; i < nextStates.length && result.length < 4 /* MaxNext */; i += 2) {
              let s = nextStates[i + 1];
              if (s == this.state)
                  continue;
              let stack = this.split();
              stack.pushState(s, this.pos);
              stack.storeNode(0 /* Err */, stack.pos, stack.pos, 4, true);
              stack.shiftContext(nextStates[i], this.pos);
              stack.score -= 200 /* Insert */;
              result.push(stack);
          }
          return result;
      }
      // Force a reduce, if possible. Return false if that can't
      // be done.
      /// @internal
      forceReduce() {
          let reduce = this.p.parser.stateSlot(this.state, 5 /* ForcedReduce */);
          if ((reduce & 65536 /* ReduceFlag */) == 0)
              return false;
          let { parser } = this.p;
          if (!parser.validAction(this.state, reduce)) {
              let depth = reduce >> 19 /* ReduceDepthShift */, term = reduce & 65535 /* ValueMask */;
              let target = this.stack.length - depth * 3;
              if (target < 0 || parser.getGoto(this.stack[target], term, false) < 0)
                  return false;
              this.storeNode(0 /* Err */, this.reducePos, this.reducePos, 4, true);
              this.score -= 100 /* Reduce */;
          }
          this.reducePos = this.pos;
          this.reduce(reduce);
          return true;
      }
      /// @internal
      forceAll() {
          while (!this.p.parser.stateFlag(this.state, 2 /* Accepting */)) {
              if (!this.forceReduce()) {
                  this.storeNode(0 /* Err */, this.pos, this.pos, 4, true);
                  break;
              }
          }
          return this;
      }
      /// Check whether this state has no further actions (assumed to be a direct descendant of the
      /// top state, since any other states must be able to continue
      /// somehow). @internal
      get deadEnd() {
          if (this.stack.length != 3)
              return false;
          let { parser } = this.p;
          return parser.data[parser.stateSlot(this.state, 1 /* Actions */)] == 65535 /* End */ &&
              !parser.stateSlot(this.state, 4 /* DefaultReduce */);
      }
      /// Restart the stack (put it back in its start state). Only safe
      /// when this.stack.length == 3 (state is directly below the top
      /// state). @internal
      restart() {
          this.state = this.stack[0];
          this.stack.length = 0;
      }
      /// @internal
      sameState(other) {
          if (this.state != other.state || this.stack.length != other.stack.length)
              return false;
          for (let i = 0; i < this.stack.length; i += 3)
              if (this.stack[i] != other.stack[i])
                  return false;
          return true;
      }
      /// Get the parser used by this stack.
      get parser() { return this.p.parser; }
      /// Test whether a given dialect (by numeric ID, as exported from
      /// the terms file) is enabled.
      dialectEnabled(dialectID) { return this.p.parser.dialect.flags[dialectID]; }
      shiftContext(term, start) {
          if (this.curContext)
              this.updateContext(this.curContext.tracker.shift(this.curContext.context, term, this, this.p.stream.reset(start)));
      }
      reduceContext(term, start) {
          if (this.curContext)
              this.updateContext(this.curContext.tracker.reduce(this.curContext.context, term, this, this.p.stream.reset(start)));
      }
      /// @internal
      emitContext() {
          let last = this.buffer.length - 1;
          if (last < 0 || this.buffer[last] != -3)
              this.buffer.push(this.curContext.hash, this.reducePos, this.reducePos, -3);
      }
      /// @internal
      emitLookAhead() {
          let last = this.buffer.length - 1;
          if (last < 0 || this.buffer[last] != -4)
              this.buffer.push(this.lookAhead, this.reducePos, this.reducePos, -4);
      }
      updateContext(context) {
          if (context != this.curContext.context) {
              let newCx = new StackContext(this.curContext.tracker, context);
              if (newCx.hash != this.curContext.hash)
                  this.emitContext();
              this.curContext = newCx;
          }
      }
      /// @internal
      setLookAhead(lookAhead) {
          if (lookAhead > this.lookAhead) {
              this.emitLookAhead();
              this.lookAhead = lookAhead;
          }
      }
      /// @internal
      close() {
          if (this.curContext && this.curContext.tracker.strict)
              this.emitContext();
          if (this.lookAhead > 0)
              this.emitLookAhead();
      }
  }
  class StackContext {
      constructor(tracker, context) {
          this.tracker = tracker;
          this.context = context;
          this.hash = tracker.strict ? tracker.hash(context) : 0;
      }
  }
  var Recover;
  (function (Recover) {
      Recover[Recover["Insert"] = 200] = "Insert";
      Recover[Recover["Delete"] = 190] = "Delete";
      Recover[Recover["Reduce"] = 100] = "Reduce";
      Recover[Recover["MaxNext"] = 4] = "MaxNext";
      Recover[Recover["MaxInsertStackDepth"] = 300] = "MaxInsertStackDepth";
      Recover[Recover["DampenInsertStackDepth"] = 120] = "DampenInsertStackDepth";
  })(Recover || (Recover = {}));
  // Used to cheaply run some reductions to scan ahead without mutating
  // an entire stack
  class SimulatedStack {
      constructor(start) {
          this.start = start;
          this.state = start.state;
          this.stack = start.stack;
          this.base = this.stack.length;
      }
      reduce(action) {
          let term = action & 65535 /* ValueMask */, depth = action >> 19 /* ReduceDepthShift */;
          if (depth == 0) {
              if (this.stack == this.start.stack)
                  this.stack = this.stack.slice();
              this.stack.push(this.state, 0, 0);
              this.base += 3;
          }
          else {
              this.base -= (depth - 1) * 3;
          }
          let goto = this.start.p.parser.getGoto(this.stack[this.base - 3], term, true);
          this.state = goto;
      }
  }
  // This is given to `Tree.build` to build a buffer, and encapsulates
  // the parent-stack-walking necessary to read the nodes.
  class StackBufferCursor {
      constructor(stack, pos, index) {
          this.stack = stack;
          this.pos = pos;
          this.index = index;
          this.buffer = stack.buffer;
          if (this.index == 0)
              this.maybeNext();
      }
      static create(stack, pos = stack.bufferBase + stack.buffer.length) {
          return new StackBufferCursor(stack, pos, pos - stack.bufferBase);
      }
      maybeNext() {
          let next = this.stack.parent;
          if (next != null) {
              this.index = this.stack.bufferBase - next.bufferBase;
              this.stack = next;
              this.buffer = next.buffer;
          }
      }
      get id() { return this.buffer[this.index - 4]; }
      get start() { return this.buffer[this.index - 3]; }
      get end() { return this.buffer[this.index - 2]; }
      get size() { return this.buffer[this.index - 1]; }
      next() {
          this.index -= 4;
          this.pos -= 4;
          if (this.index == 0)
              this.maybeNext();
      }
      fork() {
          return new StackBufferCursor(this.stack, this.pos, this.index);
      }
  }

  class CachedToken {
      constructor() {
          this.start = -1;
          this.value = -1;
          this.end = -1;
          this.extended = -1;
          this.lookAhead = 0;
          this.mask = 0;
          this.context = 0;
      }
  }
  const nullToken = new CachedToken;
  /// [Tokenizers](#lr.ExternalTokenizer) interact with the input
  /// through this interface. It presents the input as a stream of
  /// characters, tracking lookahead and hiding the complexity of
  /// [ranges](#common.Parser.parse^ranges) from tokenizer code.
  class InputStream {
      /// @internal
      constructor(
      /// @internal
      input, 
      /// @internal
      ranges) {
          this.input = input;
          this.ranges = ranges;
          /// @internal
          this.chunk = "";
          /// @internal
          this.chunkOff = 0;
          /// Backup chunk
          this.chunk2 = "";
          this.chunk2Pos = 0;
          /// The character code of the next code unit in the input, or -1
          /// when the stream is at the end of the input.
          this.next = -1;
          /// @internal
          this.token = nullToken;
          this.rangeIndex = 0;
          this.pos = this.chunkPos = ranges[0].from;
          this.range = ranges[0];
          this.end = ranges[ranges.length - 1].to;
          this.readNext();
      }
      /// @internal
      resolveOffset(offset, assoc) {
          let range = this.range, index = this.rangeIndex;
          let pos = this.pos + offset;
          while (pos < range.from) {
              if (!index)
                  return null;
              let next = this.ranges[--index];
              pos -= range.from - next.to;
              range = next;
          }
          while (assoc < 0 ? pos > range.to : pos >= range.to) {
              if (index == this.ranges.length - 1)
                  return null;
              let next = this.ranges[++index];
              pos += next.from - range.to;
              range = next;
          }
          return pos;
      }
      /// @internal
      clipPos(pos) {
          if (pos >= this.range.from && pos < this.range.to)
              return pos;
          for (let range of this.ranges)
              if (range.to > pos)
                  return Math.max(pos, range.from);
          return this.end;
      }
      /// Look at a code unit near the stream position. `.peek(0)` equals
      /// `.next`, `.peek(-1)` gives you the previous character, and so
      /// on.
      ///
      /// Note that looking around during tokenizing creates dependencies
      /// on potentially far-away content, which may reduce the
      /// effectiveness incremental parsing—when looking forward—or even
      /// cause invalid reparses when looking backward more than 25 code
      /// units, since the library does not track lookbehind.
      peek(offset) {
          let idx = this.chunkOff + offset, pos, result;
          if (idx >= 0 && idx < this.chunk.length) {
              pos = this.pos + offset;
              result = this.chunk.charCodeAt(idx);
          }
          else {
              let resolved = this.resolveOffset(offset, 1);
              if (resolved == null)
                  return -1;
              pos = resolved;
              if (pos >= this.chunk2Pos && pos < this.chunk2Pos + this.chunk2.length) {
                  result = this.chunk2.charCodeAt(pos - this.chunk2Pos);
              }
              else {
                  let i = this.rangeIndex, range = this.range;
                  while (range.to <= pos)
                      range = this.ranges[++i];
                  this.chunk2 = this.input.chunk(this.chunk2Pos = pos);
                  if (pos + this.chunk2.length > range.to)
                      this.chunk2 = this.chunk2.slice(0, range.to - pos);
                  result = this.chunk2.charCodeAt(0);
              }
          }
          if (pos >= this.token.lookAhead)
              this.token.lookAhead = pos + 1;
          return result;
      }
      /// Accept a token. By default, the end of the token is set to the
      /// current stream position, but you can pass an offset (relative to
      /// the stream position) to change that.
      acceptToken(token, endOffset = 0) {
          let end = endOffset ? this.resolveOffset(endOffset, -1) : this.pos;
          if (end == null || end < this.token.start)
              throw new RangeError("Token end out of bounds");
          this.token.value = token;
          this.token.end = end;
      }
      getChunk() {
          if (this.pos >= this.chunk2Pos && this.pos < this.chunk2Pos + this.chunk2.length) {
              let { chunk, chunkPos } = this;
              this.chunk = this.chunk2;
              this.chunkPos = this.chunk2Pos;
              this.chunk2 = chunk;
              this.chunk2Pos = chunkPos;
              this.chunkOff = this.pos - this.chunkPos;
          }
          else {
              this.chunk2 = this.chunk;
              this.chunk2Pos = this.chunkPos;
              let nextChunk = this.input.chunk(this.pos);
              let end = this.pos + nextChunk.length;
              this.chunk = end > this.range.to ? nextChunk.slice(0, this.range.to - this.pos) : nextChunk;
              this.chunkPos = this.pos;
              this.chunkOff = 0;
          }
      }
      readNext() {
          if (this.chunkOff >= this.chunk.length) {
              this.getChunk();
              if (this.chunkOff == this.chunk.length)
                  return this.next = -1;
          }
          return this.next = this.chunk.charCodeAt(this.chunkOff);
      }
      /// Move the stream forward N (defaults to 1) code units. Returns
      /// the new value of [`next`](#lr.InputStream.next).
      advance(n = 1) {
          this.chunkOff += n;
          while (this.pos + n >= this.range.to) {
              if (this.rangeIndex == this.ranges.length - 1)
                  return this.setDone();
              n -= this.range.to - this.pos;
              this.range = this.ranges[++this.rangeIndex];
              this.pos = this.range.from;
          }
          this.pos += n;
          if (this.pos >= this.token.lookAhead)
              this.token.lookAhead = this.pos + 1;
          return this.readNext();
      }
      setDone() {
          this.pos = this.chunkPos = this.end;
          this.range = this.ranges[this.rangeIndex = this.ranges.length - 1];
          this.chunk = "";
          return this.next = -1;
      }
      /// @internal
      reset(pos, token) {
          if (token) {
              this.token = token;
              token.start = pos;
              token.lookAhead = pos + 1;
              token.value = token.extended = -1;
          }
          else {
              this.token = nullToken;
          }
          if (this.pos != pos) {
              this.pos = pos;
              if (pos == this.end) {
                  this.setDone();
                  return this;
              }
              while (pos < this.range.from)
                  this.range = this.ranges[--this.rangeIndex];
              while (pos >= this.range.to)
                  this.range = this.ranges[++this.rangeIndex];
              if (pos >= this.chunkPos && pos < this.chunkPos + this.chunk.length) {
                  this.chunkOff = pos - this.chunkPos;
              }
              else {
                  this.chunk = "";
                  this.chunkOff = 0;
              }
              this.readNext();
          }
          return this;
      }
      /// @internal
      read(from, to) {
          if (from >= this.chunkPos && to <= this.chunkPos + this.chunk.length)
              return this.chunk.slice(from - this.chunkPos, to - this.chunkPos);
          if (from >= this.chunk2Pos && to <= this.chunk2Pos + this.chunk2.length)
              return this.chunk2.slice(from - this.chunk2Pos, to - this.chunk2Pos);
          if (from >= this.range.from && to <= this.range.to)
              return this.input.read(from, to);
          let result = "";
          for (let r of this.ranges) {
              if (r.from >= to)
                  break;
              if (r.to > from)
                  result += this.input.read(Math.max(r.from, from), Math.min(r.to, to));
          }
          return result;
      }
  }
  /// @internal
  class TokenGroup {
      constructor(data, id) {
          this.data = data;
          this.id = id;
      }
      token(input, stack) { readToken(this.data, input, stack, this.id); }
  }
  TokenGroup.prototype.contextual = TokenGroup.prototype.fallback = TokenGroup.prototype.extend = false;
  /// `@external tokens` declarations in the grammar should resolve to
  /// an instance of this class.
  class ExternalTokenizer {
      /// Create a tokenizer. The first argument is the function that,
      /// given an input stream, scans for the types of tokens it
      /// recognizes at the stream's position, and calls
      /// [`acceptToken`](#lr.InputStream.acceptToken) when it finds
      /// one.
      constructor(
      /// @internal
      token, options = {}) {
          this.token = token;
          this.contextual = !!options.contextual;
          this.fallback = !!options.fallback;
          this.extend = !!options.extend;
      }
  }
  // Tokenizer data is stored a big uint16 array containing, for each
  // state:
  //
  //  - A group bitmask, indicating what token groups are reachable from
  //    this state, so that paths that can only lead to tokens not in
  //    any of the current groups can be cut off early.
  //
  //  - The position of the end of the state's sequence of accepting
  //    tokens
  //
  //  - The number of outgoing edges for the state
  //
  //  - The accepting tokens, as (token id, group mask) pairs
  //
  //  - The outgoing edges, as (start character, end character, state
  //    index) triples, with end character being exclusive
  //
  // This function interprets that data, running through a stream as
  // long as new states with the a matching group mask can be reached,
  // and updating `input.token` when it matches a token.
  function readToken(data, input, stack, group) {
      let state = 0, groupMask = 1 << group, { parser } = stack.p, { dialect } = parser;
      scan: for (;;) {
          if ((groupMask & data[state]) == 0)
              break;
          let accEnd = data[state + 1];
          // Check whether this state can lead to a token in the current group
          // Accept tokens in this state, possibly overwriting
          // lower-precedence / shorter tokens
          for (let i = state + 3; i < accEnd; i += 2)
              if ((data[i + 1] & groupMask) > 0) {
                  let term = data[i];
                  if (dialect.allows(term) &&
                      (input.token.value == -1 || input.token.value == term || parser.overrides(term, input.token.value))) {
                      input.acceptToken(term);
                      break;
                  }
              }
          let next = input.next, low = 0, high = data[state + 2];
          // Special case for EOF
          if (input.next < 0 && high > low && data[accEnd + high * 3 - 3] == 65535 /* End */ && data[accEnd + high * 3 - 3] == 65535 /* End */) {
              state = data[accEnd + high * 3 - 1];
              continue scan;
          }
          // Do a binary search on the state's edges
          for (; low < high;) {
              let mid = (low + high) >> 1;
              let index = accEnd + mid + (mid << 1);
              let from = data[index], to = data[index + 1] || 0x10000;
              if (next < from)
                  high = mid;
              else if (next >= to)
                  low = mid + 1;
              else {
                  state = data[index + 2];
                  input.advance();
                  continue scan;
              }
          }
          break;
      }
  }

  // See lezer-generator/src/encode.ts for comments about the encoding
  // used here
  function decodeArray(input, Type = Uint16Array) {
      if (typeof input != "string")
          return input;
      let array = null;
      for (let pos = 0, out = 0; pos < input.length;) {
          let value = 0;
          for (;;) {
              let next = input.charCodeAt(pos++), stop = false;
              if (next == 126 /* BigValCode */) {
                  value = 65535 /* BigVal */;
                  break;
              }
              if (next >= 92 /* Gap2 */)
                  next--;
              if (next >= 34 /* Gap1 */)
                  next--;
              let digit = next - 32 /* Start */;
              if (digit >= 46 /* Base */) {
                  digit -= 46 /* Base */;
                  stop = true;
              }
              value += digit;
              if (stop)
                  break;
              value *= 46 /* Base */;
          }
          if (array)
              array[out++] = value;
          else
              array = new Type(value);
      }
      return array;
  }

  // Environment variable used to control console output
  const verbose = typeof process != "undefined" && process.env && /\bparse\b/.test(process.env.LOG);
  let stackIDs = null;
  var Safety;
  (function (Safety) {
      Safety[Safety["Margin"] = 25] = "Margin";
  })(Safety || (Safety = {}));
  function cutAt(tree, pos, side) {
      let cursor = tree.cursor(IterMode.IncludeAnonymous);
      cursor.moveTo(pos);
      for (;;) {
          if (!(side < 0 ? cursor.childBefore(pos) : cursor.childAfter(pos)))
              for (;;) {
                  if ((side < 0 ? cursor.to < pos : cursor.from > pos) && !cursor.type.isError)
                      return side < 0 ? Math.max(0, Math.min(cursor.to - 1, pos - 25 /* Margin */))
                          : Math.min(tree.length, Math.max(cursor.from + 1, pos + 25 /* Margin */));
                  if (side < 0 ? cursor.prevSibling() : cursor.nextSibling())
                      break;
                  if (!cursor.parent())
                      return side < 0 ? 0 : tree.length;
              }
      }
  }
  let FragmentCursor$1 = class FragmentCursor {
      constructor(fragments, nodeSet) {
          this.fragments = fragments;
          this.nodeSet = nodeSet;
          this.i = 0;
          this.fragment = null;
          this.safeFrom = -1;
          this.safeTo = -1;
          this.trees = [];
          this.start = [];
          this.index = [];
          this.nextFragment();
      }
      nextFragment() {
          let fr = this.fragment = this.i == this.fragments.length ? null : this.fragments[this.i++];
          if (fr) {
              this.safeFrom = fr.openStart ? cutAt(fr.tree, fr.from + fr.offset, 1) - fr.offset : fr.from;
              this.safeTo = fr.openEnd ? cutAt(fr.tree, fr.to + fr.offset, -1) - fr.offset : fr.to;
              while (this.trees.length) {
                  this.trees.pop();
                  this.start.pop();
                  this.index.pop();
              }
              this.trees.push(fr.tree);
              this.start.push(-fr.offset);
              this.index.push(0);
              this.nextStart = this.safeFrom;
          }
          else {
              this.nextStart = 1e9;
          }
      }
      // `pos` must be >= any previously given `pos` for this cursor
      nodeAt(pos) {
          if (pos < this.nextStart)
              return null;
          while (this.fragment && this.safeTo <= pos)
              this.nextFragment();
          if (!this.fragment)
              return null;
          for (;;) {
              let last = this.trees.length - 1;
              if (last < 0) { // End of tree
                  this.nextFragment();
                  return null;
              }
              let top = this.trees[last], index = this.index[last];
              if (index == top.children.length) {
                  this.trees.pop();
                  this.start.pop();
                  this.index.pop();
                  continue;
              }
              let next = top.children[index];
              let start = this.start[last] + top.positions[index];
              if (start > pos) {
                  this.nextStart = start;
                  return null;
              }
              if (next instanceof Tree) {
                  if (start == pos) {
                      if (start < this.safeFrom)
                          return null;
                      let end = start + next.length;
                      if (end <= this.safeTo) {
                          let lookAhead = next.prop(NodeProp.lookAhead);
                          if (!lookAhead || end + lookAhead < this.fragment.to)
                              return next;
                      }
                  }
                  this.index[last]++;
                  if (start + next.length >= Math.max(this.safeFrom, pos)) { // Enter this node
                      this.trees.push(next);
                      this.start.push(start);
                      this.index.push(0);
                  }
              }
              else {
                  this.index[last]++;
                  this.nextStart = start + next.length;
              }
          }
      }
  };
  class TokenCache {
      constructor(parser, stream) {
          this.stream = stream;
          this.tokens = [];
          this.mainToken = null;
          this.actions = [];
          this.tokens = parser.tokenizers.map(_ => new CachedToken);
      }
      getActions(stack) {
          let actionIndex = 0;
          let main = null;
          let { parser } = stack.p, { tokenizers } = parser;
          let mask = parser.stateSlot(stack.state, 3 /* TokenizerMask */);
          let context = stack.curContext ? stack.curContext.hash : 0;
          let lookAhead = 0;
          for (let i = 0; i < tokenizers.length; i++) {
              if (((1 << i) & mask) == 0)
                  continue;
              let tokenizer = tokenizers[i], token = this.tokens[i];
              if (main && !tokenizer.fallback)
                  continue;
              if (tokenizer.contextual || token.start != stack.pos || token.mask != mask || token.context != context) {
                  this.updateCachedToken(token, tokenizer, stack);
                  token.mask = mask;
                  token.context = context;
              }
              if (token.lookAhead > token.end + 25 /* Margin */)
                  lookAhead = Math.max(token.lookAhead, lookAhead);
              if (token.value != 0 /* Err */) {
                  let startIndex = actionIndex;
                  if (token.extended > -1)
                      actionIndex = this.addActions(stack, token.extended, token.end, actionIndex);
                  actionIndex = this.addActions(stack, token.value, token.end, actionIndex);
                  if (!tokenizer.extend) {
                      main = token;
                      if (actionIndex > startIndex)
                          break;
                  }
              }
          }
          while (this.actions.length > actionIndex)
              this.actions.pop();
          if (lookAhead)
              stack.setLookAhead(lookAhead);
          if (!main && stack.pos == this.stream.end) {
              main = new CachedToken;
              main.value = stack.p.parser.eofTerm;
              main.start = main.end = stack.pos;
              actionIndex = this.addActions(stack, main.value, main.end, actionIndex);
          }
          this.mainToken = main;
          return this.actions;
      }
      getMainToken(stack) {
          if (this.mainToken)
              return this.mainToken;
          let main = new CachedToken, { pos, p } = stack;
          main.start = pos;
          main.end = Math.min(pos + 1, p.stream.end);
          main.value = pos == p.stream.end ? p.parser.eofTerm : 0 /* Err */;
          return main;
      }
      updateCachedToken(token, tokenizer, stack) {
          let start = this.stream.clipPos(stack.pos);
          tokenizer.token(this.stream.reset(start, token), stack);
          if (token.value > -1) {
              let { parser } = stack.p;
              for (let i = 0; i < parser.specialized.length; i++)
                  if (parser.specialized[i] == token.value) {
                      let result = parser.specializers[i](this.stream.read(token.start, token.end), stack);
                      if (result >= 0 && stack.p.parser.dialect.allows(result >> 1)) {
                          if ((result & 1) == 0 /* Specialize */)
                              token.value = result >> 1;
                          else
                              token.extended = result >> 1;
                          break;
                      }
                  }
          }
          else {
              token.value = 0 /* Err */;
              token.end = this.stream.clipPos(start + 1);
          }
      }
      putAction(action, token, end, index) {
          // Don't add duplicate actions
          for (let i = 0; i < index; i += 3)
              if (this.actions[i] == action)
                  return index;
          this.actions[index++] = action;
          this.actions[index++] = token;
          this.actions[index++] = end;
          return index;
      }
      addActions(stack, token, end, index) {
          let { state } = stack, { parser } = stack.p, { data } = parser;
          for (let set = 0; set < 2; set++) {
              for (let i = parser.stateSlot(state, set ? 2 /* Skip */ : 1 /* Actions */);; i += 3) {
                  if (data[i] == 65535 /* End */) {
                      if (data[i + 1] == 1 /* Next */) {
                          i = pair(data, i + 2);
                      }
                      else {
                          if (index == 0 && data[i + 1] == 2 /* Other */)
                              index = this.putAction(pair(data, i + 2), token, end, index);
                          break;
                      }
                  }
                  if (data[i] == token)
                      index = this.putAction(pair(data, i + 1), token, end, index);
              }
          }
          return index;
      }
  }
  var Rec;
  (function (Rec) {
      Rec[Rec["Distance"] = 5] = "Distance";
      Rec[Rec["MaxRemainingPerStep"] = 3] = "MaxRemainingPerStep";
      // When two stacks have been running independently long enough to
      // add this many elements to their buffers, prune one.
      Rec[Rec["MinBufferLengthPrune"] = 500] = "MinBufferLengthPrune";
      Rec[Rec["ForceReduceLimit"] = 10] = "ForceReduceLimit";
      // Once a stack reaches this depth (in .stack.length) force-reduce
      // it back to CutTo to avoid creating trees that overflow the stack
      // on recursive traversal.
      Rec[Rec["CutDepth"] = 15000] = "CutDepth";
      Rec[Rec["CutTo"] = 9000] = "CutTo";
  })(Rec || (Rec = {}));
  class Parse {
      constructor(parser, input, fragments, ranges) {
          this.parser = parser;
          this.input = input;
          this.ranges = ranges;
          this.recovering = 0;
          this.nextStackID = 0x2654; // ♔, ♕, ♖, ♗, ♘, ♙, ♠, ♡, ♢, ♣, ♤, ♥, ♦, ♧
          this.minStackPos = 0;
          this.reused = [];
          this.stoppedAt = null;
          this.stream = new InputStream(input, ranges);
          this.tokens = new TokenCache(parser, this.stream);
          this.topTerm = parser.top[1];
          let { from } = ranges[0];
          this.stacks = [Stack.start(this, parser.top[0], from)];
          this.fragments = fragments.length && this.stream.end - from > parser.bufferLength * 4
              ? new FragmentCursor$1(fragments, parser.nodeSet) : null;
      }
      get parsedPos() {
          return this.minStackPos;
      }
      // Move the parser forward. This will process all parse stacks at
      // `this.pos` and try to advance them to a further position. If no
      // stack for such a position is found, it'll start error-recovery.
      //
      // When the parse is finished, this will return a syntax tree. When
      // not, it returns `null`.
      advance() {
          let stacks = this.stacks, pos = this.minStackPos;
          // This will hold stacks beyond `pos`.
          let newStacks = this.stacks = [];
          let stopped, stoppedTokens;
          // Keep advancing any stacks at `pos` until they either move
          // forward or can't be advanced. Gather stacks that can't be
          // advanced further in `stopped`.
          for (let i = 0; i < stacks.length; i++) {
              let stack = stacks[i];
              for (;;) {
                  this.tokens.mainToken = null;
                  if (stack.pos > pos) {
                      newStacks.push(stack);
                  }
                  else if (this.advanceStack(stack, newStacks, stacks)) {
                      continue;
                  }
                  else {
                      if (!stopped) {
                          stopped = [];
                          stoppedTokens = [];
                      }
                      stopped.push(stack);
                      let tok = this.tokens.getMainToken(stack);
                      stoppedTokens.push(tok.value, tok.end);
                  }
                  break;
              }
          }
          if (!newStacks.length) {
              let finished = stopped && findFinished(stopped);
              if (finished)
                  return this.stackToTree(finished);
              if (this.parser.strict) {
                  if (verbose && stopped)
                      console.log("Stuck with token " + (this.tokens.mainToken ? this.parser.getName(this.tokens.mainToken.value) : "none"));
                  throw new SyntaxError("No parse at " + pos);
              }
              if (!this.recovering)
                  this.recovering = 5 /* Distance */;
          }
          if (this.recovering && stopped) {
              let finished = this.stoppedAt != null && stopped[0].pos > this.stoppedAt ? stopped[0]
                  : this.runRecovery(stopped, stoppedTokens, newStacks);
              if (finished)
                  return this.stackToTree(finished.forceAll());
          }
          if (this.recovering) {
              let maxRemaining = this.recovering == 1 ? 1 : this.recovering * 3 /* MaxRemainingPerStep */;
              if (newStacks.length > maxRemaining) {
                  newStacks.sort((a, b) => b.score - a.score);
                  while (newStacks.length > maxRemaining)
                      newStacks.pop();
              }
              if (newStacks.some(s => s.reducePos > pos))
                  this.recovering--;
          }
          else if (newStacks.length > 1) {
              // Prune stacks that are in the same state, or that have been
              // running without splitting for a while, to avoid getting stuck
              // with multiple successful stacks running endlessly on.
              outer: for (let i = 0; i < newStacks.length - 1; i++) {
                  let stack = newStacks[i];
                  for (let j = i + 1; j < newStacks.length; j++) {
                      let other = newStacks[j];
                      if (stack.sameState(other) ||
                          stack.buffer.length > 500 /* MinBufferLengthPrune */ && other.buffer.length > 500 /* MinBufferLengthPrune */) {
                          if (((stack.score - other.score) || (stack.buffer.length - other.buffer.length)) > 0) {
                              newStacks.splice(j--, 1);
                          }
                          else {
                              newStacks.splice(i--, 1);
                              continue outer;
                          }
                      }
                  }
              }
          }
          this.minStackPos = newStacks[0].pos;
          for (let i = 1; i < newStacks.length; i++)
              if (newStacks[i].pos < this.minStackPos)
                  this.minStackPos = newStacks[i].pos;
          return null;
      }
      stopAt(pos) {
          if (this.stoppedAt != null && this.stoppedAt < pos)
              throw new RangeError("Can't move stoppedAt forward");
          this.stoppedAt = pos;
      }
      // Returns an updated version of the given stack, or null if the
      // stack can't advance normally. When `split` and `stacks` are
      // given, stacks split off by ambiguous operations will be pushed to
      // `split`, or added to `stacks` if they move `pos` forward.
      advanceStack(stack, stacks, split) {
          let start = stack.pos, { parser } = this;
          let base = verbose ? this.stackID(stack) + " -> " : "";
          if (this.stoppedAt != null && start > this.stoppedAt)
              return stack.forceReduce() ? stack : null;
          if (this.fragments) {
              let strictCx = stack.curContext && stack.curContext.tracker.strict, cxHash = strictCx ? stack.curContext.hash : 0;
              for (let cached = this.fragments.nodeAt(start); cached;) {
                  let match = this.parser.nodeSet.types[cached.type.id] == cached.type ? parser.getGoto(stack.state, cached.type.id) : -1;
                  if (match > -1 && cached.length && (!strictCx || (cached.prop(NodeProp.contextHash) || 0) == cxHash)) {
                      stack.useNode(cached, match);
                      if (verbose)
                          console.log(base + this.stackID(stack) + ` (via reuse of ${parser.getName(cached.type.id)})`);
                      return true;
                  }
                  if (!(cached instanceof Tree) || cached.children.length == 0 || cached.positions[0] > 0)
                      break;
                  let inner = cached.children[0];
                  if (inner instanceof Tree && cached.positions[0] == 0)
                      cached = inner;
                  else
                      break;
              }
          }
          let defaultReduce = parser.stateSlot(stack.state, 4 /* DefaultReduce */);
          if (defaultReduce > 0) {
              stack.reduce(defaultReduce);
              if (verbose)
                  console.log(base + this.stackID(stack) + ` (via always-reduce ${parser.getName(defaultReduce & 65535 /* ValueMask */)})`);
              return true;
          }
          if (stack.stack.length >= 15000 /* CutDepth */) {
              while (stack.stack.length > 9000 /* CutTo */ && stack.forceReduce()) { }
          }
          let actions = this.tokens.getActions(stack);
          for (let i = 0; i < actions.length;) {
              let action = actions[i++], term = actions[i++], end = actions[i++];
              let last = i == actions.length || !split;
              let localStack = last ? stack : stack.split();
              localStack.apply(action, term, end);
              if (verbose)
                  console.log(base + this.stackID(localStack) + ` (via ${(action & 65536 /* ReduceFlag */) == 0 ? "shift"
                    : `reduce of ${parser.getName(action & 65535 /* ValueMask */)}`} for ${parser.getName(term)} @ ${start}${localStack == stack ? "" : ", split"})`);
              if (last)
                  return true;
              else if (localStack.pos > start)
                  stacks.push(localStack);
              else
                  split.push(localStack);
          }
          return false;
      }
      // Advance a given stack forward as far as it will go. Returns the
      // (possibly updated) stack if it got stuck, or null if it moved
      // forward and was given to `pushStackDedup`.
      advanceFully(stack, newStacks) {
          let pos = stack.pos;
          for (;;) {
              if (!this.advanceStack(stack, null, null))
                  return false;
              if (stack.pos > pos) {
                  pushStackDedup(stack, newStacks);
                  return true;
              }
          }
      }
      runRecovery(stacks, tokens, newStacks) {
          let finished = null, restarted = false;
          for (let i = 0; i < stacks.length; i++) {
              let stack = stacks[i], token = tokens[i << 1], tokenEnd = tokens[(i << 1) + 1];
              let base = verbose ? this.stackID(stack) + " -> " : "";
              if (stack.deadEnd) {
                  if (restarted)
                      continue;
                  restarted = true;
                  stack.restart();
                  if (verbose)
                      console.log(base + this.stackID(stack) + " (restarted)");
                  let done = this.advanceFully(stack, newStacks);
                  if (done)
                      continue;
              }
              let force = stack.split(), forceBase = base;
              for (let j = 0; force.forceReduce() && j < 10 /* ForceReduceLimit */; j++) {
                  if (verbose)
                      console.log(forceBase + this.stackID(force) + " (via force-reduce)");
                  let done = this.advanceFully(force, newStacks);
                  if (done)
                      break;
                  if (verbose)
                      forceBase = this.stackID(force) + " -> ";
              }
              for (let insert of stack.recoverByInsert(token)) {
                  if (verbose)
                      console.log(base + this.stackID(insert) + " (via recover-insert)");
                  this.advanceFully(insert, newStacks);
              }
              if (this.stream.end > stack.pos) {
                  if (tokenEnd == stack.pos) {
                      tokenEnd++;
                      token = 0 /* Err */;
                  }
                  stack.recoverByDelete(token, tokenEnd);
                  if (verbose)
                      console.log(base + this.stackID(stack) + ` (via recover-delete ${this.parser.getName(token)})`);
                  pushStackDedup(stack, newStacks);
              }
              else if (!finished || finished.score < stack.score) {
                  finished = stack;
              }
          }
          return finished;
      }
      // Convert the stack's buffer to a syntax tree.
      stackToTree(stack) {
          stack.close();
          return Tree.build({ buffer: StackBufferCursor.create(stack),
              nodeSet: this.parser.nodeSet,
              topID: this.topTerm,
              maxBufferLength: this.parser.bufferLength,
              reused: this.reused,
              start: this.ranges[0].from,
              length: stack.pos - this.ranges[0].from,
              minRepeatType: this.parser.minRepeatTerm });
      }
      stackID(stack) {
          let id = (stackIDs || (stackIDs = new WeakMap)).get(stack);
          if (!id)
              stackIDs.set(stack, id = String.fromCodePoint(this.nextStackID++));
          return id + stack;
      }
  }
  function pushStackDedup(stack, newStacks) {
      for (let i = 0; i < newStacks.length; i++) {
          let other = newStacks[i];
          if (other.pos == stack.pos && other.sameState(stack)) {
              if (newStacks[i].score < stack.score)
                  newStacks[i] = stack;
              return;
          }
      }
      newStacks.push(stack);
  }
  class Dialect {
      constructor(source, flags, disabled) {
          this.source = source;
          this.flags = flags;
          this.disabled = disabled;
      }
      allows(term) { return !this.disabled || this.disabled[term] == 0; }
  }
  const id = x => x;
  /// Context trackers are used to track stateful context (such as
  /// indentation in the Python grammar, or parent elements in the XML
  /// grammar) needed by external tokenizers. You declare them in a
  /// grammar file as `@context exportName from "module"`.
  ///
  /// Context values should be immutable, and can be updated (replaced)
  /// on shift or reduce actions.
  ///
  /// The export used in a `@context` declaration should be of this
  /// type.
  class ContextTracker {
      /// Define a context tracker.
      constructor(spec) {
          this.start = spec.start;
          this.shift = spec.shift || id;
          this.reduce = spec.reduce || id;
          this.reuse = spec.reuse || id;
          this.hash = spec.hash || (() => 0);
          this.strict = spec.strict !== false;
      }
  }
  /// Holds the parse tables for a given grammar, as generated by
  /// `lezer-generator`, and provides [methods](#common.Parser) to parse
  /// content with.
  class LRParser extends Parser {
      /// @internal
      constructor(spec) {
          super();
          /// @internal
          this.wrappers = [];
          if (spec.version != 14 /* Version */)
              throw new RangeError(`Parser version (${spec.version}) doesn't match runtime version (${14 /* Version */})`);
          let nodeNames = spec.nodeNames.split(" ");
          this.minRepeatTerm = nodeNames.length;
          for (let i = 0; i < spec.repeatNodeCount; i++)
              nodeNames.push("");
          let topTerms = Object.keys(spec.topRules).map(r => spec.topRules[r][1]);
          let nodeProps = [];
          for (let i = 0; i < nodeNames.length; i++)
              nodeProps.push([]);
          function setProp(nodeID, prop, value) {
              nodeProps[nodeID].push([prop, prop.deserialize(String(value))]);
          }
          if (spec.nodeProps)
              for (let propSpec of spec.nodeProps) {
                  let prop = propSpec[0];
                  if (typeof prop == "string")
                      prop = NodeProp[prop];
                  for (let i = 1; i < propSpec.length;) {
                      let next = propSpec[i++];
                      if (next >= 0) {
                          setProp(next, prop, propSpec[i++]);
                      }
                      else {
                          let value = propSpec[i + -next];
                          for (let j = -next; j > 0; j--)
                              setProp(propSpec[i++], prop, value);
                          i++;
                      }
                  }
              }
          this.nodeSet = new NodeSet(nodeNames.map((name, i) => NodeType.define({
              name: i >= this.minRepeatTerm ? undefined : name,
              id: i,
              props: nodeProps[i],
              top: topTerms.indexOf(i) > -1,
              error: i == 0,
              skipped: spec.skippedNodes && spec.skippedNodes.indexOf(i) > -1
          })));
          if (spec.propSources)
              this.nodeSet = this.nodeSet.extend(...spec.propSources);
          this.strict = false;
          this.bufferLength = DefaultBufferLength;
          let tokenArray = decodeArray(spec.tokenData);
          this.context = spec.context;
          this.specializerSpecs = spec.specialized || [];
          this.specialized = new Uint16Array(this.specializerSpecs.length);
          for (let i = 0; i < this.specializerSpecs.length; i++)
              this.specialized[i] = this.specializerSpecs[i].term;
          this.specializers = this.specializerSpecs.map(getSpecializer);
          this.states = decodeArray(spec.states, Uint32Array);
          this.data = decodeArray(spec.stateData);
          this.goto = decodeArray(spec.goto);
          this.maxTerm = spec.maxTerm;
          this.tokenizers = spec.tokenizers.map(value => typeof value == "number" ? new TokenGroup(tokenArray, value) : value);
          this.topRules = spec.topRules;
          this.dialects = spec.dialects || {};
          this.dynamicPrecedences = spec.dynamicPrecedences || null;
          this.tokenPrecTable = spec.tokenPrec;
          this.termNames = spec.termNames || null;
          this.maxNode = this.nodeSet.types.length - 1;
          this.dialect = this.parseDialect();
          this.top = this.topRules[Object.keys(this.topRules)[0]];
      }
      createParse(input, fragments, ranges) {
          let parse = new Parse(this, input, fragments, ranges);
          for (let w of this.wrappers)
              parse = w(parse, input, fragments, ranges);
          return parse;
      }
      /// Get a goto table entry @internal
      getGoto(state, term, loose = false) {
          let table = this.goto;
          if (term >= table[0])
              return -1;
          for (let pos = table[term + 1];;) {
              let groupTag = table[pos++], last = groupTag & 1;
              let target = table[pos++];
              if (last && loose)
                  return target;
              for (let end = pos + (groupTag >> 1); pos < end; pos++)
                  if (table[pos] == state)
                      return target;
              if (last)
                  return -1;
          }
      }
      /// Check if this state has an action for a given terminal @internal
      hasAction(state, terminal) {
          let data = this.data;
          for (let set = 0; set < 2; set++) {
              for (let i = this.stateSlot(state, set ? 2 /* Skip */ : 1 /* Actions */), next;; i += 3) {
                  if ((next = data[i]) == 65535 /* End */) {
                      if (data[i + 1] == 1 /* Next */)
                          next = data[i = pair(data, i + 2)];
                      else if (data[i + 1] == 2 /* Other */)
                          return pair(data, i + 2);
                      else
                          break;
                  }
                  if (next == terminal || next == 0 /* Err */)
                      return pair(data, i + 1);
              }
          }
          return 0;
      }
      /// @internal
      stateSlot(state, slot) {
          return this.states[(state * 6 /* Size */) + slot];
      }
      /// @internal
      stateFlag(state, flag) {
          return (this.stateSlot(state, 0 /* Flags */) & flag) > 0;
      }
      /// @internal
      validAction(state, action) {
          if (action == this.stateSlot(state, 4 /* DefaultReduce */))
              return true;
          for (let i = this.stateSlot(state, 1 /* Actions */);; i += 3) {
              if (this.data[i] == 65535 /* End */) {
                  if (this.data[i + 1] == 1 /* Next */)
                      i = pair(this.data, i + 2);
                  else
                      return false;
              }
              if (action == pair(this.data, i + 1))
                  return true;
          }
      }
      /// Get the states that can follow this one through shift actions or
      /// goto jumps. @internal
      nextStates(state) {
          let result = [];
          for (let i = this.stateSlot(state, 1 /* Actions */);; i += 3) {
              if (this.data[i] == 65535 /* End */) {
                  if (this.data[i + 1] == 1 /* Next */)
                      i = pair(this.data, i + 2);
                  else
                      break;
              }
              if ((this.data[i + 2] & (65536 /* ReduceFlag */ >> 16)) == 0) {
                  let value = this.data[i + 1];
                  if (!result.some((v, i) => (i & 1) && v == value))
                      result.push(this.data[i], value);
              }
          }
          return result;
      }
      /// @internal
      overrides(token, prev) {
          let iPrev = findOffset(this.data, this.tokenPrecTable, prev);
          return iPrev < 0 || findOffset(this.data, this.tokenPrecTable, token) < iPrev;
      }
      /// Configure the parser. Returns a new parser instance that has the
      /// given settings modified. Settings not provided in `config` are
      /// kept from the original parser.
      configure(config) {
          // Hideous reflection-based kludge to make it easy to create a
          // slightly modified copy of a parser.
          let copy = Object.assign(Object.create(LRParser.prototype), this);
          if (config.props)
              copy.nodeSet = this.nodeSet.extend(...config.props);
          if (config.top) {
              let info = this.topRules[config.top];
              if (!info)
                  throw new RangeError(`Invalid top rule name ${config.top}`);
              copy.top = info;
          }
          if (config.tokenizers)
              copy.tokenizers = this.tokenizers.map(t => {
                  let found = config.tokenizers.find(r => r.from == t);
                  return found ? found.to : t;
              });
          if (config.specializers) {
              copy.specializers = this.specializers.slice();
              copy.specializerSpecs = this.specializerSpecs.map((s, i) => {
                  let found = config.specializers.find(r => r.from == s.external);
                  if (!found)
                      return s;
                  let spec = Object.assign(Object.assign({}, s), { external: found.to });
                  copy.specializers[i] = getSpecializer(spec);
                  return spec;
              });
          }
          if (config.contextTracker)
              copy.context = config.contextTracker;
          if (config.dialect)
              copy.dialect = this.parseDialect(config.dialect);
          if (config.strict != null)
              copy.strict = config.strict;
          if (config.wrap)
              copy.wrappers = copy.wrappers.concat(config.wrap);
          if (config.bufferLength != null)
              copy.bufferLength = config.bufferLength;
          return copy;
      }
      /// Tells you whether any [parse wrappers](#lr.ParserConfig.wrap)
      /// are registered for this parser.
      hasWrappers() {
          return this.wrappers.length > 0;
      }
      /// Returns the name associated with a given term. This will only
      /// work for all terms when the parser was generated with the
      /// `--names` option. By default, only the names of tagged terms are
      /// stored.
      getName(term) {
          return this.termNames ? this.termNames[term] : String(term <= this.maxNode && this.nodeSet.types[term].name || term);
      }
      /// The eof term id is always allocated directly after the node
      /// types. @internal
      get eofTerm() { return this.maxNode + 1; }
      /// The type of top node produced by the parser.
      get topNode() { return this.nodeSet.types[this.top[1]]; }
      /// @internal
      dynamicPrecedence(term) {
          let prec = this.dynamicPrecedences;
          return prec == null ? 0 : prec[term] || 0;
      }
      /// @internal
      parseDialect(dialect) {
          let values = Object.keys(this.dialects), flags = values.map(() => false);
          if (dialect)
              for (let part of dialect.split(" ")) {
                  let id = values.indexOf(part);
                  if (id >= 0)
                      flags[id] = true;
              }
          let disabled = null;
          for (let i = 0; i < values.length; i++)
              if (!flags[i]) {
                  for (let j = this.dialects[values[i]], id; (id = this.data[j++]) != 65535 /* End */;)
                      (disabled || (disabled = new Uint8Array(this.maxTerm + 1)))[id] = 1;
              }
          return new Dialect(dialect, flags, disabled);
      }
      /// Used by the output of the parser generator. Not available to
      /// user code.
      static deserialize(spec) {
          return new LRParser(spec);
      }
  }
  function pair(data, off) { return data[off] | (data[off + 1] << 16); }
  function findOffset(data, start, term) {
      for (let i = start, next; (next = data[i]) != 65535 /* End */; i++)
          if (next == term)
              return i - start;
      return -1;
  }
  function findFinished(stacks) {
      let best = null;
      for (let stack of stacks) {
          let stopped = stack.p.stoppedAt;
          if ((stack.pos == stack.p.stream.end || stopped != null && stack.pos > stopped) &&
              stack.p.parser.stateFlag(stack.state, 2 /* Accepting */) &&
              (!best || best.score < stack.score))
              best = stack;
      }
      return best;
  }
  function getSpecializer(spec) {
      if (spec.external) {
          let mask = spec.extend ? 1 /* Extend */ : 0 /* Specialize */;
          return (value, stack) => (spec.external(value, stack) << 1) | mask;
      }
      return spec.get;
  }

  let nextTagID = 0;
  /**
  Highlighting tags are markers that denote a highlighting category.
  They are [associated](#highlight.styleTags) with parts of a syntax
  tree by a language mode, and then mapped to an actual CSS style by
  a [highlighter](#highlight.Highlighter).

  Because syntax tree node types and highlight styles have to be
  able to talk the same language, CodeMirror uses a mostly _closed_
  [vocabulary](#highlight.tags) of syntax tags (as opposed to
  traditional open string-based systems, which make it hard for
  highlighting themes to cover all the tokens produced by the
  various languages).

  It _is_ possible to [define](#highlight.Tag^define) your own
  highlighting tags for system-internal use (where you control both
  the language package and the highlighter), but such tags will not
  be picked up by regular highlighters (though you can derive them
  from standard tags to allow highlighters to fall back to those).
  */
  class Tag {
      /**
      @internal
      */
      constructor(
      /**
      The set of this tag and all its parent tags, starting with
      this one itself and sorted in order of decreasing specificity.
      */
      set, 
      /**
      The base unmodified tag that this one is based on, if it's
      modified @internal
      */
      base, 
      /**
      The modifiers applied to this.base @internal
      */
      modified) {
          this.set = set;
          this.base = base;
          this.modified = modified;
          /**
          @internal
          */
          this.id = nextTagID++;
      }
      /**
      Define a new tag. If `parent` is given, the tag is treated as a
      sub-tag of that parent, and
      [highlighters](#highlight.tagHighlighter) that don't mention
      this tag will try to fall back to the parent tag (or grandparent
      tag, etc).
      */
      static define(parent) {
          if (parent === null || parent === void 0 ? void 0 : parent.base)
              throw new Error("Can not derive from a modified tag");
          let tag = new Tag([], null, []);
          tag.set.push(tag);
          if (parent)
              for (let t of parent.set)
                  tag.set.push(t);
          return tag;
      }
      /**
      Define a tag _modifier_, which is a function that, given a tag,
      will return a tag that is a subtag of the original. Applying the
      same modifier to a twice tag will return the same value (`m1(t1)
      == m1(t1)`) and applying multiple modifiers will, regardless or
      order, produce the same tag (`m1(m2(t1)) == m2(m1(t1))`).
      
      When multiple modifiers are applied to a given base tag, each
      smaller set of modifiers is registered as a parent, so that for
      example `m1(m2(m3(t1)))` is a subtype of `m1(m2(t1))`,
      `m1(m3(t1)`, and so on.
      */
      static defineModifier() {
          let mod = new Modifier;
          return (tag) => {
              if (tag.modified.indexOf(mod) > -1)
                  return tag;
              return Modifier.get(tag.base || tag, tag.modified.concat(mod).sort((a, b) => a.id - b.id));
          };
      }
  }
  let nextModifierID = 0;
  class Modifier {
      constructor() {
          this.instances = [];
          this.id = nextModifierID++;
      }
      static get(base, mods) {
          if (!mods.length)
              return base;
          let exists = mods[0].instances.find(t => t.base == base && sameArray$1(mods, t.modified));
          if (exists)
              return exists;
          let set = [], tag = new Tag(set, base, mods);
          for (let m of mods)
              m.instances.push(tag);
          let configs = powerSet(mods);
          for (let parent of base.set)
              if (!parent.modified.length)
                  for (let config of configs)
                      set.push(Modifier.get(parent, config));
          return tag;
      }
  }
  function sameArray$1(a, b) {
      return a.length == b.length && a.every((x, i) => x == b[i]);
  }
  function powerSet(array) {
      let sets = [[]];
      for (let i = 0; i < array.length; i++) {
          for (let j = 0, e = sets.length; j < e; j++) {
              sets.push(sets[j].concat(array[i]));
          }
      }
      return sets.sort((a, b) => b.length - a.length);
  }
  /**
  This function is used to add a set of tags to a language syntax
  via [`NodeSet.extend`](#common.NodeSet.extend) or
  [`LRParser.configure`](#lr.LRParser.configure).

  The argument object maps node selectors to [highlighting
  tags](#highlight.Tag) or arrays of tags.

  Node selectors may hold one or more (space-separated) node paths.
  Such a path can be a [node name](#common.NodeType.name), or
  multiple node names (or `*` wildcards) separated by slash
  characters, as in `"Block/Declaration/VariableName"`. Such a path
  matches the final node but only if its direct parent nodes are the
  other nodes mentioned. A `*` in such a path matches any parent,
  but only a single level—wildcards that match multiple parents
  aren't supported, both for efficiency reasons and because Lezer
  trees make it rather hard to reason about what they would match.)

  A path can be ended with `/...` to indicate that the tag assigned
  to the node should also apply to all child nodes, even if they
  match their own style (by default, only the innermost style is
  used).

  When a path ends in `!`, as in `Attribute!`, no further matching
  happens for the node's child nodes, and the entire node gets the
  given style.

  In this notation, node names that contain `/`, `!`, `*`, or `...`
  must be quoted as JSON strings.

  For example:

  ```javascript
  parser.withProps(
    styleTags({
      // Style Number and BigNumber nodes
      "Number BigNumber": tags.number,
      // Style Escape nodes whose parent is String
      "String/Escape": tags.escape,
      // Style anything inside Attributes nodes
      "Attributes!": tags.meta,
      // Add a style to all content inside Italic nodes
      "Italic/...": tags.emphasis,
      // Style InvalidString nodes as both `string` and `invalid`
      "InvalidString": [tags.string, tags.invalid],
      // Style the node named "/" as punctuation
      '"/"': tags.punctuation
    })
  )
  ```
  */
  function styleTags(spec) {
      let byName = Object.create(null);
      for (let prop in spec) {
          let tags = spec[prop];
          if (!Array.isArray(tags))
              tags = [tags];
          for (let part of prop.split(" "))
              if (part) {
                  let pieces = [], mode = 2 /* Normal */, rest = part;
                  for (let pos = 0;;) {
                      if (rest == "..." && pos > 0 && pos + 3 == part.length) {
                          mode = 1 /* Inherit */;
                          break;
                      }
                      let m = /^"(?:[^"\\]|\\.)*?"|[^\/!]+/.exec(rest);
                      if (!m)
                          throw new RangeError("Invalid path: " + part);
                      pieces.push(m[0] == "*" ? "" : m[0][0] == '"' ? JSON.parse(m[0]) : m[0]);
                      pos += m[0].length;
                      if (pos == part.length)
                          break;
                      let next = part[pos++];
                      if (pos == part.length && next == "!") {
                          mode = 0 /* Opaque */;
                          break;
                      }
                      if (next != "/")
                          throw new RangeError("Invalid path: " + part);
                      rest = part.slice(pos);
                  }
                  let last = pieces.length - 1, inner = pieces[last];
                  if (!inner)
                      throw new RangeError("Invalid path: " + part);
                  let rule = new Rule(tags, mode, last > 0 ? pieces.slice(0, last) : null);
                  byName[inner] = rule.sort(byName[inner]);
              }
      }
      return ruleNodeProp.add(byName);
  }
  const ruleNodeProp = new NodeProp();
  class Rule {
      constructor(tags, mode, context, next) {
          this.tags = tags;
          this.mode = mode;
          this.context = context;
          this.next = next;
      }
      get opaque() { return this.mode == 0 /* Opaque */; }
      get inherit() { return this.mode == 1 /* Inherit */; }
      sort(other) {
          if (!other || other.depth < this.depth) {
              this.next = other;
              return this;
          }
          other.next = this.sort(other.next);
          return other;
      }
      get depth() { return this.context ? this.context.length : 0; }
  }
  Rule.empty = new Rule([], 2 /* Normal */, null);
  /**
  Define a [highlighter](#highlight.Highlighter) from an array of
  tag/class pairs. Classes associated with more specific tags will
  take precedence.
  */
  function tagHighlighter(tags, options) {
      let map = Object.create(null);
      for (let style of tags) {
          if (!Array.isArray(style.tag))
              map[style.tag.id] = style.class;
          else
              for (let tag of style.tag)
                  map[tag.id] = style.class;
      }
      let { scope, all = null } = options || {};
      return {
          style: (tags) => {
              let cls = all;
              for (let tag of tags) {
                  for (let sub of tag.set) {
                      let tagClass = map[sub.id];
                      if (tagClass) {
                          cls = cls ? cls + " " + tagClass : tagClass;
                          break;
                      }
                  }
              }
              return cls;
          },
          scope
      };
  }
  function highlightTags(highlighters, tags) {
      let result = null;
      for (let highlighter of highlighters) {
          let value = highlighter.style(tags);
          if (value)
              result = result ? result + " " + value : value;
      }
      return result;
  }
  /**
  Highlight the given [tree](#common.Tree) with the given
  [highlighter](#highlight.Highlighter).
  */
  function highlightTree(tree, highlighter, 
  /**
  Assign styling to a region of the text. Will be called, in order
  of position, for any ranges where more than zero classes apply.
  `classes` is a space separated string of CSS classes.
  */
  putStyle, 
  /**
  The start of the range to highlight.
  */
  from = 0, 
  /**
  The end of the range.
  */
  to = tree.length) {
      let builder = new HighlightBuilder(from, Array.isArray(highlighter) ? highlighter : [highlighter], putStyle);
      builder.highlightRange(tree.cursor(), from, to, "", builder.highlighters);
      builder.flush(to);
  }
  class HighlightBuilder {
      constructor(at, highlighters, span) {
          this.at = at;
          this.highlighters = highlighters;
          this.span = span;
          this.class = "";
      }
      startSpan(at, cls) {
          if (cls != this.class) {
              this.flush(at);
              if (at > this.at)
                  this.at = at;
              this.class = cls;
          }
      }
      flush(to) {
          if (to > this.at && this.class)
              this.span(this.at, to, this.class);
      }
      highlightRange(cursor, from, to, inheritedClass, highlighters) {
          let { type, from: start, to: end } = cursor;
          if (start >= to || end <= from)
              return;
          if (type.isTop)
              highlighters = this.highlighters.filter(h => !h.scope || h.scope(type));
          let cls = inheritedClass;
          let rule = getStyleTags(cursor) || Rule.empty;
          let tagCls = highlightTags(highlighters, rule.tags);
          if (tagCls) {
              if (cls)
                  cls += " ";
              cls += tagCls;
              if (rule.mode == 1 /* Inherit */)
                  inheritedClass += (inheritedClass ? " " : "") + tagCls;
          }
          this.startSpan(cursor.from, cls);
          if (rule.opaque)
              return;
          let mounted = cursor.tree && cursor.tree.prop(NodeProp.mounted);
          if (mounted && mounted.overlay) {
              let inner = cursor.node.enter(mounted.overlay[0].from + start, 1);
              let innerHighlighters = this.highlighters.filter(h => !h.scope || h.scope(mounted.tree.type));
              let hasChild = cursor.firstChild();
              for (let i = 0, pos = start;; i++) {
                  let next = i < mounted.overlay.length ? mounted.overlay[i] : null;
                  let nextPos = next ? next.from + start : end;
                  let rangeFrom = Math.max(from, pos), rangeTo = Math.min(to, nextPos);
                  if (rangeFrom < rangeTo && hasChild) {
                      while (cursor.from < rangeTo) {
                          this.highlightRange(cursor, rangeFrom, rangeTo, inheritedClass, highlighters);
                          this.startSpan(Math.min(rangeTo, cursor.to), cls);
                          if (cursor.to >= nextPos || !cursor.nextSibling())
                              break;
                      }
                  }
                  if (!next || nextPos > to)
                      break;
                  pos = next.to + start;
                  if (pos > from) {
                      this.highlightRange(inner.cursor(), Math.max(from, next.from + start), Math.min(to, pos), inheritedClass, innerHighlighters);
                      this.startSpan(pos, cls);
                  }
              }
              if (hasChild)
                  cursor.parent();
          }
          else if (cursor.firstChild()) {
              do {
                  if (cursor.to <= from)
                      continue;
                  if (cursor.from >= to)
                      break;
                  this.highlightRange(cursor, from, to, inheritedClass, highlighters);
                  this.startSpan(Math.min(to, cursor.to), cls);
              } while (cursor.nextSibling());
              cursor.parent();
          }
      }
  }
  /**
  Match a syntax node's [highlight rules](#highlight.styleTags). If
  there's a match, return its set of tags, and whether it is
  opaque (uses a `!`) or applies to all child nodes (`/...`).
  */
  function getStyleTags(node) {
      let rule = node.type.prop(ruleNodeProp);
      while (rule && rule.context && !node.matchContext(rule.context))
          rule = rule.next;
      return rule || null;
  }
  const t$2 = Tag.define;
  const comment = t$2(), name = t$2(), typeName = t$2(name), propertyName = t$2(name), literal = t$2(), string = t$2(literal), number = t$2(literal), content = t$2(), heading = t$2(content), keyword = t$2(), operator = t$2(), punctuation = t$2(), bracket = t$2(punctuation), meta = t$2();
  /**
  The default set of highlighting [tags](#highlight.Tag).

  This collection is heavily biased towards programming languages,
  and necessarily incomplete. A full ontology of syntactic
  constructs would fill a stack of books, and be impractical to
  write themes for. So try to make do with this set. If all else
  fails, [open an
  issue](https://github.com/codemirror/codemirror.next) to propose a
  new tag, or [define](#highlight.Tag^define) a local custom tag for
  your use case.

  Note that it is not obligatory to always attach the most specific
  tag possible to an element—if your grammar can't easily
  distinguish a certain type of element (such as a local variable),
  it is okay to style it as its more general variant (a variable).

  For tags that extend some parent tag, the documentation links to
  the parent.
  */
  const tags = {
      /**
      A comment.
      */
      comment,
      /**
      A line [comment](#highlight.tags.comment).
      */
      lineComment: t$2(comment),
      /**
      A block [comment](#highlight.tags.comment).
      */
      blockComment: t$2(comment),
      /**
      A documentation [comment](#highlight.tags.comment).
      */
      docComment: t$2(comment),
      /**
      Any kind of identifier.
      */
      name,
      /**
      The [name](#highlight.tags.name) of a variable.
      */
      variableName: t$2(name),
      /**
      A type [name](#highlight.tags.name).
      */
      typeName: typeName,
      /**
      A tag name (subtag of [`typeName`](#highlight.tags.typeName)).
      */
      tagName: t$2(typeName),
      /**
      A property or field [name](#highlight.tags.name).
      */
      propertyName: propertyName,
      /**
      An attribute name (subtag of [`propertyName`](#highlight.tags.propertyName)).
      */
      attributeName: t$2(propertyName),
      /**
      The [name](#highlight.tags.name) of a class.
      */
      className: t$2(name),
      /**
      A label [name](#highlight.tags.name).
      */
      labelName: t$2(name),
      /**
      A namespace [name](#highlight.tags.name).
      */
      namespace: t$2(name),
      /**
      The [name](#highlight.tags.name) of a macro.
      */
      macroName: t$2(name),
      /**
      A literal value.
      */
      literal,
      /**
      A string [literal](#highlight.tags.literal).
      */
      string,
      /**
      A documentation [string](#highlight.tags.string).
      */
      docString: t$2(string),
      /**
      A character literal (subtag of [string](#highlight.tags.string)).
      */
      character: t$2(string),
      /**
      An attribute value (subtag of [string](#highlight.tags.string)).
      */
      attributeValue: t$2(string),
      /**
      A number [literal](#highlight.tags.literal).
      */
      number,
      /**
      An integer [number](#highlight.tags.number) literal.
      */
      integer: t$2(number),
      /**
      A floating-point [number](#highlight.tags.number) literal.
      */
      float: t$2(number),
      /**
      A boolean [literal](#highlight.tags.literal).
      */
      bool: t$2(literal),
      /**
      Regular expression [literal](#highlight.tags.literal).
      */
      regexp: t$2(literal),
      /**
      An escape [literal](#highlight.tags.literal), for example a
      backslash escape in a string.
      */
      escape: t$2(literal),
      /**
      A color [literal](#highlight.tags.literal).
      */
      color: t$2(literal),
      /**
      A URL [literal](#highlight.tags.literal).
      */
      url: t$2(literal),
      /**
      A language keyword.
      */
      keyword,
      /**
      The [keyword](#highlight.tags.keyword) for the self or this
      object.
      */
      self: t$2(keyword),
      /**
      The [keyword](#highlight.tags.keyword) for null.
      */
      null: t$2(keyword),
      /**
      A [keyword](#highlight.tags.keyword) denoting some atomic value.
      */
      atom: t$2(keyword),
      /**
      A [keyword](#highlight.tags.keyword) that represents a unit.
      */
      unit: t$2(keyword),
      /**
      A modifier [keyword](#highlight.tags.keyword).
      */
      modifier: t$2(keyword),
      /**
      A [keyword](#highlight.tags.keyword) that acts as an operator.
      */
      operatorKeyword: t$2(keyword),
      /**
      A control-flow related [keyword](#highlight.tags.keyword).
      */
      controlKeyword: t$2(keyword),
      /**
      A [keyword](#highlight.tags.keyword) that defines something.
      */
      definitionKeyword: t$2(keyword),
      /**
      A [keyword](#highlight.tags.keyword) related to defining or
      interfacing with modules.
      */
      moduleKeyword: t$2(keyword),
      /**
      An operator.
      */
      operator,
      /**
      An [operator](#highlight.tags.operator) that dereferences something.
      */
      derefOperator: t$2(operator),
      /**
      Arithmetic-related [operator](#highlight.tags.operator).
      */
      arithmeticOperator: t$2(operator),
      /**
      Logical [operator](#highlight.tags.operator).
      */
      logicOperator: t$2(operator),
      /**
      Bit [operator](#highlight.tags.operator).
      */
      bitwiseOperator: t$2(operator),
      /**
      Comparison [operator](#highlight.tags.operator).
      */
      compareOperator: t$2(operator),
      /**
      [Operator](#highlight.tags.operator) that updates its operand.
      */
      updateOperator: t$2(operator),
      /**
      [Operator](#highlight.tags.operator) that defines something.
      */
      definitionOperator: t$2(operator),
      /**
      Type-related [operator](#highlight.tags.operator).
      */
      typeOperator: t$2(operator),
      /**
      Control-flow [operator](#highlight.tags.operator).
      */
      controlOperator: t$2(operator),
      /**
      Program or markup punctuation.
      */
      punctuation,
      /**
      [Punctuation](#highlight.tags.punctuation) that separates
      things.
      */
      separator: t$2(punctuation),
      /**
      Bracket-style [punctuation](#highlight.tags.punctuation).
      */
      bracket,
      /**
      Angle [brackets](#highlight.tags.bracket) (usually `<` and `>`
      tokens).
      */
      angleBracket: t$2(bracket),
      /**
      Square [brackets](#highlight.tags.bracket) (usually `[` and `]`
      tokens).
      */
      squareBracket: t$2(bracket),
      /**
      Parentheses (usually `(` and `)` tokens). Subtag of
      [bracket](#highlight.tags.bracket).
      */
      paren: t$2(bracket),
      /**
      Braces (usually `{` and `}` tokens). Subtag of
      [bracket](#highlight.tags.bracket).
      */
      brace: t$2(bracket),
      /**
      Content, for example plain text in XML or markup documents.
      */
      content,
      /**
      [Content](#highlight.tags.content) that represents a heading.
      */
      heading,
      /**
      A level 1 [heading](#highlight.tags.heading).
      */
      heading1: t$2(heading),
      /**
      A level 2 [heading](#highlight.tags.heading).
      */
      heading2: t$2(heading),
      /**
      A level 3 [heading](#highlight.tags.heading).
      */
      heading3: t$2(heading),
      /**
      A level 4 [heading](#highlight.tags.heading).
      */
      heading4: t$2(heading),
      /**
      A level 5 [heading](#highlight.tags.heading).
      */
      heading5: t$2(heading),
      /**
      A level 6 [heading](#highlight.tags.heading).
      */
      heading6: t$2(heading),
      /**
      A prose separator (such as a horizontal rule).
      */
      contentSeparator: t$2(content),
      /**
      [Content](#highlight.tags.content) that represents a list.
      */
      list: t$2(content),
      /**
      [Content](#highlight.tags.content) that represents a quote.
      */
      quote: t$2(content),
      /**
      [Content](#highlight.tags.content) that is emphasized.
      */
      emphasis: t$2(content),
      /**
      [Content](#highlight.tags.content) that is styled strong.
      */
      strong: t$2(content),
      /**
      [Content](#highlight.tags.content) that is part of a link.
      */
      link: t$2(content),
      /**
      [Content](#highlight.tags.content) that is styled as code or
      monospace.
      */
      monospace: t$2(content),
      /**
      [Content](#highlight.tags.content) that has a strike-through
      style.
      */
      strikethrough: t$2(content),
      /**
      Inserted text in a change-tracking format.
      */
      inserted: t$2(),
      /**
      Deleted text.
      */
      deleted: t$2(),
      /**
      Changed text.
      */
      changed: t$2(),
      /**
      An invalid or unsyntactic element.
      */
      invalid: t$2(),
      /**
      Metadata or meta-instruction.
      */
      meta,
      /**
      [Metadata](#highlight.tags.meta) that applies to the entire
      document.
      */
      documentMeta: t$2(meta),
      /**
      [Metadata](#highlight.tags.meta) that annotates or adds
      attributes to a given syntactic element.
      */
      annotation: t$2(meta),
      /**
      Processing instruction or preprocessor directive. Subtag of
      [meta](#highlight.tags.meta).
      */
      processingInstruction: t$2(meta),
      /**
      [Modifier](#highlight.Tag^defineModifier) that indicates that a
      given element is being defined. Expected to be used with the
      various [name](#highlight.tags.name) tags.
      */
      definition: Tag.defineModifier(),
      /**
      [Modifier](#highlight.Tag^defineModifier) that indicates that
      something is constant. Mostly expected to be used with
      [variable names](#highlight.tags.variableName).
      */
      constant: Tag.defineModifier(),
      /**
      [Modifier](#highlight.Tag^defineModifier) used to indicate that
      a [variable](#highlight.tags.variableName) or [property
      name](#highlight.tags.propertyName) is being called or defined
      as a function.
      */
      function: Tag.defineModifier(),
      /**
      [Modifier](#highlight.Tag^defineModifier) that can be applied to
      [names](#highlight.tags.name) to indicate that they belong to
      the language's standard environment.
      */
      standard: Tag.defineModifier(),
      /**
      [Modifier](#highlight.Tag^defineModifier) that indicates a given
      [names](#highlight.tags.name) is local to some scope.
      */
      local: Tag.defineModifier(),
      /**
      A generic variant [modifier](#highlight.Tag^defineModifier) that
      can be used to tag language-specific alternative variants of
      some common tag. It is recommended for themes to define special
      forms of at least the [string](#highlight.tags.string) and
      [variable name](#highlight.tags.variableName) tags, since those
      come up a lot.
      */
      special: Tag.defineModifier()
  };
  /**
  This is a highlighter that adds stable, predictable classes to
  tokens, for styling with external CSS.

  The following tags are mapped to their name prefixed with `"tok-"`
  (for example `"tok-comment"`):

  * [`link`](#highlight.tags.link)
  * [`heading`](#highlight.tags.heading)
  * [`emphasis`](#highlight.tags.emphasis)
  * [`strong`](#highlight.tags.strong)
  * [`keyword`](#highlight.tags.keyword)
  * [`atom`](#highlight.tags.atom)
  * [`bool`](#highlight.tags.bool)
  * [`url`](#highlight.tags.url)
  * [`labelName`](#highlight.tags.labelName)
  * [`inserted`](#highlight.tags.inserted)
  * [`deleted`](#highlight.tags.deleted)
  * [`literal`](#highlight.tags.literal)
  * [`string`](#highlight.tags.string)
  * [`number`](#highlight.tags.number)
  * [`variableName`](#highlight.tags.variableName)
  * [`typeName`](#highlight.tags.typeName)
  * [`namespace`](#highlight.tags.namespace)
  * [`className`](#highlight.tags.className)
  * [`macroName`](#highlight.tags.macroName)
  * [`propertyName`](#highlight.tags.propertyName)
  * [`operator`](#highlight.tags.operator)
  * [`comment`](#highlight.tags.comment)
  * [`meta`](#highlight.tags.meta)
  * [`punctuation`](#highlight.tags.punctuation)
  * [`invalid`](#highlight.tags.invalid)

  In addition, these mappings are provided:

  * [`regexp`](#highlight.tags.regexp),
    [`escape`](#highlight.tags.escape), and
    [`special`](#highlight.tags.special)[`(string)`](#highlight.tags.string)
    are mapped to `"tok-string2"`
  * [`special`](#highlight.tags.special)[`(variableName)`](#highlight.tags.variableName)
    to `"tok-variableName2"`
  * [`local`](#highlight.tags.local)[`(variableName)`](#highlight.tags.variableName)
    to `"tok-variableName tok-local"`
  * [`definition`](#highlight.tags.definition)[`(variableName)`](#highlight.tags.variableName)
    to `"tok-variableName tok-definition"`
  * [`definition`](#highlight.tags.definition)[`(propertyName)`](#highlight.tags.propertyName)
    to `"tok-propertyName tok-definition"`
  */
  tagHighlighter([
      { tag: tags.link, class: "tok-link" },
      { tag: tags.heading, class: "tok-heading" },
      { tag: tags.emphasis, class: "tok-emphasis" },
      { tag: tags.strong, class: "tok-strong" },
      { tag: tags.keyword, class: "tok-keyword" },
      { tag: tags.atom, class: "tok-atom" },
      { tag: tags.bool, class: "tok-bool" },
      { tag: tags.url, class: "tok-url" },
      { tag: tags.labelName, class: "tok-labelName" },
      { tag: tags.inserted, class: "tok-inserted" },
      { tag: tags.deleted, class: "tok-deleted" },
      { tag: tags.literal, class: "tok-literal" },
      { tag: tags.string, class: "tok-string" },
      { tag: tags.number, class: "tok-number" },
      { tag: [tags.regexp, tags.escape, tags.special(tags.string)], class: "tok-string2" },
      { tag: tags.variableName, class: "tok-variableName" },
      { tag: tags.local(tags.variableName), class: "tok-variableName tok-local" },
      { tag: tags.definition(tags.variableName), class: "tok-variableName tok-definition" },
      { tag: tags.special(tags.variableName), class: "tok-variableName2" },
      { tag: tags.definition(tags.propertyName), class: "tok-propertyName tok-definition" },
      { tag: tags.typeName, class: "tok-typeName" },
      { tag: tags.namespace, class: "tok-namespace" },
      { tag: tags.className, class: "tok-className" },
      { tag: tags.macroName, class: "tok-macroName" },
      { tag: tags.propertyName, class: "tok-propertyName" },
      { tag: tags.operator, class: "tok-operator" },
      { tag: tags.comment, class: "tok-comment" },
      { tag: tags.meta, class: "tok-meta" },
      { tag: tags.invalid, class: "tok-invalid" },
      { tag: tags.punctuation, class: "tok-punctuation" }
  ]);

  // This file was generated by lezer-generator. You probably shouldn't edit it.
  const propertyIdentifier$1 = 147,
    identifier$1 = 148,
    nameIdentifier$1 = 149,
    insertSemi$1 = 150,
    expression0$1 = 154,
    ForExpression$1 = 4,
    forExpressionStart$1 = 157,
    ForInExpression$1 = 7,
    Name$1 = 8,
    Identifier$1 = 9,
    AdditionalIdentifier$1 = 10,
    forExpressionBodyStart$1 = 165,
    IfExpression$1 = 18,
    ifExpressionStart$1 = 166,
    QuantifiedExpression$1 = 22,
    quantifiedExpressionStart$1 = 167,
    QuantifiedInExpression$1 = 26,
    PositiveUnaryTest$1 = 36,
    ArithmeticExpression$1 = 40,
    arithmeticPlusStart$1 = 171,
    arithmeticTimesStart$1 = 172,
    arithmeticExpStart$1 = 173,
    arithmeticUnaryStart$1 = 174,
    VariableName$1 = 47,
    PathExpression$1 = 67,
    pathExpressionStart$1 = 179,
    FilterExpression$1 = 69,
    filterExpressionStart$1 = 180,
    FunctionInvocation$1 = 71,
    functionInvocationStart$1 = 181,
    ParameterName$1 = 103,
    nil$1 = 186,
    NumericLiteral$1 = 106,
    StringLiteral$1 = 107,
    BooleanLiteral$1 = 108,
    FunctionDefinition$1 = 117,
    functionDefinitionStart$1 = 194,
    Context$1 = 124,
    contextStart$1 = 196,
    ContextEntry$1 = 125,
    PropertyName$1 = 127,
    PropertyIdentifier$1 = 128;

  const LOG_PARSE$1 = typeof process != 'undefined' && process.env && /\bfparse(:dbg)?\b/.test(process.env.LOG);
  const LOG_PARSE_DEBUG$1 = typeof process != 'undefined' && process.env && /\fparse:dbg\b/.test(process.env.LOG);
  const LOG_VARS$1 = typeof process != 'undefined' && process.env && /\bcontext?\b/.test(process.env.LOG);

  const spaceChars$1 = [
    9, 11, 12, 32, 133, 160,
    5760, 8192, 8193, 8194, 8195, 8196, 8197, 8198,
    8199, 8200, 8201, 8202, 8232, 8233, 8239, 8287, 12288
  ];

  const newlineChars$1 = chars$3('\n\r');

  const additionalNameChars$1 = chars$3("'./-+*");

  /**
   * @param { string } str
   * @return { number[] }
   */
  function chars$3(str) {
    return Array.from(str).map(s => s.charCodeAt(0));
  }

  /**
   * @param { number } ch
   * @return { boolean }
   */
  function isStartChar$1(ch) {
    return (
      ch === 63 // ?
    ) || (
      ch === 95 // _
    ) || (
      ch >= 65 && ch <= 90 // A-Z
    ) || (
      ch >= 97 && ch <= 122 // a-z
    ) || (
      ch >= 161 && !isPartChar$1(ch) && !isSpace$1(ch)
    );
  }

  /**
   * @param { number } ch
   * @return { boolean }
   */
  function isAdditional$1(ch) {
    return additionalNameChars$1.includes(ch);
  }

  /**
   * @param { number } ch
   * @return { boolean }
   */
  function isPartChar$1(ch) {
    return (
      ch >= 48 && ch <= 57 // 0-9
    ) || (
      ch === 0xB7
    ) || (
      ch >= 0x0300 && ch <= 0x036F
    ) || (
      ch >= 0x203F && ch <= 0x2040
    );
  }

  /**
   * @param { number } ch
   * @return { boolean }
   */
  function isSpace$1(ch) {
    return spaceChars$1.includes(ch);
  }

  // eslint-disable-next-line
  function indent$1(str, spaces) {
    return spaces.concat(
      str.split(/\n/g).join('\n' + spaces)
    );
  }

  /**
   * @param { import('@lezer/lr').InputStream } input
   * @param  { number } [offset]
   * @param { boolean } [includeOperators]
   *
   * @return { { token: string, offset: number } | null }
   */
  function parseAdditionalSymbol$1(input, offset = 0) {

    const next = input.peek(offset);

    if (isAdditional$1(next)) {
      return {
        offset: 1,
        token: String.fromCharCode(next)
      };
    }

    return null;
  }

  /**
   * @param { import('@lezer/lr').InputStream } input
   * @param { number } [offset]
   * @param { boolean } [namePart]
   *
   * @return { { token: string, offset: number } | null }
   */
  function parseIdentifier$1(input, offset = 0, namePart = false) {
    for (let inside = false, chars = [], i = 0;; i++) {
      const next = input.peek(offset + i);

      if (isStartChar$1(next) || ((inside || namePart) && isPartChar$1(next))) {
        if (!inside) {
          inside = true;
        }

        chars.push(next);
      } else {

        if (chars.length) {
          return {
            token: String.fromCharCode(...chars),
            offset: i
          };
        }

        return null;
      }
    }
  }

  /**
   * @param { import('@lezer/lr').InputStream } input
   * @param  { number } offset
   *
   * @return { { token: string, offset: number } | null }
   */
  function parseSpaces$1(input, offset) {

    for (let inside = false, i = 0;; i++) {
      let next = input.peek(offset + i);

      if (isSpace$1(next)) {
        if (!inside) {
          inside = true;
        }
      } else {
        if (inside) {
          return {
            token: ' ',
            offset: i
          };
        }

        return null;
      }
    }
  }

  /**
   * Parse a name from the input and return the first match, if any.
   *
   * @param { import('@lezer/lr').InputStream } input
   * @param { Variables } variables
   *
   * @return { { token: string, offset: number, term: number } | null }
   */
  function parseName$1(input, variables) {
    const contextKeys = variables.contextKeys();

    const start = variables.tokens;

    for (let i = 0, tokens = [], nextMatch = null;;) {

      const namePart = (start.length + tokens.length) > 0;
      const maybeSpace = tokens.length > 0;

      const match = (
        parseIdentifier$1(input, i, namePart) ||
        namePart && parseAdditionalSymbol$1(input, i) ||
        maybeSpace && parseSpaces$1(input, i)
      );

      // match is required
      if (!match) {
        return nextMatch;
      }

      const {
        token,
        offset
      } = match;

      i += offset;

      if (token === ' ') {
        continue;
      }

      tokens = [ ...tokens, token ];

      const name = [ ...start, ...tokens ].join(' ');

      if (contextKeys.some(el => el === name)) {
        const token = tokens[0];

        nextMatch = {
          token,
          offset: token.length,
          term: nameIdentifier$1
        };
      }

      if (dateTimeIdentifiers$1.some(el => el === name)) {
        const token = tokens[0];

        // parse date time identifiers as normal
        // identifiers to allow specialization to kick in
        //
        // cf. https://github.com/nikku/lezer-feel/issues/8
        nextMatch = {
          token,
          offset: token.length,
          term: identifier$1
        };
      }

      if (
        !contextKeys.some(el => el.startsWith(name)) &&
        !dateTimeIdentifiers$1.some(el => el.startsWith(name))
      ) {
        return nextMatch;
      }
    }

  }

  const identifiersMap$1 = {
    [ identifier$1 ]: 'identifier',
    [ nameIdentifier$1 ]: 'nameIdentifier'
  };

  const identifiers$1 = new ExternalTokenizer((input, stack) => {

    LOG_PARSE_DEBUG$1 && console.log('%s: T <identifier | nameIdentifier>', input.pos);

    const nameMatch = parseName$1(input, stack.context);

    const start = stack.context.tokens;

    const match = nameMatch || parseIdentifier$1(input, 0, start.length > 0);

    if (match) {
      input.advance(match.offset);
      input.acceptToken(nameMatch ? nameMatch.term : identifier$1);

      LOG_PARSE$1 && console.log('%s: MATCH <%s> <%s>', input.pos, nameMatch ? identifiersMap$1[nameMatch.term] : 'identifier', match.token);
    }
  }, { contextual: true });


  const propertyIdentifiers$1 = new ExternalTokenizer((input, stack) => {

    LOG_PARSE_DEBUG$1 && console.log('%s: T <propertyIdentifier>', input.pos);

    const start = stack.context.tokens;

    const match = parseIdentifier$1(input, 0, start.length > 0);

    if (match) {
      input.advance(match.offset);
      input.acceptToken(propertyIdentifier$1);

      LOG_PARSE$1 && console.log('%s: MATCH <propertyIdentifier> <%s>', input.pos, match.token);
    }
  });


  const insertSemicolon$1 = new ExternalTokenizer((input, stack) => {

    LOG_PARSE_DEBUG$1 && console.log('%s: T <insertSemi>', input.pos);

    let offset;
    let insert = false;

    for (offset = 0;; offset++) {
      const char = input.peek(offset);

      if (spaceChars$1.includes(char)) {
        continue;
      }

      if (newlineChars$1.includes(char)) {
        insert = true;
      }

      break;
    }

    if (insert) {

      const identifier = parseIdentifier$1(input, offset + 1);
      const spaces = parseSpaces$1(input, offset + 1);

      if (spaces || identifier && /^(then|else|return|satisfies)$/.test(identifier.token)) {
        return;
      }

      LOG_PARSE$1 && console.log('%s: MATCH <insertSemi>', input.pos);
      input.acceptToken(insertSemi$1);
    }
  });

  const prefixedContextStarts$1 = {
    [ functionInvocationStart$1 ]: 'FunctionInvocation',
    [ filterExpressionStart$1 ]: 'FilterExpression',
    [ pathExpressionStart$1 ]: 'PathExpression'
  };

  const contextStarts$1 = {
    [ contextStart$1 ]: 'Context',
    [ functionDefinitionStart$1 ]: 'FunctionDefinition',
    [ forExpressionStart$1 ]: 'ForExpression',
    [ ifExpressionStart$1 ]: 'IfExpression',
    [ quantifiedExpressionStart$1 ]: 'QuantifiedExpression'
  };

  const contextEnds$1 = {
    [ Context$1 ]: 'Context',
    [ FunctionDefinition$1 ]: 'FunctionDefinition',
    [ ForExpression$1 ]: 'ForExpression',
    [ IfExpression$1 ]: 'IfExpression',
    [ QuantifiedExpression$1 ]: 'QuantifiedExpression',
    [ PathExpression$1 ]: 'PathExpression',
    [ FunctionInvocation$1 ]: 'FunctionInvocation',
    [ FilterExpression$1 ]: 'FilterExpression',
    [ ArithmeticExpression$1 ]: 'ArithmeticExpression'
  };

  let ValueProducer$1 = class ValueProducer {

    /**
     * @param { Function } fn
     */
    constructor(fn) {
      this.fn = fn;
    }

    get(variables) {
      return this.fn(variables);
    }

    /**
     * @param { Function }
     *
     * @return { ValueProducer }
     */
    static of(fn) {
      return new ValueProducer$1(fn);
    }

  };

  const dateTimeLiterals$1 = {
    'date and time': 1,
    'date': 1,
    'time': 1,
    'duration': 1
  };

  const dateTimeIdentifiers$1 = Object.keys(dateTimeLiterals$1);

  let Variables$1 = class Variables {

    constructor({
      name = 'Expressions',
      tokens = [],
      children = [],
      parent = null,
      context = { },
      value,
      raw
    } = {}) {
      this.name = name;
      this.tokens = tokens;
      this.children = children;
      this.parent = parent;
      this.context = context;
      this.value = value;
      this.raw = raw;
    }

    enterScope(name) {

      const childScope = this.of({
        name,
        parent: this
      });

      LOG_VARS$1 && console.log('[%s] enter', childScope.path, childScope.context);

      return childScope;
    }

    exitScope(str) {

      if (!this.parent) {
        LOG_VARS$1 && console.log('[%s] NO exit %o\n%s', this.path, this.context, indent$1(str, '  '));

        return this;
      }

      LOG_VARS$1 && console.log('[%s] exit %o\n%s', this.path, this.context, indent$1(str, '  '));

      return this.parent.pushChild(this);
    }

    token(part) {

      LOG_VARS$1 && console.log('[%s] token <%s> + <%s>', this.path, this.tokens.join(' '), part);

      return this.assign({
        tokens: [ ...this.tokens, part ]
      });
    }

    literal(value) {

      LOG_VARS$1 && console.log('[%s] literal %o', this.path, value);

      return this.pushChild(this.of({
        name: 'Literal',
        value
      }));
    }

    /**
     * Return computed scope value
     *
     * @return {any}
     */
    computedValue() {
      for (let scope = this;;scope = scope.children.slice(-1)[0]) {

        if (!scope) {
          return null;
        }

        if (scope.value) {
          return scope.value;
        }
      }
    }

    contextKeys() {
      return Object.keys(this.context).map(normalizeContextKey$1);
    }

    get path() {
      return this.parent?.path?.concat(' > ', this.name) || this.name;
    }

    /**
     * Return value of variable.
     *
     * @param { string } variable
     * @return { any } value
     */
    get(variable) {

      const names = [ variable, variable && normalizeContextKey$1(variable) ];

      const contextKey = Object.keys(this.context).find(
        key => names.includes(normalizeContextKey$1(key))
      );

      if (typeof contextKey === 'undefined') {
        return undefined;
      }

      const val = this.context[contextKey];

      if (val instanceof ValueProducer$1) {
        return val.get(this);
      } else {
        return val;
      }
    }

    resolveName() {

      const variable = this.tokens.join(' ');
      const tokens = [];

      const parentScope = this.assign({
        tokens
      });

      const variableScope = this.of({
        name: 'VariableName',
        parent: parentScope,
        value: this.get(variable),
        raw: variable
      });

      LOG_VARS$1 && console.log('[%s] resolve name <%s=%s>', variableScope.path, variable, this.get(variable));

      return parentScope.pushChild(variableScope);
    }

    pushChild(child) {

      if (!child) {
        return this;
      }

      const parent = this.assign({
        children: [ ...this.children, child ]
      });

      child.parent = parent;

      return parent;
    }

    pushChildren(children) {

      let parent = this;

      for (const child of children) {
        parent = parent.pushChild(child);
      }

      return parent;
    }

    declareName() {

      if (this.tokens.length === 0) {
        throw Error('no tokens to declare name');
      }

      const variableName = this.tokens.join(' ');

      LOG_VARS$1 && console.log('[%s] declareName <%s>', this.path, variableName);

      return this.assign({
        tokens: []
      }).pushChild(
        this.of({
          name: 'Name',
          value: variableName
        })
      );
    }

    define(name, value) {

      if (typeof name !== 'string') {
        LOG_VARS$1 && console.log('[%s] no define <%s=%s>', this.path, name, value);

        return this;
      }

      LOG_VARS$1 && console.log('[%s] define <%s=%s>', this.path, name, value);

      const context = {
        ...this.context,
        [name]: value
      };

      return this.assign({
        context
      });
    }

    /**
     * @param { Record<string, any> } [options]
     *
     * @return { Variables }
     */
    assign(options = {}) {

      return Variables$1.of({
        ...this,
        ...options
      });
    }

    /**
     * @param { Record<string, any> } [options]
     *
     * @return { Variables }
     */
    of(options = {}) {

      const defaultOptions = {
        context: this.context,
        parent: this.parent
      };

      return Variables$1.of({
        ...defaultOptions,
        ...options
      });
    }

    static of(options) {
      const {
        name,
        tokens = [],
        children = [],
        parent = null,
        context = {},
        value,
        raw
      } = options;

      return new Variables$1({
        name,
        tokens: [ ...tokens ],
        children: [ ...children ],
        context: {
          ...context
        },
        parent,
        value,
        raw
      });
    }

  };

  /**
   * @param { string } name
   *
   * @return { string } normalizedName
   */
  function normalizeContextKey$1(name) {
    return name.replace(/\s*([./\-'+*])\s*/g, ' $1 ').replace(/\s{2,}/g, ' ').trim();
  }

  /**
   * Wrap children of variables under the given named child.
   *
   * @param { Variables } variables
   * @param { string } name
   * @param { string } code
   * @return { Variables }
   */
  function wrap$1(variables, scopeName, code) {

    const parts = variables.children.filter(c => c.name !== scopeName);
    const children = variables.children.filter(c => c.name === scopeName);

    const namePart = parts[0];
    const valuePart = parts[Math.max(1, parts.length - 1)];

    const name = namePart.computedValue();
    const value = valuePart?.computedValue() || null;

    return variables
      .assign({
        children
      })
      .enterScope(scopeName)
      .pushChildren(parts)
      .exitScope(code)
      .define(name, value);
  }

  /**
   * @param { any } context
   *
   * @return { ContextTracker<Variables> }
   */
  function trackVariables$1(context = {}) {

    const start = Variables$1.of({
      context
    });

    return new ContextTracker({
      start,
      reduce(variables, term, stack, input) {

        if (term === Context$1) {
          variables = variables.assign({
            value: variables.context
          });
        }

        if (term === IfExpression$1) {
          const [ thenPart, elsePart ] = variables.children.slice(-2);

          variables = variables.assign({
            value: {
              ...thenPart?.computedValue(),
              ...elsePart?.computedValue()
            }
          });
        }

        if (term === FilterExpression$1) {
          const [ sourcePart, _ ] = variables.children.slice(-2);

          variables = variables.assign({
            value: sourcePart?.computedValue()
          });
        }

        if (term === FunctionInvocation$1) {

          const [
            name,
            ...args
          ] = variables.children;

          // preserve type information through `get value(context, key)` utility
          if (name?.raw === 'get value') {
            variables = getContextValue$1(variables, args);
          }
        }

        const start = contextStarts$1[term];

        if (start) {
          return variables.enterScope(start);
        }

        const prefixedStart = prefixedContextStarts$1[term];

        // pull <expression> into new <prefixedStart> context
        if (prefixedStart) {

          const children = variables.children.slice(0, -1);
          const lastChild = variables.children.slice(-1)[0];

          return variables.assign({
            children
          }).enterScope(prefixedStart).pushChild(lastChild).assign({
            context: {
              ...variables.context,
              ...lastChild?.computedValue()
            }
          });
        }

        const code = input.read(input.pos, stack.pos);

        const end = contextEnds$1[term];

        if (end) {
          return variables.exitScope(code);
        }

        if (term === ContextEntry$1) {
          return wrap$1(variables, 'ContextEntry', code);
        }

        if (
          term === ForInExpression$1 ||
          term === QuantifiedInExpression$1
        ) {
          return wrap$1(variables, 'InExpression', code);
        }

        // define <partial> within ForExpression body
        if (term === forExpressionBodyStart$1) {

          return variables.define(
            'partial',
            ValueProducer$1.of(variables => {
              return variables.children[variables.children.length - 1]?.computedValue();
            })
          );
        }

        if (
          term === ParameterName$1
        ) {
          const [ left ] = variables.children.slice(-1);

          const name = left.computedValue();

          // TODO: attach type information
          return variables.define(name, 1);
        }

        // pull <expression> into ArithmeticExpression child
        if (
          term === arithmeticPlusStart$1 ||
          term === arithmeticTimesStart$1 ||
          term === arithmeticExpStart$1
        ) {
          const children = variables.children.slice(0, -1);
          const lastChild = variables.children.slice(-1)[0];

          return variables.assign({
            children
          }).enterScope('ArithmeticExpression').pushChild(lastChild);
        }

        if (term === arithmeticUnaryStart$1) {
          return variables.enterScope('ArithmeticExpression');
        }

        if (
          term === Identifier$1 ||
          term === AdditionalIdentifier$1 ||
          term === PropertyIdentifier$1
        ) {
          return variables.token(code);
        }

        if (
          term === StringLiteral$1
        ) {
          return variables.literal(code.replace(/^"|"$/g, ''));
        }

        if (term === BooleanLiteral$1) {
          return variables.literal(code === 'true' ? true : false);
        }

        if (term === NumericLiteral$1) {
          return variables.literal(parseFloat(code));
        }

        if (term === nil$1) {
          return variables.literal(null);
        }

        if (
          term === VariableName$1
        ) {
          return variables.resolveName();
        }

        if (
          term === Name$1 ||
          term === PropertyName$1
        ) {
          return variables.declareName();
        }

        if (
          term === expression0$1 ||
          term === PositiveUnaryTest$1
        ) {
          if (variables.tokens.length > 0) {
            throw new Error('uncleared name');
          }
        }

        if (term === expression0$1) {

          let parent = variables;

          while (parent.parent) {
            parent = parent.exitScope(code);
          }

          return parent;
        }

        return variables;
      }
    });
  }

  const variableTracker$1 = trackVariables$1({});


  // helpers //////////////

  function getContextValue$1(variables, args) {

    if (!args.length) {
      return variables.assign({
        value: null
      });
    }

    if (args[0].name === 'Name') {
      args = extractNamedArgs$1(args, [ 'm', 'key' ]);
    }

    if (args.length !== 2) {
      return variables.assign({
        value: null
      });
    }

    const [
      context,
      key
    ] = args;

    const keyValue = key?.computedValue();
    const contextValue = context?.computedValue();

    if (
      (!contextValue || typeof contextValue !== 'object') || typeof keyValue !== 'string'
    ) {
      return variables.assign({
        value: null
      });
    }

    return variables.assign({
      value: [ normalizeContextKey$1(keyValue), keyValue ].reduce((value, keyValue) => {
        if (keyValue in contextValue) {
          return contextValue[keyValue];
        }

        return value;
      }, null)
    });
  }

  function extractNamedArgs$1(args, argNames) {

    const context = {};

    for (let i = 0; i < args.length; i += 2) {
      const [ name, value ] = args.slice(i, i + 2);

      context[name.value] = value;
    }

    return argNames.map(name => context[name]);
  }

  const feelHighlighting$1 = styleTags({
    'StringLiteral': tags.string,
    'NumericLiteral': tags.number,
    'BooleanLiteral': tags.bool,
    'Name QualifiedName': tags.name,
    'CompareOp': tags.compareOperator,
    'ArithOp': tags.arithmeticOperator,
    'PropertyName PathExpression/Name Key': tags.propertyName,
    'for if then else some every satisfies between': tags.controlKeyword,
    'in return instance of and or': tags.operatorKeyword,
    'function': tags.definitionKeyword,
    'FormalParameter/Type!': tags.typeName,
    'as': tags.keyword,
    'Wildcard': tags.special,
    'null': tags.null,
    ',': tags.separator,
    '[ ]': tags.squareBracket,
    '{ }': tags.brace,
    '( )': tags.paren,
    'LineComment': tags.lineComment,
    'BlockComment': tags.blockComment,
    'ParameterName VariableName ?': tags.variableName,
    'DateTimeConstructor! SpecialFunctionName BuiltInFunctionName': tags.function(tags.special(tags.variableName)),
    'FunctionInvocation/VariableName': tags.function(tags.variableName),
    'List Interval': tags.list,
    'BuiltInType ListType ContextType FunctionType': tags.function(tags.typeName),
    'Context': tags.definition(tags.literal),
    'ContextEntry/Key': tags.variableName,
    'InExpression/Name': tags.local(tags.variableName),
    'ParameterName/Name': tags.local(tags.variableName),
    'IterationContext/".." Interval/".." "."': tags.punctuation
  });

  // This file was generated by lezer-generator. You probably shouldn't edit it.
  const spec_identifier$1 = {__proto__:null,for:10, in:30, return:34, if:38, then:40, else:42, some:46, every:48, satisfies:55, or:58, and:62, between:70, instance:86, of:89, days:99, time:101, duration:103, years:105, months:107, date:109, list:115, context:121, function:128, string:147, length:149, upper:151, case:153, lower:155, substring:157, before:159, after:161, starts:163, with:165, ends:167, contains:169, insert:171, index:173, distinct:175, values:177, met:179, by:181, overlaps:183, finished:185, started:187, day:189, year:191, week:193, month:195, get:197, value:199, entries:201, null:210, true:380, false:380, "?":224, external:240, not:263};
  const parser$2 = LRParser.deserialize({
    version: 14,
    states: "!&nO`QYOOO&wQYOOOOQU'#Ce'#CeO'RQYO'#C`O([Q^O'#FlOOQQ'#GQ'#GQO*|QYO'#GQO`QYO'#DUOOQU'#FZ'#FZO-rQ^O'#D]OOQO'#GX'#GXO1yQWO'#DuOOQU'#Ej'#EjOOQU'#Ek'#EkOOQU'#El'#ElO2OOWO'#EoO1yQWO'#EmOOQU'#Em'#EmOOQU'#G_'#G_OOQU'#G]'#G]O2TQYO'#ErO`QYO'#EsO2uQYO'#EtO2TQYO'#EqOOQU'#Eq'#EqOOQU'#Fn'#FnO4ZQ^O'#FnO6uQWO'#EuOOQP'#Gh'#GhO6zQXO'#E|OOQU'#Ge'#GeOOQU'#Fm'#FmOOQQ'#FU'#FUQ`QYOOOOQQ'#Fo'#FoOOQQ'#Fx'#FxO`QYO'#CnOOQQ'#Fy'#FyO'RQYO'#CrO7VQYO'#DvO7[QYO'#DvO7aQYO'#DvO7fQYO'#DvO7nQYO'#DvO7sQYO'#DvO7xQYO'#DvO7}QYO'#DvO8SQYO'#DvO8XQYO'#DvO8^QYO'#DvO8cQYO'#DvO8hQYO'#DvOOQU'#G^'#G^O8pQYO'#EnOOQO'#En'#EnOOQO'#Gf'#GfO:SQYO'#DQO:jQWO'#F|OOQO'#DS'#DSO:uQYO'#GQQOQWOOO:|QWOOO;pQYO'#CdO;}QYO'#FqOOQQ'#Cc'#CcO<SQYO'#FpOOQQ'#Cb'#CbO<[QYO,58zO`QYO,59hOOQQ'#F}'#F}OOQQ'#GO'#GOOOQQ'#GP'#GPO`QYO,59pO`QYO,59pO`QYO,59pOOQQ'#GV'#GVO'RQYO,5:]OOQQ'#GW'#GWO`QYO,5:_OOQQ,5<W,5<WO`QYO,59dO`QYO,59fO`QYO,59hO<aQYO,59hO?VQYO,59rOOQU,5;U,5;UO?[Q^O,59pOOQU-E9X-E9XOCcQYO'#GYOOQU,5:a,5:aOOQU,5;Z,5;ZOOQU,5;X,5;XOCjQ^O'#D[OGqQWO'#EjOOQU'#Gd'#GdOGvQWO,5;^OG{QYO,5;_OJZQ^O'#D[OJeQYO'#G]OLsQYO'#G[OMQQWO,5;`OOQU,5;],5;]OOQU,5<Y,5<YOMVQYO,5;aOOQP'#FQ'#FQOMyQXO'#FPOOQO'#FO'#FOONQQWO'#E}ONVQWO'#GiON_QWO,5;hOOQQ-E9S-E9SONdQYO,59YO;}QYO'#F{OOQQ'#Cv'#CvONkQYO'#FzOOQQ'#Cu'#CuONsQYO,59^ONxQYO,5:bOOQO,5:b,5:bON}QYO,5:bO! VQYO,5:bO! [QYO,5;YO`QYO'#FYO! aQWO,5<hO`QYOOOOQR'#Cf'#CfOOQQ'#FV'#FVO!!WQYO,59OO`QYO,5<]OOQQ'#Ft'#FtO'RQYO'#FWO!!hQYO,5<[O`QYO1G.fOOQQ'#Fw'#FwO!!pQ^O1G/SO!&_Q^O1G/[O!)|Q^O1G/[O!1YQ^O1G/[OOQU1G/w1G/wO!1vQYO1G/yO!5]Q^O1G/OO!9RQ^O1G/QO!:aQYO1G/SO`QYO1G/SOOQU1G/S1G/SO!:hQYO1G/^O!;SQ^O'#CdOOQO'#Eg'#EgO!<fQWO'#EfO!<kQWO'#GZOOQO'#Ee'#EeOOQO'#Eh'#EhO!<sQWO,5<tO'RQYO'#F[O!<xQ^O,59vO2TQYO1G0xOOQU1G0y1G0yO`QYO'#F`O!APQWO,5<vOOQU1G0z1G0zO!A[QWO'#EwO!AgQWO'#GgOOQO'#Ev'#EvO!AoQWO1G0{OOQP'#Fb'#FbO!AtQXO,5;kO`QYO,5;iO!A{QXO'#FcO!BTQWO,5=TOOQU1G1S1G1SO`QYO1G.tO`QYO,5<gO'RQYO'#FXO!B]QYO,5<fO`QYO1G.xO!BeQYO1G/|OOQO1G/|1G/|OOQO1G0t1G0tOOQO,5;t,5;tOOQO-E9W-E9WO!BjQWOOOOQQ-E9T-E9TO!BoQYO'#ClOOQQ1G1w1G1wOOQQ,5;r,5;rOOQQ-E9U-E9UO!B|Q^O7+$QOOQU7+%e7+%eO`QYO7+$nO!EnQYO,5;_O!EuQWO7+$nOOQU'#DZ'#DZO!EzQYO'#D^O!FPQYO'#D^O!FUQYO'#D^O!FZQ`O'#DfO!F`Q`O'#DiO!FeQ`O'#DmOOQU7+$x7+$xO`QYO,5;QO'RQYO'#F_O!FjQWO,5<uOOQU1G2`1G2`OOQU,5;v,5;vOOQU-E9Y-E9YO!FrQWO7+&dO!F}QYO,5;zOOQO-E9^-E9^O!:hQYO,5;cO'RQYO'#FaO!G[QWO,5=RO!GdQYO7+&gOOQP-E9`-E9`O!GkQYO1G1TOOQO,5;},5;}OOQO-E9a-E9aO!KdQ^O7+$`O!KkQYO1G2ROOQQ,5;s,5;sOOQQ-E9V-E9VO!KuQ^O7+$dOOQO7+%h7+%hO`QYO,59WO!NgQ^O<<HYOOQU<<HY<<HYO#$UQYO,59xO#$ZQYO,59xO#$`QYO,59xO#$eQYO,5:QO'RQYO,5:TO#%PQbO,5:XO#%WQYO1G0lOOQO,5;y,5;yOOQO-E9]-E9]OOQU<<JO<<JOOOQO1G0}1G0}OOQO,5;{,5;{OOQO-E9_-E9_O#%bQ^O'#EyOOQU<<JR<<JRO`QYO<<JRO`QYO<<GzO#(SQYO1G.rO#(^QYO1G/dOOQU1G/d1G/dO#(cQbO'#D]O#(tQ`O'#D[O#)PQ`O1G/lO#)UQWO'#DlO#)ZQ`O'#GROOQO'#Dk'#DkO#)cQ`O1G/oOOQO'#Dp'#DpO#)hQ`O'#GTOOQO'#Do'#DoO#)pQ`O1G/sOOQUAN?mAN?mO#)uQ^OAN=fOOQU7+%O7+%OO#,gQ`O,59vOOQU7+%W7+%WO#$eQYO,5:WO'RQYO'#F]O#,rQ`O,5<mOOQU7+%Z7+%ZO#$eQYO'#F^O#,zQ`O,5<oO#-SQ`O7+%_OOQO1G/r1G/rOOQO,5;w,5;wOOQO-E9Z-E9ZOOQO,5;x,5;xOOQO-E9[-E9[O!:hQYO<<HyOOQUAN>eAN>eO#-XQ^O'#FnO`QYO,59hO`QYO,59pO`QYO,59pO`QYO,59pO`QYO,59dO`QYO,59fO<aQYO,59hO`QYO1G.fO#-rQYO1G/SO#/`QYO1G/[O#0|QYO1G/[O#3wQYO1G/OO#5lQYO1G/QO'RQYO'#F[O`QYO1G.tO`QYO1G.xO#5|QYO7+$QO`QYO7+$nO#6mQYO7+&gO#8bQYO7+$`O#8iQYO7+$`O#8pQ^O7+$`O#8wQYO7+$dO#9hQYO<<HYO#;UQYO'#EyO`QYO<<JRO`QYO<<GzO#;uQYOAN=fO#$eQYO<<HyO`QYO'#DUO#<fQ^O'#DQO<[QYO,58zO#?WQYO,59YO#?_QYO,59^O#?dQYO1G/SO#?kQWO1G0{O`QYO1G.tO#?pQ`O7+%_O`QYO1G.tO'RQYO'#C`O`QYO'#CnO'RQYO'#CrO`QYO,59hOMVQYO,5;aO#?uQYO,59YO#?|Q`O1G/sO#@RQYO,59YO#@YQWO'#EuO`QYO'#CnO`QYO,59hO`QYO,59pO`QYO,59pO`QYO,59pO`QYO,59dO`QYO,59fO<aQYO,59hO`QYO1G.fO#@_Q^O1G/SO#@fQ^O1G/[O#@mQ^O1G/[O#@tQ^O1G/OO#A[Q^O1G/QO`QYO1G.xO#BpQ^O7+$QO`QYO7+$nO#EeQYO7+&gO#ElQ^O7+$dO#HaQ^O<<HYO#%PQbO,5:XO#HhQ^O'#EyO`QYO<<JRP`QYO<<GzP#K]Q^OAN=fO#L`Q^O'#DQO`QYO'#CnO`QYO'#DUO<[QYO,58zO$ TQYO,59^O$ YQYO1G/SO$ aQWO1G0{O$ fQ`O'#DmO`QYO,59hO`QYO,59pO`QYO,59pO`QYO,59pO`QYO,59dO`QYO,59fO<aQYO,59hO`QYO1G.fO$ kQYO1G/SO$ rQYO1G/[O$ yQYO1G/[O$!QQYO1G/OO$!hQYO1G/QO`QYO1G.xO$#|QYO7+$QO`QYO7+$nO$$pQYO7+&gO$$wQYO7+$dO$%kQYO<<HYO$%rQYO'#EyO`QYO<<JRP`QYO<<GzP$&fQYOAN=fO`QYO'#DUO<[QYO,58zO$'iQYO,59^O$'nQYO1G/SO$'uQWO1G0{O'RQYO'#C`O'RQYO'#CrO`QYO,59hOMVQYO,5;aO$'zQWO'#EuO'RQYO'#C`O'RQYO'#CrO$(PQYO'#DQO`QYO,59hOMVQYO,5;aO$(jQWO'#Eu",
    stateData: "$(o~O$^OS$_OSPOSQOS~OTrOZUO[TOcsOguOhuOrgOueO!S!WO!T!WO!UwO!W!VO!Z|O!b!XO!fdO!hfO!kxO!myO!oyO!pzO!s{O!u{O!w}O!x!OO!y!PO!{!QO!}zO#O!QO#P!QO#Q!RO#S!SO#T!SO#U!TO#]!UO#diO#olO$YQO$ZQO%S[O%T]O%U^O%V_O~OTrO[TOcsOguOhuOrgOueO!S!WO!T!WO!UwO!W!VO!Z|O!b!XO!fdO!hfO!kxO!myO!oyO!pzO!s{O!u{O!w}O!x!OO!y!PO!{!QO!}zO#O!QO#P!QO#Q!RO#S!SO#T!SO#U!TO#]!UO#diO#olO$YQO$ZQO%S[O%T]O%U^O%V_O~OZ!]O#w!_O~P$UO$YQO$ZQO~OZ!gO[!gO]!hO^!hO_!uOm!rOo!sOq!fOr!fOs!tOy!iO{!vO!h!oO$f!mOu${X~O$[!qO%^!qOT$`Xc$`Xg$`Xh$`X!S$`X!T$`X!U$`X!W$`X!Z$`X!b$`X!f$`X!k$`X!m$`X!o$`X!p$`X!s$`X!u$`X!w$`X!x$`X!y$`X!{$`X!}$`X#O$`X#P$`X#Q$`X#S$`X#T$`X#U$`X#]$`X#d$`X#o$`X$W$`X$Y$`X$Z$`X%S$`X%T$`X%U$`X%V$`X~P'ZO%S!wOT$tXZ$tX[$tXc$tXg$tXh$tXr$tXu$tX!S$tX!T$tX!U$tX!W$tX!Z$tX!b$tX!f$tX!h$tX!k$tX!m$tX!o$tX!p$tX!s$tX!u$tX!w$tX!x$tX!y$tX!{$tX!}$tX#O$tX#P$tX#Q$tX#S$tX#T$tX#U$tX#]$tX#d$tX#o$tX$Y$tX$Z$tX%T$tX%U$tX%V$tX~O$YQO$ZQOT!PXZ!PX[!PX]!PX^!PX_!PXc!PXg!PXh!PXm!PXo!PXq!PXr!PXs!PXu!PXy!PX{!PX!S!PX!T!PX!U!PX!W!PX!Z!PX!b!PX!f!PX!h!PX!k!PX!m!PX!o!PX!p!PX!s!PX!u!PX!w!PX!x!PX!y!PX!{!PX!}!PX#O!PX#P!PX#Q!PX#S!PX#T!PX#U!PX#]!PX#d!PX#o!PX$W!PX$[!PX$f!PX%S!PX%T!PX%U!PX%V!PX%^!PX$j!PX$i!PXw!PXd!PXa!PX#n!PXe!PXk!PX~Ou!zO~O%T]O~OZ#PO!S!WO!T!WO!W!VO$YQO$ZQO%S[O%T]O%U^O%V_O~O!f%OP~P`O$[#YOZ$bX[$bX]$bX^$bX_$bXm$bXo$bXq$bXr$bXs$bXu$bXy$bX{$bX!f$bX!h$bX$W$bX$f$bXe$bX~OT$bXc$bXg$bXh$bX!S$bX!T$bX!U$bX!W$bX!Z$bX!b$bX!k$bX!m$bX!o$bX!p$bX!s$bX!u$bX!w$bX!x$bX!y$bX!{$bX!}$bX#O$bX#P$bX#Q$bX#S$bX#T$bX#U$bX#]$bX#d$bX#o$bX$Y$bX$Z$bX$[$bX%S$bX%T$bX%U$bX%V$bX%^$bX~P2|Ou#ZO~O$X#[O%T]O#n%]P~Oo#iO~O!l#jO~O!n#jO~O!q#jO!r#jO~O!t#jO~O!v#jO~O!q#jO~O|#jO~O!z#jO~O!|#jO~O|#kO~O|#lO~O#V#jO#W#jO~Oo#mOu#bX~OZ!gO[!gO]!hO^!hOy!iO{!vO!h!oO$f!mOu${X$WtX$jtXwtX!ftXdtXatX$itX#ntXktX~O_'UOm'SOo'TOq'OOr'OOs'zO~P8xO$j#nO$W$pXw$pX~O$W#vX~P*|Ou#pO~OZ#qO[#qO]#qO^#qO$YQO$ZQO$f#qO$g#qO$vWX~O_WXwWX$jWX~P;RO_#uO~O$j#vOa$dX~Oa#yO~OTrOZUO[TOcsOguOhuOrgOu$TO!S!WO!T!WO!UwO!W!VO!Z|O!b!XO!fdO!hfO!kxO!myO!oyO!pzO!s{O!u{O!w}O!x!OO!y!PO!{!QO!}zO#O!QO#P!QO#Q!RO#S!SO#T!SO#U!TO#]!UO#diO#olO$YQO$ZQO%S[O%T]O%U^O%V_O~O|$VO~O{!vO!h!oO$f!mOTxaZxa[xa]xa^xa_xacxagxahxamxaoxaqxarxasxau${Xyxa!Sxa!Txa!Uxa!Wxa!Zxa!bxa!fxa!kxa!mxa!oxa!pxa!sxa!uxa!wxa!xxa!yxa!{xa!}xa#Oxa#Pxa#Qxa#Sxa#Txa#Uxa#]xa#dxa#oxa$Wxa$Yxa$Zxa$[xa%Sxa%Txa%Uxa%Vxa%^xa$jxawxadxaaxa$ixa#nxaexakxa~Ow%OP~P`O$f$_O$i!OXT!OXZ!OX[!OX]!OX^!OX_!OXc!OXg!OXh!OXm!OXo!OXq!OXr!OXs!OXu!OXy!OX{!OX!S!OX!T!OX!U!OX!W!OX!Z!OX!b!OX!f!OX!h!OX!k!OX!m!OX!o!OX!p!OX!s!OX!u!OX!w!OX!x!OX!y!OX!{!OX!}!OX#O!OX#P!OX#Q!OX#S!OX#T!OX#U!OX#]!OX#d!OX#o!OX$W!OX$Y!OX$Z!OX$[!OX%S!OX%T!OX%U!OX%V!OX%^!OX$j!OXw!OXd!OXa!OX#n!OXe!OXk!OX~O%S!wO~O$i$aO~OZ!gO[!gO]!hO^!hO_'UOm'SOo'TOq'OOr'OOs'zOw$bOy!iO{!vO!h!oO$f!mOu${X~O$[#YOZ$bX[$bX]$bX^$bX_$bXm$bXo$bXq$bXr$bXs$bXu$bXw$bXy$bX{$bX!h$bX!f$bX$j$bX~O$f$_O$i!OX~PIPOZ%PX[%PX]%PX^%PX_%PXm%PXo%PXq%PXr%PXs%PXu%PXw%PXy%PX{%PX!h%PX$f%PX$i%WX!f%PX$j%PX~OZ!gO[!gO]!hO^!hO_'UOm'SOo'TOq'OOr'OOs'zOy!iO{!vO!h!oO$f!mOu${X~O$j$cO!f%OXw%OX~PKrO!f$eO~O$YQO$ZQOw%ZP~OZ#qO[#qO]#qO^#qO$X#[O$f#qO$g#qO~O$v#sX~PMbO$v$lO~O$j$mO#n%]X~O#n$oO~Od$pO~PKrO$j$rOk$nX~Ok$tO~O!V$uO~O#R$vO#S$vO~O#R$vO~O!S$wO~O$j#nO$W$paw$pa~OZ#qO[#qO]#qO^#qO$YQO$ZQO$f#qO$g#qO~O_Wa$vWawWa$jWa~P! lO$j#vOa$da~OZ!gO[!gO]!hO^!hOy!iO{!vO!h!oO$f!mOTpi_picpigpihpimpiopiqpirpispiu${X!Spi!Tpi!Upi!Wpi!Zpi!bpi!fpi!kpi!mpi!opi!ppi!spi!upi!wpi!xpi!ypi!{pi!}pi#Opi#Ppi#Qpi#Spi#Tpi#Upi#]pi#dpi#opi$Wpi$Ypi$Zpi$[pi%Spi%Tpi%Upi%Vpi%^pi~O]!hO^!hOy!iO{!vO!h!oO$f!mOTxiZxi[xi_xicxigxihximxioxiqxirxisxiu${X!Sxi!Txi!Uxi!Wxi!Zxi!bxi!fxi!kxi!mxi!oxi!pxi!sxi!uxi!wxi!xxi!yxi!{xi!}xi#Oxi#Pxi#Qxi#Sxi#Txi#Uxi#]xi#dxi#oxi$Wxi$Yxi$Zxi$[xi%Sxi%Txi%Uxi%Vxi%^xi~Oy!iO{!vO!h!oO$f!mOTxiZxi[xi]xi^xi_xicxigxihximxioxiqxirxisxiu${X!Sxi!Txi!Uxi!Wxi!Zxi!bxi!fxi!kxi!mxi!oxi!pxi!sxi!uxi!wxi!xxi!yxi!{xi!}xi#Oxi#Pxi#Qxi#Sxi#Txi#Uxi#]xi#dxi#oxi$Wxi$Yxi$Zxi$[xi%Sxi%Txi%Uxi%Vxi%^xi~O{!vO!h!oO$f!mOTxiZxi[xi]xi^xi_xicxigxihximxioxiqxirxisxiu${X!Sxi!Txi!Uxi!Wxi!Zxi!bxi!fxi!kxi!mxi!oxi!pxi!sxi!uxi!wxi!xxi!yxi!{xi!}xi#Oxi#Pxi#Qxi#Sxi#Txi#Uxi#]xi#dxi#oxi$Wxi$Yxi$Zxi$[xi%Sxi%Txi%Uxi%Vxi%^xiexi~Oyxi$jxiwxidxiaxi$ixi#nxikxi~P!-kO!f%RO~PKrOZ!gO[!gO]!hO^!hOy!iO{!vO!h!oO$f!mOTlicliglihlimliu${X!Sli!Tli!Uli!Wli!Zli!bli!fli!kli!mli!oli!pli!sli!uli!wli!xli!yli!{li!}li#Oli#Pli#Qli#Sli#Tli#Uli#]li#dli#oli$Wli$Yli$Zli$[li%Sli%Tli%Uli%Vli%^li~O_!uOo!sOq!fOr!fOs!tO~P!1}OZ!gO[!gO]!hO^!hOy!iO{!vO!h!oO$f!mOTnicnignihnimnioniu${X!Sni!Tni!Uni!Wni!Zni!bni!fni!kni!mni!oni!pni!sni!uni!wni!xni!yni!{ni!}ni#Oni#Pni#Qni#Sni#Tni#Uni#]ni#dni#oni$Wni$Yni$Zni$[ni%Sni%Tni%Uni%Vni%^ni~O_!uOq!fOr!fOs!tO~P!5pOZ!gO[!gO]!hO^!hO_'UOm'SOq'OOr'OOs'zOy!iO{!vO!h!oO$f!mOu${X~Oo%SO~P!9cO!R%WO!U%XO!W%YO!Z%ZO!^%[O!b%]O$YQO$ZQO~OZ#}X[#}X]#}X^#}X_#}Xm#}Xo#}Xq#}Xr#}Xs#}Xu#}Xw#}Xy#}X{#}X!h#}X$Y#}X$Z#}X$[#}X$f#}X$j#}X~P;RO$v%_O~O$j%`Ow$}X~Ow%bO~O$f$_O$i!OaT!OaZ!Oa[!Oa]!Oa^!Oa_!Oac!Oag!Oah!Oam!Oao!Oaq!Oar!Oas!Oau!Oay!Oa{!Oa!S!Oa!T!Oa!U!Oa!W!Oa!Z!Oa!b!Oa!f!Oa!h!Oa!k!Oa!m!Oa!o!Oa!p!Oa!s!Oa!u!Oa!w!Oa!x!Oa!y!Oa!{!Oa!}!Oa#O!Oa#P!Oa#Q!Oa#S!Oa#T!Oa#U!Oa#]!Oa#d!Oa#o!Oa$W!Oa$Y!Oa$Z!Oa$[!Oa%S!Oa%T!Oa%U!Oa%V!Oa%^!Oa$j!Oaw!Oad!Oaa!Oa#n!Oae!Oak!Oa~O$j$cO!f%Oaw%Oa~O$v%hOw#kX$j#kX~O$j%iOw%ZX~Ow%kO~O$v#sa~PMbO$X#[O%T]O~O$j$mO#n%]a~O$j$rOk$na~O!T%uO~Ow!^O~O$i%vOa`X$j`X~PKrOTSqcSqgSqhSq!SSq!TSq!USq!WSq!ZSq!bSq!fSq!kSq!mSq!oSq!pSq!sSq!uSq!wSq!xSq!ySq!{Sq!}Sq#OSq#PSq#QSq#SSq#TSq#USq#]Sq#dSq#oSq$WSq$YSq$ZSq$[Sq%SSq%TSq%USq%VSq%^Sq~P'ZO$jtX~PG{Ow%xO~Oo%yO~Oo%zO~Oo%{O~O![%|O~O![%}O~O![&OO~O$j%`Ow$}a~Ow&SO!f&SO!h&SO~O!f$Sa$j$Saw$Sa~PKrO$j%iOw%Za~O#l&YO~P`O#n#qi$j#qi~PKrOZ!gO[!gO]!hO^!hO_(XOm(VOo(WOq(ROr(ROs)iOy!iO{!vO!h!oO$f!mOTbqcbqgbqhbqu${X!Sbq!Tbq!Ubq!Wbq!Zbq!bbq!fbq!kbq!mbq!obq!pbq!sbq!ubq!wbq!xbq!ybq!{bq!}bq#Obq#Pbq#Qbq#Sbq#Tbq#Ubq#]bq#dbq#obq$Wbq$Ybq$Zbq$[bq%Sbq%Tbq%Ubq%Vbq%^bq~Oe&ZO~P!GuOk$oi$j$oi~PKrOTfqcfqgfqhfq!Sfq!Tfq!Ufq!Wfq!Zfq!bfq!ffq!kfq!mfq!ofq!pfq!sfq!ufq!wfq!xfq!yfq!{fq!}fq#Ofq#Pfq#Qfq#Sfq#Tfq#Ufq#]fq#dfq#ofq$Wfq$Yfq$Zfq$[fq%Sfq%Tfq%Ufq%Vfq%^fq~P'ZOZ!gO[!gO]!hO^!hOy!iO{!vO!h!oO$f!mOTpy_pycpygpyhpympyopyqpyrpyspyu${X!Spy!Tpy!Upy!Wpy!Zpy!bpy!fpy!kpy!mpy!opy!ppy!spy!upy!wpy!xpy!ypy!{py!}py#Opy#Ppy#Qpy#Spy#Tpy#Upy#]py#dpy#opy$Wpy$Ypy$Zpy$[py%Spy%Tpy%Upy%Vpy%^py~O!S&]O~O!V&]O~O!S&^O~O!R%WO!U%XO!W%YO!Z%ZO!^%[O!b(rO$YQO$ZQO~O!X$wP~P#$eOw#Yi$j#Yi~PKrOT#mXc#mXg#mXh#mX!S#mX!T#mX!U#mX!W#mX!Z#mX!b#mX!f#mX!k#mX!m#mX!o#mX!p#mX!s#mX!u#mX!w#mX!x#mX!y#mX!{#mX!}#mX#O#mX#P#mX#Q#mX#S#mX#T#mX#U#mX#]#mX#d#mX#o#mX$W#mX$Y#mX$Z#mX$[#mX%S#mX%T#mX%U#mX%V#mX%^#mX~P'ZOa`i$j`i~PKrO!T&lO~O$YQO$ZQO!X!PX$f!PX$j!PX~O$f']O!X!OX$j!OX~O!X&nO~O$v&oO~O$j&pO!X$uX~O!X&rO~O$j&sO!X$wX~O!X&uO~OTb!Rcb!Rgb!Rhb!R!Sb!R!Tb!R!Ub!R!Wb!R!Zb!R!bb!R!fb!R!kb!R!mb!R!ob!R!pb!R!sb!R!ub!R!wb!R!xb!R!yb!R!{b!R!}b!R#Ob!R#Pb!R#Qb!R#Sb!R#Tb!R#Ub!R#]b!R#db!R#ob!R$Wb!R$Yb!R$Zb!R$[b!R%Sb!R%Tb!R%Ub!R%Vb!R%^b!R~P'ZO$f']O!X!Oa$j!Oa~O$j&pO!X$ua~O$j&sO!X$wa~O$x&{O~O$j$bXd$bXw$bXa$bX$i$bX#n$bXk$bX~P2|OZ!gO[!gO]!hO^!hOy!iO{!vO!h!oO$f!mO_pimpiopiqpirpispiu${X$Wpi$jpiwpi!fpidpiapi$ipi#npikpi~O]!hO^!hOy!iO{!vO!h!oO$f!mOZxi[xi_ximxioxiqxirxisxiu${X$Wxi$jxiwxi!fxidxiaxi$ixi#nxikxi~Oy!iO{!vO!h!oO$f!mOZxi[xi]xi^xi_ximxioxiqxirxisxiu${X$Wxi$jxiwxi!fxidxiaxi$ixi#nxikxi~OZ!gO[!gO]!hO^!hOy!iO{!vO!h!oO$f!mOmliu${X$Wli$jliwli!flidliali$ili#nlikli~O_'UOo'TOq'OOr'OOs'zO~P#2jOZ!gO[!gO]!hO^!hOy!iO{!vO!h!oO$f!mOmnioniu${X$Wni$jniwni!fnidniani$ini#nnikni~O_'UOq'OOr'OOs'zO~P#4[O$WSq$jSqwSq!fSqdSqaSq$iSq#nSqkSq~PKrO#l'iO~P`OZ!gO[!gO]!hO^!hO_(yOm(wOo(xOq(sOr(sOs)cOy!iO{!vO!h!oO$f!mOu${X$Wbq$jbqwbq!fbqdbqabq$ibq#nbqkbq~Oe'jO~P#6tOebq~P#6tOebq~P!GuO$Wfq$jfqwfq!ffqdfqafq$ifq#nfqkfq~PKrOZ!gO[!gO]!hO^!hOy!iO{!vO!h!oO$f!mO_pympyopyqpyrpyspyu${X$Wpy$jpywpy!fpydpyapy$ipy#npykpy~O$W#mX$j#mXw#mX!f#mXd#mXa#mX$i#mX#n#mXk#mX~PKrO$Wb!R$jb!Rwb!R!fb!Rdb!Rab!R$ib!R#nb!Rkb!R~PKrOTtXctXgtXhtX!StX!TtX!UtX!WtX!ZtX!btX!ftX!ktX!mtX!otX!ptX!stX!utX!wtX!xtX!ytX!{tX!}tX#OtX#PtX#QtX#StX#TtX#UtX#]tX#dtX#otX$WtX$YtX$ZtX$[tX%StX%TtX%UtX%VtX%^tX~P'ZOd'^O~PKrOk'_O~Oo'aO~P!9cOw'bO~O$x'lO~Od'tO~PKrO!X'uO~Od'vO~PKrOu'{O~Oepi~P!!pOexi~P!&_Oexi~P!)|O_(XOo(WOq(ROr(ROs)iOeli~P!1}O_(XOq(ROr(ROs)iOeni~P!5pOZ!gO[!gO]!hO^!hO_(XOm(VOo(WOq(ROr(ROs)iOy!iO{!vO!h!oO$f!mOu${X~OTSqcSqeSqgSqhSq!SSq!TSq!USq!WSq!ZSq!bSq!fSq!kSq!mSq!oSq!pSq!sSq!uSq!wSq!xSq!ySq!{Sq!}Sq#OSq#PSq#QSq#SSq#TSq#USq#]Sq#dSq#oSq$WSq$YSq$ZSq$[Sq%SSq%TSq%USq%VSq%^Sq~P#AoO#l(hO~P`OTfqcfqefqgfqhfq!Sfq!Tfq!Ufq!Wfq!Zfq!bfq!ffq!kfq!mfq!ofq!pfq!sfq!ufq!wfq!xfq!yfq!{fq!}fq#Ofq#Pfq#Qfq#Sfq#Tfq#Ufq#]fq#dfq#ofq$Wfq$Yfq$Zfq$[fq%Sfq%Tfq%Ufq%Vfq%^fq~P#AoOepy~P!NgOT#mXc#mXe#mXg#mXh#mX!S#mX!T#mX!U#mX!W#mX!Z#mX!b#mX!f#mX!k#mX!m#mX!o#mX!p#mX!s#mX!u#mX!w#mX!x#mX!y#mX!{#mX!}#mX#O#mX#P#mX#Q#mX#S#mX#T#mX#U#mX#]#mX#d#mX#o#mX$W#mX$Y#mX$Z#mX$[#mX%S#mX%T#mX%U#mX%V#mX%^#mX~P#AoOZ!gO[!gO]!hO^!hO_(XOm(VOo(WOq(ROr(ROs)iOy!iO{!vO!h!oO$f!mOu${X~Qb!RTtXctXetXgtXhtX!StX!TtX!UtX!WtX!ZtX!btX!ftX!ktX!mtX!otX!ptX!stX!utX!wtX!xtX!ytX!{tX!}tX#OtX#PtX#QtX#StX#TtX#UtX#]tX#dtX#otX$WtX$YtX$ZtX$[tX%StX%TtX%UtX%VtX%^tX~P#AoOk(`O~Oo(bO~P!9cOw(cO~O![(fO~Oepi~P#-rOexi~P#/`Oexi~P#0|O_(yOo(xOq(sOr(sOs)cOeli~P#2jO_(yOq(sOr(sOs)cOeni~P#4[OZ!gO[!gO]!hO^!hO_(yOm(wOo(xOq(sOr(sOs)cOy!iO{!vO!h!oO$f!mOu${X~OeSq$WSq$jSqwSq!fSqdSqaSq$iSq#nSqkSq~P$!{O#l)XO~P`Oefq$Wfq$jfqwfq!ffqdfqafq$ifq#nfqkfq~P$!{Oepy~P#9hOe#mX$W#mX$j#mXw#mX!f#mXd#mXa#mX$i#mX#n#mXk#mX~P$!{OZ!gO[!gO]!hO^!hO_(yOm(wOo(xOq(sOr(sOs)cOy!iO{!vO!h!oO$f!mOu${X~Qb!Rk)QO~Oo)SO~P!9cOw)TO~Ou)dO~O_(yOm(wOo(xOq(sOr(sOs)cOetX~P8xOu)jO~O",
    goto: "!7f%^PPPP%_P'X'e'n(Z+RPPPPP+[P%_PPP%_PP+_+kP%_P%_P%_PPP+tP,SP%_P%_PP,],r-V,zPPPPPPP,zPP,zP/l/o,zP/u/{%_P%_P%_0SPPPPPPPPPPPPPPPPPPPPPPPPPPPP1|2P2V1|P2b4_2b2b6c8`P%_:]%_<V<V>P>]P>fPP<V>r>x6_>|P?UP?X?_?f?l?r?xBXBdBjBpBvB|CSCYPPPPPPPPC`CdH[JULUL[PPLcPPLiLuNu!!u!!{!#S!#X!$n!&X!'v!)vP!)yP!)}!+h!-R!.{!/R!/U%_!/[!1UPPPP!3VH[!3c!5c!5i!7c$oiOPVefqt!f!j!k!l!p!r!s!t!u!z#n#p#t#x$T$c$l$p$q$t%S%_%k%v&Y&Z'O'P'Q'R'S'T'U'V'^'_'a'b'i'j'm't'v'x'z(Q(R(S(T(U(V(W(X(Y(`(b(c(h(i(l(m(s(t(u(v(w(x(y(z)Q)S)T)X)Y)[)c)iQ!eRQ'o'wQ(n)fR)])aW!cR'w)a)fR%O#vY!aR#v'w)a)fY#dv$r'y)b)g^$X!z#Z%`%i'{)d)jT&b%}&p%`WOPVXdefgqt!f!j!k!l!n!p!r!s!t!u#n#p#t#x$T$V$_$a$c$l$p$q$t%S%_%h%k%v%|&O&Y&Z&_&o&s&{'O'P'Q'R'S'T'U'V']'^'_'a'b'i'j'l'm't'v'x'z(Q(R(S(T(U(V(W(X(Y(`(b(c(f(h(i(l(m(s(t(u(v(w(x(y(z)Q)S)T)X)Y)[)c)it!`Rv#Z#v$r%`%i%}&p'w'y'{)a)b)d)f)g)jU#r!`#s$WR$W!zU#r!`#s$WT$j#]$kR$}#tQ#hvQ'q'yQ(o)gR)^)bW#fv'y)b)gR%r$rU!ZP#p$TW$U!u'U(X(yR$x#nQ!^PQ$z#pR%U$TQ%^$VQ&T%hQ&a%|U&f&O&s(fQ&v&oT&|&{'l[#Qdefg$T$ac%V$V%h%|&O&o&s&{'l(f!bjOVq!f!j!k!l!r!s!u#x$p$t%S%k&Y&Z't(R(S(T(U(V(W(X(Y(`(b(c(h(i(m[#Odg$V$a%h&{U#Tef$TQ$O!nS%c$_'][&`%|&O&o&s'l(f#V&}Pt!p!t!z#n#p#t$c$l$q%_%v'O'P'Q'R'S'T'U'V'^'_'a'b'i'j'm'v'x'z(Q(l(s(t(u(v(w(x(y(z)Q)S)T)X)Y)[)c)iR&e%}Q&c%}R&w&pQ&i&OR'}(fS&g&O(fR&y&s$oYOPVefqt!f!j!k!l!p!r!s!t!u!z#n#p#t#x$T$c$l$p$q$t%S%_%k%v&Y&Z'O'P'Q'R'S'T'U'V'^'_'a'b'i'j'm't'v'x'z(Q(R(S(T(U(V(W(X(Y(`(b(c(h(i(l(m(s(t(u(v(w(x(y(z)Q)S)T)X)Y)[)c)iR$^!zQ$Z!zR&Q%`S$Y!z%`Z$f#Z%i'{)d)j$ubOPVdefgqt!f!j!k!l!p!r!s!t!u!z#n#p#t#x$T$a$c$l$p$q$t%S%_%k%v&Y&Z'O'P'Q'R'S'T'U'V'^'_'a'b'i'j'm't'v'x'z(Q(R(S(T(U(V(W(X(Y(`(b(c(h(i(l(m(s(t(u(v(w(x(y(z)Q)S)T)X)Y)[)c)i$tbOPVdefgqt!f!j!k!l!p!r!s!t!u!z#n#p#t#x$T$a$c$l$p$q$t%S%_%k%v&Y&Z'O'P'Q'R'S'T'U'V'^'_'a'b'i'j'm't'v'x'z(Q(R(S(T(U(V(W(X(Y(`(b(c(h(i(l(m(s(t(u(v(w(x(y(z)Q)S)T)X)Y)[)c)iQ!|_T#^m$m$u`OPVdefgqt!f!j!k!l!p!r!s!t!u!z#n#p#t#x$T$a$c$l$p$q$t%S%_%k%v&Y&Z'O'P'Q'R'S'T'U'V'^'_'a'b'i'j'm't'v'x'z(Q(R(S(T(U(V(W(X(Y(`(b(c(h(i(l(m(s(t(u(v(w(x(y(z)Q)S)T)X)Y)[)c)i$uaOPVdefgqt!f!j!k!l!p!r!s!t!u!z#n#p#t#x$T$a$c$l$p$q$t%S%_%k%v&Y&Z'O'P'Q'R'S'T'U'V'^'_'a'b'i'j'm't'v'x'z(Q(R(S(T(U(V(W(X(Y(`(b(c(h(i(l(m(s(t(u(v(w(x(y(z)Q)S)T)X)Y)[)c)i$ohOPVefqt!f!j!k!l!p!r!s!t!u!z#n#p#t#x$T$c$l$p$q$t%S%_%k%v&Y&Z'O'P'Q'R'S'T'U'V'^'_'a'b'i'j'm't'v'x'z(Q(R(S(T(U(V(W(X(Y(`(b(c(h(i(l(m(s(t(u(v(w(x(y(z)Q)S)T)X)Y)[)c)i$onOPVefqt!f!j!k!l!p!r!s!t!u!z#n#p#t#x$T$c$l$p$q$t%S%_%k%v&Y&Z'O'P'Q'R'S'T'U'V'^'_'a'b'i'j'm't'v'x'z(Q(R(S(T(U(V(W(X(Y(`(b(c(h(i(l(m(s(t(u(v(w(x(y(z)Q)S)T)X)Y)[)c)iQ$i#ZQ's'{Q(q)jR)`)dW$g#Z'{)d)jR&U%iW&X%k'b(c)TX&j&Y'i(h)XQ#`mR%n$mT#_m$mS#]m$mT$j#]$kR!^PQqOR#bqS#s!`$WR${#sQ#w!cR%P#wQ$s#fR%s$sQ#o!ZR$y#o%OXOPVdefgqt!f!j!k!l!n!p!r!s!t!u!z#n#p#t#x$T$V$_$a$c$l$p$q$t%S%_%h%k%v&Y&Z&{'O'P'Q'R'S'T'U'V'^'_'a'b'i'j'm't'v'x'z(Q(R(S(T(U(V(W(X(Y(`(b(c(h(i(l(m(s(t(u(v(w(x(y(z)Q)S)T)X)Y)[)c)iS!yX&__&_%|&O&o&s']'l(fS$`#O#TS%d$`&mR&m&`Q&q&cR&x&qQ&t&gR&z&tQ%a$ZR&R%aQ$d#VR%g$dQ%j$gR&V%jQ$k#]R%l$kQ$n#`R%o$nTpOqSSOqW!YP#n#p'UW!xV'm(m)[Q#SeS#Vf!zQ#ctQ#z!fQ#{!jQ#|!kW#}!l'R(U(vQ$P!pQ$Q!rQ$R!sQ$S!tQ$|#tQ%Q#xQ%T$TQ%f$cQ%m$lQ%p$pQ%q$qQ%t$tQ%w%SQ&P%_S&W%k&YQ&[%vQ&k&ZQ'W'OQ'X'PQ'Y'QQ'Z'SQ'['TQ'`'VQ'c'^Q'd'vQ'e'tQ'f'_Q'g'aS'h'b'iQ'k'jQ'n!uQ'p'xQ'r'zQ'|(QQ(O(lQ(Z(RQ([(SQ(](TQ(^(VQ(_(WQ(a(YQ(d(`Q(e(bS(g(c(hQ(j(iQ(k(XQ(p)iQ({(sQ(|(tQ(}(uQ)O(wQ)P(xQ)R(zQ)U)QQ)V)SS)W)T)XQ)Z)YQ)_)cR)h(y$ooOPVefqt!f!j!k!l!p!r!s!t!u!z#n#p#t#x$T$c$l$p$q$t%S%_%k%v&Y&Z'O'P'Q'R'S'T'U'V'^'_'a'b'i'j'm't'v'x'z(Q(R(S(T(U(V(W(X(Y(`(b(c(h(i(l(m(s(t(u(v(w(x(y(z)Q)S)T)X)Y)[)c)ipROVq!f!j!k!l!r!s!u#x$t%S%k&Y&Z!j'wPeft!p!t!z#n#p#t$T$c$l$q%_%v'O'P'Q'R'S'T'U'V'_'a'b'i'j'm'x'z(Q(l)c)ip)a'^'v(s(t(u(v(w(x(y(z)Q)S)T)X)Y)[q)f$p't(R(S(T(U(V(W(X(Y(`(b(c(h(i(mX!dR'w)a)fZ!bR#v'w)a)fQ#t!aR$q#dQ#x!eQ'V'oQ(Y(nR(z)]ptOVq!f!j!k!l!r!s!u#x$t%S%k&Y&Z!j'xPeft!p!t!z#n#p#t$T$c$l$q%_%v'O'P'Q'R'S'T'U'V'_'a'b'i'j'm'x'z(Q(l)c)ip(Q$p't(R(S(T(U(V(W(X(Y(`(b(c(h(i(mq(l'^'v(s(t(u(v(w(x(y(z)Q)S)T)X)Y)[pvOVq!f!j!k!l!r!s!u#x$t%S%k&Y&Z!j'yPeft!p!t!z#n#p#t$T$c$l$q%_%v'O'P'Q'R'S'T'U'V'_'a'b'i'j'm'x'z(Q(l)c)ip)b'^'v(s(t(u(v(w(x(y(z)Q)S)T)X)Y)[q)g$p't(R(S(T(U(V(W(X(Y(`(b(c(h(i(mX#gv'y)b)gZ#ev$r'y)b)gV![P#p$Td!jS#z$Q$R%Q%t%w&W&k'n!W'P!Y#S#V#c$P$S$|%T%f%m%q&P&['W'Z'['`'f'g'h'k'p'r'|(O(p)_f(S%p'e(Z(^(_(a(d(e(g(j(kg(t'c'd({)O)P)R)U)V)W)Z)hf!kS#z#{$Q$R%Q%t%w&W&k'n!Y'Q!Y#S#V#c$P$S$|%T%f%m%q&P&['W'X'Z'['`'f'g'h'k'p'r'|(O(p)_h(T%p'e(Z([(^(_(a(d(e(g(j(ki(u'c'd({(|)O)P)R)U)V)W)Z)hh!lS#z#{#|$Q$R%Q%t%w&W&k'n!['R!Y#S#V#c$P$S$|%T%f%m%q&P&['W'X'Y'Z'['`'f'g'h'k'p'r'|(O(p)_j(U%p'e(Z([(](^(_(a(d(e(g(j(kk(v'c'd({(|(})O)P)R)U)V)W)Z)hpVOVq!f!j!k!l!r!s!u#x$t%S%k&Y&Z!j'mPeft!p!t!z#n#p#t$T$c$l$q%_%v'O'P'Q'R'S'T'U'V'_'a'b'i'j'm'x'z(Q(l)c)ip(m$p't(R(S(T(U(V(W(X(Y(`(b(c(h(i(mq)['^'v(s(t(u(v(w(x(y(z)Q)S)T)X)Y)[R&d%}T&h&O(f$P!nS!Y!x#S#V#c#z#{#|#}$P$Q$R$S$|%Q%T%f%m%p%q%t%w&P&W&[&k'W'X'Y'Z'['`'c'd'e'f'g'h'k'n'p'r'|(O(Z([(](^(_(a(d(e(g(j(k(p({(|(})O)P)R)U)V)W)Z)_)h$P!pS!Y!x#S#V#c#z#{#|#}$P$Q$R$S$|%Q%T%f%m%p%q%t%w&P&W&[&k'W'X'Y'Z'['`'c'd'e'f'g'h'k'n'p'r'|(O(Z([(](^(_(a(d(e(g(j(k(p({(|(})O)P)R)U)V)W)Z)_)h$oZOPVefqt!f!j!k!l!p!r!s!t!u!z#n#p#t#x$T$c$l$p$q$t%S%_%k%v&Y&Z'O'P'Q'R'S'T'U'V'^'_'a'b'i'j'm't'v'x'z(Q(R(S(T(U(V(W(X(Y(`(b(c(h(i(l(m(s(t(u(v(w(x(y(z)Q)S)T)X)Y)[)c)iQ!{ZR!}`R$[!zQ#WfR$]!z$ocOPVefqt!f!j!k!l!p!r!s!t!u!z#n#p#t#x$T$c$l$p$q$t%S%_%k%v&Y&Z'O'P'Q'R'S'T'U'V'^'_'a'b'i'j'm't'v'x'z(Q(R(S(T(U(V(W(X(Y(`(b(c(h(i(l(m(s(t(u(v(w(x(y(z)Q)S)T)X)Y)[)c)i$hcOPVqt!f!j!k!l!p!r!s!t!u!z#n#p#t#x$c$l$p$q$t%S%_%k%v&Y&Z'O'P'Q'R'S'T'U'V'^'_'a'b'i'j'm't'v'x'z(Q(R(S(T(U(V(W(X(Y(`(b(c(h(i(l(m(s(t(u(v(w(x(y(z)Q)S)T)X)Y)[)c)iU#Qdg$aV#Uef$TW#Rdef$TQ#XgR%e$apkOVq!f!j!k!l!r!s!u#x$t%S%k&Y&Z!j(PPeft!p!t!z#n#p#t$T$c$l$q%_%v'O'P'Q'R'S'T'U'V'_'a'b'i'j'm'x'z(Q(l)c)ip)e'^'v(s(t(u(v(w(x(y(z)Q)S)T)X)Y)[q)k$p't(R(S(T(U(V(W(X(Y(`(b(c(h(i(mX$h#Z'{)d)j$omOPVefqt!f!j!k!l!p!r!s!t!u!z#n#p#t#x$T$c$l$p$q$t%S%_%k%v&Y&Z'O'P'Q'R'S'T'U'V'^'_'a'b'i'j'm't'v'x'z(Q(R(S(T(U(V(W(X(Y(`(b(c(h(i(l(m(s(t(u(v(w(x(y(z)Q)S)T)X)Y)[)c)iR#am",
    nodeNames: "⚠ LineComment BlockComment Expressions ForExpression for InExpressions InExpression Name Identifier Identifier ArithOp ArithOp ArithOp ArithOp in IterationContext return IfExpression if then else QuantifiedExpression some every InExpressions InExpression satisfies Disjunction or Conjunction and Comparison CompareOp CompareOp between PositiveUnaryTest ( PositiveUnaryTests ) ArithmeticExpression ArithOp InstanceOfExpression instance of Type QualifiedName VariableName SpecialType days time duration years months date > ListType list < ContextType context ContextEntryTypes ContextEntryType FunctionType function ArgumentTypes ArgumentType PathExpression ] FilterExpression [ FunctionInvocation SpecialFunctionName string length upper case lower substring before after starts with ends contains insert index distinct values met by overlaps finished started day year week month get value entries NamedParameters NamedParameter ParameterName PositionalParameters null NumericLiteral StringLiteral BooleanLiteral DateTimeLiteral DateTimeConstructor AtLiteral ? SimplePositiveUnaryTest Interval ParenthesizedExpression List FunctionDefinition FormalParameters FormalParameter external FunctionBody } { Context ContextEntry Key Name Identifier UnaryTests Wildcard not",
    maxTerm: 198,
    context: variableTracker$1,
    nodeProps: [
      ["group", -17,4,18,22,28,30,32,40,42,67,69,71,112,113,115,116,117,124,"Expression",47,"Expression Expression",-5,105,106,107,108,109,"Expression Literal"],
      ["closedBy", 37,")",70,"]",123,"}"],
      ["openedBy", 39,"(",68,"[",122,"{"]
    ],
    propSources: [feelHighlighting$1],
    skippedNodes: [0,1,2],
    repeatNodeCount: 14,
    tokenData: ")x~RuXY#fYZ$ZZ[#f]^$Zpq#fqr$`rs$kwx%_xy%dyz%iz{%n{|%{|}&Q}!O&V!O!P&d!P!Q&|!Q![(X![!](j!]!^(o!^!_(t!_!`$f!`!a)T!b!c)_!}#O)d#P#Q)i#Q#R%v#o#p)n#q#r)s$f$g#f#BY#BZ#f$IS$I_#f$I|$I}$Z$I}$JO$Z$JT$JU#f$KV$KW#f&FU&FV#f?HT?HU#f~#kY$^~XY#fZ[#fpq#f$f$g#f#BY#BZ#f$IS$I_#f$JT$JU#f$KV$KW#f&FU&FV#f?HT?HU#f~$`O$_~~$cP!_!`$f~$kOq~~$pU%T~OY$kZr$krs%Ss#O$k#O#P%X#P~$k~%XO%T~~%[PO~$k~%dO$g~~%iOu~~%nOw~~%sP^~z{%v~%{Oy~~&QO[~~&VO$j~R&[PZP!`!a&_Q&dO$xQ~&iQ$f~!O!P&o!Q![&t~&tO$i~~&yP%S~!Q![&t~'RQ]~z{'X!P!Q'|~'[ROz'Xz{'e{~'X~'hTOz'Xz{'e{!P'X!P!Q'w!Q~'X~'|OQ~~(RQP~OY'|Z~'|~(^Q%S~!O!P(d!Q![(X~(gP!Q![&t~(oO$v~~(tO%^~R({P![QrP!_!`)OP)TOrPR)[P!XQrP!_!`)O~)dO%V~~)iO!h~~)nO!f~~)sO#o~~)xO#n~",
    tokenizers: [propertyIdentifiers$1, identifiers$1, insertSemicolon$1, 0, 1],
    topRules: {"Expressions":[0,3],"UnaryTests":[1,129]},
    dynamicPrecedences: {"30":-1,"71":-1,"101":-1,"154":-1},
    specialized: [{term: 148, get: value => spec_identifier$1[value] || -1}],
    tokenPrec: 0
  });

  function parseParameterNames$1(fn) {
      if (Array.isArray(fn.$args)) {
          return fn.$args;
      }
      const code = fn.toString();
      const match = /^(?:[^(]*\s*)?\(([^)]+)?\)/.exec(code);
      if (!match) {
          throw new Error('failed to parse params: ' + code);
      }
      const [_, params] = match;
      if (!params) {
          return [];
      }
      return params.split(',').map(p => p.trim());
  }
  function notImplemented$1(thing) {
      return new Error(`not implemented: ${thing}`);
  }
  /**
   * @param {string} name
   * @param {Record<string, any>} context
   *
   * @return {any}
   */
  function getFromContext$1(name, context) {
      if (['nil', 'boolean', 'number', 'string'].includes(getType$1(context))) {
          return null;
      }
      if (name in context) {
          return context[name];
      }
      const normalizedName = normalizeContextKey$1(name);
      if (normalizedName in context) {
          return context[normalizedName];
      }
      const entry = Object.entries(context).find(([key]) => normalizedName === normalizeContextKey$1(key));
      if (entry) {
          return entry[1];
      }
      return null;
  }

  function isDateTime$1(obj) {
      return DateTime.isDateTime(obj);
  }
  function isDuration$1(obj) {
      return Duration.isDuration(obj);
  }
  function duration$2(opts) {
      if (typeof opts === 'number') {
          return Duration.fromMillis(opts);
      }
      return Duration.fromISO(opts);
  }
  function date$1(str = null, time = null, zone = null) {
      if (time) {
          if (str) {
              throw new Error('<str> and <time> provided');
          }
          return date$1(`1900-01-01T${time}`);
      }
      if (typeof str === 'string') {
          if (str.startsWith('-')) {
              throw notImplemented$1('negative date');
          }
          if (!str.includes('T')) {
              // raw dates are in UTC time zone
              return date$1(str + 'T00:00:00.000Z');
          }
          if (str.includes('@')) {
              if (zone) {
                  throw new Error('<zone> already provided');
              }
              const [datePart, zonePart] = str.split('@');
              return date$1(datePart, null, Info.normalizeZone(zonePart));
          }
          return DateTime.fromISO(str.toUpperCase(), {
              setZone: true,
              zone
          });
      }
      return DateTime.now();
  }

  function isContext$1(e) {
      return Object.getPrototypeOf(e) === Object.prototype;
  }
  function isArray$2(e) {
      return Array.isArray(e);
  }
  function isBoolean$1(e) {
      return typeof e === 'boolean';
  }
  function getType$1(e) {
      if (e === null || e === undefined) {
          return 'nil';
      }
      if (isBoolean$1(e)) {
          return 'boolean';
      }
      if (isNumber$1(e)) {
          return 'number';
      }
      if (isString$1(e)) {
          return 'string';
      }
      if (isContext$1(e)) {
          return 'context';
      }
      if (isArray$2(e)) {
          return 'list';
      }
      if (isDuration$1(e)) {
          return 'duration';
      }
      if (isDateTime$1(e)) {
          if (e.year === 1900 &&
              e.month === 1 &&
              e.day === 1) {
              return 'time';
          }
          if (e.hour === 0 &&
              e.minute === 0 &&
              e.second === 0 &&
              e.millisecond === 0 &&
              e.zone === FixedOffsetZone.utcInstance) {
              return 'date';
          }
          return 'date time';
      }
      if (e instanceof Range$2) {
          return 'range';
      }
      if (e instanceof FunctionWrapper$1) {
          return 'function';
      }
      return 'literal';
  }
  function isType$1(el, type) {
      return getType$1(el) === type;
  }
  function typeCast$1(obj, type) {
      if (isDateTime$1(obj)) {
          if (type === 'time') {
              return obj.set({
                  year: 1900,
                  month: 1,
                  day: 1
              });
          }
          if (type === 'date') {
              return obj.setZone('utc', { keepLocalTime: true }).startOf('day');
          }
          if (type === 'date time') {
              return obj;
          }
      }
      return null;
  }
  let Range$2 = class Range {
      constructor(props) {
          Object.assign(this, props);
      }
  };
  function isNumber$1(obj) {
      return typeof obj === 'number';
  }
  function isString$1(obj) {
      return typeof obj === 'string';
  }
  function equals$1(a, b) {
      if (a === null && b !== null ||
          a !== null && b === null) {
          return false;
      }
      if (isArray$2(a) && a.length < 2) {
          a = a[0];
      }
      if (isArray$2(b) && b.length < 2) {
          b = b[0];
      }
      const aType = getType$1(a);
      const bType = getType$1(b);
      if (aType !== bType) {
          return null;
      }
      if (aType === 'nil') {
          return true;
      }
      if (aType === 'list') {
          if (a.length !== b.length) {
              return false;
          }
          return a.every((element, idx) => equals$1(element, b[idx]));
      }
      if (aType === 'date time' || aType === 'time' || aType === 'date') {
          return (a.toUTC().valueOf() === b.toUTC().valueOf());
      }
      if (aType === 'duration') {
          // years and months duration -> months
          if (Math.abs(a.as('days')) > 180) {
              return Math.trunc(a.minus(b).as('months')) === 0;
          }
          // days and time duration -> seconds
          else {
              return Math.trunc(a.minus(b).as('seconds')) === 0;
          }
      }
      if (aType === 'context') {
          const aEntries = Object.entries(a);
          const bEntries = Object.entries(b);
          if (aEntries.length !== bEntries.length) {
              return false;
          }
          return aEntries.every(([key, value]) => key in b && equals$1(value, b[key]));
      }
      if (aType === 'range') {
          return [
              [a.start, b.start],
              [a.end, b.end],
              [a['start included'], b['start included']],
              [a['end included'], b['end included']]
          ].every(([a, b]) => a === b);
      }
      if (a == b) {
          return true;
      }
      return aType === bType ? false : null;
  }
  let FunctionWrapper$1 = class FunctionWrapper {
      constructor(fn, parameterNames) {
          this.fn = fn;
          this.parameterNames = parameterNames;
      }
      invoke(contextOrArgs) {
          let params;
          if (isArray$2(contextOrArgs)) {
              params = contextOrArgs;
          }
          else {
              params = this.parameterNames.map(n => contextOrArgs[n]);
          }
          return this.fn.call(null, ...params);
      }
  };

  // 10.3.4 Built-in functions
  const builtins$1 = {
      // 10.3.4.1 Conversion functions
      'number': function () {
          throw notImplemented$1('number');
      },
      'string': fn$1(function (from) {
          if (arguments.length !== 1) {
              return null;
          }
          return toString$1(from);
      }, ['any']),
      // date(from) => date string
      // date(from) => date and time
      // date(year, month, day)
      'date': fn$1(function (year, month, day, from) {
          if (!from && !isNumber$1(year)) {
              from = year;
              year = null;
          }
          let d;
          if (isString$1(from)) {
              d = date$1(from);
          }
          if (isDateTime$1(from)) {
              d = from;
          }
          if (year) {
              d = date$1().setZone('utc').set({
                  year,
                  month,
                  day
              });
          }
          return d && ifValid$1(d.setZone('utc').startOf('day')) || null;
      }, ['any?', 'number?', 'number?', 'any?']),
      // date and time(from) => date time string
      // date and time(date, time)
      'date and time': fn$1(function (d, time, from) {
          let dt;
          if (isDateTime$1(d) && isDateTime$1(time)) {
              dt = time.set({
                  year: d.year,
                  month: d.month,
                  day: d.day
              });
          }
          if (isString$1(d)) {
              from = d;
              d = null;
          }
          if (isString$1(from)) {
              dt = date$1(from);
          }
          return dt && ifValid$1(dt) || null;
      }, ['any?', 'time?', 'string?'], ['date', 'time', 'from']),
      // time(from) => time string
      // time(from) => time, date and time
      // time(hour, minute, second, offset?) => ...
      'time': fn$1(function (hour, minute, second, offset, from) {
          let t;
          if (offset) {
              throw notImplemented$1('time(..., offset)');
          }
          if (isString$1(hour) || isDateTime$1(hour)) {
              from = hour;
              hour = null;
          }
          if (isString$1(from)) {
              t = date$1(null, from);
          }
          if (isDateTime$1(from)) {
              t = from.set({
                  year: 1900,
                  month: 1,
                  day: 1
              });
          }
          if (isNumber$1(hour)) {
              // TODO: support offset = days and time duration
              t = date$1().set({
                  hour,
                  minute,
                  second
              }).set({
                  year: 1900,
                  month: 1,
                  day: 1,
                  millisecond: 0
              });
          }
          return t && ifValid$1(t) || null;
      }, ['any?', 'number?', 'number?', 'any?', 'any?']),
      'duration': fn$1(function (from) {
          return ifValid$1(duration$2(from));
      }, ['string']),
      'years and months duration': fn$1(function (from, to) {
          return ifValid$1(to.diff(from, ['years', 'months']));
      }, ['date', 'date']),
      '@': fn$1(function (string) {
          let t;
          if (/^-?P/.test(string)) {
              t = duration$2(string);
          }
          else if (/^[\d]{1,2}:[\d]{1,2}:[\d]{1,2}/.test(string)) {
              t = date$1(null, string);
          }
          else {
              t = date$1(string);
          }
          return t && ifValid$1(t) || null;
      }, ['string']),
      'now': fn$1(function () {
          return date$1();
      }, []),
      'today': fn$1(function () {
          return date$1().startOf('day');
      }, []),
      // 10.3.4.2 Boolean function
      'not': fn$1(function (bool) {
          return isType$1(bool, 'boolean') ? !bool : null;
      }, ['any']),
      // 10.3.4.3 String functions
      'substring': fn$1(function (string, start, length) {
          const _start = (start < 0 ? string.length + start : start - 1);
          const arr = Array.from(string);
          return (typeof length !== 'undefined'
              ? arr.slice(_start, _start + length)
              : arr.slice(_start)).join('');
      }, ['string', 'number', 'number?'], ['string', 'start position', 'length']),
      'string length': fn$1(function (string) {
          return countSymbols$1(string);
      }, ['string']),
      'upper case': fn$1(function (string) {
          return string.toUpperCase();
      }, ['string']),
      'lower case': fn$1(function (string) {
          return string.toLowerCase();
      }, ['string']),
      'substring before': fn$1(function (string, match) {
          const index = string.indexOf(match);
          if (index === -1) {
              return '';
          }
          return string.substring(0, index);
      }, ['string', 'string']),
      'substring after': fn$1(function (string, match) {
          const index = string.indexOf(match);
          if (index === -1) {
              return '';
          }
          return string.substring(index + match.length);
      }, ['string', 'string']),
      'replace': fn$1(function (input, pattern, replacement, flags) {
          return input.replace(new RegExp(pattern, 'ug' + (flags || '').replace(/[x]/g, '')), replacement.replace(/\$0/g, '$$&'));
      }, ['string', 'string', 'string', 'string?']),
      'contains': fn$1(function (string, match) {
          return string.includes(match);
      }, ['string', 'string']),
      'starts with': fn$1(function (string, match) {
          return string.startsWith(match);
      }, ['string', 'string']),
      'ends with': fn$1(function (string, match) {
          return string.endsWith(match);
      }, ['string', 'string']),
      'split': fn$1(function (string, delimiter) {
          return string.split(new RegExp(delimiter, 'u'));
      }, ['string', 'string']),
      // 10.3.4.4 List functions
      'list contains': fn$1(function (list, element) {
          return list.some(el => matches$1(el, element));
      }, ['list', 'any?']),
      'count': fn$1(function (list) {
          return list.length;
      }, ['list']),
      'min': listFn$1(function (list) {
          return list.reduce((min, el) => min === null ? el : Math.min(min, el), null);
      }, 'number'),
      'max': listFn$1(function (list) {
          return list.reduce((max, el) => max === null ? el : Math.max(max, el), null);
      }, 'number'),
      'sum': listFn$1(function (list) {
          return sum$1(list);
      }, 'number'),
      'mean': listFn$1(function (list) {
          const s = sum$1(list);
          return s === null ? s : s / list.length;
      }, 'number'),
      'all': listFn$1(function (list) {
          let nonBool = false;
          for (const o of list) {
              if (o === false) {
                  return false;
              }
              if (typeof o !== 'boolean') {
                  nonBool = true;
              }
          }
          return nonBool ? null : true;
      }, 'any?'),
      'any': listFn$1(function (list) {
          let nonBool = false;
          for (const o of list) {
              if (o === true) {
                  return true;
              }
              if (typeof o !== 'boolean') {
                  nonBool = true;
              }
          }
          return nonBool ? null : false;
      }, 'any?'),
      'sublist': fn$1(function (list, start, length) {
          const _start = (start < 0 ? list.length + start : start - 1);
          return (typeof length !== 'undefined'
              ? list.slice(_start, _start + length)
              : list.slice(_start));
      }, ['list', 'number', 'number?']),
      'append': fn$1(function (list, ...items) {
          return list.concat(items);
      }, ['list', 'any?']),
      'concatenate': fn$1(function (...args) {
          return args.reduce((result, arg) => {
              return result.concat(arg);
          }, []);
      }, ['any']),
      'insert before': fn$1(function (list, position, newItem) {
          return list.slice(0, position - 1).concat([newItem], list.slice(position - 1));
      }, ['list', 'number', 'any?']),
      'remove': fn$1(function (list, position) {
          return list.slice(0, position - 1).concat(list.slice(position));
      }, ['list', 'number']),
      'reverse': fn$1(function (list) {
          return list.slice().reverse();
      }, ['list']),
      'index of': fn$1(function (list, match) {
          return list.reduce(function (result, element, index) {
              if (matches$1(element, match)) {
                  result.push(index + 1);
              }
              return result;
          }, []);
      }, ['list', 'any']),
      'union': fn$1(function (..._lists) {
          throw notImplemented$1('union');
      }, ['list']),
      'distinct values': fn$1(function (_list) {
          throw notImplemented$1('distinct values');
      }, ['list']),
      'flatten': fn$1(function (list) {
          return flatten$2(list);
      }, ['list']),
      'product': listFn$1(function (list) {
          if (list.length === 0) {
              return null;
          }
          return list.reduce((result, n) => {
              return result * n;
          }, 1);
      }, 'number'),
      'median': listFn$1(function (list) {
          if (list.length === 0) {
              return null;
          }
          return median$1(list);
      }, 'number'),
      'stddev': listFn$1(function (list) {
          if (list.length < 2) {
              return null;
          }
          return stddev$1(list);
      }, 'number'),
      'mode': listFn$1(function (list) {
          return mode$1(list);
      }, 'number'),
      // 10.3.4.5 Numeric functions
      'decimal': fn$1(function (n, scale) {
          if (!scale) {
              return round$2(n);
          }
          const offset = Math.pow(10, scale);
          return round$2(n * offset) / (offset);
      }, ['number', 'number']),
      'floor': fn$1(function (n) {
          return Math.floor(n);
      }, ['number']),
      'ceiling': fn$1(function (n) {
          return Math.ceil(n) + 0;
      }, ['number']),
      'abs': fn$1(function (n) {
          if (typeof n !== 'number') {
              return null;
          }
          return Math.abs(n);
      }, ['number']),
      'modulo': fn$1(function (dividend, divisor) {
          if (!divisor) {
              return null;
          }
          const adjust = 1000000000;
          // cf. https://dustinpfister.github.io/2017/09/02/js-whats-wrong-with-modulo/
          //
          // need to round here as using this custom modulo
          // variant is prone to rounding errors
          return Math.round((dividend % divisor + divisor) % divisor * adjust) / adjust;
      }, ['number', 'number']),
      'sqrt': fn$1(function (number) {
          if (number < 0) {
              return null;
          }
          return Math.sqrt(number);
      }, ['number']),
      'log': fn$1(function (number) {
          if (number <= 0) {
              return null;
          }
          return Math.log(number);
      }, ['number']),
      'exp': fn$1(function (number) {
          return Math.exp(number);
      }, ['number']),
      'odd': fn$1(function (number) {
          return Math.abs(number) % 2 === 1;
      }, ['number']),
      'even': fn$1(function (number) {
          return Math.abs(number) % 2 === 0;
      }, ['number']),
      // 10.3.4.6 Date and time functions
      'is': fn$1(function (value1, value2) {
          if (typeof value1 === 'undefined' || typeof value2 === 'undefined') {
              return false;
          }
          return equals$1(value1, value2);
      }, ['any?', 'any?']),
      // 10.3.4.7 Range Functions
      'before': fn$1(function (a, b) {
          return before$1(a, b);
      }, ['any', 'any']),
      'after': fn$1(function (a, b) {
          return before$1(b, a);
      }, ['any', 'any']),
      'meets': fn$1(function (a, b) {
          return meets$1(a, b);
      }, ['range', 'range']),
      'met by': fn$1(function (a, b) {
          return meets$1(b, a);
      }, ['range', 'range']),
      'overlaps': fn$1(function () {
          throw notImplemented$1('overlaps');
      }, ['any?']),
      'overlaps before': fn$1(function () {
          throw notImplemented$1('overlaps before');
      }, ['any?']),
      'overlaps after': fn$1(function () {
          throw notImplemented$1('overlaps after');
      }, ['any?']),
      'finishes': fn$1(function () {
          throw notImplemented$1('finishes');
      }, ['any?']),
      'finished by': fn$1(function () {
          throw notImplemented$1('finished by');
      }, ['any?']),
      'includes': fn$1(function () {
          throw notImplemented$1('includes');
      }, ['any?']),
      'during': fn$1(function () {
          throw notImplemented$1('during');
      }, ['any?']),
      'starts': fn$1(function () {
          throw notImplemented$1('starts');
      }, ['any?']),
      'started by': fn$1(function () {
          throw notImplemented$1('started by');
      }, ['any?']),
      'coincides': fn$1(function () {
          throw notImplemented$1('coincides');
      }, ['any?']),
      // 10.3.4.8 Temporal built-in functions
      'day of year': fn$1(function () {
          throw notImplemented$1('day of year');
      }, ['any?']),
      'day of week': fn$1(function () {
          throw notImplemented$1('day of week');
      }, ['any?']),
      'month of year': fn$1(function () {
          throw notImplemented$1('month of year');
      }, ['any?']),
      'week of year': fn$1(function () {
          throw notImplemented$1('week of year');
      }, ['any?']),
      // 10.3.4.9 Sort
      'sort': function () {
          throw notImplemented$1('sort');
      },
      // 10.3.4.10 Context function
      'get value': fn$1(function (m, key) {
          return getFromContext$1(key, m);
      }, ['context', 'string']),
      'get entries': fn$1(function (m) {
          if (arguments.length !== 1) {
              return null;
          }
          if (Array.isArray(m)) {
              return null;
          }
          return Object.entries(m).map(([key, value]) => ({ key, value }));
      }, ['context']),
      'context': listFn$1(function (_contexts) {
          throw notImplemented$1('context');
      }, 'context'),
      'context merge': listFn$1(function (_contexts) {
          throw notImplemented$1('context merge');
      }, 'context'),
      'context put': fn$1(function (_context, _keys, _value) {
          throw notImplemented$1('context put');
      }, ['context', 'list', 'any'])
  };
  function matches$1(a, b) {
      return a === b;
  }
  const FALSE$1 = {};
  function createArgTester$1(arg) {
      const optional = arg.endsWith('?');
      const type = optional ? arg.substring(0, arg.length - 1) : arg;
      return function (obj) {
          const arr = Array.isArray(obj);
          if (type === 'list') {
              if (arr || optional && typeof obj === 'undefined') {
                  return obj;
              }
              else {
                  // implicit conversion obj => [ obj ]
                  return [obj];
              }
          }
          if (type !== 'any' && arr && obj.length === 1) {
              // implicit conversion [ obj ] => obj
              obj = obj[0];
          }
          if (type === 'range') {
              return obj instanceof Range$2 ? obj : FALSE$1;
          }
          const objType = getType$1(obj);
          if (objType === 'nil') {
              return (optional ? obj : FALSE$1);
          }
          if (type === 'any' || type === objType) {
              return obj;
          }
          return typeCast$1(obj, type) || FALSE$1;
      };
  }
  function createArgsValidator$1(argDefinitions) {
      const tests = argDefinitions.map(createArgTester$1);
      return function (args) {
          while (args.length < argDefinitions.length) {
              args.push(undefined);
          }
          return args.reduce((result, arg, index) => {
              if (result === false) {
                  return result;
              }
              const test = tests[index];
              const conversion = test ? test(arg) : arg;
              if (conversion === FALSE$1) {
                  return false;
              }
              result.push(conversion);
              return result;
          }, []);
      };
  }
  /**
   * @param {Function} fnDefinition
   * @param {string} type
   * @param {string[]} [parameterNames]
   *
   * @return {Function}
   */
  function listFn$1(fnDefinition, type, parameterNames = null) {
      const tester = createArgTester$1(type);
      const wrappedFn = function (...args) {
          if (args.length === 0) {
              return null;
          }
          // unwrap first arg
          if (Array.isArray(args[0]) && args.length === 1) {
              args = args[0];
          }
          if (!args.every(arg => tester(arg) !== FALSE$1)) {
              return null;
          }
          return fnDefinition(args);
      };
      wrappedFn.$args = parameterNames || parseParameterNames$1(fnDefinition);
      return wrappedFn;
  }
  /**
   * @param {Function} fnDefinition
   * @param {string[]} argDefinitions
   * @param {string[]} [parameterNames]
   *
   * @return {Function}
   */
  function fn$1(fnDefinition, argDefinitions, parameterNames = null) {
      const checkArgs = createArgsValidator$1(argDefinitions);
      parameterNames = parameterNames || parseParameterNames$1(fnDefinition);
      const wrappedFn = function (...args) {
          const convertedArgs = checkArgs(args);
          if (!convertedArgs) {
              return null;
          }
          return fnDefinition(...convertedArgs);
      };
      wrappedFn.$args = parameterNames;
      return wrappedFn;
  }
  function meets$1(a, b) {
      return [
          (a.end === b.start),
          (a['end included'] === true),
          (b['start included'] === true)
      ].every(v => v);
  }
  function before$1(a, b) {
      if (a instanceof Range$2 && b instanceof Range$2) {
          return (a.end < b.start || (!a['end included'] || !b['start included']) && a.end == b.start);
      }
      if (a instanceof Range$2) {
          return (a.end < b || (!a['end included'] && a.end === b));
      }
      if (b instanceof Range$2) {
          return (b.start > a || (!b['start included'] && b.start === a));
      }
      return a < b;
  }
  function sum$1(list) {
      return list.reduce((sum, el) => sum === null ? el : sum + el, null);
  }
  function flatten$2([x, ...xs]) {
      return (x !== undefined
          ? [...Array.isArray(x) ? flatten$2(x) : [x], ...flatten$2(xs)]
          : []);
  }
  function toKeyString$1(key) {
      if (typeof key === 'string' && /\W/.test(key)) {
          return toString$1(key, true);
      }
      return key;
  }
  function toDeepString$1(obj) {
      return toString$1(obj, true);
  }
  function escapeStr$1(str) {
      return str.replace(/("|\\)/g, '\\$1');
  }
  function toString$1(obj, wrap = false) {
      var _a, _b, _c, _d;
      const type = getType$1(obj);
      if (type === 'nil') {
          return 'null';
      }
      if (type === 'string') {
          return wrap ? `"${escapeStr$1(obj)}"` : obj;
      }
      if (type === 'boolean' || type === 'number') {
          return String(obj);
      }
      if (type === 'list') {
          return '[' + obj.map(toDeepString$1).join(', ') + ']';
      }
      if (type === 'context') {
          return '{' + Object.entries(obj).map(([key, value]) => {
              return toKeyString$1(key) + ': ' + toDeepString$1(value);
          }).join(', ') + '}';
      }
      if (type === 'duration') {
          return obj.shiftTo('years', 'months', 'days', 'hours', 'minutes', 'seconds').normalize().toISO();
      }
      if (type === 'date time') {
          if ((_a = obj.zone) === null || _a === void 0 ? void 0 : _a.zoneName) {
              return obj.toISO({ suppressMilliseconds: true, includeOffset: false }) + '@' + ((_b = obj.zone) === null || _b === void 0 ? void 0 : _b.zoneName);
          }
          return obj.toISO({ suppressMilliseconds: true });
      }
      if (type === 'date') {
          return obj.toISODate();
      }
      if (type === 'range') {
          return '<range>';
      }
      if (type === 'time') {
          if ((_c = obj.zone) === null || _c === void 0 ? void 0 : _c.zoneName) {
              return obj.toISOTime({ suppressMilliseconds: true, includeOffset: false }) + '@' + ((_d = obj.zone) === null || _d === void 0 ? void 0 : _d.zoneName);
          }
          return obj.toISOTime({ suppressMilliseconds: true });
      }
      if (type === 'function') {
          return '<function>';
      }
      throw notImplemented$1('string(' + type + ')');
  }
  function countSymbols$1(str) {
      // cf. https://mathiasbynens.be/notes/javascript-unicode
      return str.replace(/[\uD800-\uDBFF][\uDC00-\uDFFF]/g, '_').length;
  }
  function round$2(n) {
      const integral = Math.trunc(n);
      if (n - integral > .5) {
          return integral + 1;
      }
      else {
          return integral;
      }
  }
  // adapted from https://stackoverflow.com/a/53577159
  function stddev$1(array) {
      const n = array.length;
      const mean = array.reduce((a, b) => a + b) / n;
      return Math.sqrt(array.map(x => Math.pow(x - mean, 2)).reduce((a, b) => a + b) / (n - 1));
  }
  function median$1(array) {
      const n = array.length;
      const sorted = array.slice().sort();
      const mid = n / 2 - 1;
      const index = Math.ceil(mid);
      // even
      if (mid === index) {
          return (sorted[index] + sorted[index + 1]) / 2;
      }
      // uneven
      return sorted[index];
  }
  function mode$1(array) {
      if (array.length < 2) {
          return array;
      }
      const buckets = {};
      for (const n of array) {
          buckets[n] = (buckets[n] || 0) + 1;
      }
      const sorted = Object.entries(buckets).sort((a, b) => b[1] - a[1]);
      return sorted.filter(s => s[1] === sorted[0][1]).map(e => +e[0]);
  }
  function ifValid$1(o) {
      return o.isValid ? o : null;
  }

  function parseExpressions$1(expression, context = {}) {
      return parser$2.configure({
          top: 'Expressions',
          contextTracker: trackVariables$1(context)
      }).parse(expression);
  }
  function parseUnaryTests$1(expression, context = {}) {
      return parser$2.configure({
          top: 'UnaryTests',
          contextTracker: trackVariables$1(context)
      }).parse(expression);
  }

  let Interpreter$1 = class Interpreter {
      _buildExecutionTree(tree, input) {
          const root = { args: [], nodeInput: input };
          const stack = [root];
          tree.iterate({
              enter(nodeRef) {
                  const { isError, isSkipped } = nodeRef.type;
                  const { from, to } = nodeRef;
                  if (isError) {
                      throw new Error(`Statement unparseable at [${from}, ${to}]`);
                  }
                  if (isSkipped) {
                      return false;
                  }
                  const nodeInput = input.slice(from, to);
                  stack.push({
                      nodeInput,
                      args: []
                  });
              },
              leave(nodeRef) {
                  if (nodeRef.type.isSkipped) {
                      return;
                  }
                  const { nodeInput, args } = stack.pop();
                  const parent = stack[stack.length - 1];
                  const expr = evalNode$1(nodeRef, nodeInput, args);
                  parent.args.push(expr);
              }
          });
          return root.args[root.args.length - 1];
      }
      evaluate(expression, context = {}) {
          const parseTree = parseExpressions$1(expression, context);
          const root = this._buildExecutionTree(parseTree, expression);
          return {
              parseTree,
              root
          };
      }
      unaryTest(expression, context = {}) {
          const parseTree = parseUnaryTests$1(expression, context);
          const root = this._buildExecutionTree(parseTree, expression);
          return {
              parseTree,
              root
          };
      }
  };
  const interpreter$1 = new Interpreter$1();
  function unaryTest(expression, context = {}) {
      const value = context['?'] || null;
      const { root } = interpreter$1.unaryTest(expression, context);
      // root = fn(ctx) => test(val)
      const test = root(context);
      return test(value);
  }
  function evaluate$2(expression, context = {}) {
      const { root } = interpreter$1.evaluate(expression, context);
      // root = [ fn(ctx) ]
      const results = root(context);
      if (results.length === 1) {
          return results[0];
      }
      else {
          return results;
      }
  }
  function evalNode$1(node, input, args) {
      switch (node.name) {
          case 'ArithOp': return (context) => {
              const nullable = (op, types = ['number']) => (a, b) => {
                  const left = a(context);
                  const right = b(context);
                  if (isArray$2(left)) {
                      return null;
                  }
                  if (isArray$2(right)) {
                      return null;
                  }
                  const leftType = getType$1(left);
                  const rightType = getType$1(right);
                  if (leftType !== rightType ||
                      !types.includes(leftType)) {
                      return null;
                  }
                  return op(left, right);
              };
              switch (input) {
                  case '+': return nullable((a, b) => a + b, ['string', 'number']);
                  case '-': return nullable((a, b) => a - b);
                  case '*': return nullable((a, b) => a * b);
                  case '/': return nullable((a, b) => !b ? null : a / b);
                  case '**':
                  case '^': return nullable((a, b) => Math.pow(a, b));
              }
          };
          case 'CompareOp': return tag$1(() => {
              switch (input) {
                  case '>': return (b) => createRange$1(b, null, false, false);
                  case '>=': return (b) => createRange$1(b, null, true, false);
                  case '<': return (b) => createRange$1(null, b, false, false);
                  case '<=': return (b) => createRange$1(null, b, false, true);
                  case '=': return (b) => (a) => equals$1(a, b);
                  case '!=': return (b) => (a) => !equals$1(a, b);
              }
          }, Test$1('boolean'));
          case 'Wildcard': return (_context) => true;
          case 'null': return (_context) => {
              return null;
          };
          case 'Disjunction': return tag$1((context) => {
              const left = args[0](context);
              const right = args[2](context);
              const matrix = [
                  [true, true, true],
                  [true, false, true],
                  [true, null, true],
                  [false, true, true],
                  [false, false, false],
                  [false, null, null],
                  [null, true, true],
                  [null, false, null],
                  [null, null, null],
              ];
              const a = typeof left === 'boolean' ? left : null;
              const b = typeof right === 'boolean' ? right : null;
              return matrix.find(el => el[0] === a && el[1] === b)[2];
          }, Test$1('boolean'));
          case 'Conjunction': return tag$1((context) => {
              const left = args[0](context);
              const right = args[2](context);
              const matrix = [
                  [true, true, true],
                  [true, false, false],
                  [true, null, null],
                  [false, true, false],
                  [false, false, false],
                  [false, null, false],
                  [null, true, null],
                  [null, false, false],
                  [null, null, null],
              ];
              const a = typeof left === 'boolean' ? left : null;
              const b = typeof right === 'boolean' ? right : null;
              return matrix.find(el => el[0] === a && el[1] === b)[2];
          }, Test$1('boolean'));
          case 'Context': return (context) => {
              return args.slice(1, -1).reduce((obj, arg) => {
                  const [key, value] = arg(Object.assign(Object.assign({}, context), obj));
                  return Object.assign(Object.assign({}, obj), { [key]: value });
              }, {});
          };
          case 'FunctionBody': return args[0];
          case 'FormalParameters': return args;
          case 'FormalParameter': return args[0];
          case 'ParameterName': return args.join(' ');
          case 'FunctionDefinition': return (context) => {
              const parameterNames = args[2];
              const fnBody = args[4];
              return wrapFunction$1((...args) => {
                  const fnContext = parameterNames.reduce((context, name, idx) => {
                      // support positional parameters
                      context[name] = args[idx];
                      return context;
                  }, Object.assign({}, context));
                  return fnBody(fnContext);
              }, parameterNames);
          };
          case 'ContextEntry': return (context) => {
              const key = typeof args[0] === 'function' ? args[0](context) : args[0];
              const value = args[1](context);
              return [key, value];
          };
          case 'Key': return args[0];
          case 'Identifier': return input;
          case 'SpecialFunctionName': return (context) => getBuiltin$1(input);
          // preserve spaces in name, but compact multiple
          // spaces into one (token)
          case 'Name': return input.replace(/\s{2,}/g, ' ');
          case 'VariableName': return (context) => {
              const name = args.join(' ');
              return getBuiltin$1(name) || getFromContext$1(name, context);
          };
          case 'QualifiedName': return (context) => {
              return args.reduce((context, arg) => arg(context), context);
          };
          case '?': return (context) => getFromContext$1('?', context);
          // expression
          // expression ".." expression
          case 'IterationContext': return (context) => {
              const a = args[0](context);
              const b = args[1] && args[1](context);
              return b ? createRange$1(a, b) : a;
          };
          case 'Type': return args[0];
          case 'InExpressions': return (context) => {
              const iterationContexts = args.map(ctx => ctx(context));
              if (iterationContexts.some(ctx => getType$1(ctx) !== 'list')) {
                  return null;
              }
              return cartesianProduct$1(iterationContexts).map(ctx => {
                  if (!isArray$2(ctx)) {
                      ctx = [ctx];
                  }
                  return Object.assign({}, context, ...ctx);
              });
          };
          // Name kw<"in"> Expr
          case 'InExpression': return (context) => {
              return extractValue$1(context, args[0], args[2]);
          };
          case 'SpecialType': throw notImplemented$1('SpecialType');
          case 'InstanceOfExpression': return tag$1((context) => {
              const a = args[0](context);
              const b = args[3](context);
              return a instanceof b;
          }, Test$1('boolean'));
          case 'every': return tag$1((context) => {
              return (_contexts, _condition) => {
                  const contexts = _contexts(context);
                  if (getType$1(contexts) !== 'list') {
                      return contexts;
                  }
                  return contexts.every(ctx => isTruthy$1(_condition(ctx)));
              };
          }, Test$1('boolean'));
          case 'some': return tag$1((context) => {
              return (_contexts, _condition) => {
                  const contexts = _contexts(context);
                  if (getType$1(contexts) !== 'list') {
                      return contexts;
                  }
                  return contexts.some(ctx => isTruthy$1(_condition(ctx)));
              };
          }, Test$1('boolean'));
          case 'NumericLiteral': return tag$1((_context) => input.includes('.') ? parseFloat(input) : parseInt(input), 'number');
          case 'BooleanLiteral': return tag$1((_context) => input === 'true' ? true : false, 'boolean');
          case 'StringLiteral': return tag$1((_context) => parseString$1(input), 'string');
          case 'PositionalParameters': return (context) => args.map(arg => arg(context));
          case 'NamedParameter': return (context) => {
              const name = args[0];
              const value = args[1](context);
              return [name, value];
          };
          case 'NamedParameters': return (context) => args.reduce((args, arg) => {
              const [name, value] = arg(context);
              args[name] = value;
              return args;
          }, {});
          case 'DateTimeConstructor': return (context) => {
              return getBuiltin$1(input);
          };
          case 'DateTimeLiteral': return (context) => {
              // AtLiteral
              if (args.length === 1) {
                  return args[0](context);
              }
              // FunctionInvocation
              else {
                  const wrappedFn = wrapFunction$1(args[0](context));
                  if (!wrappedFn) {
                      throw new Error(`Failed to evaluate ${input}: Target is not a function`);
                  }
                  const contextOrArgs = args[2](context);
                  return wrappedFn.invoke(contextOrArgs);
              }
          };
          case 'AtLiteral': return (context) => {
              const wrappedFn = wrapFunction$1(getBuiltin$1('@'));
              if (!wrappedFn) {
                  throw new Error(`Failed to evaluate ${input}: Target is not a function`);
              }
              return wrappedFn.invoke([args[0](context)]);
          };
          case 'FunctionInvocation': return (context) => {
              const wrappedFn = wrapFunction$1(args[0](context));
              if (!wrappedFn) {
                  throw new Error(`Failed to evaluate ${input}: Target is not a function`);
              }
              const contextOrArgs = args[2](context);
              return wrappedFn.invoke(contextOrArgs);
          };
          case 'IfExpression': return (function () {
              const ifCondition = args[1];
              const thenValue = args[3];
              const elseValue = args[5];
              const type = coalecenseTypes$1(thenValue, elseValue);
              return tag$1((context) => {
                  if (isTruthy$1(ifCondition(context))) {
                      return thenValue(context);
                  }
                  else {
                      return elseValue ? elseValue(context) : null;
                  }
              }, type);
          })();
          case 'Parameters': return args.length === 3 ? args[1] : (_context) => [];
          case 'Comparison': return (context) => {
              const operator = args[1];
              // expression !compare kw<"in"> PositiveUnaryTest |
              // expression !compare kw<"in"> !unaryTest "(" PositiveUnaryTests ")"
              if (operator === 'in') {
                  return compareIn$1(args[0](context), (args[3] || args[2])(context));
              }
              // expression !compare kw<"between"> expression kw<"and"> expression
              if (operator === 'between') {
                  const start = args[2](context);
                  const end = args[4](context);
                  if (start === null || end === null) {
                      return null;
                  }
                  return createRange$1(start, end).includes(args[0](context));
              }
              // expression !compare CompareOp<"=" | "!="> expression |
              // expression !compare CompareOp<Gt | Gte | Lt | Lte> expression |
              const left = args[0](context);
              const right = args[2](context);
              const test = operator()(right);
              return compareValue$1(test, left);
          };
          case 'QuantifiedExpression': return (context) => {
              const testFn = args[0](context);
              const contexts = args[1];
              const condition = args[3];
              return testFn(contexts, condition);
          };
          // DMN 1.2 - 10.3.2.14
          // kw<"for"> commaSep1<InExpression<IterationContext>> kw<"return"> expression
          case 'ForExpression': return (context) => {
              const extractor = args[args.length - 1];
              const iterationContexts = args[1](context);
              if (getType$1(iterationContexts) !== 'list') {
                  return iterationContexts;
              }
              const partial = [];
              for (const ctx of iterationContexts) {
                  partial.push(extractor(Object.assign(Object.assign({}, ctx), { partial })));
              }
              return partial;
          };
          case 'ArithmeticExpression': return (function () {
              // binary expression (a + b)
              if (args.length === 3) {
                  const [a, op, b] = args;
                  return tag$1((context) => {
                      return op(context)(a, b);
                  }, coalecenseTypes$1(a, b));
              }
              // unary expression (-b)
              if (args.length === 2) {
                  const [op, value] = args;
                  return tag$1((context) => {
                      return op(context)(() => 0, value);
                  }, value.type);
              }
          })();
          case 'PositiveUnaryTest': return args[0];
          case 'ParenthesizedExpression': return args[1];
          case 'PathExpression': return (context) => {
              const pathTarget = args[0](context);
              const pathProp = args[1];
              if (isArray$2(pathTarget)) {
                  return coerceSingleton$1(pathTarget.map(pathProp));
              }
              else {
                  return pathProp(pathTarget);
              }
          };
          // expression !filter "[" expression "]"
          case 'FilterExpression': return (context) => {
              const target = args[0](context);
              const filterFn = args[2];
              const filterTarget = isArray$2(target) ? target : [target];
              // null[..]
              if (target === null) {
                  return null;
              }
              // a[1]
              if (filterFn.type === 'number') {
                  const idx = filterFn(context);
                  const value = filterTarget[idx < 0 ? filterTarget.length + idx : idx - 1];
                  if (typeof value === 'undefined') {
                      return null;
                  }
                  else {
                      return value;
                  }
              }
              // a[true]
              if (filterFn.type === 'boolean') {
                  if (filterFn(context)) {
                      return filterTarget;
                  }
                  else {
                      return [];
                  }
              }
              if (filterFn.type === 'string') {
                  const value = filterFn(context);
                  return filterTarget.filter(el => el === value);
              }
              // a[test]
              return filterTarget.map(el => {
                  const iterationContext = Object.assign(Object.assign(Object.assign({}, context), { item: el }), el);
                  let result = filterFn(iterationContext);
                  // test is fn(val) => boolean SimpleUnaryTest
                  if (typeof result === 'function') {
                      result = result(el);
                  }
                  if (result instanceof Range$2) {
                      result = result.includes(el);
                  }
                  if (result === true) {
                      return el;
                  }
                  return result;
              }).filter(isTruthy$1);
          };
          case 'SimplePositiveUnaryTest': return tag$1((context) => {
              // <Interval>
              if (args.length === 1) {
                  return args[0](context);
              }
              // <CompareOp> <Expr>
              return args[0](context)(args[1](context));
          }, 'test');
          case 'List': return (context) => {
              return args.slice(1, -1).map(arg => arg(context));
          };
          case 'Interval': return tag$1((context) => {
              const left = args[1](context);
              const right = args[2](context);
              const startIncluded = left !== null && args[0] === '[';
              const endIncluded = right !== null && args[3] === ']';
              return createRange$1(left, right, startIncluded, endIncluded);
          }, Test$1('boolean'));
          case 'PositiveUnaryTests':
          case 'Expressions': return (context) => {
              return args.map(a => a(context));
          };
          case 'UnaryTests': return (context) => {
              return (value = null) => {
                  const negate = args[0] === 'not';
                  const tests = negate ? args.slice(2, -1) : args;
                  const matches = tests.map(test => test(context)).flat(1).map(test => {
                      if (isArray$2(test)) {
                          return test.includes(value);
                      }
                      if (test === null) {
                          return null;
                      }
                      if (typeof test === 'boolean') {
                          return test;
                      }
                      return compareValue$1(test, value);
                  }).reduce(combineResult$1, undefined);
                  return matches === null ? null : (negate ? !matches : matches);
              };
          };
          default: return node.name;
      }
  }
  function getBuiltin$1(name, _context) {
      return getFromContext$1(name, builtins$1);
  }
  function extractValue$1(context, prop, _target) {
      const target = _target(context);
      if (['list', 'range'].includes(getType$1(target))) {
          return target.map(t => ({ [prop]: t }));
      }
      return null;
  }
  function compareIn$1(value, tests) {
      if (!isArray$2(tests)) {
          if (getType$1(tests) === 'nil') {
              return null;
          }
          tests = [tests];
      }
      return tests.some(test => compareValue$1(test, value));
  }
  function compareValue$1(test, value) {
      if (typeof test === 'function') {
          return test(value);
      }
      if (test instanceof Range$2) {
          return test.includes(value);
      }
      return equals$1(test, value);
  }
  const chars$2 = Array.from('abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ');
  function isTyped$1(type, values) {
      return (values.some(e => getType$1(e) === type) &&
          values.every(e => e === null || getType$1(e) === type));
  }
  function createRange$1(start, end, startIncluded = true, endIncluded = true) {
      if (isTyped$1('string', [start, end])) {
          return createStringRange$1(start, end, startIncluded, endIncluded);
      }
      if (isTyped$1('number', [start, end])) {
          return createNumberRange$1(start, end, startIncluded, endIncluded);
      }
      if (isTyped$1('duration', [start, end])) {
          throw notImplemented$1('range<duration>');
      }
      if (isTyped$1('time', [start, end])) {
          throw notImplemented$1('range<time>');
      }
      if (isTyped$1('date time', [start, end])) {
          throw notImplemented$1('range<date and time>');
      }
      if (isTyped$1('date', [start, end])) {
          throw notImplemented$1('range<date>');
      }
      throw new Error(`unsupported range: ${start}..${end}`);
  }
  function noopMap$1() {
      return () => {
          throw new Error('unsupported range operation: map');
      };
  }
  function valuesMap$1(values) {
      return (fn) => values.map(fn);
  }
  function valuesIncludes$1(values) {
      return (value) => values.includes(value);
  }
  function numberMap$1(start, end, startIncluded, endIncluded) {
      const direction = start > end ? -1 : 1;
      return (fn) => {
          const result = [];
          for (let i = start;; i += direction) {
              if (i === 0 && !startIncluded) {
                  continue;
              }
              if (i === end && !endIncluded) {
                  break;
              }
              result.push(fn(i));
              if (i === end) {
                  break;
              }
          }
          return result;
      };
  }
  function includesStart$1(n, inclusive) {
      if (inclusive) {
          return (value) => n <= value;
      }
      else {
          return (value) => n < value;
      }
  }
  function includesEnd$1(n, inclusive) {
      if (inclusive) {
          return (value) => n >= value;
      }
      else {
          return (value) => n > value;
      }
  }
  function anyIncludes$1(start, end, startIncluded, endIncluded) {
      let tests = [];
      if (start !== null && end !== null) {
          if (start > end) {
              tests = [
                  includesStart$1(end, endIncluded),
                  includesEnd$1(start, startIncluded)
              ];
          }
          else {
              tests = [
                  includesStart$1(start, startIncluded),
                  includesEnd$1(end, endIncluded)
              ];
          }
      }
      else if (end !== null) {
          tests = [
              includesEnd$1(end, endIncluded)
          ];
      }
      else if (start !== null) {
          tests = [
              includesStart$1(start, startIncluded)
          ];
      }
      return (value) => tests.every(t => t(value));
  }
  function createStringRange$1(start, end, startIncluded = true, endIncluded = true) {
      if (start !== null && !chars$2.includes(start)) {
          throw new Error('illegal range start: ' + start);
      }
      if (end !== null && !chars$2.includes(end)) {
          throw new Error('illegal range end: ' + end);
      }
      let values;
      if (start !== null && end !== null) {
          let startIdx = chars$2.indexOf(start);
          let endIdx = chars$2.indexOf(end);
          const direction = startIdx > endIdx ? -1 : 1;
          if (startIncluded === false) {
              startIdx += direction;
          }
          if (endIncluded === false) {
              endIdx -= direction;
          }
          values = chars$2.slice(startIdx, endIdx + 1);
      }
      const map = values ? valuesMap$1(values) : noopMap$1();
      const includes = values ? valuesIncludes$1(values) : anyIncludes$1(start, end, startIncluded, endIncluded);
      return new Range$2({
          start,
          end,
          'start included': startIncluded,
          'end included': endIncluded,
          map,
          includes
      });
  }
  function createNumberRange$1(start, end, startIncluded, endIncluded) {
      const map = start !== null && end !== null ? numberMap$1(start, end, startIncluded, endIncluded) : noopMap$1();
      const includes = anyIncludes$1(start, end, startIncluded, endIncluded);
      return new Range$2({
          start,
          end,
          'start included': startIncluded,
          'end included': endIncluded,
          map,
          includes
      });
  }
  function cartesianProduct$1(arrays) {
      if (arrays.some(arr => getType$1(arr) === 'nil')) {
          return null;
      }
      const f = (a, b) => [].concat(...a.map(d => b.map(e => [].concat(d, e))));
      const cartesian = (a, b, ...c) => (b ? cartesian(f(a, b), ...c) : a || []);
      return cartesian(...arrays);
  }
  function coalecenseTypes$1(a, b) {
      if (!b) {
          return a.type;
      }
      if (a.type === b.type) {
          return a.type;
      }
      return 'any';
  }
  function tag$1(fn, type) {
      return Object.assign(fn, {
          type,
          toString() {
              return `TaggedFunction[${type}] ${Function.prototype.toString.call(fn)}`;
          }
      });
  }
  function combineResult$1(result, match) {
      if (!result) {
          return match;
      }
      return result;
  }
  function isTruthy$1(obj) {
      return obj !== false && obj !== null;
  }
  function Test$1(type) {
      return `Test<${type}>`;
  }
  /**
   * @param {Function} fn
   * @param {string[]} [parameterNames]
   *
   * @return {FunctionWrapper}
   */
  function wrapFunction$1(fn, parameterNames = null) {
      if (!fn) {
          return null;
      }
      if (fn instanceof FunctionWrapper$1) {
          return fn;
      }
      if (fn instanceof Range$2) {
          return new FunctionWrapper$1((value) => fn.includes(value), ['value']);
      }
      return new FunctionWrapper$1(fn, parameterNames || parseParameterNames$1(fn));
  }
  function coerceSingleton$1(values) {
      if (Array.isArray(values) && values.length === 1) {
          return values[0];
      }
      else {
          return values;
      }
  }
  function parseString$1(str) {
      if (str.startsWith('"')) {
          str = str.slice(1);
      }
      if (str.endsWith('"')) {
          str = str.slice(0, -1);
      }
      return str.replace(/(\\")|(\\\\)|(\\u[a-fA-F0-9]{5,6})|((?:\\u[a-fA-F0-9]{1,4})+)/ig, function (substring, ...groups) {
          const [quotes, escape, codePoint, charCodes] = groups;
          if (quotes) {
              return '"';
          }
          if (escape) {
              return '\\';
          }
          const escapePattern = /\\u([a-fA-F0-9]+)/ig;
          if (codePoint) {
              const codePointMatch = escapePattern.exec(codePoint);
              return String.fromCodePoint(parseInt(codePointMatch[1], 16));
          }
          if (charCodes) {
              const chars = [];
              let charCodeMatch;
              while ((charCodeMatch = escapePattern.exec(substring)) !== null) {
                  chars.push(parseInt(charCodeMatch[1], 16));
              }
              return String.fromCharCode(...chars);
          }
          throw new Error('illegal match');
      });
  }

  // This file was generated by lezer-generator. You probably shouldn't edit it.
  const propertyIdentifier = 119,
    identifier = 120,
    nameIdentifier = 121,
    insertSemi = 122,
    expression0 = 126,
    ForExpression = 4,
    forExpressionStart = 129,
    ForInExpression = 7,
    Name = 8,
    Identifier = 9,
    AdditionalIdentifier = 10,
    forExpressionBodyStart = 137,
    IfExpression = 18,
    ifExpressionStart = 138,
    QuantifiedExpression = 22,
    quantifiedExpressionStart = 139,
    QuantifiedInExpression = 26,
    PositiveUnaryTest = 36,
    ArithmeticExpression = 40,
    arithmeticPlusStart = 143,
    arithmeticTimesStart = 144,
    arithmeticExpStart = 145,
    arithmeticUnaryStart = 146,
    VariableName = 47,
    PathExpression = 67,
    pathExpressionStart = 151,
    FilterExpression = 69,
    filterExpressionStart = 152,
    FunctionInvocation = 71,
    functionInvocationStart = 153,
    ParameterName = 75,
    nil = 158,
    NumericLiteral = 78,
    StringLiteral = 79,
    BooleanLiteral = 80,
    List = 88,
    listStart = 169,
    FunctionDefinition = 89,
    functionDefinitionStart = 171,
    Context = 96,
    contextStart = 173,
    ContextEntry = 97,
    PropertyName = 99,
    PropertyIdentifier = 100;

  /* global console,process */

  const LOG_PARSE = typeof process != 'undefined' && process.env && /\bfparse(:dbg)?\b/.test(process.env.LOG);
  const LOG_PARSE_DEBUG = typeof process != 'undefined' && process.env && /\bfparse:dbg\b/.test(process.env.LOG);
  const LOG_VARS = typeof process != 'undefined' && process.env && /\bcontext\b/.test(process.env.LOG);

  const spaceChars = [
    9, 11, 12, 32, 133, 160,
    5760, 8192, 8193, 8194, 8195, 8196, 8197, 8198,
    8199, 8200, 8201, 8202, 8232, 8233, 8239, 8287, 12288
  ];

  const newlineChars = chars$1('\n\r');

  const additionalNameChars = chars$1("'./-+*");

  /**
   * @param { string } str
   * @return { number[] }
   */
  function chars$1(str) {
    return Array.from(str).map(s => s.charCodeAt(0));
  }

  /**
   * @param { number } ch
   * @return { boolean }
   */
  function isStartChar(ch) {
    return (
      ch === 63 // ?
    ) || (
      ch === 95 // _
    ) || (
      ch >= 65 && ch <= 90 // A-Z
    ) || (
      ch >= 97 && ch <= 122 // a-z
    ) || (
      ch >= 161 && !isPartChar(ch) && !isSpace(ch)
    );
  }

  /**
   * @param { number } ch
   * @return { boolean }
   */
  function isAdditional(ch) {
    return additionalNameChars.includes(ch);
  }

  /**
   * @param { number } ch
   * @return { boolean }
   */
  function isPartChar(ch) {
    return (
      ch >= 48 && ch <= 57 // 0-9
    ) || (
      ch === 0xB7
    ) || (
      ch >= 0x0300 && ch <= 0x036F
    ) || (
      ch >= 0x203F && ch <= 0x2040
    );
  }

  /**
   * @param { number } ch
   * @return { boolean }
   */
  function isSpace(ch) {
    return spaceChars.includes(ch);
  }

  // eslint-disable-next-line
  function indent(str, spaces) {
    return spaces.concat(
      str.split(/\n/g).join('\n' + spaces)
    );
  }

  /**
   * @param { import('@lezer/lr').InputStream } input
   * @param  { number } [offset]
   * @param { boolean } [includeOperators]
   *
   * @return { { token: string, offset: number } | null }
   */
  function parseAdditionalSymbol(input, offset = 0) {

    const next = input.peek(offset);

    if (isAdditional(next)) {
      return {
        offset: 1,
        token: String.fromCharCode(next)
      };
    }

    return null;
  }

  /**
   * @param { import('@lezer/lr').InputStream } input
   * @param { number } [offset]
   * @param { boolean } [namePart]
   *
   * @return { { token: string, offset: number } | null }
   */
  function parseIdentifier(input, offset = 0, namePart = false) {
    for (let inside = false, chars = [], i = 0;; i++) {
      const next = input.peek(offset + i);

      if (isStartChar(next) || ((inside || namePart) && isPartChar(next))) {
        if (!inside) {
          inside = true;
        }

        chars.push(next);
      } else {

        if (chars.length) {
          return {
            token: String.fromCharCode(...chars),
            offset: i
          };
        }

        return null;
      }
    }
  }

  /**
   * @param { import('@lezer/lr').InputStream } input
   * @param  { number } offset
   *
   * @return { { token: string, offset: number } | null }
   */
  function parseSpaces(input, offset) {

    for (let inside = false, i = 0;; i++) {
      let next = input.peek(offset + i);

      if (isSpace(next)) {
        if (!inside) {
          inside = true;
        }
      } else {
        if (inside) {
          return {
            token: ' ',
            offset: i
          };
        }

        return null;
      }
    }
  }

  /**
   * Parse a name from the input and return the first match, if any.
   *
   * @param { import('@lezer/lr').InputStream } input
   * @param { Variables } variables
   *
   * @return { { token: string, offset: number, term: number } | null }
   */
  function parseName(input, variables) {
    const contextKeys = variables.contextKeys();

    const start = variables.tokens;

    for (let i = 0, tokens = [], nextMatch = null;;) {

      const namePart = (start.length + tokens.length) > 0;
      const maybeSpace = tokens.length > 0;

      const match = (
        parseIdentifier(input, i, namePart) ||
        namePart && parseAdditionalSymbol(input, i) ||
        maybeSpace && parseSpaces(input, i)
      );

      // match is required
      if (!match) {
        return nextMatch;
      }

      const {
        token,
        offset
      } = match;

      i += offset;

      if (token === ' ') {
        continue;
      }

      tokens = [ ...tokens, token ];

      const name = [ ...start, ...tokens ].join(' ');

      if (contextKeys.some(el => el === name)) {
        const token = tokens[0];

        nextMatch = {
          token,
          offset: token.length,
          term: nameIdentifier
        };
      }

      if (dateTimeIdentifiers.some(el => el === name)) {
        const token = tokens[0];

        // parse date time identifiers as normal
        // identifiers to allow specialization to kick in
        //
        // cf. https://github.com/nikku/lezer-feel/issues/8
        nextMatch = {
          token,
          offset: token.length,
          term: identifier
        };
      }

      if (
        !contextKeys.some(el => el.startsWith(name)) &&
        !dateTimeIdentifiers.some(el => el.startsWith(name))
      ) {
        return nextMatch;
      }
    }

  }

  const identifiersMap = {
    [ identifier ]: 'identifier',
    [ nameIdentifier ]: 'nameIdentifier'
  };

  const identifiers = new ExternalTokenizer((input, stack) => {

    LOG_PARSE_DEBUG && console.log('%s: T <identifier | nameIdentifier>', input.pos);

    const nameMatch = parseName(input, stack.context);

    const start = stack.context.tokens;

    const match = nameMatch || parseIdentifier(input, 0, start.length > 0);

    if (match) {
      input.advance(match.offset);
      input.acceptToken(nameMatch ? nameMatch.term : identifier);

      LOG_PARSE && console.log('%s: MATCH <%s> <%s>', input.pos, nameMatch ? identifiersMap[nameMatch.term] : 'identifier', match.token);
    }
  }, { contextual: true });


  const propertyIdentifiers = new ExternalTokenizer((input, stack) => {

    LOG_PARSE_DEBUG && console.log('%s: T <propertyIdentifier>', input.pos);

    const start = stack.context.tokens;

    const match = parseIdentifier(input, 0, start.length > 0);

    if (match) {
      input.advance(match.offset);
      input.acceptToken(propertyIdentifier);

      LOG_PARSE && console.log('%s: MATCH <propertyIdentifier> <%s>', input.pos, match.token);
    }
  });


  const insertSemicolon = new ExternalTokenizer((input, stack) => {

    LOG_PARSE_DEBUG && console.log('%s: T <insertSemi>', input.pos);

    let offset;
    let insert = false;

    for (offset = 0;; offset++) {
      const char = input.peek(offset);

      if (spaceChars.includes(char)) {
        continue;
      }

      if (newlineChars.includes(char)) {
        insert = true;
      }

      break;
    }

    if (insert) {

      const identifier = parseIdentifier(input, offset + 1);
      const spaces = parseSpaces(input, offset + 1);

      if (spaces || identifier && /^(then|else|return|satisfies)$/.test(identifier.token)) {
        return;
      }

      LOG_PARSE && console.log('%s: MATCH <insertSemi>', input.pos);
      input.acceptToken(insertSemi);
    }
  });

  const prefixedContextStarts = {
    [ functionInvocationStart ]: 'FunctionInvocation',
    [ filterExpressionStart ]: 'FilterExpression',
    [ pathExpressionStart ]: 'PathExpression'
  };

  const contextStarts = {
    [ contextStart ]: 'Context',
    [ functionDefinitionStart ]: 'FunctionDefinition',
    [ forExpressionStart ]: 'ForExpression',
    [ listStart ]: 'List',
    [ ifExpressionStart ]: 'IfExpression',
    [ quantifiedExpressionStart ]: 'QuantifiedExpression'
  };

  const contextEnds = {
    [ Context ]: 'Context',
    [ FunctionDefinition ]: 'FunctionDefinition',
    [ ForExpression ]: 'ForExpression',
    [ List ]: 'List',
    [ IfExpression ]: 'IfExpression',
    [ QuantifiedExpression ]: 'QuantifiedExpression',
    [ PathExpression ]: 'PathExpression',
    [ FunctionInvocation ]: 'FunctionInvocation',
    [ FilterExpression ]: 'FilterExpression',
    [ ArithmeticExpression ]: 'ArithmeticExpression'
  };

  class ValueProducer {

    /**
     * @param { Function } fn
     */
    constructor(fn) {
      this.fn = fn;
    }

    get(variables) {
      return this.fn(variables);
    }

    /**
     * @param { Function }
     *
     * @return { ValueProducer }
     */
    static of(fn) {
      return new ValueProducer(fn);
    }

  }

  const dateTimeLiterals = {
    'date and time': 1,
    'date': 1,
    'time': 1,
    'duration': 1
  };

  const dateTimeIdentifiers = Object.keys(dateTimeLiterals);

  class Variables {

    constructor({
      name = 'Expressions',
      tokens = [],
      children = [],
      parent = null,
      context = { },
      value,
      raw
    } = {}) {
      this.name = name;
      this.tokens = tokens;
      this.children = children;
      this.parent = parent;
      this.context = context;
      this.value = value;
      this.raw = raw;
    }

    enterScope(name) {

      const childScope = this.of({
        name,
        parent: this
      });

      LOG_VARS && console.log('[%s] enter', childScope.path, childScope.context);

      return childScope;
    }

    exitScope(str) {

      if (!this.parent) {
        LOG_VARS && console.log('[%s] NO exit %o\n%s', this.path, this.context, indent(str, '  '));

        return this;
      }

      LOG_VARS && console.log('[%s] exit %o\n%s', this.path, this.context, indent(str, '  '));

      return this.parent.pushChild(this);
    }

    token(part) {

      LOG_VARS && console.log('[%s] token <%s> + <%s>', this.path, this.tokens.join(' '), part);

      return this.assign({
        tokens: [ ...this.tokens, part ]
      });
    }

    literal(value) {

      LOG_VARS && console.log('[%s] literal %o', this.path, value);

      return this.pushChild(this.of({
        name: 'Literal',
        value
      }));
    }

    /**
     * Return computed scope value
     *
     * @return {any}
     */
    computedValue() {
      for (let scope = this;;scope = last(scope.children)) {

        if (!scope) {
          return null;
        }

        if (scope.value) {
          return scope.value;
        }
      }
    }

    contextKeys() {
      return Object.keys(this.context).map(normalizeContextKey);
    }

    get path() {
      return this.parent?.path?.concat(' > ', this.name) || this.name;
    }

    /**
     * Return value of variable.
     *
     * @param { string } variable
     * @return { any } value
     */
    get(variable) {

      const names = [ variable, variable && normalizeContextKey(variable) ];

      const contextKey = Object.keys(this.context).find(
        key => names.includes(normalizeContextKey(key))
      );

      if (typeof contextKey === 'undefined') {
        return undefined;
      }

      const val = this.context[contextKey];

      if (val instanceof ValueProducer) {
        return val.get(this);
      } else {
        return val;
      }
    }

    resolveName() {

      const variable = this.tokens.join(' ');
      const tokens = [];

      const parentScope = this.assign({
        tokens
      });

      const variableScope = this.of({
        name: 'VariableName',
        parent: parentScope,
        value: this.get(variable),
        raw: variable
      });

      LOG_VARS && console.log('[%s] resolve name <%s=%s>', variableScope.path, variable, this.get(variable));

      return parentScope.pushChild(variableScope);
    }

    pushChild(child) {

      if (!child) {
        return this;
      }

      const parent = this.assign({
        children: [ ...this.children, child ]
      });

      child.parent = parent;

      return parent;
    }

    pushChildren(children) {

      let parent = this;

      for (const child of children) {
        parent = parent.pushChild(child);
      }

      return parent;
    }

    declareName() {

      if (this.tokens.length === 0) {
        throw Error('no tokens to declare name');
      }

      const variableName = this.tokens.join(' ');

      LOG_VARS && console.log('[%s] declareName <%s>', this.path, variableName);

      return this.assign({
        tokens: []
      }).pushChild(
        this.of({
          name: 'Name',
          value: variableName
        })
      );
    }

    define(name, value) {

      if (typeof name !== 'string') {
        LOG_VARS && console.log('[%s] no define <%s=%s>', this.path, name, value);

        return this;
      }

      LOG_VARS && console.log('[%s] define <%s=%s>', this.path, name, value);

      const context = {
        ...this.context,
        [name]: value
      };

      return this.assign({
        context
      });
    }

    /**
     * @param { Record<string, any> } [options]
     *
     * @return { Variables }
     */
    assign(options = {}) {

      return Variables.of({
        ...this,
        ...options
      });
    }

    /**
     * @param { Record<string, any> } [options]
     *
     * @return { Variables }
     */
    of(options = {}) {

      const defaultOptions = {
        context: this.context,
        parent: this.parent
      };

      return Variables.of({
        ...defaultOptions,
        ...options
      });
    }

    static of(options) {
      const {
        name,
        tokens = [],
        children = [],
        parent = null,
        context = {},
        value,
        raw
      } = options;

      return new Variables({
        name,
        tokens: [ ...tokens ],
        children: [ ...children ],
        context: {
          ...context
        },
        parent,
        value,
        raw
      });
    }

  }

  /**
   * @param { string } name
   *
   * @return { string } normalizedName
   */
  function normalizeContextKey(name) {
    return name.replace(/\s*([./\-'+*])\s*/g, ' $1 ').replace(/\s{2,}/g, ' ').trim();
  }

  /**
   * Wrap children of variables under the given named child.
   *
   * @param { Variables } variables
   * @param { string } name
   * @param { string } code
   * @return { Variables }
   */
  function wrap(variables, scopeName, code) {

    const parts = variables.children.filter(c => c.name !== scopeName);
    const children = variables.children.filter(c => c.name === scopeName);

    const namePart = parts[0];
    const valuePart = parts[Math.max(1, parts.length - 1)];

    const name = namePart.computedValue();
    const value = valuePart?.computedValue() || null;

    return variables
      .assign({
        children
      })
      .enterScope(scopeName)
      .pushChildren(parts)
      .exitScope(code)
      .define(name, value);
  }

  /**
   * @param { any } context
   *
   * @return { ContextTracker<Variables> }
   */
  function trackVariables(context = {}) {

    const start = Variables.of({
      context
    });

    return new ContextTracker({
      start,
      reduce(variables, term, stack, input) {

        if (term === IfExpression) {
          const [ thenPart, elsePart ] = variables.children.slice(-2);

          variables = variables.assign({
            value: {
              ...thenPart?.computedValue(),
              ...elsePart?.computedValue()
            }
          });
        }

        if (term === List) {
          variables = variables.assign({
            value: variables.children.reduce((value, child) => {
              return {
                ...value,
                ...child?.computedValue()
              };
            }, {})
          });
        }

        if (term === FilterExpression) {
          const [ sourcePart, _ ] = variables.children.slice(-2);

          variables = variables.assign({
            value: sourcePart?.computedValue()
          });
        }

        if (term === FunctionInvocation) {

          const [
            name,
            ...args
          ] = variables.children;

          // preserve type information through `get value(context, key)` utility
          if (name?.raw === 'get value') {
            variables = getContextValue(variables, args);
          }
        }

        const start = contextStarts[term];

        if (start) {
          return variables.enterScope(start);
        }

        const prefixedStart = prefixedContextStarts[term];

        // pull <expression> into new <prefixedStart> context
        if (prefixedStart) {

          const {
            children: currentChildren,
            context: currentContext,
          } = variables;

          const children = currentChildren.slice(0, -1);
          const lastChild = last(currentChildren);

          let newContext = null;

          if (term === pathExpressionStart) {
            newContext = lastChild?.computedValue();
          }

          if (term === filterExpressionStart) {
            newContext = {
              ...currentContext,
              ...lastChild?.computedValue(),
              item: lastChild?.computedValue()
            };
          }

          return variables.assign({
            children
          }).enterScope(prefixedStart).pushChild(lastChild).assign({
            context: newContext || currentContext
          });
        }

        const code = input.read(input.pos, stack.pos);

        const end = contextEnds[term];

        if (end) {
          return variables.exitScope(code);
        }

        if (term === ContextEntry) {
          const parts = variables.children.filter(c => c.name !== 'ContextEntry');

          const name = parts[0];
          const value = last(parts);

          return wrap(variables, 'ContextEntry', code).assign(
            {
              value: {
                ...variables.value,
                [name.computedValue()] : value?.computedValue()
              }
            }
          );
        }

        if (
          term === ForInExpression ||
          term === QuantifiedInExpression
        ) {
          return wrap(variables, 'InExpression', code);
        }

        // define <partial> within ForExpression body
        if (term === forExpressionBodyStart) {

          return variables.define(
            'partial',
            ValueProducer.of(variables => {
              return last(variables.children)?.computedValue();
            })
          );
        }

        if (
          term === ParameterName
        ) {
          const name = last(variables.children).computedValue();

          // TODO: attach type information
          return variables.define(name, 1);
        }

        // pull <expression> into ArithmeticExpression child
        if (
          term === arithmeticPlusStart ||
          term === arithmeticTimesStart ||
          term === arithmeticExpStart
        ) {
          const children = variables.children.slice(0, -1);
          const lastChild = last(variables.children);

          return variables.assign({
            children
          }).enterScope('ArithmeticExpression').pushChild(lastChild);
        }

        if (term === arithmeticUnaryStart) {
          return variables.enterScope('ArithmeticExpression');
        }

        if (
          term === Identifier ||
          term === AdditionalIdentifier ||
          term === PropertyIdentifier
        ) {
          return variables.token(code);
        }

        if (
          term === StringLiteral
        ) {
          return variables.literal(code.replace(/^"|"$/g, ''));
        }

        if (term === BooleanLiteral) {
          return variables.literal(code === 'true' ? true : false);
        }

        if (term === NumericLiteral) {
          return variables.literal(parseFloat(code));
        }

        if (term === nil) {
          return variables.literal(null);
        }

        if (
          term === VariableName
        ) {
          return variables.resolveName();
        }

        if (
          term === Name ||
          term === PropertyName
        ) {
          return variables.declareName();
        }

        if (
          term === expression0 ||
          term === PositiveUnaryTest
        ) {
          if (variables.tokens.length > 0) {
            throw new Error('uncleared name');
          }
        }

        if (term === expression0) {

          let parent = variables;

          while (parent.parent) {
            parent = parent.exitScope(code);
          }

          return parent;
        }

        return variables;
      }
    });
  }

  const variableTracker = trackVariables({});


  // helpers //////////////

  function getContextValue(variables, args) {

    if (!args.length) {
      return variables.assign({
        value: null
      });
    }

    if (args[0].name === 'Name') {
      args = extractNamedArgs(args, [ 'm', 'key' ]);
    }

    if (args.length !== 2) {
      return variables.assign({
        value: null
      });
    }

    const [
      context,
      key
    ] = args;

    const keyValue = key?.computedValue();
    const contextValue = context?.computedValue();

    if (
      (!contextValue || typeof contextValue !== 'object') || typeof keyValue !== 'string'
    ) {
      return variables.assign({
        value: null
      });
    }

    return variables.assign({
      value: [ normalizeContextKey(keyValue), keyValue ].reduce((value, keyValue) => {
        if (keyValue in contextValue) {
          return contextValue[keyValue];
        }

        return value;
      }, null)
    });
  }

  function extractNamedArgs(args, argNames) {

    const context = {};

    for (let i = 0; i < args.length; i += 2) {
      const [ name, value ] = args.slice(i, i + 2);

      context[name.value] = value;
    }

    return argNames.map(name => context[name]);
  }

  function last(arr) {
    return arr[arr.length - 1];
  }

  const feelHighlighting = styleTags({
    StringLiteral: tags.string,
    NumericLiteral: tags.number,
    BooleanLiteral: tags.bool,
    'AtLiteral!': tags.special(tags.string),
    CompareOp: tags.compareOperator,
    ArithOp: tags.arithmeticOperator,
    'for if then else some every satisfies between return': tags.controlKeyword,
    'in instance of and or': tags.operatorKeyword,
    function: tags.definitionKeyword,
    as: tags.keyword,
    'Type/...': tags.typeName,
    Wildcard: tags.special,
    null: tags.null,
    LineComment: tags.lineComment,
    BlockComment: tags.blockComment,
    'VariableName! "?"': tags.variableName,
    'DateTimeConstructor! SpecialFunctionName!': tags.function(tags.special(tags.variableName)),
    'List Interval': tags.list,
    Context: tags.definition(tags.literal),
    'Name!': tags.definition(tags.variableName),
    'Key/Name! ContextEntryType/Name!': tags.definition(tags.propertyName),
    'PathExpression/VariableName!': tags.function(tags.propertyName),
    'FormalParameter/ParameterName!': tags.function(tags.definition(tags.variableName)),
    '( )': tags.paren,
    '[ ]': tags.squareBracket,
    '{ }': tags.brace,
    '.': tags.derefOperator,
    ', ;': tags.separator,
    '..': tags.punctuation
  });

  // This file was generated by lezer-generator. You probably shouldn't edit it.
  const spec_identifier = {__proto__:null,for:10, in:30, return:34, if:38, then:40, else:42, some:46, every:48, satisfies:55, or:58, and:62, between:70, instance:86, of:89, days:99, time:101, duration:103, years:105, months:107, date:109, list:115, context:121, function:128, null:154, true:324, false:324, "?":168, external:184, not:207};
  const parser$1 = LRParser.deserialize({
    version: 14,
    states: "IWO`QYOOO$gQYOOOOQU'#Ce'#CeO$qQYO'#C`O%zQ^O'#FOOOQQ'#Fd'#FdO'dQYO'#FdO`QYO'#DUOOQU'#Em'#EmO)QQ^O'#D]OOQO'#Fk'#FkO,PQWO'#DuOOQU'#D|'#D|OOQU'#D}'#D}OOQU'#EO'#EOO,UOWO'#ERO,PQWO'#EPOOQU'#EP'#EPOOQU'#Fq'#FqOOQU'#Fo'#FoOOQQ'#Fv'#FvO.yQYO'#FvO0wQYO'#FvOOQU'#ET'#ETO2sQYO'#EVOOQU'#FQ'#FQO4UQ^O'#FQO5hQYO'#EWO5rQWO'#EXOOQP'#GP'#GPO5wQXO'#E`OOQU'#Fz'#FzOOQU'#FP'#FPOOQQ'#Eh'#EhQ`QYOOOOQQ'#FR'#FROOQQ'#F['#F[O2sQYO'#CnOOQQ'#F]'#F]O$qQYO'#CrO6SQYO'#DvOOQU'#Fp'#FpO6XQYO'#EQOOQO'#EQ'#EQO2sQYO'#EUO`QYO'#ETOOQO'#F}'#F}O7bQYO'#DQO8UQWO'#F`OOQO'#DS'#DSO8aQYO'#FdQOQWOOO8hQWOOO9[QYO'#CdO9iQYO'#FTOOQQ'#Cc'#CcO9nQYO'#FSOOQQ'#Cb'#CbO9vQYO,58zO`QYO,59hOOQQ'#Fa'#FaOOQQ'#Fb'#FbOOQQ'#Fc'#FcO`QYO,59pO`QYO,59pO`QYO,59pOOQQ'#Fi'#FiO$qQYO,5:]OOQQ'#Fj'#FjO2sQYO,5:_OOQQ,5;j,5;jO`QYO,59dO`QYO,59fO2sQYO,59hO;fQYO,59hO;mQYO,59rOOQU,5:h,5:hO;rQ^O,59pOOQU-E8k-E8kO>qQYO'#FlOOQU,5:a,5:aOOQU,5:m,5:mOOQU,5:k,5:kO>{QYO,5:qOOQU,5;l,5;lO?SQYO'#FnO?aQWO,5:rO?fQYO,5:sOOQP'#Ed'#EdO@YQXO'#EcOOQO'#Eb'#EbO@aQWO'#EaO@fQWO'#GQO@nQWO,5:zOOQQ-E8f-E8fO@sQYO,59YO9iQYO'#F_OOQQ'#Cv'#CvO@zQYO'#F^OOQQ'#Cu'#CuOASQYO,59^OAXQYO,5:bOA^QYO,5:lOAcQYO,5:pOAjQ^O,5:oO2sQYO'#ElOCSQWO,5;zO2sQYOOOOQR'#Cf'#CfOOQQ'#Ei'#EiOCyQYO,59OO2sQYO,5;oOOQQ'#FW'#FWO$qQYO'#EjODZQYO,5;nO`QYO1G.fOOQQ'#FZ'#FZOEjQ^O1G/SOI]Q^O1G/[OIgQ^O1G/[OK_Q^O1G/[OOQU1G/w1G/wOLtQYO1G/yOMyQ^O1G/OO!!aQ^O1G/QO!$}QYO1G/SO!%UQYO1G/SOOQU1G/S1G/SO!&tQYO1G/^O!'`Q^O'#CdOOQO'#Dy'#DyO!(rQWO'#DxO!(wQWO'#FmOOQO'#Dw'#DwOOQO'#Dz'#DzO!)PQWO,5<WOOQU'#Fy'#FyOOQU1G0]1G0]O2sQYO'#ErO!)UQWO,5<YOOQU'#F|'#F|OOQU1G0^1G0^O!)aQWO'#EZO!)lQWO'#GOOOQO'#EY'#EYO!)tQWO1G0_OOQP'#Et'#EtO!)yQXO,5:}O2sQYO,5:{O!*QQXO'#EuO!*YQWO,5<lOOQU1G0f1G0fO2sQYO1G.tO2sQYO,5;yO$qQYO'#EkO!*bQYO,5;xO`QYO1G.xO!*jQYO1G/|OOQO1G0W1G0WO2sQYO1G0[OOQO,5;W,5;WOOQO-E8j-E8jO!*oQWOOOOQQ-E8g-E8gO!*tQYO'#ClOOQQ1G1Z1G1ZOOQQ,5;U,5;UOOQQ-E8h-E8hO!+RQ^O7+$QOOQU7+%e7+%eO`QYO7+$nO!,kQWO7+$nO!,pQ^O'#D[OOQU'#DZ'#DZO!/oQYO'#D^O!/tQYO'#D^O!/yQYO'#D^O!0OQ`O'#DfO!0TQ`O'#DiO!0YQ`O'#DmOOQU7+$x7+$xO2sQYO,5:dO$qQYO'#EqO!0_QWO,5<XOOQU1G1r1G1rO!0gQYO,5;^OOQO-E8p-E8pO!&tQYO,5:uO$qQYO'#EsO!0tQWO,5<jO!0|QYO7+%yOOQP-E8r-E8rO!1TQYO1G0gOOQO,5;a,5;aOOQO-E8s-E8sO!1_QYO7+$`O!1fQYO1G1eOOQQ,5;V,5;VOOQQ-E8i-E8iO!1pQ^O7+$dOOQO7+%h7+%hO!4WQYO7+%vO2sQYO,59WO!5lQ^O<<HYOOQU<<HY<<HYO$qQYO'#EnO!7OQ^O,59vO!9}QYO,59xO!:SQYO,59xO!:XQYO,59xO!:^QYO,5:QO$qQYO,5:TO!:xQbO,5:XO!;PQYO1G0OOOQO,5;],5;]OOQO-E8o-E8oOOQO1G0a1G0aOOQO,5;_,5;_OOQO-E8q-E8qO!;ZQ^O'#E]OOQU<<Ie<<IeO`QYO<<IeO`QYO<<GzO!<sQ^O'#FjOOQU'#Fw'#FwOOQU<<Ib<<IbO!?rQYO1G.rOOQU,5;Y,5;YOOQU-E8l-E8lO!?|QYO1G/dOOQU1G/d1G/dO!@RQbO'#D]O!@dQ`O'#D[O!@oQ`O1G/lO!@tQWO'#DlO!@yQ`O'#FeOOQO'#Dk'#DkO!ARQ`O1G/oOOQO'#Dp'#DpO!AWQ`O'#FgOOQO'#Do'#DoO!A`Q`O1G/sOOQUAN?PAN?PO!AeQ^OAN=fOOQU7+%O7+%OO!B}Q`O,59vOOQU7+%W7+%WO!:^QYO,5:WO$qQYO'#EoO!CYQ`O,5<POOQU7+%Z7+%ZO!:^QYO'#EpO!CbQ`O,5<RO!CjQ`O7+%_OOQO1G/r1G/rOOQO,5;Z,5;ZOOQO-E8m-E8mOOQO,5;[,5;[OOQO-E8n-E8nO!&tQYO<<HyOOQUAN>eAN>eO!CoQ^O'#FQO2sQYO'#ETO2sQYO,59hO2sQYO,59pO2sQYO,59pO2sQYO,59pO2sQYO,59dO2sQYO,59fO!EvQYO,59hO!E}QYO,5:oO2sQYO1G.fO!FqQYO1G/SO!HxQYO1G/[O!ISQYO1G/[O!JXQYO1G/OO!KyQYO1G/QO2sQYO1G.xO!LsQYO7+$QO2sQYO7+$nO!MgQYO7+%yO!MqQYO7+$dO!NeQYO<<HYO$qQYO'#EnO# RQYO'#E]O2sQYO<<IeO2sQYO<<GzO# uQYOAN=fO!:^QYO<<HyO2sQYO'#DUO#!iQ^O'#DQO9vQYO,58zO#$RQYO,59^O#$WQYO1G/SO#$_QWO1G0_O#$dQYO7+$`O#$kQ`O7+%_O$qQYO'#C`O$qQYO'#CrO2sQYO,59hO?fQYO,5:sO2sQYO1G.tO#$pQ`O1G/sO#$uQWO'#EXO#$zQYO,59YO!:xQbO,5:XO2sQYO'#CnO#%RQ`O'#Dm",
    stateData: "#%W~O#pOS#qOSPOSQOS~OTsOZUO[TOctOgvOhvOr}OueO!S{O!T{O!UxO!WzO!b!OO!fdO!hfO!oyO!viO#RmO#lQO#mQO$f[O$g]O$h^O$i_O~OTsO[TOctOgvOhvOr&pOueO!S{O!T{O!UxO!WzO!b!OO!fdO!hfO!oyO!viO#RmO#lQO#mQO$f[O$g]O$h^O$i_O~OZ!SO#Z!UO~P!|O#lQO#mQO~OZ!^O[!^O]!_O^!_O_!lOm!iOo!jOq!]Or!]Os!kOy!`O{!mO!h!fO#x!dOu$_X~O#n!hO$u!hOT#rXc#rXg#rXh#rX!S#rX!T#rX!U#rX!W#rX!b#rX!f#rX!o#rX!v#rX#R#rX#j#rX#l#rX#m#rX$f#rX$g#rX$h#rX$i#rX~P$yO$f!nOT$WXZ$WX[$WXc$WXg$WXh$WXr$WXu$WX!S$WX!T$WX!U$WX!W$WX!b$WX!f$WX!h$WX!o$WX!v$WX#R$WX#l$WX#m$WX$g$WX$h$WX$i$WX~O#lQO#mQOT!PXZ!PX[!PX]!PX^!PX_!PXc!PXg!PXh!PXm!PXo!PXq!PXr!PXs!PXu!PXy!PX{!PX!S!PX!T!PX!U!PX!W!PX!b!PX!f!PX!h!PX!o!PX!v!PX#R!PX#j!PX#n!PX#x!PX$f!PX$g!PX$h!PX$i!PX$u!PX#|!PXw!PXd!PX#{!PXa!PX#Q!PXe!PXk!PX~Ou!qO~O$g]O~OT$jXT$lXc$jXc$lXg$jXg$lXh$jXh$lXr$jXr$lXu$jXu$lX!S$jX!S$lX!T$jX!T$lX!U$jX!U$lX!W$jX!W$lX!b$jX!b$lX!f$jX!f$lX!h$jX!h$lX!o$jX!o$lX!v$jX!v$lX#R$jX#R$lX$f$jX$f$lX$g$jX$g$lX$h$jX$h$lX$i$jX$i$lX~OZ$jXZ$lX[$jX[$lX#l$jX#l$lX#m$jX#m$lX~P,ZOT$jXc$jXg$jXh$jXr$jXu$jX!S$jX!T$jX!U$jX!W$jX!b$jX!f$jX!h$jX!o$jX!v$jX#R$jX$f$jX$g$jX$h$jX$i$jX~OT$oXZ$jXZ$oX[$jX[$oXc$oXg$oXh$oXr$oXu$oX!S$oX!T$oX!U$oX!W$oX!b$oX!f$oX!h$oX!o$oX!v$oX#R$oX#l$jX#l$oX#m$jX#m$oX$f$oX$g$oX$h$oX$i$oX~P/gOZUO~P!|O#n!vOZ#tX[#tX]#tX^#tX_#tXm#tXo#tXq#tXr#tXs#tXu#tXy#tX{#tX!f#tX!h#tX#j#tX#x#tX~OT#tXc#tXg#tXh#tX!S#tX!T#tX!U#tX!W#tX!b#tX!o#tX!v#tX#R#tX#l#tX#m#tX#n#tX$f#tX$g#tX$h#tX$i#tX$u#tX~P2zOZUO!f$bP~P!|Ou!yO~O#k!zO$g]O#Q$tP~Oo#XO~Oo#YOu!tX~OZ!^O[!^O]!_O^!_O_&wOm&uOo&vOq&qOr&qOs'gOy!`O{!mO!h!fO#x!dOu$_X~O#jtX#|tXwtX!ftXdtX#{tXatX#QtXetXktX~P6aO#|#]O#j$SXw$SX~O#j#YX~P'dOu#_O~OZ#`O[#`O]#`O^#`O#lQO#mQO#x#`O#y#`O$YWX~O_WXwWX#|WX~P8mO_#dO~O#|#eOa#vX~Oa#hO~OTsOZUO[TOctOgvOhvOr}O!S{O!T{O!UxO!WzO!b!OO!fdO!hfO!oyO!viO#RmO#lQO#mQO$f[O$g]O$h^O$i_O~Ou#rO~P9{O|#tO~O{!mO!h!fO#x!dOTxaZxa[xa]xa^xa_xacxagxahxamxaoxaqxarxasxau$_Xyxa!Sxa!Txa!Uxa!Wxa!bxa!fxa!oxa!vxa#Rxa#jxa#lxa#mxa#nxa$fxa$gxa$hxa$ixa$uxa#|xawxadxa#{xaaxa#Qxaexakxa~OZUOw$bP~P!|Ow#|O~P6aO#|$OO!f$bXw$bX~P6aO!f$QO~O#lQO#mQOw$rP~OZ#`O[#`O]#`O^#`O#k!zO#x#`O#y#`O~O$Y#VX~P?qO$Y$YO~O#|$ZO#Q$tX~O#Q$]O~Od$^O~P6aO#|$`Ok$QX~Ok$bO~O!V$cO~O!S$dO~O#{$eO~P6aOT!wac!wag!wah!wa!S!wa!T!wa!U!wa!W!wa!b!wa!f!wa!o!wa!v!wa#R!wa#j!wa#l!wa#m!wa#n!wa$f!wa$g!wa$h!wa$i!wa$u!wa~P$yO#|#]O#j$Saw$Sa~OZ#`O[#`O]#`O^#`O#lQO#mQO#x#`O#y#`O~O_Wa$YWawWa#|Wa~PC_O#|#eOa#va~OZ!^O[!^O]!_O^!_Oy!`O{!mO!h!fO#x!dO_pimpiopiqpirpispiu$_X!fpi#jpi~OTpicpigpihpi!Spi!Tpi!Upi!Wpi!bpi!opi!vpi#Rpi#lpi#mpi#npi$fpi$gpi$hpi$ipi$upi~PDcOy!`O{!mO!h!fO#x!dOTxiZxi[xi_xicxigxihximxioxiqxirxisxiu$_X!Sxi!Txi!Uxi!Wxi!bxi!fxi!oxi!vxi#Rxi#jxi#lxi#mxi#nxi$fxi$gxi$hxi$ixi$uxi~O]!_O^!_O~PF|O]xi^xi~PF|O{!mO!h!fO#x!dOZxi[xi]xi^xi_ximxioxiqxirxisxiu$_X!fxi#jxi#|xiwxidxi#{xiaxi#Qxiexikxi~OTxicxigxihxiyxi!Sxi!Txi!Uxi!Wxi!bxi!oxi!vxi#Rxi#lxi#mxi#nxi$fxi$gxi$hxi$ixi$uxi~PIqO!f$oO~P6aOZ!^O[!^O]!_O^!_O_!lOo!jOq!]Or!]Os!kOy!`O{!mO!h!fO#x!dOu$_X~OTlicliglihlimli!Sli!Tli!Uli!Wli!bli!fli!oli!vli#Rli#jli#lli#mli#nli$fli$gli$hli$ili$uli~PL{OZ!^O[!^O]!_O^!_O_!lOq!]Or!]Os!kOy!`O{!mO!h!fO#x!dOu$_X~OTnicnignihnimnioni!Sni!Tni!Uni!Wni!bni!fni!oni!vni#Rni#jni#lni#mni#nni$fni$gni$hni$ini$uni~P! fOZ!^O[!^O]!_O^!_O_&wOm&uOq&qOr&qOs'gOy!`O{!mO!h!fO#x!dOu$_X~Oo$pO~P!$POTsOZUO[TOctOgvOhvOr&pOueO!S{O!T{O!UxO!WzO!b!OO!fdO!hfO!oyO!viO#RmO#lQO#mQO$f[O$g]O$h^O$i_O~P,ZO!R$tO!U$uO!W$vO!Z$wO!^$xO!b$yO#lQO#mQO~OZ#aX[#aX]#aX^#aX_#aXm#aXo#aXq#aXr#aXs#aXu#aXw#aXy#aX{#aX!h#aX#l#aX#m#aX#n#aX#x#aX#|#aX~P8mO$Y${O~O#|$|Ow$aX~Ow%OO~O#|$OO!f$baw$ba~O$Y%ROw!}X#|!}X~O#|%SOw$rX~Ow%UO~O$Y#Va~P?qO#k!zO$g]O~O#|$ZO#Q$ta~O#|$`Ok$Qa~O!T%`O~Ow!TO~O#{%bOa`X#|`X~P6aOTSqcSqgSqhSq!SSq!TSq!USq!WSq!bSq!fSq!oSq!vSq#RSq#jSq#lSq#mSq#nSq$fSq$gSq$hSq$iSq$uSq~P$yOw%dO~O#x%eOT!OXZ!OX[!OX]!OX^!OX_!OXc!OXg!OXh!OXm!OXo!OXq!OXr!OXs!OXu!OXy!OX{!OX!S!OX!T!OX!U!OX!W!OX!b!OX!f!OX!h!OX!o!OX!v!OX#R!OX#j!OX#l!OX#m!OX#n!OX$f!OX$g!OX$h!OX$i!OX$u!OX#|!OXw!OXd!OX#{!OXa!OX#Q!OXe!OXk!OX~Oo%gO~Oo%hO~Oo%iO~O![%jO~O![%kO~O![%lO~O#|$|Ow$aa~O!f#fa#|#faw#fa~P6aO#|%SOw$ra~O#O%uO~P`O#Q#Ti#|#Ti~P6aOe%vO~P6aOk$Ri#|$Ri~P6aOTfqcfqgfqhfq!Sfq!Tfq!Ufq!Wfq!bfq!ffq!ofq!vfq#Rfq#jfq#lfq#mfq#nfq$ffq$gfq$hfq$ifq$ufq~P$yOZ!^O[!^O]!_O^!_O_&wOm&uOo&vOq&qOr&qOs'gOy!`O{!mO#x!dOu$_X~Ow%xO!f%xO!h%wO~P!3YOZ!^O[!^O]!_O^!_Oy!`O{!mO!h!fO#x!dO_pympyopyqpyrpyspyu$_X!fpy#jpy~OTpycpygpyhpy!Spy!Tpy!Upy!Wpy!bpy!opy!vpy#Rpy#lpy#mpy#npy$fpy$gpy$hpy$ipy$upy~P!4eO#x%eOT!OaZ!Oa[!Oa]!Oa^!Oa_!Oac!Oag!Oah!Oam!Oao!Oaq!Oar!Oas!Oau!Oay!Oa{!Oa!S!Oa!T!Oa!U!Oa!W!Oa!b!Oa!f!Oa!h!Oa!o!Oa!v!Oa#R!Oa#j!Oa#l!Oa#m!Oa#n!Oa$f!Oa$g!Oa$h!Oa$i!Oa$u!Oa#|!Oaw!Oad!Oa#{!Oaa!Oa#Q!Oae!Oak!Oa~O!S%}O~O!V%}O~O!S&OO~O!R$tO!U$uO!W$vO!Z$wO!^$xO!b'oO#lQO#mQO~O!X$ZP~P!:^Ow!li#|!li~P6aOT#PXc#PXg#PXh#PX!S#PX!T#PX!U#PX!W#PX!b#PX!f#PX!o#PX!v#PX#R#PX#j#PX#l#PX#m#PX#n#PX$f#PX$g#PX$h#PX$i#PX$u#PX~P$yOT$^XZ$^X[$^X]$kX^$kX_$kXc$^Xg$^Xh$^Xm$kXo$kXq$kXr$^Xs$kXu$^Xy$kX{$kX!S$^X!T$^X!U$^X!W$^X!b$^X!f$^X!h$^X!o$^X!v$^X#R$^X#j$kX#l$^X#m$^X#n$kX#x$kX$f$^X$g$^X$h$^X$i$^X$u$kX#|$kXw$kXd$kX#{$kXa$kX#Q$kXe$kXk$kX~Oa`i#|`i~P6aO!T&^O~O#lQO#mQO!X!PX#x!PX#|!PX~O#x'VO!X!OX#|!OX~O!X&`O~O$Y&aO~O#|&bO!X$XX~O!X&dO~O#|&eO!X$ZX~O!X&gO~OTb!Rcb!Rgb!Rhb!R!Sb!R!Tb!R!Ub!R!Wb!R!bb!R!fb!R!ob!R!vb!R#Rb!R#jb!R#lb!R#mb!R#nb!R$fb!R$gb!R$hb!R$ib!R$ub!R~P$yO#x'VO!X!Oa#|!Oa~O#|&bO!X$Xa~O#|&eO!X$Za~O$[&mO~O#|#tXw#tXd#tX#{#tXa#tX#Q#tXe#tXk#tX~P2zOTsOZUO[TOctOgvOhvOr&pO!S{O!T{O!UxO!WzO!b!OO!fdO!hfO!oyO!viO#RmO#lQO#mQO$f[O$g]O$h^O$i_O~Ou#rO~P!D]O#j!wa#|!waw!wa!f!wad!wa#{!waa!wa#Q!wae!wak!wa~P6aO#|piwpidpi#{piapi#Qpiepikpi~PDcOy!`O{!mO!h!fO#x!dOZxi[xi_ximxioxiqxirxisxiu$_X#jxi#|xiwxi!fxidxi#{xiaxi#Qxiexikxi~O]!_O^!_O~P!G_Oy!`O~PIqOZ!^O[!^O]!_O^!_O_&wOo&vOq&qOr&qOs'gOy!`O{!mO!h!fO#x!dOu$_X~Omli#jli#|liwli!flidli#{liali#Qlielikli~P!IZOZ!^O[!^O]!_O^!_O_&wOq&qOr&qOs'gOy!`O{!mO!h!fO#x!dOu$_X~Omnioni#jni#|niwni!fnidni#{niani#Qnienikni~P!KOO#jSq#|SqwSq!fSqdSq#{SqaSq#QSqeSqkSq~P6aOZUO#O'XO~P!|O#jfq#|fqwfq!ffqdfq#{fqafq#Qfqefqkfq~P6aO#|pywpydpy#{pyapy#Qpyepykpy~P!4eO#j#PX#|#PXw#PX!f#PXd#PX#{#PXa#PX#Q#PXe#PXk#PX~P6aO#jb!R#|b!Rwb!R!fb!Rdb!R#{b!Rab!R#Qb!Reb!Rkb!R~P6aOTtXctXgtXhtX!StX!TtX!UtX!WtX!btX!ftX!otX!vtX#RtX#jtX#ltX#mtX#ntX$ftX$gtX$htX$itX$utX~P$yOk'PO~Oo'RO~P!$POw'SO~Oe'YO~P6aO$['[O~O!X'dO~Ou'hO~Od'iO~P6aO!['mO~O",
    goto: "!$f$uPPPP$vP&Q&W&_&t(sPPPPP(|P$vPPP$vPP)P)VP$vP$vP$vPPP)^P)jP$vP$vPP)s*Y*e*YPPPPPPP*YPP*YP,V,Y*YP,`,f$vP$vP$v,m-w-z.Q-wP.Z/e.Z.Z0v2QP$v3[$v4f4f5p5vP5}PP4f6V6]0r6aP6iP6l6r6y7P7V7]8y9T9Z9a9g9m9s9yPPPPPPPP:P:T=X>c?o?sPP?xPP@O@UAbBnBrBwB|CxDvEvGSPGVPGZHZIZJeJkJn$vJtJtPPPPLOMYM]Ng=XNj! t! w!#T!#X!$c#aiOPVhkru|}!]!a!b!c!g!i!j!k!l!q#]#_#c#g#r$O$Y$^$_$b$e$p${%U%b%u%v&p&q&r&s&t&u&v&w&y'P'R'S'X'Y']'g'i'nQ![RR'_'eS!YR'eR$l#eU!WR#e'eU#Sw$`'fY#v!q!y$|%S'hT&S%k&b#zWOPVXhkru|}!]!a!b!c!e!g!i!j!k!l#]#_#c#g#r#t$O$Y$^$_$b$e$p${%R%U%b%e%j%l%u%v&P&a&e&m&p&q&r&s&t&u&v&w&y'P'R'S'V'X'Y'[']'g'i'm'nh!VRw!y#e$`$|%S%k&b'e'f'hU#a!V#b#uR#u!qU#a!V#b#uT$W!{$XR$k#cQ#WwR'`'fS#Uw'fR%]$`U!QP#_#rS#s!l&wR$f#]Q!TPQ$h#_R$q#rQ$z#tQ%p%RQ&R%jU&W%l&e'mQ&h&aT&n&m'[c$s#t%R%j%l&a&e&m'['mrjOVr}!]!a!b!c!i!j!l#g$b$p%U%u%vQ#m!eU$r#t%R&mS%{%e'V[&Q%j%l&a&e'['m!m&oPhku|!g!k!q#]#_#c#r$O$Y$^$_$e${%b&p&q&r&s&t&u&v&w&y'P'R'S'X'Y']'g'i'nR&V%kQ&T%kR&i&bQ&Z%lR'j'mS&X%l'mR&k&e#aYOPVhkru|}!]!a!b!c!g!i!j!k!l!q#]#_#c#g#r$O$Y$^$_$b$e$p${%U%b%u%v&p&q&r&s&t&u&v&w&y'P'R'S'X'Y']'g'i'nR#{!qQ#x!qR%n$|S#w!q$|V$S!y%S'h#abOPVhkru|}!]!a!b!c!g!i!j!k!l!q#]#_#c#g#r$O$Y$^$_$b$e$p${%U%b%u%v&p&q&r&s&t&u&v&w&y'P'R'S'X'Y']'g'i'n#`bOPVhkru|}!]!a!b!c!g!i!j!k!l!q#]#_#c#g#r$O$Y$^$_$b$e$p${%U%b%u%v&p&q&r&s&t&u&v&w&y'P'R'S'X'Y']'g'i'nQ!s_T!|n$Z#a`OPVhkru|}!]!a!b!c!g!i!j!k!l!q#]#_#c#g#r$O$Y$^$_$b$e$p${%U%b%u%v&p&q&r&s&t&u&v&w&y'P'R'S'X'Y']'g'i'n#aaOPVhkru|}!]!a!b!c!g!i!j!k!l!q#]#_#c#g#r$O$Y$^$_$b$e$p${%U%b%u%v&p&q&r&s&t&u&v&w&y'P'R'S'X'Y']'g'i'n#agOPVhkru|}!]!a!b!c!g!i!j!k!l!q#]#_#c#g#r$O$Y$^$_$b$e$p${%U%b%u%v&p&q&r&s&t&u&v&w&y'P'R'S'X'Y']'g'i'n#aoOPVhkru|}!]!a!b!c!g!i!j!k!l!q#]#_#c#g#r$O$Y$^$_$b$e$p${%U%b%u%v&p&q&r&s&t&u&v&w&y'P'R'S'X'Y']'g'i'nQ$V!yR'b'hS$T!y'hR%q%SS%t%U'ST&[%u'XQ#OnR%X$ZT!}n$ZS!{n$ZT$W!{$XR!TPQrOR#QrS#b!V#uR$i#bQ#f!YR$m#fQ$a#UR%^$aQ#^!QR$g#^#jXOPVhkru|}!]!a!b!c!e!g!i!j!k!l!q#]#_#c#g#r#t$O$Y$^$_$b$e$p${%R%U%b%e%u%v&m&p&q&r&s&t&u&v&w&y'P'R'S'X'Y']'g'i'nS!pX&P_&P%j%l&a&e'V'['mQ%f$rS%|%f&_R&_&QQ&c&TR&j&cQ&f&XR&l&fQ$}#xR%o$}Q$P!wR%Q$PQ%T$TR%r%TQ$X!{R%V$XQ$[#OR%Y$[TqOrSSOrY!PP#]#_#r&wS!oV']Q!uhS!wk!qQ#RuQ#Z|Q#[}Q#i!]Q#j!aQ#k!bS#l!c&tQ#n!gQ#o!iQ#p!jQ#q!kQ$j#cQ$n#gQ%P$OQ%W$YQ%Z$^Q%[$_Q%_$bQ%a$eQ%c$pQ%m${S%s%U%uQ%z%bQ&]%vQ&x&pQ&z&qQ&{&rQ&|&sQ&}&uQ'O&vQ'Q&yQ'T'PQ'U'RS'W'S'XQ'Z'YQ'^!lQ'a'gQ'c'iR'l'n#apOPVhkru|}!]!a!b!c!g!i!j!k!l!q#]#_#c#g#r$O$Y$^$_$b$e$p${%U%b%u%v&p&q&r&s&t&u&v&w&y'P'R'S'X'Y']'g'i'nrROVr}!]!a!b!c!i!j!l#g$b$p%U%u%v!m'ePhku|!g!k!q#]#_#c#r$O$Y$^$_$e${%b&p&q&r&s&t&u&v&w&y'P'R'S'X'Y']'g'i'nT!ZR'eV!XR#e'eQ#c!WR$_#SQ#g![R&y'_ruOVr}!]!a!b!c!i!j!l#g$b$p%U%u%v!m'nPhku|!g!k!q#]#_#c#r$O$Y$^$_$e${%b&p&q&r&s&t&u&v&w&y'P'R'S'X'Y']'g'i'nrwOVr}!]!a!b!c!i!j!l#g$b$p%U%u%v!m'fPhku|!g!k!q#]#_#c#r$O$Y$^$_$e${%b&p&q&r&s&t&u&v&w&y'P'R'S'X'Y']'g'i'nT#Vw'fV#Tw$`'fV!RP#_#rf!aS#[#i#o#p$n%_%c%s&]'^!X&r!P!u!w#R#Z#n#q$j%P%W%Z%[%a%m%z&x&z&}'O'Q'T'U'W'Z'a'c'lh!bS#[#i#j#o#p$n%_%c%s&]'^!Z&s!P!u!w#R#Z#n#q$j%P%W%Z%[%a%m%z&x&z&{&}'O'Q'T'U'W'Z'a'c'lj!cS#[#i#j#k#o#p$n%_%c%s&]'^!]&t!P!u!w#R#Z#n#q$j%P%W%Z%[%a%m%z&x&z&{&|&}'O'Q'T'U'W'Z'a'c'lrVOVr}!]!a!b!c!i!j!l#g$b$p%U%u%v!m']Phku|!g!k!q#]#_#c#r$O$Y$^$_$e${%b&p&q&r&s&t&u&v&w&y'P'R'S'X'Y']'g'i'nR&U%kT&Y%l'm!{!eS!P!o!u!w#R#Z#[#i#j#k#l#n#o#p#q$j$n%P%W%Z%[%_%a%c%m%s%z&]&x&z&{&|&}'O'Q'T'U'W'Z'^'a'c'l!{!gS!P!o!u!w#R#Z#[#i#j#k#l#n#o#p#q$j$n%P%W%Z%[%_%a%c%m%s%z&]&x&z&{&|&}'O'Q'T'U'W'Z'^'a'c'l#aZOPVhkru|}!]!a!b!c!g!i!j!k!l!q#]#_#c#g#r$O$Y$^$_$b$e$p${%U%b%u%v&p&q&r&s&t&u&v&w&y'P'R'S'X'Y']'g'i'nQ!rZR!t`R#y!qQ!xkR#z!q#acOPVhkru|}!]!a!b!c!g!i!j!k!l!q#]#_#c#g#r$O$Y$^$_$b$e$p${%U%b%u%v&p&q&r&s&t&u&v&w&y'P'R'S'X'Y']'g'i'n#a|OPVhkru|}!]!a!b!c!g!i!j!k!l!q#]#_#c#g#r$O$Y$^$_$b$e$p${%U%b%u%v&p&q&r&s&t&u&v&w&y'P'R'S'X'Y']'g'i'nR%y%a#ahOPVhkru|}!]!a!b!c!g!i!j!k!l!q#]#_#c#g#r$O$Y$^$_$b$e$p${%U%b%u%v&p&q&r&s&t&u&v&w&y'P'R'S'X'Y']'g'i'nR#}!u#akOPVhkru|}!]!a!b!c!g!i!j!k!l!q#]#_#c#g#r$O$Y$^$_$b$e$p${%U%b%u%v&p&q&r&s&t&u&v&w&y'P'R'S'X'Y']'g'i'nR$R!xrlOVr}!]!a!b!c!i!j!l#g$b$p%U%u%v!m'kPhku|!g!k!q#]#_#c#r$O$Y$^$_$e${%b&p&q&r&s&t&u&v&w&y'P'R'S'X'Y']'g'i'nT$U!y'h#anOPVhkru|}!]!a!b!c!g!i!j!k!l!q#]#_#c#g#r$O$Y$^$_$b$e$p${%U%b%u%v&p&q&r&s&t&u&v&w&y'P'R'S'X'Y']'g'i'nR#Pn",
    nodeNames: "⚠ LineComment BlockComment Expressions ForExpression for InExpressions InExpression Name Identifier Identifier ArithOp ArithOp ArithOp ArithOp in IterationContext return IfExpression if then else QuantifiedExpression some every InExpressions InExpression satisfies Disjunction or Conjunction and Comparison CompareOp CompareOp between PositiveUnaryTest ( PositiveUnaryTests ) ArithmeticExpression ArithOp InstanceOfExpression instance of Type QualifiedName VariableName SpecialType days time duration years months date > ListType list < ContextType context ContextEntryTypes ContextEntryType FunctionType function ArgumentTypes ArgumentType PathExpression ] FilterExpression [ FunctionInvocation SpecialFunctionName NamedParameters NamedParameter ParameterName PositionalParameters null NumericLiteral StringLiteral BooleanLiteral DateTimeLiteral DateTimeConstructor AtLiteral ? SimplePositiveUnaryTest Interval ParenthesizedExpression List FunctionDefinition FormalParameters FormalParameter external FunctionBody } { Context ContextEntry Key Name Identifier UnaryTests Wildcard not",
    maxTerm: 175,
    context: variableTracker,
    nodeProps: [
      ["group", -17,4,18,22,28,30,32,40,42,67,69,71,84,85,87,88,89,96,"Expression",47,"Expression Expression",-5,77,78,79,80,81,"Expression Literal"],
      ["closedBy", 37,")",70,"]",95,"}"],
      ["openedBy", 39,"(",68,"[",94,"{"]
    ],
    propSources: [feelHighlighting],
    skippedNodes: [0,1,2],
    repeatNodeCount: 14,
    tokenData: "+l~RuXY#fYZ$ZZ[#f]^$Zpq#fqr$`rs$kwx&cxy&hyz&mz{&r{|'P|}'U}!O'Z!O!P'h!P!Q(Q!Q![){![!]*^!]!^*c!^!_*h!_!`$f!`!a*w!b!c+R!}#O+W#P#Q+]#Q#R&z#o#p+b#q#r+g$f$g#f#BY#BZ#f$IS$I_#f$I|$I}$Z$I}$JO$Z$JT$JU#f$KV$KW#f&FU&FV#f?HT?HU#f~#kY#p~XY#fZ[#fpq#f$f$g#f#BY#BZ#f$IS$I_#f$JT$JU#f$KV$KW#f&FU&FV#f?HT?HU#f~$`O#q~~$cP!_!`$f~$kOq~~$pW$g~OY$kZr$krs%Ys#O$k#O#P%_#P;'S$k;'S;=`&]<%lO$k~%_O$g~~%bRO;'S$k;'S;=`%k;=`O$k~%pX$g~OY$kZr$krs%Ys#O$k#O#P%_#P;'S$k;'S;=`&];=`<%l$k<%lO$k~&`P;=`<%l$k~&hO#y~~&mOu~~&rOw~~&wP^~z{&z~'POy~~'UO[~~'ZO#|~R'`PZP!`!a'cQ'hO$[Q~'mQ#x~!O!P's!Q!['x~'xO#{~~'}P$f~!Q!['x~(VQ]~z{(]!P!Q)d~(`TOz(]z{(o{;'S(];'S;=`)^<%lO(]~(rVOz(]z{(o{!P(]!P!Q)X!Q;'S(];'S;=`)^<%lO(]~)^OQ~~)aP;=`<%l(]~)iSP~OY)dZ;'S)d;'S;=`)u<%lO)d~)xP;=`<%l)d~*QQ$f~!O!P*W!Q![){~*ZP!Q!['x~*cO$Y~~*hO$u~R*oP![QrP!_!`*rP*wOrPR+OP!XQrP!_!`*r~+WO$i~~+]O!h~~+bO!f~~+gO#R~~+lO#Q~",
    tokenizers: [propertyIdentifiers, identifiers, insertSemicolon, 0, 1],
    topRules: {"Expressions":[0,3],"UnaryTests":[1,101]},
    dynamicPrecedences: {"30":-1,"71":-1,"73":-1,"126":-1},
    specialized: [{term: 120, get: value => spec_identifier[value] || -1}],
    tokenPrec: 0
  });

  function isContext(e) {
      return Object.getPrototypeOf(e) === Object.prototype;
  }
  function isDateTime(obj) {
      return DateTime.isDateTime(obj);
  }
  function isDuration(obj) {
      return Duration.isDuration(obj);
  }
  function isArray$1(e) {
      return Array.isArray(e);
  }
  function isBoolean(e) {
      return typeof e === 'boolean';
  }
  function getType(e) {
      if (e === null || e === undefined) {
          return 'nil';
      }
      if (isBoolean(e)) {
          return 'boolean';
      }
      if (isNumber(e)) {
          return 'number';
      }
      if (isString(e)) {
          return 'string';
      }
      if (isContext(e)) {
          return 'context';
      }
      if (isArray$1(e)) {
          return 'list';
      }
      if (isDuration(e)) {
          return 'duration';
      }
      if (isDateTime(e)) {
          if (e.year === 1900 &&
              e.month === 1 &&
              e.day === 1) {
              return 'time';
          }
          if (e.hour === 0 &&
              e.minute === 0 &&
              e.second === 0 &&
              e.millisecond === 0 &&
              e.zone === FixedOffsetZone.utcInstance) {
              return 'date';
          }
          return 'date time';
      }
      if (e instanceof Range$1) {
          return 'range';
      }
      if (e instanceof FunctionWrapper) {
          return 'function';
      }
      return 'literal';
  }
  function isType(el, type) {
      return getType(el) === type;
  }
  function typeCast(obj, type) {
      if (isDateTime(obj)) {
          if (type === 'time') {
              return obj.set({
                  year: 1900,
                  month: 1,
                  day: 1
              });
          }
          if (type === 'date') {
              return obj.setZone('utc', { keepLocalTime: true }).startOf('day');
          }
          if (type === 'date time') {
              return obj;
          }
      }
      return null;
  }
  let Range$1 = class Range {
      constructor(props) {
          Object.assign(this, props);
      }
  };
  function isNumber(obj) {
      return typeof obj === 'number';
  }
  function isString(obj) {
      return typeof obj === 'string';
  }
  function equals(a, b) {
      if (a === null && b !== null ||
          a !== null && b === null) {
          return false;
      }
      if (isArray$1(a) && a.length < 2) {
          a = a[0];
      }
      if (isArray$1(b) && b.length < 2) {
          b = b[0];
      }
      const aType = getType(a);
      const bType = getType(b);
      if (aType !== bType) {
          return null;
      }
      if (aType === 'nil') {
          return true;
      }
      if (aType === 'list') {
          if (a.length !== b.length) {
              return false;
          }
          return a.every((element, idx) => equals(element, b[idx]));
      }
      if (aType === 'date time' || aType === 'time' || aType === 'date') {
          return (a.toUTC().valueOf() === b.toUTC().valueOf());
      }
      if (aType === 'duration') {
          // years and months duration -> months
          if (Math.abs(a.as('days')) > 180) {
              return Math.trunc(a.minus(b).as('months')) === 0;
          }
          // days and time duration -> seconds
          else {
              return Math.trunc(a.minus(b).as('seconds')) === 0;
          }
      }
      if (aType === 'context') {
          const aEntries = Object.entries(a);
          const bEntries = Object.entries(b);
          if (aEntries.length !== bEntries.length) {
              return false;
          }
          return aEntries.every(([key, value]) => key in b && equals(value, b[key]));
      }
      if (aType === 'range') {
          return [
              [a.start, b.start],
              [a.end, b.end],
              [a['start included'], b['start included']],
              [a['end included'], b['end included']]
          ].every(([a, b]) => a === b);
      }
      if (a == b) {
          return true;
      }
      return aType === bType ? false : null;
  }
  class FunctionWrapper {
      constructor(fn, parameterNames) {
          this.fn = fn;
          this.parameterNames = parameterNames;
      }
      invoke(contextOrArgs) {
          let params;
          if (isArray$1(contextOrArgs)) {
              params = contextOrArgs;
          }
          else {
              params = this.parameterNames.map(n => contextOrArgs[n]);
          }
          return this.fn.call(null, ...params);
      }
  }

  function parseParameterNames(fn) {
      if (Array.isArray(fn.$args)) {
          return fn.$args;
      }
      const code = fn.toString();
      const match = /^(?:[^(]*\s*)?\(([^)]+)?\)/.exec(code);
      if (!match) {
          throw new Error('failed to parse params: ' + code);
      }
      const [_, params] = match;
      if (!params) {
          return [];
      }
      return params.split(',').map(p => p.trim());
  }
  function notImplemented(thing) {
      return new Error(`not implemented: ${thing}`);
  }
  /**
   * @param {string} name
   * @param {Record<string, any>} context
   *
   * @return {any}
   */
  function getFromContext(name, context) {
      if (['nil', 'boolean', 'number', 'string'].includes(getType(context))) {
          return null;
      }
      if (name in context) {
          return context[name];
      }
      const normalizedName = normalizeContextKey(name);
      if (normalizedName in context) {
          return context[normalizedName];
      }
      const entry = Object.entries(context).find(([key]) => normalizedName === normalizeContextKey(key));
      if (entry) {
          return entry[1];
      }
      return null;
  }

  function duration$1(opts) {
      if (typeof opts === 'number') {
          return Duration.fromMillis(opts);
      }
      return Duration.fromISO(opts);
  }
  function date(str = null, time = null, zone = null) {
      if (time) {
          if (str) {
              throw new Error('<str> and <time> provided');
          }
          return date(`1900-01-01T${time}`);
      }
      if (typeof str === 'string') {
          if (str.startsWith('-')) {
              throw notImplemented('negative date');
          }
          if (!str.includes('T')) {
              // raw dates are in UTC time zone
              return date(str + 'T00:00:00.000Z');
          }
          if (str.includes('@')) {
              if (zone) {
                  throw new Error('<zone> already provided');
              }
              const [datePart, zonePart] = str.split('@');
              return date(datePart, null, Info.normalizeZone(zonePart));
          }
          return DateTime.fromISO(str.toUpperCase(), {
              setZone: true,
              zone
          });
      }
      return DateTime.now();
  }

  // 10.3.4 Built-in functions
  const builtins = {
      // 10.3.4.1 Conversion functions
      'number': function () {
          throw notImplemented('number');
      },
      'string': fn(function (from) {
          if (arguments.length !== 1) {
              return null;
          }
          return toString(from);
      }, ['any']),
      // date(from) => date string
      // date(from) => date and time
      // date(year, month, day)
      'date': fn(function (year, month, day, from) {
          if (!from && !isNumber(year)) {
              from = year;
              year = null;
          }
          let d;
          if (isString(from)) {
              d = date(from);
          }
          if (isDateTime(from)) {
              d = from;
          }
          if (year) {
              d = date().setZone('utc').set({
                  year,
                  month,
                  day
              });
          }
          return d && ifValid(d.setZone('utc').startOf('day')) || null;
      }, ['any?', 'number?', 'number?', 'any?']),
      // date and time(from) => date time string
      // date and time(date, time)
      'date and time': fn(function (d, time, from) {
          let dt;
          if (isDateTime(d) && isDateTime(time)) {
              dt = time.set({
                  year: d.year,
                  month: d.month,
                  day: d.day
              });
          }
          if (isString(d)) {
              from = d;
              d = null;
          }
          if (isString(from)) {
              dt = date(from);
          }
          return dt && ifValid(dt) || null;
      }, ['any?', 'time?', 'string?'], ['date', 'time', 'from']),
      // time(from) => time string
      // time(from) => time, date and time
      // time(hour, minute, second, offset?) => ...
      'time': fn(function (hour, minute, second, offset, from) {
          let t;
          if (offset) {
              throw notImplemented('time(..., offset)');
          }
          if (isString(hour) || isDateTime(hour)) {
              from = hour;
              hour = null;
          }
          if (isString(from)) {
              t = date(null, from);
          }
          if (isDateTime(from)) {
              t = from.set({
                  year: 1900,
                  month: 1,
                  day: 1
              });
          }
          if (isNumber(hour)) {
              // TODO: support offset = days and time duration
              t = date().set({
                  hour,
                  minute,
                  second
              }).set({
                  year: 1900,
                  month: 1,
                  day: 1,
                  millisecond: 0
              });
          }
          return t && ifValid(t) || null;
      }, ['any?', 'number?', 'number?', 'any?', 'any?']),
      'duration': fn(function (from) {
          return ifValid(duration$1(from));
      }, ['string']),
      'years and months duration': fn(function (from, to) {
          return ifValid(to.diff(from, ['years', 'months']));
      }, ['date', 'date']),
      '@': fn(function (string) {
          let t;
          if (/^-?P/.test(string)) {
              t = duration$1(string);
          }
          else if (/^[\d]{1,2}:[\d]{1,2}:[\d]{1,2}/.test(string)) {
              t = date(null, string);
          }
          else {
              t = date(string);
          }
          return t && ifValid(t) || null;
      }, ['string']),
      'now': fn(function () {
          return date();
      }, []),
      'today': fn(function () {
          return date().startOf('day');
      }, []),
      // 10.3.4.2 Boolean function
      'not': fn(function (bool) {
          return isType(bool, 'boolean') ? !bool : null;
      }, ['any']),
      // 10.3.4.3 String functions
      'substring': fn(function (string, start, length) {
          const _start = (start < 0 ? string.length + start : start - 1);
          const arr = Array.from(string);
          return (typeof length !== 'undefined'
              ? arr.slice(_start, _start + length)
              : arr.slice(_start)).join('');
      }, ['string', 'number', 'number?'], ['string', 'start position', 'length']),
      'string length': fn(function (string) {
          return countSymbols(string);
      }, ['string']),
      'upper case': fn(function (string) {
          return string.toUpperCase();
      }, ['string']),
      'lower case': fn(function (string) {
          return string.toLowerCase();
      }, ['string']),
      'substring before': fn(function (string, match) {
          const index = string.indexOf(match);
          if (index === -1) {
              return '';
          }
          return string.substring(0, index);
      }, ['string', 'string']),
      'substring after': fn(function (string, match) {
          const index = string.indexOf(match);
          if (index === -1) {
              return '';
          }
          return string.substring(index + match.length);
      }, ['string', 'string']),
      'replace': fn(function (input, pattern, replacement, flags) {
          return input.replace(new RegExp(pattern, 'ug' + (flags || '').replace(/[x]/g, '')), replacement.replace(/\$0/g, '$$&'));
      }, ['string', 'string', 'string', 'string?']),
      'contains': fn(function (string, match) {
          return string.includes(match);
      }, ['string', 'string']),
      'starts with': fn(function (string, match) {
          return string.startsWith(match);
      }, ['string', 'string']),
      'ends with': fn(function (string, match) {
          return string.endsWith(match);
      }, ['string', 'string']),
      'split': fn(function (string, delimiter) {
          return string.split(new RegExp(delimiter, 'u'));
      }, ['string', 'string']),
      // 10.3.4.4 List functions
      'list contains': fn(function (list, element) {
          return list.some(el => matches(el, element));
      }, ['list', 'any?']),
      'count': fn(function (list) {
          return list.length;
      }, ['list']),
      'min': listFn(function (list) {
          return list.reduce((min, el) => min === null ? el : Math.min(min, el), null);
      }, 'number'),
      'max': listFn(function (list) {
          return list.reduce((max, el) => max === null ? el : Math.max(max, el), null);
      }, 'number'),
      'sum': listFn(function (list) {
          return sum(list);
      }, 'number'),
      'mean': listFn(function (list) {
          const s = sum(list);
          return s === null ? s : s / list.length;
      }, 'number'),
      'all': listFn(function (list) {
          let nonBool = false;
          for (const o of list) {
              if (o === false) {
                  return false;
              }
              if (typeof o !== 'boolean') {
                  nonBool = true;
              }
          }
          return nonBool ? null : true;
      }, 'any?'),
      'any': listFn(function (list) {
          let nonBool = false;
          for (const o of list) {
              if (o === true) {
                  return true;
              }
              if (typeof o !== 'boolean') {
                  nonBool = true;
              }
          }
          return nonBool ? null : false;
      }, 'any?'),
      'sublist': fn(function (list, start, length) {
          const _start = (start < 0 ? list.length + start : start - 1);
          return (typeof length !== 'undefined'
              ? list.slice(_start, _start + length)
              : list.slice(_start));
      }, ['list', 'number', 'number?']),
      'append': fn(function (list, ...items) {
          return list.concat(items);
      }, ['list', 'any?']),
      'concatenate': fn(function (...args) {
          return args.reduce((result, arg) => {
              return result.concat(arg);
          }, []);
      }, ['any']),
      'insert before': fn(function (list, position, newItem) {
          return list.slice(0, position - 1).concat([newItem], list.slice(position - 1));
      }, ['list', 'number', 'any?']),
      'remove': fn(function (list, position) {
          return list.slice(0, position - 1).concat(list.slice(position));
      }, ['list', 'number']),
      'reverse': fn(function (list) {
          return list.slice().reverse();
      }, ['list']),
      'index of': fn(function (list, match) {
          return list.reduce(function (result, element, index) {
              if (matches(element, match)) {
                  result.push(index + 1);
              }
              return result;
          }, []);
      }, ['list', 'any']),
      'union': fn(function (..._lists) {
          throw notImplemented('union');
      }, ['list']),
      'distinct values': fn(function (_list) {
          throw notImplemented('distinct values');
      }, ['list']),
      'flatten': fn(function (list) {
          return flatten$1(list);
      }, ['list']),
      'product': listFn(function (list) {
          if (list.length === 0) {
              return null;
          }
          return list.reduce((result, n) => {
              return result * n;
          }, 1);
      }, 'number'),
      'median': listFn(function (list) {
          if (list.length === 0) {
              return null;
          }
          return median(list);
      }, 'number'),
      'stddev': listFn(function (list) {
          if (list.length < 2) {
              return null;
          }
          return stddev(list);
      }, 'number'),
      'mode': listFn(function (list) {
          return mode(list);
      }, 'number'),
      // 10.3.4.5 Numeric functions
      'decimal': fn(function (n, scale) {
          if (!scale) {
              return round$1(n);
          }
          const offset = Math.pow(10, scale);
          return round$1(n * offset) / (offset);
      }, ['number', 'number']),
      'floor': fn(function (n) {
          return Math.floor(n);
      }, ['number']),
      'ceiling': fn(function (n) {
          return Math.ceil(n) + 0;
      }, ['number']),
      'abs': fn(function (n) {
          if (typeof n !== 'number') {
              return null;
          }
          return Math.abs(n);
      }, ['number']),
      'modulo': fn(function (dividend, divisor) {
          if (!divisor) {
              return null;
          }
          const adjust = 1000000000;
          // cf. https://dustinpfister.github.io/2017/09/02/js-whats-wrong-with-modulo/
          //
          // need to round here as using this custom modulo
          // variant is prone to rounding errors
          return Math.round((dividend % divisor + divisor) % divisor * adjust) / adjust;
      }, ['number', 'number']),
      'sqrt': fn(function (number) {
          if (number < 0) {
              return null;
          }
          return Math.sqrt(number);
      }, ['number']),
      'log': fn(function (number) {
          if (number <= 0) {
              return null;
          }
          return Math.log(number);
      }, ['number']),
      'exp': fn(function (number) {
          return Math.exp(number);
      }, ['number']),
      'odd': fn(function (number) {
          return Math.abs(number) % 2 === 1;
      }, ['number']),
      'even': fn(function (number) {
          return Math.abs(number) % 2 === 0;
      }, ['number']),
      // 10.3.4.6 Date and time functions
      'is': fn(function (value1, value2) {
          if (typeof value1 === 'undefined' || typeof value2 === 'undefined') {
              return false;
          }
          return equals(value1, value2);
      }, ['any?', 'any?']),
      // 10.3.4.7 Range Functions
      'before': fn(function (a, b) {
          return before(a, b);
      }, ['any', 'any']),
      'after': fn(function (a, b) {
          return before(b, a);
      }, ['any', 'any']),
      'meets': fn(function (a, b) {
          return meets(a, b);
      }, ['range', 'range']),
      'met by': fn(function (a, b) {
          return meets(b, a);
      }, ['range', 'range']),
      'overlaps': fn(function () {
          throw notImplemented('overlaps');
      }, ['any?']),
      'overlaps before': fn(function () {
          throw notImplemented('overlaps before');
      }, ['any?']),
      'overlaps after': fn(function () {
          throw notImplemented('overlaps after');
      }, ['any?']),
      'finishes': fn(function () {
          throw notImplemented('finishes');
      }, ['any?']),
      'finished by': fn(function () {
          throw notImplemented('finished by');
      }, ['any?']),
      'includes': fn(function () {
          throw notImplemented('includes');
      }, ['any?']),
      'during': fn(function () {
          throw notImplemented('during');
      }, ['any?']),
      'starts': fn(function () {
          throw notImplemented('starts');
      }, ['any?']),
      'started by': fn(function () {
          throw notImplemented('started by');
      }, ['any?']),
      'coincides': fn(function () {
          throw notImplemented('coincides');
      }, ['any?']),
      // 10.3.4.8 Temporal built-in functions
      'day of year': fn(function () {
          throw notImplemented('day of year');
      }, ['any?']),
      'day of week': fn(function () {
          throw notImplemented('day of week');
      }, ['any?']),
      'month of year': fn(function () {
          throw notImplemented('month of year');
      }, ['any?']),
      'week of year': fn(function () {
          throw notImplemented('week of year');
      }, ['any?']),
      // 10.3.4.9 Sort
      'sort': function () {
          throw notImplemented('sort');
      },
      // 10.3.4.10 Context function
      'get value': fn(function (m, key) {
          return getFromContext(key, m);
      }, ['context', 'string']),
      'get entries': fn(function (m) {
          if (arguments.length !== 1) {
              return null;
          }
          if (Array.isArray(m)) {
              return null;
          }
          return Object.entries(m).map(([key, value]) => ({ key, value }));
      }, ['context']),
      'context': listFn(function (_contexts) {
          throw notImplemented('context');
      }, 'context'),
      'context merge': listFn(function (_contexts) {
          throw notImplemented('context merge');
      }, 'context'),
      'context put': fn(function (_context, _keys, _value) {
          throw notImplemented('context put');
      }, ['context', 'list', 'any'])
  };
  function matches(a, b) {
      return a === b;
  }
  const FALSE = {};
  function createArgTester(arg) {
      const optional = arg.endsWith('?');
      const type = optional ? arg.substring(0, arg.length - 1) : arg;
      return function (obj) {
          const arr = Array.isArray(obj);
          if (type === 'list') {
              if (arr || optional && typeof obj === 'undefined') {
                  return obj;
              }
              else {
                  // implicit conversion obj => [ obj ]
                  return [obj];
              }
          }
          if (type !== 'any' && arr && obj.length === 1) {
              // implicit conversion [ obj ] => obj
              obj = obj[0];
          }
          if (type === 'range') {
              return obj instanceof Range$1 ? obj : FALSE;
          }
          const objType = getType(obj);
          if (objType === 'nil') {
              return (optional ? obj : FALSE);
          }
          if (type === 'any' || type === objType) {
              return obj;
          }
          return typeCast(obj, type) || FALSE;
      };
  }
  function createArgsValidator(argDefinitions) {
      const tests = argDefinitions.map(createArgTester);
      return function (args) {
          while (args.length < argDefinitions.length) {
              args.push(undefined);
          }
          return args.reduce((result, arg, index) => {
              if (result === false) {
                  return result;
              }
              const test = tests[index];
              const conversion = test ? test(arg) : arg;
              if (conversion === FALSE) {
                  return false;
              }
              result.push(conversion);
              return result;
          }, []);
      };
  }
  /**
   * @param {Function} fnDefinition
   * @param {string} type
   * @param {string[]} [parameterNames]
   *
   * @return {Function}
   */
  function listFn(fnDefinition, type, parameterNames = null) {
      const tester = createArgTester(type);
      const wrappedFn = function (...args) {
          if (args.length === 0) {
              return null;
          }
          // unwrap first arg
          if (Array.isArray(args[0]) && args.length === 1) {
              args = args[0];
          }
          if (!args.every(arg => tester(arg) !== FALSE)) {
              return null;
          }
          return fnDefinition(args);
      };
      wrappedFn.$args = parameterNames || parseParameterNames(fnDefinition);
      return wrappedFn;
  }
  /**
   * @param {Function} fnDefinition
   * @param {string[]} argDefinitions
   * @param {string[]} [parameterNames]
   *
   * @return {Function}
   */
  function fn(fnDefinition, argDefinitions, parameterNames = null) {
      const checkArgs = createArgsValidator(argDefinitions);
      parameterNames = parameterNames || parseParameterNames(fnDefinition);
      const wrappedFn = function (...args) {
          const convertedArgs = checkArgs(args);
          if (!convertedArgs) {
              return null;
          }
          return fnDefinition(...convertedArgs);
      };
      wrappedFn.$args = parameterNames;
      return wrappedFn;
  }
  function meets(a, b) {
      return [
          (a.end === b.start),
          (a['end included'] === true),
          (b['start included'] === true)
      ].every(v => v);
  }
  function before(a, b) {
      if (a instanceof Range$1 && b instanceof Range$1) {
          return (a.end < b.start || (!a['end included'] || !b['start included']) && a.end == b.start);
      }
      if (a instanceof Range$1) {
          return (a.end < b || (!a['end included'] && a.end === b));
      }
      if (b instanceof Range$1) {
          return (b.start > a || (!b['start included'] && b.start === a));
      }
      return a < b;
  }
  function sum(list) {
      return list.reduce((sum, el) => sum === null ? el : sum + el, null);
  }
  function flatten$1([x, ...xs]) {
      return (x !== undefined
          ? [...Array.isArray(x) ? flatten$1(x) : [x], ...flatten$1(xs)]
          : []);
  }
  function toKeyString(key) {
      if (typeof key === 'string' && /\W/.test(key)) {
          return toString(key, true);
      }
      return key;
  }
  function toDeepString(obj) {
      return toString(obj, true);
  }
  function escapeStr(str) {
      return str.replace(/("|\\)/g, '\\$1');
  }
  function toString(obj, wrap = false) {
      var _a, _b, _c, _d;
      const type = getType(obj);
      if (type === 'nil') {
          return 'null';
      }
      if (type === 'string') {
          return wrap ? `"${escapeStr(obj)}"` : obj;
      }
      if (type === 'boolean' || type === 'number') {
          return String(obj);
      }
      if (type === 'list') {
          return '[' + obj.map(toDeepString).join(', ') + ']';
      }
      if (type === 'context') {
          return '{' + Object.entries(obj).map(([key, value]) => {
              return toKeyString(key) + ': ' + toDeepString(value);
          }).join(', ') + '}';
      }
      if (type === 'duration') {
          return obj.shiftTo('years', 'months', 'days', 'hours', 'minutes', 'seconds').normalize().toISO();
      }
      if (type === 'date time') {
          if ((_a = obj.zone) === null || _a === void 0 ? void 0 : _a.zoneName) {
              return obj.toISO({ suppressMilliseconds: true, includeOffset: false }) + '@' + ((_b = obj.zone) === null || _b === void 0 ? void 0 : _b.zoneName);
          }
          return obj.toISO({ suppressMilliseconds: true });
      }
      if (type === 'date') {
          return obj.toISODate();
      }
      if (type === 'range') {
          return '<range>';
      }
      if (type === 'time') {
          if ((_c = obj.zone) === null || _c === void 0 ? void 0 : _c.zoneName) {
              return obj.toISOTime({ suppressMilliseconds: true, includeOffset: false }) + '@' + ((_d = obj.zone) === null || _d === void 0 ? void 0 : _d.zoneName);
          }
          return obj.toISOTime({ suppressMilliseconds: true });
      }
      if (type === 'function') {
          return '<function>';
      }
      throw notImplemented('string(' + type + ')');
  }
  function countSymbols(str) {
      // cf. https://mathiasbynens.be/notes/javascript-unicode
      return str.replace(/[\uD800-\uDBFF][\uDC00-\uDFFF]/g, '_').length;
  }
  function round$1(n) {
      const integral = Math.trunc(n);
      if (n - integral > .5) {
          return integral + 1;
      }
      else {
          return integral;
      }
  }
  // adapted from https://stackoverflow.com/a/53577159
  function stddev(array) {
      const n = array.length;
      const mean = array.reduce((a, b) => a + b) / n;
      return Math.sqrt(array.map(x => Math.pow(x - mean, 2)).reduce((a, b) => a + b) / (n - 1));
  }
  function median(array) {
      const n = array.length;
      const sorted = array.slice().sort();
      const mid = n / 2 - 1;
      const index = Math.ceil(mid);
      // even
      if (mid === index) {
          return (sorted[index] + sorted[index + 1]) / 2;
      }
      // uneven
      return sorted[index];
  }
  function mode(array) {
      if (array.length < 2) {
          return array;
      }
      const buckets = {};
      for (const n of array) {
          buckets[n] = (buckets[n] || 0) + 1;
      }
      const sorted = Object.entries(buckets).sort((a, b) => b[1] - a[1]);
      return sorted.filter(s => s[1] === sorted[0][1]).map(e => +e[0]);
  }
  function ifValid(o) {
      return o.isValid ? o : null;
  }

  function parseExpressions(expression, context = {}) {
      return parser$1.configure({
          top: 'Expressions',
          contextTracker: trackVariables(context)
      }).parse(expression);
  }
  function parseUnaryTests(expression, context = {}) {
      return parser$1.configure({
          top: 'UnaryTests',
          contextTracker: trackVariables(context)
      }).parse(expression);
  }

  class Interpreter {
      _buildExecutionTree(tree, input) {
          const root = { args: [], nodeInput: input };
          const stack = [root];
          tree.iterate({
              enter(nodeRef) {
                  const { isError, isSkipped } = nodeRef.type;
                  const { from, to } = nodeRef;
                  if (isError) {
                      throw new Error(`Statement unparseable at [${from}, ${to}]`);
                  }
                  if (isSkipped) {
                      return false;
                  }
                  const nodeInput = input.slice(from, to);
                  stack.push({
                      nodeInput,
                      args: []
                  });
              },
              leave(nodeRef) {
                  if (nodeRef.type.isSkipped) {
                      return;
                  }
                  const { nodeInput, args } = stack.pop();
                  const parent = stack[stack.length - 1];
                  const expr = evalNode(nodeRef, nodeInput, args);
                  parent.args.push(expr);
              }
          });
          return root.args[root.args.length - 1];
      }
      evaluate(expression, context = {}) {
          const parseTree = parseExpressions(expression, context);
          const root = this._buildExecutionTree(parseTree, expression);
          return {
              parseTree,
              root
          };
      }
      unaryTest(expression, context = {}) {
          const parseTree = parseUnaryTests(expression, context);
          const root = this._buildExecutionTree(parseTree, expression);
          return {
              parseTree,
              root
          };
      }
  }
  const interpreter = new Interpreter();
  function evaluate$1(expression, context = {}) {
      const { root } = interpreter.evaluate(expression, context);
      // root = [ fn(ctx) ]
      const results = root(context);
      if (results.length === 1) {
          return results[0];
      }
      else {
          return results;
      }
  }
  function evalNode(node, input, args) {
      switch (node.name) {
          case 'ArithOp': return (context) => {
              const nullable = (op, types = ['number']) => (a, b) => {
                  const left = a(context);
                  const right = b(context);
                  if (isArray$1(left)) {
                      return null;
                  }
                  if (isArray$1(right)) {
                      return null;
                  }
                  const leftType = getType(left);
                  const rightType = getType(right);
                  if (leftType !== rightType ||
                      !types.includes(leftType)) {
                      return null;
                  }
                  return op(left, right);
              };
              switch (input) {
                  case '+': return nullable((a, b) => a + b, ['string', 'number']);
                  case '-': return nullable((a, b) => a - b);
                  case '*': return nullable((a, b) => a * b);
                  case '/': return nullable((a, b) => !b ? null : a / b);
                  case '**':
                  case '^': return nullable((a, b) => Math.pow(a, b));
              }
          };
          case 'CompareOp': return tag(() => {
              switch (input) {
                  case '>': return (b) => createRange(b, null, false, false);
                  case '>=': return (b) => createRange(b, null, true, false);
                  case '<': return (b) => createRange(null, b, false, false);
                  case '<=': return (b) => createRange(null, b, false, true);
                  case '=': return (b) => (a) => equals(a, b);
                  case '!=': return (b) => (a) => !equals(a, b);
              }
          }, Test('boolean'));
          case 'Wildcard': return (_context) => true;
          case 'null': return (_context) => {
              return null;
          };
          case 'Disjunction': return tag((context) => {
              const left = args[0](context);
              const right = args[2](context);
              const matrix = [
                  [true, true, true],
                  [true, false, true],
                  [true, null, true],
                  [false, true, true],
                  [false, false, false],
                  [false, null, null],
                  [null, true, true],
                  [null, false, null],
                  [null, null, null],
              ];
              const a = typeof left === 'boolean' ? left : null;
              const b = typeof right === 'boolean' ? right : null;
              return matrix.find(el => el[0] === a && el[1] === b)[2];
          }, Test('boolean'));
          case 'Conjunction': return tag((context) => {
              const left = args[0](context);
              const right = args[2](context);
              const matrix = [
                  [true, true, true],
                  [true, false, false],
                  [true, null, null],
                  [false, true, false],
                  [false, false, false],
                  [false, null, false],
                  [null, true, null],
                  [null, false, false],
                  [null, null, null],
              ];
              const a = typeof left === 'boolean' ? left : null;
              const b = typeof right === 'boolean' ? right : null;
              return matrix.find(el => el[0] === a && el[1] === b)[2];
          }, Test('boolean'));
          case 'Context': return (context) => {
              return args.slice(1, -1).reduce((obj, arg) => {
                  const [key, value] = arg(Object.assign(Object.assign({}, context), obj));
                  return Object.assign(Object.assign({}, obj), { [key]: value });
              }, {});
          };
          case 'FunctionBody': return args[0];
          case 'FormalParameters': return args;
          case 'FormalParameter': return args[0];
          case 'ParameterName': return args.join(' ');
          case 'FunctionDefinition': return (context) => {
              const parameterNames = args[2];
              const fnBody = args[4];
              return wrapFunction((...args) => {
                  const fnContext = parameterNames.reduce((context, name, idx) => {
                      // support positional parameters
                      context[name] = args[idx];
                      return context;
                  }, Object.assign({}, context));
                  return fnBody(fnContext);
              }, parameterNames);
          };
          case 'ContextEntry': return (context) => {
              const key = typeof args[0] === 'function' ? args[0](context) : args[0];
              const value = args[1](context);
              return [key, value];
          };
          case 'Key': return args[0];
          case 'Identifier': return input;
          case 'SpecialFunctionName': return (context) => getBuiltin(input);
          // preserve spaces in name, but compact multiple
          // spaces into one (token)
          case 'Name': return input.replace(/\s{2,}/g, ' ');
          case 'VariableName': return (context) => {
              const name = args.join(' ');
              return getBuiltin(name) || getFromContext(name, context);
          };
          case 'QualifiedName': return (context) => {
              return args.reduce((context, arg) => arg(context), context);
          };
          case '?': return (context) => getFromContext('?', context);
          // expression
          // expression ".." expression
          case 'IterationContext': return (context) => {
              const a = args[0](context);
              const b = args[1] && args[1](context);
              return b ? createRange(a, b) : a;
          };
          case 'Type': return args[0];
          case 'InExpressions': return (context) => {
              const iterationContexts = args.map(ctx => ctx(context));
              if (iterationContexts.some(ctx => getType(ctx) !== 'list')) {
                  return null;
              }
              return cartesianProduct(iterationContexts).map(ctx => {
                  if (!isArray$1(ctx)) {
                      ctx = [ctx];
                  }
                  return Object.assign({}, context, ...ctx);
              });
          };
          // Name kw<"in"> Expr
          case 'InExpression': return (context) => {
              return extractValue(context, args[0], args[2]);
          };
          case 'SpecialType': throw notImplemented('SpecialType');
          case 'InstanceOfExpression': return tag((context) => {
              const a = args[0](context);
              const b = args[3](context);
              return a instanceof b;
          }, Test('boolean'));
          case 'every': return tag((context) => {
              return (_contexts, _condition) => {
                  const contexts = _contexts(context);
                  if (getType(contexts) !== 'list') {
                      return contexts;
                  }
                  return contexts.every(ctx => isTruthy(_condition(ctx)));
              };
          }, Test('boolean'));
          case 'some': return tag((context) => {
              return (_contexts, _condition) => {
                  const contexts = _contexts(context);
                  if (getType(contexts) !== 'list') {
                      return contexts;
                  }
                  return contexts.some(ctx => isTruthy(_condition(ctx)));
              };
          }, Test('boolean'));
          case 'NumericLiteral': return tag((_context) => input.includes('.') ? parseFloat(input) : parseInt(input), 'number');
          case 'BooleanLiteral': return tag((_context) => input === 'true' ? true : false, 'boolean');
          case 'StringLiteral': return tag((_context) => parseString(input), 'string');
          case 'PositionalParameters': return (context) => args.map(arg => arg(context));
          case 'NamedParameter': return (context) => {
              const name = args[0];
              const value = args[1](context);
              return [name, value];
          };
          case 'NamedParameters': return (context) => args.reduce((args, arg) => {
              const [name, value] = arg(context);
              args[name] = value;
              return args;
          }, {});
          case 'DateTimeConstructor': return (context) => {
              return getBuiltin(input);
          };
          case 'DateTimeLiteral': return (context) => {
              // AtLiteral
              if (args.length === 1) {
                  return args[0](context);
              }
              // FunctionInvocation
              else {
                  const wrappedFn = wrapFunction(args[0](context));
                  if (!wrappedFn) {
                      throw new Error(`Failed to evaluate ${input}: Target is not a function`);
                  }
                  const contextOrArgs = args[2](context);
                  return wrappedFn.invoke(contextOrArgs);
              }
          };
          case 'AtLiteral': return (context) => {
              const wrappedFn = wrapFunction(getBuiltin('@'));
              if (!wrappedFn) {
                  throw new Error(`Failed to evaluate ${input}: Target is not a function`);
              }
              return wrappedFn.invoke([args[0](context)]);
          };
          case 'FunctionInvocation': return (context) => {
              const wrappedFn = wrapFunction(args[0](context));
              if (!wrappedFn) {
                  throw new Error(`Failed to evaluate ${input}: Target is not a function`);
              }
              const contextOrArgs = args[2](context);
              return wrappedFn.invoke(contextOrArgs);
          };
          case 'IfExpression': return (function () {
              const ifCondition = args[1];
              const thenValue = args[3];
              const elseValue = args[5];
              const type = coalecenseTypes(thenValue, elseValue);
              return tag((context) => {
                  if (isTruthy(ifCondition(context))) {
                      return thenValue(context);
                  }
                  else {
                      return elseValue ? elseValue(context) : null;
                  }
              }, type);
          })();
          case 'Parameters': return args.length === 3 ? args[1] : (_context) => [];
          case 'Comparison': return (context) => {
              const operator = args[1];
              // expression !compare kw<"in"> PositiveUnaryTest |
              // expression !compare kw<"in"> !unaryTest "(" PositiveUnaryTests ")"
              if (operator === 'in') {
                  return compareIn(args[0](context), (args[3] || args[2])(context));
              }
              // expression !compare kw<"between"> expression kw<"and"> expression
              if (operator === 'between') {
                  const start = args[2](context);
                  const end = args[4](context);
                  if (start === null || end === null) {
                      return null;
                  }
                  return createRange(start, end).includes(args[0](context));
              }
              // expression !compare CompareOp<"=" | "!="> expression |
              // expression !compare CompareOp<Gt | Gte | Lt | Lte> expression |
              const left = args[0](context);
              const right = args[2](context);
              const test = operator()(right);
              return compareValue(test, left);
          };
          case 'QuantifiedExpression': return (context) => {
              const testFn = args[0](context);
              const contexts = args[1];
              const condition = args[3];
              return testFn(contexts, condition);
          };
          // DMN 1.2 - 10.3.2.14
          // kw<"for"> commaSep1<InExpression<IterationContext>> kw<"return"> expression
          case 'ForExpression': return (context) => {
              const extractor = args[args.length - 1];
              const iterationContexts = args[1](context);
              if (getType(iterationContexts) !== 'list') {
                  return iterationContexts;
              }
              const partial = [];
              for (const ctx of iterationContexts) {
                  partial.push(extractor(Object.assign(Object.assign({}, ctx), { partial })));
              }
              return partial;
          };
          case 'ArithmeticExpression': return (function () {
              // binary expression (a + b)
              if (args.length === 3) {
                  const [a, op, b] = args;
                  return tag((context) => {
                      return op(context)(a, b);
                  }, coalecenseTypes(a, b));
              }
              // unary expression (-b)
              if (args.length === 2) {
                  const [op, value] = args;
                  return tag((context) => {
                      return op(context)(() => 0, value);
                  }, value.type);
              }
          })();
          case 'PositiveUnaryTest': return args[0];
          case 'ParenthesizedExpression': return args[1];
          case 'PathExpression': return (context) => {
              const pathTarget = coerceSingleton(args[0](context));
              const pathProp = args[1];
              if (isArray$1(pathTarget)) {
                  return pathTarget.map(pathProp).filter(e => e !== null);
              }
              else {
                  return pathProp(pathTarget);
              }
          };
          // expression !filter "[" expression "]"
          case 'FilterExpression': return (context) => {
              const target = args[0](context);
              const filterFn = args[2];
              const filterTarget = isArray$1(target) ? target : [target];
              // null[..]
              if (target === null) {
                  return null;
              }
              // a[1]
              if (filterFn.type === 'number') {
                  const idx = filterFn(context);
                  const value = filterTarget[idx < 0 ? filterTarget.length + idx : idx - 1];
                  if (typeof value === 'undefined') {
                      return null;
                  }
                  else {
                      return value;
                  }
              }
              // a[true]
              if (filterFn.type === 'boolean') {
                  if (filterFn(context)) {
                      return filterTarget;
                  }
                  else {
                      return [];
                  }
              }
              if (filterFn.type === 'string') {
                  const value = filterFn(context);
                  return filterTarget.filter(el => el === value);
              }
              // a[test]
              return filterTarget.map(el => {
                  const iterationContext = Object.assign(Object.assign(Object.assign({}, context), { item: el }), el);
                  let result = filterFn(iterationContext);
                  // test is fn(val) => boolean SimpleUnaryTest
                  if (typeof result === 'function') {
                      result = result(el);
                  }
                  if (result instanceof Range$1) {
                      result = result.includes(el);
                  }
                  if (result === true) {
                      return el;
                  }
                  return result;
              }).filter(isTruthy);
          };
          case 'SimplePositiveUnaryTest': return tag((context) => {
              // <Interval>
              if (args.length === 1) {
                  return args[0](context);
              }
              // <CompareOp> <Expr>
              return args[0](context)(args[1](context));
          }, 'test');
          case 'List': return (context) => {
              return args.slice(1, -1).map(arg => arg(context));
          };
          case 'Interval': return tag((context) => {
              const left = args[1](context);
              const right = args[2](context);
              const startIncluded = left !== null && args[0] === '[';
              const endIncluded = right !== null && args[3] === ']';
              return createRange(left, right, startIncluded, endIncluded);
          }, Test('boolean'));
          case 'PositiveUnaryTests':
          case 'Expressions': return (context) => {
              return args.map(a => a(context));
          };
          case 'UnaryTests': return (context) => {
              return (value = null) => {
                  const negate = args[0] === 'not';
                  const tests = negate ? args.slice(2, -1) : args;
                  const matches = tests.map(test => test(context)).flat(1).map(test => {
                      if (isArray$1(test)) {
                          return test.includes(value);
                      }
                      if (test === null) {
                          return null;
                      }
                      if (typeof test === 'boolean') {
                          return test;
                      }
                      return compareValue(test, value);
                  }).reduce(combineResult, undefined);
                  return matches === null ? null : (negate ? !matches : matches);
              };
          };
          default: return node.name;
      }
  }
  function getBuiltin(name, _context) {
      return getFromContext(name, builtins);
  }
  function extractValue(context, prop, _target) {
      const target = _target(context);
      if (['list', 'range'].includes(getType(target))) {
          return target.map(t => ({ [prop]: t }));
      }
      return null;
  }
  function compareIn(value, tests) {
      if (!isArray$1(tests)) {
          if (getType(tests) === 'nil') {
              return null;
          }
          tests = [tests];
      }
      return tests.some(test => compareValue(test, value));
  }
  function compareValue(test, value) {
      if (typeof test === 'function') {
          return test(value);
      }
      if (test instanceof Range$1) {
          return test.includes(value);
      }
      return equals(test, value);
  }
  const chars = Array.from('abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ');
  function isTyped(type, values) {
      return (values.some(e => getType(e) === type) &&
          values.every(e => e === null || getType(e) === type));
  }
  const nullRange = new Range$1({
      start: null,
      end: null,
      'start included': false,
      'end included': false,
      map() {
          return [];
      },
      includes() {
          return null;
      }
  });
  function createRange(start, end, startIncluded = true, endIncluded = true) {
      if (isTyped('string', [start, end])) {
          return createStringRange(start, end, startIncluded, endIncluded);
      }
      if (isTyped('number', [start, end])) {
          return createNumberRange(start, end, startIncluded, endIncluded);
      }
      if (isTyped('duration', [start, end])) {
          throw notImplemented('range<duration>');
      }
      if (isTyped('time', [start, end])) {
          throw notImplemented('range<time>');
      }
      if (isTyped('date time', [start, end])) {
          throw notImplemented('range<date and time>');
      }
      if (isTyped('date', [start, end])) {
          throw notImplemented('range<date>');
      }
      if (start === null && end === null) {
          return nullRange;
      }
      throw new Error(`unsupported range: ${start}..${end}`);
  }
  function noopMap() {
      return () => {
          throw new Error('unsupported range operation: map');
      };
  }
  function valuesMap(values) {
      return (fn) => values.map(fn);
  }
  function valuesIncludes(values) {
      return (value) => values.includes(value);
  }
  function numberMap(start, end, startIncluded, endIncluded) {
      const direction = start > end ? -1 : 1;
      return (fn) => {
          const result = [];
          for (let i = start;; i += direction) {
              if (i === 0 && !startIncluded) {
                  continue;
              }
              if (i === end && !endIncluded) {
                  break;
              }
              result.push(fn(i));
              if (i === end) {
                  break;
              }
          }
          return result;
      };
  }
  function includesStart(n, inclusive) {
      if (inclusive) {
          return (value) => n <= value;
      }
      else {
          return (value) => n < value;
      }
  }
  function includesEnd(n, inclusive) {
      if (inclusive) {
          return (value) => n >= value;
      }
      else {
          return (value) => n > value;
      }
  }
  function anyIncludes(start, end, startIncluded, endIncluded) {
      let tests = [];
      if (start === null && end === null) {
          return () => null;
      }
      if (start !== null && end !== null) {
          if (start > end) {
              tests = [
                  includesStart(end, endIncluded),
                  includesEnd(start, startIncluded)
              ];
          }
          else {
              tests = [
                  includesStart(start, startIncluded),
                  includesEnd(end, endIncluded)
              ];
          }
      }
      else if (end !== null) {
          tests = [
              includesEnd(end, endIncluded)
          ];
      }
      else if (start !== null) {
          tests = [
              includesStart(start, startIncluded)
          ];
      }
      return (value) => value === null ? null : tests.every(t => t(value));
  }
  function createStringRange(start, end, startIncluded = true, endIncluded = true) {
      if (start !== null && !chars.includes(start)) {
          throw new Error('illegal range start: ' + start);
      }
      if (end !== null && !chars.includes(end)) {
          throw new Error('illegal range end: ' + end);
      }
      let values;
      if (start !== null && end !== null) {
          let startIdx = chars.indexOf(start);
          let endIdx = chars.indexOf(end);
          const direction = startIdx > endIdx ? -1 : 1;
          if (startIncluded === false) {
              startIdx += direction;
          }
          if (endIncluded === false) {
              endIdx -= direction;
          }
          values = chars.slice(startIdx, endIdx + 1);
      }
      const map = values ? valuesMap(values) : noopMap();
      const includes = values ? valuesIncludes(values) : anyIncludes(start, end, startIncluded, endIncluded);
      return new Range$1({
          start,
          end,
          'start included': startIncluded,
          'end included': endIncluded,
          map,
          includes
      });
  }
  function createNumberRange(start, end, startIncluded, endIncluded) {
      const map = start !== null && end !== null ? numberMap(start, end, startIncluded, endIncluded) : noopMap();
      const includes = anyIncludes(start, end, startIncluded, endIncluded);
      return new Range$1({
          start,
          end,
          'start included': startIncluded,
          'end included': endIncluded,
          map,
          includes
      });
  }
  function cartesianProduct(arrays) {
      if (arrays.some(arr => getType(arr) === 'nil')) {
          return null;
      }
      const f = (a, b) => [].concat(...a.map(d => b.map(e => [].concat(d, e))));
      const cartesian = (a, b, ...c) => (b ? cartesian(f(a, b), ...c) : a || []);
      return cartesian(...arrays);
  }
  function coalecenseTypes(a, b) {
      if (!b) {
          return a.type;
      }
      if (a.type === b.type) {
          return a.type;
      }
      return 'any';
  }
  function tag(fn, type) {
      return Object.assign(fn, {
          type,
          toString() {
              return `TaggedFunction[${type}] ${Function.prototype.toString.call(fn)}`;
          }
      });
  }
  function combineResult(result, match) {
      if (!result) {
          return match;
      }
      return result;
  }
  function isTruthy(obj) {
      return obj !== false && obj !== null;
  }
  function Test(type) {
      return `Test<${type}>`;
  }
  /**
   * @param {Function} fn
   * @param {string[]} [parameterNames]
   *
   * @return {FunctionWrapper}
   */
  function wrapFunction(fn, parameterNames = null) {
      if (!fn) {
          return null;
      }
      if (fn instanceof FunctionWrapper) {
          return fn;
      }
      if (fn instanceof Range$1) {
          return new FunctionWrapper((value) => fn.includes(value), ['value']);
      }
      return new FunctionWrapper(fn, parameterNames || parseParameterNames(fn));
  }
  function coerceSingleton(values) {
      if (Array.isArray(values) && values.length === 1) {
          return values[0];
      }
      else {
          return values;
      }
  }
  function parseString(str) {
      if (str.startsWith('"')) {
          str = str.slice(1);
      }
      if (str.endsWith('"')) {
          str = str.slice(0, -1);
      }
      return str.replace(/(\\")|(\\\\)|(\\u[a-fA-F0-9]{5,6})|((?:\\u[a-fA-F0-9]{1,4})+)/ig, function (substring, ...groups) {
          const [quotes, escape, codePoint, charCodes] = groups;
          if (quotes) {
              return '"';
          }
          if (escape) {
              return '\\';
          }
          const escapePattern = /\\u([a-fA-F0-9]+)/ig;
          if (codePoint) {
              const codePointMatch = escapePattern.exec(codePoint);
              return String.fromCodePoint(parseInt(codePointMatch[1], 16));
          }
          if (charCodes) {
              const chars = [];
              let charCodeMatch;
              while ((charCodeMatch = escapePattern.exec(substring)) !== null) {
                  chars.push(parseInt(charCodeMatch[1], 16));
              }
              return String.fromCharCode(...chars);
          }
          throw new Error('illegal match');
      });
  }

  /**
  The data structure for documents. @nonabstract
  */
  let Text$1 = class Text {
      /**
      @internal
      */
      constructor() { }
      /**
      Get the line description around the given position.
      */
      lineAt(pos) {
          if (pos < 0 || pos > this.length)
              throw new RangeError(`Invalid position ${pos} in document of length ${this.length}`);
          return this.lineInner(pos, false, 1, 0);
      }
      /**
      Get the description for the given (1-based) line number.
      */
      line(n) {
          if (n < 1 || n > this.lines)
              throw new RangeError(`Invalid line number ${n} in ${this.lines}-line document`);
          return this.lineInner(n, true, 1, 0);
      }
      /**
      Replace a range of the text with the given content.
      */
      replace(from, to, text) {
          let parts = [];
          this.decompose(0, from, parts, 2 /* Open.To */);
          if (text.length)
              text.decompose(0, text.length, parts, 1 /* Open.From */ | 2 /* Open.To */);
          this.decompose(to, this.length, parts, 1 /* Open.From */);
          return TextNode.from(parts, this.length - (to - from) + text.length);
      }
      /**
      Append another document to this one.
      */
      append(other) {
          return this.replace(this.length, this.length, other);
      }
      /**
      Retrieve the text between the given points.
      */
      slice(from, to = this.length) {
          let parts = [];
          this.decompose(from, to, parts, 0);
          return TextNode.from(parts, to - from);
      }
      /**
      Test whether this text is equal to another instance.
      */
      eq(other) {
          if (other == this)
              return true;
          if (other.length != this.length || other.lines != this.lines)
              return false;
          let start = this.scanIdentical(other, 1), end = this.length - this.scanIdentical(other, -1);
          let a = new RawTextCursor(this), b = new RawTextCursor(other);
          for (let skip = start, pos = start;;) {
              a.next(skip);
              b.next(skip);
              skip = 0;
              if (a.lineBreak != b.lineBreak || a.done != b.done || a.value != b.value)
                  return false;
              pos += a.value.length;
              if (a.done || pos >= end)
                  return true;
          }
      }
      /**
      Iterate over the text. When `dir` is `-1`, iteration happens
      from end to start. This will return lines and the breaks between
      them as separate strings.
      */
      iter(dir = 1) { return new RawTextCursor(this, dir); }
      /**
      Iterate over a range of the text. When `from` > `to`, the
      iterator will run in reverse.
      */
      iterRange(from, to = this.length) { return new PartialTextCursor(this, from, to); }
      /**
      Return a cursor that iterates over the given range of lines,
      _without_ returning the line breaks between, and yielding empty
      strings for empty lines.
      
      When `from` and `to` are given, they should be 1-based line numbers.
      */
      iterLines(from, to) {
          let inner;
          if (from == null) {
              inner = this.iter();
          }
          else {
              if (to == null)
                  to = this.lines + 1;
              let start = this.line(from).from;
              inner = this.iterRange(start, Math.max(start, to == this.lines + 1 ? this.length : to <= 1 ? 0 : this.line(to - 1).to));
          }
          return new LineCursor(inner);
      }
      /**
      @internal
      */
      toString() { return this.sliceString(0); }
      /**
      Convert the document to an array of lines (which can be
      deserialized again via [`Text.of`](https://codemirror.net/6/docs/ref/#state.Text^of)).
      */
      toJSON() {
          let lines = [];
          this.flatten(lines);
          return lines;
      }
      /**
      Create a `Text` instance for the given array of lines.
      */
      static of(text) {
          if (text.length == 0)
              throw new RangeError("A document must have at least one line");
          if (text.length == 1 && !text[0])
              return Text$1.empty;
          return text.length <= 32 /* Tree.Branch */ ? new TextLeaf(text) : TextNode.from(TextLeaf.split(text, []));
      }
  };
  // Leaves store an array of line strings. There are always line breaks
  // between these strings. Leaves are limited in size and have to be
  // contained in TextNode instances for bigger documents.
  class TextLeaf extends Text$1 {
      constructor(text, length = textLength(text)) {
          super();
          this.text = text;
          this.length = length;
      }
      get lines() { return this.text.length; }
      get children() { return null; }
      lineInner(target, isLine, line, offset) {
          for (let i = 0;; i++) {
              let string = this.text[i], end = offset + string.length;
              if ((isLine ? line : end) >= target)
                  return new Line$1(offset, end, line, string);
              offset = end + 1;
              line++;
          }
      }
      decompose(from, to, target, open) {
          let text = from <= 0 && to >= this.length ? this
              : new TextLeaf(sliceText(this.text, from, to), Math.min(to, this.length) - Math.max(0, from));
          if (open & 1 /* Open.From */) {
              let prev = target.pop();
              let joined = appendText(text.text, prev.text.slice(), 0, text.length);
              if (joined.length <= 32 /* Tree.Branch */) {
                  target.push(new TextLeaf(joined, prev.length + text.length));
              }
              else {
                  let mid = joined.length >> 1;
                  target.push(new TextLeaf(joined.slice(0, mid)), new TextLeaf(joined.slice(mid)));
              }
          }
          else {
              target.push(text);
          }
      }
      replace(from, to, text) {
          if (!(text instanceof TextLeaf))
              return super.replace(from, to, text);
          let lines = appendText(this.text, appendText(text.text, sliceText(this.text, 0, from)), to);
          let newLen = this.length + text.length - (to - from);
          if (lines.length <= 32 /* Tree.Branch */)
              return new TextLeaf(lines, newLen);
          return TextNode.from(TextLeaf.split(lines, []), newLen);
      }
      sliceString(from, to = this.length, lineSep = "\n") {
          let result = "";
          for (let pos = 0, i = 0; pos <= to && i < this.text.length; i++) {
              let line = this.text[i], end = pos + line.length;
              if (pos > from && i)
                  result += lineSep;
              if (from < end && to > pos)
                  result += line.slice(Math.max(0, from - pos), to - pos);
              pos = end + 1;
          }
          return result;
      }
      flatten(target) {
          for (let line of this.text)
              target.push(line);
      }
      scanIdentical() { return 0; }
      static split(text, target) {
          let part = [], len = -1;
          for (let line of text) {
              part.push(line);
              len += line.length + 1;
              if (part.length == 32 /* Tree.Branch */) {
                  target.push(new TextLeaf(part, len));
                  part = [];
                  len = -1;
              }
          }
          if (len > -1)
              target.push(new TextLeaf(part, len));
          return target;
      }
  }
  // Nodes provide the tree structure of the `Text` type. They store a
  // number of other nodes or leaves, taking care to balance themselves
  // on changes. There are implied line breaks _between_ the children of
  // a node (but not before the first or after the last child).
  class TextNode extends Text$1 {
      constructor(children, length) {
          super();
          this.children = children;
          this.length = length;
          this.lines = 0;
          for (let child of children)
              this.lines += child.lines;
      }
      lineInner(target, isLine, line, offset) {
          for (let i = 0;; i++) {
              let child = this.children[i], end = offset + child.length, endLine = line + child.lines - 1;
              if ((isLine ? endLine : end) >= target)
                  return child.lineInner(target, isLine, line, offset);
              offset = end + 1;
              line = endLine + 1;
          }
      }
      decompose(from, to, target, open) {
          for (let i = 0, pos = 0; pos <= to && i < this.children.length; i++) {
              let child = this.children[i], end = pos + child.length;
              if (from <= end && to >= pos) {
                  let childOpen = open & ((pos <= from ? 1 /* Open.From */ : 0) | (end >= to ? 2 /* Open.To */ : 0));
                  if (pos >= from && end <= to && !childOpen)
                      target.push(child);
                  else
                      child.decompose(from - pos, to - pos, target, childOpen);
              }
              pos = end + 1;
          }
      }
      replace(from, to, text) {
          if (text.lines < this.lines)
              for (let i = 0, pos = 0; i < this.children.length; i++) {
                  let child = this.children[i], end = pos + child.length;
                  // Fast path: if the change only affects one child and the
                  // child's size remains in the acceptable range, only update
                  // that child
                  if (from >= pos && to <= end) {
                      let updated = child.replace(from - pos, to - pos, text);
                      let totalLines = this.lines - child.lines + updated.lines;
                      if (updated.lines < (totalLines >> (5 /* Tree.BranchShift */ - 1)) &&
                          updated.lines > (totalLines >> (5 /* Tree.BranchShift */ + 1))) {
                          let copy = this.children.slice();
                          copy[i] = updated;
                          return new TextNode(copy, this.length - (to - from) + text.length);
                      }
                      return super.replace(pos, end, updated);
                  }
                  pos = end + 1;
              }
          return super.replace(from, to, text);
      }
      sliceString(from, to = this.length, lineSep = "\n") {
          let result = "";
          for (let i = 0, pos = 0; i < this.children.length && pos <= to; i++) {
              let child = this.children[i], end = pos + child.length;
              if (pos > from && i)
                  result += lineSep;
              if (from < end && to > pos)
                  result += child.sliceString(from - pos, to - pos, lineSep);
              pos = end + 1;
          }
          return result;
      }
      flatten(target) {
          for (let child of this.children)
              child.flatten(target);
      }
      scanIdentical(other, dir) {
          if (!(other instanceof TextNode))
              return 0;
          let length = 0;
          let [iA, iB, eA, eB] = dir > 0 ? [0, 0, this.children.length, other.children.length]
              : [this.children.length - 1, other.children.length - 1, -1, -1];
          for (;; iA += dir, iB += dir) {
              if (iA == eA || iB == eB)
                  return length;
              let chA = this.children[iA], chB = other.children[iB];
              if (chA != chB)
                  return length + chA.scanIdentical(chB, dir);
              length += chA.length + 1;
          }
      }
      static from(children, length = children.reduce((l, ch) => l + ch.length + 1, -1)) {
          let lines = 0;
          for (let ch of children)
              lines += ch.lines;
          if (lines < 32 /* Tree.Branch */) {
              let flat = [];
              for (let ch of children)
                  ch.flatten(flat);
              return new TextLeaf(flat, length);
          }
          let chunk = Math.max(32 /* Tree.Branch */, lines >> 5 /* Tree.BranchShift */), maxChunk = chunk << 1, minChunk = chunk >> 1;
          let chunked = [], currentLines = 0, currentLen = -1, currentChunk = [];
          function add(child) {
              let last;
              if (child.lines > maxChunk && child instanceof TextNode) {
                  for (let node of child.children)
                      add(node);
              }
              else if (child.lines > minChunk && (currentLines > minChunk || !currentLines)) {
                  flush();
                  chunked.push(child);
              }
              else if (child instanceof TextLeaf && currentLines &&
                  (last = currentChunk[currentChunk.length - 1]) instanceof TextLeaf &&
                  child.lines + last.lines <= 32 /* Tree.Branch */) {
                  currentLines += child.lines;
                  currentLen += child.length + 1;
                  currentChunk[currentChunk.length - 1] = new TextLeaf(last.text.concat(child.text), last.length + 1 + child.length);
              }
              else {
                  if (currentLines + child.lines > chunk)
                      flush();
                  currentLines += child.lines;
                  currentLen += child.length + 1;
                  currentChunk.push(child);
              }
          }
          function flush() {
              if (currentLines == 0)
                  return;
              chunked.push(currentChunk.length == 1 ? currentChunk[0] : TextNode.from(currentChunk, currentLen));
              currentLen = -1;
              currentLines = currentChunk.length = 0;
          }
          for (let child of children)
              add(child);
          flush();
          return chunked.length == 1 ? chunked[0] : new TextNode(chunked, length);
      }
  }
  Text$1.empty = /*@__PURE__*/new TextLeaf([""], 0);
  function textLength(text) {
      let length = -1;
      for (let line of text)
          length += line.length + 1;
      return length;
  }
  function appendText(text, target, from = 0, to = 1e9) {
      for (let pos = 0, i = 0, first = true; i < text.length && pos <= to; i++) {
          let line = text[i], end = pos + line.length;
          if (end >= from) {
              if (end > to)
                  line = line.slice(0, to - pos);
              if (pos < from)
                  line = line.slice(from - pos);
              if (first) {
                  target[target.length - 1] += line;
                  first = false;
              }
              else
                  target.push(line);
          }
          pos = end + 1;
      }
      return target;
  }
  function sliceText(text, from, to) {
      return appendText(text, [""], from, to);
  }
  class RawTextCursor {
      constructor(text, dir = 1) {
          this.dir = dir;
          this.done = false;
          this.lineBreak = false;
          this.value = "";
          this.nodes = [text];
          this.offsets = [dir > 0 ? 1 : (text instanceof TextLeaf ? text.text.length : text.children.length) << 1];
      }
      nextInner(skip, dir) {
          this.done = this.lineBreak = false;
          for (;;) {
              let last = this.nodes.length - 1;
              let top = this.nodes[last], offsetValue = this.offsets[last], offset = offsetValue >> 1;
              let size = top instanceof TextLeaf ? top.text.length : top.children.length;
              if (offset == (dir > 0 ? size : 0)) {
                  if (last == 0) {
                      this.done = true;
                      this.value = "";
                      return this;
                  }
                  if (dir > 0)
                      this.offsets[last - 1]++;
                  this.nodes.pop();
                  this.offsets.pop();
              }
              else if ((offsetValue & 1) == (dir > 0 ? 0 : 1)) {
                  this.offsets[last] += dir;
                  if (skip == 0) {
                      this.lineBreak = true;
                      this.value = "\n";
                      return this;
                  }
                  skip--;
              }
              else if (top instanceof TextLeaf) {
                  // Move to the next string
                  let next = top.text[offset + (dir < 0 ? -1 : 0)];
                  this.offsets[last] += dir;
                  if (next.length > Math.max(0, skip)) {
                      this.value = skip == 0 ? next : dir > 0 ? next.slice(skip) : next.slice(0, next.length - skip);
                      return this;
                  }
                  skip -= next.length;
              }
              else {
                  let next = top.children[offset + (dir < 0 ? -1 : 0)];
                  if (skip > next.length) {
                      skip -= next.length;
                      this.offsets[last] += dir;
                  }
                  else {
                      if (dir < 0)
                          this.offsets[last]--;
                      this.nodes.push(next);
                      this.offsets.push(dir > 0 ? 1 : (next instanceof TextLeaf ? next.text.length : next.children.length) << 1);
                  }
              }
          }
      }
      next(skip = 0) {
          if (skip < 0) {
              this.nextInner(-skip, (-this.dir));
              skip = this.value.length;
          }
          return this.nextInner(skip, this.dir);
      }
  }
  class PartialTextCursor {
      constructor(text, start, end) {
          this.value = "";
          this.done = false;
          this.cursor = new RawTextCursor(text, start > end ? -1 : 1);
          this.pos = start > end ? text.length : 0;
          this.from = Math.min(start, end);
          this.to = Math.max(start, end);
      }
      nextInner(skip, dir) {
          if (dir < 0 ? this.pos <= this.from : this.pos >= this.to) {
              this.value = "";
              this.done = true;
              return this;
          }
          skip += Math.max(0, dir < 0 ? this.pos - this.to : this.from - this.pos);
          let limit = dir < 0 ? this.pos - this.from : this.to - this.pos;
          if (skip > limit)
              skip = limit;
          limit -= skip;
          let { value } = this.cursor.next(skip);
          this.pos += (value.length + skip) * dir;
          this.value = value.length <= limit ? value : dir < 0 ? value.slice(value.length - limit) : value.slice(0, limit);
          this.done = !this.value;
          return this;
      }
      next(skip = 0) {
          if (skip < 0)
              skip = Math.max(skip, this.from - this.pos);
          else if (skip > 0)
              skip = Math.min(skip, this.to - this.pos);
          return this.nextInner(skip, this.cursor.dir);
      }
      get lineBreak() { return this.cursor.lineBreak && this.value != ""; }
  }
  class LineCursor {
      constructor(inner) {
          this.inner = inner;
          this.afterBreak = true;
          this.value = "";
          this.done = false;
      }
      next(skip = 0) {
          let { done, lineBreak, value } = this.inner.next(skip);
          if (done) {
              this.done = true;
              this.value = "";
          }
          else if (lineBreak) {
              if (this.afterBreak) {
                  this.value = "";
              }
              else {
                  this.afterBreak = true;
                  this.next();
              }
          }
          else {
              this.value = value;
              this.afterBreak = false;
          }
          return this;
      }
      get lineBreak() { return false; }
  }
  if (typeof Symbol != "undefined") {
      Text$1.prototype[Symbol.iterator] = function () { return this.iter(); };
      RawTextCursor.prototype[Symbol.iterator] = PartialTextCursor.prototype[Symbol.iterator] =
          LineCursor.prototype[Symbol.iterator] = function () { return this; };
  }
  /**
  This type describes a line in the document. It is created
  on-demand when lines are [queried](https://codemirror.net/6/docs/ref/#state.Text.lineAt).
  */
  let Line$1 = class Line {
      /**
      @internal
      */
      constructor(
      /**
      The position of the start of the line.
      */
      from, 
      /**
      The position at the end of the line (_before_ the line break,
      or at the end of document for the last line).
      */
      to, 
      /**
      This line's line number (1-based).
      */
      number, 
      /**
      The line's content.
      */
      text) {
          this.from = from;
          this.to = to;
          this.number = number;
          this.text = text;
      }
      /**
      The length of the line (not including any line break after it).
      */
      get length() { return this.to - this.from; }
  };

  // Compressed representation of the Grapheme_Cluster_Break=Extend
  // information from
  // http://www.unicode.org/Public/13.0.0/ucd/auxiliary/GraphemeBreakProperty.txt.
  // Each pair of elements represents a range, as an offet from the
  // previous range and a length. Numbers are in base-36, with the empty
  // string being a shorthand for 1.
  let extend = /*@__PURE__*/"lc,34,7n,7,7b,19,,,,2,,2,,,20,b,1c,l,g,,2t,7,2,6,2,2,,4,z,,u,r,2j,b,1m,9,9,,o,4,,9,,3,,5,17,3,3b,f,,w,1j,,,,4,8,4,,3,7,a,2,t,,1m,,,,2,4,8,,9,,a,2,q,,2,2,1l,,4,2,4,2,2,3,3,,u,2,3,,b,2,1l,,4,5,,2,4,,k,2,m,6,,,1m,,,2,,4,8,,7,3,a,2,u,,1n,,,,c,,9,,14,,3,,1l,3,5,3,,4,7,2,b,2,t,,1m,,2,,2,,3,,5,2,7,2,b,2,s,2,1l,2,,,2,4,8,,9,,a,2,t,,20,,4,,2,3,,,8,,29,,2,7,c,8,2q,,2,9,b,6,22,2,r,,,,,,1j,e,,5,,2,5,b,,10,9,,2u,4,,6,,2,2,2,p,2,4,3,g,4,d,,2,2,6,,f,,jj,3,qa,3,t,3,t,2,u,2,1s,2,,7,8,,2,b,9,,19,3,3b,2,y,,3a,3,4,2,9,,6,3,63,2,2,,1m,,,7,,,,,2,8,6,a,2,,1c,h,1r,4,1c,7,,,5,,14,9,c,2,w,4,2,2,,3,1k,,,2,3,,,3,1m,8,2,2,48,3,,d,,7,4,,6,,3,2,5i,1m,,5,ek,,5f,x,2da,3,3x,,2o,w,fe,6,2x,2,n9w,4,,a,w,2,28,2,7k,,3,,4,,p,2,5,,47,2,q,i,d,,12,8,p,b,1a,3,1c,,2,4,2,2,13,,1v,6,2,2,2,2,c,,8,,1b,,1f,,,3,2,2,5,2,,,16,2,8,,6m,,2,,4,,fn4,,kh,g,g,g,a6,2,gt,,6a,,45,5,1ae,3,,2,5,4,14,3,4,,4l,2,fx,4,ar,2,49,b,4w,,1i,f,1k,3,1d,4,2,2,1x,3,10,5,,8,1q,,c,2,1g,9,a,4,2,,2n,3,2,,,2,6,,4g,,3,8,l,2,1l,2,,,,,m,,e,7,3,5,5f,8,2,3,,,n,,29,,2,6,,,2,,,2,,2,6j,,2,4,6,2,,2,r,2,2d,8,2,,,2,2y,,,,2,6,,,2t,3,2,4,,5,77,9,,2,6t,,a,2,,,4,,40,4,2,2,4,,w,a,14,6,2,4,8,,9,6,2,3,1a,d,,2,ba,7,,6,,,2a,m,2,7,,2,,2,3e,6,3,,,2,,7,,,20,2,3,,,,9n,2,f0b,5,1n,7,t4,,1r,4,29,,f5k,2,43q,,,3,4,5,8,8,2,7,u,4,44,3,1iz,1j,4,1e,8,,e,,m,5,,f,11s,7,,h,2,7,,2,,5,79,7,c5,4,15s,7,31,7,240,5,gx7k,2o,3k,6o".split(",").map(s => s ? parseInt(s, 36) : 1);
  // Convert offsets into absolute values
  for (let i = 1; i < extend.length; i++)
      extend[i] += extend[i - 1];
  function isExtendingChar(code) {
      for (let i = 1; i < extend.length; i += 2)
          if (extend[i] > code)
              return extend[i - 1] <= code;
      return false;
  }
  function isRegionalIndicator(code) {
      return code >= 0x1F1E6 && code <= 0x1F1FF;
  }
  const ZWJ = 0x200d;
  /**
  Returns a next grapheme cluster break _after_ (not equal to)
  `pos`, if `forward` is true, or before otherwise. Returns `pos`
  itself if no further cluster break is available in the string.
  Moves across surrogate pairs, extending characters (when
  `includeExtending` is true), characters joined with zero-width
  joiners, and flag emoji.
  */
  function findClusterBreak(str, pos, forward = true, includeExtending = true) {
      return (forward ? nextClusterBreak : prevClusterBreak)(str, pos, includeExtending);
  }
  function nextClusterBreak(str, pos, includeExtending) {
      if (pos == str.length)
          return pos;
      // If pos is in the middle of a surrogate pair, move to its start
      if (pos && surrogateLow(str.charCodeAt(pos)) && surrogateHigh(str.charCodeAt(pos - 1)))
          pos--;
      let prev = codePointAt(str, pos);
      pos += codePointSize(prev);
      while (pos < str.length) {
          let next = codePointAt(str, pos);
          if (prev == ZWJ || next == ZWJ || includeExtending && isExtendingChar(next)) {
              pos += codePointSize(next);
              prev = next;
          }
          else if (isRegionalIndicator(next)) {
              let countBefore = 0, i = pos - 2;
              while (i >= 0 && isRegionalIndicator(codePointAt(str, i))) {
                  countBefore++;
                  i -= 2;
              }
              if (countBefore % 2 == 0)
                  break;
              else
                  pos += 2;
          }
          else {
              break;
          }
      }
      return pos;
  }
  function prevClusterBreak(str, pos, includeExtending) {
      while (pos > 0) {
          let found = nextClusterBreak(str, pos - 2, includeExtending);
          if (found < pos)
              return found;
          pos--;
      }
      return 0;
  }
  function surrogateLow(ch) { return ch >= 0xDC00 && ch < 0xE000; }
  function surrogateHigh(ch) { return ch >= 0xD800 && ch < 0xDC00; }
  /**
  Find the code point at the given position in a string (like the
  [`codePointAt`](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/String/codePointAt)
  string method).
  */
  function codePointAt(str, pos) {
      let code0 = str.charCodeAt(pos);
      if (!surrogateHigh(code0) || pos + 1 == str.length)
          return code0;
      let code1 = str.charCodeAt(pos + 1);
      if (!surrogateLow(code1))
          return code0;
      return ((code0 - 0xd800) << 10) + (code1 - 0xdc00) + 0x10000;
  }
  /**
  The amount of positions a character takes up a JavaScript string.
  */
  function codePointSize(code) { return code < 0x10000 ? 1 : 2; }

  const DefaultSplit = /\r\n?|\n/;
  /**
  Distinguishes different ways in which positions can be mapped.
  */
  var MapMode = /*@__PURE__*/(function (MapMode) {
      /**
      Map a position to a valid new position, even when its context
      was deleted.
      */
      MapMode[MapMode["Simple"] = 0] = "Simple";
      /**
      Return null if deletion happens across the position.
      */
      MapMode[MapMode["TrackDel"] = 1] = "TrackDel";
      /**
      Return null if the character _before_ the position is deleted.
      */
      MapMode[MapMode["TrackBefore"] = 2] = "TrackBefore";
      /**
      Return null if the character _after_ the position is deleted.
      */
      MapMode[MapMode["TrackAfter"] = 3] = "TrackAfter";
  return MapMode})(MapMode || (MapMode = {}));
  /**
  A change description is a variant of [change set](https://codemirror.net/6/docs/ref/#state.ChangeSet)
  that doesn't store the inserted text. As such, it can't be
  applied, but is cheaper to store and manipulate.
  */
  class ChangeDesc {
      // Sections are encoded as pairs of integers. The first is the
      // length in the current document, and the second is -1 for
      // unaffected sections, and the length of the replacement content
      // otherwise. So an insertion would be (0, n>0), a deletion (n>0,
      // 0), and a replacement two positive numbers.
      /**
      @internal
      */
      constructor(
      /**
      @internal
      */
      sections) {
          this.sections = sections;
      }
      /**
      The length of the document before the change.
      */
      get length() {
          let result = 0;
          for (let i = 0; i < this.sections.length; i += 2)
              result += this.sections[i];
          return result;
      }
      /**
      The length of the document after the change.
      */
      get newLength() {
          let result = 0;
          for (let i = 0; i < this.sections.length; i += 2) {
              let ins = this.sections[i + 1];
              result += ins < 0 ? this.sections[i] : ins;
          }
          return result;
      }
      /**
      False when there are actual changes in this set.
      */
      get empty() { return this.sections.length == 0 || this.sections.length == 2 && this.sections[1] < 0; }
      /**
      Iterate over the unchanged parts left by these changes. `posA`
      provides the position of the range in the old document, `posB`
      the new position in the changed document.
      */
      iterGaps(f) {
          for (let i = 0, posA = 0, posB = 0; i < this.sections.length;) {
              let len = this.sections[i++], ins = this.sections[i++];
              if (ins < 0) {
                  f(posA, posB, len);
                  posB += len;
              }
              else {
                  posB += ins;
              }
              posA += len;
          }
      }
      /**
      Iterate over the ranges changed by these changes. (See
      [`ChangeSet.iterChanges`](https://codemirror.net/6/docs/ref/#state.ChangeSet.iterChanges) for a
      variant that also provides you with the inserted text.)
      `fromA`/`toA` provides the extent of the change in the starting
      document, `fromB`/`toB` the extent of the replacement in the
      changed document.
      
      When `individual` is true, adjacent changes (which are kept
      separate for [position mapping](https://codemirror.net/6/docs/ref/#state.ChangeDesc.mapPos)) are
      reported separately.
      */
      iterChangedRanges(f, individual = false) {
          iterChanges(this, f, individual);
      }
      /**
      Get a description of the inverted form of these changes.
      */
      get invertedDesc() {
          let sections = [];
          for (let i = 0; i < this.sections.length;) {
              let len = this.sections[i++], ins = this.sections[i++];
              if (ins < 0)
                  sections.push(len, ins);
              else
                  sections.push(ins, len);
          }
          return new ChangeDesc(sections);
      }
      /**
      Compute the combined effect of applying another set of changes
      after this one. The length of the document after this set should
      match the length before `other`.
      */
      composeDesc(other) { return this.empty ? other : other.empty ? this : composeSets(this, other); }
      /**
      Map this description, which should start with the same document
      as `other`, over another set of changes, so that it can be
      applied after it. When `before` is true, map as if the changes
      in `other` happened before the ones in `this`.
      */
      mapDesc(other, before = false) { return other.empty ? this : mapSet(this, other, before); }
      mapPos(pos, assoc = -1, mode = MapMode.Simple) {
          let posA = 0, posB = 0;
          for (let i = 0; i < this.sections.length;) {
              let len = this.sections[i++], ins = this.sections[i++], endA = posA + len;
              if (ins < 0) {
                  if (endA > pos)
                      return posB + (pos - posA);
                  posB += len;
              }
              else {
                  if (mode != MapMode.Simple && endA >= pos &&
                      (mode == MapMode.TrackDel && posA < pos && endA > pos ||
                          mode == MapMode.TrackBefore && posA < pos ||
                          mode == MapMode.TrackAfter && endA > pos))
                      return null;
                  if (endA > pos || endA == pos && assoc < 0 && !len)
                      return pos == posA || assoc < 0 ? posB : posB + ins;
                  posB += ins;
              }
              posA = endA;
          }
          if (pos > posA)
              throw new RangeError(`Position ${pos} is out of range for changeset of length ${posA}`);
          return posB;
      }
      /**
      Check whether these changes touch a given range. When one of the
      changes entirely covers the range, the string `"cover"` is
      returned.
      */
      touchesRange(from, to = from) {
          for (let i = 0, pos = 0; i < this.sections.length && pos <= to;) {
              let len = this.sections[i++], ins = this.sections[i++], end = pos + len;
              if (ins >= 0 && pos <= to && end >= from)
                  return pos < from && end > to ? "cover" : true;
              pos = end;
          }
          return false;
      }
      /**
      @internal
      */
      toString() {
          let result = "";
          for (let i = 0; i < this.sections.length;) {
              let len = this.sections[i++], ins = this.sections[i++];
              result += (result ? " " : "") + len + (ins >= 0 ? ":" + ins : "");
          }
          return result;
      }
      /**
      Serialize this change desc to a JSON-representable value.
      */
      toJSON() { return this.sections; }
      /**
      Create a change desc from its JSON representation (as produced
      by [`toJSON`](https://codemirror.net/6/docs/ref/#state.ChangeDesc.toJSON).
      */
      static fromJSON(json) {
          if (!Array.isArray(json) || json.length % 2 || json.some(a => typeof a != "number"))
              throw new RangeError("Invalid JSON representation of ChangeDesc");
          return new ChangeDesc(json);
      }
      /**
      @internal
      */
      static create(sections) { return new ChangeDesc(sections); }
  }
  /**
  A change set represents a group of modifications to a document. It
  stores the document length, and can only be applied to documents
  with exactly that length.
  */
  class ChangeSet extends ChangeDesc {
      constructor(sections, 
      /**
      @internal
      */
      inserted) {
          super(sections);
          this.inserted = inserted;
      }
      /**
      Apply the changes to a document, returning the modified
      document.
      */
      apply(doc) {
          if (this.length != doc.length)
              throw new RangeError("Applying change set to a document with the wrong length");
          iterChanges(this, (fromA, toA, fromB, _toB, text) => doc = doc.replace(fromB, fromB + (toA - fromA), text), false);
          return doc;
      }
      mapDesc(other, before = false) { return mapSet(this, other, before, true); }
      /**
      Given the document as it existed _before_ the changes, return a
      change set that represents the inverse of this set, which could
      be used to go from the document created by the changes back to
      the document as it existed before the changes.
      */
      invert(doc) {
          let sections = this.sections.slice(), inserted = [];
          for (let i = 0, pos = 0; i < sections.length; i += 2) {
              let len = sections[i], ins = sections[i + 1];
              if (ins >= 0) {
                  sections[i] = ins;
                  sections[i + 1] = len;
                  let index = i >> 1;
                  while (inserted.length < index)
                      inserted.push(Text$1.empty);
                  inserted.push(len ? doc.slice(pos, pos + len) : Text$1.empty);
              }
              pos += len;
          }
          return new ChangeSet(sections, inserted);
      }
      /**
      Combine two subsequent change sets into a single set. `other`
      must start in the document produced by `this`. If `this` goes
      `docA` → `docB` and `other` represents `docB` → `docC`, the
      returned value will represent the change `docA` → `docC`.
      */
      compose(other) { return this.empty ? other : other.empty ? this : composeSets(this, other, true); }
      /**
      Given another change set starting in the same document, maps this
      change set over the other, producing a new change set that can be
      applied to the document produced by applying `other`. When
      `before` is `true`, order changes as if `this` comes before
      `other`, otherwise (the default) treat `other` as coming first.
      
      Given two changes `A` and `B`, `A.compose(B.map(A))` and
      `B.compose(A.map(B, true))` will produce the same document. This
      provides a basic form of [operational
      transformation](https://en.wikipedia.org/wiki/Operational_transformation),
      and can be used for collaborative editing.
      */
      map(other, before = false) { return other.empty ? this : mapSet(this, other, before, true); }
      /**
      Iterate over the changed ranges in the document, calling `f` for
      each, with the range in the original document (`fromA`-`toA`)
      and the range that replaces it in the new document
      (`fromB`-`toB`).
      
      When `individual` is true, adjacent changes are reported
      separately.
      */
      iterChanges(f, individual = false) {
          iterChanges(this, f, individual);
      }
      /**
      Get a [change description](https://codemirror.net/6/docs/ref/#state.ChangeDesc) for this change
      set.
      */
      get desc() { return ChangeDesc.create(this.sections); }
      /**
      @internal
      */
      filter(ranges) {
          let resultSections = [], resultInserted = [], filteredSections = [];
          let iter = new SectionIter(this);
          done: for (let i = 0, pos = 0;;) {
              let next = i == ranges.length ? 1e9 : ranges[i++];
              while (pos < next || pos == next && iter.len == 0) {
                  if (iter.done)
                      break done;
                  let len = Math.min(iter.len, next - pos);
                  addSection(filteredSections, len, -1);
                  let ins = iter.ins == -1 ? -1 : iter.off == 0 ? iter.ins : 0;
                  addSection(resultSections, len, ins);
                  if (ins > 0)
                      addInsert(resultInserted, resultSections, iter.text);
                  iter.forward(len);
                  pos += len;
              }
              let end = ranges[i++];
              while (pos < end) {
                  if (iter.done)
                      break done;
                  let len = Math.min(iter.len, end - pos);
                  addSection(resultSections, len, -1);
                  addSection(filteredSections, len, iter.ins == -1 ? -1 : iter.off == 0 ? iter.ins : 0);
                  iter.forward(len);
                  pos += len;
              }
          }
          return { changes: new ChangeSet(resultSections, resultInserted),
              filtered: ChangeDesc.create(filteredSections) };
      }
      /**
      Serialize this change set to a JSON-representable value.
      */
      toJSON() {
          let parts = [];
          for (let i = 0; i < this.sections.length; i += 2) {
              let len = this.sections[i], ins = this.sections[i + 1];
              if (ins < 0)
                  parts.push(len);
              else if (ins == 0)
                  parts.push([len]);
              else
                  parts.push([len].concat(this.inserted[i >> 1].toJSON()));
          }
          return parts;
      }
      /**
      Create a change set for the given changes, for a document of the
      given length, using `lineSep` as line separator.
      */
      static of(changes, length, lineSep) {
          let sections = [], inserted = [], pos = 0;
          let total = null;
          function flush(force = false) {
              if (!force && !sections.length)
                  return;
              if (pos < length)
                  addSection(sections, length - pos, -1);
              let set = new ChangeSet(sections, inserted);
              total = total ? total.compose(set.map(total)) : set;
              sections = [];
              inserted = [];
              pos = 0;
          }
          function process(spec) {
              if (Array.isArray(spec)) {
                  for (let sub of spec)
                      process(sub);
              }
              else if (spec instanceof ChangeSet) {
                  if (spec.length != length)
                      throw new RangeError(`Mismatched change set length (got ${spec.length}, expected ${length})`);
                  flush();
                  total = total ? total.compose(spec.map(total)) : spec;
              }
              else {
                  let { from, to = from, insert } = spec;
                  if (from > to || from < 0 || to > length)
                      throw new RangeError(`Invalid change range ${from} to ${to} (in doc of length ${length})`);
                  let insText = !insert ? Text$1.empty : typeof insert == "string" ? Text$1.of(insert.split(lineSep || DefaultSplit)) : insert;
                  let insLen = insText.length;
                  if (from == to && insLen == 0)
                      return;
                  if (from < pos)
                      flush();
                  if (from > pos)
                      addSection(sections, from - pos, -1);
                  addSection(sections, to - from, insLen);
                  addInsert(inserted, sections, insText);
                  pos = to;
              }
          }
          process(changes);
          flush(!total);
          return total;
      }
      /**
      Create an empty changeset of the given length.
      */
      static empty(length) {
          return new ChangeSet(length ? [length, -1] : [], []);
      }
      /**
      Create a changeset from its JSON representation (as produced by
      [`toJSON`](https://codemirror.net/6/docs/ref/#state.ChangeSet.toJSON).
      */
      static fromJSON(json) {
          if (!Array.isArray(json))
              throw new RangeError("Invalid JSON representation of ChangeSet");
          let sections = [], inserted = [];
          for (let i = 0; i < json.length; i++) {
              let part = json[i];
              if (typeof part == "number") {
                  sections.push(part, -1);
              }
              else if (!Array.isArray(part) || typeof part[0] != "number" || part.some((e, i) => i && typeof e != "string")) {
                  throw new RangeError("Invalid JSON representation of ChangeSet");
              }
              else if (part.length == 1) {
                  sections.push(part[0], 0);
              }
              else {
                  while (inserted.length < i)
                      inserted.push(Text$1.empty);
                  inserted[i] = Text$1.of(part.slice(1));
                  sections.push(part[0], inserted[i].length);
              }
          }
          return new ChangeSet(sections, inserted);
      }
      /**
      @internal
      */
      static createSet(sections, inserted) {
          return new ChangeSet(sections, inserted);
      }
  }
  function addSection(sections, len, ins, forceJoin = false) {
      if (len == 0 && ins <= 0)
          return;
      let last = sections.length - 2;
      if (last >= 0 && ins <= 0 && ins == sections[last + 1])
          sections[last] += len;
      else if (len == 0 && sections[last] == 0)
          sections[last + 1] += ins;
      else if (forceJoin) {
          sections[last] += len;
          sections[last + 1] += ins;
      }
      else
          sections.push(len, ins);
  }
  function addInsert(values, sections, value) {
      if (value.length == 0)
          return;
      let index = (sections.length - 2) >> 1;
      if (index < values.length) {
          values[values.length - 1] = values[values.length - 1].append(value);
      }
      else {
          while (values.length < index)
              values.push(Text$1.empty);
          values.push(value);
      }
  }
  function iterChanges(desc, f, individual) {
      let inserted = desc.inserted;
      for (let posA = 0, posB = 0, i = 0; i < desc.sections.length;) {
          let len = desc.sections[i++], ins = desc.sections[i++];
          if (ins < 0) {
              posA += len;
              posB += len;
          }
          else {
              let endA = posA, endB = posB, text = Text$1.empty;
              for (;;) {
                  endA += len;
                  endB += ins;
                  if (ins && inserted)
                      text = text.append(inserted[(i - 2) >> 1]);
                  if (individual || i == desc.sections.length || desc.sections[i + 1] < 0)
                      break;
                  len = desc.sections[i++];
                  ins = desc.sections[i++];
              }
              f(posA, endA, posB, endB, text);
              posA = endA;
              posB = endB;
          }
      }
  }
  function mapSet(setA, setB, before, mkSet = false) {
      // Produce a copy of setA that applies to the document after setB
      // has been applied (assuming both start at the same document).
      let sections = [], insert = mkSet ? [] : null;
      let a = new SectionIter(setA), b = new SectionIter(setB);
      // Iterate over both sets in parallel. inserted tracks, for changes
      // in A that have to be processed piece-by-piece, whether their
      // content has been inserted already, and refers to the section
      // index.
      for (let inserted = -1;;) {
          if (a.ins == -1 && b.ins == -1) {
              // Move across ranges skipped by both sets.
              let len = Math.min(a.len, b.len);
              addSection(sections, len, -1);
              a.forward(len);
              b.forward(len);
          }
          else if (b.ins >= 0 && (a.ins < 0 || inserted == a.i || a.off == 0 && (b.len < a.len || b.len == a.len && !before))) {
              // If there's a change in B that comes before the next change in
              // A (ordered by start pos, then len, then before flag), skip
              // that (and process any changes in A it covers).
              let len = b.len;
              addSection(sections, b.ins, -1);
              while (len) {
                  let piece = Math.min(a.len, len);
                  if (a.ins >= 0 && inserted < a.i && a.len <= piece) {
                      addSection(sections, 0, a.ins);
                      if (insert)
                          addInsert(insert, sections, a.text);
                      inserted = a.i;
                  }
                  a.forward(piece);
                  len -= piece;
              }
              b.next();
          }
          else if (a.ins >= 0) {
              // Process the part of a change in A up to the start of the next
              // non-deletion change in B (if overlapping).
              let len = 0, left = a.len;
              while (left) {
                  if (b.ins == -1) {
                      let piece = Math.min(left, b.len);
                      len += piece;
                      left -= piece;
                      b.forward(piece);
                  }
                  else if (b.ins == 0 && b.len < left) {
                      left -= b.len;
                      b.next();
                  }
                  else {
                      break;
                  }
              }
              addSection(sections, len, inserted < a.i ? a.ins : 0);
              if (insert && inserted < a.i)
                  addInsert(insert, sections, a.text);
              inserted = a.i;
              a.forward(a.len - left);
          }
          else if (a.done && b.done) {
              return insert ? ChangeSet.createSet(sections, insert) : ChangeDesc.create(sections);
          }
          else {
              throw new Error("Mismatched change set lengths");
          }
      }
  }
  function composeSets(setA, setB, mkSet = false) {
      let sections = [];
      let insert = mkSet ? [] : null;
      let a = new SectionIter(setA), b = new SectionIter(setB);
      for (let open = false;;) {
          if (a.done && b.done) {
              return insert ? ChangeSet.createSet(sections, insert) : ChangeDesc.create(sections);
          }
          else if (a.ins == 0) { // Deletion in A
              addSection(sections, a.len, 0, open);
              a.next();
          }
          else if (b.len == 0 && !b.done) { // Insertion in B
              addSection(sections, 0, b.ins, open);
              if (insert)
                  addInsert(insert, sections, b.text);
              b.next();
          }
          else if (a.done || b.done) {
              throw new Error("Mismatched change set lengths");
          }
          else {
              let len = Math.min(a.len2, b.len), sectionLen = sections.length;
              if (a.ins == -1) {
                  let insB = b.ins == -1 ? -1 : b.off ? 0 : b.ins;
                  addSection(sections, len, insB, open);
                  if (insert && insB)
                      addInsert(insert, sections, b.text);
              }
              else if (b.ins == -1) {
                  addSection(sections, a.off ? 0 : a.len, len, open);
                  if (insert)
                      addInsert(insert, sections, a.textBit(len));
              }
              else {
                  addSection(sections, a.off ? 0 : a.len, b.off ? 0 : b.ins, open);
                  if (insert && !b.off)
                      addInsert(insert, sections, b.text);
              }
              open = (a.ins > len || b.ins >= 0 && b.len > len) && (open || sections.length > sectionLen);
              a.forward2(len);
              b.forward(len);
          }
      }
  }
  class SectionIter {
      constructor(set) {
          this.set = set;
          this.i = 0;
          this.next();
      }
      next() {
          let { sections } = this.set;
          if (this.i < sections.length) {
              this.len = sections[this.i++];
              this.ins = sections[this.i++];
          }
          else {
              this.len = 0;
              this.ins = -2;
          }
          this.off = 0;
      }
      get done() { return this.ins == -2; }
      get len2() { return this.ins < 0 ? this.len : this.ins; }
      get text() {
          let { inserted } = this.set, index = (this.i - 2) >> 1;
          return index >= inserted.length ? Text$1.empty : inserted[index];
      }
      textBit(len) {
          let { inserted } = this.set, index = (this.i - 2) >> 1;
          return index >= inserted.length && !len ? Text$1.empty
              : inserted[index].slice(this.off, len == null ? undefined : this.off + len);
      }
      forward(len) {
          if (len == this.len)
              this.next();
          else {
              this.len -= len;
              this.off += len;
          }
      }
      forward2(len) {
          if (this.ins == -1)
              this.forward(len);
          else if (len == this.ins)
              this.next();
          else {
              this.ins -= len;
              this.off += len;
          }
      }
  }

  /**
  A single selection range. When
  [`allowMultipleSelections`](https://codemirror.net/6/docs/ref/#state.EditorState^allowMultipleSelections)
  is enabled, a [selection](https://codemirror.net/6/docs/ref/#state.EditorSelection) may hold
  multiple ranges. By default, selections hold exactly one range.
  */
  class SelectionRange {
      constructor(
      /**
      The lower boundary of the range.
      */
      from, 
      /**
      The upper boundary of the range.
      */
      to, flags) {
          this.from = from;
          this.to = to;
          this.flags = flags;
      }
      /**
      The anchor of the range—the side that doesn't move when you
      extend it.
      */
      get anchor() { return this.flags & 16 /* RangeFlag.Inverted */ ? this.to : this.from; }
      /**
      The head of the range, which is moved when the range is
      [extended](https://codemirror.net/6/docs/ref/#state.SelectionRange.extend).
      */
      get head() { return this.flags & 16 /* RangeFlag.Inverted */ ? this.from : this.to; }
      /**
      True when `anchor` and `head` are at the same position.
      */
      get empty() { return this.from == this.to; }
      /**
      If this is a cursor that is explicitly associated with the
      character on one of its sides, this returns the side. -1 means
      the character before its position, 1 the character after, and 0
      means no association.
      */
      get assoc() { return this.flags & 4 /* RangeFlag.AssocBefore */ ? -1 : this.flags & 8 /* RangeFlag.AssocAfter */ ? 1 : 0; }
      /**
      The bidirectional text level associated with this cursor, if
      any.
      */
      get bidiLevel() {
          let level = this.flags & 3 /* RangeFlag.BidiLevelMask */;
          return level == 3 ? null : level;
      }
      /**
      The goal column (stored vertical offset) associated with a
      cursor. This is used to preserve the vertical position when
      [moving](https://codemirror.net/6/docs/ref/#view.EditorView.moveVertically) across
      lines of different length.
      */
      get goalColumn() {
          let value = this.flags >> 5 /* RangeFlag.GoalColumnOffset */;
          return value == 33554431 /* RangeFlag.NoGoalColumn */ ? undefined : value;
      }
      /**
      Map this range through a change, producing a valid range in the
      updated document.
      */
      map(change, assoc = -1) {
          let from, to;
          if (this.empty) {
              from = to = change.mapPos(this.from, assoc);
          }
          else {
              from = change.mapPos(this.from, 1);
              to = change.mapPos(this.to, -1);
          }
          return from == this.from && to == this.to ? this : new SelectionRange(from, to, this.flags);
      }
      /**
      Extend this range to cover at least `from` to `to`.
      */
      extend(from, to = from) {
          if (from <= this.anchor && to >= this.anchor)
              return EditorSelection.range(from, to);
          let head = Math.abs(from - this.anchor) > Math.abs(to - this.anchor) ? from : to;
          return EditorSelection.range(this.anchor, head);
      }
      /**
      Compare this range to another range.
      */
      eq(other) {
          return this.anchor == other.anchor && this.head == other.head;
      }
      /**
      Return a JSON-serializable object representing the range.
      */
      toJSON() { return { anchor: this.anchor, head: this.head }; }
      /**
      Convert a JSON representation of a range to a `SelectionRange`
      instance.
      */
      static fromJSON(json) {
          if (!json || typeof json.anchor != "number" || typeof json.head != "number")
              throw new RangeError("Invalid JSON representation for SelectionRange");
          return EditorSelection.range(json.anchor, json.head);
      }
      /**
      @internal
      */
      static create(from, to, flags) {
          return new SelectionRange(from, to, flags);
      }
  }
  /**
  An editor selection holds one or more selection ranges.
  */
  class EditorSelection {
      constructor(
      /**
      The ranges in the selection, sorted by position. Ranges cannot
      overlap (but they may touch, if they aren't empty).
      */
      ranges, 
      /**
      The index of the _main_ range in the selection (which is
      usually the range that was added last).
      */
      mainIndex) {
          this.ranges = ranges;
          this.mainIndex = mainIndex;
      }
      /**
      Map a selection through a change. Used to adjust the selection
      position for changes.
      */
      map(change, assoc = -1) {
          if (change.empty)
              return this;
          return EditorSelection.create(this.ranges.map(r => r.map(change, assoc)), this.mainIndex);
      }
      /**
      Compare this selection to another selection.
      */
      eq(other) {
          if (this.ranges.length != other.ranges.length ||
              this.mainIndex != other.mainIndex)
              return false;
          for (let i = 0; i < this.ranges.length; i++)
              if (!this.ranges[i].eq(other.ranges[i]))
                  return false;
          return true;
      }
      /**
      Get the primary selection range. Usually, you should make sure
      your code applies to _all_ ranges, by using methods like
      [`changeByRange`](https://codemirror.net/6/docs/ref/#state.EditorState.changeByRange).
      */
      get main() { return this.ranges[this.mainIndex]; }
      /**
      Make sure the selection only has one range. Returns a selection
      holding only the main range from this selection.
      */
      asSingle() {
          return this.ranges.length == 1 ? this : new EditorSelection([this.main], 0);
      }
      /**
      Extend this selection with an extra range.
      */
      addRange(range, main = true) {
          return EditorSelection.create([range].concat(this.ranges), main ? 0 : this.mainIndex + 1);
      }
      /**
      Replace a given range with another range, and then normalize the
      selection to merge and sort ranges if necessary.
      */
      replaceRange(range, which = this.mainIndex) {
          let ranges = this.ranges.slice();
          ranges[which] = range;
          return EditorSelection.create(ranges, this.mainIndex);
      }
      /**
      Convert this selection to an object that can be serialized to
      JSON.
      */
      toJSON() {
          return { ranges: this.ranges.map(r => r.toJSON()), main: this.mainIndex };
      }
      /**
      Create a selection from a JSON representation.
      */
      static fromJSON(json) {
          if (!json || !Array.isArray(json.ranges) || typeof json.main != "number" || json.main >= json.ranges.length)
              throw new RangeError("Invalid JSON representation for EditorSelection");
          return new EditorSelection(json.ranges.map((r) => SelectionRange.fromJSON(r)), json.main);
      }
      /**
      Create a selection holding a single range.
      */
      static single(anchor, head = anchor) {
          return new EditorSelection([EditorSelection.range(anchor, head)], 0);
      }
      /**
      Sort and merge the given set of ranges, creating a valid
      selection.
      */
      static create(ranges, mainIndex = 0) {
          if (ranges.length == 0)
              throw new RangeError("A selection needs at least one range");
          for (let pos = 0, i = 0; i < ranges.length; i++) {
              let range = ranges[i];
              if (range.empty ? range.from <= pos : range.from < pos)
                  return EditorSelection.normalized(ranges.slice(), mainIndex);
              pos = range.to;
          }
          return new EditorSelection(ranges, mainIndex);
      }
      /**
      Create a cursor selection range at the given position. You can
      safely ignore the optional arguments in most situations.
      */
      static cursor(pos, assoc = 0, bidiLevel, goalColumn) {
          return SelectionRange.create(pos, pos, (assoc == 0 ? 0 : assoc < 0 ? 4 /* RangeFlag.AssocBefore */ : 8 /* RangeFlag.AssocAfter */) |
              (bidiLevel == null ? 3 : Math.min(2, bidiLevel)) |
              ((goalColumn !== null && goalColumn !== void 0 ? goalColumn : 33554431 /* RangeFlag.NoGoalColumn */) << 5 /* RangeFlag.GoalColumnOffset */));
      }
      /**
      Create a selection range.
      */
      static range(anchor, head, goalColumn) {
          let goal = (goalColumn !== null && goalColumn !== void 0 ? goalColumn : 33554431 /* RangeFlag.NoGoalColumn */) << 5 /* RangeFlag.GoalColumnOffset */;
          return head < anchor ? SelectionRange.create(head, anchor, 16 /* RangeFlag.Inverted */ | goal | 8 /* RangeFlag.AssocAfter */)
              : SelectionRange.create(anchor, head, goal | (head > anchor ? 4 /* RangeFlag.AssocBefore */ : 0));
      }
      /**
      @internal
      */
      static normalized(ranges, mainIndex = 0) {
          let main = ranges[mainIndex];
          ranges.sort((a, b) => a.from - b.from);
          mainIndex = ranges.indexOf(main);
          for (let i = 1; i < ranges.length; i++) {
              let range = ranges[i], prev = ranges[i - 1];
              if (range.empty ? range.from <= prev.to : range.from < prev.to) {
                  let from = prev.from, to = Math.max(range.to, prev.to);
                  if (i <= mainIndex)
                      mainIndex--;
                  ranges.splice(--i, 2, range.anchor > range.head ? EditorSelection.range(to, from) : EditorSelection.range(from, to));
              }
          }
          return new EditorSelection(ranges, mainIndex);
      }
  }
  function checkSelection(selection, docLength) {
      for (let range of selection.ranges)
          if (range.to > docLength)
              throw new RangeError("Selection points outside of document");
  }

  let nextID = 0;
  /**
  A facet is a labeled value that is associated with an editor
  state. It takes inputs from any number of extensions, and combines
  those into a single output value.

  Examples of uses of facets are the [tab
  size](https://codemirror.net/6/docs/ref/#state.EditorState^tabSize), [editor
  attributes](https://codemirror.net/6/docs/ref/#view.EditorView^editorAttributes), and [update
  listeners](https://codemirror.net/6/docs/ref/#view.EditorView^updateListener).
  */
  class Facet {
      constructor(
      /**
      @internal
      */
      combine, 
      /**
      @internal
      */
      compareInput, 
      /**
      @internal
      */
      compare, isStatic, enables) {
          this.combine = combine;
          this.compareInput = compareInput;
          this.compare = compare;
          this.isStatic = isStatic;
          /**
          @internal
          */
          this.id = nextID++;
          this.default = combine([]);
          this.extensions = typeof enables == "function" ? enables(this) : enables;
      }
      /**
      Define a new facet.
      */
      static define(config = {}) {
          return new Facet(config.combine || ((a) => a), config.compareInput || ((a, b) => a === b), config.compare || (!config.combine ? sameArray : (a, b) => a === b), !!config.static, config.enables);
      }
      /**
      Returns an extension that adds the given value to this facet.
      */
      of(value) {
          return new FacetProvider([], this, 0 /* Provider.Static */, value);
      }
      /**
      Create an extension that computes a value for the facet from a
      state. You must take care to declare the parts of the state that
      this value depends on, since your function is only called again
      for a new state when one of those parts changed.
      
      In cases where your value depends only on a single field, you'll
      want to use the [`from`](https://codemirror.net/6/docs/ref/#state.Facet.from) method instead.
      */
      compute(deps, get) {
          if (this.isStatic)
              throw new Error("Can't compute a static facet");
          return new FacetProvider(deps, this, 1 /* Provider.Single */, get);
      }
      /**
      Create an extension that computes zero or more values for this
      facet from a state.
      */
      computeN(deps, get) {
          if (this.isStatic)
              throw new Error("Can't compute a static facet");
          return new FacetProvider(deps, this, 2 /* Provider.Multi */, get);
      }
      from(field, get) {
          if (!get)
              get = x => x;
          return this.compute([field], state => get(state.field(field)));
      }
  }
  function sameArray(a, b) {
      return a == b || a.length == b.length && a.every((e, i) => e === b[i]);
  }
  class FacetProvider {
      constructor(dependencies, facet, type, value) {
          this.dependencies = dependencies;
          this.facet = facet;
          this.type = type;
          this.value = value;
          this.id = nextID++;
      }
      dynamicSlot(addresses) {
          var _a;
          let getter = this.value;
          let compare = this.facet.compareInput;
          let id = this.id, idx = addresses[id] >> 1, multi = this.type == 2 /* Provider.Multi */;
          let depDoc = false, depSel = false, depAddrs = [];
          for (let dep of this.dependencies) {
              if (dep == "doc")
                  depDoc = true;
              else if (dep == "selection")
                  depSel = true;
              else if ((((_a = addresses[dep.id]) !== null && _a !== void 0 ? _a : 1) & 1) == 0)
                  depAddrs.push(addresses[dep.id]);
          }
          return {
              create(state) {
                  state.values[idx] = getter(state);
                  return 1 /* SlotStatus.Changed */;
              },
              update(state, tr) {
                  if ((depDoc && tr.docChanged) || (depSel && (tr.docChanged || tr.selection)) || ensureAll(state, depAddrs)) {
                      let newVal = getter(state);
                      if (multi ? !compareArray(newVal, state.values[idx], compare) : !compare(newVal, state.values[idx])) {
                          state.values[idx] = newVal;
                          return 1 /* SlotStatus.Changed */;
                      }
                  }
                  return 0;
              },
              reconfigure: (state, oldState) => {
                  let newVal, oldAddr = oldState.config.address[id];
                  if (oldAddr != null) {
                      let oldVal = getAddr(oldState, oldAddr);
                      if (this.dependencies.every(dep => {
                          return dep instanceof Facet ? oldState.facet(dep) === state.facet(dep) :
                              dep instanceof StateField ? oldState.field(dep, false) == state.field(dep, false) : true;
                      }) || (multi ? compareArray(newVal = getter(state), oldVal, compare) : compare(newVal = getter(state), oldVal))) {
                          state.values[idx] = oldVal;
                          return 0;
                      }
                  }
                  else {
                      newVal = getter(state);
                  }
                  state.values[idx] = newVal;
                  return 1 /* SlotStatus.Changed */;
              }
          };
      }
  }
  function compareArray(a, b, compare) {
      if (a.length != b.length)
          return false;
      for (let i = 0; i < a.length; i++)
          if (!compare(a[i], b[i]))
              return false;
      return true;
  }
  function ensureAll(state, addrs) {
      let changed = false;
      for (let addr of addrs)
          if (ensureAddr(state, addr) & 1 /* SlotStatus.Changed */)
              changed = true;
      return changed;
  }
  function dynamicFacetSlot(addresses, facet, providers) {
      let providerAddrs = providers.map(p => addresses[p.id]);
      let providerTypes = providers.map(p => p.type);
      let dynamic = providerAddrs.filter(p => !(p & 1));
      let idx = addresses[facet.id] >> 1;
      function get(state) {
          let values = [];
          for (let i = 0; i < providerAddrs.length; i++) {
              let value = getAddr(state, providerAddrs[i]);
              if (providerTypes[i] == 2 /* Provider.Multi */)
                  for (let val of value)
                      values.push(val);
              else
                  values.push(value);
          }
          return facet.combine(values);
      }
      return {
          create(state) {
              for (let addr of providerAddrs)
                  ensureAddr(state, addr);
              state.values[idx] = get(state);
              return 1 /* SlotStatus.Changed */;
          },
          update(state, tr) {
              if (!ensureAll(state, dynamic))
                  return 0;
              let value = get(state);
              if (facet.compare(value, state.values[idx]))
                  return 0;
              state.values[idx] = value;
              return 1 /* SlotStatus.Changed */;
          },
          reconfigure(state, oldState) {
              let depChanged = ensureAll(state, providerAddrs);
              let oldProviders = oldState.config.facets[facet.id], oldValue = oldState.facet(facet);
              if (oldProviders && !depChanged && sameArray(providers, oldProviders)) {
                  state.values[idx] = oldValue;
                  return 0;
              }
              let value = get(state);
              if (facet.compare(value, oldValue)) {
                  state.values[idx] = oldValue;
                  return 0;
              }
              state.values[idx] = value;
              return 1 /* SlotStatus.Changed */;
          }
      };
  }
  const initField = /*@__PURE__*/Facet.define({ static: true });
  /**
  Fields can store additional information in an editor state, and
  keep it in sync with the rest of the state.
  */
  class StateField {
      constructor(
      /**
      @internal
      */
      id, createF, updateF, compareF, 
      /**
      @internal
      */
      spec) {
          this.id = id;
          this.createF = createF;
          this.updateF = updateF;
          this.compareF = compareF;
          this.spec = spec;
          /**
          @internal
          */
          this.provides = undefined;
      }
      /**
      Define a state field.
      */
      static define(config) {
          let field = new StateField(nextID++, config.create, config.update, config.compare || ((a, b) => a === b), config);
          if (config.provide)
              field.provides = config.provide(field);
          return field;
      }
      create(state) {
          let init = state.facet(initField).find(i => i.field == this);
          return ((init === null || init === void 0 ? void 0 : init.create) || this.createF)(state);
      }
      /**
      @internal
      */
      slot(addresses) {
          let idx = addresses[this.id] >> 1;
          return {
              create: (state) => {
                  state.values[idx] = this.create(state);
                  return 1 /* SlotStatus.Changed */;
              },
              update: (state, tr) => {
                  let oldVal = state.values[idx];
                  let value = this.updateF(oldVal, tr);
                  if (this.compareF(oldVal, value))
                      return 0;
                  state.values[idx] = value;
                  return 1 /* SlotStatus.Changed */;
              },
              reconfigure: (state, oldState) => {
                  if (oldState.config.address[this.id] != null) {
                      state.values[idx] = oldState.field(this);
                      return 0;
                  }
                  state.values[idx] = this.create(state);
                  return 1 /* SlotStatus.Changed */;
              }
          };
      }
      /**
      Returns an extension that enables this field and overrides the
      way it is initialized. Can be useful when you need to provide a
      non-default starting value for the field.
      */
      init(create) {
          return [this, initField.of({ field: this, create })];
      }
      /**
      State field instances can be used as
      [`Extension`](https://codemirror.net/6/docs/ref/#state.Extension) values to enable the field in a
      given state.
      */
      get extension() { return this; }
  }
  const Prec_ = { lowest: 4, low: 3, default: 2, high: 1, highest: 0 };
  function prec(value) {
      return (ext) => new PrecExtension(ext, value);
  }
  /**
  By default extensions are registered in the order they are found
  in the flattened form of nested array that was provided.
  Individual extension values can be assigned a precedence to
  override this. Extensions that do not have a precedence set get
  the precedence of the nearest parent with a precedence, or
  [`default`](https://codemirror.net/6/docs/ref/#state.Prec.default) if there is no such parent. The
  final ordering of extensions is determined by first sorting by
  precedence and then by order within each precedence.
  */
  const Prec = {
      /**
      The highest precedence level, for extensions that should end up
      near the start of the precedence ordering.
      */
      highest: /*@__PURE__*/prec(Prec_.highest),
      /**
      A higher-than-default precedence, for extensions that should
      come before those with default precedence.
      */
      high: /*@__PURE__*/prec(Prec_.high),
      /**
      The default precedence, which is also used for extensions
      without an explicit precedence.
      */
      default: /*@__PURE__*/prec(Prec_.default),
      /**
      A lower-than-default precedence.
      */
      low: /*@__PURE__*/prec(Prec_.low),
      /**
      The lowest precedence level. Meant for things that should end up
      near the end of the extension order.
      */
      lowest: /*@__PURE__*/prec(Prec_.lowest)
  };
  class PrecExtension {
      constructor(inner, prec) {
          this.inner = inner;
          this.prec = prec;
      }
  }
  /**
  Extension compartments can be used to make a configuration
  dynamic. By [wrapping](https://codemirror.net/6/docs/ref/#state.Compartment.of) part of your
  configuration in a compartment, you can later
  [replace](https://codemirror.net/6/docs/ref/#state.Compartment.reconfigure) that part through a
  transaction.
  */
  class Compartment {
      /**
      Create an instance of this compartment to add to your [state
      configuration](https://codemirror.net/6/docs/ref/#state.EditorStateConfig.extensions).
      */
      of(ext) { return new CompartmentInstance(this, ext); }
      /**
      Create an [effect](https://codemirror.net/6/docs/ref/#state.TransactionSpec.effects) that
      reconfigures this compartment.
      */
      reconfigure(content) {
          return Compartment.reconfigure.of({ compartment: this, extension: content });
      }
      /**
      Get the current content of the compartment in the state, or
      `undefined` if it isn't present.
      */
      get(state) {
          return state.config.compartments.get(this);
      }
  }
  class CompartmentInstance {
      constructor(compartment, inner) {
          this.compartment = compartment;
          this.inner = inner;
      }
  }
  class Configuration {
      constructor(base, compartments, dynamicSlots, address, staticValues, facets) {
          this.base = base;
          this.compartments = compartments;
          this.dynamicSlots = dynamicSlots;
          this.address = address;
          this.staticValues = staticValues;
          this.facets = facets;
          this.statusTemplate = [];
          while (this.statusTemplate.length < dynamicSlots.length)
              this.statusTemplate.push(0 /* SlotStatus.Unresolved */);
      }
      staticFacet(facet) {
          let addr = this.address[facet.id];
          return addr == null ? facet.default : this.staticValues[addr >> 1];
      }
      static resolve(base, compartments, oldState) {
          let fields = [];
          let facets = Object.create(null);
          let newCompartments = new Map();
          for (let ext of flatten(base, compartments, newCompartments)) {
              if (ext instanceof StateField)
                  fields.push(ext);
              else
                  (facets[ext.facet.id] || (facets[ext.facet.id] = [])).push(ext);
          }
          let address = Object.create(null);
          let staticValues = [];
          let dynamicSlots = [];
          for (let field of fields) {
              address[field.id] = dynamicSlots.length << 1;
              dynamicSlots.push(a => field.slot(a));
          }
          let oldFacets = oldState === null || oldState === void 0 ? void 0 : oldState.config.facets;
          for (let id in facets) {
              let providers = facets[id], facet = providers[0].facet;
              let oldProviders = oldFacets && oldFacets[id] || [];
              if (providers.every(p => p.type == 0 /* Provider.Static */)) {
                  address[facet.id] = (staticValues.length << 1) | 1;
                  if (sameArray(oldProviders, providers)) {
                      staticValues.push(oldState.facet(facet));
                  }
                  else {
                      let value = facet.combine(providers.map(p => p.value));
                      staticValues.push(oldState && facet.compare(value, oldState.facet(facet)) ? oldState.facet(facet) : value);
                  }
              }
              else {
                  for (let p of providers) {
                      if (p.type == 0 /* Provider.Static */) {
                          address[p.id] = (staticValues.length << 1) | 1;
                          staticValues.push(p.value);
                      }
                      else {
                          address[p.id] = dynamicSlots.length << 1;
                          dynamicSlots.push(a => p.dynamicSlot(a));
                      }
                  }
                  address[facet.id] = dynamicSlots.length << 1;
                  dynamicSlots.push(a => dynamicFacetSlot(a, facet, providers));
              }
          }
          let dynamic = dynamicSlots.map(f => f(address));
          return new Configuration(base, newCompartments, dynamic, address, staticValues, facets);
      }
  }
  function flatten(extension, compartments, newCompartments) {
      let result = [[], [], [], [], []];
      let seen = new Map();
      function inner(ext, prec) {
          let known = seen.get(ext);
          if (known != null) {
              if (known <= prec)
                  return;
              let found = result[known].indexOf(ext);
              if (found > -1)
                  result[known].splice(found, 1);
              if (ext instanceof CompartmentInstance)
                  newCompartments.delete(ext.compartment);
          }
          seen.set(ext, prec);
          if (Array.isArray(ext)) {
              for (let e of ext)
                  inner(e, prec);
          }
          else if (ext instanceof CompartmentInstance) {
              if (newCompartments.has(ext.compartment))
                  throw new RangeError(`Duplicate use of compartment in extensions`);
              let content = compartments.get(ext.compartment) || ext.inner;
              newCompartments.set(ext.compartment, content);
              inner(content, prec);
          }
          else if (ext instanceof PrecExtension) {
              inner(ext.inner, ext.prec);
          }
          else if (ext instanceof StateField) {
              result[prec].push(ext);
              if (ext.provides)
                  inner(ext.provides, prec);
          }
          else if (ext instanceof FacetProvider) {
              result[prec].push(ext);
              if (ext.facet.extensions)
                  inner(ext.facet.extensions, Prec_.default);
          }
          else {
              let content = ext.extension;
              if (!content)
                  throw new Error(`Unrecognized extension value in extension set (${ext}). This sometimes happens because multiple instances of @codemirror/state are loaded, breaking instanceof checks.`);
              inner(content, prec);
          }
      }
      inner(extension, Prec_.default);
      return result.reduce((a, b) => a.concat(b));
  }
  function ensureAddr(state, addr) {
      if (addr & 1)
          return 2 /* SlotStatus.Computed */;
      let idx = addr >> 1;
      let status = state.status[idx];
      if (status == 4 /* SlotStatus.Computing */)
          throw new Error("Cyclic dependency between fields and/or facets");
      if (status & 2 /* SlotStatus.Computed */)
          return status;
      state.status[idx] = 4 /* SlotStatus.Computing */;
      let changed = state.computeSlot(state, state.config.dynamicSlots[idx]);
      return state.status[idx] = 2 /* SlotStatus.Computed */ | changed;
  }
  function getAddr(state, addr) {
      return addr & 1 ? state.config.staticValues[addr >> 1] : state.values[addr >> 1];
  }

  const languageData = /*@__PURE__*/Facet.define();
  const allowMultipleSelections = /*@__PURE__*/Facet.define({
      combine: values => values.some(v => v),
      static: true
  });
  const lineSeparator = /*@__PURE__*/Facet.define({
      combine: values => values.length ? values[0] : undefined,
      static: true
  });
  const changeFilter = /*@__PURE__*/Facet.define();
  const transactionFilter = /*@__PURE__*/Facet.define();
  const transactionExtender = /*@__PURE__*/Facet.define();
  const readOnly = /*@__PURE__*/Facet.define({
      combine: values => values.length ? values[0] : false
  });

  /**
  Annotations are tagged values that are used to add metadata to
  transactions in an extensible way. They should be used to model
  things that effect the entire transaction (such as its [time
  stamp](https://codemirror.net/6/docs/ref/#state.Transaction^time) or information about its
  [origin](https://codemirror.net/6/docs/ref/#state.Transaction^userEvent)). For effects that happen
  _alongside_ the other changes made by the transaction, [state
  effects](https://codemirror.net/6/docs/ref/#state.StateEffect) are more appropriate.
  */
  class Annotation {
      /**
      @internal
      */
      constructor(
      /**
      The annotation type.
      */
      type, 
      /**
      The value of this annotation.
      */
      value) {
          this.type = type;
          this.value = value;
      }
      /**
      Define a new type of annotation.
      */
      static define() { return new AnnotationType(); }
  }
  /**
  Marker that identifies a type of [annotation](https://codemirror.net/6/docs/ref/#state.Annotation).
  */
  class AnnotationType {
      /**
      Create an instance of this annotation.
      */
      of(value) { return new Annotation(this, value); }
  }
  /**
  Representation of a type of state effect. Defined with
  [`StateEffect.define`](https://codemirror.net/6/docs/ref/#state.StateEffect^define).
  */
  class StateEffectType {
      /**
      @internal
      */
      constructor(
      // The `any` types in these function types are there to work
      // around TypeScript issue #37631, where the type guard on
      // `StateEffect.is` mysteriously stops working when these properly
      // have type `Value`.
      /**
      @internal
      */
      map) {
          this.map = map;
      }
      /**
      Create a [state effect](https://codemirror.net/6/docs/ref/#state.StateEffect) instance of this
      type.
      */
      of(value) { return new StateEffect(this, value); }
  }
  /**
  State effects can be used to represent additional effects
  associated with a [transaction](https://codemirror.net/6/docs/ref/#state.Transaction.effects). They
  are often useful to model changes to custom [state
  fields](https://codemirror.net/6/docs/ref/#state.StateField), when those changes aren't implicit in
  document or selection changes.
  */
  class StateEffect {
      /**
      @internal
      */
      constructor(
      /**
      @internal
      */
      type, 
      /**
      The value of this effect.
      */
      value) {
          this.type = type;
          this.value = value;
      }
      /**
      Map this effect through a position mapping. Will return
      `undefined` when that ends up deleting the effect.
      */
      map(mapping) {
          let mapped = this.type.map(this.value, mapping);
          return mapped === undefined ? undefined : mapped == this.value ? this : new StateEffect(this.type, mapped);
      }
      /**
      Tells you whether this effect object is of a given
      [type](https://codemirror.net/6/docs/ref/#state.StateEffectType).
      */
      is(type) { return this.type == type; }
      /**
      Define a new effect type. The type parameter indicates the type
      of values that his effect holds.
      */
      static define(spec = {}) {
          return new StateEffectType(spec.map || (v => v));
      }
      /**
      Map an array of effects through a change set.
      */
      static mapEffects(effects, mapping) {
          if (!effects.length)
              return effects;
          let result = [];
          for (let effect of effects) {
              let mapped = effect.map(mapping);
              if (mapped)
                  result.push(mapped);
          }
          return result;
      }
  }
  /**
  This effect can be used to reconfigure the root extensions of
  the editor. Doing this will discard any extensions
  [appended](https://codemirror.net/6/docs/ref/#state.StateEffect^appendConfig), but does not reset
  the content of [reconfigured](https://codemirror.net/6/docs/ref/#state.Compartment.reconfigure)
  compartments.
  */
  StateEffect.reconfigure = /*@__PURE__*/StateEffect.define();
  /**
  Append extensions to the top-level configuration of the editor.
  */
  StateEffect.appendConfig = /*@__PURE__*/StateEffect.define();
  /**
  Changes to the editor state are grouped into transactions.
  Typically, a user action creates a single transaction, which may
  contain any number of document changes, may change the selection,
  or have other effects. Create a transaction by calling
  [`EditorState.update`](https://codemirror.net/6/docs/ref/#state.EditorState.update), or immediately
  dispatch one by calling
  [`EditorView.dispatch`](https://codemirror.net/6/docs/ref/#view.EditorView.dispatch).
  */
  class Transaction {
      constructor(
      /**
      The state from which the transaction starts.
      */
      startState, 
      /**
      The document changes made by this transaction.
      */
      changes, 
      /**
      The selection set by this transaction, or undefined if it
      doesn't explicitly set a selection.
      */
      selection, 
      /**
      The effects added to the transaction.
      */
      effects, 
      /**
      @internal
      */
      annotations, 
      /**
      Whether the selection should be scrolled into view after this
      transaction is dispatched.
      */
      scrollIntoView) {
          this.startState = startState;
          this.changes = changes;
          this.selection = selection;
          this.effects = effects;
          this.annotations = annotations;
          this.scrollIntoView = scrollIntoView;
          /**
          @internal
          */
          this._doc = null;
          /**
          @internal
          */
          this._state = null;
          if (selection)
              checkSelection(selection, changes.newLength);
          if (!annotations.some((a) => a.type == Transaction.time))
              this.annotations = annotations.concat(Transaction.time.of(Date.now()));
      }
      /**
      @internal
      */
      static create(startState, changes, selection, effects, annotations, scrollIntoView) {
          return new Transaction(startState, changes, selection, effects, annotations, scrollIntoView);
      }
      /**
      The new document produced by the transaction. Contrary to
      [`.state`](https://codemirror.net/6/docs/ref/#state.Transaction.state)`.doc`, accessing this won't
      force the entire new state to be computed right away, so it is
      recommended that [transaction
      filters](https://codemirror.net/6/docs/ref/#state.EditorState^transactionFilter) use this getter
      when they need to look at the new document.
      */
      get newDoc() {
          return this._doc || (this._doc = this.changes.apply(this.startState.doc));
      }
      /**
      The new selection produced by the transaction. If
      [`this.selection`](https://codemirror.net/6/docs/ref/#state.Transaction.selection) is undefined,
      this will [map](https://codemirror.net/6/docs/ref/#state.EditorSelection.map) the start state's
      current selection through the changes made by the transaction.
      */
      get newSelection() {
          return this.selection || this.startState.selection.map(this.changes);
      }
      /**
      The new state created by the transaction. Computed on demand
      (but retained for subsequent access), so it is recommended not to
      access it in [transaction
      filters](https://codemirror.net/6/docs/ref/#state.EditorState^transactionFilter) when possible.
      */
      get state() {
          if (!this._state)
              this.startState.applyTransaction(this);
          return this._state;
      }
      /**
      Get the value of the given annotation type, if any.
      */
      annotation(type) {
          for (let ann of this.annotations)
              if (ann.type == type)
                  return ann.value;
          return undefined;
      }
      /**
      Indicates whether the transaction changed the document.
      */
      get docChanged() { return !this.changes.empty; }
      /**
      Indicates whether this transaction reconfigures the state
      (through a [configuration compartment](https://codemirror.net/6/docs/ref/#state.Compartment) or
      with a top-level configuration
      [effect](https://codemirror.net/6/docs/ref/#state.StateEffect^reconfigure).
      */
      get reconfigured() { return this.startState.config != this.state.config; }
      /**
      Returns true if the transaction has a [user
      event](https://codemirror.net/6/docs/ref/#state.Transaction^userEvent) annotation that is equal to
      or more specific than `event`. For example, if the transaction
      has `"select.pointer"` as user event, `"select"` and
      `"select.pointer"` will match it.
      */
      isUserEvent(event) {
          let e = this.annotation(Transaction.userEvent);
          return !!(e && (e == event || e.length > event.length && e.slice(0, event.length) == event && e[event.length] == "."));
      }
  }
  /**
  Annotation used to store transaction timestamps. Automatically
  added to every transaction, holding `Date.now()`.
  */
  Transaction.time = /*@__PURE__*/Annotation.define();
  /**
  Annotation used to associate a transaction with a user interface
  event. Holds a string identifying the event, using a
  dot-separated format to support attaching more specific
  information. The events used by the core libraries are:

   - `"input"` when content is entered
     - `"input.type"` for typed input
       - `"input.type.compose"` for composition
     - `"input.paste"` for pasted input
     - `"input.drop"` when adding content with drag-and-drop
     - `"input.complete"` when autocompleting
   - `"delete"` when the user deletes content
     - `"delete.selection"` when deleting the selection
     - `"delete.forward"` when deleting forward from the selection
     - `"delete.backward"` when deleting backward from the selection
     - `"delete.cut"` when cutting to the clipboard
   - `"move"` when content is moved
     - `"move.drop"` when content is moved within the editor through drag-and-drop
   - `"select"` when explicitly changing the selection
     - `"select.pointer"` when selecting with a mouse or other pointing device
   - `"undo"` and `"redo"` for history actions

  Use [`isUserEvent`](https://codemirror.net/6/docs/ref/#state.Transaction.isUserEvent) to check
  whether the annotation matches a given event.
  */
  Transaction.userEvent = /*@__PURE__*/Annotation.define();
  /**
  Annotation indicating whether a transaction should be added to
  the undo history or not.
  */
  Transaction.addToHistory = /*@__PURE__*/Annotation.define();
  /**
  Annotation indicating (when present and true) that a transaction
  represents a change made by some other actor, not the user. This
  is used, for example, to tag other people's changes in
  collaborative editing.
  */
  Transaction.remote = /*@__PURE__*/Annotation.define();
  function joinRanges(a, b) {
      let result = [];
      for (let iA = 0, iB = 0;;) {
          let from, to;
          if (iA < a.length && (iB == b.length || b[iB] >= a[iA])) {
              from = a[iA++];
              to = a[iA++];
          }
          else if (iB < b.length) {
              from = b[iB++];
              to = b[iB++];
          }
          else
              return result;
          if (!result.length || result[result.length - 1] < from)
              result.push(from, to);
          else if (result[result.length - 1] < to)
              result[result.length - 1] = to;
      }
  }
  function mergeTransaction(a, b, sequential) {
      var _a;
      let mapForA, mapForB, changes;
      if (sequential) {
          mapForA = b.changes;
          mapForB = ChangeSet.empty(b.changes.length);
          changes = a.changes.compose(b.changes);
      }
      else {
          mapForA = b.changes.map(a.changes);
          mapForB = a.changes.mapDesc(b.changes, true);
          changes = a.changes.compose(mapForA);
      }
      return {
          changes,
          selection: b.selection ? b.selection.map(mapForB) : (_a = a.selection) === null || _a === void 0 ? void 0 : _a.map(mapForA),
          effects: StateEffect.mapEffects(a.effects, mapForA).concat(StateEffect.mapEffects(b.effects, mapForB)),
          annotations: a.annotations.length ? a.annotations.concat(b.annotations) : b.annotations,
          scrollIntoView: a.scrollIntoView || b.scrollIntoView
      };
  }
  function resolveTransactionInner(state, spec, docSize) {
      let sel = spec.selection, annotations = asArray(spec.annotations);
      if (spec.userEvent)
          annotations = annotations.concat(Transaction.userEvent.of(spec.userEvent));
      return {
          changes: spec.changes instanceof ChangeSet ? spec.changes
              : ChangeSet.of(spec.changes || [], docSize, state.facet(lineSeparator)),
          selection: sel && (sel instanceof EditorSelection ? sel : EditorSelection.single(sel.anchor, sel.head)),
          effects: asArray(spec.effects),
          annotations,
          scrollIntoView: !!spec.scrollIntoView
      };
  }
  function resolveTransaction(state, specs, filter) {
      let s = resolveTransactionInner(state, specs.length ? specs[0] : {}, state.doc.length);
      if (specs.length && specs[0].filter === false)
          filter = false;
      for (let i = 1; i < specs.length; i++) {
          if (specs[i].filter === false)
              filter = false;
          let seq = !!specs[i].sequential;
          s = mergeTransaction(s, resolveTransactionInner(state, specs[i], seq ? s.changes.newLength : state.doc.length), seq);
      }
      let tr = Transaction.create(state, s.changes, s.selection, s.effects, s.annotations, s.scrollIntoView);
      return extendTransaction(filter ? filterTransaction(tr) : tr);
  }
  // Finish a transaction by applying filters if necessary.
  function filterTransaction(tr) {
      let state = tr.startState;
      // Change filters
      let result = true;
      for (let filter of state.facet(changeFilter)) {
          let value = filter(tr);
          if (value === false) {
              result = false;
              break;
          }
          if (Array.isArray(value))
              result = result === true ? value : joinRanges(result, value);
      }
      if (result !== true) {
          let changes, back;
          if (result === false) {
              back = tr.changes.invertedDesc;
              changes = ChangeSet.empty(state.doc.length);
          }
          else {
              let filtered = tr.changes.filter(result);
              changes = filtered.changes;
              back = filtered.filtered.mapDesc(filtered.changes).invertedDesc;
          }
          tr = Transaction.create(state, changes, tr.selection && tr.selection.map(back), StateEffect.mapEffects(tr.effects, back), tr.annotations, tr.scrollIntoView);
      }
      // Transaction filters
      let filters = state.facet(transactionFilter);
      for (let i = filters.length - 1; i >= 0; i--) {
          let filtered = filters[i](tr);
          if (filtered instanceof Transaction)
              tr = filtered;
          else if (Array.isArray(filtered) && filtered.length == 1 && filtered[0] instanceof Transaction)
              tr = filtered[0];
          else
              tr = resolveTransaction(state, asArray(filtered), false);
      }
      return tr;
  }
  function extendTransaction(tr) {
      let state = tr.startState, extenders = state.facet(transactionExtender), spec = tr;
      for (let i = extenders.length - 1; i >= 0; i--) {
          let extension = extenders[i](tr);
          if (extension && Object.keys(extension).length)
              spec = mergeTransaction(spec, resolveTransactionInner(state, extension, tr.changes.newLength), true);
      }
      return spec == tr ? tr : Transaction.create(state, tr.changes, tr.selection, spec.effects, spec.annotations, spec.scrollIntoView);
  }
  const none$1 = [];
  function asArray(value) {
      return value == null ? none$1 : Array.isArray(value) ? value : [value];
  }

  /**
  The categories produced by a [character
  categorizer](https://codemirror.net/6/docs/ref/#state.EditorState.charCategorizer). These are used
  do things like selecting by word.
  */
  var CharCategory = /*@__PURE__*/(function (CharCategory) {
      /**
      Word characters.
      */
      CharCategory[CharCategory["Word"] = 0] = "Word";
      /**
      Whitespace.
      */
      CharCategory[CharCategory["Space"] = 1] = "Space";
      /**
      Anything else.
      */
      CharCategory[CharCategory["Other"] = 2] = "Other";
  return CharCategory})(CharCategory || (CharCategory = {}));
  const nonASCIISingleCaseWordChar = /[\u00df\u0587\u0590-\u05f4\u0600-\u06ff\u3040-\u309f\u30a0-\u30ff\u3400-\u4db5\u4e00-\u9fcc\uac00-\ud7af]/;
  let wordChar;
  try {
      wordChar = /*@__PURE__*/new RegExp("[\\p{Alphabetic}\\p{Number}_]", "u");
  }
  catch (_) { }
  function hasWordChar(str) {
      if (wordChar)
          return wordChar.test(str);
      for (let i = 0; i < str.length; i++) {
          let ch = str[i];
          if (/\w/.test(ch) || ch > "\x80" && (ch.toUpperCase() != ch.toLowerCase() || nonASCIISingleCaseWordChar.test(ch)))
              return true;
      }
      return false;
  }
  function makeCategorizer(wordChars) {
      return (char) => {
          if (!/\S/.test(char))
              return CharCategory.Space;
          if (hasWordChar(char))
              return CharCategory.Word;
          for (let i = 0; i < wordChars.length; i++)
              if (char.indexOf(wordChars[i]) > -1)
                  return CharCategory.Word;
          return CharCategory.Other;
      };
  }

  /**
  The editor state class is a persistent (immutable) data structure.
  To update a state, you [create](https://codemirror.net/6/docs/ref/#state.EditorState.update) a
  [transaction](https://codemirror.net/6/docs/ref/#state.Transaction), which produces a _new_ state
  instance, without modifying the original object.

  As such, _never_ mutate properties of a state directly. That'll
  just break things.
  */
  class EditorState {
      constructor(
      /**
      @internal
      */
      config, 
      /**
      The current document.
      */
      doc, 
      /**
      The current selection.
      */
      selection, 
      /**
      @internal
      */
      values, computeSlot, tr) {
          this.config = config;
          this.doc = doc;
          this.selection = selection;
          this.values = values;
          this.status = config.statusTemplate.slice();
          this.computeSlot = computeSlot;
          // Fill in the computed state immediately, so that further queries
          // for it made during the update return this state
          if (tr)
              tr._state = this;
          for (let i = 0; i < this.config.dynamicSlots.length; i++)
              ensureAddr(this, i << 1);
          this.computeSlot = null;
      }
      field(field, require = true) {
          let addr = this.config.address[field.id];
          if (addr == null) {
              if (require)
                  throw new RangeError("Field is not present in this state");
              return undefined;
          }
          ensureAddr(this, addr);
          return getAddr(this, addr);
      }
      /**
      Create a [transaction](https://codemirror.net/6/docs/ref/#state.Transaction) that updates this
      state. Any number of [transaction specs](https://codemirror.net/6/docs/ref/#state.TransactionSpec)
      can be passed. Unless
      [`sequential`](https://codemirror.net/6/docs/ref/#state.TransactionSpec.sequential) is set, the
      [changes](https://codemirror.net/6/docs/ref/#state.TransactionSpec.changes) (if any) of each spec
      are assumed to start in the _current_ document (not the document
      produced by previous specs), and its
      [selection](https://codemirror.net/6/docs/ref/#state.TransactionSpec.selection) and
      [effects](https://codemirror.net/6/docs/ref/#state.TransactionSpec.effects) are assumed to refer
      to the document created by its _own_ changes. The resulting
      transaction contains the combined effect of all the different
      specs. For [selection](https://codemirror.net/6/docs/ref/#state.TransactionSpec.selection), later
      specs take precedence over earlier ones.
      */
      update(...specs) {
          return resolveTransaction(this, specs, true);
      }
      /**
      @internal
      */
      applyTransaction(tr) {
          let conf = this.config, { base, compartments } = conf;
          for (let effect of tr.effects) {
              if (effect.is(Compartment.reconfigure)) {
                  if (conf) {
                      compartments = new Map;
                      conf.compartments.forEach((val, key) => compartments.set(key, val));
                      conf = null;
                  }
                  compartments.set(effect.value.compartment, effect.value.extension);
              }
              else if (effect.is(StateEffect.reconfigure)) {
                  conf = null;
                  base = effect.value;
              }
              else if (effect.is(StateEffect.appendConfig)) {
                  conf = null;
                  base = asArray(base).concat(effect.value);
              }
          }
          let startValues;
          if (!conf) {
              conf = Configuration.resolve(base, compartments, this);
              let intermediateState = new EditorState(conf, this.doc, this.selection, conf.dynamicSlots.map(() => null), (state, slot) => slot.reconfigure(state, this), null);
              startValues = intermediateState.values;
          }
          else {
              startValues = tr.startState.values.slice();
          }
          new EditorState(conf, tr.newDoc, tr.newSelection, startValues, (state, slot) => slot.update(state, tr), tr);
      }
      /**
      Create a [transaction spec](https://codemirror.net/6/docs/ref/#state.TransactionSpec) that
      replaces every selection range with the given content.
      */
      replaceSelection(text) {
          if (typeof text == "string")
              text = this.toText(text);
          return this.changeByRange(range => ({ changes: { from: range.from, to: range.to, insert: text },
              range: EditorSelection.cursor(range.from + text.length) }));
      }
      /**
      Create a set of changes and a new selection by running the given
      function for each range in the active selection. The function
      can return an optional set of changes (in the coordinate space
      of the start document), plus an updated range (in the coordinate
      space of the document produced by the call's own changes). This
      method will merge all the changes and ranges into a single
      changeset and selection, and return it as a [transaction
      spec](https://codemirror.net/6/docs/ref/#state.TransactionSpec), which can be passed to
      [`update`](https://codemirror.net/6/docs/ref/#state.EditorState.update).
      */
      changeByRange(f) {
          let sel = this.selection;
          let result1 = f(sel.ranges[0]);
          let changes = this.changes(result1.changes), ranges = [result1.range];
          let effects = asArray(result1.effects);
          for (let i = 1; i < sel.ranges.length; i++) {
              let result = f(sel.ranges[i]);
              let newChanges = this.changes(result.changes), newMapped = newChanges.map(changes);
              for (let j = 0; j < i; j++)
                  ranges[j] = ranges[j].map(newMapped);
              let mapBy = changes.mapDesc(newChanges, true);
              ranges.push(result.range.map(mapBy));
              changes = changes.compose(newMapped);
              effects = StateEffect.mapEffects(effects, newMapped).concat(StateEffect.mapEffects(asArray(result.effects), mapBy));
          }
          return {
              changes,
              selection: EditorSelection.create(ranges, sel.mainIndex),
              effects
          };
      }
      /**
      Create a [change set](https://codemirror.net/6/docs/ref/#state.ChangeSet) from the given change
      description, taking the state's document length and line
      separator into account.
      */
      changes(spec = []) {
          if (spec instanceof ChangeSet)
              return spec;
          return ChangeSet.of(spec, this.doc.length, this.facet(EditorState.lineSeparator));
      }
      /**
      Using the state's [line
      separator](https://codemirror.net/6/docs/ref/#state.EditorState^lineSeparator), create a
      [`Text`](https://codemirror.net/6/docs/ref/#state.Text) instance from the given string.
      */
      toText(string) {
          return Text$1.of(string.split(this.facet(EditorState.lineSeparator) || DefaultSplit));
      }
      /**
      Return the given range of the document as a string.
      */
      sliceDoc(from = 0, to = this.doc.length) {
          return this.doc.sliceString(from, to, this.lineBreak);
      }
      /**
      Get the value of a state [facet](https://codemirror.net/6/docs/ref/#state.Facet).
      */
      facet(facet) {
          let addr = this.config.address[facet.id];
          if (addr == null)
              return facet.default;
          ensureAddr(this, addr);
          return getAddr(this, addr);
      }
      /**
      Convert this state to a JSON-serializable object. When custom
      fields should be serialized, you can pass them in as an object
      mapping property names (in the resulting object, which should
      not use `doc` or `selection`) to fields.
      */
      toJSON(fields) {
          let result = {
              doc: this.sliceDoc(),
              selection: this.selection.toJSON()
          };
          if (fields)
              for (let prop in fields) {
                  let value = fields[prop];
                  if (value instanceof StateField && this.config.address[value.id] != null)
                      result[prop] = value.spec.toJSON(this.field(fields[prop]), this);
              }
          return result;
      }
      /**
      Deserialize a state from its JSON representation. When custom
      fields should be deserialized, pass the same object you passed
      to [`toJSON`](https://codemirror.net/6/docs/ref/#state.EditorState.toJSON) when serializing as
      third argument.
      */
      static fromJSON(json, config = {}, fields) {
          if (!json || typeof json.doc != "string")
              throw new RangeError("Invalid JSON representation for EditorState");
          let fieldInit = [];
          if (fields)
              for (let prop in fields) {
                  if (Object.prototype.hasOwnProperty.call(json, prop)) {
                      let field = fields[prop], value = json[prop];
                      fieldInit.push(field.init(state => field.spec.fromJSON(value, state)));
                  }
              }
          return EditorState.create({
              doc: json.doc,
              selection: EditorSelection.fromJSON(json.selection),
              extensions: config.extensions ? fieldInit.concat([config.extensions]) : fieldInit
          });
      }
      /**
      Create a new state. You'll usually only need this when
      initializing an editor—updated states are created by applying
      transactions.
      */
      static create(config = {}) {
          let configuration = Configuration.resolve(config.extensions || [], new Map);
          let doc = config.doc instanceof Text$1 ? config.doc
              : Text$1.of((config.doc || "").split(configuration.staticFacet(EditorState.lineSeparator) || DefaultSplit));
          let selection = !config.selection ? EditorSelection.single(0)
              : config.selection instanceof EditorSelection ? config.selection
                  : EditorSelection.single(config.selection.anchor, config.selection.head);
          checkSelection(selection, doc.length);
          if (!configuration.staticFacet(allowMultipleSelections))
              selection = selection.asSingle();
          return new EditorState(configuration, doc, selection, configuration.dynamicSlots.map(() => null), (state, slot) => slot.create(state), null);
      }
      /**
      The size (in columns) of a tab in the document, determined by
      the [`tabSize`](https://codemirror.net/6/docs/ref/#state.EditorState^tabSize) facet.
      */
      get tabSize() { return this.facet(EditorState.tabSize); }
      /**
      Get the proper [line-break](https://codemirror.net/6/docs/ref/#state.EditorState^lineSeparator)
      string for this state.
      */
      get lineBreak() { return this.facet(EditorState.lineSeparator) || "\n"; }
      /**
      Returns true when the editor is
      [configured](https://codemirror.net/6/docs/ref/#state.EditorState^readOnly) to be read-only.
      */
      get readOnly() { return this.facet(readOnly); }
      /**
      Look up a translation for the given phrase (via the
      [`phrases`](https://codemirror.net/6/docs/ref/#state.EditorState^phrases) facet), or return the
      original string if no translation is found.
      
      If additional arguments are passed, they will be inserted in
      place of markers like `$1` (for the first value) and `$2`, etc.
      A single `$` is equivalent to `$1`, and `$$` will produce a
      literal dollar sign.
      */
      phrase(phrase, ...insert) {
          for (let map of this.facet(EditorState.phrases))
              if (Object.prototype.hasOwnProperty.call(map, phrase)) {
                  phrase = map[phrase];
                  break;
              }
          if (insert.length)
              phrase = phrase.replace(/\$(\$|\d*)/g, (m, i) => {
                  if (i == "$")
                      return "$";
                  let n = +(i || 1);
                  return !n || n > insert.length ? m : insert[n - 1];
              });
          return phrase;
      }
      /**
      Find the values for a given language data field, provided by the
      the [`languageData`](https://codemirror.net/6/docs/ref/#state.EditorState^languageData) facet.
      
      Examples of language data fields are...
      
      - [`"commentTokens"`](https://codemirror.net/6/docs/ref/#commands.CommentTokens) for specifying
        comment syntax.
      - [`"autocomplete"`](https://codemirror.net/6/docs/ref/#autocomplete.autocompletion^config.override)
        for providing language-specific completion sources.
      - [`"wordChars"`](https://codemirror.net/6/docs/ref/#state.EditorState.charCategorizer) for adding
        characters that should be considered part of words in this
        language.
      - [`"closeBrackets"`](https://codemirror.net/6/docs/ref/#autocomplete.CloseBracketConfig) controls
        bracket closing behavior.
      */
      languageDataAt(name, pos, side = -1) {
          let values = [];
          for (let provider of this.facet(languageData)) {
              for (let result of provider(this, pos, side)) {
                  if (Object.prototype.hasOwnProperty.call(result, name))
                      values.push(result[name]);
              }
          }
          return values;
      }
      /**
      Return a function that can categorize strings (expected to
      represent a single [grapheme cluster](https://codemirror.net/6/docs/ref/#state.findClusterBreak))
      into one of:
      
       - Word (contains an alphanumeric character or a character
         explicitly listed in the local language's `"wordChars"`
         language data, which should be a string)
       - Space (contains only whitespace)
       - Other (anything else)
      */
      charCategorizer(at) {
          return makeCategorizer(this.languageDataAt("wordChars", at).join(""));
      }
      /**
      Find the word at the given position, meaning the range
      containing all [word](https://codemirror.net/6/docs/ref/#state.CharCategory.Word) characters
      around it. If no word characters are adjacent to the position,
      this returns null.
      */
      wordAt(pos) {
          let { text, from, length } = this.doc.lineAt(pos);
          let cat = this.charCategorizer(pos);
          let start = pos - from, end = pos - from;
          while (start > 0) {
              let prev = findClusterBreak(text, start, false);
              if (cat(text.slice(prev, start)) != CharCategory.Word)
                  break;
              start = prev;
          }
          while (end < length) {
              let next = findClusterBreak(text, end);
              if (cat(text.slice(end, next)) != CharCategory.Word)
                  break;
              end = next;
          }
          return start == end ? null : EditorSelection.range(start + from, end + from);
      }
  }
  /**
  A facet that, when enabled, causes the editor to allow multiple
  ranges to be selected. Be careful though, because by default the
  editor relies on the native DOM selection, which cannot handle
  multiple selections. An extension like
  [`drawSelection`](https://codemirror.net/6/docs/ref/#view.drawSelection) can be used to make
  secondary selections visible to the user.
  */
  EditorState.allowMultipleSelections = allowMultipleSelections;
  /**
  Configures the tab size to use in this state. The first
  (highest-precedence) value of the facet is used. If no value is
  given, this defaults to 4.
  */
  EditorState.tabSize = /*@__PURE__*/Facet.define({
      combine: values => values.length ? values[0] : 4
  });
  /**
  The line separator to use. By default, any of `"\n"`, `"\r\n"`
  and `"\r"` is treated as a separator when splitting lines, and
  lines are joined with `"\n"`.

  When you configure a value here, only that precise separator
  will be used, allowing you to round-trip documents through the
  editor without normalizing line separators.
  */
  EditorState.lineSeparator = lineSeparator;
  /**
  This facet controls the value of the
  [`readOnly`](https://codemirror.net/6/docs/ref/#state.EditorState.readOnly) getter, which is
  consulted by commands and extensions that implement editing
  functionality to determine whether they should apply. It
  defaults to false, but when its highest-precedence value is
  `true`, such functionality disables itself.

  Not to be confused with
  [`EditorView.editable`](https://codemirror.net/6/docs/ref/#view.EditorView^editable), which
  controls whether the editor's DOM is set to be editable (and
  thus focusable).
  */
  EditorState.readOnly = readOnly;
  /**
  Registers translation phrases. The
  [`phrase`](https://codemirror.net/6/docs/ref/#state.EditorState.phrase) method will look through
  all objects registered with this facet to find translations for
  its argument.
  */
  EditorState.phrases = /*@__PURE__*/Facet.define({
      compare(a, b) {
          let kA = Object.keys(a), kB = Object.keys(b);
          return kA.length == kB.length && kA.every(k => a[k] == b[k]);
      }
  });
  /**
  A facet used to register [language
  data](https://codemirror.net/6/docs/ref/#state.EditorState.languageDataAt) providers.
  */
  EditorState.languageData = languageData;
  /**
  Facet used to register change filters, which are called for each
  transaction (unless explicitly
  [disabled](https://codemirror.net/6/docs/ref/#state.TransactionSpec.filter)), and can suppress
  part of the transaction's changes.

  Such a function can return `true` to indicate that it doesn't
  want to do anything, `false` to completely stop the changes in
  the transaction, or a set of ranges in which changes should be
  suppressed. Such ranges are represented as an array of numbers,
  with each pair of two numbers indicating the start and end of a
  range. So for example `[10, 20, 100, 110]` suppresses changes
  between 10 and 20, and between 100 and 110.
  */
  EditorState.changeFilter = changeFilter;
  /**
  Facet used to register a hook that gets a chance to update or
  replace transaction specs before they are applied. This will
  only be applied for transactions that don't have
  [`filter`](https://codemirror.net/6/docs/ref/#state.TransactionSpec.filter) set to `false`. You
  can either return a single transaction spec (possibly the input
  transaction), or an array of specs (which will be combined in
  the same way as the arguments to
  [`EditorState.update`](https://codemirror.net/6/docs/ref/#state.EditorState.update)).

  When possible, it is recommended to avoid accessing
  [`Transaction.state`](https://codemirror.net/6/docs/ref/#state.Transaction.state) in a filter,
  since it will force creation of a state that will then be
  discarded again, if the transaction is actually filtered.

  (This functionality should be used with care. Indiscriminately
  modifying transaction is likely to break something or degrade
  the user experience.)
  */
  EditorState.transactionFilter = transactionFilter;
  /**
  This is a more limited form of
  [`transactionFilter`](https://codemirror.net/6/docs/ref/#state.EditorState^transactionFilter),
  which can only add
  [annotations](https://codemirror.net/6/docs/ref/#state.TransactionSpec.annotations) and
  [effects](https://codemirror.net/6/docs/ref/#state.TransactionSpec.effects). _But_, this type
  of filter runs even if the transaction has disabled regular
  [filtering](https://codemirror.net/6/docs/ref/#state.TransactionSpec.filter), making it suitable
  for effects that don't need to touch the changes or selection,
  but do want to process every transaction.

  Extenders run _after_ filters, when both are present.
  */
  EditorState.transactionExtender = transactionExtender;
  Compartment.reconfigure = /*@__PURE__*/StateEffect.define();

  /**
  Utility function for combining behaviors to fill in a config
  object from an array of provided configs. `defaults` should hold
  default values for all optional fields in `Config`.

  The function will, by default, error
  when a field gets two values that aren't `===`-equal, but you can
  provide combine functions per field to do something else.
  */
  function combineConfig(configs, defaults, // Should hold only the optional properties of Config, but I haven't managed to express that
  combine = {}) {
      let result = {};
      for (let config of configs)
          for (let key of Object.keys(config)) {
              let value = config[key], current = result[key];
              if (current === undefined)
                  result[key] = value;
              else if (current === value || value === undefined) ; // No conflict
              else if (Object.hasOwnProperty.call(combine, key))
                  result[key] = combine[key](current, value);
              else
                  throw new Error("Config merge conflict for field " + key);
          }
      for (let key in defaults)
          if (result[key] === undefined)
              result[key] = defaults[key];
      return result;
  }

  /**
  Each range is associated with a value, which must inherit from
  this class.
  */
  class RangeValue {
      /**
      Compare this value with another value. Used when comparing
      rangesets. The default implementation compares by identity.
      Unless you are only creating a fixed number of unique instances
      of your value type, it is a good idea to implement this
      properly.
      */
      eq(other) { return this == other; }
      /**
      Create a [range](https://codemirror.net/6/docs/ref/#state.Range) with this value.
      */
      range(from, to = from) { return Range.create(from, to, this); }
  }
  RangeValue.prototype.startSide = RangeValue.prototype.endSide = 0;
  RangeValue.prototype.point = false;
  RangeValue.prototype.mapMode = MapMode.TrackDel;
  /**
  A range associates a value with a range of positions.
  */
  class Range {
      constructor(
      /**
      The range's start position.
      */
      from, 
      /**
      Its end position.
      */
      to, 
      /**
      The value associated with this range.
      */
      value) {
          this.from = from;
          this.to = to;
          this.value = value;
      }
      /**
      @internal
      */
      static create(from, to, value) {
          return new Range(from, to, value);
      }
  }
  function cmpRange(a, b) {
      return a.from - b.from || a.value.startSide - b.value.startSide;
  }
  class Chunk {
      constructor(from, to, value, 
      // Chunks are marked with the largest point that occurs
      // in them (or -1 for no points), so that scans that are
      // only interested in points (such as the
      // heightmap-related logic) can skip range-only chunks.
      maxPoint) {
          this.from = from;
          this.to = to;
          this.value = value;
          this.maxPoint = maxPoint;
      }
      get length() { return this.to[this.to.length - 1]; }
      // Find the index of the given position and side. Use the ranges'
      // `from` pos when `end == false`, `to` when `end == true`.
      findIndex(pos, side, end, startAt = 0) {
          let arr = end ? this.to : this.from;
          for (let lo = startAt, hi = arr.length;;) {
              if (lo == hi)
                  return lo;
              let mid = (lo + hi) >> 1;
              let diff = arr[mid] - pos || (end ? this.value[mid].endSide : this.value[mid].startSide) - side;
              if (mid == lo)
                  return diff >= 0 ? lo : hi;
              if (diff >= 0)
                  hi = mid;
              else
                  lo = mid + 1;
          }
      }
      between(offset, from, to, f) {
          for (let i = this.findIndex(from, -1000000000 /* C.Far */, true), e = this.findIndex(to, 1000000000 /* C.Far */, false, i); i < e; i++)
              if (f(this.from[i] + offset, this.to[i] + offset, this.value[i]) === false)
                  return false;
      }
      map(offset, changes) {
          let value = [], from = [], to = [], newPos = -1, maxPoint = -1;
          for (let i = 0; i < this.value.length; i++) {
              let val = this.value[i], curFrom = this.from[i] + offset, curTo = this.to[i] + offset, newFrom, newTo;
              if (curFrom == curTo) {
                  let mapped = changes.mapPos(curFrom, val.startSide, val.mapMode);
                  if (mapped == null)
                      continue;
                  newFrom = newTo = mapped;
                  if (val.startSide != val.endSide) {
                      newTo = changes.mapPos(curFrom, val.endSide);
                      if (newTo < newFrom)
                          continue;
                  }
              }
              else {
                  newFrom = changes.mapPos(curFrom, val.startSide);
                  newTo = changes.mapPos(curTo, val.endSide);
                  if (newFrom > newTo || newFrom == newTo && val.startSide > 0 && val.endSide <= 0)
                      continue;
              }
              if ((newTo - newFrom || val.endSide - val.startSide) < 0)
                  continue;
              if (newPos < 0)
                  newPos = newFrom;
              if (val.point)
                  maxPoint = Math.max(maxPoint, newTo - newFrom);
              value.push(val);
              from.push(newFrom - newPos);
              to.push(newTo - newPos);
          }
          return { mapped: value.length ? new Chunk(from, to, value, maxPoint) : null, pos: newPos };
      }
  }
  /**
  A range set stores a collection of [ranges](https://codemirror.net/6/docs/ref/#state.Range) in a
  way that makes them efficient to [map](https://codemirror.net/6/docs/ref/#state.RangeSet.map) and
  [update](https://codemirror.net/6/docs/ref/#state.RangeSet.update). This is an immutable data
  structure.
  */
  class RangeSet {
      constructor(
      /**
      @internal
      */
      chunkPos, 
      /**
      @internal
      */
      chunk, 
      /**
      @internal
      */
      nextLayer, 
      /**
      @internal
      */
      maxPoint) {
          this.chunkPos = chunkPos;
          this.chunk = chunk;
          this.nextLayer = nextLayer;
          this.maxPoint = maxPoint;
      }
      /**
      @internal
      */
      static create(chunkPos, chunk, nextLayer, maxPoint) {
          return new RangeSet(chunkPos, chunk, nextLayer, maxPoint);
      }
      /**
      @internal
      */
      get length() {
          let last = this.chunk.length - 1;
          return last < 0 ? 0 : Math.max(this.chunkEnd(last), this.nextLayer.length);
      }
      /**
      The number of ranges in the set.
      */
      get size() {
          if (this.isEmpty)
              return 0;
          let size = this.nextLayer.size;
          for (let chunk of this.chunk)
              size += chunk.value.length;
          return size;
      }
      /**
      @internal
      */
      chunkEnd(index) {
          return this.chunkPos[index] + this.chunk[index].length;
      }
      /**
      Update the range set, optionally adding new ranges or filtering
      out existing ones.
      
      (Note: The type parameter is just there as a kludge to work
      around TypeScript variance issues that prevented `RangeSet<X>`
      from being a subtype of `RangeSet<Y>` when `X` is a subtype of
      `Y`.)
      */
      update(updateSpec) {
          let { add = [], sort = false, filterFrom = 0, filterTo = this.length } = updateSpec;
          let filter = updateSpec.filter;
          if (add.length == 0 && !filter)
              return this;
          if (sort)
              add = add.slice().sort(cmpRange);
          if (this.isEmpty)
              return add.length ? RangeSet.of(add) : this;
          let cur = new LayerCursor(this, null, -1).goto(0), i = 0, spill = [];
          let builder = new RangeSetBuilder();
          while (cur.value || i < add.length) {
              if (i < add.length && (cur.from - add[i].from || cur.startSide - add[i].value.startSide) >= 0) {
                  let range = add[i++];
                  if (!builder.addInner(range.from, range.to, range.value))
                      spill.push(range);
              }
              else if (cur.rangeIndex == 1 && cur.chunkIndex < this.chunk.length &&
                  (i == add.length || this.chunkEnd(cur.chunkIndex) < add[i].from) &&
                  (!filter || filterFrom > this.chunkEnd(cur.chunkIndex) || filterTo < this.chunkPos[cur.chunkIndex]) &&
                  builder.addChunk(this.chunkPos[cur.chunkIndex], this.chunk[cur.chunkIndex])) {
                  cur.nextChunk();
              }
              else {
                  if (!filter || filterFrom > cur.to || filterTo < cur.from || filter(cur.from, cur.to, cur.value)) {
                      if (!builder.addInner(cur.from, cur.to, cur.value))
                          spill.push(Range.create(cur.from, cur.to, cur.value));
                  }
                  cur.next();
              }
          }
          return builder.finishInner(this.nextLayer.isEmpty && !spill.length ? RangeSet.empty
              : this.nextLayer.update({ add: spill, filter, filterFrom, filterTo }));
      }
      /**
      Map this range set through a set of changes, return the new set.
      */
      map(changes) {
          if (changes.empty || this.isEmpty)
              return this;
          let chunks = [], chunkPos = [], maxPoint = -1;
          for (let i = 0; i < this.chunk.length; i++) {
              let start = this.chunkPos[i], chunk = this.chunk[i];
              let touch = changes.touchesRange(start, start + chunk.length);
              if (touch === false) {
                  maxPoint = Math.max(maxPoint, chunk.maxPoint);
                  chunks.push(chunk);
                  chunkPos.push(changes.mapPos(start));
              }
              else if (touch === true) {
                  let { mapped, pos } = chunk.map(start, changes);
                  if (mapped) {
                      maxPoint = Math.max(maxPoint, mapped.maxPoint);
                      chunks.push(mapped);
                      chunkPos.push(pos);
                  }
              }
          }
          let next = this.nextLayer.map(changes);
          return chunks.length == 0 ? next : new RangeSet(chunkPos, chunks, next || RangeSet.empty, maxPoint);
      }
      /**
      Iterate over the ranges that touch the region `from` to `to`,
      calling `f` for each. There is no guarantee that the ranges will
      be reported in any specific order. When the callback returns
      `false`, iteration stops.
      */
      between(from, to, f) {
          if (this.isEmpty)
              return;
          for (let i = 0; i < this.chunk.length; i++) {
              let start = this.chunkPos[i], chunk = this.chunk[i];
              if (to >= start && from <= start + chunk.length &&
                  chunk.between(start, from - start, to - start, f) === false)
                  return;
          }
          this.nextLayer.between(from, to, f);
      }
      /**
      Iterate over the ranges in this set, in order, including all
      ranges that end at or after `from`.
      */
      iter(from = 0) {
          return HeapCursor.from([this]).goto(from);
      }
      /**
      @internal
      */
      get isEmpty() { return this.nextLayer == this; }
      /**
      Iterate over the ranges in a collection of sets, in order,
      starting from `from`.
      */
      static iter(sets, from = 0) {
          return HeapCursor.from(sets).goto(from);
      }
      /**
      Iterate over two groups of sets, calling methods on `comparator`
      to notify it of possible differences.
      */
      static compare(oldSets, newSets, 
      /**
      This indicates how the underlying data changed between these
      ranges, and is needed to synchronize the iteration. `from` and
      `to` are coordinates in the _new_ space, after these changes.
      */
      textDiff, comparator, 
      /**
      Can be used to ignore all non-point ranges, and points below
      the given size. When -1, all ranges are compared.
      */
      minPointSize = -1) {
          let a = oldSets.filter(set => set.maxPoint > 0 || !set.isEmpty && set.maxPoint >= minPointSize);
          let b = newSets.filter(set => set.maxPoint > 0 || !set.isEmpty && set.maxPoint >= minPointSize);
          let sharedChunks = findSharedChunks(a, b, textDiff);
          let sideA = new SpanCursor(a, sharedChunks, minPointSize);
          let sideB = new SpanCursor(b, sharedChunks, minPointSize);
          textDiff.iterGaps((fromA, fromB, length) => compare(sideA, fromA, sideB, fromB, length, comparator));
          if (textDiff.empty && textDiff.length == 0)
              compare(sideA, 0, sideB, 0, 0, comparator);
      }
      /**
      Compare the contents of two groups of range sets, returning true
      if they are equivalent in the given range.
      */
      static eq(oldSets, newSets, from = 0, to) {
          if (to == null)
              to = 1000000000 /* C.Far */ - 1;
          let a = oldSets.filter(set => !set.isEmpty && newSets.indexOf(set) < 0);
          let b = newSets.filter(set => !set.isEmpty && oldSets.indexOf(set) < 0);
          if (a.length != b.length)
              return false;
          if (!a.length)
              return true;
          let sharedChunks = findSharedChunks(a, b);
          let sideA = new SpanCursor(a, sharedChunks, 0).goto(from), sideB = new SpanCursor(b, sharedChunks, 0).goto(from);
          for (;;) {
              if (sideA.to != sideB.to ||
                  !sameValues(sideA.active, sideB.active) ||
                  sideA.point && (!sideB.point || !sideA.point.eq(sideB.point)))
                  return false;
              if (sideA.to > to)
                  return true;
              sideA.next();
              sideB.next();
          }
      }
      /**
      Iterate over a group of range sets at the same time, notifying
      the iterator about the ranges covering every given piece of
      content. Returns the open count (see
      [`SpanIterator.span`](https://codemirror.net/6/docs/ref/#state.SpanIterator.span)) at the end
      of the iteration.
      */
      static spans(sets, from, to, iterator, 
      /**
      When given and greater than -1, only points of at least this
      size are taken into account.
      */
      minPointSize = -1) {
          let cursor = new SpanCursor(sets, null, minPointSize).goto(from), pos = from;
          let openRanges = cursor.openStart;
          for (;;) {
              let curTo = Math.min(cursor.to, to);
              if (cursor.point) {
                  let active = cursor.activeForPoint(cursor.to);
                  let openCount = cursor.pointFrom < from ? active.length + 1 : Math.min(active.length, openRanges);
                  iterator.point(pos, curTo, cursor.point, active, openCount, cursor.pointRank);
                  openRanges = Math.min(cursor.openEnd(curTo), active.length);
              }
              else if (curTo > pos) {
                  iterator.span(pos, curTo, cursor.active, openRanges);
                  openRanges = cursor.openEnd(curTo);
              }
              if (cursor.to > to)
                  return openRanges + (cursor.point && cursor.to > to ? 1 : 0);
              pos = cursor.to;
              cursor.next();
          }
      }
      /**
      Create a range set for the given range or array of ranges. By
      default, this expects the ranges to be _sorted_ (by start
      position and, if two start at the same position,
      `value.startSide`). You can pass `true` as second argument to
      cause the method to sort them.
      */
      static of(ranges, sort = false) {
          let build = new RangeSetBuilder();
          for (let range of ranges instanceof Range ? [ranges] : sort ? lazySort(ranges) : ranges)
              build.add(range.from, range.to, range.value);
          return build.finish();
      }
  }
  /**
  The empty set of ranges.
  */
  RangeSet.empty = /*@__PURE__*/new RangeSet([], [], null, -1);
  function lazySort(ranges) {
      if (ranges.length > 1)
          for (let prev = ranges[0], i = 1; i < ranges.length; i++) {
              let cur = ranges[i];
              if (cmpRange(prev, cur) > 0)
                  return ranges.slice().sort(cmpRange);
              prev = cur;
          }
      return ranges;
  }
  RangeSet.empty.nextLayer = RangeSet.empty;
  /**
  A range set builder is a data structure that helps build up a
  [range set](https://codemirror.net/6/docs/ref/#state.RangeSet) directly, without first allocating
  an array of [`Range`](https://codemirror.net/6/docs/ref/#state.Range) objects.
  */
  class RangeSetBuilder {
      /**
      Create an empty builder.
      */
      constructor() {
          this.chunks = [];
          this.chunkPos = [];
          this.chunkStart = -1;
          this.last = null;
          this.lastFrom = -1000000000 /* C.Far */;
          this.lastTo = -1000000000 /* C.Far */;
          this.from = [];
          this.to = [];
          this.value = [];
          this.maxPoint = -1;
          this.setMaxPoint = -1;
          this.nextLayer = null;
      }
      finishChunk(newArrays) {
          this.chunks.push(new Chunk(this.from, this.to, this.value, this.maxPoint));
          this.chunkPos.push(this.chunkStart);
          this.chunkStart = -1;
          this.setMaxPoint = Math.max(this.setMaxPoint, this.maxPoint);
          this.maxPoint = -1;
          if (newArrays) {
              this.from = [];
              this.to = [];
              this.value = [];
          }
      }
      /**
      Add a range. Ranges should be added in sorted (by `from` and
      `value.startSide`) order.
      */
      add(from, to, value) {
          if (!this.addInner(from, to, value))
              (this.nextLayer || (this.nextLayer = new RangeSetBuilder)).add(from, to, value);
      }
      /**
      @internal
      */
      addInner(from, to, value) {
          let diff = from - this.lastTo || value.startSide - this.last.endSide;
          if (diff <= 0 && (from - this.lastFrom || value.startSide - this.last.startSide) < 0)
              throw new Error("Ranges must be added sorted by `from` position and `startSide`");
          if (diff < 0)
              return false;
          if (this.from.length == 250 /* C.ChunkSize */)
              this.finishChunk(true);
          if (this.chunkStart < 0)
              this.chunkStart = from;
          this.from.push(from - this.chunkStart);
          this.to.push(to - this.chunkStart);
          this.last = value;
          this.lastFrom = from;
          this.lastTo = to;
          this.value.push(value);
          if (value.point)
              this.maxPoint = Math.max(this.maxPoint, to - from);
          return true;
      }
      /**
      @internal
      */
      addChunk(from, chunk) {
          if ((from - this.lastTo || chunk.value[0].startSide - this.last.endSide) < 0)
              return false;
          if (this.from.length)
              this.finishChunk(true);
          this.setMaxPoint = Math.max(this.setMaxPoint, chunk.maxPoint);
          this.chunks.push(chunk);
          this.chunkPos.push(from);
          let last = chunk.value.length - 1;
          this.last = chunk.value[last];
          this.lastFrom = chunk.from[last] + from;
          this.lastTo = chunk.to[last] + from;
          return true;
      }
      /**
      Finish the range set. Returns the new set. The builder can't be
      used anymore after this has been called.
      */
      finish() { return this.finishInner(RangeSet.empty); }
      /**
      @internal
      */
      finishInner(next) {
          if (this.from.length)
              this.finishChunk(false);
          if (this.chunks.length == 0)
              return next;
          let result = RangeSet.create(this.chunkPos, this.chunks, this.nextLayer ? this.nextLayer.finishInner(next) : next, this.setMaxPoint);
          this.from = null; // Make sure further `add` calls produce errors
          return result;
      }
  }
  function findSharedChunks(a, b, textDiff) {
      let inA = new Map();
      for (let set of a)
          for (let i = 0; i < set.chunk.length; i++)
              if (set.chunk[i].maxPoint <= 0)
                  inA.set(set.chunk[i], set.chunkPos[i]);
      let shared = new Set();
      for (let set of b)
          for (let i = 0; i < set.chunk.length; i++) {
              let known = inA.get(set.chunk[i]);
              if (known != null && (textDiff ? textDiff.mapPos(known) : known) == set.chunkPos[i] &&
                  !(textDiff === null || textDiff === void 0 ? void 0 : textDiff.touchesRange(known, known + set.chunk[i].length)))
                  shared.add(set.chunk[i]);
          }
      return shared;
  }
  class LayerCursor {
      constructor(layer, skip, minPoint, rank = 0) {
          this.layer = layer;
          this.skip = skip;
          this.minPoint = minPoint;
          this.rank = rank;
      }
      get startSide() { return this.value ? this.value.startSide : 0; }
      get endSide() { return this.value ? this.value.endSide : 0; }
      goto(pos, side = -1000000000 /* C.Far */) {
          this.chunkIndex = this.rangeIndex = 0;
          this.gotoInner(pos, side, false);
          return this;
      }
      gotoInner(pos, side, forward) {
          while (this.chunkIndex < this.layer.chunk.length) {
              let next = this.layer.chunk[this.chunkIndex];
              if (!(this.skip && this.skip.has(next) ||
                  this.layer.chunkEnd(this.chunkIndex) < pos ||
                  next.maxPoint < this.minPoint))
                  break;
              this.chunkIndex++;
              forward = false;
          }
          if (this.chunkIndex < this.layer.chunk.length) {
              let rangeIndex = this.layer.chunk[this.chunkIndex].findIndex(pos - this.layer.chunkPos[this.chunkIndex], side, true);
              if (!forward || this.rangeIndex < rangeIndex)
                  this.setRangeIndex(rangeIndex);
          }
          this.next();
      }
      forward(pos, side) {
          if ((this.to - pos || this.endSide - side) < 0)
              this.gotoInner(pos, side, true);
      }
      next() {
          for (;;) {
              if (this.chunkIndex == this.layer.chunk.length) {
                  this.from = this.to = 1000000000 /* C.Far */;
                  this.value = null;
                  break;
              }
              else {
                  let chunkPos = this.layer.chunkPos[this.chunkIndex], chunk = this.layer.chunk[this.chunkIndex];
                  let from = chunkPos + chunk.from[this.rangeIndex];
                  this.from = from;
                  this.to = chunkPos + chunk.to[this.rangeIndex];
                  this.value = chunk.value[this.rangeIndex];
                  this.setRangeIndex(this.rangeIndex + 1);
                  if (this.minPoint < 0 || this.value.point && this.to - this.from >= this.minPoint)
                      break;
              }
          }
      }
      setRangeIndex(index) {
          if (index == this.layer.chunk[this.chunkIndex].value.length) {
              this.chunkIndex++;
              if (this.skip) {
                  while (this.chunkIndex < this.layer.chunk.length && this.skip.has(this.layer.chunk[this.chunkIndex]))
                      this.chunkIndex++;
              }
              this.rangeIndex = 0;
          }
          else {
              this.rangeIndex = index;
          }
      }
      nextChunk() {
          this.chunkIndex++;
          this.rangeIndex = 0;
          this.next();
      }
      compare(other) {
          return this.from - other.from || this.startSide - other.startSide || this.rank - other.rank ||
              this.to - other.to || this.endSide - other.endSide;
      }
  }
  class HeapCursor {
      constructor(heap) {
          this.heap = heap;
      }
      static from(sets, skip = null, minPoint = -1) {
          let heap = [];
          for (let i = 0; i < sets.length; i++) {
              for (let cur = sets[i]; !cur.isEmpty; cur = cur.nextLayer) {
                  if (cur.maxPoint >= minPoint)
                      heap.push(new LayerCursor(cur, skip, minPoint, i));
              }
          }
          return heap.length == 1 ? heap[0] : new HeapCursor(heap);
      }
      get startSide() { return this.value ? this.value.startSide : 0; }
      goto(pos, side = -1000000000 /* C.Far */) {
          for (let cur of this.heap)
              cur.goto(pos, side);
          for (let i = this.heap.length >> 1; i >= 0; i--)
              heapBubble(this.heap, i);
          this.next();
          return this;
      }
      forward(pos, side) {
          for (let cur of this.heap)
              cur.forward(pos, side);
          for (let i = this.heap.length >> 1; i >= 0; i--)
              heapBubble(this.heap, i);
          if ((this.to - pos || this.value.endSide - side) < 0)
              this.next();
      }
      next() {
          if (this.heap.length == 0) {
              this.from = this.to = 1000000000 /* C.Far */;
              this.value = null;
              this.rank = -1;
          }
          else {
              let top = this.heap[0];
              this.from = top.from;
              this.to = top.to;
              this.value = top.value;
              this.rank = top.rank;
              if (top.value)
                  top.next();
              heapBubble(this.heap, 0);
          }
      }
  }
  function heapBubble(heap, index) {
      for (let cur = heap[index];;) {
          let childIndex = (index << 1) + 1;
          if (childIndex >= heap.length)
              break;
          let child = heap[childIndex];
          if (childIndex + 1 < heap.length && child.compare(heap[childIndex + 1]) >= 0) {
              child = heap[childIndex + 1];
              childIndex++;
          }
          if (cur.compare(child) < 0)
              break;
          heap[childIndex] = cur;
          heap[index] = child;
          index = childIndex;
      }
  }
  class SpanCursor {
      constructor(sets, skip, minPoint) {
          this.minPoint = minPoint;
          this.active = [];
          this.activeTo = [];
          this.activeRank = [];
          this.minActive = -1;
          // A currently active point range, if any
          this.point = null;
          this.pointFrom = 0;
          this.pointRank = 0;
          this.to = -1000000000 /* C.Far */;
          this.endSide = 0;
          // The amount of open active ranges at the start of the iterator.
          // Not including points.
          this.openStart = -1;
          this.cursor = HeapCursor.from(sets, skip, minPoint);
      }
      goto(pos, side = -1000000000 /* C.Far */) {
          this.cursor.goto(pos, side);
          this.active.length = this.activeTo.length = this.activeRank.length = 0;
          this.minActive = -1;
          this.to = pos;
          this.endSide = side;
          this.openStart = -1;
          this.next();
          return this;
      }
      forward(pos, side) {
          while (this.minActive > -1 && (this.activeTo[this.minActive] - pos || this.active[this.minActive].endSide - side) < 0)
              this.removeActive(this.minActive);
          this.cursor.forward(pos, side);
      }
      removeActive(index) {
          remove(this.active, index);
          remove(this.activeTo, index);
          remove(this.activeRank, index);
          this.minActive = findMinIndex(this.active, this.activeTo);
      }
      addActive(trackOpen) {
          let i = 0, { value, to, rank } = this.cursor;
          while (i < this.activeRank.length && this.activeRank[i] <= rank)
              i++;
          insert(this.active, i, value);
          insert(this.activeTo, i, to);
          insert(this.activeRank, i, rank);
          if (trackOpen)
              insert(trackOpen, i, this.cursor.from);
          this.minActive = findMinIndex(this.active, this.activeTo);
      }
      // After calling this, if `this.point` != null, the next range is a
      // point. Otherwise, it's a regular range, covered by `this.active`.
      next() {
          let from = this.to, wasPoint = this.point;
          this.point = null;
          let trackOpen = this.openStart < 0 ? [] : null;
          for (;;) {
              let a = this.minActive;
              if (a > -1 && (this.activeTo[a] - this.cursor.from || this.active[a].endSide - this.cursor.startSide) < 0) {
                  if (this.activeTo[a] > from) {
                      this.to = this.activeTo[a];
                      this.endSide = this.active[a].endSide;
                      break;
                  }
                  this.removeActive(a);
                  if (trackOpen)
                      remove(trackOpen, a);
              }
              else if (!this.cursor.value) {
                  this.to = this.endSide = 1000000000 /* C.Far */;
                  break;
              }
              else if (this.cursor.from > from) {
                  this.to = this.cursor.from;
                  this.endSide = this.cursor.startSide;
                  break;
              }
              else {
                  let nextVal = this.cursor.value;
                  if (!nextVal.point) { // Opening a range
                      this.addActive(trackOpen);
                      this.cursor.next();
                  }
                  else if (wasPoint && this.cursor.to == this.to && this.cursor.from < this.cursor.to) {
                      // Ignore any non-empty points that end precisely at the end of the prev point
                      this.cursor.next();
                  }
                  else { // New point
                      this.point = nextVal;
                      this.pointFrom = this.cursor.from;
                      this.pointRank = this.cursor.rank;
                      this.to = this.cursor.to;
                      this.endSide = nextVal.endSide;
                      this.cursor.next();
                      this.forward(this.to, this.endSide);
                      break;
                  }
              }
          }
          if (trackOpen) {
              this.openStart = 0;
              for (let i = trackOpen.length - 1; i >= 0 && trackOpen[i] < from; i--)
                  this.openStart++;
          }
      }
      activeForPoint(to) {
          if (!this.active.length)
              return this.active;
          let active = [];
          for (let i = this.active.length - 1; i >= 0; i--) {
              if (this.activeRank[i] < this.pointRank)
                  break;
              if (this.activeTo[i] > to || this.activeTo[i] == to && this.active[i].endSide >= this.point.endSide)
                  active.push(this.active[i]);
          }
          return active.reverse();
      }
      openEnd(to) {
          let open = 0;
          for (let i = this.activeTo.length - 1; i >= 0 && this.activeTo[i] > to; i--)
              open++;
          return open;
      }
  }
  function compare(a, startA, b, startB, length, comparator) {
      a.goto(startA);
      b.goto(startB);
      let endB = startB + length;
      let pos = startB, dPos = startB - startA;
      for (;;) {
          let diff = (a.to + dPos) - b.to || a.endSide - b.endSide;
          let end = diff < 0 ? a.to + dPos : b.to, clipEnd = Math.min(end, endB);
          if (a.point || b.point) {
              if (!(a.point && b.point && (a.point == b.point || a.point.eq(b.point)) &&
                  sameValues(a.activeForPoint(a.to + dPos), b.activeForPoint(b.to))))
                  comparator.comparePoint(pos, clipEnd, a.point, b.point);
          }
          else {
              if (clipEnd > pos && !sameValues(a.active, b.active))
                  comparator.compareRange(pos, clipEnd, a.active, b.active);
          }
          if (end > endB)
              break;
          pos = end;
          if (diff <= 0)
              a.next();
          if (diff >= 0)
              b.next();
      }
  }
  function sameValues(a, b) {
      if (a.length != b.length)
          return false;
      for (let i = 0; i < a.length; i++)
          if (a[i] != b[i] && !a[i].eq(b[i]))
              return false;
      return true;
  }
  function remove(array, index) {
      for (let i = index, e = array.length - 1; i < e; i++)
          array[i] = array[i + 1];
      array.pop();
  }
  function insert(array, index, value) {
      for (let i = array.length - 1; i >= index; i--)
          array[i + 1] = array[i];
      array[index] = value;
  }
  function findMinIndex(value, array) {
      let found = -1, foundPos = 1000000000 /* C.Far */;
      for (let i = 0; i < array.length; i++)
          if ((array[i] - foundPos || value[i].endSide - value[found].endSide) < 0) {
              found = i;
              foundPos = array[i];
          }
      return found;
  }
  /**
  Find the offset that corresponds to the given column position in a
  string, taking extending characters and tab size into account. By
  default, the string length is returned when it is too short to
  reach the column. Pass `strict` true to make it return -1 in that
  situation.
  */
  function findColumn(string, col, tabSize, strict) {
      for (let i = 0, n = 0;;) {
          if (n >= col)
              return i;
          if (i == string.length)
              break;
          n += string.charCodeAt(i) == 9 ? tabSize - (n % tabSize) : 1;
          i = findClusterBreak(string, i);
      }
      return strict === true ? -1 : string.length;
  }

  const C$2 = "\u037c";
  const COUNT = typeof Symbol == "undefined" ? "__" + C$2 : Symbol.for(C$2);
  const SET = typeof Symbol == "undefined" ? "__styleSet" + Math.floor(Math.random() * 1e8) : Symbol("styleSet");
  const top = typeof globalThis != "undefined" ? globalThis : typeof window != "undefined" ? window : {};

  // :: - Style modules encapsulate a set of CSS rules defined from
  // JavaScript. Their definitions are only available in a given DOM
  // root after it has been _mounted_ there with `StyleModule.mount`.
  //
  // Style modules should be created once and stored somewhere, as
  // opposed to re-creating them every time you need them. The amount of
  // CSS rules generated for a given DOM root is bounded by the amount
  // of style modules that were used. So to avoid leaking rules, don't
  // create these dynamically, but treat them as one-time allocations.
  class StyleModule {
    // :: (Object<Style>, ?{finish: ?(string) → string})
    // Create a style module from the given spec.
    //
    // When `finish` is given, it is called on regular (non-`@`)
    // selectors (after `&` expansion) to compute the final selector.
    constructor(spec, options) {
      this.rules = [];
      let {finish} = options || {};

      function splitSelector(selector) {
        return /^@/.test(selector) ? [selector] : selector.split(/,\s*/)
      }

      function render(selectors, spec, target, isKeyframes) {
        let local = [], isAt = /^@(\w+)\b/.exec(selectors[0]), keyframes = isAt && isAt[1] == "keyframes";
        if (isAt && spec == null) return target.push(selectors[0] + ";")
        for (let prop in spec) {
          let value = spec[prop];
          if (/&/.test(prop)) {
            render(prop.split(/,\s*/).map(part => selectors.map(sel => part.replace(/&/, sel))).reduce((a, b) => a.concat(b)),
                   value, target);
          } else if (value && typeof value == "object") {
            if (!isAt) throw new RangeError("The value of a property (" + prop + ") should be a primitive value.")
            render(splitSelector(prop), value, local, keyframes);
          } else if (value != null) {
            local.push(prop.replace(/_.*/, "").replace(/[A-Z]/g, l => "-" + l.toLowerCase()) + ": " + value + ";");
          }
        }
        if (local.length || keyframes) {
          target.push((finish && !isAt && !isKeyframes ? selectors.map(finish) : selectors).join(", ") +
                      " {" + local.join(" ") + "}");
        }
      }

      for (let prop in spec) render(splitSelector(prop), spec[prop], this.rules);
    }

    // :: () → string
    // Returns a string containing the module's CSS rules.
    getRules() { return this.rules.join("\n") }

    // :: () → string
    // Generate a new unique CSS class name.
    static newName() {
      let id = top[COUNT] || 1;
      top[COUNT] = id + 1;
      return C$2 + id.toString(36)
    }

    // :: (union<Document, ShadowRoot>, union<[StyleModule], StyleModule>)
    //
    // Mount the given set of modules in the given DOM root, which ensures
    // that the CSS rules defined by the module are available in that
    // context.
    //
    // Rules are only added to the document once per root.
    //
    // Rule order will follow the order of the modules, so that rules from
    // modules later in the array take precedence of those from earlier
    // modules. If you call this function multiple times for the same root
    // in a way that changes the order of already mounted modules, the old
    // order will be changed.
    static mount(root, modules) {
      (root[SET] || new StyleSet(root)).mount(Array.isArray(modules) ? modules : [modules]);
    }
  }

  let adoptedSet = null;

  class StyleSet {
    constructor(root) {
      if (!root.head && root.adoptedStyleSheets && typeof CSSStyleSheet != "undefined") {
        if (adoptedSet) {
          root.adoptedStyleSheets = [adoptedSet.sheet].concat(root.adoptedStyleSheets);
          return root[SET] = adoptedSet
        }
        this.sheet = new CSSStyleSheet;
        root.adoptedStyleSheets = [this.sheet].concat(root.adoptedStyleSheets);
        adoptedSet = this;
      } else {
        this.styleTag = (root.ownerDocument || root).createElement("style");
        let target = root.head || root;
        target.insertBefore(this.styleTag, target.firstChild);
      }
      this.modules = [];
      root[SET] = this;
    }

    mount(modules) {
      let sheet = this.sheet;
      let pos = 0 /* Current rule offset */, j = 0; /* Index into this.modules */
      for (let i = 0; i < modules.length; i++) {
        let mod = modules[i], index = this.modules.indexOf(mod);
        if (index < j && index > -1) { // Ordering conflict
          this.modules.splice(index, 1);
          j--;
          index = -1;
        }
        if (index == -1) {
          this.modules.splice(j++, 0, mod);
          if (sheet) for (let k = 0; k < mod.rules.length; k++)
            sheet.insertRule(mod.rules[k], pos++);
        } else {
          while (j < index) pos += this.modules[j++].rules.length;
          pos += mod.rules.length;
          j++;
        }
      }

      if (!sheet) {
        let text = "";
        for (let i = 0; i < this.modules.length; i++)
          text += this.modules[i].getRules() + "\n";
        this.styleTag.textContent = text;
      }
    }
  }

  // Style::Object<union<Style,string>>
  //
  // A style is an object that, in the simple case, maps CSS property
  // names to strings holding their values, as in `{color: "red",
  // fontWeight: "bold"}`. The property names can be given in
  // camel-case—the library will insert a dash before capital letters
  // when converting them to CSS.
  //
  // If you include an underscore in a property name, it and everything
  // after it will be removed from the output, which can be useful when
  // providing a property multiple times, for browser compatibility
  // reasons.
  //
  // A property in a style object can also be a sub-selector, which
  // extends the current context to add a pseudo-selector or a child
  // selector. Such a property should contain a `&` character, which
  // will be replaced by the current selector. For example `{"&:before":
  // {content: '"hi"'}}`. Sub-selectors and regular properties can
  // freely be mixed in a given object. Any property containing a `&` is
  // assumed to be a sub-selector.
  //
  // Finally, a property can specify an @-block to be wrapped around the
  // styles defined inside the object that's the property's value. For
  // example to create a media query you can do `{"@media screen and
  // (min-width: 400px)": {...}}`.

  var base = {
    8: "Backspace",
    9: "Tab",
    10: "Enter",
    12: "NumLock",
    13: "Enter",
    16: "Shift",
    17: "Control",
    18: "Alt",
    20: "CapsLock",
    27: "Escape",
    32: " ",
    33: "PageUp",
    34: "PageDown",
    35: "End",
    36: "Home",
    37: "ArrowLeft",
    38: "ArrowUp",
    39: "ArrowRight",
    40: "ArrowDown",
    44: "PrintScreen",
    45: "Insert",
    46: "Delete",
    59: ";",
    61: "=",
    91: "Meta",
    92: "Meta",
    106: "*",
    107: "+",
    108: ",",
    109: "-",
    110: ".",
    111: "/",
    144: "NumLock",
    145: "ScrollLock",
    160: "Shift",
    161: "Shift",
    162: "Control",
    163: "Control",
    164: "Alt",
    165: "Alt",
    173: "-",
    186: ";",
    187: "=",
    188: ",",
    189: "-",
    190: ".",
    191: "/",
    192: "`",
    219: "[",
    220: "\\",
    221: "]",
    222: "'"
  };

  var shift = {
    48: ")",
    49: "!",
    50: "@",
    51: "#",
    52: "$",
    53: "%",
    54: "^",
    55: "&",
    56: "*",
    57: "(",
    59: ":",
    61: "+",
    173: "_",
    186: ":",
    187: "+",
    188: "<",
    189: "_",
    190: ">",
    191: "?",
    192: "~",
    219: "{",
    220: "|",
    221: "}",
    222: "\""
  };

  var chrome$1 = typeof navigator != "undefined" && /Chrome\/(\d+)/.exec(navigator.userAgent);
  var mac = typeof navigator != "undefined" && /Mac/.test(navigator.platform);
  mac || chrome$1 && +chrome$1[1] < 57;

  // Fill in the digit keys
  for (var i$2 = 0; i$2 < 10; i$2++) base[48 + i$2] = base[96 + i$2] = String(i$2);

  // The function keys
  for (var i$2 = 1; i$2 <= 24; i$2++) base[i$2 + 111] = "F" + i$2;

  // And the alphabetic keys
  for (var i$2 = 65; i$2 <= 90; i$2++) {
    base[i$2] = String.fromCharCode(i$2 + 32);
    shift[i$2] = String.fromCharCode(i$2);
  }

  // For each code that doesn't have a shift-equivalent, copy the base name
  for (var code in base) if (!shift.hasOwnProperty(code)) shift[code] = base[code];

  function getSelection(root) {
      let target;
      // Browsers differ on whether shadow roots have a getSelection
      // method. If it exists, use that, otherwise, call it on the
      // document.
      if (root.nodeType == 11) { // Shadow root
          target = root.getSelection ? root : root.ownerDocument;
      }
      else {
          target = root;
      }
      return target.getSelection();
  }
  function contains(dom, node) {
      return node ? dom == node || dom.contains(node.nodeType != 1 ? node.parentNode : node) : false;
  }
  function deepActiveElement(doc) {
      let elt = doc.activeElement;
      while (elt && elt.shadowRoot)
          elt = elt.shadowRoot.activeElement;
      return elt;
  }
  function hasSelection(dom, selection) {
      if (!selection.anchorNode)
          return false;
      try {
          // Firefox will raise 'permission denied' errors when accessing
          // properties of `sel.anchorNode` when it's in a generated CSS
          // element.
          return contains(dom, selection.anchorNode);
      }
      catch (_) {
          return false;
      }
  }
  function clientRectsFor(dom) {
      if (dom.nodeType == 3)
          return textRange(dom, 0, dom.nodeValue.length).getClientRects();
      else if (dom.nodeType == 1)
          return dom.getClientRects();
      else
          return [];
  }
  // Scans forward and backward through DOM positions equivalent to the
  // given one to see if the two are in the same place (i.e. after a
  // text node vs at the end of that text node)
  function isEquivalentPosition(node, off, targetNode, targetOff) {
      return targetNode ? (scanFor(node, off, targetNode, targetOff, -1) ||
          scanFor(node, off, targetNode, targetOff, 1)) : false;
  }
  function domIndex(node) {
      for (var index = 0;; index++) {
          node = node.previousSibling;
          if (!node)
              return index;
      }
  }
  function scanFor(node, off, targetNode, targetOff, dir) {
      for (;;) {
          if (node == targetNode && off == targetOff)
              return true;
          if (off == (dir < 0 ? 0 : maxOffset(node))) {
              if (node.nodeName == "DIV")
                  return false;
              let parent = node.parentNode;
              if (!parent || parent.nodeType != 1)
                  return false;
              off = domIndex(node) + (dir < 0 ? 0 : 1);
              node = parent;
          }
          else if (node.nodeType == 1) {
              node = node.childNodes[off + (dir < 0 ? -1 : 0)];
              if (node.nodeType == 1 && node.contentEditable == "false")
                  return false;
              off = dir < 0 ? maxOffset(node) : 0;
          }
          else {
              return false;
          }
      }
  }
  function maxOffset(node) {
      return node.nodeType == 3 ? node.nodeValue.length : node.childNodes.length;
  }
  const Rect0 = { left: 0, right: 0, top: 0, bottom: 0 };
  function flattenRect(rect, left) {
      let x = left ? rect.left : rect.right;
      return { left: x, right: x, top: rect.top, bottom: rect.bottom };
  }
  function windowRect(win) {
      return { left: 0, right: win.innerWidth,
          top: 0, bottom: win.innerHeight };
  }
  function scrollRectIntoView(dom, rect, side, x, y, xMargin, yMargin, ltr) {
      let doc = dom.ownerDocument, win = doc.defaultView || window;
      for (let cur = dom; cur;) {
          if (cur.nodeType == 1) { // Element
              let bounding, top = cur == doc.body;
              if (top) {
                  bounding = windowRect(win);
              }
              else {
                  if (cur.scrollHeight <= cur.clientHeight && cur.scrollWidth <= cur.clientWidth) {
                      cur = cur.assignedSlot || cur.parentNode;
                      continue;
                  }
                  let rect = cur.getBoundingClientRect();
                  // Make sure scrollbar width isn't included in the rectangle
                  bounding = { left: rect.left, right: rect.left + cur.clientWidth,
                      top: rect.top, bottom: rect.top + cur.clientHeight };
              }
              let moveX = 0, moveY = 0;
              if (y == "nearest") {
                  if (rect.top < bounding.top) {
                      moveY = -(bounding.top - rect.top + yMargin);
                      if (side > 0 && rect.bottom > bounding.bottom + moveY)
                          moveY = rect.bottom - bounding.bottom + moveY + yMargin;
                  }
                  else if (rect.bottom > bounding.bottom) {
                      moveY = rect.bottom - bounding.bottom + yMargin;
                      if (side < 0 && (rect.top - moveY) < bounding.top)
                          moveY = -(bounding.top + moveY - rect.top + yMargin);
                  }
              }
              else {
                  let rectHeight = rect.bottom - rect.top, boundingHeight = bounding.bottom - bounding.top;
                  let targetTop = y == "center" && rectHeight <= boundingHeight ? rect.top + rectHeight / 2 - boundingHeight / 2 :
                      y == "start" || y == "center" && side < 0 ? rect.top - yMargin :
                          rect.bottom - boundingHeight + yMargin;
                  moveY = targetTop - bounding.top;
              }
              if (x == "nearest") {
                  if (rect.left < bounding.left) {
                      moveX = -(bounding.left - rect.left + xMargin);
                      if (side > 0 && rect.right > bounding.right + moveX)
                          moveX = rect.right - bounding.right + moveX + xMargin;
                  }
                  else if (rect.right > bounding.right) {
                      moveX = rect.right - bounding.right + xMargin;
                      if (side < 0 && rect.left < bounding.left + moveX)
                          moveX = -(bounding.left + moveX - rect.left + xMargin);
                  }
              }
              else {
                  let targetLeft = x == "center" ? rect.left + (rect.right - rect.left) / 2 - (bounding.right - bounding.left) / 2 :
                      (x == "start") == ltr ? rect.left - xMargin :
                          rect.right - (bounding.right - bounding.left) + xMargin;
                  moveX = targetLeft - bounding.left;
              }
              if (moveX || moveY) {
                  if (top) {
                      win.scrollBy(moveX, moveY);
                  }
                  else {
                      let movedX = 0, movedY = 0;
                      if (moveY) {
                          let start = cur.scrollTop;
                          cur.scrollTop += moveY;
                          movedY = cur.scrollTop - start;
                      }
                      if (moveX) {
                          let start = cur.scrollLeft;
                          cur.scrollLeft += moveX;
                          movedX = cur.scrollLeft - start;
                      }
                      rect = { left: rect.left - movedX, top: rect.top - movedY,
                          right: rect.right - movedX, bottom: rect.bottom - movedY };
                      if (movedX && Math.abs(movedX - moveX) < 1)
                          x = "nearest";
                      if (movedY && Math.abs(movedY - moveY) < 1)
                          y = "nearest";
                  }
              }
              if (top)
                  break;
              cur = cur.assignedSlot || cur.parentNode;
          }
          else if (cur.nodeType == 11) { // A shadow root
              cur = cur.host;
          }
          else {
              break;
          }
      }
  }
  class DOMSelectionState {
      constructor() {
          this.anchorNode = null;
          this.anchorOffset = 0;
          this.focusNode = null;
          this.focusOffset = 0;
      }
      eq(domSel) {
          return this.anchorNode == domSel.anchorNode && this.anchorOffset == domSel.anchorOffset &&
              this.focusNode == domSel.focusNode && this.focusOffset == domSel.focusOffset;
      }
      setRange(range) {
          this.set(range.anchorNode, range.anchorOffset, range.focusNode, range.focusOffset);
      }
      set(anchorNode, anchorOffset, focusNode, focusOffset) {
          this.anchorNode = anchorNode;
          this.anchorOffset = anchorOffset;
          this.focusNode = focusNode;
          this.focusOffset = focusOffset;
      }
  }
  let preventScrollSupported = null;
  // Feature-detects support for .focus({preventScroll: true}), and uses
  // a fallback kludge when not supported.
  function focusPreventScroll(dom) {
      if (dom.setActive)
          return dom.setActive(); // in IE
      if (preventScrollSupported)
          return dom.focus(preventScrollSupported);
      let stack = [];
      for (let cur = dom; cur; cur = cur.parentNode) {
          stack.push(cur, cur.scrollTop, cur.scrollLeft);
          if (cur == cur.ownerDocument)
              break;
      }
      dom.focus(preventScrollSupported == null ? {
          get preventScroll() {
              preventScrollSupported = { preventScroll: true };
              return true;
          }
      } : undefined);
      if (!preventScrollSupported) {
          preventScrollSupported = false;
          for (let i = 0; i < stack.length;) {
              let elt = stack[i++], top = stack[i++], left = stack[i++];
              if (elt.scrollTop != top)
                  elt.scrollTop = top;
              if (elt.scrollLeft != left)
                  elt.scrollLeft = left;
          }
      }
  }
  let scratchRange;
  function textRange(node, from, to = from) {
      let range = scratchRange || (scratchRange = document.createRange());
      range.setEnd(node, to);
      range.setStart(node, from);
      return range;
  }
  function dispatchKey(elt, name, code) {
      let options = { key: name, code: name, keyCode: code, which: code, cancelable: true };
      let down = new KeyboardEvent("keydown", options);
      down.synthetic = true;
      elt.dispatchEvent(down);
      let up = new KeyboardEvent("keyup", options);
      up.synthetic = true;
      elt.dispatchEvent(up);
      return down.defaultPrevented || up.defaultPrevented;
  }
  function getRoot(node) {
      while (node) {
          if (node && (node.nodeType == 9 || node.nodeType == 11 && node.host))
              return node;
          node = node.assignedSlot || node.parentNode;
      }
      return null;
  }
  function clearAttributes(node) {
      while (node.attributes.length)
          node.removeAttributeNode(node.attributes[0]);
  }
  function atElementStart(doc, selection) {
      let node = selection.focusNode, offset = selection.focusOffset;
      if (!node || selection.anchorNode != node || selection.anchorOffset != offset)
          return false;
      for (;;) {
          if (offset) {
              if (node.nodeType != 1)
                  return false;
              let prev = node.childNodes[offset - 1];
              if (prev.contentEditable == "false")
                  offset--;
              else {
                  node = prev;
                  offset = maxOffset(node);
              }
          }
          else if (node == doc) {
              return true;
          }
          else {
              offset = domIndex(node);
              node = node.parentNode;
          }
      }
  }

  class DOMPos {
      constructor(node, offset, precise = true) {
          this.node = node;
          this.offset = offset;
          this.precise = precise;
      }
      static before(dom, precise) { return new DOMPos(dom.parentNode, domIndex(dom), precise); }
      static after(dom, precise) { return new DOMPos(dom.parentNode, domIndex(dom) + 1, precise); }
  }
  const noChildren = [];
  class ContentView {
      constructor() {
          this.parent = null;
          this.dom = null;
          this.dirty = 2 /* Dirty.Node */;
      }
      get editorView() {
          if (!this.parent)
              throw new Error("Accessing view in orphan content view");
          return this.parent.editorView;
      }
      get overrideDOMText() { return null; }
      get posAtStart() {
          return this.parent ? this.parent.posBefore(this) : 0;
      }
      get posAtEnd() {
          return this.posAtStart + this.length;
      }
      posBefore(view) {
          let pos = this.posAtStart;
          for (let child of this.children) {
              if (child == view)
                  return pos;
              pos += child.length + child.breakAfter;
          }
          throw new RangeError("Invalid child in posBefore");
      }
      posAfter(view) {
          return this.posBefore(view) + view.length;
      }
      // Will return a rectangle directly before (when side < 0), after
      // (side > 0) or directly on (when the browser supports it) the
      // given position.
      coordsAt(_pos, _side) { return null; }
      sync(track) {
          if (this.dirty & 2 /* Dirty.Node */) {
              let parent = this.dom;
              let prev = null, next;
              for (let child of this.children) {
                  if (child.dirty) {
                      if (!child.dom && (next = prev ? prev.nextSibling : parent.firstChild)) {
                          let contentView = ContentView.get(next);
                          if (!contentView || !contentView.parent && contentView.canReuseDOM(child))
                              child.reuseDOM(next);
                      }
                      child.sync(track);
                      child.dirty = 0 /* Dirty.Not */;
                  }
                  next = prev ? prev.nextSibling : parent.firstChild;
                  if (track && !track.written && track.node == parent && next != child.dom)
                      track.written = true;
                  if (child.dom.parentNode == parent) {
                      while (next && next != child.dom)
                          next = rm$1(next);
                  }
                  else {
                      parent.insertBefore(child.dom, next);
                  }
                  prev = child.dom;
              }
              next = prev ? prev.nextSibling : parent.firstChild;
              if (next && track && track.node == parent)
                  track.written = true;
              while (next)
                  next = rm$1(next);
          }
          else if (this.dirty & 1 /* Dirty.Child */) {
              for (let child of this.children)
                  if (child.dirty) {
                      child.sync(track);
                      child.dirty = 0 /* Dirty.Not */;
                  }
          }
      }
      reuseDOM(_dom) { }
      localPosFromDOM(node, offset) {
          let after;
          if (node == this.dom) {
              after = this.dom.childNodes[offset];
          }
          else {
              let bias = maxOffset(node) == 0 ? 0 : offset == 0 ? -1 : 1;
              for (;;) {
                  let parent = node.parentNode;
                  if (parent == this.dom)
                      break;
                  if (bias == 0 && parent.firstChild != parent.lastChild) {
                      if (node == parent.firstChild)
                          bias = -1;
                      else
                          bias = 1;
                  }
                  node = parent;
              }
              if (bias < 0)
                  after = node;
              else
                  after = node.nextSibling;
          }
          if (after == this.dom.firstChild)
              return 0;
          while (after && !ContentView.get(after))
              after = after.nextSibling;
          if (!after)
              return this.length;
          for (let i = 0, pos = 0;; i++) {
              let child = this.children[i];
              if (child.dom == after)
                  return pos;
              pos += child.length + child.breakAfter;
          }
      }
      domBoundsAround(from, to, offset = 0) {
          let fromI = -1, fromStart = -1, toI = -1, toEnd = -1;
          for (let i = 0, pos = offset, prevEnd = offset; i < this.children.length; i++) {
              let child = this.children[i], end = pos + child.length;
              if (pos < from && end > to)
                  return child.domBoundsAround(from, to, pos);
              if (end >= from && fromI == -1) {
                  fromI = i;
                  fromStart = pos;
              }
              if (pos > to && child.dom.parentNode == this.dom) {
                  toI = i;
                  toEnd = prevEnd;
                  break;
              }
              prevEnd = end;
              pos = end + child.breakAfter;
          }
          return { from: fromStart, to: toEnd < 0 ? offset + this.length : toEnd,
              startDOM: (fromI ? this.children[fromI - 1].dom.nextSibling : null) || this.dom.firstChild,
              endDOM: toI < this.children.length && toI >= 0 ? this.children[toI].dom : null };
      }
      markDirty(andParent = false) {
          this.dirty |= 2 /* Dirty.Node */;
          this.markParentsDirty(andParent);
      }
      markParentsDirty(childList) {
          for (let parent = this.parent; parent; parent = parent.parent) {
              if (childList)
                  parent.dirty |= 2 /* Dirty.Node */;
              if (parent.dirty & 1 /* Dirty.Child */)
                  return;
              parent.dirty |= 1 /* Dirty.Child */;
              childList = false;
          }
      }
      setParent(parent) {
          if (this.parent != parent) {
              this.parent = parent;
              if (this.dirty)
                  this.markParentsDirty(true);
          }
      }
      setDOM(dom) {
          if (this.dom)
              this.dom.cmView = null;
          this.dom = dom;
          dom.cmView = this;
      }
      get rootView() {
          for (let v = this;;) {
              let parent = v.parent;
              if (!parent)
                  return v;
              v = parent;
          }
      }
      replaceChildren(from, to, children = noChildren) {
          this.markDirty();
          for (let i = from; i < to; i++) {
              let child = this.children[i];
              if (child.parent == this)
                  child.destroy();
          }
          this.children.splice(from, to - from, ...children);
          for (let i = 0; i < children.length; i++)
              children[i].setParent(this);
      }
      ignoreMutation(_rec) { return false; }
      ignoreEvent(_event) { return false; }
      childCursor(pos = this.length) {
          return new ChildCursor(this.children, pos, this.children.length);
      }
      childPos(pos, bias = 1) {
          return this.childCursor().findPos(pos, bias);
      }
      toString() {
          let name = this.constructor.name.replace("View", "");
          return name + (this.children.length ? "(" + this.children.join() + ")" :
              this.length ? "[" + (name == "Text" ? this.text : this.length) + "]" : "") +
              (this.breakAfter ? "#" : "");
      }
      static get(node) { return node.cmView; }
      get isEditable() { return true; }
      merge(from, to, source, hasStart, openStart, openEnd) {
          return false;
      }
      become(other) { return false; }
      canReuseDOM(other) { return other.constructor == this.constructor; }
      // When this is a zero-length view with a side, this should return a
      // number <= 0 to indicate it is before its position, or a
      // number > 0 when after its position.
      getSide() { return 0; }
      destroy() {
          this.parent = null;
      }
  }
  ContentView.prototype.breakAfter = 0;
  // Remove a DOM node and return its next sibling.
  function rm$1(dom) {
      let next = dom.nextSibling;
      dom.parentNode.removeChild(dom);
      return next;
  }
  class ChildCursor {
      constructor(children, pos, i) {
          this.children = children;
          this.pos = pos;
          this.i = i;
          this.off = 0;
      }
      findPos(pos, bias = 1) {
          for (;;) {
              if (pos > this.pos || pos == this.pos &&
                  (bias > 0 || this.i == 0 || this.children[this.i - 1].breakAfter)) {
                  this.off = pos - this.pos;
                  return this;
              }
              let next = this.children[--this.i];
              this.pos -= next.length + next.breakAfter;
          }
      }
  }
  function replaceRange(parent, fromI, fromOff, toI, toOff, insert, breakAtStart, openStart, openEnd) {
      let { children } = parent;
      let before = children.length ? children[fromI] : null;
      let last = insert.length ? insert[insert.length - 1] : null;
      let breakAtEnd = last ? last.breakAfter : breakAtStart;
      // Change within a single child
      if (fromI == toI && before && !breakAtStart && !breakAtEnd && insert.length < 2 &&
          before.merge(fromOff, toOff, insert.length ? last : null, fromOff == 0, openStart, openEnd))
          return;
      if (toI < children.length) {
          let after = children[toI];
          // Make sure the end of the child after the update is preserved in `after`
          if (after && toOff < after.length) {
              // If we're splitting a child, separate part of it to avoid that
              // being mangled when updating the child before the update.
              if (fromI == toI) {
                  after = after.split(toOff);
                  toOff = 0;
              }
              // If the element after the replacement should be merged with
              // the last replacing element, update `content`
              if (!breakAtEnd && last && after.merge(0, toOff, last, true, 0, openEnd)) {
                  insert[insert.length - 1] = after;
              }
              else {
                  // Remove the start of the after element, if necessary, and
                  // add it to `content`.
                  if (toOff)
                      after.merge(0, toOff, null, false, 0, openEnd);
                  insert.push(after);
              }
          }
          else if (after === null || after === void 0 ? void 0 : after.breakAfter) {
              // The element at `toI` is entirely covered by this range.
              // Preserve its line break, if any.
              if (last)
                  last.breakAfter = 1;
              else
                  breakAtStart = 1;
          }
          // Since we've handled the next element from the current elements
          // now, make sure `toI` points after that.
          toI++;
      }
      if (before) {
          before.breakAfter = breakAtStart;
          if (fromOff > 0) {
              if (!breakAtStart && insert.length && before.merge(fromOff, before.length, insert[0], false, openStart, 0)) {
                  before.breakAfter = insert.shift().breakAfter;
              }
              else if (fromOff < before.length || before.children.length && before.children[before.children.length - 1].length == 0) {
                  before.merge(fromOff, before.length, null, false, openStart, 0);
              }
              fromI++;
          }
      }
      // Try to merge widgets on the boundaries of the replacement
      while (fromI < toI && insert.length) {
          if (children[toI - 1].become(insert[insert.length - 1])) {
              toI--;
              insert.pop();
              openEnd = insert.length ? 0 : openStart;
          }
          else if (children[fromI].become(insert[0])) {
              fromI++;
              insert.shift();
              openStart = insert.length ? 0 : openEnd;
          }
          else {
              break;
          }
      }
      if (!insert.length && fromI && toI < children.length && !children[fromI - 1].breakAfter &&
          children[toI].merge(0, 0, children[fromI - 1], false, openStart, openEnd))
          fromI--;
      if (fromI < toI || insert.length)
          parent.replaceChildren(fromI, toI, insert);
  }
  function mergeChildrenInto(parent, from, to, insert, openStart, openEnd) {
      let cur = parent.childCursor();
      let { i: toI, off: toOff } = cur.findPos(to, 1);
      let { i: fromI, off: fromOff } = cur.findPos(from, -1);
      let dLen = from - to;
      for (let view of insert)
          dLen += view.length;
      parent.length += dLen;
      replaceRange(parent, fromI, fromOff, toI, toOff, insert, 0, openStart, openEnd);
  }

  let nav = typeof navigator != "undefined" ? navigator : { userAgent: "", vendor: "", platform: "" };
  let doc = typeof document != "undefined" ? document : { documentElement: { style: {} } };
  const ie_edge = /*@__PURE__*//Edge\/(\d+)/.exec(nav.userAgent);
  const ie_upto10 = /*@__PURE__*//MSIE \d/.test(nav.userAgent);
  const ie_11up = /*@__PURE__*//Trident\/(?:[7-9]|\d{2,})\..*rv:(\d+)/.exec(nav.userAgent);
  const ie = !!(ie_upto10 || ie_11up || ie_edge);
  const gecko = !ie && /*@__PURE__*//gecko\/(\d+)/i.test(nav.userAgent);
  const chrome = !ie && /*@__PURE__*//Chrome\/(\d+)/.exec(nav.userAgent);
  const webkit = "webkitFontSmoothing" in doc.documentElement.style;
  const safari = !ie && /*@__PURE__*//Apple Computer/.test(nav.vendor);
  const ios = safari && (/*@__PURE__*//Mobile\/\w+/.test(nav.userAgent) || nav.maxTouchPoints > 2);
  var browser = {
      mac: ios || /*@__PURE__*//Mac/.test(nav.platform),
      windows: /*@__PURE__*//Win/.test(nav.platform),
      linux: /*@__PURE__*//Linux|X11/.test(nav.platform),
      ie,
      ie_version: ie_upto10 ? doc.documentMode || 6 : ie_11up ? +ie_11up[1] : ie_edge ? +ie_edge[1] : 0,
      gecko,
      gecko_version: gecko ? +(/*@__PURE__*//Firefox\/(\d+)/.exec(nav.userAgent) || [0, 0])[1] : 0,
      chrome: !!chrome,
      chrome_version: chrome ? +chrome[1] : 0,
      ios,
      android: /*@__PURE__*//Android\b/.test(nav.userAgent),
      webkit,
      safari,
      webkit_version: webkit ? +(/*@__PURE__*//\bAppleWebKit\/(\d+)/.exec(navigator.userAgent) || [0, 0])[1] : 0,
      tabSize: doc.documentElement.style.tabSize != null ? "tab-size" : "-moz-tab-size"
  };

  const MaxJoinLen = 256;
  class TextView extends ContentView {
      constructor(text) {
          super();
          this.text = text;
      }
      get length() { return this.text.length; }
      createDOM(textDOM) {
          this.setDOM(textDOM || document.createTextNode(this.text));
      }
      sync(track) {
          if (!this.dom)
              this.createDOM();
          if (this.dom.nodeValue != this.text) {
              if (track && track.node == this.dom)
                  track.written = true;
              this.dom.nodeValue = this.text;
          }
      }
      reuseDOM(dom) {
          if (dom.nodeType == 3)
              this.createDOM(dom);
      }
      merge(from, to, source) {
          if (source && (!(source instanceof TextView) || this.length - (to - from) + source.length > MaxJoinLen))
              return false;
          this.text = this.text.slice(0, from) + (source ? source.text : "") + this.text.slice(to);
          this.markDirty();
          return true;
      }
      split(from) {
          let result = new TextView(this.text.slice(from));
          this.text = this.text.slice(0, from);
          this.markDirty();
          return result;
      }
      localPosFromDOM(node, offset) {
          return node == this.dom ? offset : offset ? this.text.length : 0;
      }
      domAtPos(pos) { return new DOMPos(this.dom, pos); }
      domBoundsAround(_from, _to, offset) {
          return { from: offset, to: offset + this.length, startDOM: this.dom, endDOM: this.dom.nextSibling };
      }
      coordsAt(pos, side) {
          return textCoords(this.dom, pos, side);
      }
  }
  class MarkView extends ContentView {
      constructor(mark, children = [], length = 0) {
          super();
          this.mark = mark;
          this.children = children;
          this.length = length;
          for (let ch of children)
              ch.setParent(this);
      }
      setAttrs(dom) {
          clearAttributes(dom);
          if (this.mark.class)
              dom.className = this.mark.class;
          if (this.mark.attrs)
              for (let name in this.mark.attrs)
                  dom.setAttribute(name, this.mark.attrs[name]);
          return dom;
      }
      reuseDOM(node) {
          if (node.nodeName == this.mark.tagName.toUpperCase()) {
              this.setDOM(node);
              this.dirty |= 4 /* Dirty.Attrs */ | 2 /* Dirty.Node */;
          }
      }
      sync(track) {
          if (!this.dom)
              this.setDOM(this.setAttrs(document.createElement(this.mark.tagName)));
          else if (this.dirty & 4 /* Dirty.Attrs */)
              this.setAttrs(this.dom);
          super.sync(track);
      }
      merge(from, to, source, _hasStart, openStart, openEnd) {
          if (source && (!(source instanceof MarkView && source.mark.eq(this.mark)) ||
              (from && openStart <= 0) || (to < this.length && openEnd <= 0)))
              return false;
          mergeChildrenInto(this, from, to, source ? source.children : [], openStart - 1, openEnd - 1);
          this.markDirty();
          return true;
      }
      split(from) {
          let result = [], off = 0, detachFrom = -1, i = 0;
          for (let elt of this.children) {
              let end = off + elt.length;
              if (end > from)
                  result.push(off < from ? elt.split(from - off) : elt);
              if (detachFrom < 0 && off >= from)
                  detachFrom = i;
              off = end;
              i++;
          }
          let length = this.length - from;
          this.length = from;
          if (detachFrom > -1) {
              this.children.length = detachFrom;
              this.markDirty();
          }
          return new MarkView(this.mark, result, length);
      }
      domAtPos(pos) {
          return inlineDOMAtPos(this, pos);
      }
      coordsAt(pos, side) {
          return coordsInChildren(this, pos, side);
      }
  }
  function textCoords(text, pos, side) {
      let length = text.nodeValue.length;
      if (pos > length)
          pos = length;
      let from = pos, to = pos, flatten = 0;
      if (pos == 0 && side < 0 || pos == length && side >= 0) {
          if (!(browser.chrome || browser.gecko)) { // These browsers reliably return valid rectangles for empty ranges
              if (pos) {
                  from--;
                  flatten = 1;
              } // FIXME this is wrong in RTL text
              else if (to < length) {
                  to++;
                  flatten = -1;
              }
          }
      }
      else {
          if (side < 0)
              from--;
          else if (to < length)
              to++;
      }
      let rects = textRange(text, from, to).getClientRects();
      if (!rects.length)
          return Rect0;
      let rect = rects[(flatten ? flatten < 0 : side >= 0) ? 0 : rects.length - 1];
      if (browser.safari && !flatten && rect.width == 0)
          rect = Array.prototype.find.call(rects, r => r.width) || rect;
      return flatten ? flattenRect(rect, flatten < 0) : rect || null;
  }
  // Also used for collapsed ranges that don't have a placeholder widget!
  class WidgetView extends ContentView {
      constructor(widget, length, side) {
          super();
          this.widget = widget;
          this.length = length;
          this.side = side;
          this.prevWidget = null;
      }
      static create(widget, length, side) {
          return new (widget.customView || WidgetView)(widget, length, side);
      }
      split(from) {
          let result = WidgetView.create(this.widget, this.length - from, this.side);
          this.length -= from;
          return result;
      }
      sync() {
          if (!this.dom || !this.widget.updateDOM(this.dom)) {
              if (this.dom && this.prevWidget)
                  this.prevWidget.destroy(this.dom);
              this.prevWidget = null;
              this.setDOM(this.widget.toDOM(this.editorView));
              this.dom.contentEditable = "false";
          }
      }
      getSide() { return this.side; }
      merge(from, to, source, hasStart, openStart, openEnd) {
          if (source && (!(source instanceof WidgetView) || !this.widget.compare(source.widget) ||
              from > 0 && openStart <= 0 || to < this.length && openEnd <= 0))
              return false;
          this.length = from + (source ? source.length : 0) + (this.length - to);
          return true;
      }
      become(other) {
          if (other.length == this.length && other instanceof WidgetView && other.side == this.side) {
              if (this.widget.constructor == other.widget.constructor) {
                  if (!this.widget.eq(other.widget))
                      this.markDirty(true);
                  if (this.dom && !this.prevWidget)
                      this.prevWidget = this.widget;
                  this.widget = other.widget;
                  return true;
              }
          }
          return false;
      }
      ignoreMutation() { return true; }
      ignoreEvent(event) { return this.widget.ignoreEvent(event); }
      get overrideDOMText() {
          if (this.length == 0)
              return Text$1.empty;
          let top = this;
          while (top.parent)
              top = top.parent;
          let view = top.editorView, text = view && view.state.doc, start = this.posAtStart;
          return text ? text.slice(start, start + this.length) : Text$1.empty;
      }
      domAtPos(pos) {
          return pos == 0 ? DOMPos.before(this.dom) : DOMPos.after(this.dom, pos == this.length);
      }
      domBoundsAround() { return null; }
      coordsAt(pos, side) {
          let rects = this.dom.getClientRects(), rect = null;
          if (!rects.length)
              return Rect0;
          for (let i = pos > 0 ? rects.length - 1 : 0;; i += (pos > 0 ? -1 : 1)) {
              rect = rects[i];
              if (pos > 0 ? i == 0 : i == rects.length - 1 || rect.top < rect.bottom)
                  break;
          }
          return this.length ? rect : flattenRect(rect, this.side > 0);
      }
      get isEditable() { return false; }
      destroy() {
          super.destroy();
          if (this.dom)
              this.widget.destroy(this.dom);
      }
  }
  class CompositionView extends WidgetView {
      domAtPos(pos) {
          let { topView, text } = this.widget;
          if (!topView)
              return new DOMPos(text, Math.min(pos, text.nodeValue.length));
          return scanCompositionTree(pos, 0, topView, text, (v, p) => v.domAtPos(p), p => new DOMPos(text, Math.min(p, text.nodeValue.length)));
      }
      sync() { this.setDOM(this.widget.toDOM()); }
      localPosFromDOM(node, offset) {
          let { topView, text } = this.widget;
          if (!topView)
              return Math.min(offset, this.length);
          return posFromDOMInCompositionTree(node, offset, topView, text);
      }
      ignoreMutation() { return false; }
      get overrideDOMText() { return null; }
      coordsAt(pos, side) {
          let { topView, text } = this.widget;
          if (!topView)
              return textCoords(text, pos, side);
          return scanCompositionTree(pos, side, topView, text, (v, pos, side) => v.coordsAt(pos, side), (pos, side) => textCoords(text, pos, side));
      }
      destroy() {
          var _a;
          super.destroy();
          (_a = this.widget.topView) === null || _a === void 0 ? void 0 : _a.destroy();
      }
      get isEditable() { return true; }
      canReuseDOM() { return true; }
  }
  // Uses the old structure of a chunk of content view frozen for
  // composition to try and find a reasonable DOM location for the given
  // offset.
  function scanCompositionTree(pos, side, view, text, enterView, fromText) {
      if (view instanceof MarkView) {
          for (let child = view.dom.firstChild; child; child = child.nextSibling) {
              let desc = ContentView.get(child);
              if (!desc)
                  return fromText(pos, side);
              let hasComp = contains(child, text);
              let len = desc.length + (hasComp ? text.nodeValue.length : 0);
              if (pos < len || pos == len && desc.getSide() <= 0)
                  return hasComp ? scanCompositionTree(pos, side, desc, text, enterView, fromText) : enterView(desc, pos, side);
              pos -= len;
          }
          return enterView(view, view.length, -1);
      }
      else if (view.dom == text) {
          return fromText(pos, side);
      }
      else {
          return enterView(view, pos, side);
      }
  }
  function posFromDOMInCompositionTree(node, offset, view, text) {
      if (view instanceof MarkView) {
          for (let child of view.children) {
              let pos = 0, hasComp = contains(child.dom, text);
              if (contains(child.dom, node))
                  return pos + (hasComp ? posFromDOMInCompositionTree(node, offset, child, text) : child.localPosFromDOM(node, offset));
              pos += hasComp ? text.nodeValue.length : child.length;
          }
      }
      else if (view.dom == text) {
          return Math.min(offset, text.nodeValue.length);
      }
      return view.localPosFromDOM(node, offset);
  }
  // These are drawn around uneditable widgets to avoid a number of
  // browser bugs that show up when the cursor is directly next to
  // uneditable inline content.
  class WidgetBufferView extends ContentView {
      constructor(side) {
          super();
          this.side = side;
      }
      get length() { return 0; }
      merge() { return false; }
      become(other) {
          return other instanceof WidgetBufferView && other.side == this.side;
      }
      split() { return new WidgetBufferView(this.side); }
      sync() {
          if (!this.dom) {
              let dom = document.createElement("img");
              dom.className = "cm-widgetBuffer";
              dom.setAttribute("aria-hidden", "true");
              this.setDOM(dom);
          }
      }
      getSide() { return this.side; }
      domAtPos(pos) { return DOMPos.before(this.dom); }
      localPosFromDOM() { return 0; }
      domBoundsAround() { return null; }
      coordsAt(pos) {
          let imgRect = this.dom.getBoundingClientRect();
          // Since the <img> height doesn't correspond to text height, try
          // to borrow the height from some sibling node.
          let siblingRect = inlineSiblingRect(this, this.side > 0 ? -1 : 1);
          return siblingRect && siblingRect.top < imgRect.bottom && siblingRect.bottom > imgRect.top
              ? { left: imgRect.left, right: imgRect.right, top: siblingRect.top, bottom: siblingRect.bottom } : imgRect;
      }
      get overrideDOMText() {
          return Text$1.empty;
      }
  }
  TextView.prototype.children = WidgetView.prototype.children = WidgetBufferView.prototype.children = noChildren;
  function inlineSiblingRect(view, side) {
      let parent = view.parent, index = parent ? parent.children.indexOf(view) : -1;
      while (parent && index >= 0) {
          if (side < 0 ? index > 0 : index < parent.children.length) {
              let next = parent.children[index + side];
              if (next instanceof TextView) {
                  let nextRect = next.coordsAt(side < 0 ? next.length : 0, side);
                  if (nextRect)
                      return nextRect;
              }
              index += side;
          }
          else if (parent instanceof MarkView && parent.parent) {
              index = parent.parent.children.indexOf(parent) + (side < 0 ? 0 : 1);
              parent = parent.parent;
          }
          else {
              let last = parent.dom.lastChild;
              if (last && last.nodeName == "BR")
                  return last.getClientRects()[0];
              break;
          }
      }
      return undefined;
  }
  function inlineDOMAtPos(parent, pos) {
      let dom = parent.dom, { children } = parent, i = 0;
      for (let off = 0; i < children.length; i++) {
          let child = children[i], end = off + child.length;
          if (end == off && child.getSide() <= 0)
              continue;
          if (pos > off && pos < end && child.dom.parentNode == dom)
              return child.domAtPos(pos - off);
          if (pos <= off)
              break;
          off = end;
      }
      for (let j = i; j > 0; j--) {
          let prev = children[j - 1];
          if (prev.dom.parentNode == dom)
              return prev.domAtPos(prev.length);
      }
      for (let j = i; j < children.length; j++) {
          let next = children[j];
          if (next.dom.parentNode == dom)
              return next.domAtPos(0);
      }
      return new DOMPos(dom, 0);
  }
  // Assumes `view`, if a mark view, has precisely 1 child.
  function joinInlineInto(parent, view, open) {
      let last, { children } = parent;
      if (open > 0 && view instanceof MarkView && children.length &&
          (last = children[children.length - 1]) instanceof MarkView && last.mark.eq(view.mark)) {
          joinInlineInto(last, view.children[0], open - 1);
      }
      else {
          children.push(view);
          view.setParent(parent);
      }
      parent.length += view.length;
  }
  function coordsInChildren(view, pos, side) {
      let before = null, beforePos = -1, after = null, afterPos = -1;
      function scan(view, pos) {
          for (let i = 0, off = 0; i < view.children.length && off <= pos; i++) {
              let child = view.children[i], end = off + child.length;
              if (end >= pos) {
                  if (child.children.length) {
                      scan(child, pos - off);
                  }
                  else if (!after && (end > pos || off == end && child.getSide() > 0)) {
                      after = child;
                      afterPos = pos - off;
                  }
                  else if (off < pos || (off == end && child.getSide() < 0)) {
                      before = child;
                      beforePos = pos - off;
                  }
              }
              off = end;
          }
      }
      scan(view, pos);
      let target = (side < 0 ? before : after) || before || after;
      if (target)
          return target.coordsAt(Math.max(0, target == before ? beforePos : afterPos), side);
      return fallbackRect(view);
  }
  function fallbackRect(view) {
      let last = view.dom.lastChild;
      if (!last)
          return view.dom.getBoundingClientRect();
      let rects = clientRectsFor(last);
      return rects[rects.length - 1] || null;
  }

  function combineAttrs(source, target) {
      for (let name in source) {
          if (name == "class" && target.class)
              target.class += " " + source.class;
          else if (name == "style" && target.style)
              target.style += ";" + source.style;
          else
              target[name] = source[name];
      }
      return target;
  }
  function attrsEq(a, b) {
      if (a == b)
          return true;
      if (!a || !b)
          return false;
      let keysA = Object.keys(a), keysB = Object.keys(b);
      if (keysA.length != keysB.length)
          return false;
      for (let key of keysA) {
          if (keysB.indexOf(key) == -1 || a[key] !== b[key])
              return false;
      }
      return true;
  }
  function updateAttrs(dom, prev, attrs) {
      let changed = null;
      if (prev)
          for (let name in prev)
              if (!(attrs && name in attrs))
                  dom.removeAttribute(changed = name);
      if (attrs)
          for (let name in attrs)
              if (!(prev && prev[name] == attrs[name]))
                  dom.setAttribute(changed = name, attrs[name]);
      return !!changed;
  }

  /**
  Widgets added to the content are described by subclasses of this
  class. Using a description object like that makes it possible to
  delay creating of the DOM structure for a widget until it is
  needed, and to avoid redrawing widgets even if the decorations
  that define them are recreated.
  */
  class WidgetType {
      /**
      Compare this instance to another instance of the same type.
      (TypeScript can't express this, but only instances of the same
      specific class will be passed to this method.) This is used to
      avoid redrawing widgets when they are replaced by a new
      decoration of the same type. The default implementation just
      returns `false`, which will cause new instances of the widget to
      always be redrawn.
      */
      eq(widget) { return false; }
      /**
      Update a DOM element created by a widget of the same type (but
      different, non-`eq` content) to reflect this widget. May return
      true to indicate that it could update, false to indicate it
      couldn't (in which case the widget will be redrawn). The default
      implementation just returns false.
      */
      updateDOM(dom) { return false; }
      /**
      @internal
      */
      compare(other) {
          return this == other || this.constructor == other.constructor && this.eq(other);
      }
      /**
      The estimated height this widget will have, to be used when
      estimating the height of content that hasn't been drawn. May
      return -1 to indicate you don't know. The default implementation
      returns -1.
      */
      get estimatedHeight() { return -1; }
      /**
      Can be used to configure which kinds of events inside the widget
      should be ignored by the editor. The default is to ignore all
      events.
      */
      ignoreEvent(event) { return true; }
      /**
      @internal
      */
      get customView() { return null; }
      /**
      This is called when the an instance of the widget is removed
      from the editor view.
      */
      destroy(dom) { }
  }
  /**
  The different types of blocks that can occur in an editor view.
  */
  var BlockType = /*@__PURE__*/(function (BlockType) {
      /**
      A line of text.
      */
      BlockType[BlockType["Text"] = 0] = "Text";
      /**
      A block widget associated with the position after it.
      */
      BlockType[BlockType["WidgetBefore"] = 1] = "WidgetBefore";
      /**
      A block widget associated with the position before it.
      */
      BlockType[BlockType["WidgetAfter"] = 2] = "WidgetAfter";
      /**
      A block widget [replacing](https://codemirror.net/6/docs/ref/#view.Decoration^replace) a range of content.
      */
      BlockType[BlockType["WidgetRange"] = 3] = "WidgetRange";
  return BlockType})(BlockType || (BlockType = {}));
  /**
  A decoration provides information on how to draw or style a piece
  of content. You'll usually use it wrapped in a
  [`Range`](https://codemirror.net/6/docs/ref/#state.Range), which adds a start and end position.
  @nonabstract
  */
  class Decoration extends RangeValue {
      constructor(
      /**
      @internal
      */
      startSide, 
      /**
      @internal
      */
      endSide, 
      /**
      @internal
      */
      widget, 
      /**
      The config object used to create this decoration. You can
      include additional properties in there to store metadata about
      your decoration.
      */
      spec) {
          super();
          this.startSide = startSide;
          this.endSide = endSide;
          this.widget = widget;
          this.spec = spec;
      }
      /**
      @internal
      */
      get heightRelevant() { return false; }
      /**
      Create a mark decoration, which influences the styling of the
      content in its range. Nested mark decorations will cause nested
      DOM elements to be created. Nesting order is determined by
      precedence of the [facet](https://codemirror.net/6/docs/ref/#view.EditorView^decorations), with
      the higher-precedence decorations creating the inner DOM nodes.
      Such elements are split on line boundaries and on the boundaries
      of lower-precedence decorations.
      */
      static mark(spec) {
          return new MarkDecoration(spec);
      }
      /**
      Create a widget decoration, which displays a DOM element at the
      given position.
      */
      static widget(spec) {
          let side = spec.side || 0, block = !!spec.block;
          side += block ? (side > 0 ? 300000000 /* Side.BlockAfter */ : -400000000 /* Side.BlockBefore */) : (side > 0 ? 100000000 /* Side.InlineAfter */ : -100000000 /* Side.InlineBefore */);
          return new PointDecoration(spec, side, side, block, spec.widget || null, false);
      }
      /**
      Create a replace decoration which replaces the given range with
      a widget, or simply hides it.
      */
      static replace(spec) {
          let block = !!spec.block, startSide, endSide;
          if (spec.isBlockGap) {
              startSide = -500000000 /* Side.GapStart */;
              endSide = 400000000 /* Side.GapEnd */;
          }
          else {
              let { start, end } = getInclusive(spec, block);
              startSide = (start ? (block ? -300000000 /* Side.BlockIncStart */ : -1 /* Side.InlineIncStart */) : 500000000 /* Side.NonIncStart */) - 1;
              endSide = (end ? (block ? 200000000 /* Side.BlockIncEnd */ : 1 /* Side.InlineIncEnd */) : -600000000 /* Side.NonIncEnd */) + 1;
          }
          return new PointDecoration(spec, startSide, endSide, block, spec.widget || null, true);
      }
      /**
      Create a line decoration, which can add DOM attributes to the
      line starting at the given position.
      */
      static line(spec) {
          return new LineDecoration(spec);
      }
      /**
      Build a [`DecorationSet`](https://codemirror.net/6/docs/ref/#view.DecorationSet) from the given
      decorated range or ranges. If the ranges aren't already sorted,
      pass `true` for `sort` to make the library sort them for you.
      */
      static set(of, sort = false) {
          return RangeSet.of(of, sort);
      }
      /**
      @internal
      */
      hasHeight() { return this.widget ? this.widget.estimatedHeight > -1 : false; }
  }
  /**
  The empty set of decorations.
  */
  Decoration.none = RangeSet.empty;
  class MarkDecoration extends Decoration {
      constructor(spec) {
          let { start, end } = getInclusive(spec);
          super(start ? -1 /* Side.InlineIncStart */ : 500000000 /* Side.NonIncStart */, end ? 1 /* Side.InlineIncEnd */ : -600000000 /* Side.NonIncEnd */, null, spec);
          this.tagName = spec.tagName || "span";
          this.class = spec.class || "";
          this.attrs = spec.attributes || null;
      }
      eq(other) {
          return this == other ||
              other instanceof MarkDecoration &&
                  this.tagName == other.tagName &&
                  this.class == other.class &&
                  attrsEq(this.attrs, other.attrs);
      }
      range(from, to = from) {
          if (from >= to)
              throw new RangeError("Mark decorations may not be empty");
          return super.range(from, to);
      }
  }
  MarkDecoration.prototype.point = false;
  class LineDecoration extends Decoration {
      constructor(spec) {
          super(-200000000 /* Side.Line */, -200000000 /* Side.Line */, null, spec);
      }
      eq(other) {
          return other instanceof LineDecoration && attrsEq(this.spec.attributes, other.spec.attributes);
      }
      range(from, to = from) {
          if (to != from)
              throw new RangeError("Line decoration ranges must be zero-length");
          return super.range(from, to);
      }
  }
  LineDecoration.prototype.mapMode = MapMode.TrackBefore;
  LineDecoration.prototype.point = true;
  class PointDecoration extends Decoration {
      constructor(spec, startSide, endSide, block, widget, isReplace) {
          super(startSide, endSide, widget, spec);
          this.block = block;
          this.isReplace = isReplace;
          this.mapMode = !block ? MapMode.TrackDel : startSide <= 0 ? MapMode.TrackBefore : MapMode.TrackAfter;
      }
      // Only relevant when this.block == true
      get type() {
          return this.startSide < this.endSide ? BlockType.WidgetRange
              : this.startSide <= 0 ? BlockType.WidgetBefore : BlockType.WidgetAfter;
      }
      get heightRelevant() { return this.block || !!this.widget && this.widget.estimatedHeight >= 5; }
      eq(other) {
          return other instanceof PointDecoration &&
              widgetsEq(this.widget, other.widget) &&
              this.block == other.block &&
              this.startSide == other.startSide && this.endSide == other.endSide;
      }
      range(from, to = from) {
          if (this.isReplace && (from > to || (from == to && this.startSide > 0 && this.endSide <= 0)))
              throw new RangeError("Invalid range for replacement decoration");
          if (!this.isReplace && to != from)
              throw new RangeError("Widget decorations can only have zero-length ranges");
          return super.range(from, to);
      }
  }
  PointDecoration.prototype.point = true;
  function getInclusive(spec, block = false) {
      let { inclusiveStart: start, inclusiveEnd: end } = spec;
      if (start == null)
          start = spec.inclusive;
      if (end == null)
          end = spec.inclusive;
      return { start: start !== null && start !== void 0 ? start : block, end: end !== null && end !== void 0 ? end : block };
  }
  function widgetsEq(a, b) {
      return a == b || !!(a && b && a.compare(b));
  }
  function addRange(from, to, ranges, margin = 0) {
      let last = ranges.length - 1;
      if (last >= 0 && ranges[last] + margin >= from)
          ranges[last] = Math.max(ranges[last], to);
      else
          ranges.push(from, to);
  }

  class LineView extends ContentView {
      constructor() {
          super(...arguments);
          this.children = [];
          this.length = 0;
          this.prevAttrs = undefined;
          this.attrs = null;
          this.breakAfter = 0;
      }
      // Consumes source
      merge(from, to, source, hasStart, openStart, openEnd) {
          if (source) {
              if (!(source instanceof LineView))
                  return false;
              if (!this.dom)
                  source.transferDOM(this); // Reuse source.dom when appropriate
          }
          if (hasStart)
              this.setDeco(source ? source.attrs : null);
          mergeChildrenInto(this, from, to, source ? source.children : [], openStart, openEnd);
          return true;
      }
      split(at) {
          let end = new LineView;
          end.breakAfter = this.breakAfter;
          if (this.length == 0)
              return end;
          let { i, off } = this.childPos(at);
          if (off) {
              end.append(this.children[i].split(off), 0);
              this.children[i].merge(off, this.children[i].length, null, false, 0, 0);
              i++;
          }
          for (let j = i; j < this.children.length; j++)
              end.append(this.children[j], 0);
          while (i > 0 && this.children[i - 1].length == 0)
              this.children[--i].destroy();
          this.children.length = i;
          this.markDirty();
          this.length = at;
          return end;
      }
      transferDOM(other) {
          if (!this.dom)
              return;
          this.markDirty();
          other.setDOM(this.dom);
          other.prevAttrs = this.prevAttrs === undefined ? this.attrs : this.prevAttrs;
          this.prevAttrs = undefined;
          this.dom = null;
      }
      setDeco(attrs) {
          if (!attrsEq(this.attrs, attrs)) {
              if (this.dom) {
                  this.prevAttrs = this.attrs;
                  this.markDirty();
              }
              this.attrs = attrs;
          }
      }
      append(child, openStart) {
          joinInlineInto(this, child, openStart);
      }
      // Only called when building a line view in ContentBuilder
      addLineDeco(deco) {
          let attrs = deco.spec.attributes, cls = deco.spec.class;
          if (attrs)
              this.attrs = combineAttrs(attrs, this.attrs || {});
          if (cls)
              this.attrs = combineAttrs({ class: cls }, this.attrs || {});
      }
      domAtPos(pos) {
          return inlineDOMAtPos(this, pos);
      }
      reuseDOM(node) {
          if (node.nodeName == "DIV") {
              this.setDOM(node);
              this.dirty |= 4 /* Dirty.Attrs */ | 2 /* Dirty.Node */;
          }
      }
      sync(track) {
          var _a;
          if (!this.dom) {
              this.setDOM(document.createElement("div"));
              this.dom.className = "cm-line";
              this.prevAttrs = this.attrs ? null : undefined;
          }
          else if (this.dirty & 4 /* Dirty.Attrs */) {
              clearAttributes(this.dom);
              this.dom.className = "cm-line";
              this.prevAttrs = this.attrs ? null : undefined;
          }
          if (this.prevAttrs !== undefined) {
              updateAttrs(this.dom, this.prevAttrs, this.attrs);
              this.dom.classList.add("cm-line");
              this.prevAttrs = undefined;
          }
          super.sync(track);
          let last = this.dom.lastChild;
          while (last && ContentView.get(last) instanceof MarkView)
              last = last.lastChild;
          if (!last || !this.length ||
              last.nodeName != "BR" && ((_a = ContentView.get(last)) === null || _a === void 0 ? void 0 : _a.isEditable) == false &&
                  (!browser.ios || !this.children.some(ch => ch instanceof TextView))) {
              let hack = document.createElement("BR");
              hack.cmIgnore = true;
              this.dom.appendChild(hack);
          }
      }
      measureTextSize() {
          if (this.children.length == 0 || this.length > 20)
              return null;
          let totalWidth = 0;
          for (let child of this.children) {
              if (!(child instanceof TextView) || /[^ -~]/.test(child.text))
                  return null;
              let rects = clientRectsFor(child.dom);
              if (rects.length != 1)
                  return null;
              totalWidth += rects[0].width;
          }
          return !totalWidth ? null : {
              lineHeight: this.dom.getBoundingClientRect().height,
              charWidth: totalWidth / this.length
          };
      }
      coordsAt(pos, side) {
          return coordsInChildren(this, pos, side);
      }
      become(_other) { return false; }
      get type() { return BlockType.Text; }
      static find(docView, pos) {
          for (let i = 0, off = 0; i < docView.children.length; i++) {
              let block = docView.children[i], end = off + block.length;
              if (end >= pos) {
                  if (block instanceof LineView)
                      return block;
                  if (end > pos)
                      break;
              }
              off = end + block.breakAfter;
          }
          return null;
      }
  }
  class BlockWidgetView extends ContentView {
      constructor(widget, length, type) {
          super();
          this.widget = widget;
          this.length = length;
          this.type = type;
          this.breakAfter = 0;
          this.prevWidget = null;
      }
      merge(from, to, source, _takeDeco, openStart, openEnd) {
          if (source && (!(source instanceof BlockWidgetView) || !this.widget.compare(source.widget) ||
              from > 0 && openStart <= 0 || to < this.length && openEnd <= 0))
              return false;
          this.length = from + (source ? source.length : 0) + (this.length - to);
          return true;
      }
      domAtPos(pos) {
          return pos == 0 ? DOMPos.before(this.dom) : DOMPos.after(this.dom, pos == this.length);
      }
      split(at) {
          let len = this.length - at;
          this.length = at;
          let end = new BlockWidgetView(this.widget, len, this.type);
          end.breakAfter = this.breakAfter;
          return end;
      }
      get children() { return noChildren; }
      sync() {
          if (!this.dom || !this.widget.updateDOM(this.dom)) {
              if (this.dom && this.prevWidget)
                  this.prevWidget.destroy(this.dom);
              this.prevWidget = null;
              this.setDOM(this.widget.toDOM(this.editorView));
              this.dom.contentEditable = "false";
          }
      }
      get overrideDOMText() {
          return this.parent ? this.parent.view.state.doc.slice(this.posAtStart, this.posAtEnd) : Text$1.empty;
      }
      domBoundsAround() { return null; }
      become(other) {
          if (other instanceof BlockWidgetView && other.type == this.type &&
              other.widget.constructor == this.widget.constructor) {
              if (!other.widget.eq(this.widget))
                  this.markDirty(true);
              if (this.dom && !this.prevWidget)
                  this.prevWidget = this.widget;
              this.widget = other.widget;
              this.length = other.length;
              this.breakAfter = other.breakAfter;
              return true;
          }
          return false;
      }
      ignoreMutation() { return true; }
      ignoreEvent(event) { return this.widget.ignoreEvent(event); }
      destroy() {
          super.destroy();
          if (this.dom)
              this.widget.destroy(this.dom);
      }
  }

  class ContentBuilder {
      constructor(doc, pos, end, disallowBlockEffectsFor) {
          this.doc = doc;
          this.pos = pos;
          this.end = end;
          this.disallowBlockEffectsFor = disallowBlockEffectsFor;
          this.content = [];
          this.curLine = null;
          this.breakAtStart = 0;
          this.pendingBuffer = 0 /* Buf.No */;
          this.bufferMarks = [];
          // Set to false directly after a widget that covers the position after it
          this.atCursorPos = true;
          this.openStart = -1;
          this.openEnd = -1;
          this.text = "";
          this.textOff = 0;
          this.cursor = doc.iter();
          this.skip = pos;
      }
      posCovered() {
          if (this.content.length == 0)
              return !this.breakAtStart && this.doc.lineAt(this.pos).from != this.pos;
          let last = this.content[this.content.length - 1];
          return !last.breakAfter && !(last instanceof BlockWidgetView && last.type == BlockType.WidgetBefore);
      }
      getLine() {
          if (!this.curLine) {
              this.content.push(this.curLine = new LineView);
              this.atCursorPos = true;
          }
          return this.curLine;
      }
      flushBuffer(active = this.bufferMarks) {
          if (this.pendingBuffer) {
              this.curLine.append(wrapMarks(new WidgetBufferView(-1), active), active.length);
              this.pendingBuffer = 0 /* Buf.No */;
          }
      }
      addBlockWidget(view) {
          this.flushBuffer();
          this.curLine = null;
          this.content.push(view);
      }
      finish(openEnd) {
          if (this.pendingBuffer && openEnd <= this.bufferMarks.length)
              this.flushBuffer();
          else
              this.pendingBuffer = 0 /* Buf.No */;
          if (!this.posCovered())
              this.getLine();
      }
      buildText(length, active, openStart) {
          while (length > 0) {
              if (this.textOff == this.text.length) {
                  let { value, lineBreak, done } = this.cursor.next(this.skip);
                  this.skip = 0;
                  if (done)
                      throw new Error("Ran out of text content when drawing inline views");
                  if (lineBreak) {
                      if (!this.posCovered())
                          this.getLine();
                      if (this.content.length)
                          this.content[this.content.length - 1].breakAfter = 1;
                      else
                          this.breakAtStart = 1;
                      this.flushBuffer();
                      this.curLine = null;
                      this.atCursorPos = true;
                      length--;
                      continue;
                  }
                  else {
                      this.text = value;
                      this.textOff = 0;
                  }
              }
              let take = Math.min(this.text.length - this.textOff, length, 512 /* T.Chunk */);
              this.flushBuffer(active.slice(active.length - openStart));
              this.getLine().append(wrapMarks(new TextView(this.text.slice(this.textOff, this.textOff + take)), active), openStart);
              this.atCursorPos = true;
              this.textOff += take;
              length -= take;
              openStart = 0;
          }
      }
      span(from, to, active, openStart) {
          this.buildText(to - from, active, openStart);
          this.pos = to;
          if (this.openStart < 0)
              this.openStart = openStart;
      }
      point(from, to, deco, active, openStart, index) {
          if (this.disallowBlockEffectsFor[index] && deco instanceof PointDecoration) {
              if (deco.block)
                  throw new RangeError("Block decorations may not be specified via plugins");
              if (to > this.doc.lineAt(this.pos).to)
                  throw new RangeError("Decorations that replace line breaks may not be specified via plugins");
          }
          let len = to - from;
          if (deco instanceof PointDecoration) {
              if (deco.block) {
                  let { type } = deco;
                  if (type == BlockType.WidgetAfter && !this.posCovered())
                      this.getLine();
                  this.addBlockWidget(new BlockWidgetView(deco.widget || new NullWidget("div"), len, type));
              }
              else {
                  let view = WidgetView.create(deco.widget || new NullWidget("span"), len, len ? 0 : deco.startSide);
                  let cursorBefore = this.atCursorPos && !view.isEditable && openStart <= active.length && (from < to || deco.startSide > 0);
                  let cursorAfter = !view.isEditable && (from < to || openStart > active.length || deco.startSide <= 0);
                  let line = this.getLine();
                  if (this.pendingBuffer == 2 /* Buf.IfCursor */ && !cursorBefore)
                      this.pendingBuffer = 0 /* Buf.No */;
                  this.flushBuffer(active);
                  if (cursorBefore) {
                      line.append(wrapMarks(new WidgetBufferView(1), active), openStart);
                      openStart = active.length + Math.max(0, openStart - active.length);
                  }
                  line.append(wrapMarks(view, active), openStart);
                  this.atCursorPos = cursorAfter;
                  this.pendingBuffer = !cursorAfter ? 0 /* Buf.No */ : from < to || openStart > active.length ? 1 /* Buf.Yes */ : 2 /* Buf.IfCursor */;
                  if (this.pendingBuffer)
                      this.bufferMarks = active.slice();
              }
          }
          else if (this.doc.lineAt(this.pos).from == this.pos) { // Line decoration
              this.getLine().addLineDeco(deco);
          }
          if (len) {
              // Advance the iterator past the replaced content
              if (this.textOff + len <= this.text.length) {
                  this.textOff += len;
              }
              else {
                  this.skip += len - (this.text.length - this.textOff);
                  this.text = "";
                  this.textOff = 0;
              }
              this.pos = to;
          }
          if (this.openStart < 0)
              this.openStart = openStart;
      }
      static build(text, from, to, decorations, dynamicDecorationMap) {
          let builder = new ContentBuilder(text, from, to, dynamicDecorationMap);
          builder.openEnd = RangeSet.spans(decorations, from, to, builder);
          if (builder.openStart < 0)
              builder.openStart = builder.openEnd;
          builder.finish(builder.openEnd);
          return builder;
      }
  }
  function wrapMarks(view, active) {
      for (let mark of active)
          view = new MarkView(mark, [view], view.length);
      return view;
  }
  class NullWidget extends WidgetType {
      constructor(tag) {
          super();
          this.tag = tag;
      }
      eq(other) { return other.tag == this.tag; }
      toDOM() { return document.createElement(this.tag); }
      updateDOM(elt) { return elt.nodeName.toLowerCase() == this.tag; }
  }

  const clickAddsSelectionRange = /*@__PURE__*/Facet.define();
  const dragMovesSelection$1 = /*@__PURE__*/Facet.define();
  const mouseSelectionStyle = /*@__PURE__*/Facet.define();
  const exceptionSink = /*@__PURE__*/Facet.define();
  const updateListener = /*@__PURE__*/Facet.define();
  const inputHandler = /*@__PURE__*/Facet.define();
  const perLineTextDirection = /*@__PURE__*/Facet.define({
      combine: values => values.some(x => x)
  });
  const nativeSelectionHidden = /*@__PURE__*/Facet.define({
      combine: values => values.some(x => x)
  });
  class ScrollTarget {
      constructor(range, y = "nearest", x = "nearest", yMargin = 5, xMargin = 5) {
          this.range = range;
          this.y = y;
          this.x = x;
          this.yMargin = yMargin;
          this.xMargin = xMargin;
      }
      map(changes) {
          return changes.empty ? this : new ScrollTarget(this.range.map(changes), this.y, this.x, this.yMargin, this.xMargin);
      }
  }
  const scrollIntoView = /*@__PURE__*/StateEffect.define({ map: (t, ch) => t.map(ch) });
  /**
  Log or report an unhandled exception in client code. Should
  probably only be used by extension code that allows client code to
  provide functions, and calls those functions in a context where an
  exception can't be propagated to calling code in a reasonable way
  (for example when in an event handler).

  Either calls a handler registered with
  [`EditorView.exceptionSink`](https://codemirror.net/6/docs/ref/#view.EditorView^exceptionSink),
  `window.onerror`, if defined, or `console.error` (in which case
  it'll pass `context`, when given, as first argument).
  */
  function logException(state, exception, context) {
      let handler = state.facet(exceptionSink);
      if (handler.length)
          handler[0](exception);
      else if (window.onerror)
          window.onerror(String(exception), context, undefined, undefined, exception);
      else if (context)
          console.error(context + ":", exception);
      else
          console.error(exception);
  }
  const editable = /*@__PURE__*/Facet.define({ combine: values => values.length ? values[0] : true });
  let nextPluginID = 0;
  const viewPlugin = /*@__PURE__*/Facet.define();
  /**
  View plugins associate stateful values with a view. They can
  influence the way the content is drawn, and are notified of things
  that happen in the view.
  */
  class ViewPlugin {
      constructor(
      /**
      @internal
      */
      id, 
      /**
      @internal
      */
      create, 
      /**
      @internal
      */
      domEventHandlers, buildExtensions) {
          this.id = id;
          this.create = create;
          this.domEventHandlers = domEventHandlers;
          this.extension = buildExtensions(this);
      }
      /**
      Define a plugin from a constructor function that creates the
      plugin's value, given an editor view.
      */
      static define(create, spec) {
          const { eventHandlers, provide, decorations: deco } = spec || {};
          return new ViewPlugin(nextPluginID++, create, eventHandlers, plugin => {
              let ext = [viewPlugin.of(plugin)];
              if (deco)
                  ext.push(decorations.of(view => {
                      let pluginInst = view.plugin(plugin);
                      return pluginInst ? deco(pluginInst) : Decoration.none;
                  }));
              if (provide)
                  ext.push(provide(plugin));
              return ext;
          });
      }
      /**
      Create a plugin for a class whose constructor takes a single
      editor view as argument.
      */
      static fromClass(cls, spec) {
          return ViewPlugin.define(view => new cls(view), spec);
      }
  }
  class PluginInstance {
      constructor(spec) {
          this.spec = spec;
          // When starting an update, all plugins have this field set to the
          // update object, indicating they need to be updated. When finished
          // updating, it is set to `false`. Retrieving a plugin that needs to
          // be updated with `view.plugin` forces an eager update.
          this.mustUpdate = null;
          // This is null when the plugin is initially created, but
          // initialized on the first update.
          this.value = null;
      }
      update(view) {
          if (!this.value) {
              if (this.spec) {
                  try {
                      this.value = this.spec.create(view);
                  }
                  catch (e) {
                      logException(view.state, e, "CodeMirror plugin crashed");
                      this.deactivate();
                  }
              }
          }
          else if (this.mustUpdate) {
              let update = this.mustUpdate;
              this.mustUpdate = null;
              if (this.value.update) {
                  try {
                      this.value.update(update);
                  }
                  catch (e) {
                      logException(update.state, e, "CodeMirror plugin crashed");
                      if (this.value.destroy)
                          try {
                              this.value.destroy();
                          }
                          catch (_) { }
                      this.deactivate();
                  }
              }
          }
          return this;
      }
      destroy(view) {
          var _a;
          if ((_a = this.value) === null || _a === void 0 ? void 0 : _a.destroy) {
              try {
                  this.value.destroy();
              }
              catch (e) {
                  logException(view.state, e, "CodeMirror plugin crashed");
              }
          }
      }
      deactivate() {
          this.spec = this.value = null;
      }
  }
  const editorAttributes = /*@__PURE__*/Facet.define();
  const contentAttributes = /*@__PURE__*/Facet.define();
  // Provide decorations
  const decorations = /*@__PURE__*/Facet.define();
  const atomicRanges = /*@__PURE__*/Facet.define();
  const scrollMargins = /*@__PURE__*/Facet.define();
  const styleModule = /*@__PURE__*/Facet.define();
  class ChangedRange {
      constructor(fromA, toA, fromB, toB) {
          this.fromA = fromA;
          this.toA = toA;
          this.fromB = fromB;
          this.toB = toB;
      }
      join(other) {
          return new ChangedRange(Math.min(this.fromA, other.fromA), Math.max(this.toA, other.toA), Math.min(this.fromB, other.fromB), Math.max(this.toB, other.toB));
      }
      addToSet(set) {
          let i = set.length, me = this;
          for (; i > 0; i--) {
              let range = set[i - 1];
              if (range.fromA > me.toA)
                  continue;
              if (range.toA < me.fromA)
                  break;
              me = me.join(range);
              set.splice(i - 1, 1);
          }
          set.splice(i, 0, me);
          return set;
      }
      static extendWithRanges(diff, ranges) {
          if (ranges.length == 0)
              return diff;
          let result = [];
          for (let dI = 0, rI = 0, posA = 0, posB = 0;; dI++) {
              let next = dI == diff.length ? null : diff[dI], off = posA - posB;
              let end = next ? next.fromB : 1e9;
              while (rI < ranges.length && ranges[rI] < end) {
                  let from = ranges[rI], to = ranges[rI + 1];
                  let fromB = Math.max(posB, from), toB = Math.min(end, to);
                  if (fromB <= toB)
                      new ChangedRange(fromB + off, toB + off, fromB, toB).addToSet(result);
                  if (to > end)
                      break;
                  else
                      rI += 2;
              }
              if (!next)
                  return result;
              new ChangedRange(next.fromA, next.toA, next.fromB, next.toB).addToSet(result);
              posA = next.toA;
              posB = next.toB;
          }
      }
  }
  /**
  View [plugins](https://codemirror.net/6/docs/ref/#view.ViewPlugin) are given instances of this
  class, which describe what happened, whenever the view is updated.
  */
  class ViewUpdate {
      constructor(
      /**
      The editor view that the update is associated with.
      */
      view, 
      /**
      The new editor state.
      */
      state, 
      /**
      The transactions involved in the update. May be empty.
      */
      transactions) {
          this.view = view;
          this.state = state;
          this.transactions = transactions;
          /**
          @internal
          */
          this.flags = 0;
          this.startState = view.state;
          this.changes = ChangeSet.empty(this.startState.doc.length);
          for (let tr of transactions)
              this.changes = this.changes.compose(tr.changes);
          let changedRanges = [];
          this.changes.iterChangedRanges((fromA, toA, fromB, toB) => changedRanges.push(new ChangedRange(fromA, toA, fromB, toB)));
          this.changedRanges = changedRanges;
          let focus = view.hasFocus;
          if (focus != view.inputState.notifiedFocused) {
              view.inputState.notifiedFocused = focus;
              this.flags |= 1 /* UpdateFlag.Focus */;
          }
      }
      /**
      @internal
      */
      static create(view, state, transactions) {
          return new ViewUpdate(view, state, transactions);
      }
      /**
      Tells you whether the [viewport](https://codemirror.net/6/docs/ref/#view.EditorView.viewport) or
      [visible ranges](https://codemirror.net/6/docs/ref/#view.EditorView.visibleRanges) changed in this
      update.
      */
      get viewportChanged() {
          return (this.flags & 4 /* UpdateFlag.Viewport */) > 0;
      }
      /**
      Indicates whether the height of a block element in the editor
      changed in this update.
      */
      get heightChanged() {
          return (this.flags & 2 /* UpdateFlag.Height */) > 0;
      }
      /**
      Returns true when the document was modified or the size of the
      editor, or elements within the editor, changed.
      */
      get geometryChanged() {
          return this.docChanged || (this.flags & (8 /* UpdateFlag.Geometry */ | 2 /* UpdateFlag.Height */)) > 0;
      }
      /**
      True when this update indicates a focus change.
      */
      get focusChanged() {
          return (this.flags & 1 /* UpdateFlag.Focus */) > 0;
      }
      /**
      Whether the document changed in this update.
      */
      get docChanged() {
          return !this.changes.empty;
      }
      /**
      Whether the selection was explicitly set in this update.
      */
      get selectionSet() {
          return this.transactions.some(tr => tr.selection);
      }
      /**
      @internal
      */
      get empty() { return this.flags == 0 && this.transactions.length == 0; }
  }

  /**
  Used to indicate [text direction](https://codemirror.net/6/docs/ref/#view.EditorView.textDirection).
  */
  var Direction = /*@__PURE__*/(function (Direction) {
      // (These are chosen to match the base levels, in bidi algorithm
      // terms, of spans in that direction.)
      /**
      Left-to-right.
      */
      Direction[Direction["LTR"] = 0] = "LTR";
      /**
      Right-to-left.
      */
      Direction[Direction["RTL"] = 1] = "RTL";
  return Direction})(Direction || (Direction = {}));
  const LTR = Direction.LTR, RTL = Direction.RTL;
  // Decode a string with each type encoded as log2(type)
  function dec(str) {
      let result = [];
      for (let i = 0; i < str.length; i++)
          result.push(1 << +str[i]);
      return result;
  }
  // Character types for codepoints 0 to 0xf8
  const LowTypes = /*@__PURE__*/dec("88888888888888888888888888888888888666888888787833333333337888888000000000000000000000000008888880000000000000000000000000088888888888888888888888888888888888887866668888088888663380888308888800000000000000000000000800000000000000000000000000000008");
  // Character types for codepoints 0x600 to 0x6f9
  const ArabicTypes = /*@__PURE__*/dec("4444448826627288999999999992222222222222222222222222222222222222222222222229999999999999999999994444444444644222822222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222222999999949999999229989999223333333333");
  const Brackets = /*@__PURE__*/Object.create(null), BracketStack = [];
  // There's a lot more in
  // https://www.unicode.org/Public/UCD/latest/ucd/BidiBrackets.txt,
  // which are left out to keep code size down.
  for (let p of ["()", "[]", "{}"]) {
      let l = /*@__PURE__*/p.charCodeAt(0), r = /*@__PURE__*/p.charCodeAt(1);
      Brackets[l] = r;
      Brackets[r] = -l;
  }
  function charType(ch) {
      return ch <= 0xf7 ? LowTypes[ch] :
          0x590 <= ch && ch <= 0x5f4 ? 2 /* T.R */ :
              0x600 <= ch && ch <= 0x6f9 ? ArabicTypes[ch - 0x600] :
                  0x6ee <= ch && ch <= 0x8ac ? 4 /* T.AL */ :
                      0x2000 <= ch && ch <= 0x200b ? 256 /* T.NI */ :
                          0xfb50 <= ch && ch <= 0xfdff ? 4 /* T.AL */ :
                              ch == 0x200c ? 256 /* T.NI */ : 1 /* T.L */;
  }
  const BidiRE = /[\u0590-\u05f4\u0600-\u06ff\u0700-\u08ac\ufb50-\ufdff]/;
  /**
  Represents a contiguous range of text that has a single direction
  (as in left-to-right or right-to-left).
  */
  class BidiSpan {
      /**
      @internal
      */
      constructor(
      /**
      The start of the span (relative to the start of the line).
      */
      from, 
      /**
      The end of the span.
      */
      to, 
      /**
      The ["bidi
      level"](https://unicode.org/reports/tr9/#Basic_Display_Algorithm)
      of the span (in this context, 0 means
      left-to-right, 1 means right-to-left, 2 means left-to-right
      number inside right-to-left text).
      */
      level) {
          this.from = from;
          this.to = to;
          this.level = level;
      }
      /**
      The direction of this span.
      */
      get dir() { return this.level % 2 ? RTL : LTR; }
      /**
      @internal
      */
      side(end, dir) { return (this.dir == dir) == end ? this.to : this.from; }
      /**
      @internal
      */
      static find(order, index, level, assoc) {
          let maybe = -1;
          for (let i = 0; i < order.length; i++) {
              let span = order[i];
              if (span.from <= index && span.to >= index) {
                  if (span.level == level)
                      return i;
                  // When multiple spans match, if assoc != 0, take the one that
                  // covers that side, otherwise take the one with the minimum
                  // level.
                  if (maybe < 0 || (assoc != 0 ? (assoc < 0 ? span.from < index : span.to > index) : order[maybe].level > span.level))
                      maybe = i;
              }
          }
          if (maybe < 0)
              throw new RangeError("Index out of range");
          return maybe;
      }
  }
  // Reused array of character types
  const types = [];
  function computeOrder(line, direction) {
      let len = line.length, outerType = direction == LTR ? 1 /* T.L */ : 2 /* T.R */, oppositeType = direction == LTR ? 2 /* T.R */ : 1 /* T.L */;
      if (!line || outerType == 1 /* T.L */ && !BidiRE.test(line))
          return trivialOrder(len);
      // W1. Examine each non-spacing mark (NSM) in the level run, and
      // change the type of the NSM to the type of the previous
      // character. If the NSM is at the start of the level run, it will
      // get the type of sor.
      // W2. Search backwards from each instance of a European number
      // until the first strong type (R, L, AL, or sor) is found. If an
      // AL is found, change the type of the European number to Arabic
      // number.
      // W3. Change all ALs to R.
      // (Left after this: L, R, EN, AN, ET, CS, NI)
      for (let i = 0, prev = outerType, prevStrong = outerType; i < len; i++) {
          let type = charType(line.charCodeAt(i));
          if (type == 512 /* T.NSM */)
              type = prev;
          else if (type == 8 /* T.EN */ && prevStrong == 4 /* T.AL */)
              type = 16 /* T.AN */;
          types[i] = type == 4 /* T.AL */ ? 2 /* T.R */ : type;
          if (type & 7 /* T.Strong */)
              prevStrong = type;
          prev = type;
      }
      // W5. A sequence of European terminators adjacent to European
      // numbers changes to all European numbers.
      // W6. Otherwise, separators and terminators change to Other
      // Neutral.
      // W7. Search backwards from each instance of a European number
      // until the first strong type (R, L, or sor) is found. If an L is
      // found, then change the type of the European number to L.
      // (Left after this: L, R, EN+AN, NI)
      for (let i = 0, prev = outerType, prevStrong = outerType; i < len; i++) {
          let type = types[i];
          if (type == 128 /* T.CS */) {
              if (i < len - 1 && prev == types[i + 1] && (prev & 24 /* T.Num */))
                  type = types[i] = prev;
              else
                  types[i] = 256 /* T.NI */;
          }
          else if (type == 64 /* T.ET */) {
              let end = i + 1;
              while (end < len && types[end] == 64 /* T.ET */)
                  end++;
              let replace = (i && prev == 8 /* T.EN */) || (end < len && types[end] == 8 /* T.EN */) ? (prevStrong == 1 /* T.L */ ? 1 /* T.L */ : 8 /* T.EN */) : 256 /* T.NI */;
              for (let j = i; j < end; j++)
                  types[j] = replace;
              i = end - 1;
          }
          else if (type == 8 /* T.EN */ && prevStrong == 1 /* T.L */) {
              types[i] = 1 /* T.L */;
          }
          prev = type;
          if (type & 7 /* T.Strong */)
              prevStrong = type;
      }
      // N0. Process bracket pairs in an isolating run sequence
      // sequentially in the logical order of the text positions of the
      // opening paired brackets using the logic given below. Within this
      // scope, bidirectional types EN and AN are treated as R.
      for (let i = 0, sI = 0, context = 0, ch, br, type; i < len; i++) {
          // Keeps [startIndex, type, strongSeen] triples for each open
          // bracket on BracketStack.
          if (br = Brackets[ch = line.charCodeAt(i)]) {
              if (br < 0) { // Closing bracket
                  for (let sJ = sI - 3; sJ >= 0; sJ -= 3) {
                      if (BracketStack[sJ + 1] == -br) {
                          let flags = BracketStack[sJ + 2];
                          let type = (flags & 2 /* Bracketed.EmbedInside */) ? outerType :
                              !(flags & 4 /* Bracketed.OppositeInside */) ? 0 :
                                  (flags & 1 /* Bracketed.OppositeBefore */) ? oppositeType : outerType;
                          if (type)
                              types[i] = types[BracketStack[sJ]] = type;
                          sI = sJ;
                          break;
                      }
                  }
              }
              else if (BracketStack.length == 189 /* Bracketed.MaxDepth */) {
                  break;
              }
              else {
                  BracketStack[sI++] = i;
                  BracketStack[sI++] = ch;
                  BracketStack[sI++] = context;
              }
          }
          else if ((type = types[i]) == 2 /* T.R */ || type == 1 /* T.L */) {
              let embed = type == outerType;
              context = embed ? 0 : 1 /* Bracketed.OppositeBefore */;
              for (let sJ = sI - 3; sJ >= 0; sJ -= 3) {
                  let cur = BracketStack[sJ + 2];
                  if (cur & 2 /* Bracketed.EmbedInside */)
                      break;
                  if (embed) {
                      BracketStack[sJ + 2] |= 2 /* Bracketed.EmbedInside */;
                  }
                  else {
                      if (cur & 4 /* Bracketed.OppositeInside */)
                          break;
                      BracketStack[sJ + 2] |= 4 /* Bracketed.OppositeInside */;
                  }
              }
          }
      }
      // N1. A sequence of neutrals takes the direction of the
      // surrounding strong text if the text on both sides has the same
      // direction. European and Arabic numbers act as if they were R in
      // terms of their influence on neutrals. Start-of-level-run (sor)
      // and end-of-level-run (eor) are used at level run boundaries.
      // N2. Any remaining neutrals take the embedding direction.
      // (Left after this: L, R, EN+AN)
      for (let i = 0; i < len; i++) {
          if (types[i] == 256 /* T.NI */) {
              let end = i + 1;
              while (end < len && types[end] == 256 /* T.NI */)
                  end++;
              let beforeL = (i ? types[i - 1] : outerType) == 1 /* T.L */;
              let afterL = (end < len ? types[end] : outerType) == 1 /* T.L */;
              let replace = beforeL == afterL ? (beforeL ? 1 /* T.L */ : 2 /* T.R */) : outerType;
              for (let j = i; j < end; j++)
                  types[j] = replace;
              i = end - 1;
          }
      }
      // Here we depart from the documented algorithm, in order to avoid
      // building up an actual levels array. Since there are only three
      // levels (0, 1, 2) in an implementation that doesn't take
      // explicit embedding into account, we can build up the order on
      // the fly, without following the level-based algorithm.
      let order = [];
      if (outerType == 1 /* T.L */) {
          for (let i = 0; i < len;) {
              let start = i, rtl = types[i++] != 1 /* T.L */;
              while (i < len && rtl == (types[i] != 1 /* T.L */))
                  i++;
              if (rtl) {
                  for (let j = i; j > start;) {
                      let end = j, l = types[--j] != 2 /* T.R */;
                      while (j > start && l == (types[j - 1] != 2 /* T.R */))
                          j--;
                      order.push(new BidiSpan(j, end, l ? 2 : 1));
                  }
              }
              else {
                  order.push(new BidiSpan(start, i, 0));
              }
          }
      }
      else {
          for (let i = 0; i < len;) {
              let start = i, rtl = types[i++] == 2 /* T.R */;
              while (i < len && rtl == (types[i] == 2 /* T.R */))
                  i++;
              order.push(new BidiSpan(start, i, rtl ? 1 : 2));
          }
      }
      return order;
  }
  function trivialOrder(length) {
      return [new BidiSpan(0, length, 0)];
  }
  let movedOver = "";
  function moveVisually(line, order, dir, start, forward) {
      var _a;
      let startIndex = start.head - line.from, spanI = -1;
      if (startIndex == 0) {
          if (!forward || !line.length)
              return null;
          if (order[0].level != dir) {
              startIndex = order[0].side(false, dir);
              spanI = 0;
          }
      }
      else if (startIndex == line.length) {
          if (forward)
              return null;
          let last = order[order.length - 1];
          if (last.level != dir) {
              startIndex = last.side(true, dir);
              spanI = order.length - 1;
          }
      }
      if (spanI < 0)
          spanI = BidiSpan.find(order, startIndex, (_a = start.bidiLevel) !== null && _a !== void 0 ? _a : -1, start.assoc);
      let span = order[spanI];
      // End of span. (But not end of line--that was checked for above.)
      if (startIndex == span.side(forward, dir)) {
          span = order[spanI += forward ? 1 : -1];
          startIndex = span.side(!forward, dir);
      }
      let indexForward = forward == (span.dir == dir);
      let nextIndex = findClusterBreak(line.text, startIndex, indexForward);
      movedOver = line.text.slice(Math.min(startIndex, nextIndex), Math.max(startIndex, nextIndex));
      if (nextIndex != span.side(forward, dir))
          return EditorSelection.cursor(nextIndex + line.from, indexForward ? -1 : 1, span.level);
      let nextSpan = spanI == (forward ? order.length - 1 : 0) ? null : order[spanI + (forward ? 1 : -1)];
      if (!nextSpan && span.level != dir)
          return EditorSelection.cursor(forward ? line.to : line.from, forward ? -1 : 1, dir);
      if (nextSpan && nextSpan.level < span.level)
          return EditorSelection.cursor(nextSpan.side(!forward, dir) + line.from, forward ? 1 : -1, nextSpan.level);
      return EditorSelection.cursor(nextIndex + line.from, forward ? -1 : 1, span.level);
  }

  const LineBreakPlaceholder = "\uffff";
  class DOMReader {
      constructor(points, state) {
          this.points = points;
          this.text = "";
          this.lineSeparator = state.facet(EditorState.lineSeparator);
      }
      append(text) {
          this.text += text;
      }
      lineBreak() {
          this.text += LineBreakPlaceholder;
      }
      readRange(start, end) {
          if (!start)
              return this;
          let parent = start.parentNode;
          for (let cur = start;;) {
              this.findPointBefore(parent, cur);
              this.readNode(cur);
              let next = cur.nextSibling;
              if (next == end)
                  break;
              let view = ContentView.get(cur), nextView = ContentView.get(next);
              if (view && nextView ? view.breakAfter :
                  (view ? view.breakAfter : isBlockElement(cur)) ||
                      (isBlockElement(next) && (cur.nodeName != "BR" || cur.cmIgnore)))
                  this.lineBreak();
              cur = next;
          }
          this.findPointBefore(parent, end);
          return this;
      }
      readTextNode(node) {
          let text = node.nodeValue;
          for (let point of this.points)
              if (point.node == node)
                  point.pos = this.text.length + Math.min(point.offset, text.length);
          for (let off = 0, re = this.lineSeparator ? null : /\r\n?|\n/g;;) {
              let nextBreak = -1, breakSize = 1, m;
              if (this.lineSeparator) {
                  nextBreak = text.indexOf(this.lineSeparator, off);
                  breakSize = this.lineSeparator.length;
              }
              else if (m = re.exec(text)) {
                  nextBreak = m.index;
                  breakSize = m[0].length;
              }
              this.append(text.slice(off, nextBreak < 0 ? text.length : nextBreak));
              if (nextBreak < 0)
                  break;
              this.lineBreak();
              if (breakSize > 1)
                  for (let point of this.points)
                      if (point.node == node && point.pos > this.text.length)
                          point.pos -= breakSize - 1;
              off = nextBreak + breakSize;
          }
      }
      readNode(node) {
          if (node.cmIgnore)
              return;
          let view = ContentView.get(node);
          let fromView = view && view.overrideDOMText;
          if (fromView != null) {
              this.findPointInside(node, fromView.length);
              for (let i = fromView.iter(); !i.next().done;) {
                  if (i.lineBreak)
                      this.lineBreak();
                  else
                      this.append(i.value);
              }
          }
          else if (node.nodeType == 3) {
              this.readTextNode(node);
          }
          else if (node.nodeName == "BR") {
              if (node.nextSibling)
                  this.lineBreak();
          }
          else if (node.nodeType == 1) {
              this.readRange(node.firstChild, null);
          }
      }
      findPointBefore(node, next) {
          for (let point of this.points)
              if (point.node == node && node.childNodes[point.offset] == next)
                  point.pos = this.text.length;
      }
      findPointInside(node, maxLen) {
          for (let point of this.points)
              if (node.nodeType == 3 ? point.node == node : node.contains(point.node))
                  point.pos = this.text.length + Math.min(maxLen, point.offset);
      }
  }
  function isBlockElement(node) {
      return node.nodeType == 1 && /^(DIV|P|LI|UL|OL|BLOCKQUOTE|DD|DT|H\d|SECTION|PRE)$/.test(node.nodeName);
  }
  class DOMPoint {
      constructor(node, offset) {
          this.node = node;
          this.offset = offset;
          this.pos = -1;
      }
  }

  class DocView extends ContentView {
      constructor(view) {
          super();
          this.view = view;
          this.compositionDeco = Decoration.none;
          this.decorations = [];
          this.dynamicDecorationMap = [];
          // Track a minimum width for the editor. When measuring sizes in
          // measureVisibleLineHeights, this is updated to point at the width
          // of a given element and its extent in the document. When a change
          // happens in that range, these are reset. That way, once we've seen
          // a line/element of a given length, we keep the editor wide enough
          // to fit at least that element, until it is changed, at which point
          // we forget it again.
          this.minWidth = 0;
          this.minWidthFrom = 0;
          this.minWidthTo = 0;
          // Track whether the DOM selection was set in a lossy way, so that
          // we don't mess it up when reading it back it
          this.impreciseAnchor = null;
          this.impreciseHead = null;
          this.forceSelection = false;
          // Used by the resize observer to ignore resizes that we caused
          // ourselves
          this.lastUpdate = Date.now();
          this.setDOM(view.contentDOM);
          this.children = [new LineView];
          this.children[0].setParent(this);
          this.updateDeco();
          this.updateInner([new ChangedRange(0, 0, 0, view.state.doc.length)], 0);
      }
      get editorView() { return this.view; }
      get length() { return this.view.state.doc.length; }
      // Update the document view to a given state. scrollIntoView can be
      // used as a hint to compute a new viewport that includes that
      // position, if we know the editor is going to scroll that position
      // into view.
      update(update) {
          let changedRanges = update.changedRanges;
          if (this.minWidth > 0 && changedRanges.length) {
              if (!changedRanges.every(({ fromA, toA }) => toA < this.minWidthFrom || fromA > this.minWidthTo)) {
                  this.minWidth = this.minWidthFrom = this.minWidthTo = 0;
              }
              else {
                  this.minWidthFrom = update.changes.mapPos(this.minWidthFrom, 1);
                  this.minWidthTo = update.changes.mapPos(this.minWidthTo, 1);
              }
          }
          if (this.view.inputState.composing < 0)
              this.compositionDeco = Decoration.none;
          else if (update.transactions.length || this.dirty)
              this.compositionDeco = computeCompositionDeco(this.view, update.changes);
          // When the DOM nodes around the selection are moved to another
          // parent, Chrome sometimes reports a different selection through
          // getSelection than the one that it actually shows to the user.
          // This forces a selection update when lines are joined to work
          // around that. Issue #54
          if ((browser.ie || browser.chrome) && !this.compositionDeco.size && update &&
              update.state.doc.lines != update.startState.doc.lines)
              this.forceSelection = true;
          let prevDeco = this.decorations, deco = this.updateDeco();
          let decoDiff = findChangedDeco(prevDeco, deco, update.changes);
          changedRanges = ChangedRange.extendWithRanges(changedRanges, decoDiff);
          if (this.dirty == 0 /* Dirty.Not */ && changedRanges.length == 0) {
              return false;
          }
          else {
              this.updateInner(changedRanges, update.startState.doc.length);
              if (update.transactions.length)
                  this.lastUpdate = Date.now();
              return true;
          }
      }
      // Used by update and the constructor do perform the actual DOM
      // update
      updateInner(changes, oldLength) {
          this.view.viewState.mustMeasureContent = true;
          this.updateChildren(changes, oldLength);
          let { observer } = this.view;
          observer.ignore(() => {
              // Lock the height during redrawing, since Chrome sometimes
              // messes with the scroll position during DOM mutation (though
              // no relayout is triggered and I cannot imagine how it can
              // recompute the scroll position without a layout)
              this.dom.style.height = this.view.viewState.contentHeight + "px";
              this.dom.style.flexBasis = this.minWidth ? this.minWidth + "px" : "";
              // Chrome will sometimes, when DOM mutations occur directly
              // around the selection, get confused and report a different
              // selection from the one it displays (issue #218). This tries
              // to detect that situation.
              let track = browser.chrome || browser.ios ? { node: observer.selectionRange.focusNode, written: false } : undefined;
              this.sync(track);
              this.dirty = 0 /* Dirty.Not */;
              if (track && (track.written || observer.selectionRange.focusNode != track.node))
                  this.forceSelection = true;
              this.dom.style.height = "";
          });
          let gaps = [];
          if (this.view.viewport.from || this.view.viewport.to < this.view.state.doc.length)
              for (let child of this.children)
                  if (child instanceof BlockWidgetView && child.widget instanceof BlockGapWidget)
                      gaps.push(child.dom);
          observer.updateGaps(gaps);
      }
      updateChildren(changes, oldLength) {
          let cursor = this.childCursor(oldLength);
          for (let i = changes.length - 1;; i--) {
              let next = i >= 0 ? changes[i] : null;
              if (!next)
                  break;
              let { fromA, toA, fromB, toB } = next;
              let { content, breakAtStart, openStart, openEnd } = ContentBuilder.build(this.view.state.doc, fromB, toB, this.decorations, this.dynamicDecorationMap);
              let { i: toI, off: toOff } = cursor.findPos(toA, 1);
              let { i: fromI, off: fromOff } = cursor.findPos(fromA, -1);
              replaceRange(this, fromI, fromOff, toI, toOff, content, breakAtStart, openStart, openEnd);
          }
      }
      // Sync the DOM selection to this.state.selection
      updateSelection(mustRead = false, fromPointer = false) {
          if (mustRead || !this.view.observer.selectionRange.focusNode)
              this.view.observer.readSelectionRange();
          if (!(fromPointer || this.mayControlSelection()))
              return;
          let force = this.forceSelection;
          this.forceSelection = false;
          let main = this.view.state.selection.main;
          // FIXME need to handle the case where the selection falls inside a block range
          let anchor = this.domAtPos(main.anchor);
          let head = main.empty ? anchor : this.domAtPos(main.head);
          // Always reset on Firefox when next to an uneditable node to
          // avoid invisible cursor bugs (#111)
          if (browser.gecko && main.empty && betweenUneditable(anchor)) {
              let dummy = document.createTextNode("");
              this.view.observer.ignore(() => anchor.node.insertBefore(dummy, anchor.node.childNodes[anchor.offset] || null));
              anchor = head = new DOMPos(dummy, 0);
              force = true;
          }
          let domSel = this.view.observer.selectionRange;
          // If the selection is already here, or in an equivalent position, don't touch it
          if (force || !domSel.focusNode ||
              !isEquivalentPosition(anchor.node, anchor.offset, domSel.anchorNode, domSel.anchorOffset) ||
              !isEquivalentPosition(head.node, head.offset, domSel.focusNode, domSel.focusOffset)) {
              this.view.observer.ignore(() => {
                  // Chrome Android will hide the virtual keyboard when tapping
                  // inside an uneditable node, and not bring it back when we
                  // move the cursor to its proper position. This tries to
                  // restore the keyboard by cycling focus.
                  if (browser.android && browser.chrome && this.dom.contains(domSel.focusNode) &&
                      inUneditable(domSel.focusNode, this.dom)) {
                      this.dom.blur();
                      this.dom.focus({ preventScroll: true });
                  }
                  let rawSel = getSelection(this.view.root);
                  if (!rawSel) ;
                  else if (main.empty) {
                      // Work around https://bugzilla.mozilla.org/show_bug.cgi?id=1612076
                      if (browser.gecko) {
                          let nextTo = nextToUneditable(anchor.node, anchor.offset);
                          if (nextTo && nextTo != (1 /* NextTo.Before */ | 2 /* NextTo.After */)) {
                              let text = nearbyTextNode(anchor.node, anchor.offset, nextTo == 1 /* NextTo.Before */ ? 1 : -1);
                              if (text)
                                  anchor = new DOMPos(text, nextTo == 1 /* NextTo.Before */ ? 0 : text.nodeValue.length);
                          }
                      }
                      rawSel.collapse(anchor.node, anchor.offset);
                      if (main.bidiLevel != null && domSel.cursorBidiLevel != null)
                          domSel.cursorBidiLevel = main.bidiLevel;
                  }
                  else if (rawSel.extend) {
                      // Selection.extend can be used to create an 'inverted' selection
                      // (one where the focus is before the anchor), but not all
                      // browsers support it yet.
                      rawSel.collapse(anchor.node, anchor.offset);
                      // Safari will ignore the call above when the editor is
                      // hidden, and then raise an error on the call to extend
                      // (#940).
                      try {
                          rawSel.extend(head.node, head.offset);
                      }
                      catch (_) { }
                  }
                  else {
                      // Primitive (IE) way
                      let range = document.createRange();
                      if (main.anchor > main.head)
                          [anchor, head] = [head, anchor];
                      range.setEnd(head.node, head.offset);
                      range.setStart(anchor.node, anchor.offset);
                      rawSel.removeAllRanges();
                      rawSel.addRange(range);
                  }
              });
              this.view.observer.setSelectionRange(anchor, head);
          }
          this.impreciseAnchor = anchor.precise ? null : new DOMPos(domSel.anchorNode, domSel.anchorOffset);
          this.impreciseHead = head.precise ? null : new DOMPos(domSel.focusNode, domSel.focusOffset);
      }
      enforceCursorAssoc() {
          if (this.compositionDeco.size)
              return;
          let { view } = this, cursor = view.state.selection.main;
          let sel = getSelection(view.root);
          let { anchorNode, anchorOffset } = view.observer.selectionRange;
          if (!sel || !cursor.empty || !cursor.assoc || !sel.modify)
              return;
          let line = LineView.find(this, cursor.head);
          if (!line)
              return;
          let lineStart = line.posAtStart;
          if (cursor.head == lineStart || cursor.head == lineStart + line.length)
              return;
          let before = this.coordsAt(cursor.head, -1), after = this.coordsAt(cursor.head, 1);
          if (!before || !after || before.bottom > after.top)
              return;
          let dom = this.domAtPos(cursor.head + cursor.assoc);
          sel.collapse(dom.node, dom.offset);
          sel.modify("move", cursor.assoc < 0 ? "forward" : "backward", "lineboundary");
          // This can go wrong in corner cases like single-character lines,
          // so check and reset if necessary.
          view.observer.readSelectionRange();
          let newRange = view.observer.selectionRange;
          if (view.docView.posFromDOM(newRange.anchorNode, newRange.anchorOffset) != cursor.from)
              sel.collapse(anchorNode, anchorOffset);
      }
      mayControlSelection() {
          let active = this.view.root.activeElement;
          return active == this.dom ||
              hasSelection(this.dom, this.view.observer.selectionRange) && !(active && this.dom.contains(active));
      }
      nearest(dom) {
          for (let cur = dom; cur;) {
              let domView = ContentView.get(cur);
              if (domView && domView.rootView == this)
                  return domView;
              cur = cur.parentNode;
          }
          return null;
      }
      posFromDOM(node, offset) {
          let view = this.nearest(node);
          if (!view)
              throw new RangeError("Trying to find position for a DOM position outside of the document");
          return view.localPosFromDOM(node, offset) + view.posAtStart;
      }
      domAtPos(pos) {
          let { i, off } = this.childCursor().findPos(pos, -1);
          for (; i < this.children.length - 1;) {
              let child = this.children[i];
              if (off < child.length || child instanceof LineView)
                  break;
              i++;
              off = 0;
          }
          return this.children[i].domAtPos(off);
      }
      coordsAt(pos, side) {
          for (let off = this.length, i = this.children.length - 1;; i--) {
              let child = this.children[i], start = off - child.breakAfter - child.length;
              if (pos > start ||
                  (pos == start && child.type != BlockType.WidgetBefore && child.type != BlockType.WidgetAfter &&
                      (!i || side == 2 || this.children[i - 1].breakAfter ||
                          (this.children[i - 1].type == BlockType.WidgetBefore && side > -2))))
                  return child.coordsAt(pos - start, side);
              off = start;
          }
      }
      measureVisibleLineHeights(viewport) {
          let result = [], { from, to } = viewport;
          let contentWidth = this.view.contentDOM.clientWidth;
          let isWider = contentWidth > Math.max(this.view.scrollDOM.clientWidth, this.minWidth) + 1;
          let widest = -1, ltr = this.view.textDirection == Direction.LTR;
          for (let pos = 0, i = 0; i < this.children.length; i++) {
              let child = this.children[i], end = pos + child.length;
              if (end > to)
                  break;
              if (pos >= from) {
                  let childRect = child.dom.getBoundingClientRect();
                  result.push(childRect.height);
                  if (isWider) {
                      let last = child.dom.lastChild;
                      let rects = last ? clientRectsFor(last) : [];
                      if (rects.length) {
                          let rect = rects[rects.length - 1];
                          let width = ltr ? rect.right - childRect.left : childRect.right - rect.left;
                          if (width > widest) {
                              widest = width;
                              this.minWidth = contentWidth;
                              this.minWidthFrom = pos;
                              this.minWidthTo = end;
                          }
                      }
                  }
              }
              pos = end + child.breakAfter;
          }
          return result;
      }
      textDirectionAt(pos) {
          let { i } = this.childPos(pos, 1);
          return getComputedStyle(this.children[i].dom).direction == "rtl" ? Direction.RTL : Direction.LTR;
      }
      measureTextSize() {
          for (let child of this.children) {
              if (child instanceof LineView) {
                  let measure = child.measureTextSize();
                  if (measure)
                      return measure;
              }
          }
          // If no workable line exists, force a layout of a measurable element
          let dummy = document.createElement("div"), lineHeight, charWidth;
          dummy.className = "cm-line";
          dummy.style.width = "99999px";
          dummy.textContent = "abc def ghi jkl mno pqr stu";
          this.view.observer.ignore(() => {
              this.dom.appendChild(dummy);
              let rect = clientRectsFor(dummy.firstChild)[0];
              lineHeight = dummy.getBoundingClientRect().height;
              charWidth = rect ? rect.width / 27 : 7;
              dummy.remove();
          });
          return { lineHeight, charWidth };
      }
      childCursor(pos = this.length) {
          // Move back to start of last element when possible, so that
          // `ChildCursor.findPos` doesn't have to deal with the edge case
          // of being after the last element.
          let i = this.children.length;
          if (i)
              pos -= this.children[--i].length;
          return new ChildCursor(this.children, pos, i);
      }
      computeBlockGapDeco() {
          let deco = [], vs = this.view.viewState;
          for (let pos = 0, i = 0;; i++) {
              let next = i == vs.viewports.length ? null : vs.viewports[i];
              let end = next ? next.from - 1 : this.length;
              if (end > pos) {
                  let height = vs.lineBlockAt(end).bottom - vs.lineBlockAt(pos).top;
                  deco.push(Decoration.replace({
                      widget: new BlockGapWidget(height),
                      block: true,
                      inclusive: true,
                      isBlockGap: true,
                  }).range(pos, end));
              }
              if (!next)
                  break;
              pos = next.to + 1;
          }
          return Decoration.set(deco);
      }
      updateDeco() {
          let allDeco = this.view.state.facet(decorations).map((d, i) => {
              let dynamic = this.dynamicDecorationMap[i] = typeof d == "function";
              return dynamic ? d(this.view) : d;
          });
          for (let i = allDeco.length; i < allDeco.length + 3; i++)
              this.dynamicDecorationMap[i] = false;
          return this.decorations = [
              ...allDeco,
              this.compositionDeco,
              this.computeBlockGapDeco(),
              this.view.viewState.lineGapDeco
          ];
      }
      scrollIntoView(target) {
          let { range } = target;
          let rect = this.coordsAt(range.head, range.empty ? range.assoc : range.head > range.anchor ? -1 : 1), other;
          if (!rect)
              return;
          if (!range.empty && (other = this.coordsAt(range.anchor, range.anchor > range.head ? -1 : 1)))
              rect = { left: Math.min(rect.left, other.left), top: Math.min(rect.top, other.top),
                  right: Math.max(rect.right, other.right), bottom: Math.max(rect.bottom, other.bottom) };
          let mLeft = 0, mRight = 0, mTop = 0, mBottom = 0;
          for (let margins of this.view.state.facet(scrollMargins).map(f => f(this.view)))
              if (margins) {
                  let { left, right, top, bottom } = margins;
                  if (left != null)
                      mLeft = Math.max(mLeft, left);
                  if (right != null)
                      mRight = Math.max(mRight, right);
                  if (top != null)
                      mTop = Math.max(mTop, top);
                  if (bottom != null)
                      mBottom = Math.max(mBottom, bottom);
              }
          let targetRect = {
              left: rect.left - mLeft, top: rect.top - mTop,
              right: rect.right + mRight, bottom: rect.bottom + mBottom
          };
          scrollRectIntoView(this.view.scrollDOM, targetRect, range.head < range.anchor ? -1 : 1, target.x, target.y, target.xMargin, target.yMargin, this.view.textDirection == Direction.LTR);
      }
  }
  function betweenUneditable(pos) {
      return pos.node.nodeType == 1 && pos.node.firstChild &&
          (pos.offset == 0 || pos.node.childNodes[pos.offset - 1].contentEditable == "false") &&
          (pos.offset == pos.node.childNodes.length || pos.node.childNodes[pos.offset].contentEditable == "false");
  }
  class BlockGapWidget extends WidgetType {
      constructor(height) {
          super();
          this.height = height;
      }
      toDOM() {
          let elt = document.createElement("div");
          this.updateDOM(elt);
          return elt;
      }
      eq(other) { return other.height == this.height; }
      updateDOM(elt) {
          elt.style.height = this.height + "px";
          return true;
      }
      get estimatedHeight() { return this.height; }
  }
  function compositionSurroundingNode(view) {
      let sel = view.observer.selectionRange;
      let textNode = sel.focusNode && nearbyTextNode(sel.focusNode, sel.focusOffset, 0);
      if (!textNode)
          return null;
      let cView = view.docView.nearest(textNode);
      if (!cView)
          return null;
      if (cView instanceof LineView) {
          let topNode = textNode;
          while (topNode.parentNode != cView.dom)
              topNode = topNode.parentNode;
          let prev = topNode.previousSibling;
          while (prev && !ContentView.get(prev))
              prev = prev.previousSibling;
          let pos = prev ? ContentView.get(prev).posAtEnd : cView.posAtStart;
          return { from: pos, to: pos, node: topNode, text: textNode };
      }
      else {
          for (;;) {
              let { parent } = cView;
              if (!parent)
                  return null;
              if (parent instanceof LineView)
                  break;
              cView = parent;
          }
          let from = cView.posAtStart;
          return { from, to: from + cView.length, node: cView.dom, text: textNode };
      }
  }
  function computeCompositionDeco(view, changes) {
      let surrounding = compositionSurroundingNode(view);
      if (!surrounding)
          return Decoration.none;
      let { from, to, node, text: textNode } = surrounding;
      let newFrom = changes.mapPos(from, 1), newTo = Math.max(newFrom, changes.mapPos(to, -1));
      let { state } = view, text = node.nodeType == 3 ? node.nodeValue :
          new DOMReader([], state).readRange(node.firstChild, null).text;
      if (newTo - newFrom < text.length) {
          if (state.doc.sliceString(newFrom, Math.min(state.doc.length, newFrom + text.length), LineBreakPlaceholder) == text)
              newTo = newFrom + text.length;
          else if (state.doc.sliceString(Math.max(0, newTo - text.length), newTo, LineBreakPlaceholder) == text)
              newFrom = newTo - text.length;
          else
              return Decoration.none;
      }
      else if (state.doc.sliceString(newFrom, newTo, LineBreakPlaceholder) != text) {
          return Decoration.none;
      }
      let topView = ContentView.get(node);
      if (topView instanceof CompositionView)
          topView = topView.widget.topView;
      else if (topView)
          topView.parent = null;
      return Decoration.set(Decoration.replace({ widget: new CompositionWidget(node, textNode, topView), inclusive: true })
          .range(newFrom, newTo));
  }
  class CompositionWidget extends WidgetType {
      constructor(top, text, topView) {
          super();
          this.top = top;
          this.text = text;
          this.topView = topView;
      }
      eq(other) { return this.top == other.top && this.text == other.text; }
      toDOM() { return this.top; }
      ignoreEvent() { return false; }
      get customView() { return CompositionView; }
  }
  function nearbyTextNode(node, offset, side) {
      for (;;) {
          if (node.nodeType == 3)
              return node;
          if (node.nodeType == 1 && offset > 0 && side <= 0) {
              node = node.childNodes[offset - 1];
              offset = maxOffset(node);
          }
          else if (node.nodeType == 1 && offset < node.childNodes.length && side >= 0) {
              node = node.childNodes[offset];
              offset = 0;
          }
          else {
              return null;
          }
      }
  }
  function nextToUneditable(node, offset) {
      if (node.nodeType != 1)
          return 0;
      return (offset && node.childNodes[offset - 1].contentEditable == "false" ? 1 /* NextTo.Before */ : 0) |
          (offset < node.childNodes.length && node.childNodes[offset].contentEditable == "false" ? 2 /* NextTo.After */ : 0);
  }
  class DecorationComparator$1 {
      constructor() {
          this.changes = [];
      }
      compareRange(from, to) { addRange(from, to, this.changes); }
      comparePoint(from, to) { addRange(from, to, this.changes); }
  }
  function findChangedDeco(a, b, diff) {
      let comp = new DecorationComparator$1;
      RangeSet.compare(a, b, diff, comp);
      return comp.changes;
  }
  function inUneditable(node, inside) {
      for (let cur = node; cur && cur != inside; cur = cur.assignedSlot || cur.parentNode) {
          if (cur.nodeType == 1 && cur.contentEditable == 'false') {
              return true;
          }
      }
      return false;
  }

  function groupAt(state, pos, bias = 1) {
      let categorize = state.charCategorizer(pos);
      let line = state.doc.lineAt(pos), linePos = pos - line.from;
      if (line.length == 0)
          return EditorSelection.cursor(pos);
      if (linePos == 0)
          bias = 1;
      else if (linePos == line.length)
          bias = -1;
      let from = linePos, to = linePos;
      if (bias < 0)
          from = findClusterBreak(line.text, linePos, false);
      else
          to = findClusterBreak(line.text, linePos);
      let cat = categorize(line.text.slice(from, to));
      while (from > 0) {
          let prev = findClusterBreak(line.text, from, false);
          if (categorize(line.text.slice(prev, from)) != cat)
              break;
          from = prev;
      }
      while (to < line.length) {
          let next = findClusterBreak(line.text, to);
          if (categorize(line.text.slice(to, next)) != cat)
              break;
          to = next;
      }
      return EditorSelection.range(from + line.from, to + line.from);
  }
  // Search the DOM for the {node, offset} position closest to the given
  // coordinates. Very inefficient and crude, but can usually be avoided
  // by calling caret(Position|Range)FromPoint instead.
  function getdx(x, rect) {
      return rect.left > x ? rect.left - x : Math.max(0, x - rect.right);
  }
  function getdy(y, rect) {
      return rect.top > y ? rect.top - y : Math.max(0, y - rect.bottom);
  }
  function yOverlap(a, b) {
      return a.top < b.bottom - 1 && a.bottom > b.top + 1;
  }
  function upTop(rect, top) {
      return top < rect.top ? { top, left: rect.left, right: rect.right, bottom: rect.bottom } : rect;
  }
  function upBot(rect, bottom) {
      return bottom > rect.bottom ? { top: rect.top, left: rect.left, right: rect.right, bottom } : rect;
  }
  function domPosAtCoords(parent, x, y) {
      let closest, closestRect, closestX, closestY, closestOverlap = false;
      let above, below, aboveRect, belowRect;
      for (let child = parent.firstChild; child; child = child.nextSibling) {
          let rects = clientRectsFor(child);
          for (let i = 0; i < rects.length; i++) {
              let rect = rects[i];
              if (closestRect && yOverlap(closestRect, rect))
                  rect = upTop(upBot(rect, closestRect.bottom), closestRect.top);
              let dx = getdx(x, rect), dy = getdy(y, rect);
              if (dx == 0 && dy == 0)
                  return child.nodeType == 3 ? domPosInText(child, x, y) : domPosAtCoords(child, x, y);
              if (!closest || closestY > dy || closestY == dy && closestX > dx) {
                  closest = child;
                  closestRect = rect;
                  closestX = dx;
                  closestY = dy;
                  closestOverlap = !dx || (dx > 0 ? i < rects.length - 1 : i > 0);
              }
              if (dx == 0) {
                  if (y > rect.bottom && (!aboveRect || aboveRect.bottom < rect.bottom)) {
                      above = child;
                      aboveRect = rect;
                  }
                  else if (y < rect.top && (!belowRect || belowRect.top > rect.top)) {
                      below = child;
                      belowRect = rect;
                  }
              }
              else if (aboveRect && yOverlap(aboveRect, rect)) {
                  aboveRect = upBot(aboveRect, rect.bottom);
              }
              else if (belowRect && yOverlap(belowRect, rect)) {
                  belowRect = upTop(belowRect, rect.top);
              }
          }
      }
      if (aboveRect && aboveRect.bottom >= y) {
          closest = above;
          closestRect = aboveRect;
      }
      else if (belowRect && belowRect.top <= y) {
          closest = below;
          closestRect = belowRect;
      }
      if (!closest)
          return { node: parent, offset: 0 };
      let clipX = Math.max(closestRect.left, Math.min(closestRect.right, x));
      if (closest.nodeType == 3)
          return domPosInText(closest, clipX, y);
      if (closestOverlap && closest.contentEditable != "false")
          return domPosAtCoords(closest, clipX, y);
      let offset = Array.prototype.indexOf.call(parent.childNodes, closest) +
          (x >= (closestRect.left + closestRect.right) / 2 ? 1 : 0);
      return { node: parent, offset };
  }
  function domPosInText(node, x, y) {
      let len = node.nodeValue.length;
      let closestOffset = -1, closestDY = 1e9, generalSide = 0;
      for (let i = 0; i < len; i++) {
          let rects = textRange(node, i, i + 1).getClientRects();
          for (let j = 0; j < rects.length; j++) {
              let rect = rects[j];
              if (rect.top == rect.bottom)
                  continue;
              if (!generalSide)
                  generalSide = x - rect.left;
              let dy = (rect.top > y ? rect.top - y : y - rect.bottom) - 1;
              if (rect.left - 1 <= x && rect.right + 1 >= x && dy < closestDY) {
                  let right = x >= (rect.left + rect.right) / 2, after = right;
                  if (browser.chrome || browser.gecko) {
                      // Check for RTL on browsers that support getting client
                      // rects for empty ranges.
                      let rectBefore = textRange(node, i).getBoundingClientRect();
                      if (rectBefore.left == rect.right)
                          after = !right;
                  }
                  if (dy <= 0)
                      return { node, offset: i + (after ? 1 : 0) };
                  closestOffset = i + (after ? 1 : 0);
                  closestDY = dy;
              }
          }
      }
      return { node, offset: closestOffset > -1 ? closestOffset : generalSide > 0 ? node.nodeValue.length : 0 };
  }
  function posAtCoords(view, { x, y }, precise, bias = -1) {
      var _a;
      let content = view.contentDOM.getBoundingClientRect(), docTop = content.top + view.viewState.paddingTop;
      let block, { docHeight } = view.viewState;
      let yOffset = y - docTop;
      if (yOffset < 0)
          return 0;
      if (yOffset > docHeight)
          return view.state.doc.length;
      // Scan for a text block near the queried y position
      for (let halfLine = view.defaultLineHeight / 2, bounced = false;;) {
          block = view.elementAtHeight(yOffset);
          if (block.type == BlockType.Text)
              break;
          for (;;) {
              // Move the y position out of this block
              yOffset = bias > 0 ? block.bottom + halfLine : block.top - halfLine;
              if (yOffset >= 0 && yOffset <= docHeight)
                  break;
              // If the document consists entirely of replaced widgets, we
              // won't find a text block, so return 0
              if (bounced)
                  return precise ? null : 0;
              bounced = true;
              bias = -bias;
          }
      }
      y = docTop + yOffset;
      let lineStart = block.from;
      // If this is outside of the rendered viewport, we can't determine a position
      if (lineStart < view.viewport.from)
          return view.viewport.from == 0 ? 0 : precise ? null : posAtCoordsImprecise(view, content, block, x, y);
      if (lineStart > view.viewport.to)
          return view.viewport.to == view.state.doc.length ? view.state.doc.length :
              precise ? null : posAtCoordsImprecise(view, content, block, x, y);
      // Prefer ShadowRootOrDocument.elementFromPoint if present, fall back to document if not
      let doc = view.dom.ownerDocument;
      let root = view.root.elementFromPoint ? view.root : doc;
      let element = root.elementFromPoint(x, y);
      if (element && !view.contentDOM.contains(element))
          element = null;
      // If the element is unexpected, clip x at the sides of the content area and try again
      if (!element) {
          x = Math.max(content.left + 1, Math.min(content.right - 1, x));
          element = root.elementFromPoint(x, y);
          if (element && !view.contentDOM.contains(element))
              element = null;
      }
      // There's visible editor content under the point, so we can try
      // using caret(Position|Range)FromPoint as a shortcut
      let node, offset = -1;
      if (element && ((_a = view.docView.nearest(element)) === null || _a === void 0 ? void 0 : _a.isEditable) != false) {
          if (doc.caretPositionFromPoint) {
              let pos = doc.caretPositionFromPoint(x, y);
              if (pos)
                  ({ offsetNode: node, offset } = pos);
          }
          else if (doc.caretRangeFromPoint) {
              let range = doc.caretRangeFromPoint(x, y);
              if (range) {
                  ({ startContainer: node, startOffset: offset } = range);
                  if (!view.contentDOM.contains(node) ||
                      browser.safari && isSuspiciousSafariCaretResult(node, offset, x) ||
                      browser.chrome && isSuspiciousChromeCaretResult(node, offset, x))
                      node = undefined;
              }
          }
      }
      // No luck, do our own (potentially expensive) search
      if (!node || !view.docView.dom.contains(node)) {
          let line = LineView.find(view.docView, lineStart);
          if (!line)
              return yOffset > block.top + block.height / 2 ? block.to : block.from;
          ({ node, offset } = domPosAtCoords(line.dom, x, y));
      }
      return view.docView.posFromDOM(node, offset);
  }
  function posAtCoordsImprecise(view, contentRect, block, x, y) {
      let into = Math.round((x - contentRect.left) * view.defaultCharacterWidth);
      if (view.lineWrapping && block.height > view.defaultLineHeight * 1.5) {
          let line = Math.floor((y - block.top) / view.defaultLineHeight);
          into += line * view.viewState.heightOracle.lineLength;
      }
      let content = view.state.sliceDoc(block.from, block.to);
      return block.from + findColumn(content, into, view.state.tabSize);
  }
  // In case of a high line height, Safari's caretRangeFromPoint treats
  // the space between lines as belonging to the last character of the
  // line before. This is used to detect such a result so that it can be
  // ignored (issue #401).
  function isSuspiciousSafariCaretResult(node, offset, x) {
      let len;
      if (node.nodeType != 3 || offset != (len = node.nodeValue.length))
          return false;
      for (let next = node.nextSibling; next; next = next.nextSibling)
          if (next.nodeType != 1 || next.nodeName != "BR")
              return false;
      return textRange(node, len - 1, len).getBoundingClientRect().left > x;
  }
  // Chrome will move positions between lines to the start of the next line
  function isSuspiciousChromeCaretResult(node, offset, x) {
      if (offset != 0)
          return false;
      for (let cur = node;;) {
          let parent = cur.parentNode;
          if (!parent || parent.nodeType != 1 || parent.firstChild != cur)
              return false;
          if (parent.classList.contains("cm-line"))
              break;
          cur = parent;
      }
      let rect = node.nodeType == 1 ? node.getBoundingClientRect()
          : textRange(node, 0, Math.max(node.nodeValue.length, 1)).getBoundingClientRect();
      return x - rect.left > 5;
  }
  function moveToLineBoundary(view, start, forward, includeWrap) {
      let line = view.state.doc.lineAt(start.head);
      let coords = !includeWrap || !view.lineWrapping ? null
          : view.coordsAtPos(start.assoc < 0 && start.head > line.from ? start.head - 1 : start.head);
      if (coords) {
          let editorRect = view.dom.getBoundingClientRect();
          let direction = view.textDirectionAt(line.from);
          let pos = view.posAtCoords({ x: forward == (direction == Direction.LTR) ? editorRect.right - 1 : editorRect.left + 1,
              y: (coords.top + coords.bottom) / 2 });
          if (pos != null)
              return EditorSelection.cursor(pos, forward ? -1 : 1);
      }
      let lineView = LineView.find(view.docView, start.head);
      let end = lineView ? (forward ? lineView.posAtEnd : lineView.posAtStart) : (forward ? line.to : line.from);
      return EditorSelection.cursor(end, forward ? -1 : 1);
  }
  function moveByChar(view, start, forward, by) {
      let line = view.state.doc.lineAt(start.head), spans = view.bidiSpans(line);
      let direction = view.textDirectionAt(line.from);
      for (let cur = start, check = null;;) {
          let next = moveVisually(line, spans, direction, cur, forward), char = movedOver;
          if (!next) {
              if (line.number == (forward ? view.state.doc.lines : 1))
                  return cur;
              char = "\n";
              line = view.state.doc.line(line.number + (forward ? 1 : -1));
              spans = view.bidiSpans(line);
              next = EditorSelection.cursor(forward ? line.from : line.to);
          }
          if (!check) {
              if (!by)
                  return next;
              check = by(char);
          }
          else if (!check(char)) {
              return cur;
          }
          cur = next;
      }
  }
  function byGroup(view, pos, start) {
      let categorize = view.state.charCategorizer(pos);
      let cat = categorize(start);
      return (next) => {
          let nextCat = categorize(next);
          if (cat == CharCategory.Space)
              cat = nextCat;
          return cat == nextCat;
      };
  }
  function moveVertically(view, start, forward, distance) {
      let startPos = start.head, dir = forward ? 1 : -1;
      if (startPos == (forward ? view.state.doc.length : 0))
          return EditorSelection.cursor(startPos, start.assoc);
      let goal = start.goalColumn, startY;
      let rect = view.contentDOM.getBoundingClientRect();
      let startCoords = view.coordsAtPos(startPos), docTop = view.documentTop;
      if (startCoords) {
          if (goal == null)
              goal = startCoords.left - rect.left;
          startY = dir < 0 ? startCoords.top : startCoords.bottom;
      }
      else {
          let line = view.viewState.lineBlockAt(startPos);
          if (goal == null)
              goal = Math.min(rect.right - rect.left, view.defaultCharacterWidth * (startPos - line.from));
          startY = (dir < 0 ? line.top : line.bottom) + docTop;
      }
      let resolvedGoal = rect.left + goal;
      let dist = distance !== null && distance !== void 0 ? distance : (view.defaultLineHeight >> 1);
      for (let extra = 0;; extra += 10) {
          let curY = startY + (dist + extra) * dir;
          let pos = posAtCoords(view, { x: resolvedGoal, y: curY }, false, dir);
          if (curY < rect.top || curY > rect.bottom || (dir < 0 ? pos < startPos : pos > startPos))
              return EditorSelection.cursor(pos, start.assoc, undefined, goal);
      }
  }
  function skipAtoms(view, oldPos, pos) {
      let atoms = view.state.facet(atomicRanges).map(f => f(view));
      for (;;) {
          let moved = false;
          for (let set of atoms) {
              set.between(pos.from - 1, pos.from + 1, (from, to, value) => {
                  if (pos.from > from && pos.from < to) {
                      pos = oldPos.head > pos.from ? EditorSelection.cursor(from, 1) : EditorSelection.cursor(to, -1);
                      moved = true;
                  }
              });
          }
          if (!moved)
              return pos;
      }
  }

  // This will also be where dragging info and such goes
  class InputState {
      constructor(view) {
          this.lastKeyCode = 0;
          this.lastKeyTime = 0;
          this.lastTouchTime = 0;
          this.lastFocusTime = 0;
          this.lastScrollTop = 0;
          this.lastScrollLeft = 0;
          this.chromeScrollHack = -1;
          // On iOS, some keys need to have their default behavior happen
          // (after which we retroactively handle them and reset the DOM) to
          // avoid messing up the virtual keyboard state.
          this.pendingIOSKey = undefined;
          this.lastSelectionOrigin = null;
          this.lastSelectionTime = 0;
          this.lastEscPress = 0;
          this.lastContextMenu = 0;
          this.scrollHandlers = [];
          this.registeredEvents = [];
          this.customHandlers = [];
          // -1 means not in a composition. Otherwise, this counts the number
          // of changes made during the composition. The count is used to
          // avoid treating the start state of the composition, before any
          // changes have been made, as part of the composition.
          this.composing = -1;
          // Tracks whether the next change should be marked as starting the
          // composition (null means no composition, true means next is the
          // first, false means first has already been marked for this
          // composition)
          this.compositionFirstChange = null;
          this.compositionEndedAt = 0;
          this.mouseSelection = null;
          for (let type in handlers) {
              let handler = handlers[type];
              view.contentDOM.addEventListener(type, (event) => {
                  if (!eventBelongsToEditor(view, event) || this.ignoreDuringComposition(event))
                      return;
                  if (type == "keydown" && this.keydown(view, event))
                      return;
                  if (this.mustFlushObserver(event))
                      view.observer.forceFlush();
                  if (this.runCustomHandlers(type, view, event))
                      event.preventDefault();
                  else
                      handler(view, event);
              }, handlerOptions[type]);
              this.registeredEvents.push(type);
          }
          if (browser.chrome && browser.chrome_version == 102) { // FIXME remove at some point
              // On Chrome 102, viewport updates somehow stop wheel-based
              // scrolling. Turning off pointer events during the scroll seems
              // to avoid the issue.
              view.scrollDOM.addEventListener("wheel", () => {
                  if (this.chromeScrollHack < 0)
                      view.contentDOM.style.pointerEvents = "none";
                  else
                      window.clearTimeout(this.chromeScrollHack);
                  this.chromeScrollHack = setTimeout(() => {
                      this.chromeScrollHack = -1;
                      view.contentDOM.style.pointerEvents = "";
                  }, 100);
              }, { passive: true });
          }
          this.notifiedFocused = view.hasFocus;
          // On Safari adding an input event handler somehow prevents an
          // issue where the composition vanishes when you press enter.
          if (browser.safari)
              view.contentDOM.addEventListener("input", () => null);
      }
      setSelectionOrigin(origin) {
          this.lastSelectionOrigin = origin;
          this.lastSelectionTime = Date.now();
      }
      ensureHandlers(view, plugins) {
          var _a;
          let handlers;
          this.customHandlers = [];
          for (let plugin of plugins)
              if (handlers = (_a = plugin.update(view).spec) === null || _a === void 0 ? void 0 : _a.domEventHandlers) {
                  this.customHandlers.push({ plugin: plugin.value, handlers });
                  for (let type in handlers)
                      if (this.registeredEvents.indexOf(type) < 0 && type != "scroll") {
                          this.registeredEvents.push(type);
                          view.contentDOM.addEventListener(type, (event) => {
                              if (!eventBelongsToEditor(view, event))
                                  return;
                              if (this.runCustomHandlers(type, view, event))
                                  event.preventDefault();
                          });
                      }
              }
      }
      runCustomHandlers(type, view, event) {
          for (let set of this.customHandlers) {
              let handler = set.handlers[type];
              if (handler) {
                  try {
                      if (handler.call(set.plugin, event, view) || event.defaultPrevented)
                          return true;
                  }
                  catch (e) {
                      logException(view.state, e);
                  }
              }
          }
          return false;
      }
      runScrollHandlers(view, event) {
          this.lastScrollTop = view.scrollDOM.scrollTop;
          this.lastScrollLeft = view.scrollDOM.scrollLeft;
          for (let set of this.customHandlers) {
              let handler = set.handlers.scroll;
              if (handler) {
                  try {
                      handler.call(set.plugin, event, view);
                  }
                  catch (e) {
                      logException(view.state, e);
                  }
              }
          }
      }
      keydown(view, event) {
          // Must always run, even if a custom handler handled the event
          this.lastKeyCode = event.keyCode;
          this.lastKeyTime = Date.now();
          if (event.keyCode == 9 && Date.now() < this.lastEscPress + 2000)
              return true;
          // Chrome for Android usually doesn't fire proper key events, but
          // occasionally does, usually surrounded by a bunch of complicated
          // composition changes. When an enter or backspace key event is
          // seen, hold off on handling DOM events for a bit, and then
          // dispatch it.
          if (browser.android && browser.chrome && !event.synthetic &&
              (event.keyCode == 13 || event.keyCode == 8)) {
              view.observer.delayAndroidKey(event.key, event.keyCode);
              return true;
          }
          // Prevent the default behavior of Enter on iOS makes the
          // virtual keyboard get stuck in the wrong (lowercase)
          // state. So we let it go through, and then, in
          // applyDOMChange, notify key handlers of it and reset to
          // the state they produce.
          let pending;
          if (browser.ios && !event.synthetic && !event.altKey && !event.metaKey &&
              ((pending = PendingKeys.find(key => key.keyCode == event.keyCode)) && !event.ctrlKey ||
                  EmacsyPendingKeys.indexOf(event.key) > -1 && event.ctrlKey && !event.shiftKey)) {
              this.pendingIOSKey = pending || event;
              setTimeout(() => this.flushIOSKey(view), 250);
              return true;
          }
          return false;
      }
      flushIOSKey(view) {
          let key = this.pendingIOSKey;
          if (!key)
              return false;
          this.pendingIOSKey = undefined;
          return dispatchKey(view.contentDOM, key.key, key.keyCode);
      }
      ignoreDuringComposition(event) {
          if (!/^key/.test(event.type))
              return false;
          if (this.composing > 0)
              return true;
          // See https://www.stum.de/2016/06/24/handling-ime-events-in-javascript/.
          // On some input method editors (IMEs), the Enter key is used to
          // confirm character selection. On Safari, when Enter is pressed,
          // compositionend and keydown events are sometimes emitted in the
          // wrong order. The key event should still be ignored, even when
          // it happens after the compositionend event.
          if (browser.safari && !browser.ios && Date.now() - this.compositionEndedAt < 100) {
              this.compositionEndedAt = 0;
              return true;
          }
          return false;
      }
      mustFlushObserver(event) {
          return event.type == "keydown" && event.keyCode != 229;
      }
      startMouseSelection(mouseSelection) {
          if (this.mouseSelection)
              this.mouseSelection.destroy();
          this.mouseSelection = mouseSelection;
      }
      update(update) {
          if (this.mouseSelection)
              this.mouseSelection.update(update);
          if (update.transactions.length)
              this.lastKeyCode = this.lastSelectionTime = 0;
      }
      destroy() {
          if (this.mouseSelection)
              this.mouseSelection.destroy();
      }
  }
  const PendingKeys = [
      { key: "Backspace", keyCode: 8, inputType: "deleteContentBackward" },
      { key: "Enter", keyCode: 13, inputType: "insertParagraph" },
      { key: "Delete", keyCode: 46, inputType: "deleteContentForward" }
  ];
  const EmacsyPendingKeys = "dthko";
  // Key codes for modifier keys
  const modifierCodes = [16, 17, 18, 20, 91, 92, 224, 225];
  class MouseSelection {
      constructor(view, startEvent, style, mustSelect) {
          this.view = view;
          this.style = style;
          this.mustSelect = mustSelect;
          this.lastEvent = startEvent;
          let doc = view.contentDOM.ownerDocument;
          doc.addEventListener("mousemove", this.move = this.move.bind(this));
          doc.addEventListener("mouseup", this.up = this.up.bind(this));
          this.extend = startEvent.shiftKey;
          this.multiple = view.state.facet(EditorState.allowMultipleSelections) && addsSelectionRange(view, startEvent);
          this.dragMove = dragMovesSelection(view, startEvent);
          this.dragging = isInPrimarySelection(view, startEvent) && getClickType(startEvent) == 1 ? null : false;
          // When clicking outside of the selection, immediately apply the
          // effect of starting the selection
          if (this.dragging === false) {
              startEvent.preventDefault();
              this.select(startEvent);
          }
      }
      move(event) {
          if (event.buttons == 0)
              return this.destroy();
          if (this.dragging !== false)
              return;
          this.select(this.lastEvent = event);
      }
      up(event) {
          if (this.dragging == null)
              this.select(this.lastEvent);
          if (!this.dragging)
              event.preventDefault();
          this.destroy();
      }
      destroy() {
          let doc = this.view.contentDOM.ownerDocument;
          doc.removeEventListener("mousemove", this.move);
          doc.removeEventListener("mouseup", this.up);
          this.view.inputState.mouseSelection = null;
      }
      select(event) {
          let selection = this.style.get(event, this.extend, this.multiple);
          if (this.mustSelect || !selection.eq(this.view.state.selection) ||
              selection.main.assoc != this.view.state.selection.main.assoc)
              this.view.dispatch({
                  selection,
                  userEvent: "select.pointer",
                  scrollIntoView: true
              });
          this.mustSelect = false;
      }
      update(update) {
          if (update.docChanged && this.dragging)
              this.dragging = this.dragging.map(update.changes);
          if (this.style.update(update))
              setTimeout(() => this.select(this.lastEvent), 20);
      }
  }
  function addsSelectionRange(view, event) {
      let facet = view.state.facet(clickAddsSelectionRange);
      return facet.length ? facet[0](event) : browser.mac ? event.metaKey : event.ctrlKey;
  }
  function dragMovesSelection(view, event) {
      let facet = view.state.facet(dragMovesSelection$1);
      return facet.length ? facet[0](event) : browser.mac ? !event.altKey : !event.ctrlKey;
  }
  function isInPrimarySelection(view, event) {
      let { main } = view.state.selection;
      if (main.empty)
          return false;
      // On boundary clicks, check whether the coordinates are inside the
      // selection's client rectangles
      let sel = getSelection(view.root);
      if (!sel || sel.rangeCount == 0)
          return true;
      let rects = sel.getRangeAt(0).getClientRects();
      for (let i = 0; i < rects.length; i++) {
          let rect = rects[i];
          if (rect.left <= event.clientX && rect.right >= event.clientX &&
              rect.top <= event.clientY && rect.bottom >= event.clientY)
              return true;
      }
      return false;
  }
  function eventBelongsToEditor(view, event) {
      if (!event.bubbles)
          return true;
      if (event.defaultPrevented)
          return false;
      for (let node = event.target, cView; node != view.contentDOM; node = node.parentNode)
          if (!node || node.nodeType == 11 || ((cView = ContentView.get(node)) && cView.ignoreEvent(event)))
              return false;
      return true;
  }
  const handlers = /*@__PURE__*/Object.create(null);
  const handlerOptions = /*@__PURE__*/Object.create(null);
  // This is very crude, but unfortunately both these browsers _pretend_
  // that they have a clipboard API—all the objects and methods are
  // there, they just don't work, and they are hard to test.
  const brokenClipboardAPI = (browser.ie && browser.ie_version < 15) ||
      (browser.ios && browser.webkit_version < 604);
  function capturePaste(view) {
      let parent = view.dom.parentNode;
      if (!parent)
          return;
      let target = parent.appendChild(document.createElement("textarea"));
      target.style.cssText = "position: fixed; left: -10000px; top: 10px";
      target.focus();
      setTimeout(() => {
          view.focus();
          target.remove();
          doPaste(view, target.value);
      }, 50);
  }
  function doPaste(view, input) {
      let { state } = view, changes, i = 1, text = state.toText(input);
      let byLine = text.lines == state.selection.ranges.length;
      let linewise = lastLinewiseCopy != null && state.selection.ranges.every(r => r.empty) && lastLinewiseCopy == text.toString();
      if (linewise) {
          let lastLine = -1;
          changes = state.changeByRange(range => {
              let line = state.doc.lineAt(range.from);
              if (line.from == lastLine)
                  return { range };
              lastLine = line.from;
              let insert = state.toText((byLine ? text.line(i++).text : input) + state.lineBreak);
              return { changes: { from: line.from, insert },
                  range: EditorSelection.cursor(range.from + insert.length) };
          });
      }
      else if (byLine) {
          changes = state.changeByRange(range => {
              let line = text.line(i++);
              return { changes: { from: range.from, to: range.to, insert: line.text },
                  range: EditorSelection.cursor(range.from + line.length) };
          });
      }
      else {
          changes = state.replaceSelection(text);
      }
      view.dispatch(changes, {
          userEvent: "input.paste",
          scrollIntoView: true
      });
  }
  handlers.keydown = (view, event) => {
      view.inputState.setSelectionOrigin("select");
      if (event.keyCode == 27)
          view.inputState.lastEscPress = Date.now();
      else if (modifierCodes.indexOf(event.keyCode) < 0)
          view.inputState.lastEscPress = 0;
  };
  handlers.touchstart = (view, e) => {
      view.inputState.lastTouchTime = Date.now();
      view.inputState.setSelectionOrigin("select.pointer");
  };
  handlers.touchmove = view => {
      view.inputState.setSelectionOrigin("select.pointer");
  };
  handlerOptions.touchstart = handlerOptions.touchmove = { passive: true };
  handlers.mousedown = (view, event) => {
      view.observer.flush();
      if (view.inputState.lastTouchTime > Date.now() - 2000)
          return; // Ignore touch interaction
      let style = null;
      for (let makeStyle of view.state.facet(mouseSelectionStyle)) {
          style = makeStyle(view, event);
          if (style)
              break;
      }
      if (!style && event.button == 0)
          style = basicMouseSelection(view, event);
      if (style) {
          let mustFocus = view.root.activeElement != view.contentDOM;
          if (mustFocus)
              view.observer.ignore(() => focusPreventScroll(view.contentDOM));
          view.inputState.startMouseSelection(new MouseSelection(view, event, style, mustFocus));
      }
  };
  function rangeForClick(view, pos, bias, type) {
      if (type == 1) { // Single click
          return EditorSelection.cursor(pos, bias);
      }
      else if (type == 2) { // Double click
          return groupAt(view.state, pos, bias);
      }
      else { // Triple click
          let visual = LineView.find(view.docView, pos), line = view.state.doc.lineAt(visual ? visual.posAtEnd : pos);
          let from = visual ? visual.posAtStart : line.from, to = visual ? visual.posAtEnd : line.to;
          if (to < view.state.doc.length && to == line.to)
              to++;
          return EditorSelection.range(from, to);
      }
  }
  let insideY = (y, rect) => y >= rect.top && y <= rect.bottom;
  let inside = (x, y, rect) => insideY(y, rect) && x >= rect.left && x <= rect.right;
  // Try to determine, for the given coordinates, associated with the
  // given position, whether they are related to the element before or
  // the element after the position.
  function findPositionSide(view, pos, x, y) {
      let line = LineView.find(view.docView, pos);
      if (!line)
          return 1;
      let off = pos - line.posAtStart;
      // Line boundaries point into the line
      if (off == 0)
          return 1;
      if (off == line.length)
          return -1;
      // Positions on top of an element point at that element
      let before = line.coordsAt(off, -1);
      if (before && inside(x, y, before))
          return -1;
      let after = line.coordsAt(off, 1);
      if (after && inside(x, y, after))
          return 1;
      // This is probably a line wrap point. Pick before if the point is
      // beside it.
      return before && insideY(y, before) ? -1 : 1;
  }
  function queryPos(view, event) {
      let pos = view.posAtCoords({ x: event.clientX, y: event.clientY }, false);
      return { pos, bias: findPositionSide(view, pos, event.clientX, event.clientY) };
  }
  const BadMouseDetail = browser.ie && browser.ie_version <= 11;
  let lastMouseDown = null, lastMouseDownCount = 0, lastMouseDownTime = 0;
  function getClickType(event) {
      if (!BadMouseDetail)
          return event.detail;
      let last = lastMouseDown, lastTime = lastMouseDownTime;
      lastMouseDown = event;
      lastMouseDownTime = Date.now();
      return lastMouseDownCount = !last || (lastTime > Date.now() - 400 && Math.abs(last.clientX - event.clientX) < 2 &&
          Math.abs(last.clientY - event.clientY) < 2) ? (lastMouseDownCount + 1) % 3 : 1;
  }
  function basicMouseSelection(view, event) {
      let start = queryPos(view, event), type = getClickType(event);
      let startSel = view.state.selection;
      let last = start, lastEvent = event;
      return {
          update(update) {
              if (update.docChanged) {
                  start.pos = update.changes.mapPos(start.pos);
                  startSel = startSel.map(update.changes);
                  lastEvent = null;
              }
          },
          get(event, extend, multiple) {
              let cur;
              if (lastEvent && event.clientX == lastEvent.clientX && event.clientY == lastEvent.clientY)
                  cur = last;
              else {
                  cur = last = queryPos(view, event);
                  lastEvent = event;
              }
              let range = rangeForClick(view, cur.pos, cur.bias, type);
              if (start.pos != cur.pos && !extend) {
                  let startRange = rangeForClick(view, start.pos, start.bias, type);
                  let from = Math.min(startRange.from, range.from), to = Math.max(startRange.to, range.to);
                  range = from < range.from ? EditorSelection.range(from, to) : EditorSelection.range(to, from);
              }
              if (extend)
                  return startSel.replaceRange(startSel.main.extend(range.from, range.to));
              else if (multiple && startSel.ranges.length > 1 && startSel.ranges.some(r => r.eq(range)))
                  return removeRange(startSel, range);
              else if (multiple)
                  return startSel.addRange(range);
              else
                  return EditorSelection.create([range]);
          }
      };
  }
  function removeRange(sel, range) {
      for (let i = 0;; i++) {
          if (sel.ranges[i].eq(range))
              return EditorSelection.create(sel.ranges.slice(0, i).concat(sel.ranges.slice(i + 1)), sel.mainIndex == i ? 0 : sel.mainIndex - (sel.mainIndex > i ? 1 : 0));
      }
  }
  handlers.dragstart = (view, event) => {
      let { selection: { main } } = view.state;
      let { mouseSelection } = view.inputState;
      if (mouseSelection)
          mouseSelection.dragging = main;
      if (event.dataTransfer) {
          event.dataTransfer.setData("Text", view.state.sliceDoc(main.from, main.to));
          event.dataTransfer.effectAllowed = "copyMove";
      }
  };
  function dropText(view, event, text, direct) {
      if (!text)
          return;
      let dropPos = view.posAtCoords({ x: event.clientX, y: event.clientY }, false);
      event.preventDefault();
      let { mouseSelection } = view.inputState;
      let del = direct && mouseSelection && mouseSelection.dragging && mouseSelection.dragMove ?
          { from: mouseSelection.dragging.from, to: mouseSelection.dragging.to } : null;
      let ins = { from: dropPos, insert: text };
      let changes = view.state.changes(del ? [del, ins] : ins);
      view.focus();
      view.dispatch({
          changes,
          selection: { anchor: changes.mapPos(dropPos, -1), head: changes.mapPos(dropPos, 1) },
          userEvent: del ? "move.drop" : "input.drop"
      });
  }
  handlers.drop = (view, event) => {
      if (!event.dataTransfer)
          return;
      if (view.state.readOnly)
          return event.preventDefault();
      let files = event.dataTransfer.files;
      if (files && files.length) { // For a file drop, read the file's text.
          event.preventDefault();
          let text = Array(files.length), read = 0;
          let finishFile = () => {
              if (++read == files.length)
                  dropText(view, event, text.filter(s => s != null).join(view.state.lineBreak), false);
          };
          for (let i = 0; i < files.length; i++) {
              let reader = new FileReader;
              reader.onerror = finishFile;
              reader.onload = () => {
                  if (!/[\x00-\x08\x0e-\x1f]{2}/.test(reader.result))
                      text[i] = reader.result;
                  finishFile();
              };
              reader.readAsText(files[i]);
          }
      }
      else {
          dropText(view, event, event.dataTransfer.getData("Text"), true);
      }
  };
  handlers.paste = (view, event) => {
      if (view.state.readOnly)
          return event.preventDefault();
      view.observer.flush();
      let data = brokenClipboardAPI ? null : event.clipboardData;
      if (data) {
          doPaste(view, data.getData("text/plain"));
          event.preventDefault();
      }
      else {
          capturePaste(view);
      }
  };
  function captureCopy(view, text) {
      // The extra wrapper is somehow necessary on IE/Edge to prevent the
      // content from being mangled when it is put onto the clipboard
      let parent = view.dom.parentNode;
      if (!parent)
          return;
      let target = parent.appendChild(document.createElement("textarea"));
      target.style.cssText = "position: fixed; left: -10000px; top: 10px";
      target.value = text;
      target.focus();
      target.selectionEnd = text.length;
      target.selectionStart = 0;
      setTimeout(() => {
          target.remove();
          view.focus();
      }, 50);
  }
  function copiedRange(state) {
      let content = [], ranges = [], linewise = false;
      for (let range of state.selection.ranges)
          if (!range.empty) {
              content.push(state.sliceDoc(range.from, range.to));
              ranges.push(range);
          }
      if (!content.length) {
          // Nothing selected, do a line-wise copy
          let upto = -1;
          for (let { from } of state.selection.ranges) {
              let line = state.doc.lineAt(from);
              if (line.number > upto) {
                  content.push(line.text);
                  ranges.push({ from: line.from, to: Math.min(state.doc.length, line.to + 1) });
              }
              upto = line.number;
          }
          linewise = true;
      }
      return { text: content.join(state.lineBreak), ranges, linewise };
  }
  let lastLinewiseCopy = null;
  handlers.copy = handlers.cut = (view, event) => {
      let { text, ranges, linewise } = copiedRange(view.state);
      if (!text && !linewise)
          return;
      lastLinewiseCopy = linewise ? text : null;
      let data = brokenClipboardAPI ? null : event.clipboardData;
      if (data) {
          event.preventDefault();
          data.clearData();
          data.setData("text/plain", text);
      }
      else {
          captureCopy(view, text);
      }
      if (event.type == "cut" && !view.state.readOnly)
          view.dispatch({
              changes: ranges,
              scrollIntoView: true,
              userEvent: "delete.cut"
          });
  };
  function updateForFocusChange(view) {
      setTimeout(() => {
          if (view.hasFocus != view.inputState.notifiedFocused)
              view.update([]);
      }, 10);
  }
  handlers.focus = view => {
      view.inputState.lastFocusTime = Date.now();
      // When focusing reset the scroll position, move it back to where it was
      if (!view.scrollDOM.scrollTop && (view.inputState.lastScrollTop || view.inputState.lastScrollLeft)) {
          view.scrollDOM.scrollTop = view.inputState.lastScrollTop;
          view.scrollDOM.scrollLeft = view.inputState.lastScrollLeft;
      }
      updateForFocusChange(view);
  };
  handlers.blur = view => {
      view.observer.clearSelectionRange();
      updateForFocusChange(view);
  };
  handlers.compositionstart = handlers.compositionupdate = view => {
      if (view.inputState.compositionFirstChange == null)
          view.inputState.compositionFirstChange = true;
      if (view.inputState.composing < 0) {
          // FIXME possibly set a timeout to clear it again on Android
          view.inputState.composing = 0;
      }
  };
  handlers.compositionend = view => {
      view.inputState.composing = -1;
      view.inputState.compositionEndedAt = Date.now();
      view.inputState.compositionFirstChange = null;
      if (browser.chrome && browser.android)
          view.observer.flushSoon();
      setTimeout(() => {
          // Force the composition state to be cleared if it hasn't already been
          if (view.inputState.composing < 0 && view.docView.compositionDeco.size)
              view.update([]);
      }, 50);
  };
  handlers.contextmenu = view => {
      view.inputState.lastContextMenu = Date.now();
  };
  handlers.beforeinput = (view, event) => {
      var _a;
      // Because Chrome Android doesn't fire useful key events, use
      // beforeinput to detect backspace (and possibly enter and delete,
      // but those usually don't even seem to fire beforeinput events at
      // the moment) and fake a key event for it.
      //
      // (preventDefault on beforeinput, though supported in the spec,
      // seems to do nothing at all on Chrome).
      let pending;
      if (browser.chrome && browser.android && (pending = PendingKeys.find(key => key.inputType == event.inputType))) {
          view.observer.delayAndroidKey(pending.key, pending.keyCode);
          if (pending.key == "Backspace" || pending.key == "Delete") {
              let startViewHeight = ((_a = window.visualViewport) === null || _a === void 0 ? void 0 : _a.height) || 0;
              setTimeout(() => {
                  var _a;
                  // Backspacing near uneditable nodes on Chrome Android sometimes
                  // closes the virtual keyboard. This tries to crudely detect
                  // that and refocus to get it back.
                  if ((((_a = window.visualViewport) === null || _a === void 0 ? void 0 : _a.height) || 0) > startViewHeight + 10 && view.hasFocus) {
                      view.contentDOM.blur();
                      view.focus();
                  }
              }, 100);
          }
      }
  };

  const wrappingWhiteSpace = ["pre-wrap", "normal", "pre-line", "break-spaces"];
  class HeightOracle {
      constructor(lineWrapping) {
          this.lineWrapping = lineWrapping;
          this.doc = Text$1.empty;
          this.heightSamples = {};
          this.lineHeight = 14;
          this.charWidth = 7;
          this.lineLength = 30;
          // Used to track, during updateHeight, if any actual heights changed
          this.heightChanged = false;
      }
      heightForGap(from, to) {
          let lines = this.doc.lineAt(to).number - this.doc.lineAt(from).number + 1;
          if (this.lineWrapping)
              lines += Math.ceil(((to - from) - (lines * this.lineLength * 0.5)) / this.lineLength);
          return this.lineHeight * lines;
      }
      heightForLine(length) {
          if (!this.lineWrapping)
              return this.lineHeight;
          let lines = 1 + Math.max(0, Math.ceil((length - this.lineLength) / (this.lineLength - 5)));
          return lines * this.lineHeight;
      }
      setDoc(doc) { this.doc = doc; return this; }
      mustRefreshForWrapping(whiteSpace) {
          return (wrappingWhiteSpace.indexOf(whiteSpace) > -1) != this.lineWrapping;
      }
      mustRefreshForHeights(lineHeights) {
          let newHeight = false;
          for (let i = 0; i < lineHeights.length; i++) {
              let h = lineHeights[i];
              if (h < 0) {
                  i++;
              }
              else if (!this.heightSamples[Math.floor(h * 10)]) { // Round to .1 pixels
                  newHeight = true;
                  this.heightSamples[Math.floor(h * 10)] = true;
              }
          }
          return newHeight;
      }
      refresh(whiteSpace, lineHeight, charWidth, lineLength, knownHeights) {
          let lineWrapping = wrappingWhiteSpace.indexOf(whiteSpace) > -1;
          let changed = Math.round(lineHeight) != Math.round(this.lineHeight) || this.lineWrapping != lineWrapping;
          this.lineWrapping = lineWrapping;
          this.lineHeight = lineHeight;
          this.charWidth = charWidth;
          this.lineLength = lineLength;
          if (changed) {
              this.heightSamples = {};
              for (let i = 0; i < knownHeights.length; i++) {
                  let h = knownHeights[i];
                  if (h < 0)
                      i++;
                  else
                      this.heightSamples[Math.floor(h * 10)] = true;
              }
          }
          return changed;
      }
  }
  // This object is used by `updateHeight` to make DOM measurements
  // arrive at the right nides. The `heights` array is a sequence of
  // block heights, starting from position `from`.
  class MeasuredHeights {
      constructor(from, heights) {
          this.from = from;
          this.heights = heights;
          this.index = 0;
      }
      get more() { return this.index < this.heights.length; }
  }
  /**
  Record used to represent information about a block-level element
  in the editor view.
  */
  class BlockInfo {
      /**
      @internal
      */
      constructor(
      /**
      The start of the element in the document.
      */
      from, 
      /**
      The length of the element.
      */
      length, 
      /**
      The top position of the element (relative to the top of the
      document).
      */
      top, 
      /**
      Its height.
      */
      height, 
      /**
      The type of element this is. When querying lines, this may be
      an array of all the blocks that make up the line.
      */
      type) {
          this.from = from;
          this.length = length;
          this.top = top;
          this.height = height;
          this.type = type;
      }
      /**
      The end of the element as a document position.
      */
      get to() { return this.from + this.length; }
      /**
      The bottom position of the element.
      */
      get bottom() { return this.top + this.height; }
      /**
      @internal
      */
      join(other) {
          let detail = (Array.isArray(this.type) ? this.type : [this])
              .concat(Array.isArray(other.type) ? other.type : [other]);
          return new BlockInfo(this.from, this.length + other.length, this.top, this.height + other.height, detail);
      }
  }
  var QueryType = /*@__PURE__*/(function (QueryType) {
      QueryType[QueryType["ByPos"] = 0] = "ByPos";
      QueryType[QueryType["ByHeight"] = 1] = "ByHeight";
      QueryType[QueryType["ByPosNoHeight"] = 2] = "ByPosNoHeight";
  return QueryType})(QueryType || (QueryType = {}));
  const Epsilon = 1e-3;
  class HeightMap {
      constructor(length, // The number of characters covered
      height, // Height of this part of the document
      flags = 2 /* Flag.Outdated */) {
          this.length = length;
          this.height = height;
          this.flags = flags;
      }
      get outdated() { return (this.flags & 2 /* Flag.Outdated */) > 0; }
      set outdated(value) { this.flags = (value ? 2 /* Flag.Outdated */ : 0) | (this.flags & ~2 /* Flag.Outdated */); }
      setHeight(oracle, height) {
          if (this.height != height) {
              if (Math.abs(this.height - height) > Epsilon)
                  oracle.heightChanged = true;
              this.height = height;
          }
      }
      // Base case is to replace a leaf node, which simply builds a tree
      // from the new nodes and returns that (HeightMapBranch and
      // HeightMapGap override this to actually use from/to)
      replace(_from, _to, nodes) {
          return HeightMap.of(nodes);
      }
      // Again, these are base cases, and are overridden for branch and gap nodes.
      decomposeLeft(_to, result) { result.push(this); }
      decomposeRight(_from, result) { result.push(this); }
      applyChanges(decorations, oldDoc, oracle, changes) {
          let me = this;
          for (let i = changes.length - 1; i >= 0; i--) {
              let { fromA, toA, fromB, toB } = changes[i];
              let start = me.lineAt(fromA, QueryType.ByPosNoHeight, oldDoc, 0, 0);
              let end = start.to >= toA ? start : me.lineAt(toA, QueryType.ByPosNoHeight, oldDoc, 0, 0);
              toB += end.to - toA;
              toA = end.to;
              while (i > 0 && start.from <= changes[i - 1].toA) {
                  fromA = changes[i - 1].fromA;
                  fromB = changes[i - 1].fromB;
                  i--;
                  if (fromA < start.from)
                      start = me.lineAt(fromA, QueryType.ByPosNoHeight, oldDoc, 0, 0);
              }
              fromB += start.from - fromA;
              fromA = start.from;
              let nodes = NodeBuilder.build(oracle, decorations, fromB, toB);
              me = me.replace(fromA, toA, nodes);
          }
          return me.updateHeight(oracle, 0);
      }
      static empty() { return new HeightMapText(0, 0); }
      // nodes uses null values to indicate the position of line breaks.
      // There are never line breaks at the start or end of the array, or
      // two line breaks next to each other, and the array isn't allowed
      // to be empty (same restrictions as return value from the builder).
      static of(nodes) {
          if (nodes.length == 1)
              return nodes[0];
          let i = 0, j = nodes.length, before = 0, after = 0;
          for (;;) {
              if (i == j) {
                  if (before > after * 2) {
                      let split = nodes[i - 1];
                      if (split.break)
                          nodes.splice(--i, 1, split.left, null, split.right);
                      else
                          nodes.splice(--i, 1, split.left, split.right);
                      j += 1 + split.break;
                      before -= split.size;
                  }
                  else if (after > before * 2) {
                      let split = nodes[j];
                      if (split.break)
                          nodes.splice(j, 1, split.left, null, split.right);
                      else
                          nodes.splice(j, 1, split.left, split.right);
                      j += 2 + split.break;
                      after -= split.size;
                  }
                  else {
                      break;
                  }
              }
              else if (before < after) {
                  let next = nodes[i++];
                  if (next)
                      before += next.size;
              }
              else {
                  let next = nodes[--j];
                  if (next)
                      after += next.size;
              }
          }
          let brk = 0;
          if (nodes[i - 1] == null) {
              brk = 1;
              i--;
          }
          else if (nodes[i] == null) {
              brk = 1;
              j++;
          }
          return new HeightMapBranch(HeightMap.of(nodes.slice(0, i)), brk, HeightMap.of(nodes.slice(j)));
      }
  }
  HeightMap.prototype.size = 1;
  class HeightMapBlock extends HeightMap {
      constructor(length, height, type) {
          super(length, height);
          this.type = type;
      }
      blockAt(_height, _doc, top, offset) {
          return new BlockInfo(offset, this.length, top, this.height, this.type);
      }
      lineAt(_value, _type, doc, top, offset) {
          return this.blockAt(0, doc, top, offset);
      }
      forEachLine(from, to, doc, top, offset, f) {
          if (from <= offset + this.length && to >= offset)
              f(this.blockAt(0, doc, top, offset));
      }
      updateHeight(oracle, offset = 0, _force = false, measured) {
          if (measured && measured.from <= offset && measured.more)
              this.setHeight(oracle, measured.heights[measured.index++]);
          this.outdated = false;
          return this;
      }
      toString() { return `block(${this.length})`; }
  }
  class HeightMapText extends HeightMapBlock {
      constructor(length, height) {
          super(length, height, BlockType.Text);
          this.collapsed = 0; // Amount of collapsed content in the line
          this.widgetHeight = 0; // Maximum inline widget height
      }
      replace(_from, _to, nodes) {
          let node = nodes[0];
          if (nodes.length == 1 && (node instanceof HeightMapText || node instanceof HeightMapGap && (node.flags & 4 /* Flag.SingleLine */)) &&
              Math.abs(this.length - node.length) < 10) {
              if (node instanceof HeightMapGap)
                  node = new HeightMapText(node.length, this.height);
              else
                  node.height = this.height;
              if (!this.outdated)
                  node.outdated = false;
              return node;
          }
          else {
              return HeightMap.of(nodes);
          }
      }
      updateHeight(oracle, offset = 0, force = false, measured) {
          if (measured && measured.from <= offset && measured.more)
              this.setHeight(oracle, measured.heights[measured.index++]);
          else if (force || this.outdated)
              this.setHeight(oracle, Math.max(this.widgetHeight, oracle.heightForLine(this.length - this.collapsed)));
          this.outdated = false;
          return this;
      }
      toString() {
          return `line(${this.length}${this.collapsed ? -this.collapsed : ""}${this.widgetHeight ? ":" + this.widgetHeight : ""})`;
      }
  }
  class HeightMapGap extends HeightMap {
      constructor(length) { super(length, 0); }
      lines(doc, offset) {
          let firstLine = doc.lineAt(offset).number, lastLine = doc.lineAt(offset + this.length).number;
          return { firstLine, lastLine, lineHeight: this.height / (lastLine - firstLine + 1) };
      }
      blockAt(height, doc, top, offset) {
          let { firstLine, lastLine, lineHeight } = this.lines(doc, offset);
          let line = Math.max(0, Math.min(lastLine - firstLine, Math.floor((height - top) / lineHeight)));
          let { from, length } = doc.line(firstLine + line);
          return new BlockInfo(from, length, top + lineHeight * line, lineHeight, BlockType.Text);
      }
      lineAt(value, type, doc, top, offset) {
          if (type == QueryType.ByHeight)
              return this.blockAt(value, doc, top, offset);
          if (type == QueryType.ByPosNoHeight) {
              let { from, to } = doc.lineAt(value);
              return new BlockInfo(from, to - from, 0, 0, BlockType.Text);
          }
          let { firstLine, lineHeight } = this.lines(doc, offset);
          let { from, length, number } = doc.lineAt(value);
          return new BlockInfo(from, length, top + lineHeight * (number - firstLine), lineHeight, BlockType.Text);
      }
      forEachLine(from, to, doc, top, offset, f) {
          let { firstLine, lineHeight } = this.lines(doc, offset);
          for (let pos = Math.max(from, offset), end = Math.min(offset + this.length, to); pos <= end;) {
              let line = doc.lineAt(pos);
              if (pos == from)
                  top += lineHeight * (line.number - firstLine);
              f(new BlockInfo(line.from, line.length, top, lineHeight, BlockType.Text));
              top += lineHeight;
              pos = line.to + 1;
          }
      }
      replace(from, to, nodes) {
          let after = this.length - to;
          if (after > 0) {
              let last = nodes[nodes.length - 1];
              if (last instanceof HeightMapGap)
                  nodes[nodes.length - 1] = new HeightMapGap(last.length + after);
              else
                  nodes.push(null, new HeightMapGap(after - 1));
          }
          if (from > 0) {
              let first = nodes[0];
              if (first instanceof HeightMapGap)
                  nodes[0] = new HeightMapGap(from + first.length);
              else
                  nodes.unshift(new HeightMapGap(from - 1), null);
          }
          return HeightMap.of(nodes);
      }
      decomposeLeft(to, result) {
          result.push(new HeightMapGap(to - 1), null);
      }
      decomposeRight(from, result) {
          result.push(null, new HeightMapGap(this.length - from - 1));
      }
      updateHeight(oracle, offset = 0, force = false, measured) {
          let end = offset + this.length;
          if (measured && measured.from <= offset + this.length && measured.more) {
              // Fill in part of this gap with measured lines. We know there
              // can't be widgets or collapsed ranges in those lines, because
              // they would already have been added to the heightmap (gaps
              // only contain plain text).
              let nodes = [], pos = Math.max(offset, measured.from), singleHeight = -1;
              let wasChanged = oracle.heightChanged;
              if (measured.from > offset)
                  nodes.push(new HeightMapGap(measured.from - offset - 1).updateHeight(oracle, offset));
              while (pos <= end && measured.more) {
                  let len = oracle.doc.lineAt(pos).length;
                  if (nodes.length)
                      nodes.push(null);
                  let height = measured.heights[measured.index++];
                  if (singleHeight == -1)
                      singleHeight = height;
                  else if (Math.abs(height - singleHeight) >= Epsilon)
                      singleHeight = -2;
                  let line = new HeightMapText(len, height);
                  line.outdated = false;
                  nodes.push(line);
                  pos += len + 1;
              }
              if (pos <= end)
                  nodes.push(null, new HeightMapGap(end - pos).updateHeight(oracle, pos));
              let result = HeightMap.of(nodes);
              oracle.heightChanged = wasChanged || singleHeight < 0 || Math.abs(result.height - this.height) >= Epsilon ||
                  Math.abs(singleHeight - this.lines(oracle.doc, offset).lineHeight) >= Epsilon;
              return result;
          }
          else if (force || this.outdated) {
              this.setHeight(oracle, oracle.heightForGap(offset, offset + this.length));
              this.outdated = false;
          }
          return this;
      }
      toString() { return `gap(${this.length})`; }
  }
  class HeightMapBranch extends HeightMap {
      constructor(left, brk, right) {
          super(left.length + brk + right.length, left.height + right.height, brk | (left.outdated || right.outdated ? 2 /* Flag.Outdated */ : 0));
          this.left = left;
          this.right = right;
          this.size = left.size + right.size;
      }
      get break() { return this.flags & 1 /* Flag.Break */; }
      blockAt(height, doc, top, offset) {
          let mid = top + this.left.height;
          return height < mid ? this.left.blockAt(height, doc, top, offset)
              : this.right.blockAt(height, doc, mid, offset + this.left.length + this.break);
      }
      lineAt(value, type, doc, top, offset) {
          let rightTop = top + this.left.height, rightOffset = offset + this.left.length + this.break;
          let left = type == QueryType.ByHeight ? value < rightTop : value < rightOffset;
          let base = left ? this.left.lineAt(value, type, doc, top, offset)
              : this.right.lineAt(value, type, doc, rightTop, rightOffset);
          if (this.break || (left ? base.to < rightOffset : base.from > rightOffset))
              return base;
          let subQuery = type == QueryType.ByPosNoHeight ? QueryType.ByPosNoHeight : QueryType.ByPos;
          if (left)
              return base.join(this.right.lineAt(rightOffset, subQuery, doc, rightTop, rightOffset));
          else
              return this.left.lineAt(rightOffset, subQuery, doc, top, offset).join(base);
      }
      forEachLine(from, to, doc, top, offset, f) {
          let rightTop = top + this.left.height, rightOffset = offset + this.left.length + this.break;
          if (this.break) {
              if (from < rightOffset)
                  this.left.forEachLine(from, to, doc, top, offset, f);
              if (to >= rightOffset)
                  this.right.forEachLine(from, to, doc, rightTop, rightOffset, f);
          }
          else {
              let mid = this.lineAt(rightOffset, QueryType.ByPos, doc, top, offset);
              if (from < mid.from)
                  this.left.forEachLine(from, mid.from - 1, doc, top, offset, f);
              if (mid.to >= from && mid.from <= to)
                  f(mid);
              if (to > mid.to)
                  this.right.forEachLine(mid.to + 1, to, doc, rightTop, rightOffset, f);
          }
      }
      replace(from, to, nodes) {
          let rightStart = this.left.length + this.break;
          if (to < rightStart)
              return this.balanced(this.left.replace(from, to, nodes), this.right);
          if (from > this.left.length)
              return this.balanced(this.left, this.right.replace(from - rightStart, to - rightStart, nodes));
          let result = [];
          if (from > 0)
              this.decomposeLeft(from, result);
          let left = result.length;
          for (let node of nodes)
              result.push(node);
          if (from > 0)
              mergeGaps(result, left - 1);
          if (to < this.length) {
              let right = result.length;
              this.decomposeRight(to, result);
              mergeGaps(result, right);
          }
          return HeightMap.of(result);
      }
      decomposeLeft(to, result) {
          let left = this.left.length;
          if (to <= left)
              return this.left.decomposeLeft(to, result);
          result.push(this.left);
          if (this.break) {
              left++;
              if (to >= left)
                  result.push(null);
          }
          if (to > left)
              this.right.decomposeLeft(to - left, result);
      }
      decomposeRight(from, result) {
          let left = this.left.length, right = left + this.break;
          if (from >= right)
              return this.right.decomposeRight(from - right, result);
          if (from < left)
              this.left.decomposeRight(from, result);
          if (this.break && from < right)
              result.push(null);
          result.push(this.right);
      }
      balanced(left, right) {
          if (left.size > 2 * right.size || right.size > 2 * left.size)
              return HeightMap.of(this.break ? [left, null, right] : [left, right]);
          this.left = left;
          this.right = right;
          this.height = left.height + right.height;
          this.outdated = left.outdated || right.outdated;
          this.size = left.size + right.size;
          this.length = left.length + this.break + right.length;
          return this;
      }
      updateHeight(oracle, offset = 0, force = false, measured) {
          let { left, right } = this, rightStart = offset + left.length + this.break, rebalance = null;
          if (measured && measured.from <= offset + left.length && measured.more)
              rebalance = left = left.updateHeight(oracle, offset, force, measured);
          else
              left.updateHeight(oracle, offset, force);
          if (measured && measured.from <= rightStart + right.length && measured.more)
              rebalance = right = right.updateHeight(oracle, rightStart, force, measured);
          else
              right.updateHeight(oracle, rightStart, force);
          if (rebalance)
              return this.balanced(left, right);
          this.height = this.left.height + this.right.height;
          this.outdated = false;
          return this;
      }
      toString() { return this.left + (this.break ? " " : "-") + this.right; }
  }
  function mergeGaps(nodes, around) {
      let before, after;
      if (nodes[around] == null &&
          (before = nodes[around - 1]) instanceof HeightMapGap &&
          (after = nodes[around + 1]) instanceof HeightMapGap)
          nodes.splice(around - 1, 3, new HeightMapGap(before.length + 1 + after.length));
  }
  const relevantWidgetHeight = 5;
  class NodeBuilder {
      constructor(pos, oracle) {
          this.pos = pos;
          this.oracle = oracle;
          this.nodes = [];
          this.lineStart = -1;
          this.lineEnd = -1;
          this.covering = null;
          this.writtenTo = pos;
      }
      get isCovered() {
          return this.covering && this.nodes[this.nodes.length - 1] == this.covering;
      }
      span(_from, to) {
          if (this.lineStart > -1) {
              let end = Math.min(to, this.lineEnd), last = this.nodes[this.nodes.length - 1];
              if (last instanceof HeightMapText)
                  last.length += end - this.pos;
              else if (end > this.pos || !this.isCovered)
                  this.nodes.push(new HeightMapText(end - this.pos, -1));
              this.writtenTo = end;
              if (to > end) {
                  this.nodes.push(null);
                  this.writtenTo++;
                  this.lineStart = -1;
              }
          }
          this.pos = to;
      }
      point(from, to, deco) {
          if (from < to || deco.heightRelevant) {
              let height = deco.widget ? deco.widget.estimatedHeight : 0;
              if (height < 0)
                  height = this.oracle.lineHeight;
              let len = to - from;
              if (deco.block) {
                  this.addBlock(new HeightMapBlock(len, height, deco.type));
              }
              else if (len || height >= relevantWidgetHeight) {
                  this.addLineDeco(height, len);
              }
          }
          else if (to > from) {
              this.span(from, to);
          }
          if (this.lineEnd > -1 && this.lineEnd < this.pos)
              this.lineEnd = this.oracle.doc.lineAt(this.pos).to;
      }
      enterLine() {
          if (this.lineStart > -1)
              return;
          let { from, to } = this.oracle.doc.lineAt(this.pos);
          this.lineStart = from;
          this.lineEnd = to;
          if (this.writtenTo < from) {
              if (this.writtenTo < from - 1 || this.nodes[this.nodes.length - 1] == null)
                  this.nodes.push(this.blankContent(this.writtenTo, from - 1));
              this.nodes.push(null);
          }
          if (this.pos > from)
              this.nodes.push(new HeightMapText(this.pos - from, -1));
          this.writtenTo = this.pos;
      }
      blankContent(from, to) {
          let gap = new HeightMapGap(to - from);
          if (this.oracle.doc.lineAt(from).to == to)
              gap.flags |= 4 /* Flag.SingleLine */;
          return gap;
      }
      ensureLine() {
          this.enterLine();
          let last = this.nodes.length ? this.nodes[this.nodes.length - 1] : null;
          if (last instanceof HeightMapText)
              return last;
          let line = new HeightMapText(0, -1);
          this.nodes.push(line);
          return line;
      }
      addBlock(block) {
          this.enterLine();
          if (block.type == BlockType.WidgetAfter && !this.isCovered)
              this.ensureLine();
          this.nodes.push(block);
          this.writtenTo = this.pos = this.pos + block.length;
          if (block.type != BlockType.WidgetBefore)
              this.covering = block;
      }
      addLineDeco(height, length) {
          let line = this.ensureLine();
          line.length += length;
          line.collapsed += length;
          line.widgetHeight = Math.max(line.widgetHeight, height);
          this.writtenTo = this.pos = this.pos + length;
      }
      finish(from) {
          let last = this.nodes.length == 0 ? null : this.nodes[this.nodes.length - 1];
          if (this.lineStart > -1 && !(last instanceof HeightMapText) && !this.isCovered)
              this.nodes.push(new HeightMapText(0, -1));
          else if (this.writtenTo < this.pos || last == null)
              this.nodes.push(this.blankContent(this.writtenTo, this.pos));
          let pos = from;
          for (let node of this.nodes) {
              if (node instanceof HeightMapText)
                  node.updateHeight(this.oracle, pos);
              pos += node ? node.length : 1;
          }
          return this.nodes;
      }
      // Always called with a region that on both sides either stretches
      // to a line break or the end of the document.
      // The returned array uses null to indicate line breaks, but never
      // starts or ends in a line break, or has multiple line breaks next
      // to each other.
      static build(oracle, decorations, from, to) {
          let builder = new NodeBuilder(from, oracle);
          RangeSet.spans(decorations, from, to, builder, 0);
          return builder.finish(from);
      }
  }
  function heightRelevantDecoChanges(a, b, diff) {
      let comp = new DecorationComparator;
      RangeSet.compare(a, b, diff, comp, 0);
      return comp.changes;
  }
  class DecorationComparator {
      constructor() {
          this.changes = [];
      }
      compareRange() { }
      comparePoint(from, to, a, b) {
          if (from < to || a && a.heightRelevant || b && b.heightRelevant)
              addRange(from, to, this.changes, 5);
      }
  }

  function visiblePixelRange(dom, paddingTop) {
      let rect = dom.getBoundingClientRect();
      let doc = dom.ownerDocument, win = doc.defaultView || window;
      let left = Math.max(0, rect.left), right = Math.min(win.innerWidth, rect.right);
      let top = Math.max(0, rect.top), bottom = Math.min(win.innerHeight, rect.bottom);
      for (let parent = dom.parentNode; parent && parent != doc.body;) {
          if (parent.nodeType == 1) {
              let elt = parent;
              let style = window.getComputedStyle(elt);
              if ((elt.scrollHeight > elt.clientHeight || elt.scrollWidth > elt.clientWidth) &&
                  style.overflow != "visible") {
                  let parentRect = elt.getBoundingClientRect();
                  left = Math.max(left, parentRect.left);
                  right = Math.min(right, parentRect.right);
                  top = Math.max(top, parentRect.top);
                  bottom = parent == dom.parentNode ? parentRect.bottom : Math.min(bottom, parentRect.bottom);
              }
              parent = style.position == "absolute" || style.position == "fixed" ? elt.offsetParent : elt.parentNode;
          }
          else if (parent.nodeType == 11) { // Shadow root
              parent = parent.host;
          }
          else {
              break;
          }
      }
      return { left: left - rect.left, right: Math.max(left, right) - rect.left,
          top: top - (rect.top + paddingTop), bottom: Math.max(top, bottom) - (rect.top + paddingTop) };
  }
  function fullPixelRange(dom, paddingTop) {
      let rect = dom.getBoundingClientRect();
      return { left: 0, right: rect.right - rect.left,
          top: paddingTop, bottom: rect.bottom - (rect.top + paddingTop) };
  }
  // Line gaps are placeholder widgets used to hide pieces of overlong
  // lines within the viewport, as a kludge to keep the editor
  // responsive when a ridiculously long line is loaded into it.
  class LineGap {
      constructor(from, to, size) {
          this.from = from;
          this.to = to;
          this.size = size;
      }
      static same(a, b) {
          if (a.length != b.length)
              return false;
          for (let i = 0; i < a.length; i++) {
              let gA = a[i], gB = b[i];
              if (gA.from != gB.from || gA.to != gB.to || gA.size != gB.size)
                  return false;
          }
          return true;
      }
      draw(wrapping) {
          return Decoration.replace({ widget: new LineGapWidget(this.size, wrapping) }).range(this.from, this.to);
      }
  }
  class LineGapWidget extends WidgetType {
      constructor(size, vertical) {
          super();
          this.size = size;
          this.vertical = vertical;
      }
      eq(other) { return other.size == this.size && other.vertical == this.vertical; }
      toDOM() {
          let elt = document.createElement("div");
          if (this.vertical) {
              elt.style.height = this.size + "px";
          }
          else {
              elt.style.width = this.size + "px";
              elt.style.height = "2px";
              elt.style.display = "inline-block";
          }
          return elt;
      }
      get estimatedHeight() { return this.vertical ? this.size : -1; }
  }
  class ViewState {
      constructor(state) {
          this.state = state;
          // These are contentDOM-local coordinates
          this.pixelViewport = { left: 0, right: window.innerWidth, top: 0, bottom: 0 };
          this.inView = true;
          this.paddingTop = 0;
          this.paddingBottom = 0;
          this.contentDOMWidth = 0;
          this.contentDOMHeight = 0;
          this.editorHeight = 0;
          this.editorWidth = 0;
          // See VP.MaxDOMHeight
          this.scaler = IdScaler;
          this.scrollTarget = null;
          // Briefly set to true when printing, to disable viewport limiting
          this.printing = false;
          // Flag set when editor content was redrawn, so that the next
          // measure stage knows it must read DOM layout
          this.mustMeasureContent = true;
          this.defaultTextDirection = Direction.LTR;
          this.visibleRanges = [];
          // Cursor 'assoc' is only significant when the cursor is on a line
          // wrap point, where it must stick to the character that it is
          // associated with. Since browsers don't provide a reasonable
          // interface to set or query this, when a selection is set that
          // might cause this to be significant, this flag is set. The next
          // measure phase will check whether the cursor is on a line-wrapping
          // boundary and, if so, reset it to make sure it is positioned in
          // the right place.
          this.mustEnforceCursorAssoc = false;
          let guessWrapping = state.facet(contentAttributes).some(v => typeof v != "function" && v.class == "cm-lineWrapping");
          this.heightOracle = new HeightOracle(guessWrapping);
          this.stateDeco = state.facet(decorations).filter(d => typeof d != "function");
          this.heightMap = HeightMap.empty().applyChanges(this.stateDeco, Text$1.empty, this.heightOracle.setDoc(state.doc), [new ChangedRange(0, 0, 0, state.doc.length)]);
          this.viewport = this.getViewport(0, null);
          this.updateViewportLines();
          this.updateForViewport();
          this.lineGaps = this.ensureLineGaps([]);
          this.lineGapDeco = Decoration.set(this.lineGaps.map(gap => gap.draw(false)));
          this.computeVisibleRanges();
      }
      updateForViewport() {
          let viewports = [this.viewport], { main } = this.state.selection;
          for (let i = 0; i <= 1; i++) {
              let pos = i ? main.head : main.anchor;
              if (!viewports.some(({ from, to }) => pos >= from && pos <= to)) {
                  let { from, to } = this.lineBlockAt(pos);
                  viewports.push(new Viewport(from, to));
              }
          }
          this.viewports = viewports.sort((a, b) => a.from - b.from);
          this.scaler = this.heightMap.height <= 7000000 /* VP.MaxDOMHeight */ ? IdScaler :
              new BigScaler(this.heightOracle.doc, this.heightMap, this.viewports);
      }
      updateViewportLines() {
          this.viewportLines = [];
          this.heightMap.forEachLine(this.viewport.from, this.viewport.to, this.state.doc, 0, 0, block => {
              this.viewportLines.push(this.scaler.scale == 1 ? block : scaleBlock(block, this.scaler));
          });
      }
      update(update, scrollTarget = null) {
          this.state = update.state;
          let prevDeco = this.stateDeco;
          this.stateDeco = this.state.facet(decorations).filter(d => typeof d != "function");
          let contentChanges = update.changedRanges;
          let heightChanges = ChangedRange.extendWithRanges(contentChanges, heightRelevantDecoChanges(prevDeco, this.stateDeco, update ? update.changes : ChangeSet.empty(this.state.doc.length)));
          let prevHeight = this.heightMap.height;
          this.heightMap = this.heightMap.applyChanges(this.stateDeco, update.startState.doc, this.heightOracle.setDoc(this.state.doc), heightChanges);
          if (this.heightMap.height != prevHeight)
              update.flags |= 2 /* UpdateFlag.Height */;
          let viewport = heightChanges.length ? this.mapViewport(this.viewport, update.changes) : this.viewport;
          if (scrollTarget && (scrollTarget.range.head < viewport.from || scrollTarget.range.head > viewport.to) ||
              !this.viewportIsAppropriate(viewport))
              viewport = this.getViewport(0, scrollTarget);
          let updateLines = !update.changes.empty || (update.flags & 2 /* UpdateFlag.Height */) ||
              viewport.from != this.viewport.from || viewport.to != this.viewport.to;
          this.viewport = viewport;
          this.updateForViewport();
          if (updateLines)
              this.updateViewportLines();
          if (this.lineGaps.length || this.viewport.to - this.viewport.from > (2000 /* LG.Margin */ << 1))
              this.updateLineGaps(this.ensureLineGaps(this.mapLineGaps(this.lineGaps, update.changes)));
          update.flags |= this.computeVisibleRanges();
          if (scrollTarget)
              this.scrollTarget = scrollTarget;
          if (!this.mustEnforceCursorAssoc && update.selectionSet && update.view.lineWrapping &&
              update.state.selection.main.empty && update.state.selection.main.assoc &&
              !update.state.facet(nativeSelectionHidden))
              this.mustEnforceCursorAssoc = true;
      }
      measure(view) {
          let dom = view.contentDOM, style = window.getComputedStyle(dom);
          let oracle = this.heightOracle;
          let whiteSpace = style.whiteSpace;
          this.defaultTextDirection = style.direction == "rtl" ? Direction.RTL : Direction.LTR;
          let refresh = this.heightOracle.mustRefreshForWrapping(whiteSpace);
          let measureContent = refresh || this.mustMeasureContent || this.contentDOMHeight != dom.clientHeight;
          this.contentDOMHeight = dom.clientHeight;
          this.mustMeasureContent = false;
          let result = 0, bias = 0;
          // Vertical padding
          let paddingTop = parseInt(style.paddingTop) || 0, paddingBottom = parseInt(style.paddingBottom) || 0;
          if (this.paddingTop != paddingTop || this.paddingBottom != paddingBottom) {
              this.paddingTop = paddingTop;
              this.paddingBottom = paddingBottom;
              result |= 8 /* UpdateFlag.Geometry */ | 2 /* UpdateFlag.Height */;
          }
          if (this.editorWidth != view.scrollDOM.clientWidth) {
              if (oracle.lineWrapping)
                  measureContent = true;
              this.editorWidth = view.scrollDOM.clientWidth;
              result |= 8 /* UpdateFlag.Geometry */;
          }
          // Pixel viewport
          let pixelViewport = (this.printing ? fullPixelRange : visiblePixelRange)(dom, this.paddingTop);
          let dTop = pixelViewport.top - this.pixelViewport.top, dBottom = pixelViewport.bottom - this.pixelViewport.bottom;
          this.pixelViewport = pixelViewport;
          let inView = this.pixelViewport.bottom > this.pixelViewport.top && this.pixelViewport.right > this.pixelViewport.left;
          if (inView != this.inView) {
              this.inView = inView;
              if (inView)
                  measureContent = true;
          }
          if (!this.inView && !this.scrollTarget)
              return 0;
          let contentWidth = dom.clientWidth;
          if (this.contentDOMWidth != contentWidth || this.editorHeight != view.scrollDOM.clientHeight) {
              this.contentDOMWidth = contentWidth;
              this.editorHeight = view.scrollDOM.clientHeight;
              result |= 8 /* UpdateFlag.Geometry */;
          }
          if (measureContent) {
              let lineHeights = view.docView.measureVisibleLineHeights(this.viewport);
              if (oracle.mustRefreshForHeights(lineHeights))
                  refresh = true;
              if (refresh || oracle.lineWrapping && Math.abs(contentWidth - this.contentDOMWidth) > oracle.charWidth) {
                  let { lineHeight, charWidth } = view.docView.measureTextSize();
                  refresh = lineHeight > 0 && oracle.refresh(whiteSpace, lineHeight, charWidth, contentWidth / charWidth, lineHeights);
                  if (refresh) {
                      view.docView.minWidth = 0;
                      result |= 8 /* UpdateFlag.Geometry */;
                  }
              }
              if (dTop > 0 && dBottom > 0)
                  bias = Math.max(dTop, dBottom);
              else if (dTop < 0 && dBottom < 0)
                  bias = Math.min(dTop, dBottom);
              oracle.heightChanged = false;
              for (let vp of this.viewports) {
                  let heights = vp.from == this.viewport.from ? lineHeights : view.docView.measureVisibleLineHeights(vp);
                  this.heightMap = (refresh ? HeightMap.empty().applyChanges(this.stateDeco, Text$1.empty, this.heightOracle, [new ChangedRange(0, 0, 0, view.state.doc.length)]) : this.heightMap).updateHeight(oracle, 0, refresh, new MeasuredHeights(vp.from, heights));
              }
              if (oracle.heightChanged)
                  result |= 2 /* UpdateFlag.Height */;
          }
          let viewportChange = !this.viewportIsAppropriate(this.viewport, bias) ||
              this.scrollTarget && (this.scrollTarget.range.head < this.viewport.from || this.scrollTarget.range.head > this.viewport.to);
          if (viewportChange)
              this.viewport = this.getViewport(bias, this.scrollTarget);
          this.updateForViewport();
          if ((result & 2 /* UpdateFlag.Height */) || viewportChange)
              this.updateViewportLines();
          if (this.lineGaps.length || this.viewport.to - this.viewport.from > (2000 /* LG.Margin */ << 1))
              this.updateLineGaps(this.ensureLineGaps(refresh ? [] : this.lineGaps, view));
          result |= this.computeVisibleRanges();
          if (this.mustEnforceCursorAssoc) {
              this.mustEnforceCursorAssoc = false;
              // This is done in the read stage, because moving the selection
              // to a line end is going to trigger a layout anyway, so it
              // can't be a pure write. It should be rare that it does any
              // writing.
              view.docView.enforceCursorAssoc();
          }
          return result;
      }
      get visibleTop() { return this.scaler.fromDOM(this.pixelViewport.top); }
      get visibleBottom() { return this.scaler.fromDOM(this.pixelViewport.bottom); }
      getViewport(bias, scrollTarget) {
          // This will divide VP.Margin between the top and the
          // bottom, depending on the bias (the change in viewport position
          // since the last update). It'll hold a number between 0 and 1
          let marginTop = 0.5 - Math.max(-0.5, Math.min(0.5, bias / 1000 /* VP.Margin */ / 2));
          let map = this.heightMap, doc = this.state.doc, { visibleTop, visibleBottom } = this;
          let viewport = new Viewport(map.lineAt(visibleTop - marginTop * 1000 /* VP.Margin */, QueryType.ByHeight, doc, 0, 0).from, map.lineAt(visibleBottom + (1 - marginTop) * 1000 /* VP.Margin */, QueryType.ByHeight, doc, 0, 0).to);
          // If scrollTarget is given, make sure the viewport includes that position
          if (scrollTarget) {
              let { head } = scrollTarget.range;
              if (head < viewport.from || head > viewport.to) {
                  let viewHeight = Math.min(this.editorHeight, this.pixelViewport.bottom - this.pixelViewport.top);
                  let block = map.lineAt(head, QueryType.ByPos, doc, 0, 0), topPos;
                  if (scrollTarget.y == "center")
                      topPos = (block.top + block.bottom) / 2 - viewHeight / 2;
                  else if (scrollTarget.y == "start" || scrollTarget.y == "nearest" && head < viewport.from)
                      topPos = block.top;
                  else
                      topPos = block.bottom - viewHeight;
                  viewport = new Viewport(map.lineAt(topPos - 1000 /* VP.Margin */ / 2, QueryType.ByHeight, doc, 0, 0).from, map.lineAt(topPos + viewHeight + 1000 /* VP.Margin */ / 2, QueryType.ByHeight, doc, 0, 0).to);
              }
          }
          return viewport;
      }
      mapViewport(viewport, changes) {
          let from = changes.mapPos(viewport.from, -1), to = changes.mapPos(viewport.to, 1);
          return new Viewport(this.heightMap.lineAt(from, QueryType.ByPos, this.state.doc, 0, 0).from, this.heightMap.lineAt(to, QueryType.ByPos, this.state.doc, 0, 0).to);
      }
      // Checks if a given viewport covers the visible part of the
      // document and not too much beyond that.
      viewportIsAppropriate({ from, to }, bias = 0) {
          if (!this.inView)
              return true;
          let { top } = this.heightMap.lineAt(from, QueryType.ByPos, this.state.doc, 0, 0);
          let { bottom } = this.heightMap.lineAt(to, QueryType.ByPos, this.state.doc, 0, 0);
          let { visibleTop, visibleBottom } = this;
          return (from == 0 || top <= visibleTop - Math.max(10 /* VP.MinCoverMargin */, Math.min(-bias, 250 /* VP.MaxCoverMargin */))) &&
              (to == this.state.doc.length ||
                  bottom >= visibleBottom + Math.max(10 /* VP.MinCoverMargin */, Math.min(bias, 250 /* VP.MaxCoverMargin */))) &&
              (top > visibleTop - 2 * 1000 /* VP.Margin */ && bottom < visibleBottom + 2 * 1000 /* VP.Margin */);
      }
      mapLineGaps(gaps, changes) {
          if (!gaps.length || changes.empty)
              return gaps;
          let mapped = [];
          for (let gap of gaps)
              if (!changes.touchesRange(gap.from, gap.to))
                  mapped.push(new LineGap(changes.mapPos(gap.from), changes.mapPos(gap.to), gap.size));
          return mapped;
      }
      // Computes positions in the viewport where the start or end of a
      // line should be hidden, trying to reuse existing line gaps when
      // appropriate to avoid unneccesary redraws.
      // Uses crude character-counting for the positioning and sizing,
      // since actual DOM coordinates aren't always available and
      // predictable. Relies on generous margins (see LG.Margin) to hide
      // the artifacts this might produce from the user.
      ensureLineGaps(current, mayMeasure) {
          let wrapping = this.heightOracle.lineWrapping;
          let margin = wrapping ? 10000 /* LG.MarginWrap */ : 2000 /* LG.Margin */, halfMargin = margin >> 1, doubleMargin = margin << 1;
          // The non-wrapping logic won't work at all in predominantly right-to-left text.
          if (this.defaultTextDirection != Direction.LTR && !wrapping)
              return [];
          let gaps = [];
          let addGap = (from, to, line, structure) => {
              if (to - from < halfMargin)
                  return;
              let sel = this.state.selection.main, avoid = [sel.from];
              if (!sel.empty)
                  avoid.push(sel.to);
              for (let pos of avoid) {
                  if (pos > from && pos < to) {
                      addGap(from, pos - 10 /* LG.SelectionMargin */, line, structure);
                      addGap(pos + 10 /* LG.SelectionMargin */, to, line, structure);
                      return;
                  }
              }
              let gap = find(current, gap => gap.from >= line.from && gap.to <= line.to &&
                  Math.abs(gap.from - from) < halfMargin && Math.abs(gap.to - to) < halfMargin &&
                  !avoid.some(pos => gap.from < pos && gap.to > pos));
              if (!gap) {
                  // When scrolling down, snap gap ends to line starts to avoid shifts in wrapping
                  if (to < line.to && mayMeasure && wrapping &&
                      mayMeasure.visibleRanges.some(r => r.from <= to && r.to >= to)) {
                      let lineStart = mayMeasure.moveToLineBoundary(EditorSelection.cursor(to), false, true).head;
                      if (lineStart > from)
                          to = lineStart;
                  }
                  gap = new LineGap(from, to, this.gapSize(line, from, to, structure));
              }
              gaps.push(gap);
          };
          for (let line of this.viewportLines) {
              if (line.length < doubleMargin)
                  continue;
              let structure = lineStructure(line.from, line.to, this.stateDeco);
              if (structure.total < doubleMargin)
                  continue;
              let target = this.scrollTarget ? this.scrollTarget.range.head : null;
              let viewFrom, viewTo;
              if (wrapping) {
                  let marginHeight = (margin / this.heightOracle.lineLength) * this.heightOracle.lineHeight;
                  let top, bot;
                  if (target != null) {
                      let targetFrac = findFraction(structure, target);
                      let spaceFrac = ((this.visibleBottom - this.visibleTop) / 2 + marginHeight) / line.height;
                      top = targetFrac - spaceFrac;
                      bot = targetFrac + spaceFrac;
                  }
                  else {
                      top = (this.visibleTop - line.top - marginHeight) / line.height;
                      bot = (this.visibleBottom - line.top + marginHeight) / line.height;
                  }
                  viewFrom = findPosition(structure, top);
                  viewTo = findPosition(structure, bot);
              }
              else {
                  let totalWidth = structure.total * this.heightOracle.charWidth;
                  let marginWidth = margin * this.heightOracle.charWidth;
                  let left, right;
                  if (target != null) {
                      let targetFrac = findFraction(structure, target);
                      let spaceFrac = ((this.pixelViewport.right - this.pixelViewport.left) / 2 + marginWidth) / totalWidth;
                      left = targetFrac - spaceFrac;
                      right = targetFrac + spaceFrac;
                  }
                  else {
                      left = (this.pixelViewport.left - marginWidth) / totalWidth;
                      right = (this.pixelViewport.right + marginWidth) / totalWidth;
                  }
                  viewFrom = findPosition(structure, left);
                  viewTo = findPosition(structure, right);
              }
              if (viewFrom > line.from)
                  addGap(line.from, viewFrom, line, structure);
              if (viewTo < line.to)
                  addGap(viewTo, line.to, line, structure);
          }
          return gaps;
      }
      gapSize(line, from, to, structure) {
          let fraction = findFraction(structure, to) - findFraction(structure, from);
          if (this.heightOracle.lineWrapping) {
              return line.height * fraction;
          }
          else {
              return structure.total * this.heightOracle.charWidth * fraction;
          }
      }
      updateLineGaps(gaps) {
          if (!LineGap.same(gaps, this.lineGaps)) {
              this.lineGaps = gaps;
              this.lineGapDeco = Decoration.set(gaps.map(gap => gap.draw(this.heightOracle.lineWrapping)));
          }
      }
      computeVisibleRanges() {
          let deco = this.stateDeco;
          if (this.lineGaps.length)
              deco = deco.concat(this.lineGapDeco);
          let ranges = [];
          RangeSet.spans(deco, this.viewport.from, this.viewport.to, {
              span(from, to) { ranges.push({ from, to }); },
              point() { }
          }, 20);
          let changed = ranges.length != this.visibleRanges.length ||
              this.visibleRanges.some((r, i) => r.from != ranges[i].from || r.to != ranges[i].to);
          this.visibleRanges = ranges;
          return changed ? 4 /* UpdateFlag.Viewport */ : 0;
      }
      lineBlockAt(pos) {
          return (pos >= this.viewport.from && pos <= this.viewport.to && this.viewportLines.find(b => b.from <= pos && b.to >= pos)) ||
              scaleBlock(this.heightMap.lineAt(pos, QueryType.ByPos, this.state.doc, 0, 0), this.scaler);
      }
      lineBlockAtHeight(height) {
          return scaleBlock(this.heightMap.lineAt(this.scaler.fromDOM(height), QueryType.ByHeight, this.state.doc, 0, 0), this.scaler);
      }
      elementAtHeight(height) {
          return scaleBlock(this.heightMap.blockAt(this.scaler.fromDOM(height), this.state.doc, 0, 0), this.scaler);
      }
      get docHeight() {
          return this.scaler.toDOM(this.heightMap.height);
      }
      get contentHeight() {
          return this.docHeight + this.paddingTop + this.paddingBottom;
      }
  }
  class Viewport {
      constructor(from, to) {
          this.from = from;
          this.to = to;
      }
  }
  function lineStructure(from, to, stateDeco) {
      let ranges = [], pos = from, total = 0;
      RangeSet.spans(stateDeco, from, to, {
          span() { },
          point(from, to) {
              if (from > pos) {
                  ranges.push({ from: pos, to: from });
                  total += from - pos;
              }
              pos = to;
          }
      }, 20); // We're only interested in collapsed ranges of a significant size
      if (pos < to) {
          ranges.push({ from: pos, to });
          total += to - pos;
      }
      return { total, ranges };
  }
  function findPosition({ total, ranges }, ratio) {
      if (ratio <= 0)
          return ranges[0].from;
      if (ratio >= 1)
          return ranges[ranges.length - 1].to;
      let dist = Math.floor(total * ratio);
      for (let i = 0;; i++) {
          let { from, to } = ranges[i], size = to - from;
          if (dist <= size)
              return from + dist;
          dist -= size;
      }
  }
  function findFraction(structure, pos) {
      let counted = 0;
      for (let { from, to } of structure.ranges) {
          if (pos <= to) {
              counted += pos - from;
              break;
          }
          counted += to - from;
      }
      return counted / structure.total;
  }
  function find(array, f) {
      for (let val of array)
          if (f(val))
              return val;
      return undefined;
  }
  // Don't scale when the document height is within the range of what
  // the DOM can handle.
  const IdScaler = {
      toDOM(n) { return n; },
      fromDOM(n) { return n; },
      scale: 1
  };
  // When the height is too big (> VP.MaxDOMHeight), scale down the
  // regions outside the viewports so that the total height is
  // VP.MaxDOMHeight.
  class BigScaler {
      constructor(doc, heightMap, viewports) {
          let vpHeight = 0, base = 0, domBase = 0;
          this.viewports = viewports.map(({ from, to }) => {
              let top = heightMap.lineAt(from, QueryType.ByPos, doc, 0, 0).top;
              let bottom = heightMap.lineAt(to, QueryType.ByPos, doc, 0, 0).bottom;
              vpHeight += bottom - top;
              return { from, to, top, bottom, domTop: 0, domBottom: 0 };
          });
          this.scale = (7000000 /* VP.MaxDOMHeight */ - vpHeight) / (heightMap.height - vpHeight);
          for (let obj of this.viewports) {
              obj.domTop = domBase + (obj.top - base) * this.scale;
              domBase = obj.domBottom = obj.domTop + (obj.bottom - obj.top);
              base = obj.bottom;
          }
      }
      toDOM(n) {
          for (let i = 0, base = 0, domBase = 0;; i++) {
              let vp = i < this.viewports.length ? this.viewports[i] : null;
              if (!vp || n < vp.top)
                  return domBase + (n - base) * this.scale;
              if (n <= vp.bottom)
                  return vp.domTop + (n - vp.top);
              base = vp.bottom;
              domBase = vp.domBottom;
          }
      }
      fromDOM(n) {
          for (let i = 0, base = 0, domBase = 0;; i++) {
              let vp = i < this.viewports.length ? this.viewports[i] : null;
              if (!vp || n < vp.domTop)
                  return base + (n - domBase) / this.scale;
              if (n <= vp.domBottom)
                  return vp.top + (n - vp.domTop);
              base = vp.bottom;
              domBase = vp.domBottom;
          }
      }
  }
  function scaleBlock(block, scaler) {
      if (scaler.scale == 1)
          return block;
      let bTop = scaler.toDOM(block.top), bBottom = scaler.toDOM(block.bottom);
      return new BlockInfo(block.from, block.length, bTop, bBottom - bTop, Array.isArray(block.type) ? block.type.map(b => scaleBlock(b, scaler)) : block.type);
  }

  const theme = /*@__PURE__*/Facet.define({ combine: strs => strs.join(" ") });
  const darkTheme = /*@__PURE__*/Facet.define({ combine: values => values.indexOf(true) > -1 });
  const baseThemeID = /*@__PURE__*/StyleModule.newName(), baseLightID = /*@__PURE__*/StyleModule.newName(), baseDarkID = /*@__PURE__*/StyleModule.newName();
  const lightDarkIDs = { "&light": "." + baseLightID, "&dark": "." + baseDarkID };
  function buildTheme(main, spec, scopes) {
      return new StyleModule(spec, {
          finish(sel) {
              return /&/.test(sel) ? sel.replace(/&\w*/, m => {
                  if (m == "&")
                      return main;
                  if (!scopes || !scopes[m])
                      throw new RangeError(`Unsupported selector: ${m}`);
                  return scopes[m];
              }) : main + " " + sel;
          }
      });
  }
  const baseTheme$1 = /*@__PURE__*/buildTheme("." + baseThemeID, {
      "&.cm-editor": {
          position: "relative !important",
          boxSizing: "border-box",
          "&.cm-focused": {
              // Provide a simple default outline to make sure a focused
              // editor is visually distinct. Can't leave the default behavior
              // because that will apply to the content element, which is
              // inside the scrollable container and doesn't include the
              // gutters. We also can't use an 'auto' outline, since those
              // are, for some reason, drawn behind the element content, which
              // will cause things like the active line background to cover
              // the outline (#297).
              outline: "1px dotted #212121"
          },
          display: "flex !important",
          flexDirection: "column"
      },
      ".cm-scroller": {
          display: "flex !important",
          alignItems: "flex-start !important",
          fontFamily: "monospace",
          lineHeight: 1.4,
          height: "100%",
          overflowX: "auto",
          position: "relative",
          zIndex: 0
      },
      ".cm-content": {
          margin: 0,
          flexGrow: 2,
          flexShrink: 0,
          display: "block",
          whiteSpace: "pre",
          wordWrap: "normal",
          boxSizing: "border-box",
          padding: "4px 0",
          outline: "none",
          "&[contenteditable=true]": {
              WebkitUserModify: "read-write-plaintext-only",
          }
      },
      ".cm-lineWrapping": {
          whiteSpace_fallback: "pre-wrap",
          whiteSpace: "break-spaces",
          wordBreak: "break-word",
          overflowWrap: "anywhere",
          flexShrink: 1
      },
      "&light .cm-content": { caretColor: "black" },
      "&dark .cm-content": { caretColor: "white" },
      ".cm-line": {
          display: "block",
          padding: "0 2px 0 6px"
      },
      ".cm-layer": {
          contain: "size style",
          "& > *": {
              position: "absolute"
          }
      },
      "&light .cm-selectionBackground": {
          background: "#d9d9d9"
      },
      "&dark .cm-selectionBackground": {
          background: "#222"
      },
      "&light.cm-focused .cm-selectionBackground": {
          background: "#d7d4f0"
      },
      "&dark.cm-focused .cm-selectionBackground": {
          background: "#233"
      },
      ".cm-cursorLayer": {
          pointerEvents: "none"
      },
      "&.cm-focused .cm-cursorLayer": {
          animation: "steps(1) cm-blink 1.2s infinite"
      },
      // Two animations defined so that we can switch between them to
      // restart the animation without forcing another style
      // recomputation.
      "@keyframes cm-blink": { "0%": {}, "50%": { opacity: 0 }, "100%": {} },
      "@keyframes cm-blink2": { "0%": {}, "50%": { opacity: 0 }, "100%": {} },
      ".cm-cursor, .cm-dropCursor": {
          borderLeft: "1.2px solid black",
          marginLeft: "-0.6px",
          pointerEvents: "none",
      },
      ".cm-cursor": {
          display: "none"
      },
      "&dark .cm-cursor": {
          borderLeftColor: "#444"
      },
      "&.cm-focused .cm-cursor": {
          display: "block"
      },
      "&light .cm-activeLine": { backgroundColor: "#cceeff44" },
      "&dark .cm-activeLine": { backgroundColor: "#99eeff33" },
      "&light .cm-specialChar": { color: "red" },
      "&dark .cm-specialChar": { color: "#f78" },
      ".cm-gutters": {
          flexShrink: 0,
          display: "flex",
          height: "100%",
          boxSizing: "border-box",
          left: 0,
          zIndex: 200
      },
      "&light .cm-gutters": {
          backgroundColor: "#f5f5f5",
          color: "#6c6c6c",
          borderRight: "1px solid #ddd"
      },
      "&dark .cm-gutters": {
          backgroundColor: "#333338",
          color: "#ccc"
      },
      ".cm-gutter": {
          display: "flex !important",
          flexDirection: "column",
          flexShrink: 0,
          boxSizing: "border-box",
          minHeight: "100%",
          overflow: "hidden"
      },
      ".cm-gutterElement": {
          boxSizing: "border-box"
      },
      ".cm-lineNumbers .cm-gutterElement": {
          padding: "0 3px 0 5px",
          minWidth: "20px",
          textAlign: "right",
          whiteSpace: "nowrap"
      },
      "&light .cm-activeLineGutter": {
          backgroundColor: "#e2f2ff"
      },
      "&dark .cm-activeLineGutter": {
          backgroundColor: "#222227"
      },
      ".cm-panels": {
          boxSizing: "border-box",
          position: "sticky",
          left: 0,
          right: 0
      },
      "&light .cm-panels": {
          backgroundColor: "#f5f5f5",
          color: "black"
      },
      "&light .cm-panels-top": {
          borderBottom: "1px solid #ddd"
      },
      "&light .cm-panels-bottom": {
          borderTop: "1px solid #ddd"
      },
      "&dark .cm-panels": {
          backgroundColor: "#333338",
          color: "white"
      },
      ".cm-tab": {
          display: "inline-block",
          overflow: "hidden",
          verticalAlign: "bottom"
      },
      ".cm-widgetBuffer": {
          verticalAlign: "text-top",
          height: "1em",
          width: 0,
          display: "inline"
      },
      ".cm-placeholder": {
          color: "#888",
          display: "inline-block",
          verticalAlign: "top",
      },
      ".cm-highlightSpace:before": {
          content: "attr(data-display)",
          position: "absolute",
          pointerEvents: "none",
          color: "#888"
      },
      ".cm-highlightTab": {
          backgroundImage: `url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" width="200" height="20"><path stroke="%23888" stroke-width="1" fill="none" d="M1 10H196L190 5M190 15L196 10M197 4L197 16"/></svg>')`,
          backgroundSize: "auto 100%",
          backgroundPosition: "right 90%",
          backgroundRepeat: "no-repeat"
      },
      ".cm-trailingSpace": {
          backgroundColor: "#ff332255"
      },
      ".cm-button": {
          verticalAlign: "middle",
          color: "inherit",
          fontSize: "70%",
          padding: ".2em 1em",
          borderRadius: "1px"
      },
      "&light .cm-button": {
          backgroundImage: "linear-gradient(#eff1f5, #d9d9df)",
          border: "1px solid #888",
          "&:active": {
              backgroundImage: "linear-gradient(#b4b4b4, #d0d3d6)"
          }
      },
      "&dark .cm-button": {
          backgroundImage: "linear-gradient(#393939, #111)",
          border: "1px solid #888",
          "&:active": {
              backgroundImage: "linear-gradient(#111, #333)"
          }
      },
      ".cm-textfield": {
          verticalAlign: "middle",
          color: "inherit",
          fontSize: "70%",
          border: "1px solid silver",
          padding: ".2em .5em"
      },
      "&light .cm-textfield": {
          backgroundColor: "white"
      },
      "&dark .cm-textfield": {
          border: "1px solid #555",
          backgroundColor: "inherit"
      }
  }, lightDarkIDs);

  class DOMChange {
      constructor(view, start, end, typeOver) {
          this.typeOver = typeOver;
          this.bounds = null;
          this.text = "";
          let { impreciseHead: iHead, impreciseAnchor: iAnchor } = view.docView;
          if (view.state.readOnly && start > -1) {
              // Ignore changes when the editor is read-only
              this.newSel = null;
          }
          else if (start > -1 && (this.bounds = view.docView.domBoundsAround(start, end, 0))) {
              let selPoints = iHead || iAnchor ? [] : selectionPoints(view);
              let reader = new DOMReader(selPoints, view.state);
              reader.readRange(this.bounds.startDOM, this.bounds.endDOM);
              this.text = reader.text;
              this.newSel = selectionFromPoints(selPoints, this.bounds.from);
          }
          else {
              let domSel = view.observer.selectionRange;
              let head = iHead && iHead.node == domSel.focusNode && iHead.offset == domSel.focusOffset ||
                  !contains(view.contentDOM, domSel.focusNode)
                  ? view.state.selection.main.head
                  : view.docView.posFromDOM(domSel.focusNode, domSel.focusOffset);
              let anchor = iAnchor && iAnchor.node == domSel.anchorNode && iAnchor.offset == domSel.anchorOffset ||
                  !contains(view.contentDOM, domSel.anchorNode)
                  ? view.state.selection.main.anchor
                  : view.docView.posFromDOM(domSel.anchorNode, domSel.anchorOffset);
              this.newSel = EditorSelection.single(anchor, head);
          }
      }
  }
  function applyDOMChange(view, domChange) {
      let change;
      let { newSel } = domChange, sel = view.state.selection.main;
      if (domChange.bounds) {
          let { from, to } = domChange.bounds;
          let preferredPos = sel.from, preferredSide = null;
          // Prefer anchoring to end when Backspace is pressed (or, on
          // Android, when something was deleted)
          if (view.inputState.lastKeyCode === 8 && view.inputState.lastKeyTime > Date.now() - 100 ||
              browser.android && domChange.text.length < to - from) {
              preferredPos = sel.to;
              preferredSide = "end";
          }
          let diff = findDiff(view.state.doc.sliceString(from, to, LineBreakPlaceholder), domChange.text, preferredPos - from, preferredSide);
          if (diff) {
 