// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.mothra.packer

import java.net.URLEncoder
import java.time.Instant
import java.util.Base64
import org.cert.netsa.data.net.{IPv4Address, IPv6Address}
import org.cert.netsa.io.ipfix.{InfoModel, Record, SimpleFieldExtractor}


/**
  * An implementation of the [[PackingLogic]] trait.  The `pack()`
  * method stores records into files based on a list of field names specified
  * in this class.
  *
  */
private[mothra] case class FlexiblePackLogic(infoModel: InfoModel) extends PackingLogic
{
  /** The observationDomain to use for newly created files. */
  val domain = 0

  /** The version of the packing logic; this value becomes part of the
    * path in the directory hierarchy. */
  val version = 1

  /** The names of the fields to use for packing each record; first
    * value is primary split, second value is secondary, etc. */
  val fieldNames = Seq(
    "protocolIdentifier",
    "sourceTransportPort" //,
    //"flowStartMilliseconds",
    //"sourceIPv4Address",
    //"sourceIPv6Address"
  )

  // The URL-safe version is used because it's also filename-safe.
  // It uses "-" and "_" instead of "+" and "/"
  private[this] val base64Encoder = Base64.getUrlEncoder()

  private[this] def makeExtractor(name: String):
      (Record => Option[String]) =
  {
    infoModel.get(name) match {
      case None =>
        (r: Record) => None
      case Some(ie) =>
        ie.dataTypeId match {
          // Unsigned values use the next larger number type
          case org.cert.netsa.io.ipfix.DataTypes.Unsigned8 =>
            val ex = SimpleFieldExtractor[Short](name);
            (r: Record) => ex.extractFrom(r) map {
              (n: Short) => s"${name}=${n}"
            }
          case org.cert.netsa.io.ipfix.DataTypes.Unsigned16 =>
            val ex = SimpleFieldExtractor[Int](name);
            (r: Record) => ex.extractFrom(r) map {
              (n: Int) => s"${name}=${n}"
            }
          case org.cert.netsa.io.ipfix.DataTypes.Unsigned32 =>
            val ex = SimpleFieldExtractor[Long](name);
            (r: Record) => ex.extractFrom(r) map {
              (n: Long) => s"${name}=${n}"
            }
          case org.cert.netsa.io.ipfix.DataTypes.Unsigned64 =>
            val ex = SimpleFieldExtractor[Long](name);
            (r: Record) => ex.extractFrom(r) map {
              (n: Long) => s"${name}=${n}"
            }

          // Signed values
          case org.cert.netsa.io.ipfix.DataTypes.Signed8 =>
            val ex = SimpleFieldExtractor[Byte](name);
            (r: Record) => ex.extractFrom(r) map {
              (n: Byte) => s"${name}=${n}"
            }
          case org.cert.netsa.io.ipfix.DataTypes.Signed16 =>
            val ex = SimpleFieldExtractor[Short](name);
            (r: Record) => ex.extractFrom(r) map {
              (n: Short) => s"${name}=${n}"
            }
          case org.cert.netsa.io.ipfix.DataTypes.Signed32 =>
            val ex = SimpleFieldExtractor[Int](name);
            (r: Record) => ex.extractFrom(r) map {
              (n: Int) => s"${name}=${n}"
            }
          case org.cert.netsa.io.ipfix.DataTypes.Signed64 =>
            val ex = SimpleFieldExtractor[Long](name);
            (r: Record) => ex.extractFrom(r) map {
              (n: Long) => s"${name}=${n}"
            }

          // IP Addresses
          case org.cert.netsa.io.ipfix.DataTypes.IPv4Address =>
            val ex = SimpleFieldExtractor[IPv4Address](name);
            (r: Record) => ex.extractFrom(r) map {
              (n: IPv4Address) => s"${name}=${n}"
            }
          case org.cert.netsa.io.ipfix.DataTypes.IPv6Address =>
            val ex = SimpleFieldExtractor[IPv6Address](name);
            (r: Record) => ex.extractFrom(r) map {
              (n: IPv6Address) =>
              s"${name}=" + URLEncoder.encode(n.toString, "utf-8")
            }

          // Boolean
          case org.cert.netsa.io.ipfix.DataTypes.Boolean =>
            val ex = SimpleFieldExtractor[Boolean](name);
            (r: Record) => ex.extractFrom(r) map {
              (n: Boolean) => s"${name}=${n}"
            }

          // Floating point values
          case org.cert.netsa.io.ipfix.DataTypes.Float32 =>
            val ex = SimpleFieldExtractor[Float](name);
            (r: Record) => ex.extractFrom(r) map {
              (n: Float) => s"${name}=${n}"
            }
          case org.cert.netsa.io.ipfix.DataTypes.Float64 =>
            val ex = SimpleFieldExtractor[Double](name);
            (r: Record) => ex.extractFrom(r) map {
              (n: Double) => s"${name}=${n}"
            }

          // Timestaamps
          case org.cert.netsa.io.ipfix.DataTypes.DateTimeSeconds =>
            val ex = SimpleFieldExtractor[Instant](name);
            (r: Record) => ex.extractFrom(r) map {
              (n: Instant) => s"${name}=${n}"
            }
          case org.cert.netsa.io.ipfix.DataTypes.DateTimeMilliseconds =>
            val ex = SimpleFieldExtractor[Instant](name);
            (r: Record) => ex.extractFrom(r) map {
              (n: Instant) => s"${name}=${n}"
            }
          case org.cert.netsa.io.ipfix.DataTypes.DateTimeMicroseconds =>
            val ex = SimpleFieldExtractor[Instant](name);
            (r: Record) => ex.extractFrom(r) map {
              (n: Instant) => s"${name}=${n}"
            }
          case org.cert.netsa.io.ipfix.DataTypes.DateTimeNanoseconds =>
            val ex = SimpleFieldExtractor[Instant](name);
            (r: Record) => ex.extractFrom(r) map {
              (n: Instant) => s"${name}=${n}"
            }

          // Strings and Octet Arrays
          case org.cert.netsa.io.ipfix.DataTypes.String =>
            val ex = SimpleFieldExtractor[String](name);
            (r: Record) => ex.extractFrom(r) map {
              (n: String) => s"${name}=" + URLEncoder.encode(n, "utf-8")
            }
          case org.cert.netsa.io.ipfix.DataTypes.MacAddress =>
            val ex = SimpleFieldExtractor[Array[Byte]](name);
            (r: Record) => ex.extractFrom(r) map {
              (n: Array[Byte]) => s"${name}=" + base64Encoder.encode(n)
            }
          case org.cert.netsa.io.ipfix.DataTypes.OctetArray =>
            val ex = SimpleFieldExtractor[Array[Byte]](name);
            (r: Record) => ex.extractFrom(r) map {
              (n: Array[Byte]) => s"${name}=" + base64Encoder.encode(n)
            }

          // Structured data types
          case org.cert.netsa.io.ipfix.DataTypes.BasicList =>
            {(r: Record) => None}
          case org.cert.netsa.io.ipfix.DataTypes.SubTemplateList =>
            {(r: Record) => None}
          case org.cert.netsa.io.ipfix.DataTypes.SubTemplateMultiList =>
            {(r: Record) => None}
        }
    }
  }

  /** Extracts the protocol field's value from a Record */
  private[this] val extractors: Seq[Record => Option[String]] =
    for (name <- fieldNames) yield makeExtractor(name)


  // Define the pack() method required by our trait.
  // The `packStream()` method (on [[Packer]]) calls it for each
  // IPFIX record to determines how the record is stored.
  def pack(record: Record): TraversableOnce[PackableRecord] = {
    // extract the fields
    val fields: Seq[Option[String]] = for (ex <- extractors) yield (ex(record))
    // create the output path
    val relPath = Seq[String](s"v${version}") ++ (
      for { f <- fields ; f2 <- f } yield f2)
    // write the record
    Option(PackableRecord(record, relPath.mkString("/"), domain))
  }

}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
