// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.mothra.packer

import org.cert.netsa.io.ipfix.{Record, SessionGroup}


/**
  * A method for Mothra packers to map [[org.cert.netsa.io.ipfix.Record
  * Record]]s to storage locations in a Mothra repository.  It permits
  * storing a [[org.cert.netsa.io.ipfix.Record Record]] zero, one, or
  * multiple times, and it also allows the opportunity to modify the
  * [[org.cert.netsa.io.ipfix.Record Record]] before it is stored.
  *
  * @see [[PartitionerConfigurator]] instead. While some packing tools
  * (`mothra-packer` and `mothra-repacker`) still allow general
  * [[PackingLogic]] definitions to be used, the more structured
  * approach of splitting records into a partition structure is more
  * generally useful and safe.
  *
  * To use a [[PackingLogic]] object for the packing logic configuration
  * of one of the Mothra packer tools, use a file containing something
  * like the following:
  *
  * {{{
    import org.cert.netsa.io.ipfix.Record
  * import org.cert.netsa.mothra.packer.{PackableRecord, PackingLogic}
  * new PackingLogic {
  *   override def pack(rec: Record): TraversableOnce[PackableRecord] = ...
  * }
  * }}}
  *
  * Those tools will then compile this and use the partition
  * configuration for packing records.
  *
  */
trait PackingLogic {

  /**
    * Stores an incoming [[org.cert.netsa.io.ipfix.Record Record]] to
    * some number of files.
    *
    * The packer calls this function for each
    * [[org.cert.netsa.io.ipfix.Record Record]] that it reads from a
    * stream. This function is responsible for determining how to store
    * the record in the destination file(s).
    *
    * Implementations of this function should determine the relative
    * pathname where the record is to be stored and create a
    * [[PackableRecord]] that contains the
    * [[org.cert.netsa.io.ipfix.Record Record]] and the relative path.
    * The "relative pathname" is the part of a file's pathname that
    * begins with a version number (for example, "v2/") and ends just
    * before the file's name.
    *
    * To have the record not be stored, the function may return
    * [[scala.None None]]. To store one [[org.cert.netsa.io.ipfix.Record
    * Record]], the function may wrap it in an [[scala.Option Option]].
    * To store the record multiple times, a collection object may be
    * returned. The stored records need not have any relation to the
    * record passed into the function.
    */
  def pack(record: Record): TraversableOnce[PackableRecord]

  /**
    * This is a function that the packer invokes before it begins to
    * process a stream of records. The argument is the
    * [[org.cert.netsa.io.ipfix.SessionGroup SessionGroup]] that was
    * created for the stream.
    *
    * This implementation of function does nothing, but subclasses may
    * override it to do things like register callback functions for
    * templates in the [[org.cert.netsa.io.ipfix.SessionGroup
    * SessionGroup]].
    */
  def sessionRegister(group: SessionGroup): Unit = ()

}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
