// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.mothra.packer

import java.time.{Instant, ZoneOffset, ZonedDateTime}
import org.cert.netsa.io.ipfix.{Record, SessionGroup, SimpleFieldExtractor}


/**
  * An object to assist in geting the start time for a
  * [[org.cert.netsa.io.ipfix.Record]] by checking the Record's
  * [[org.cert.netsa.io.ipfix.Template]] for several different
  * time-related [[org.cert.netsa.io.ipfix.InfoElement]] fields.  The
  * checks for a [[org.cert.netsa.io.ipfix.Template]] are cached to speed
  * processing.
  */
private[mothra] final object StartTimeCache {
  // /** Instance returned by ofRecord() when the Template does not have
  //   * any of the expected datetime-related fields. */
  // final val epochRecordTime =
  //   RecordStartTime(LocalDate.of(1970, 1, 1), LocalTime.of(0, 0))

  /** This is called automatically by the `packStream()` method in
    * [[Packer]].  */
  final def sessionRegister(group: SessionGroup): Unit = {
    // do nothing
  }

  private[this] final val flowStartMilliseconds =
    SimpleFieldExtractor[Instant]("flowStartMilliseconds")

  /**
    * Finds and returns a RecordStartTime based on a Record's start
    * time.
    *
    * If the Record does not have an expected datetime field, the
    * epochRecordTime object is returned.
    *
    * @param record The record to get the start time of.
    */
  final def ofRecord(record: Record): RecordStartTime = {
    val instant = flowStartMilliseconds.extractFrom(record).
      getOrElse(record.exportTime.getOrElse(Instant.EPOCH))
    val datetime = ZonedDateTime.ofInstant(instant, ZoneOffset.UTC)
    RecordStartTime(datetime.toLocalDate, datetime.toLocalTime)
  }
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
