// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.mothra.tools

import org.cert.netsa.mothra.packer.Version

import java.nio.file.{Path => JavaPath}
import java.time.Duration
import org.apache.hadoop.conf.{Configuration => HadoopConfiguration}
import org.apache.hadoop.fs.{Path => HadoopPath}
import org.apache.hadoop.io.compress.CompressionCodecFactory
import org.cert.netsa.io.ipfix.InfoModel
import scala.math.Numeric.Implicits._
import scala.math.Ordering.Implicits._
import scala.util.Try
import scopt.{OptionParser, Read}

abstract class ToolOptions(val toolName: String)
  extends OptionParser[Unit](toolName)
{

  def programVersion: String = s"$toolName ${Version.get()}"

  // Default header of program name and version
  head(programVersion)

  // Always include a hidden debug option
  var debug: Boolean = false
  opt[Unit]("debug").hidden().foreach(_ => debug = true)
    .text("Include verbose debugging output on stderr")

  /* Additional parsing of options for paths, compression methods, etc. */

  implicit val infoModel = InfoModel.getCERTStandardInfoModel()

  protected val hadoopConf = new HadoopConfiguration()

  protected def parseHadoopPath(s: String): HadoopPath =
    new HadoopPath(s)

  protected def parseJavaPath(s: String): JavaPath =
    java.nio.file.Paths.get(s)

  protected def parseDuration(s: String): Duration = Duration.parse(s)

  protected def parseSize(s: String): Long =
    if ( s.toLowerCase.endsWith("t") ) {
      s.init.toLong * 1024 * 1024 * 1024 * 1024
    } else if ( s.toLowerCase.endsWith("g") ) {
      s.init.toLong * 1024 * 1024 * 1024
    } else if ( s.toLowerCase.endsWith("m") ) {
      s.init.toLong * 1024 * 1024
    } else if ( s.toLowerCase.endsWith("k") ) {
      s.init.toLong * 1024
    } else {
      s.toLong
    }

  protected def parseCompression(s: String): String = s match {
    case "" | "none" => ""
    case _ => s
  }

  /** Output human-readable size if possible. */

  protected def showSize(s: Long): String =
    if ( s % 1024 != 0 ) {
      s"$s"
    } else if ( s % (1024L * 1024L) != 0 ) {
      s"${s / 1024L}k"
    } else if ( s % (1024L * 1024L * 1024L) != 0 ) {
      s"${s / (1024L * 1024L)}M"
    } else if ( s % (1024L * 1024L * 1024L * 1024L) != 0 ) {
      s"${s / (1024L * 1024L * 1024L)}G"
    } else {
      s"${s / (1024L * 1024L * 1024L * 1024L)}T"
    }

  /* Command-line option readers for new types. Mostly implicit. */

  protected implicit val hadoopPathRead: Read[HadoopPath] =
    Read.reads(parseHadoopPath)

  protected implicit val javaPathRead: Read[JavaPath] =
    Read.reads(parseJavaPath)

  protected implicit val durationRead: Read[Duration] =
    Read.reads(parseDuration)

  /* Use this one like opt[Long](names...)(sizeRead) */
  protected val sizeRead: Read[Long] =
    Read.reads(parseSize)

  protected def checkPos[T : Numeric](desc: String)(n: T): Either[String, Unit] =
    n.signum match {
      case 1 => success
      case _ => failure(s"$desc must be >= 1 but was given '$n'")
    }

  protected def checkNonNeg[T : Numeric](desc: String)(n: T): Either[String, Unit] =
    n.signum match {
      case 0 | 1 => success
      case _ => failure(s"$desc must be >= 1 but was given '$n'")
    }

  protected def checkGE[T : Numeric](desc: String, minVal: T)(n: T): Either[String, Unit] =
    if ( n >= minVal ) success else {
      failure(s"$desc must be >= $minVal but was given '$n'")
    }

  protected def checkRange[T : Numeric](desc: String, minVal: T, maxVal: T)(n: T): Either[String, Unit] =
    if ( n >= minVal && n <= maxVal ) success else {
      failure(s"$desc must be >= $minVal and <= $maxVal but was given '$n'")
    }

  protected def checkCompression(desc: String)(codecName: String): Either[String, Unit] =
    if ( codecName == "" || codecName == "none" ) success else {
      val codec = Option(
        new CompressionCodecFactory(hadoopConf).getCodecByName(codecName))
      val compressor = codec.flatMap(c => Try(Option(c.createCompressor)).toOption.flatten)
      val smokeTest = compressor.flatMap(c =>
        Try(c.setInput(Array(1, 2, 3, 4, 5), 0, 4)).toOption)
      if ( codec.isEmpty ) failure(
        s"$desc can't find compression codec '$codecName'")
      else if ( compressor.isEmpty ) failure(
        s"$desc can't instantiate compression codec '$codecName'")
      else if ( smokeTest.isEmpty ) failure(
        s"$desc can't initialize compression codec '$codecName'")
      else success
    }

  protected def checkHadoopDirExists(desc: String)(path: HadoopPath): Either[String, Unit] = {
    // These operations could fail, but the exceptions will be meaningful
    val fs = path.getFileSystem(hadoopConf)
    if ( fs.exists(path) && fs.getFileStatus(path).isDirectory() ) {
      success
    } else {
      failure(s"$desc can't find (Hadoop) directory '$path'")
    }
  }

  protected def checkHadoopFileExists(desc: String)(path: HadoopPath): Either[String, Unit] = {
    val fs = path.getFileSystem(hadoopConf)
    if ( fs.exists(path) && fs.getFileStatus(path).isFile() ) {
      success
    } else {
      failure(s"$desc can't find (Hadoop) file '$path'")
    }
  }

  protected def checkJavaDirExists(desc: String)(path: JavaPath): Either[String, Unit] = {
    if ( path.toFile.isDirectory ) success else {
      failure(s"$desc can't find (local) directory '$path'")
    }
  }

  protected def checkIEName(desc: String)(name: String): Either[String, Unit] = {
    if ( name.contains(",") ) {
      failure(s"$desc must not contain ',', but does")
    } else if ( infoModel.get(name).isDefined ) {
      success
    } else {
      failure(s"$desc can't find IPFIX Information Element '$name'")
    }
  }


}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
