// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.mothra.packer

import org.cert.netsa.io.ipfix.{Record, SimpleFieldExtractor}


/**
  * An implementation of the [[PackingLogic]] trait.  The `pack()` method
  * stores Records into files based on the record's IP Protocol and source
  * transport port.
  *
  * In this version, a directory is created for each unique protocol
  * and source port.  For a version that groups uncommon protocols and
  * ports into larger files, see [[ProtoSportPartitionerPackLogic]].
  *
  * @see [[ProtoSportPartitionerPackLogic]] for a better way to pack
  * [[org.cert.netsa.io.ipfix.Record Record]]s
  */
case class ProtoSportPackLogic() extends PackingLogic
{
  /** The observationDomain to use for newly created files. */
  val domain = 0

  /** The version of the packing logic; this value becomes part of the
    * path in the directory hierarchy. */
  val version = 1

  /** Extracts the protocol field's value from a Record */
  private[this] val protoField =
    SimpleFieldExtractor[Short]("protocolIdentifier")

  /** Extracts the source port field's value from a Record */
  private[this] val sportField =
    SimpleFieldExtractor[Int]("sourceTransportPort")


  // Define the pack() method required by our trait.
  // The `packStream()` method (on [[Packer]]) calls it for each
  // IPFIX record to determines how the record is stored.
  def pack(record: Record): TraversableOnce[PackableRecord] = {
    // create the relative path
    val relPath: String = record(protoField) match {
      case Some(proto) =>
        record(sportField) match {
          case Some(sport) =>
            f"v${version}/protocolIdentifier=${proto}%02x/sourceTransportPort=${sport}%04x"
          case None =>
            f"v${version}/protocolIdentifier=${proto}%02x"
        }
      case None =>
        record(sportField) match {
          case Some(sport) =>
            // this does not make sense...
            f"v${version}/sourceTransportPort=${sport}%04x"
          case None =>
            s"v${version}"
        }
    }
    // write the record
    Option(PackableRecord(record, relPath, domain))
  }

}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
