// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.mothra.tools.collector

import scala.util.Try

import com.typesafe.scalalogging.StrictLogging
import org.apache.hadoop.conf.Configuration
import org.apache.hadoop.fs.{Path, PathFilter}

case class DirMapping(sourceDir: Path, targetDir: Path) extends StrictLogging {
  def clearStaging()(implicit conf: Configuration): Unit = {
    val fs = targetDir.getFileSystem(conf)
    for (status <- fs.listStatus(targetDir, DirMapping.onlyDotFiles)) {
      fs.delete(status.getPath(), false)
    }
  }
  def jobs(implicit conf: Configuration): Iterable[CopyJob] =
    try {
      val sFs = sourceDir.getFileSystem(conf)
      val tFs = targetDir.getFileSystem(conf)
      for {
        sourceStatus <- sFs.listStatus(sourceDir, DirMapping.excludeDotFiles)
        if Try(sourceStatus.isFile()).getOrElse(false)
        if sourceStatus.getLen() > 0
        sourcePath = sourceStatus.getPath()
        targetPath = new Path(targetDir, sourcePath.getName())
        _ = Try(tFs.getFileStatus(targetPath)).toOption
      } yield CopyJob(sourcePath, targetPath)
    } catch {
      case ex: Exception =>
        logger.error(s"Failure reading source direcotry '$sourceDir': $ex")
        Iterable()
    }
}

object DirMapping {
  private object onlyDotFiles extends PathFilter {
    def accept(p: Path): Boolean = p.getName().startsWith(".")
  }
  private object excludeDotFiles extends PathFilter {
    def accept(p: Path): Boolean = !p.getName().startsWith(".")
  }
}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
