// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.mothra.packer

import org.cert.netsa.io.ipfix.Record

/** A [[org.cert.netsa.io.ipfix.Record Record]] and the information required to generate the file
  * name where the Record will be stored in a Mothra repository.
  *
  * @param record The record to write to the file.
  * @param relativePackedPath The relative path of the file to return; does not include time prefix
  *   or filename suffix, but does include a version number
  * @param startTime The date+time to use for time-based parts of file path
  * @param observationDomain The domain used for all IPFIX Messages in the file; this is needed in
  *   case a new file must created and opened.
  */
case class PackableRecord(
  record: Record,
  relativePackedPath: String,
  startTime: RecordStartTime,
  observationDomain: Int
)

/** A PackableRecord factory. */
object PackableRecord {

  /** The observationDomain to use if none is supplied. */
  private val defaultDomain = 0

  /** Creates a new PackableRecord with the `startTime` based on a starting time field in the
    * `record`. The supplied `observationDomain` is used if non-empty, else a domain of 0 is used.
    */
  def apply(
    record: Record,
    relativePackedPath: String,
    observationDomain: Option[Int] = None
  ): PackableRecord = {
    new PackableRecord(
      record,
      relativePackedPath,
      StartTimeCache.ofRecord(record),
      observationDomain.getOrElse(defaultDomain)
    )
  }

  /** Creates a new PackableRecord with a `startTime` based on a starting time field in the
    * `record`.
    */
  def apply(record: Record, relativePackedPath: String, observationDomain: Int): PackableRecord = {
    new PackableRecord(
      record,
      relativePackedPath,
      StartTimeCache.ofRecord(record),
      observationDomain
    )
  }
}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
