// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.mothra.packer

import java.nio.file.{Path => JPath}

import org.apache.hadoop.conf.Configuration
import org.apache.hadoop.fs.{Path => HPath}
import org.apache.hadoop.io.compress.CompressionCodec

/** PackerConfig holds the directories and compression method specified when the packer was started.
  *
  * @param rootDir The root directory of the record repository
  * @param workDir The temporary directory to use as files are created.
  * @param archiveDir If not None, the directory in which to move temporary files after their
  *   contents have been copied into the `rootDir`.
  * @param compressCodec The compression to use for output files.
  */
private[mothra] case class PackerConfig(
  rootDir: HPath,
  workDir: JPath,
  archiveDir: Option[HPath] = None,
  compressCodec: Option[CompressionCodec] = None
)(implicit hadoopConf: Configuration) {

  /** A file's compression suffix for the specified `compressCodec`. */
  val compressSuffix = compressCodec.map(c => c.getDefaultExtension).getOrElse("")

  /** The FileSystem of the `rootDir`. */
  lazy val rootDirFileSystem = rootDir.getFileSystem(hadoopConf)

  /** The FileSystem of the `archiveDir` or `null` if `archiveDir` is `None`. */
  lazy val archiveDirFileSystem = archiveDir.map(_.getFileSystem(hadoopConf)).orNull;

}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
