// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.mothra.packer

import org.cert.netsa.io.ipfix.{Record, SimpleFieldExtractor}

/** An implementation of the [[PackingLogic]] trait. The `pack()` method stores records into files
  * based on the record's IP Protocol.
  */
case class ProtocolPackLogic() extends PackingLogic {

  /** The observationDomain to use for newly created files. */
  val domain = 0

  /** The version of the packing logic; this value becomes part of the path in the directory
    * hierarchy.
    */
  val version = 1

  /** Extracts the protocol field's value from a Record */
  private val protoField = SimpleFieldExtractor[Short]("protocolIdentifier")

  // Define the pack() method required by our trait.
  // The `packStream()` method (on [[Packer]]) calls it for each
  // IPFIX record to determines how the record is stored.
  def pack(record: Record): Iterator[PackableRecord] = {
    // get the protocol
    val proto = record(protoField)
    // create the output path
    val relPath = proto match {
      case Some(p) => f"v$version/protocolIdentifier=$p%02x"
      case None    => s"v$version"
    }
    // write the record
    Iterator(PackableRecord(record, relPath, domain))
  }

}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
