// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.mothra.packer

import java.time.{Instant, LocalDate, LocalTime, ZonedDateTime, ZoneOffset}

/** The start time of a [[org.cert.netsa.io.ipfix.Record Record]]. All returned values use UTC.
  *
  * @param date The date (YEAR, MONTH, DAY) part of the start time.
  * @param time The time (HOUR, MINUTE, SECOND) part of the start time.
  */
final case class RecordStartTime(date: LocalDate, time: LocalTime) {

  /** The hour of the start time. */
  lazy val hour: Int = time.getHour

  /** A String based on the start time to use in a directory path, in the form "YEAR/MONTH/DAY" such
    * as "2017/01/04".
    */
  lazy val ymdDirname: String =
    f"${date.getYear}/${date.getMonthValue}%02d/${date.getDayOfMonth}%02d"

  private lazy val yyyymmdd = f"${date.getYear}${date.getMonthValue}%02d${date.getDayOfMonth}%02d"

  /** A String based on the start time to use in a file name, in the form "YYYYMMDD.HH" such as
    * "20170104.23".
    */
  lazy val ymdhBasename: String = f"$yyyymmdd.$hour%02d"

  /** A String for a file that covers multiple hours, assuming each file covers `numberOfHours`
    * hours. The form is similar to `ymdhBasename` with an additional hyphen and ISO 8601 style
    * duration. Examples: "20191002.14-PT2H", "20191002.12-PT3H", "20191002.12-PT12H",
    * "20191002.00-PT24H".
    *
    * The duration suffix is not included when `numberOfHours` is 1. When `numberOfHours` is not an
    * even divisor of 24, the value is rounded up to the next even divisor (5 becomes 6, 10 becomes
    * 12, 15 becomes 24).
    */
  def ymdhDuration(numberOfHours: Int): String = {
    numberOfHours match {
      case 24 => f"$yyyymmdd.00-PT24H"
      case 1  => ymdhBasename
      case _ =>
        require(numberOfHours >= 1 && numberOfHours <= 24)
        val dur = 24 / (24 / numberOfHours)
        val h = hour - hour % dur
        f"$yyyymmdd.$h%02d-PT${dur}H"
    }
  }

}

/** A RecordStartTime factory. */
object RecordStartTime {

  /** Creates a RecordStartTime from an Instant. */
  def apply(instant: Instant): RecordStartTime = {
    val datetime = ZonedDateTime.ofInstant(instant, ZoneOffset.UTC)
    new RecordStartTime(datetime.toLocalDate, datetime.toLocalTime)
  }
}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
