// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.mothra.packer

import java.time.{Instant, ZonedDateTime, ZoneOffset}
import scala.annotation.nowarn

import org.cert.netsa.io.ipfix.{Record, SessionGroup, SimpleFieldExtractor}

/** An object to assist in geting the start time for a [[org.cert.netsa.io.ipfix.Record]] by
  * checking the Record's [[org.cert.netsa.io.ipfix.Template]] for several different time-related
  * [[org.cert.netsa.io.ipfix.InfoElement]] fields. The checks for a
  * [[org.cert.netsa.io.ipfix.Template]] are cached to speed processing.
  */
private[mothra] final object StartTimeCache {
  // /** Instance returned by ofRecord() when the Template does not have
  //   * any of the expected datetime-related fields. */
  // final val epochRecordTime =
  //   RecordStartTime(LocalDate.of(1970, 1, 1), LocalTime.of(0, 0))

  /** This is called automatically by the `packStream()` method in [[Packer]]. */
  final def sessionRegister(
    @nowarn("cat=unused")
    group: SessionGroup
  ): Unit = {
    // do nothing
  }

  private final val flowStartMilliseconds = SimpleFieldExtractor[Instant]("flowStartMilliseconds")

  /** Finds and returns a RecordStartTime based on a Record's start time.
    *
    * If the Record does not have an expected datetime field, the epochRecordTime object is
    * returned.
    *
    * @param record The record to get the start time of.
    */
  final def ofRecord(record: Record): RecordStartTime = {
    val instant = flowStartMilliseconds
      .extractFrom(record)
      .getOrElse(record.exportTime.getOrElse(Instant.EPOCH))
    val datetime = ZonedDateTime.ofInstant(instant, ZoneOffset.UTC)
    RecordStartTime(datetime.toLocalDate, datetime.toLocalTime)
  }
}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
