// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.mothra.packer

import java.time.Instant

import org.cert.netsa.io.ipfix.{
  ArrayRecord, ExportSubTemplateList, Identifier, IEFieldSpecifier, InfoModel, Record, Template,
  VARLEN
}

private[mothra] trait Tombstone {

  /** The CERT Private Enterprise Number */
  protected val CERT_PEN = 6871

  /** The information model. */
  protected val infoModel = InfoModel.getCERTStandardInfoModel()

  /** Number used in Tombstone IPFIX records to denote the Mothra IPFIX Packer has operated on the
    * Tombstone Record.
    */
  protected val MOTHRA_PACKER_ID = 5

  // abstract variable
  /** The template used by a tombstone record. */
  val tombstoneTemplate: Template

  // abstract variable
  /** The template used by the SubTemplateList to hold timestamps for each tool along the record's
    * path.
    */
  val accessTemplate: Template

  // abstract method
  /** Returns a copy of `tombRecord` with an additional entry for the Mothra packer, and the
    * starting time to use when packing the Record.
    */
  def updateAccessList(tombRecord: Record): (Record, RecordStartTime)

}

private[mothra] object Tombstone {

  private val tombstoneTemplates = Map(
    TombstoneV1.tombstoneTemplate -> TombstoneV1,
    TombstoneV2.tombstoneTemplate -> TombstoneV2,
    TombstoneV2b.tombstoneTemplate -> TombstoneV2b
  )

  /** If `template` matches a Tombstone template, returns an Object that inherits from the Tombstone
    * trait as an Option. Otherwise returns None.
    */
  def isTombstoneTemplate(template: Template): Option[Tombstone] = {
    tombstoneTemplates.get(template)
  }

}

/** Represents the first iteration of the Tombstone record, used in YAF 2.10 and super_mediator
  * 1.6.0.
  */
private[mothra] object TombstoneV1 extends Tombstone {

  /** The scope to use for the Tombstone Template. */
  val TOMBSTONE_TEMPLATE_SCOPE = 2

  val tombstoneTemplate = Template.newOptionsTemplate(
    TOMBSTONE_TEMPLATE_SCOPE,
    List(
      IEFieldSpecifier(Identifier(551, CERT_PEN), 2), // exporterConfiguredId
      IEFieldSpecifier(Identifier(552, CERT_PEN), 2), // exporterUniqueId
      IEFieldSpecifier(Identifier(550, CERT_PEN), 4), // tombstoneId
      IEFieldSpecifier(Identifier(292), VARLEN)
    ), // subTemplateList
    infoModel
  )

  val accessTemplate = Template.newTemplate(
    List(
      IEFieldSpecifier(Identifier(144), 4), // exportingProcessId
      IEFieldSpecifier(Identifier(322), 4)
    ), // observationTimeSeconds
    infoModel
  )

  def updateAccessList(tombRecord: Record): (Record, RecordStartTime) = {
    assert(tombRecord.template == tombstoneTemplate)
    // copy the record
    val rec = Record(tombRecord, true)
    // get the STL
    val stl = rec(3).asInstanceOf[ExportSubTemplateList]
    // get the first element to access the STL (the originator)
    val originator = stl(0)
    // get the time of that element
    val originatorTime = originator(1).asInstanceOf[Instant]
    // create the new entry for the STL
    val currentAccess = ArrayRecord(stl.template)
    currentAccess.update(0, MOTHRA_PACKER_ID)
    currentAccess.update(1, java.time.Instant.now())
    stl.append(currentAccess)
    (rec, RecordStartTime(originatorTime))
  }

}

/** Represents the second iteration of the Tombstone record, used in YAF 2.11. */
private[mothra] object TombstoneV2 extends Tombstone {

  /** The scope to use for the Tombstone Template. */
  val TOMBSTONE_TEMPLATE_SCOPE = 3

  val tombstoneTemplate = Template.newOptionsTemplate(
    TOMBSTONE_TEMPLATE_SCOPE,
    List(
      IEFieldSpecifier(Identifier(149), 4), // observationDomainId
      IEFieldSpecifier(Identifier(144), 4), // exportingProcessId
      IEFieldSpecifier(Identifier(551, CERT_PEN), 2), // exporterConfiguredId
      IEFieldSpecifier(Identifier(210), 6), // paddingOctets[6]
      IEFieldSpecifier(Identifier(550, CERT_PEN), 4), // tombstoneId
      IEFieldSpecifier(Identifier(322), 4), // observationTimeSec
      IEFieldSpecifier(Identifier(554, CERT_PEN), VARLEN)
    ), // tombstoneAccess
    infoModel
  )

  val accessTemplate = Template.newTemplate(
    List(
      IEFieldSpecifier(Identifier(553, CERT_PEN), 4), // certToolId
      IEFieldSpecifier(Identifier(322), 4)
    ), // observationTimeSeconds
    infoModel
  )

  def updateAccessList(tombRecord: Record): (Record, RecordStartTime) = {
    assert(tombRecord.template == tombstoneTemplate)
    // copy the record
    val rec = Record(tombRecord, true)
    // get the observation time
    val observationTime = rec(5).asInstanceOf[Instant]
    // get the STL
    val stl = rec(6).asInstanceOf[ExportSubTemplateList]
    // create the new entry for the STL
    val currentAccess = ArrayRecord(stl.template)
    currentAccess.update(0, MOTHRA_PACKER_ID)
    currentAccess.update(1, java.time.Instant.now())
    stl.append(currentAccess)
    (rec, RecordStartTime(observationTime))
  }

}

/** Represents the second iteration of the Tombstone record (with a different (buggy) scope value),
  * used in super_mediator 1.7.0.
  */
private[mothra] object TombstoneV2b extends Tombstone {

  /** The scope to use for the Tombstone Template. */
  val TOMBSTONE_TEMPLATE_SCOPE = 2

  val tombstoneTemplate = Template.newOptionsTemplate(
    TOMBSTONE_TEMPLATE_SCOPE,
    List(
      IEFieldSpecifier(Identifier(149), 4), // observationDomainId
      IEFieldSpecifier(Identifier(144), 4), // exportingProcessId
      IEFieldSpecifier(Identifier(551, CERT_PEN), 2), // exporterConfiguredId
      IEFieldSpecifier(Identifier(210), 6), // paddingOctets[6]
      IEFieldSpecifier(Identifier(550, CERT_PEN), 4), // tombstoneId
      IEFieldSpecifier(Identifier(322), 4), // observationTimeSec
      IEFieldSpecifier(Identifier(554, CERT_PEN), VARLEN)
    ), // tombstoneAccess
    infoModel
  )

  val accessTemplate = Template.newTemplate(
    List(
      IEFieldSpecifier(Identifier(553, CERT_PEN), 4), // certToolId
      IEFieldSpecifier(Identifier(322), 4)
    ), // observationTimeSeconds
    infoModel
  )

  def updateAccessList(tombRecord: Record): (Record, RecordStartTime) = {
    assert(tombRecord.template == tombstoneTemplate)
    // copy the record
    val rec = Record(tombRecord, true)
    // get the observation time
    val observationTime = rec(5).asInstanceOf[Instant]
    // get the STL
    val stl = rec(6).asInstanceOf[ExportSubTemplateList]
    // create the new entry for the STL
    val currentAccess = ArrayRecord(stl.template)
    currentAccess.update(0, MOTHRA_PACKER_ID)
    currentAccess.update(1, java.time.Instant.now())
    stl.append(currentAccess)
    (rec, RecordStartTime(observationTime))
  }

}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
