// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.mothra.packer

import java.nio.channels.FileChannel
import java.nio.file.{Files, Path, StandardOpenOption}
import java.nio.file.attribute.FileTime
import java.time.Instant

import org.cert.netsa.io.ipfix.{ExportStream, InfoModel, Record, SessionGroup}

/** A wrapper over an [[org.cert.netsa.io.ipfix.ExportStream]].
  *
  * @param key The key to find this WorkFile in the file cache of the [[Packer]]. This is the
  *   nominal, relative path used when opening the file: it does not include the root directory, the
  *   leading y/m/d directories, nor the suffix used to make the path unique.
  * @param path The complete Path to the output file. This class assumes path ends with a UUID and
  *   names a non-existent file.
  * @param archivePath The path used to archive the working file after it has been copied to the
  *   repository. Does not include any archive directory.
  * @param infoModel The InfoModel used when writing the file.
  * @param observationDomain The observation domain used by the file.
  */
private[mothra] final case class WorkFile(
  key: String,
  path: Path,
  archivePath: String,
  infoModel: InfoModel,
  observationDomain: Int
) {

  /** A class to hold an ExportStream and its underlying FileChannel. */
  private class WorkStream(val outStream: FileChannel, val exportStream: ExportStream)

  /** The companion object for the WorkStream class. */
  private object WorkStream {

    /** Open a new file. */
    def open(): WorkStream = {
      Files.createDirectories(path.getParent())
      val outStream = FileChannel
        .open(path, StandardOpenOption.CREATE_NEW, StandardOpenOption.WRITE)
      val exportStream = ExportStream(outStream, session)
      for (cb <- _dataWrittenCallback) {
        exportStream.dataWrittenCallback = cb
      }
      new WorkStream(outStream, exportStream)
    }

    /** Re-open an existing file. */
    def reopen(): WorkStream = {
      val outStream = FileChannel.open(path, StandardOpenOption.WRITE, StandardOpenOption.READ)
      val exportStream = ExportStream.appendTo(outStream, infoModel)
      for (cb <- _dataWrittenCallback) {
        exportStream.dataWrittenCallback = cb
      }
      new WorkStream(outStream, exportStream)
    }
  }

  /** The SessionGroup for the output file. */
  private val sessionGroup = SessionGroup(infoModel, path)

  /** The Session used while writing the file. */
  val session = sessionGroup.getOrCreateSession(observationDomain)

  /** A callback invoked when bytes are written to the WritableByteChannel */
  private var _dataWrittenCallback: Option[ExportStream.DataWrittenCallback] = None

  /** The WorkStream being written to. This is an wrapped in an Option since it can be closed and
    * re-opened multiple times.
    */
  private var stream: Option[WorkStream] = Option(WorkStream.open())

  /** Gets the time when the file was created */
  lazy val creationTime: Instant = Files
    .getAttribute(path, "creationTime")
    .asInstanceOf[FileTime]
    .toInstant()

  /** Returns the number of seconds since the file was created. */
  def age(): Long = {
    Instant.now().getEpochSecond() - creationTime.getEpochSecond()
  }

  /** Returns the file's current size as a number of octets. */
  def size(): Long =
    synchronized {
      stream match {
        case Some(ws) => ws.outStream.position()
        case None     => Files.size(path)
      }
    }

  /** Returns the ExportStream for the output file as an Option, or None if the file is closed. */
  def exportStream: Option[ExportStream] =
    synchronized {
      stream map {
        _.exportStream
      }
    }

  /** Gets the current callback invoked when data is written to the exportStream as a
    * [[scala.Option Option]].
    * @since 1.3.1
    */
  def dataWrittenCallback: Option[ExportStream.DataWrittenCallback] = _dataWrittenCallback

  /** Sets a callback to be invoked when bytes are written to the exportStream.
    * @since 1.3.1
    */
  def dataWrittenCallback_=(callback: ExportStream.DataWrittenCallback): Unit = {
    _dataWrittenCallback = Option(callback)
    for (s <- stream) {
      s.exportStream.dataWrittenCallback = callback
    }
  }

  /** Reopens the output file if is currently closed. */
  final def reopen(): Unit =
    synchronized {
      if (stream.isEmpty) {
        stream = Option(WorkStream.reopen())
      }
    }

  /** Adds a Record to the [[org.cert.netsa.io.ipfix.ExportStream]]. Assumes the WorkFile is open.
    */
  final def writeRecord(record: Record): Unit =
    synchronized {
      stream.get.exportStream.add(record)
      ()
    }

  /** Flushes the [[org.cert.netsa.io.ipfix.ExportStream]] if it is open. Does nothing if it is not.
    */
  final def flush(): Unit =
    synchronized {
      for (s <- stream) s.exportStream.flush()
    }

  /** If the [[org.cert.netsa.io.ipfix.ExportStream]] is open, closes it and returns `true`.
    * Otherwise returns `false`.
    */
  final def close(): Boolean =
    synchronized {
      stream match {
        case Some(ws) =>
          ws.exportStream.close()
          stream = None
          true
        case None => false
      }
    }

}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
