// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.mothra.packer

import org.cert.netsa.io.ipfix.{Session, Template, VARLEN}

/** A utility object that defines a method which adds the [[Template Templates]] used by YAF and
  * super_meditor to a [[Session]].
  */
private[mothra] object YafTemplates {

  /** Defines the [[Template Templates]] used by YAF and super_meditor in `session`, which is a
    * session used for export. The purpose of this function is to ensure the templates use a
    * consistent set of IDs across all files.
    */
  def addToSession(session: Session): Unit = {
    val infoModel = session.infoModel

    {
      val name = "md_main_silk"
      val fields = Seq(
        ("flowStartMilliseconds", 8),
        ("flowEndMilliseconds", 8),
        ("octetTotalCount", 8),
        ("reverseOctetTotalCount", 8),
        ("octetDeltaCount", 8),
        ("reverseOctetDeltaCount", 8),
        ("packetTotalCount", 8),
        ("reversePacketTotalCount", 8),
        ("packetDeltaCount", 8),
        ("reversePacketDeltaCount", 8),
        ("sourceIPv6Address", 16),
        ("destinationIPv6Address", 16),
        ("sourceIPv4Address", 4),
        ("destinationIPv4Address", 4),
        ("sourceTransportPort", 2),
        ("destinationTransportPort", 2),
        ("flowAttributes", 2),
        ("reverseFlowAttributes", 2),
        ("protocolIdentifier", 1),
        ("flowEndReason", 1),
        ("silkAppLabel", 2),
        ("reverseFlowDeltaMilliseconds", 4),
        ("tcpSequenceNumber", 4),
        ("reverseTcpSequenceNumber", 4),
        ("initialTCPFlags", 1),
        ("unionTCPFlags", 1),
        ("reverseInitialTCPFlags", 1),
        ("reverseUnionTCPFlags", 1),
        ("vlanId", 2),
        ("reverseVlanId", 2),
        ("ingressInterface", 4),
        ("egressInterface", 4),
        ("ipClassOfService", 1),
        ("reverseIpClassOfService", 1),
        ("mplsTopLabelStackSection", 3),
        ("mplsLabelStackSection2", 3),
        ("mplsLabelStackSection3", 3),
        ("paddingOctets", 1),
        ("observationDomainId", 4),
        ("yafFlowKeyHash", 4),
        ("nDPIL7Protocol", 2),
        ("nDPIL7SubProtocol", 2),
        ("subTemplateMultiList", VARLEN)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xb000)
    }
    {
      val name = "md_main_total_ip4_list"
      val fields = Seq(
        ("flowStartMilliseconds", 8),
        ("flowEndMilliseconds", 8),
        ("octetTotalCount", 8),
        ("packetTotalCount", 8),
        ("sourceIPv4Address", 4),
        ("destinationIPv4Address", 4),
        ("sourceTransportPort", 2),
        ("destinationTransportPort", 2),
        ("flowAttributes", 2),
        ("protocolIdentifier", 1),
        ("flowEndReason", 1),
        ("silkAppLabel", 2),
        ("vlanId", 2),
        ("ipClassOfService", 1),
        ("observationDomainId", 4),
        ("yafFlowKeyHash", 4),
        ("nDPIL7Protocol", 2),
        ("nDPIL7SubProtocol", 2),
        ("subTemplateMultiList", VARLEN)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xb301)
    }
    {
      val name = "md_main_total_rev_ip4_list"
      val fields = Seq(
        ("flowStartMilliseconds", 8),
        ("flowEndMilliseconds", 8),
        ("octetTotalCount", 8),
        ("reverseOctetTotalCount", 8),
        ("packetTotalCount", 8),
        ("reversePacketTotalCount", 8),
        ("sourceIPv4Address", 4),
        ("destinationIPv4Address", 4),
        ("sourceTransportPort", 2),
        ("destinationTransportPort", 2),
        ("flowAttributes", 2),
        ("reverseFlowAttributes", 2),
        ("protocolIdentifier", 1),
        ("flowEndReason", 1),
        ("silkAppLabel", 2),
        ("reverseFlowDeltaMilliseconds", 4),
        ("vlanId", 2),
        ("reverseVlanId", 2),
        ("ipClassOfService", 1),
        ("reverseIpClassOfService", 1),
        ("observationDomainId", 4),
        ("yafFlowKeyHash", 4),
        ("nDPIL7Protocol", 2),
        ("nDPIL7SubProtocol", 2),
        ("subTemplateMultiList", VARLEN)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xb311)
    }
    {
      val name = "md_main_total_ip6_list"
      val fields = Seq(
        ("flowStartMilliseconds", 8),
        ("flowEndMilliseconds", 8),
        ("octetTotalCount", 8),
        ("packetTotalCount", 8),
        ("sourceIPv6Address", 16),
        ("destinationIPv6Address", 16),
        ("sourceTransportPort", 2),
        ("destinationTransportPort", 2),
        ("flowAttributes", 2),
        ("protocolIdentifier", 1),
        ("flowEndReason", 1),
        ("silkAppLabel", 2),
        ("vlanId", 2),
        ("ipClassOfService", 1),
        ("observationDomainId", 4),
        ("yafFlowKeyHash", 4),
        ("nDPIL7Protocol", 2),
        ("nDPIL7SubProtocol", 2),
        ("subTemplateMultiList", VARLEN)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xb501)
    }
    {
      val name = "md_main_total_rev_ip6_list"
      val fields = Seq(
        ("flowStartMilliseconds", 8),
        ("flowEndMilliseconds", 8),
        ("octetTotalCount", 8),
        ("reverseOctetTotalCount", 8),
        ("packetTotalCount", 8),
        ("reversePacketTotalCount", 8),
        ("sourceIPv6Address", 16),
        ("destinationIPv6Address", 16),
        ("sourceTransportPort", 2),
        ("destinationTransportPort", 2),
        ("flowAttributes", 2),
        ("reverseFlowAttributes", 2),
        ("protocolIdentifier", 1),
        ("flowEndReason", 1),
        ("silkAppLabel", 2),
        ("reverseFlowDeltaMilliseconds", 4),
        ("vlanId", 2),
        ("reverseVlanId", 2),
        ("ipClassOfService", 1),
        ("reverseIpClassOfService", 1),
        ("observationDomainId", 4),
        ("yafFlowKeyHash", 4),
        ("nDPIL7Protocol", 2),
        ("nDPIL7SubProtocol", 2),
        ("subTemplateMultiList", VARLEN)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xb511)
    }
    {
      val name = "yaf_entropy"
      val fields = Seq(("payloadEntropy", 1))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc002)
    }
    {
      val name = "yaf_tcp"
      val fields = Seq(("tcpSequenceNumber", 4), ("initialTCPFlags", 1), ("unionTCPFlags", 1))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc003)
    }
    {
      val name = "yaf_mac"
      val fields = Seq(("sourceMacAddress", 6), ("destinationMacAddress", 6))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc004)
    }
    {
      val name = "yaf_flow_stats"
      val fields = Seq(
        ("dataByteCount", 8),
        ("averageInterarrivalTime", 8),
        ("standardDeviationInterarrivalTime", 8),
        ("tcpUrgTotalCount", 4),
        ("smallPacketCount", 4),
        ("nonEmptyPacketCount", 4),
        ("largePacketCount", 4),
        ("firstNonEmptyPacketSize", 2),
        ("maxPacketSize", 2),
        ("standardDeviationPayloadLength", 2),
        ("firstEightNonEmptyPacketDirections", 1)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc005)
    }
    {
      val name = "yaf_p0f"
      val fields = Seq(("osName", VARLEN), ("osVersion", VARLEN), ("osFingerPrint", VARLEN))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc006)
    }
    {
      val name = "yaf_fpexport"
      val fields = Seq(("firstPacketBanner", VARLEN), ("secondPacketBanner", VARLEN))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc007)
    }
    {
      val name = "yaf_payload"
      val fields = Seq(("payload", VARLEN))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc008)
    }
    {
      val name = "yaf_mptcp"
      val fields = Seq(
        ("mptcpInitialDataSequenceNumber", 8),
        ("mptcpReceiverToken", 4),
        ("mptcpMaximumSegmentSize", 2),
        ("mptcpAddressID", 1),
        ("mptcpFlags", 1)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc009)
    }
    {
      val name = "yaf_entropy"
      val fields = Seq(("payloadEntropy", 1), ("reversePayloadEntropy", 1))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc012)
    }
    {
      val name = "yaf_tcp"
      val fields = Seq(
        ("tcpSequenceNumber", 4),
        ("initialTCPFlags", 1),
        ("unionTCPFlags", 1),
        ("reverseInitialTCPFlags", 1),
        ("reverseUnionTCPFlags", 1),
        ("reverseTcpSequenceNumber", 4)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc013)
    }
    {
      val name = "yaf_flow_stats"
      val fields = Seq(
        ("dataByteCount", 8),
        ("averageInterarrivalTime", 8),
        ("standardDeviationInterarrivalTime", 8),
        ("tcpUrgTotalCount", 4),
        ("smallPacketCount", 4),
        ("nonEmptyPacketCount", 4),
        ("largePacketCount", 4),
        ("firstNonEmptyPacketSize", 2),
        ("maxPacketSize", 2),
        ("standardDeviationPayloadLength", 2),
        ("firstEightNonEmptyPacketDirections", 1),
        ("paddingOctets", 1),
        ("reverseDataByteCount", 8),
        ("reverseAverageInterarrivalTime", 8),
        ("reverseStandardDeviationInterarrivalTime", 8),
        ("reverseTcpUrgTotalCount", 4),
        ("reverseSmallPacketCount", 4),
        ("reverseNonEmptyPacketCount", 4),
        ("reverseLargePacketCount", 4),
        ("reverseFirstNonEmptyPacketSize", 2),
        ("reverseMaxPacketSize", 2),
        ("reverseStandardDeviationPayloadLength", 2),
        ("paddingOctets", 2)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc015)
    }
    {
      val name = "yaf_p0f"
      val fields = Seq(
        ("osName", VARLEN),
        ("osVersion", VARLEN),
        ("osFingerPrint", VARLEN),
        ("reverseOsName", VARLEN),
        ("reverseOsVersion", VARLEN),
        ("reverseOsFingerPrint", VARLEN)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc016)
    }
    {
      val name = "yaf_fpexport"
      val fields = Seq(
        ("firstPacketBanner", VARLEN),
        ("secondPacketBanner", VARLEN),
        ("reverseFirstPacketBanner", VARLEN)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc017)
    }
    {
      val name = "yaf_payload"
      val fields = Seq(("payload", VARLEN), ("reversePayload", VARLEN))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc018)
    }
    {
      val name = "yaf_irc"
      val fields = Seq(("basicList", VARLEN))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc200)
    }
    {
      val name = "yaf_dhcp"
      val fields = Seq(("dhcpFingerPrint", VARLEN), ("dhcpVendorCode", VARLEN))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc201)
    }
    {
      val name = "yaf_dnp"
      val fields = Seq(("subTemplateList", VARLEN))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc202)
    }
    {
      val name = "yaf_dnp_rec"
      val fields = Seq(
        ("dnp3SourceAddress", 2),
        ("dnp3DestinationAddress", 2),
        ("dnp3Function", 1),
        ("paddingOctets", 3),
        ("dnp3ObjectData", VARLEN)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc203)
    }
    {
      val name = "yaf_modbus"
      val fields = Seq(("basicList", VARLEN))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc204)
    }
    {
      val name = "yaf_enip"
      val fields = Seq(("basicList", VARLEN))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc205)
    }
    {
      val name = "yaf_rtp"
      val fields = Seq(("rtpPayloadType", 1), ("reverseRtpPayloadType", 1))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc206)
    }
    {
      val name = "yaf_ssl_cert_full"
      val fields = Seq(("basicList", VARLEN))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc207)
    }
    {
      val name = "yaf_dhcp_options"
      val fields = Seq(("basicList", VARLEN), ("dhcpVendorCode", VARLEN))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc208)
    }
    {
      val name = "yaf_dhcp"
      val fields = Seq(
        ("dhcpFingerPrint", VARLEN),
        ("dhcpVendorCode", VARLEN),
        ("reverseDhcpFingerPrint", VARLEN),
        ("reverseDhcpVendorCode", VARLEN)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc211)
    }
    {
      val name = "yaf_dhcp_options"
      val fields = Seq(
        ("basicList", VARLEN),
        ("dhcpVendorCode", VARLEN),
        ("basicList", VARLEN),
        ("reverseDhcpVendorCode", VARLEN)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc218)
    }
    {
      val name = "yaf_pop3"
      val fields = Seq(("basicList", VARLEN))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc300)
    }
    {
      val name = "yaf_tftp"
      val fields = Seq(("tftpFilename", VARLEN), ("tftpMode", VARLEN))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc400)
    }
    {
      val name = "yaf_slp"
      val fields =
        Seq(("basicList", VARLEN), ("slpVersion", 1), ("slpMessageType", 1), ("paddingOctets", 6))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc500)
    }
    {
      val name = "yaf_http"
      val fields = Seq(
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc600)
    }
    {
      val name = "yaf_ftp"
      val fields = Seq(
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc700)
    }
    {
      val name = "yaf_imap"
      val fields = Seq(
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc800)
    }
    {
      val name = "yaf_rtsp"
      val fields = Seq(
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xc900)
    }
    {
      val name = "yaf_sip"
      val fields = Seq(
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xca00)
    }
    {
      val name = "yaf_new_ssl"
      val fields = Seq(
        ("basicList", VARLEN),
        ("sslServerCipher", 4),
        ("sslClientVersion", 1),
        ("sslCompressionMethod", 1),
        ("sslRecordVersion", 2),
        ("subTemplateList", VARLEN),
        ("sslServerName", VARLEN)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xca0a)
    }
    {
      val name = "yaf_new_ssl_cert"
      val fields = Seq(
        ("subTemplateList", VARLEN),
        ("subTemplateList", VARLEN),
        ("subTemplateList", VARLEN),
        ("sslCertSignature", VARLEN),
        ("sslCertSerialNumber", VARLEN),
        ("sslCertValidityNotBefore", VARLEN),
        ("sslCertValidityNotAfter", VARLEN),
        ("sslPublicKeyAlgorithm", VARLEN),
        ("sslPublicKeyLength", 2),
        ("sslCertVersion", 1),
        ("paddingOctets", 5),
        ("sslCertificateHash", VARLEN)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xca0b)
    }
    {
      val name = "yaf_smtp"
      val fields = Seq(
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN),
        ("basicList", VARLEN)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xcb00)
    }
    {
      val name = "yaf_ssh"
      val fields = Seq(("basicList", VARLEN))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xcc00)
    }
    {
      val name = "yaf_nntp"
      val fields = Seq(("basicList", VARLEN), ("basicList", VARLEN))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xcd00)
    }
    {
      val name = "yaf_dns"
      val fields = Seq(("subTemplateList", VARLEN))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xce00)
    }
    {
      val name = "yaf_dns_a"
      val fields = Seq(("sourceIPv4Address", 4))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xce01)
    }
    {
      val name = "yaf_dns_aaaa"
      val fields = Seq(("sourceIPv6Address", 16))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xce02)
    }
    {
      val name = "yaf_dns_cname"
      val fields = Seq(("dnsCName", VARLEN))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xce03)
    }
    {
      val name = "yaf_dns_mx"
      val fields = Seq(("dnsMXExchange", VARLEN), ("dnsMXPreference", 2), ("paddingOctets", 6))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xce04)
    }
    {
      val name = "yaf_dns_ns"
      val fields = Seq(("dnsNSDName", VARLEN))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xce05)
    }
    {
      val name = "yaf_dns_ptr"
      val fields = Seq(("dnsPTRDName", VARLEN))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xce06)
    }
    {
      val name = "yaf_dns_txt"
      val fields = Seq(("dnsTXTData", VARLEN))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xce07)
    }
    {
      val name = "yaf_dns_srv"
      val fields = Seq(
        ("dnsSRVTarget", VARLEN),
        ("dnsSRVPriority", 2),
        ("dnsSRVWeight", 2),
        ("dnsSRVPort", 2),
        ("paddingOctets", 2)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xce08)
    }
    {
      val name = "yaf_dns_soa"
      val fields = Seq(
        ("dnsSOAMName", VARLEN),
        ("dnsSOARName", VARLEN),
        ("dnsSOASerial", 4),
        ("dnsSOARefresh", 4),
        ("dnsSOARetry", 4),
        ("dnsSOAExpire", 4),
        ("dnsSOAMinimum", 4),
        ("paddingOctets", 4)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xce09)
    }
    {
      val name = "yaf_ssl"
      val fields = Seq(
        ("basicList", VARLEN),
        ("sslServerCipher", 4),
        ("sslClientVersion", 1),
        ("sslCompressionMethod", 1),
        ("paddingOctets", 2)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xce0a)
    }
    {
      val name = "yaf_ssl_cert"
      val fields = Seq(
        ("sslCertSignature", VARLEN),
        ("sslCertIssuerCountryName", VARLEN),
        ("sslCertIssuerOrgName", VARLEN),
        ("sslCertIssuerOrgUnitName", VARLEN),
        ("sslCertIssuerZipCode", VARLEN),
        ("sslCertIssuerState", VARLEN),
        ("sslCertIssuerCommonName", VARLEN),
        ("sslCertIssuerLocalityName", VARLEN),
        ("sslCertIssuerStreetAddress", VARLEN),
        ("sslCertSubCountryName", VARLEN),
        ("sslCertSubOrgName", VARLEN),
        ("sslCertSubOrgUnitName", VARLEN),
        ("sslCertSubZipCode", VARLEN),
        ("sslCertSubState", VARLEN),
        ("sslCertSubCommonName", VARLEN),
        ("sslCertSubLocalityName", VARLEN),
        ("sslCertSubStreetAddress", VARLEN),
        ("sslCertVersion", 1)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xce0b)
    }
    {
      val name = "yaf_mysql"
      val fields = Seq(("subTemplateList", VARLEN), ("mysqlUsername", VARLEN))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xce0c)
    }
    {
      val name = "yaf_mysql_txt"
      val fields = Seq(("mysqlCommandText", VARLEN), ("mysqlCommandCode", 1), ("paddingOctets", 7))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xce0d)
    }
    {
      val name = "yaf_dns_ds"
      val fields = Seq(
        ("dnsDigest", VARLEN),
        ("dnsKeyTag", 2),
        ("dnsAlgorithm", 1),
        ("dnsDigestType", 1),
        ("paddingOctets", 4)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xce0e)
    }
    {
      val name = "yaf_dns_sig"
      val fields = Seq(
        ("dnsSigner", VARLEN),
        ("dnsSignature", VARLEN),
        ("dnsSignatureInception", 4),
        ("dnsSignatureExpiration", 4),
        ("dnsTTL", 4),
        ("dnsKeyTag", 2),
        ("dnsTypeCovered", 2),
        ("dnsAlgorithm", 1),
        ("dnsLabels", 1),
        ("paddingOctets", 6)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xce0f)
    }
    {
      val name = "yaf_dns_nsec"
      val fields = Seq(("dnsHashData", VARLEN))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xce11)
    }
    {
      val name = "yaf_dns_key"
      val fields = Seq(
        ("dnsPublicKey", VARLEN),
        ("dnsFlags", 2),
        ("protocolIdentifier", 1),
        ("dnsAlgorithm", 1),
        ("paddingOctets", 4)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xce12)
    }
    {
      val name = "yaf_dns_nsec3"
      val fields = Seq(
        ("dnsSalt", VARLEN),
        ("dnsHashData", VARLEN),
        ("dnsIterations", 2),
        ("dnsAlgorithm", 1),
        ("paddingOctets", 5)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xce13)
    }
    {
      val name = "yaf_ssl_subcert"
      val fields = Seq(("sslObjectValue", VARLEN), ("sslObjectType", 1), ("paddingOctets", 7))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xce14)
    }
    {
      val name = "md_dns_full"
      val fields = Seq(
        ("flowStartMilliseconds", 8),
        ("flowEndMilliseconds", 8),
        ("sourceIPv4Address", 4),
        ("dnsTTL", 4),
        ("dnsQRType", 2),
        ("dnsHitCount", 2),
        ("paddingOctets", 4),
        ("dnsQName", VARLEN),
        ("dnsRName", VARLEN),
        ("observationDomainName", VARLEN)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xceef)
    }
    {
      val name = "yaf_dns_qr"
      val fields = Seq(
        ("subTemplateList", VARLEN),
        ("dnsQName", VARLEN),
        ("dnsTTL", 4),
        ("dnsQRType", 2),
        ("dnsQueryResponse", 1),
        ("dnsAuthoritative", 1),
        ("dnsNXDomain", 1),
        ("dnsRRSection", 1),
        ("dnsID", 2),
        ("paddingOctets", 4)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xcf00)
    }
    {
      val name = "tombstone_record"
      val fields = Seq(
        ("exporterConfiguredId", 2),
        ("exporterUniqueId", 2),
        ("tombstoneId", 4),
        ("subTemplateList", VARLEN)
      )
      val t = Template.newOptionsTemplate(2, infoModel, fields, name, None)
      session.add(t, 0xd001)
    }
    {
      val name = "tombstone_access_record"
      val fields = Seq(("exportingProcessId", 4), ("observationTimeSeconds", 4))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xd002)
    }
    {
      val name = "yaf_stats_options"
      val fields = Seq(
        ("observationDomainId", 4),
        ("exportingProcessId", 4),
        ("exporterIPv4Address", 4),
        ("observationTimeSeconds", 4),
        ("systemInitTimeMilliseconds", 8),
        ("exportedFlowRecordTotalCount", 8),
        ("packetTotalCount", 8),
        ("droppedPacketTotalCount", 8),
        ("ignoredPacketTotalCount", 8),
        ("notSentPacketTotalCount", 8),
        ("expiredFragmentCount", 4),
        ("assembledFragmentCount", 4),
        ("flowTableFlushEventCount", 4),
        ("flowTablePeakCount", 4),
        ("meanFlowRate", 4),
        ("meanPacketRate", 4)
      )
      val t = Template.newOptionsTemplate(2, infoModel, fields, name, None)
      session.add(t, 0xd003)
    }
    {
      val name = "tombstone_record"
      val fields = Seq(
        ("observationDomainId", 4),
        ("exportingProcessId", 4),
        ("exporterConfiguredId", 2),
        ("paddingOctets", 6),
        ("tombstoneId", 4),
        ("observationTimeSeconds", 4),
        ("tombstoneAccessList", VARLEN)
      )
      val t = Template.newOptionsTemplate(2, infoModel, fields, name, None)
      session.add(t, 0xd004)
    }
    {
      val name = "tombstone_access_record"
      val fields = Seq(("certToolId", 4), ("observationTimeSeconds", 4))
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xd005)
    }
    // 0xd006 and 0xd007 are defined by ExportStream itself
    // {
    //  val fields = Seq(
    //    ("privateEnterpriseNumber", 4),
    //    ("informationElementId", 2),
    //    ("informationElementDataType", 1),
    //    ("informationElementSemantics", 1),
    //    ("informationElementUnits", 2),
    //    ("informationElementRangeBegin", 8),
    //    ("informationElementRangeEnd", 8),
    //    ("informationElementName", VARLEN),
    //    ("informationElementDescription", VARLEN))
    //  val t = Template.newOptionsTemplate(2, infoModel, fields)
    //  session.add(t, 0xd006)
    // }
    // {
    //  val fields = Seq(
    //    ("templateId", 2),
    //    ("templateName", VARLEN),
    //    ("templateDescription", VARLEN))
    //  val t = Template.newOptionsTemplate(1, infoModel, fields)
    //  session.add(t, 0xd007)
    // }
    {
      val name = "md_dns_dedup_full"
      val fields = Seq(
        ("monitoringIntervalStartMilliSeconds", 8),
        ("monitoringIntervalEndMilliSeconds", 8),
        ("flowStartMilliseconds", 8),
        ("observedDataTotalCount", 8),
        ("sourceIPv6Address", 16),
        ("sourceIPv4Address", 4),
        ("yafFlowKeyHash", 4),
        ("observationDomainName", VARLEN),
        ("observedData", VARLEN),
        ("sslCertSerialNumber", VARLEN),
        ("sslCertIssuerCommonName", VARLEN),
        ("sslCertSerialNumber", VARLEN),
        ("sslCertIssuerCommonName", VARLEN)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xdaaa)
    }
    {
      val name = "md_ssl_dedup"
      val fields = Seq(
        ("flowStartMilliseconds", 8),
        ("flowEndMilliseconds", 8),
        ("observedDataTotalCount", 8),
        ("sslCertSerialNumber", VARLEN),
        ("sslCertIssuerCommonName", VARLEN),
        ("observationDomainName", VARLEN)
      )
      val t = Template.newTemplate(infoModel, fields, name, None)
      session.add(t, 0xdaaf)
    }
    ()
  }
}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
