// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.mothra.tools.silk_appender

import java.io.{FileNotFoundException, InputStreamReader}
import java.util.Properties
import scala.util.{Failure, Success, Try}

import org.apache.hadoop.conf.Configuration
import org.apache.hadoop.fs.Path
import org.cert.netsa.io.silk.SilkConfig

object Main extends App {

  def usage(): Unit = {
    print("""
Usage: spark-submit mothra-tool-silk-appender.jar <s1> <d1> [<s2> <d2> ...]

s1..sn: Source directories as Hadoop URIs
d1..dn: Matching destination directories as Hadoop URIs
""")
    System.exit(1)
  }

  def printVersion(): Unit = {
    Try {
      val resource = "/project-info.properties"
      val key = "projectInfo.version"
      val stream = Option(getClass.getResourceAsStream(resource)) match {
        case Some(s) => s
        case None    => throw new Exception(s"Did not find resource file '$resource'")
      }
      val props = new Properties()
      props.load(stream)
      Option(props.getProperty(key)) match {
        case Some(v) => v
        case None    => throw new Exception(s"Did not find key '$key'")
      }
    } match {
      case Success(version) => println(version)
      case Failure(err)     => print(s"Unable to get version property: ${err.toString}\n")
    }

    System.exit(0)
  }

  if (args.length == 0 || args.length % 2 != 0) {
    if (
      args.exists { (x: String) =>
        x == "-V" || x == "--version"
      }
    ) {
      printVersion()
    }
    usage()
  }

  implicit val conf: Configuration = new Configuration()

  // Get and parse the silk.conf file.  First, check the top-level of
  // 'rootDir'; if it does not exist, use the registry
  private def getSilkConfig(rootDir: Path): SilkConfig = {
    val fs = rootDir.getFileSystem(conf)
    val path = new Path(rootDir, "silk.conf")
    if (fs.exists(path)) {
      SilkConfig(new InputStreamReader(fs.open(path)))
    } else {
      Option(
        getClass().getResourceAsStream("/org/cert/netsa/mothra/tools/silk_appender/silk.conf")
      ) match {
        case Some(stream) => SilkConfig(new InputStreamReader(stream))
        case None =>
          throw new FileNotFoundException(s"silk.conf registry does not exist (resource not found)")
      }
    }
  }

  val loadMappings = args
    .grouped(2)
    .map {
      case Array(x, y) =>
        val rootDir = new Path(y)
        val silkConfig = getSilkConfig(rootDir)
        DirMapping(new Path(x), rootDir, silkConfig)
      case _ => throw new RuntimeException("Internal error: Unexpected misaligned mapping")
    }
    .toSeq

  val loader = Loader()

  // Add the mappings to the Loader
  loader.addMappings(loadMappings)
  // Run the Loader
  loader.run()

}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
