// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.mothra.tools

import org.apache.hadoop.fs.{Path => HadoopPath}
import org.cert.netsa.mothra.packer.CorePacker

class FileJoinerOptions extends ToolOptions("mothra-filejoiner") {

  note("Required options:\n")

  var targetDirs: Seq[HadoopPath] = Seq()
  arg[HadoopPath]("TARGET-1 .. TARGET-N")
    .required()
    .unbounded()
    .foreach(targetDirs :+= _)
    .text("HDFS directories in which to join files.")
    .validate(checkHadoopDirExists("Argument TARGET"))

  note("\nOptions:\n")

  help("help").text("Print this message and exit.")

  version("version").text(s"Print the version number of $toolName and exit.")

  note("")

  var minCountToJoin: Option[Int] = None
  opt[Int]("min-count-to-join")
    .valueName("N")
    .foreach(n => minCountToJoin = Some(n))
    .text(s"Only process prefixes with at least N files. (Def. 2)")
    .validate(checkPos("min-count-to-join"))

  var spawnThread: Option[String] = None
  opt[String]("spawn-thread")
    .valueName("MODE")
    .foreach(m => spawnThread = Some(m))
    .text("""Break work down in multiple threads on the basis of
                         'by-directory' or 'by-prefix'. (Def. 'by-directory')""")
    .validate {
      case "by-directory" => success
      case "by-prefix"    => success
      case _ => failure("Option --spawn-thread must be one of: 'by-directory', 'by-prefix'")
    }

  var maxThreads: Option[Int] = None
  opt[Int]("max-threads")
    .valueName("N")
    .foreach(n => maxThreads = Some(n))
    .text("Run no more than N joiner threads at once. (Def. 6)")
    .validate(checkPos("max-threads"))

  note("\nOptions that control the format of joined files:\n")

  var compression: Option[String] = None
  opt[String]("compression")
    .valueName("CODEC")
    .foreach(c => compression = Some(parseCompression(c)))
    .text(
      s"Compress joined files using this method.${if (CorePacker.DEFAULT_COMPRESSION == "") ""
        else s" (Def. ${CorePacker.DEFAULT_COMPRESSION})"}"
    )
    .validate(checkCompression("compression"))

  var maximumSize: Option[Long] = None
  opt[Long]("maximum-size")
    .valueName("N")
    .foreach(n => maximumSize = Some(n))
    .text("Close output files after their size reaches N bytes.")
    .validate(checkPos("maximum-size"))

}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
