// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.mothra.tools

import scala.util.control.NonFatal

abstract class Tool(val toolName: String, val toolPrefix: String) {

  /** Options parser for the tool. (Should be a subtype in subclasses.) */
  val options: ToolOptions

  /** Execute a tool after argument parsing. */
  def run(): Unit

  /** Handle options parsing and top-level error handling while calling run() */
  def main(args: Array[String]): Unit = {
    if (!options.parse(args)) sys.exit(1)
    try {
      run()
    } catch {
      // Short display of non-fatal exceptions unless debug is set
      case NonFatal(ex) if !options.debug =>
        Console.err.println(s"$toolName Error: $ex")
        sys.exit(1)
    }
  }

  /** Sets the named property under this [[toolPrefix]] to the value. */
  def setProp(name: String, value: String): Unit = sys.props(s"$toolPrefix.$name") = value

  /** Sets the named property under this [[toolPrefix]] to the value if `Some`. Does not change the
    * value of the property if the value is `None`. Note: this should not be changed to remove when
    * `None` is given, otherwise values given as properties will be ignored.
    */
  def setProp(name: String, value: Option[String]): Unit =
    value.foreach(sys.props(s"$toolPrefix.$name") = _)

}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
