// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.mothra.tools

import java.nio.file.{Path => JavaPath}
import java.time.Duration
import scala.math.Ordering.Implicits.*
import scala.util.Try

import org.apache.hadoop.conf.{Configuration => HadoopConfiguration}
import org.apache.hadoop.fs.{Path => HadoopPath}
import org.apache.hadoop.io.compress.CompressionCodecFactory
import org.cert.netsa.io.ipfix.InfoModel
import org.cert.netsa.mothra.packer.Version

import scopt.{OptionParser, Read}

abstract class ToolOptions(val toolName: String) extends OptionParser[Unit](toolName) {

  def programVersion: String = s"$toolName ${Version.get()}"

  // Default header of program name and version
  head(programVersion)

  // Always include a hidden debug option
  var debug: Boolean = false
  opt[Unit]("debug")
    .hidden()
    .foreach(_ => debug = true)
    .text("Include verbose debugging output on stderr")

  /* Additional parsing of options for paths, compression methods, etc. */

  implicit val infoModel: InfoModel = InfoModel.getCERTStandardInfoModel()

  protected val hadoopConf = new HadoopConfiguration()

  protected def parseHadoopPath(s: String): HadoopPath = new HadoopPath(s)

  protected def parseJavaPath(s: String): JavaPath = java.nio.file.Paths.get(s)

  protected def parseDuration(s: String): Duration = Duration.parse(s)

  protected def parseSize(s: String): Long =
    if (s.toLowerCase.endsWith("t")) {
      s.init.toLong * 1024 * 1024 * 1024 * 1024
    } else if (s.toLowerCase.endsWith("g")) {
      s.init.toLong * 1024 * 1024 * 1024
    } else if (s.toLowerCase.endsWith("m")) {
      s.init.toLong * 1024 * 1024
    } else if (s.toLowerCase.endsWith("k")) {
      s.init.toLong * 1024
    } else {
      s.toLong
    }

  protected def parseCompression(s: String): String =
    s match {
      case "" | "none" => ""
      case _           => s
    }

  /** Output human-readable size if possible. */

  protected def showSize(s: Long): String =
    if (s % 1024 != 0) {
      s"$s"
    } else if (s % (1024L * 1024L) != 0) {
      s"${s / 1024L}k"
    } else if (s % (1024L * 1024L * 1024L) != 0) {
      s"${s / (1024L * 1024L)}M"
    } else if (s % (1024L * 1024L * 1024L * 1024L) != 0) {
      s"${s / (1024L * 1024L * 1024L)}G"
    } else {
      s"${s / (1024L * 1024L * 1024L * 1024L)}T"
    }

  /* Command-line option readers for new types. Mostly implicit. */

  protected implicit val hadoopPathRead: Read[HadoopPath] = Read.reads(parseHadoopPath)

  protected implicit val javaPathRead: Read[JavaPath] = Read.reads(parseJavaPath)

  protected implicit val durationRead: Read[Duration] = Read.reads(parseDuration)

  /* Use this one like opt[Long](names...)(sizeRead) */
  protected val sizeRead: Read[Long] = Read.reads(parseSize)

  protected def checkPos[T: Numeric](desc: String)(n: T): Either[String, Unit] = {
    val zero = implicitly[Numeric[T]].zero
    if (n > zero) success
    else failure(s"$desc must be >= 1 but was given '$n'")
  }

  protected def checkNonNeg[T: Numeric](desc: String)(n: T): Either[String, Unit] = {
    val zero = implicitly[Numeric[T]].zero
    if (n >= zero) success
    else failure(s"$desc must be >= 0 but was given '$n'")
  }

  protected def checkGE[T: Numeric](desc: String, minVal: T)(n: T): Either[String, Unit] =
    if (n >= minVal) success
    else {
      failure(s"$desc must be >= $minVal but was given '$n'")
    }

  protected def checkRange[T: Numeric](desc: String, minVal: T, maxVal: T)(
    n: T
  ): Either[String, Unit] =
    if (n >= minVal && n <= maxVal) success
    else {
      failure(s"$desc must be >= $minVal and <= $maxVal but was given '$n'")
    }

  protected def checkCompression(desc: String)(codecName: String): Either[String, Unit] =
    if (codecName == "" || codecName == "none") success
    else {
      val codec = Option(new CompressionCodecFactory(hadoopConf).getCodecByName(codecName))
      val compressor = codec.flatMap(c => Try(Option(c.createCompressor)).toOption.flatten)
      val smokeTest = compressor.flatMap(c => Try(c.setInput(Array(1, 2, 3, 4, 5), 0, 4)).toOption)
      if (codec.isEmpty) failure(s"$desc can't find compression codec '$codecName'")
      else if (compressor.isEmpty)
        failure(s"$desc can't instantiate compression codec '$codecName'")
      else if (smokeTest.isEmpty) failure(s"$desc can't initialize compression codec '$codecName'")
      else success
    }

  protected def checkHadoopDirExists(desc: String)(path: HadoopPath): Either[String, Unit] = {
    // These operations could fail, but the exceptions will be meaningful
    val fs = path.getFileSystem(hadoopConf)
    if (fs.exists(path) && fs.getFileStatus(path).isDirectory()) {
      success
    } else {
      failure(s"$desc can't find (Hadoop) directory '$path'")
    }
  }

  protected def checkHadoopFileExists(desc: String)(path: HadoopPath): Either[String, Unit] = {
    val fs = path.getFileSystem(hadoopConf)
    if (fs.exists(path) && fs.getFileStatus(path).isFile()) {
      success
    } else {
      failure(s"$desc can't find (Hadoop) file '$path'")
    }
  }

  protected def checkJavaDirExists(desc: String)(path: JavaPath): Either[String, Unit] = {
    if (path.toFile.isDirectory) success
    else {
      failure(s"$desc can't find (local) directory '$path'")
    }
  }

  protected def checkIEName(desc: String)(name: String): Either[String, Unit] = {
    if (name.contains(",")) {
      failure(s"$desc must not contain ',', but does")
    } else if (infoModel.get(name).isDefined) {
      success
    } else {
      failure(s"$desc can't find IPFIX Information Element '$name'")
    }
  }

}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
