// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.mothra.packer

import enumeratum.values.{ShortEnum, ShortEnumEntry}

/**
  * Results of [[DirWatcher]] methods.
  *
  * @see [[DirWatcherResult$ The companion object]] for a complete
  * description and the list of values.
  *
  * @param value The short value representing the result.
  * @param name The name of the result.
  */
private[mothra] sealed abstract class DirWatcherResult(val value: Short, val name: String)
    extends ShortEnumEntry

/**
  * Results of [[DirWatcher]] methods.
  *
  * May be converted to and from [[scala.Short Short]] values.
  */
private[mothra] case object DirWatcherResult extends ShortEnum[DirWatcherResult] {

  /**
    * Collection of all known valid DirWatcherResult values.
    * @group Members
    */
  lazy val values = findValues
  // Note: lazy to avoid cyclic initialization problem

  /**
    * Converts a short value to a DirWatcherResult.
    * @return The DirWatcherResult represented by this value.
    * @throws java.util.NoSuchElementException if the short represents no
    *     known DirWatcherResult.
    * @group Members
    */
  def apply(v: Short): DirWatcherResult =
    withValueOpt(v).getOrElse(
      throw new NoSuchElementException(s"Unrecognized DirWatcherResult $v")
    )

  /**
    * Finds the DirWatcherResult whose name is `name`.
    *
    * The function seaches linearly through the list of units.
    *
    * @return The DirWatcherResult represented by `name`.
    * @throws java.util.NoSuchElementException if the name represents
    *     no known DirWatcherResult.
    */
  def withName(name: String): DirWatcherResult =
    values.find(_.name == name).getOrElse(
      throw new NoSuchElementException(s"Unrecognized DirWatcherResult $name")
    )

  /**
    * Finds the DirWatcherResult whose name is `name` as an Option.
    *
    * The function seaches linearly through the list of units.
    *
    * @return The DirWatcherResult represented by `name`.
    */
  def withNameOpt(name: String): Option[DirWatcherResult] =
    values.find(_.name == name)


  /**
    * DirWatcherResult indicating that scanning of the directory should
    * continue.
    * @group Values
    */
  case object CONTINUE extends DirWatcherResult(0, "Continue")
  /**
    * DirWatcherResult indicating that this scan of the directory should stop
    * and skip all remaining entries.
    * @group Values
    */
  case object SKIP extends DirWatcherResult(1, "Skip")
  /**
    * DirWatcherResult indicating that the DirWatcher should terminate.
    * @group Values
    */
  case object TERMINATE extends DirWatcherResult(2, "Terminate")
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
