// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.mothra.tools

import org.apache.hadoop.fs.{Path => HadoopPath}
import org.cert.netsa.mothra.packer.CorePacker

class InvariantPackerOptions extends ToolOptions("mothra-invariantpacker") {

  note("Required options:\n")

  var incomingDir: HadoopPath = _
  opt[HadoopPath]("incoming-dir").valueName("DIR").required()
    .foreach(incomingDir = _)
    .text("HDFS directory to watch for incoming files.")
    .validate(checkHadoopDirExists("Option --incoming-dir"))

  var outgoingDir: HadoopPath = _
  opt[HadoopPath]("outgoing-dir").valueName("DIR").required()
    .foreach(outgoingDir = _)
    .text("HDFS directory to store packed file repository.")
    .validate(checkHadoopDirExists("Option --outgoing-dir"))

  var packingLogicFile: HadoopPath = _
  opt[HadoopPath]("packing-logic").valueName("FILE").required()
    .foreach(packingLogicFile = _)
    .text("HDFS file containing packing configuration.")
    .validate(checkHadoopFileExists("Option --packing-logic"))

  note("\nOptions:\n")

  help("help")
    .text("Print this message and exit.")

  version("version")
    .text(s"Print the version number of $toolName and exit.")

  note("")

  var oneShot: Boolean = false
  opt[Unit]("one-shot")
    .foreach(_ => oneShot = true)
    .text("Process INCOMING-DIR once and then exit.")
  opt[Unit]("oneshot").hidden()
    .foreach(_ => oneShot = true)

  var maxThreads: Option[Int] = None
  opt[Int]("max-threads").valueName("N")
    .foreach(n => maxThreads = Some(n))
    .text(s"Write to N output files simultaneously. (Def. 6)")
    .validate(checkPos("Option --max-threads"))

  var pollingInterval: Option[Int] = None
  opt[Int]("polling-interval").valueName("N")
    .foreach(n => pollingInterval = Some(n))
    .text(s"Seconds between scans for incoming files. (Def. 15)")
    .validate(checkPos("Option --polling-interval"))

  note("\nOptions that control the format of packed files:\n")

  var compression: Option[String] = None
  opt[String]("compression").valueName("CODEC")
    .foreach(c => compression = Some(parseCompression(c)))
    .text(s"Compress sanitized files using this method.${
      if ( CorePacker.DEFAULT_COMPRESSION == "" ) ""
        else s" (Def. ${CorePacker.DEFAULT_COMPRESSION})"}")
    .validate(checkCompression("compression"))

  var maximumSize: Option[Long] = None
  opt[Long]("maximum-size").valueName("N")
    .foreach(n => maximumSize = Some(n))
    .text(s"""Close output files after their size reaches
                             N (>= ${1L << 19}) bytes.""")
    .validate(checkGE("Option --maximum-size", 1L << 19))

  var observationDomainId: Option[Int] = None
  opt[Int]("observation-domain-id").valueName("ID")
    .foreach(id => observationDomainId = Some(id))
    .text("Output files use ID for Observation Domain. (Def. 0)")
    .validate(checkNonNeg("Option --observation-domain-id"))

  note("\nOptions that control closing packed files:\n")

  var outputIdleTime: Option[Int] = None
  opt[Int]("output-idle-time").valueName("N")
    .foreach(n => outputIdleTime = Some(n))
    .text("""Keep output files open for N seconds after last
                             write. (Def. 900)""")
    .validate(checkGE("Option --output-idle-time", 60))

  note("\nOptions that control when output files are opened:\n")

  var maxInputAge: Option[Int] = None
  opt[Int]("max-input-age").valueName("N")
    .foreach(n => maxInputAge = Some(n))
    .text("""Open output when input is at least N seconds
                             old. (Def. 900)""")
    .validate(checkPos("Option --max-input-age"))

  var minInputCount: Option[Int] = None
  opt[Int]("min-input-count").valueName("N")
    .foreach(n => minInputCount = Some(n))
    .text("""Open output for which at least N input files
                             exist. (Def. 3)""")
    .validate(checkPos("Option --min-input-count"))

  var minInputSize: Option[Long] = None
  opt[Long]("min-input-size").valueName("N")
    .foreach(n => minInputSize = Some(n))
    .text(s"""Open output for which at least N bytes of input
                             exist. (Def. ${1L<<20})""")

  var fileCacheSize: Option[Int] = None
  opt[Int]("file-cache-size").valueName("N")
    .foreach(n => fileCacheSize = Some(n))
    .text(s"Keep up to N >= 128 files open for writing. (Def. 2000)")
    .validate({
      case n if n < 128 => failure(
        s"Option --file-cache-size must be >= 128 but was given '$n'")
      case _ => success
    })

}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
