// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix

import enumeratum.values.{ShortEnum, ShortEnumEntry}

/**
  * IPFIX Information Element Data Types as defined in RFC5102
  *
  * @see [[DataTypes$ The companion object]] for a complete
  * description and the list of values.
  *
  * @param value The short value representing the data type.
  * @param name The name of the data type.
  */
sealed abstract class DataTypes(val value: Short, val name: String)
    extends ShortEnumEntry

/**
  * IPFIX Information Element Data Types as defined in RFC5102
  *
  * A description of the abstract data type of an IPFIX Information
  * Element as defined in Section 3.1 of RFC5102.
  *
  * See https://www.iana.org/assignments/ipfix/ipfix.xhtml#ipfix-information-element-data-types
  *
  * May be converted to and from [[scala.Short Short]] values.
  */
case object DataTypes extends ShortEnum[DataTypes] {

  /**
    * Collection of all known valid information element data types.
    * @group Members
    */
  lazy val values = findValues
  // Note: lazy to avoid cyclic initialization problem

  /*
   * Maps from integers and from names to DataType objects
   */
  private[this] val (idmap, namemap) = {
    val types = List[DataType](
      org.cert.netsa.io.ipfix.datatype.OctetArray,
      org.cert.netsa.io.ipfix.datatype.Unsigned8,
      org.cert.netsa.io.ipfix.datatype.Unsigned16,
      org.cert.netsa.io.ipfix.datatype.Unsigned32,
      org.cert.netsa.io.ipfix.datatype.Unsigned64,
      org.cert.netsa.io.ipfix.datatype.Signed8,
      org.cert.netsa.io.ipfix.datatype.Signed16,
      org.cert.netsa.io.ipfix.datatype.Signed32,
      org.cert.netsa.io.ipfix.datatype.Signed64,
      org.cert.netsa.io.ipfix.datatype.Float32,
      org.cert.netsa.io.ipfix.datatype.Float64,
      org.cert.netsa.io.ipfix.datatype.Boolean,
      org.cert.netsa.io.ipfix.datatype.MacAddress,
      org.cert.netsa.io.ipfix.datatype.String,
      org.cert.netsa.io.ipfix.datatype.DateTimeSeconds,
      org.cert.netsa.io.ipfix.datatype.DateTimeMilliseconds,
      org.cert.netsa.io.ipfix.datatype.DateTimeMicroseconds,
      org.cert.netsa.io.ipfix.datatype.DateTimeNanoseconds,
      org.cert.netsa.io.ipfix.datatype.IPv4Address,
      org.cert.netsa.io.ipfix.datatype.IPv6Address,
      org.cert.netsa.io.ipfix.datatype.BasicList,
      org.cert.netsa.io.ipfix.datatype.SubTemplateList,
      org.cert.netsa.io.ipfix.datatype.SubTemplateMultiList
    )

    (Map.empty[Short, DataType] ++ (for (t <- types) yield (t.id.value, t)),
      (Map.empty[String, DataType] ++ (for (t <- types) yield (t.name, t))))
  }

  /** Returns the DataType having the specified `name`. */
  def getDataType(name: String): DataType = namemap(name)

  /** Returns the DataType having the specified `id`. */
  def getDataType(id: Short): DataType = idmap(id)

  /** Returns the DataType having the specified `id`. */
  def getDataType(id: Int): DataType = idmap(id.toShort)

  /** Returns the DataType that matches `dt`. */
  def getDataType(dt: DataTypes): DataType = idmap(dt.value)


  /**
    * Converts a short value to an information element data type.
    * @return The information element data type represented by this
    * value.
    * @throws java.util.NoSuchElementException if the short
    *     represents no known information element data type.
    * @group Members
    */
  def apply(v: Short): DataTypes =
    withValueOpt(v).getOrElse(
      throw new NoSuchElementException(s"Unrecognized data type identifier $v")
    )

  /**
    * Finds the IPFIX IE data type value whose name is `name`.
    *
    * The function seaches linearly through the list of data types.
    *
    * @return The information element data type represented by `name`.
    * @throws java.util.NoSuchElementException if the name
    *     represents no known information element data type.
    */
  def withName(name: String): DataTypes =
    values.find(_.name == name).getOrElse(
      throw new NoSuchElementException(s"Unrecognized data type name $name")
    )


  /**
    * Finds the IPFIX IE data type value whose name is `name` as an
    * Option.
    *
    * The function seaches linearly through the list of data types.
    *
    * @return The information element data type represented by `name`
    * or [[scala.None]] if `name` represents no known data type.
    */
  def withNameOpt(name: String): Option[DataTypes] = values.find(_.name == name)


  /**
    * IPFIX IE data type value denoting an octet array.
    * @group Values
    */
  case object OctetArray extends DataTypes(0, "octetArray")
  /**
    * IPFIX IE data type value denoting an 8-bit unsigned value.
    * @group Values
    */
  case object Unsigned8 extends DataTypes(1, "unsigned8")
  /**
    * IPFIX IE data type value denoting a 16-bit unsigned value.
    * @group Values
    */
  case object Unsigned16 extends DataTypes(2, "unsigned16")
  /**
    * IPFIX IE data type value denoting a 32-bit unsigned value.
    * @group Values
    */
  case object Unsigned32 extends DataTypes(3, "unsigned32")
  /**
    * IPFIX IE data type value denoting a 64-bit unsigned value.
    * @group Values
    */
  case object Unsigned64 extends DataTypes(4, "unsigned64")
  /**
    * IPFIX IE data type value denoting an 8-bit signed value.
    * @group Values
    */
  case object Signed8 extends DataTypes(5, "signed8")
  /**
    * IPFIX IE data type value denoting a 16-bit signed value.
    * @group Values
    */
  case object Signed16 extends DataTypes(6, "signed16")
  /**
    * IPFIX IE data type value denoting a 32-bit signed value.
    * @group Values
    */
  case object Signed32 extends DataTypes(7, "signed32")
  /**
    * IPFIX IE data type value denoting a 64-bit signed value.
    * @group Values
    */
  case object Signed64 extends DataTypes(8, "signed64")
  /**
    * IPFIX IE data type value denoting a 32-bit floating point value.
    * @group Values
    */
  case object Float32 extends DataTypes(9, "float32")
  /**
    * IPFIX IE data type value denoting a 64-bit floating point value.
    * @group Values
    */
  case object Float64 extends DataTypes(10, "float64")
  /**
    * IPFIX IE data type value denoting a boolean value.
    * @group Values
    */
  case object Boolean extends DataTypes(11, "boolean")
  /**
    * IPFIX IE data type value denoting a MAC address.
    * @group Values
    */
  case object MacAddress extends DataTypes(12, "macAddress")
  /**
    * IPFIX IE data type value denoting a string.
    * @group Values
    */
  case object String extends DataTypes(13, "string")
  /**
    * IPFIX IE data type value denoting a date-time represented in
    * seconds.
    * @group Values
    */
  case object DateTimeSeconds extends DataTypes(14, "dateTimeSeconds")
  /**
    * IPFIX IE data type value denoting a date-time represented in
    * milliseconds.
    * @group Values
    */
  case object DateTimeMilliseconds extends DataTypes(15, "dateTimeMilliseconds")
  /**
    * IPFIX IE data type value denoting a date-time represented in
    * microseconds.
    * @group Values
    */
  case object DateTimeMicroseconds extends DataTypes(16, "dateTimeMicroseconds")
  /**
    * IPFIX IE data type value denoting a date-time represented in
    * nanoseconds.
    * @group Values
    */
  case object DateTimeNanoseconds extends DataTypes(17, "dateTimeNanoseconds")
  /**
    * IPFIX IE data type value denoting an IPv4 address.
    * @group Values
    */
  case object IPv4Address extends DataTypes(18, "ipv4Address")
  /**
    * IPFIX IE data type value denoting an IPv6 address.
    * @group Values
    */
  case object IPv6Address extends DataTypes(19, "ipv6Address")
  /**
    * IPFIX IE data type value denoting a basicList structured data.
    * @group Values
    */
  case object BasicList extends DataTypes(20, "basicList")
  /**
    * IPFIX IE data type value denoting subTemplateList structured
    * data.
    * @group Values
    */
  case object SubTemplateList extends DataTypes(21, "subTemplateList")
  /**
    * IPFIX IE data type value denoting a subTemplateMultiList
    * structured data.
    * @group Values
    */
  case object SubTemplateMultiList extends DataTypes(22, "subTemplateMultiList")
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
