// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix

import scala.collection.immutable.VectorBuilder
import scala.collection.mutable.WeakHashMap


/**
  * Extracts the specified field from a [[Record]] or a Sub-Record and
  * casts the field to a type.
  *
  * A simple string returns the named IE in the top-level record:
  *
  * val protoEx = DeepFieldExtractor[Short]("protocolIdentifier")
  * val proto = protoEx.extractFrom(r.get)
  * proto: Option[Short] = Some(6)
  *
  * One or more template names may preceed the named IE.  Separate the
  * sequence of template names and the IE by
  * `DeepFieldExtractor$.delim`.
  *
  * val initialTcpEx = DeepFieldExtractor[Short]("yaf_tcp_rev/initialTCPFlags")
  * val initialTcp = initialTcpEx.extractFrom(r.get)
  * initialTcp: Option[Short] = Some(2)
  *
  * Prepend the `DeepFieldExtractor$.deepPrefix` to either a lone IE
  * name or a template name and IE sequence to start searching for the
  * path at any depth within the Record.
  *
  * val unionTcpEx = DeepFieldExtractor[Short]("* /unionTCPFlags")
  * val unionTcp = unionTcpEx.extractFrom(r.get)
  * unionTcp: Option[Short] = Some(28)
  *
  * To get a BasicList containing a specific IE from a Record, wrap
  * the IE's name `ieName` as "basicList[ieName]" (see
  * `DeepFieldExtractor$.wrapAsBasicList`).  For these, the type
  * parameter T should always be `BasicList`.
  *
  * val agentEx = DeepFieldExtractor[BasicList]("* /basicList[httpUserAgent]")
  * val agent = agentEx.extractFrom(r.get)
  * agent: Option[org.cert.netsa.io.ipfix.BasicList] = Some((BL ...))
  *
  * Use the BasicList's `elements` method to get the contents of the
  * BasicList:
  *
  * val v = Vector.empty[String] ++ agent.get.elements
  *
  *
  * Maintains a mapping from [[Template]] to field position to improve
  * performance when extracting a field from multiple Records that
  * share the same Template.
  *
  * @tparam T The type to cast the field to.
  * @param path The path to the field to extract.
  *
  * @see SimpleFieldExtractor for a simplified version
  */
final case class DeepFieldExtractor[T] private (path: String)
    extends FieldExtractor[T]
{
  // import from the companion object
  import DeepFieldExtractor.{deepPrefix, delim, unwrapBasicList, TemplateInfo}

  require(!path.startsWith(delim) && !path.endsWith(delim),
    s"Field path may not begin or end with the delimiter '${delim}'")

  /** Map from a template to its info */
  private[this] val tmplInfoMap = WeakHashMap.empty[Template, TemplateInfo]

  /** fieldName is the "basename" of the desired field or the name of
    * the InfoElement when looking for a BasicList.  "isList" is true
    * when a BasicList is wanted. */
  private[this] val (fieldName, isList) = {
    val ieName = path.split(delim).last
    val unwrapped = unwrapBasicList(ieName)
    (unwrapped.getOrElse(ieName), unwrapped.nonEmpty)
  }

  /** Whether the template path may begin at any level */
  private[this] val deep = path.startsWith(deepPrefix)

  /** List of Template names to search for */
  private[this] val templateNames =
    // drop leading "*/" and final ieName
    path.split(delim).drop(if ( deep ) 1 else 0).dropRight(1).toList

  /** Takes a Record and returns a pair containing the Record and the
    * TemplateInfo of the Record's Template.  Uses the tmplInfoMap to
    * get/cache the result. */
  private[this] def recordToRecInfoPair(record: Record): (Record, TemplateInfo)=
  {
    val info = synchronized {
      tmplInfoMap.getOrElseUpdate(
        record.template, TemplateInfo.ofTemplate(record.template, fieldName))
    }
    (record, info)
  }

  /** Extracts the field from this record */
  private[this] def simpleExtractor(recInfo: (Record, TemplateInfo)):
      Option[T] =
  {
    if ( !isList ) {
      recInfo._2.element.map { idx => recInfo._1(idx).asInstanceOf[T] }
    } else {
      val blIter = for {
        idx <- recInfo._2.basicLists.iterator
        bl = recInfo._1(idx).asInstanceOf[BasicList]
        if ( bl.infoElement.name == fieldName )
          } yield bl
      if ( blIter.isEmpty ) {
        None
      } else {
        Some(blIter.next().asInstanceOf[T])
      }
    }
  }

  /** Returns an iterator over the STL and STML on this record. */
  private[this] def listElements(recInfo: (Record, TemplateInfo)):
      Iterator[ListElement] =
  {
    for {
      pos <- recInfo._2.subTemplates.iterator
    } yield recInfo._1(pos).asInstanceOf[ListElement]
  }

  /** When the Template of the `stl` is named `name`, returns an
    * iterator over (Record,TemplateInfo) pairs for all Records in
    * `stl`.  Otherwise returns an empty iterator. */
  private[this] def stlIterator(stl: SubTemplateList, name: String):
      Iterator[(Record, TemplateInfo)] =
  {
    if ( stl.template.name != Option(name) ) {
      Iterator.empty
    } else {
      val info = synchronized {
        tmplInfoMap.getOrElseUpdate(
          stl.template, TemplateInfo.ofTemplate(stl.template, fieldName))
      }
      for (rec <- stl.iterator) yield (rec, info)
    }
  }

  /** Returns an iterator over (Record,TemplateInfo) pairs for all
    * Records in `stml` where the Record's Template's name is
    * `name`. */
  private[this] def stmlIterator(stml: SubTemplateMultiList, name: String):
      Iterator[(Record, TemplateInfo)] =
  {
    for {
      rec <- stml.iterator.filter(_.template.name == Option(name))
    } yield recordToRecInfoPair(rec)
  }

  /** Returns an iterator over sub-records of this record that have a
    * Template named `name`. */
  private[this] def namedRecords(recInfo: (Record, TemplateInfo), name: String):
      Iterator[(Record, TemplateInfo)] =
  {
    for {
      listElem <- listElements(recInfo)
      subRecInfo <- listElem match {
        case stl: SubTemplateList => stlIterator(stl, name)
        case stml: SubTemplateMultiList => stmlIterator(stml, name)
      }
    } yield subRecInfo
  }

  /** Returns an iterator over sub-records of this record that match the
    * given sequence of names.  When names is empty, returns this
    * record. */
  private[this] def matchingRecords(
    recInfo: (Record, TemplateInfo), names: List[String]):
      Iterator[(Record, TemplateInfo)] =
  {
    if ( names.isEmpty ) {
      Iterator.single(recInfo)
    } else {
      for {
        subRecInfo <- namedRecords(recInfo, names.head)
        subRecInfo2 <- matchingRecords(subRecInfo, names.tail)
      } yield subRecInfo2
    }
  }

  /** Finds the `templateNames` path in this Record, starting from any
    * depth. */
  private[this] def findRecords(recInfo: (Record, TemplateInfo)):
      Iterator[(Record, TemplateInfo)] =
  {
    matchingRecords(recInfo, templateNames) ++ {
      if ( !deep ) {
        Iterator.empty
      } else {
        for {
          listElem <- listElements(recInfo)
          subrec <- listElem.iterator
        } yield recordToRecInfoPair(subrec)
      }
    }
  }

  /**
    * Extracts the field from a Record as an Option.
    */
  def extractFrom(record: Record): Option[T] = {
    val matchingIEs = for {
      recInfo <- findRecords(recordToRecInfoPair(record))
      value <- simpleExtractor(recInfo)
    } yield value
    if ( matchingIEs.isEmpty ) {
      None
    } else {
      Option(matchingIEs.next())
    }
  }

}

/** Holds constants and methods used by [[DeepFieldExtractor]]. */
object DeepFieldExtractor {
  /** The delimiter between template names. */
  val delim = "/"

  /** The prefix to signify starting at any level. */
  val deepPrefix = s"*${delim}"

  /** The prefix of an InfoElement name to signify a IE contained in a
    * BasicList */
  val basicListPrefix = "basicList["

  /** The suffix for a BasicList */
  val basicListSuffix = "]"

  /** Wraps a IE name as a BasicList. */
  def wrapAsBasicList(name: String): String =
    s"${basicListPrefix}${name}${basicListSuffix}"

  /** Unwraps a BasicList.  Returns None if not a BasicList. */
  def unwrapBasicList(value: String): Option[String] =
    if (value.startsWith(basicListPrefix) && value.endsWith(basicListSuffix)) {
      Option(value.substring(
        basicListPrefix.length, value.length - basicListSuffix.length))
    } else {
      None
    }


  /**
    * TemplateInfo maintains the positions of various [[InfoElement
    * InfoElements]] in a Template.  Note: TemplateInfo does not
    * maintain a handle to the [[Template]] it was created for.
    *
    * @param element Position in a Template of the IE that is the
    * target of the DeepFieldExtractor or None if not present.
    * @param basicLists: Position(s) in a Template of BasicList elements.
    * @param subTemplates: Position(s) in a Template of
    * SubTemplateList and SubTemplateMultiList elements.
    */
  private class TemplateInfo(
    val element: Option[Int],
    val basicLists: Vector[Int],
    val subTemplates: Vector[Int])

  /**
    * Factory for [[TemplateInfo]] instances.
    */
  private object TemplateInfo {
    /** Creates a [[TemplateInfo]] object for a Template, where `name` is
      * the name of an [[InfoElement]] to find on `template`. */
    def ofTemplate(template: Template, name: String): TemplateInfo = {
      var elem = Option.empty[Int]
      val bl = new VectorBuilder[Int]()
      val stl = new VectorBuilder[Int]()
      for (i <- 0 until template.size) {
        val ie = template(i)
        if ( elem.isEmpty && ie.name == name ) {
          elem = Option(i)
        } else {
          ie.dataTypeId match {
            case DataTypes.BasicList =>
              bl += i
            case DataTypes.SubTemplateMultiList | DataTypes.SubTemplateList =>
              stl += i
            case _ =>
          }
        }
      }
      new TemplateInfo(elem, bl.result(), stl.result())
    }
  }


}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
