// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix

import java.io.DataOutputStream
import java.nio.channels.{Channels, FileChannel, WritableByteChannel}
import java.nio.file.{Path, StandardOpenOption}
import java.nio.{ByteBuffer, BufferOverflowException}
import java.time.Instant
import scala.collection.immutable.{Set => ImmutableSet}
import scala.collection.mutable.{Set => ScalaSet}

// Globally replace "//LOG//" with "/*LOG*/" to enable logging
//LOG//import com.typesafe.scalalogging.StrictLogging

/**
  * ExportStream supports writing IPFIX templates and records to a
  * file.
  *
  * The caller creates an ExportStream instance with an output channel
  * and an associated Session.  The caller may then add [[Template
  * Templates]] or [[Record Records]] to the ExportStream, and the
  * ExportStream writes the IPFIX data to the output file.
  *
  * Since the Session is part of the ExportStream's constructor, all
  * IPFIX Messages in the stream use the same observation domain.
  *
  * By default, each exported IPFIX [[Message]] uses the current time
  * as its timestamp.  Set the `makeTimestamp` variable to change this
  * behavior.
  *
  * By default, an ExportStream adds an options record to describe
  * each [[InfoElement Information Element]] that is not part of the
  * standard information model, and ExportStream ignores any such
  * options records in the input stream.  Set the `describeElements`
  * variable to `false` to change this behavor, and use
  * `elementDescriptionTID` to view or set the template ID used by
  * these records.
  *
  * By default, an ExportStream adds an options record to describe
  * each [[Template]] it exports, and ExportStream ignores any such
  * options records in the input stream.  Set the `describeTemplates`
  * variable to `false` to change this behavor, and use
  * `templateDescriptionTID` to view or set the template ID used by
  * these records.
  *
  * NOTE: When copying data from an IPFIX input stream, ExportStream
  * may change the order in which templates and records
  * appear---specifically, templates that appear after a record in the
  * input may be written before that record.  ExportStream is aware of
  * the Meta-Data Options templates and records that describe
  * Information Elements and Templates, and, when ExportStream is not
  * creating and exporting those items itself, those templates and
  * records are copied from the input and written to the output in the
  * required order.  The user may call flush() to ensure some items
  * appear in the output before others.  The authors of this package
  * may need to consider providing some way for the users to know when
  * to call flush(), such as adding a callback methods to be invoked
  * when a new [[IpfixSet]] is read.
  *
  * @param out Where to write the IPFIX data.
  * @param session The session to use.
  */
class ExportStream protected (out: WritableByteChannel, val session: Session)
    //LOG//extends StrictLogging
{
  // import from the companion object
  import ExportStream.{DataWrittenCallback, ElementDescriptionRecord,
    TemplateDescriptionRecord}

  // Technically the next two counts are the total number of templates
  // or records that have been added to the export buffer, not
  // necessarily written to the stream.  The "total number" means
  // across writes of the buffer, not the number currently in the
  // buffer.
  private var tmplCount = 0
  private var recCount = 0

  /** Whether to export option records that describe private enterprise
    * information elements.
    *
    * If this is `true`, ExportStream generates these records itself and any
    * such records given to the `add` method are ignored.  The records are
    * generated as-needed and may appear anywhere in the output.
    *
    * If this is `false`, input records that describe private enterprise
    * elements are written to the output without ExportStream examining their
    * content.  This may result in duplicate descriptions in the output.
    *
    * In either case, ExportStream attempts to use the same template ID for
    * the element metadata template as seen in the incoming data.
    */
  var describeElements = true

  /** Whether to export option records that include the name and description of
    * each template.
    *
    * If this is `true`, ExportStream generates these records itself and any
    * such records given to the `add` method are ignored.  The records are
    * generated as-needed and may appear anywhere in the output.
    *
    * If this is `false`, input records that describe templates are written to
    * the output without ExportStream examining their content.  This may
    * result in duplicate descriptions in the output.
    *
    * In either case, ExportStream attempts to use the same template ID for
    * the template metadata template as seen in the incoming data.
    */
  var describeTemplates = true

  /** Callback to generate a timestamp */
  var makeTimestamp: ExportStream.MessageTimestamp = ExportStream.TimestampNow

  /* Values to use as arguments for closeSets() */
  private[this] val ELEMENT_DESCRIPTIONS = 1
  private[this] val TEMPLATE_DESCRIPTIONS = 2
  private[this] val TEMPLATES = 3
  private[this] val RECORDS = 4
  private[this] val ALL = RECORDS

  //LOG//private[this] lazy val closeSetsArgName = Array("zero",
  //LOG//  "ELEMENT_DESCRIPTIONS", "TEMPLATE_DESCRIPTIONS",
  //LOG//  "TEMPLATES", "RECORDS")

  /** The current RecordSet for descriptions of Private Enterprise
    * Information Elements */
  private[this] var elemDescRecSet: Option[RecordSet] = None

  /** The current RecordSet for descriptions of templates */
  private[this] var tempDescRecSet: Option[RecordSet] = None

  /** The current TemplateSet */
  private[this] var tmplSet: Option[TemplateSet] = None

  /** The current RecordSet */
  private[this] var recSet: Option[RecordSet] = None

  /** The set of templates that have been exported */
  var knownTmpl = ScalaSet.empty[Template]

  /** The set of info element descriptions that have been exported */
  private[this] val knownElement = ScalaSet.empty[InfoElement]

  /** Whether the options template that describes private enterprise
    * information elements has been exported yet. */
  private var elemDescWritten = false

  /** The options template that is used by the record that describes an
    * private enterprise information element */
  private var elemDescTmpl: Option[Template] = None

  /** The template ID for the elemDescTmpl */
  private var elemDescTID: Option[Int] = None

  /** Whether the options template that describes templates has been
    * exported yet. */
  private var tempDescWritten = false

  /** The options template that is used by the record that provides the
    * name and description of a template */
  private var tempDescTmpl: Option[Template] = None

  /** The template ID for the tempDescTmpl */
  private var tempDescTID: Option[Int] = None

  /** The number of data records written to the stream; used for setting
    * the sequence number */
  private var prevCount = 0

  /** The buffer to fill and write */
  private[this] val buffer: ByteBuffer = ByteBuffer.allocate(65535)

  buffer.position(Message.headerLength)


  /** Returns the template used to describe info elements, creating it if
    * necessary and adding it to the Session with the ID specified by
    * `elemDescTID`. */
  private[this] def elementDescriptionTemplate: Template = {
    elemDescTmpl.getOrElse {
      val model = session.infoModel
      val fields = Seq(
        IEFieldSpecifier(model, "privateEnterpriseNumber", 4),
        IEFieldSpecifier(model, "informationElementId", 2),
        IEFieldSpecifier(model, "informationElementDataType", 1),
        IEFieldSpecifier(model, "informationElementSemantics", 1),
        IEFieldSpecifier(model, "informationElementUnits", 2),
        IEFieldSpecifier(model, "informationElementRangeBegin", 8),
        IEFieldSpecifier(model, "informationElementRangeEnd", 8),
        IEFieldSpecifier(model, "informationElementName", VARLEN),
        IEFieldSpecifier(model, "informationElementDescription", VARLEN)
      )
      val t = Template.newOptionsTemplate(2, fields, model)
      elemDescTID match {
        case None => elemDescTID = Option(session.getOrAdd(t))
        case Some(tid) =>
          if ( session.getTemplate(tid).nonEmpty ) {
            throw new RuntimeException(
              s"Cannot use ${tid} for elementDescription Template" +
                " since already in use")
          }
          session.add(t, tid)
      }
      elemDescTmpl = Option(t)
      //LOG//logger.trace(f"Created elemDescTmpl ${elemDescTID.get}%#06x")
      t
    }
  }

  /** Returns the template used to describe templates, creating it if
    * necessary. */
  private[this] def templateDescriptionTemplate: Template = {
    tempDescTmpl.getOrElse {
      val model = session.infoModel
      val fields = Seq(
        IEFieldSpecifier(model, "templateId", 2),
        IEFieldSpecifier(model, "templateName", VARLEN),
        IEFieldSpecifier(model, "templateDescription", VARLEN)
      )
      val t = Template.newOptionsTemplate(1, fields, model)
      tempDescTID match {
        case None => tempDescTID = Option(session.getOrAdd(t))
        case Some(tid) =>
          if ( session.getTemplate(tid).nonEmpty ) {
            throw new RuntimeException(
              s"Cannot use ${tid} for templateDescription Template" +
                " since already in use")
          }
          session.add(t, tid)
      }
      tempDescTmpl = Option(t)
      //LOG//logger.trace(f"Created tempDescTmpl ${tempDescTID.get}%#06x")
      t
    }
  }

  ///**
  //  * Writes a stream that contains only an IPFIX message header.
  //  *
  //  * Used to ensure a valid IPFIX file is created when no templates
  //  * or records are added to the ExportStream.
  //  */
  //private[this] def writeEmptyStream(): ExportStream = {
  //  val timestamp = makeTimestamp.stamp()
  //  assert(buffer.position() == Message.headerLength)
  //  buffer.putShort(0, IPFIX_VERSION.toShort)
  //    .putShort(2, Message.headerLength.toShort)
  //    .putInt(4, timestamp.toInt)
  //    .putInt(8, prevCount)
  //    .putInt(12, session.observationDomain)
  //    .limit(Message.headerLength)
  //    .flip()
  //  out.write(buffer)
  //  // FIXME: Call the DataWritten callback here.
  //  buffer.clear()
  //    .position(Message.headerLength)
  //  this
  //}

  /**
    * Adds an IPFIX message header to the beginning of the buffer and writes
    * the current content of the buffer to the output stream.  After writing
    * the content, updates the `prevCount` global, clears the buffer and sets
    * the current position to the end of the IPFIX header.  Does nothing if
    * the current buffer position is not greater than the size of an IPFIX
    * message header.  Returns the number of bytes written to the stream.
    */
  private[this] def writeBuffer(): Long = {
    if ( buffer.position() <= Message.headerLength ) {
      0L
    } else {
      val timestamp = makeTimestamp.stamp()
      buffer.flip()
      buffer.putShort(0, IPFIX_VERSION.toShort)
        .putShort(2, buffer.limit().toShort)
        .putInt(4, timestamp.toInt)
        .putInt(8, prevCount)
        .putInt(12, session.observationDomain)
      prevCount = recCount
      out.write(buffer)
      val wrote = buffer.limit()
      buffer.clear()
        .position(Message.headerLength)
      wrote
    }
  }

  /**
    * Copies the contents of `set` to the buffer.
    *
    * If there is an overflow exception when copying the set into the buffer,
    * calls writeBuffer() and then copies `set` into the empty buffer.
    * Returns the number of octets written to the output stream or 0 if no
    * write occurrs.
    */
  private[this] def closeSet(set: IpfixSet): Long = {
    val pos = buffer.position()
    assert(pos > 0)
    var wrote = 0L
    try {
      set.toBuffer(buffer)
    } catch {
      case (_ : BufferOverflowException | _ : IllegalArgumentException) =>
        buffer.position(pos)
        wrote = writeBuffer()
        set.toBuffer(buffer)
    }
    wrote
  }

  /**
    * Closes the Sets at `level` and below.  Clears the global set variables
    * as the sets are closed.  Updates the global `recCount` and `tmplCount`
    * variables.  May write data to the output stream if the IPFIX message
    * buffer fills.  Returns the number of octets written to the stream or 0
    * if no write occurs.  If a write occurs, calls the dataWrittenCallback if
    * one exists and the `callDWC` parameter is not false.
    *
    * Use the constants ELEMENT_DESCRIPTIONS, TEMPLATE_DESCRIPTIONS,
    * TEMPLATES, RECORDS, or ALL for the `level` parameter.
    *
    * @param level What is being closed
    * @param callDWC Whether to call the dataWrittenCallback when data is
    * written
    */
  private[this] def closeSets(level: Int, callDWC: Boolean = true): Long = {
    assert(level >= ELEMENT_DESCRIPTIONS && level <= ALL)
    //LOG//logger.trace(s"closeSets(${closeSetsArgName(level)}) was called")
    var wrote = 0L
    for ( set <- elemDescRecSet ) {
      wrote += closeSet(set)
      recCount += set.size
      elemDescRecSet = None
    }
    if ( level >= TEMPLATE_DESCRIPTIONS ) {
      for ( set <- tempDescRecSet ) {
        wrote += closeSet(set)
        recCount += set.size
        tempDescRecSet = None
      }
    }
    if ( level >= TEMPLATES ) {
      for ( set <- tmplSet ) {
        wrote += closeSet(set)
        tmplCount += set.size
        tmplSet = None
      }
    }
    if ( level >= RECORDS ) {
      for ( set <- recSet ) {
        wrote += closeSet(set)
        recCount += set.size
        recSet = None
      }
    }
    if ( wrote > 0 && callDWC && _dataWrittenCallback.nonEmpty ) {
      _dataWrittenCallback.get.wrote(this, wrote)
    }
    wrote
  }

  /**
    * Writes an options record describing each information element in
    * `ieList`.  The caller is responsible for ensuring `ieList` only contains
    * private enterprise elements that have not yet had their descriptions
    * written previously (ie, that are not in `knownElement`).
    *
    * Adds the elements in `ieList` to `knownElement`.
    *
    * Creates and exports the element description template if necessary.
    */
  private[this] def elementDescriptionWrite(ieList: ImmutableSet[InfoElement]):
      Unit =
  {
    if ( ieList.nonEmpty ) {
      // write the element description template if not written
      val t = elementDescriptionTemplate
      if ( !elemDescWritten ) {
        addMetadataTemplate(t, elemDescTID)
        elemDescWritten = true
      }

      // get or open a new RecordSet for the element description recs
      var set = elemDescRecSet.getOrElse(RecordSet.empty(t, session))

      // handle each element
      for ( ie <- ieList ) {
        val rec = ElementDescriptionRecord(t, ie)
        while ( !{
          try {
            set.addRecord(rec)
            true
          } catch {
            case _: MessageTooLongException =>
              elemDescRecSet = Option(set)
              closeSets(ELEMENT_DESCRIPTIONS)
              set = RecordSet.empty(t, session)
              false
          }
        }) {}
      }
      elemDescRecSet = Option(set)
      knownElement ++= ieList
    }
  }

  /**
    * Writes an options record describing `template` if `template` has a name
    * or a description.
    *
    * Creates and exports the element description template if necessary.
    */
  private[this] def templateDescriptionWrite(template: Template, tid: Int):
      Unit =
  {
    val name = template.name.getOrElse("")
    val desc = template.description.getOrElse("")
    if ( name.nonEmpty || desc.nonEmpty ) {
      val rec = TemplateDescriptionRecord(
        templateDescriptionTemplate, tid, name, desc)
      val t = rec.template
      assert(t != template)
      if ( !tempDescWritten ) {
        addTemplate(t)
        closeSets(TEMPLATES)
        tempDescWritten = true
      }
      if ( tempDescRecSet.isEmpty ) {
        tempDescRecSet = Option(RecordSet.empty(t, session).addRecord(rec))
      } else {
        try {
          tempDescRecSet.get.addRecord(rec)
        } catch {
          case _: MessageTooLongException =>
            closeSets(TEMPLATE_DESCRIPTIONS)
            tempDescRecSet = Option(RecordSet.empty(t, session).addRecord(rec))
        }
      }
    }
  }


  /**
    * Checks to see if `template` has been seen before and, if not, adds it to
    * the global template set.  The parameter `tid` is an optional preferred
    * Template ID; see the `getOrAdd()` method on [[Session]] for details.
    * Returns the ID of the Template in the ExportStream's [[Session]].
    *
    * If `describeElements` is `true`, writes an option record for each
    * InfoElement in the Template that has not been seen before.
    *
    * Creates a template set if none exists.  Closes the template set
    * and creates a new one if the options-template setting of the
    * template and the current template set differ.
    *
    * @return `true` if the template was added, `false` if the template was
    * not added because it had been previously seen
    */
  private[this] def addTemplate(template: Template, tid: Option[Int] = None):
      Boolean =
  {
    if ( knownTmpl.contains(template) ) {
      false
    } else {
      // The ExportStream has not seen this template, but the template may
      // have been added to the Session by other means.
      val id = session.getOrAdd(template, tid)
      // Create and write a record-metadata record describing each private
      // element in the template that has not been seen before
      if ( describeElements ) {
        elementDescriptionWrite(ImmutableSet.empty[InfoElement] ++ (
          for (
            (ie, len) <- template.iterator
            if ( ie.ident.enterpriseId != 0 &&
              !knownElement.contains(ie) )) yield ie ))
      }
      // Create and write a template-metadata record for this template if the
      // template has a name
      if ( describeTemplates ) {
        templateDescriptionWrite(template, id)
      }

      // Add the template to a template set.
      val isOption = template.isOptionsTemplate
      if ( tmplSet.isEmpty ) {
        tmplSet = Option(
          TemplateSet.empty(session, isOption).addTemplate(template))
      } else if ( isOption != tmplSet.get.isOptionsTemplate ) {
        closeSets(TEMPLATES)
        tmplSet = Option(
          TemplateSet.empty(session, isOption).addTemplate(template))
      } else {
        try {
          tmplSet.get.addTemplate(template)
        } catch {
          case _: MessageTooLongException =>
            closeSets(TEMPLATES)
            tmplSet = Option(
              TemplateSet.empty(session, isOption).addTemplate(template))
        }
      }
      knownTmpl += template
      true
    }
  }

  /**
    * Like `addTemplate` but for a metadata template.  This function creates a
    * TemplateSet, adds `template` to it, and closes that TemplateSet.  This
    * ensures the metadata template appears before other templates and
    * records.
    *
    * Also adds `template` to the global `knownTmpl` Set, increments the
    * `tmplCount`, and calls the dataWrittenCallback if data is written.
    */
  private[this] def addMetadataTemplate(
    template: Template, tid: Option[Int]): Unit =
  {
    if ( !knownTmpl.contains(template) ) {
      val tset = TemplateSet.empty(session, true).addTemplate(template)
      val wrote = closeSet(tset)
      if ( wrote > 0 && _dataWrittenCallback.nonEmpty ) {
        _dataWrittenCallback.get.wrote(this, wrote)
      }
      tmplCount += tset.size
      knownTmpl += template
    }
  }


  /** Throws an error if `tid` is invalid, if `tid` is already in-use,
    * or if `tmpl` already exists.
    */
  private[this] def checkTid(tid: Int, tmpl: Option[Template]): Unit = {
    if ( tid < MIN_TEMPLATE_ID || tid > MAX_TEMPLATE_ID ) {
      throw new IllegalTemplateRecordException(s"Invalid template ID: $tid")
    }
    if ( session.getTemplate(tid).nonEmpty ) {
      throw new IllegalTemplateRecordException(
        s"Cannot set template ID to $tid; ID already in use")
    }
    if ( tmpl.nonEmpty ) {
      throw new RuntimeException(
        "Cannot set template id; template already exists")
    }
  }

  /** Stores `template` in the variables meant for the element meta-data
    * description template.  This is used when `describeElements` is `false`
    * and the caller adds such a template via one of the `add()` methods. */
  private[this] def elementDescriptionStore(
    template: Template, tid: Option[Int] = None): Unit =
  {
    assert(template.isInfoElementMetadataTemplate)
    assert(!describeElements)
    if ( elemDescTmpl.isEmpty ) {
      assert(elemDescTID.isEmpty)
      elemDescTmpl = Option(template)
      elemDescTID = Option(session.getOrAdd(template, tid))
      addMetadataTemplate(template, elemDescTID)
      elemDescWritten = true
      //LOG//logger.trace(f"Stored template ${elemDescTID.get}%#06x" +
      //LOG//  " as the element description template")
    } else {
      // call addMetadataTemplate in case this template is different than the
      // one we have defined as the element description template
      addMetadataTemplate(template, tid)
    }
  }

  /**
    * Stores `template` in the variables meant for the template meta-data
    * description template, using `tid` if not `None`.  This function is used
    * when `describeTemplates` is `false` and the caller adds such a template
    * via one of the `add()` methods.
    *
    * Returns `true` if the template was added or `false` if it was not added
    * because it had already been set.  If this function returns `true`, the
    * caller should call `closeSets(TEMPLATES)` to ensure the template
    * description template is written before the records.
    */
  private[this] def templateDescriptionStore(
    template: Template, tid: Option[Int] = None): Boolean =
  {
    assert(template.isTemplateMetadataTemplate)
    assert(!describeTemplates)
    if ( tempDescTmpl.isEmpty ) {
      assert(tempDescTID.isEmpty)
      tempDescTmpl = Option(template)
      tempDescTID = Option(session.getOrAdd(template, tid))
      tempDescWritten = true
      //LOG//logger.trace(f"Stored template ${tempDescTID.get}%#06x" +
      //LOG//  " as the template description template")

      // Do not use addMetadataTemplate() since we want to be certain any
      // element description records are written before the template
      // description template.
      addTemplate(template, tempDescTID)
    } else {
      addTemplate(template, tid)
    }
  }


  /** A callback invoked when bytes are written to the WritableByteChannel */
  private[this] var _dataWrittenCallback: Option[DataWrittenCallback] = None

  /** Sets a callback to be invoked when bytes are written to the
    * outputStream.
    * @since 1.3.1 */
  def dataWrittenCallback_=(callback: DataWrittenCallback): Unit =
    _dataWrittenCallback = Option(callback)

  /** Gets the current callback invoked when bytes are written to the
    * outputStream as an [[scala.Option Option]].
    * @since 1.3.1 */
  def dataWrittenCallback: Option[DataWrittenCallback] = _dataWrittenCallback

  /** Returns the file channel to which the data is being written */
  def outputStream: WritableByteChannel = out

  /** Returns the number of templates added to the stream. */
  def templateCount: Int = synchronized {
    tmplCount + (tmplSet.map { s => s.size }).getOrElse(0)
  }

  /** Returns the number of data records added to the stream. */
  def recordCount: Int = synchronized {
    (recCount + (elemDescRecSet.map { s => s.size }).getOrElse(0) +
      (tempDescRecSet.map { s => s.size }).getOrElse(0) +
      (recSet.map { s => s.size }).getOrElse(0))
  }

  /**
    * Flushes and closes the output stream.
    */
  def close(): Unit = synchronized {
    val wrote = closeSets(ALL, false) + writeBuffer()
    out.close()
    if ( wrote > 0 && _dataWrittenCallback.nonEmpty ) {
      _dataWrittenCallback.get.closed(this, wrote)
    }
  }

  /**
    * Flushes any data currently sitting in memory.
    */
  def flush(): ExportStream = synchronized {
    val wrote = closeSets(ALL, false) + writeBuffer()
    if ( wrote > 0 && _dataWrittenCallback.nonEmpty ) {
      _dataWrittenCallback.get.wrote(this, wrote)
    }
    this
  }


  /**
    * Adds `template` to the output stream unless the template has
    * already been seen by this stream.
    *
    * If `template` appears to be a template for describing either
    * information element meta-data or template meta-data, then the
    * template is ignored if the Export Stream is exporting that
    * meta-data itself (see the `describeElements` and `describeTemplates`
    * variables).
    */
  def add(template: Template): ExportStream = synchronized {
    if ( !template.isMetadataTemplate ) {
      addTemplate(template)

    } else if ( template.isInfoElementMetadataTemplate ) {
      if ( !describeElements ) {
        elementDescriptionStore(template)
      }
      // else ExportStream is describing the elements, and it should
      // ignore this template
    } else {
      assert(template.isTemplateMetadataTemplate)
      if ( !describeTemplates ) {
        if ( templateDescriptionStore(template) ) {
          closeSets(TEMPLATES)
        }
      }
      // else ExportStream is describing the templates, and it should
      // ignore this template
    }
    this
  }


  /**
    * Adds `record` to the output stream.  Prior to adding `record`,
    * adds any [[Template Templates]] used by `record` that have not
    * been seen before.
    *
    * If `record` appears to contain either information element
    * meta-data or template meta-data, then the record is ignored if
    * the Export Stream is exporting that meta-data itself (see the
    * `describeElements` and `describeTemplates` variables).
    */
  def add(record: Record): ExportStream = synchronized {
    val template = record.template
    if ( !template.isMetadataTemplate ) {
      // an ordinary record
      if ( describeElements ) {
        // handle IEs hidden in the BasicLists
        val ieList = record.allBasicListElements
        elementDescriptionWrite(ieList.filter({ie: InfoElement => {
          ie.ident.enterpriseId != 0 && !knownElement.contains(ie)}}))
      }
      record.message match {
        case Some(msg) =>
          for ( t <- record.allTemplates ) {
            addTemplate(t, msg.session.getId(t))
          }
        case None =>
          for ( t <- record.allTemplates ) {
            addTemplate(t)
          }
      }
      if ( recSet.isEmpty ) {
        recSet = Option(RecordSet.empty(template, session).addRecord(record))
      } else if ( recSet.get.template != template ) {
        closeSets(RECORDS)
        recSet = Option(RecordSet.empty(template, session).addRecord(record))
      } else {
        try {
          recSet.get.addRecord(record)
        } catch {
          case _: MessageTooLongException =>
            closeSets(RECORDS)
            recSet = Option(
              RecordSet.empty(template, session).addRecord(record))
        }
      }

    } else if ( template.isInfoElementMetadataTemplate ) {
      if ( !describeElements ) {
        // store this record's template as the element description template if
        // we have not seen it before, then add this record to the element
        // description RecordSet
        val tid = record.message.flatMap{ msg => msg.session.getId(template) }
        elementDescriptionStore(template, tid)
        if ( elemDescRecSet.isEmpty ) {
          elemDescRecSet = Option(
            RecordSet.empty(template, session).addRecord(record))
        } else {
          try {
            elemDescRecSet.get.addRecord(record)
          } catch {
            case _: MessageTooLongException =>
              closeSets(ELEMENT_DESCRIPTIONS)
              elemDescRecSet = Option(
                RecordSet.empty(template, session).addRecord(record))
          }
        }
      } else if ( elemDescTID.isEmpty ) {
        // ignore the record, but use this record's template ID as the TID of
        // the element description template
        for (tid <- record.message.flatMap{msg => msg.session.getId(template)})
        {
          elementDescriptionTID = tid
        }
      }
      // else ExportStream is describing the elements, and it should ignore
      // this record

    } else {
      assert(template.isTemplateMetadataTemplate)
      if ( !describeTemplates ) {
        // store this record's template as the template description template if
        // we have not seen it before, then add this record to the template
        // description RecordSet
        var added = false
        record.message.map{ _.session } match {
          case Some(session) =>
            added = templateDescriptionStore(template, session.getId(template))
            for ( t <- record.allTemplates ) {
              // put the function first to ensure it is called
              added = addTemplate(t, session.getId(t)) || added
            }
          case None =>
            added = templateDescriptionStore(template, None)
            for ( t <- record.allTemplates ) {
              added = addTemplate(t) || added
            }
        }
        if ( added ) {
          closeSets(TEMPLATES)
        }
        if ( tempDescRecSet.isEmpty ) {
          tempDescRecSet = Option(
            RecordSet.empty(template, session).addRecord(record))
        } else {
          try {
            tempDescRecSet.get.addRecord(record)
          } catch {
            case _: MessageTooLongException =>
              closeSets(TEMPLATE_DESCRIPTIONS)
              tempDescRecSet = Option(
                RecordSet.empty(template, session).addRecord(record))
          }
        }
      } else if ( tempDescTmpl.isEmpty ) {
        // ignore the record, but use this record's template ID as the TID of
        // the template description template
        for (tid <- record.message.flatMap{msg => msg.session.getId(template)})
        {
          templateDescriptionTID = tid
        }
      }
      // else ExportStream is describing the templates, and it should ignore
      // this record
    }

    this
  }

  /** Add all records from a MesageReader to this ExportStream */
  def add(reader: MessageReader): ExportStream = {
    var readerSession: Session = null
    for ( message <- reader ) {
      if ( message.session != readerSession ) {
        if ( Option(readerSession).nonEmpty ) {
          throw new RuntimeException(
            "MessageReader must use a single observation domain")
        }
        readerSession = message.session
      }
      // ignore TemplateSets, the record's templates are added as needed
      for {
        set <- message.iterator
        rec <- set.recordIterator
      } {
        add(rec)
      }
    }
    this
  }

  /** Returns the template ID for the options template used by records
    * that describe private enterprise information elements.  Returns
    * [[scala.None]] if no ID has been assigned.
    */
  def elementDescriptionTID: Option[Int] = synchronized { elemDescTID }

  /** Sets the template ID for the options template used by records that
    * describe private enterprise information elements and exports the
    * template to ensure the template ID is reserved.
    *
    * @throws IllegalTemplateRecordException when `tid` is illegal or
    * is already in use by the session
    * @throws RuntimeException when `tid` cannot be used for the
    * template because the template already exists
    */
  def elementDescriptionTID_=(tid: Int): Unit = synchronized {
    checkTid(tid, elemDescTmpl)
    elemDescTID = Option(tid)
    // create the template to reserve the TID
    elementDescriptionTemplate
  }

  /** Returns the template ID for the options template used by records
    * that describe templates.  Returns [[scala.None]] if no ID has
    * been assigned.
    */
  def templateDescriptionTID: Option[Int] = synchronized { tempDescTID }

  /** Sets the template ID for the options template used by records that
    * describe templates and exports the template to ensure the
    * template ID is reserved.
    *
    * @throws IllegalTemplateRecordException when `tid` is illegal or
    * is already in use by the session
    * @throws RuntimeException when `tid` cannot be used for the
    * template because the template already exists
    */
  def templateDescriptionTID_=(tid: Int): Unit = synchronized {
    checkTid(tid, tempDescTmpl)
    tempDescTID = Option(tid)
    // create the template to reserve the TID
    templateDescriptionTemplate
  }

}

/**
  * An [[ExportStream]] factory.
  */
object ExportStream {
  /** Creates a new ExportStream.
    *
    * @param outputStream Where to write the IPFIX data.
    * @param session The session to use.
    */
  def apply(outputStream: WritableByteChannel, session: Session): ExportStream =
    new ExportStream(outputStream, session)

  /** Creates a new ExportStream
    *
    * @param outStream Where to write the IPFIX data.
    * @param session The session to use.
    */
  def apply(outputStream: DataOutputStream, session: Session): ExportStream =
    new ExportStream(Channels.newChannel(outputStream), session)


  /** Creates a new ExportStream that appends to an existing file when
    * given a FileChannel open for reading and writing to that file.
    *
    * Creates a new [[SessionGroup]] using `model`, reads the IPFIX
    * data from `channel` and adds the [[Template Templates]] to the
    * [[Session]], then creates an ExportStream to append records to
    * the file.
    *
    * @param channel An open read-write handle to the file to append
    * IPFIX records to
    * @param model The information model for elements
    * @throws RuntimeException if the underlying file contains records
    * in more than one observation domain
    */
  def appendTo(channel: FileChannel, model: InfoModel): ExportStream = {
    channel.position(0)
    val sessionGroup = SessionGroup(model, channel)
    // domain used for session when file is empty
    val defaultDomain = 0
    val msgReader = StreamMessageReader(channel, sessionGroup)
    var session: Session = null
    var elemDescTmpl: Option[Template] = None
    var tempDescTmpl: Option[Template] = None
    val knownTmpl = ScalaSet.empty[Template]
    var sequence = 0L
    var recCount = 0
    var tmplCount = 0
    for ( message <- msgReader ) {
        if ( message.session != session ) {
          if ( Option(session).nonEmpty ) {
            throw new RuntimeException(
              "file must use a single observation domain")
          }
          session = message.session
        }
        sequence = message.sequenceNumber
        for ( set <- message.iterator ) {
          set match {
            case rs: RecordSet =>
              recCount += rs.size
              sequence += rs.size
            case ts: TemplateSet =>
              tmplCount += ts.size
              for ( t <- ts.iterator ) {
                knownTmpl += t
                if ( t.isMetadataTemplate ) {
                  if ( t.isInfoElementMetadataTemplate ) {
                    elemDescTmpl = Option(t)
                  } else {
                    assert(t.isTemplateMetadataTemplate)
                    tempDescTmpl = Option(t)
                  }
                }
              }
          }
        }
    }
    if ( Option(session).isEmpty ) {
      // only happens when the file being appended to has no data
      session = sessionGroup.getOrCreateSession(defaultDomain)
    }
    val es = new ExportStream(channel, session)
    es.elemDescTmpl = elemDescTmpl
    elemDescTmpl.foreach { t =>
      es.elemDescTID = Option(session.getOrAdd(t))
      es.elemDescWritten = true
    }
    es.tempDescTmpl = tempDescTmpl
    tempDescTmpl.foreach { t =>
      es.tempDescTID = Option(session.getOrAdd(t))
      es.tempDescWritten = true
    }
    es.knownTmpl = knownTmpl
    es.recCount = recCount
    es.tmplCount = tmplCount
    es.prevCount = sequence.toInt
    es
  }

  /** Creates a new ExportStream that appends to an existing file.
    *
    * @param path The file to append IPFIX records to
    * @param model The information model for elements
    */
  def appendTo(path: Path, model: InfoModel): ExportStream =
    appendTo(FileChannel.open(path,
      StandardOpenOption.READ, StandardOpenOption.WRITE), model)


  /**
    * Describes a class that may be used to generate a timestamp when
    * creating an IPFIX message.
    */
  trait MessageTimestamp {
    /** Return a timestamp to use for an IPFIX Message. The format is the
      * number of seconds since the UNIX epoch. */
    def stamp(): Long
  }

  /** The default timestamp generator. */
  object TimestampNow extends MessageTimestamp {
    def stamp(): Long = Instant.now().getEpochSecond()
  }

  /**
    * Supports a callback mechanism when data is written to an
    * [[ExportStream]].  Enable the callback by setting the
    * `dataWrittenCallback` of an [[ExportStream]].
    * @since 1.3.1
    */
  trait DataWrittenCallback {
    /** Method invoked when bytes are written to an [[ExportStream]]; that is,
      * when an IPFIX [[Message]] is complete.  This method is not called if
      * data is written as the file as being closed; instead, the closed()
      * function is called with that count.
      * @param stream The ExportStream writing the data
      * @param count The number of bytes written to the stream in this Message
      */
    def wrote(stream: ExportStream, count: Long): Unit

    /** Method invoked for an [[ExportStream]] that has just been closed if
      * closing caused data to be written to disk.
      * @param stream The ExportStream that was just closed.
      * @param count The number of bytes written to the stream as the file was
      * closed.
      * @since 1.3.2
      */
    def closed(stream: ExportStream, count: Long): Unit
  }

  /** A class for the Record used to describe an [[InfoElement]].
    */
  private final case class ElementDescriptionRecord (
    tmpl: Template,
    ie: InfoElement)
      extends ExportRecord(tmpl)
  {
    @IPFIXExtract(name = "privateEnterpriseNumber")
    val pen = ie.enterpriseId.toInt
    @IPFIXExtract(name = "informationElementId")
    val id = ie.elementId.toShort
    @IPFIXExtract(name = "informationElementDataType")
    val dataType = ie.dataTypeId.value.toByte
    @IPFIXExtract(name = "informationElementSemantics")
    val semantics = ie.semantics.value.toByte
    @IPFIXExtract(name = "informationElementUnits")
    val units = ie.units.value
    @IPFIXExtract(name = "informationElementRangeBegin")
    val rangeMin = ie.rangeMin
    @IPFIXExtract(name = "informationElementRangeEnd")
    val rangeMax = ie.rangeMax
    @IPFIXExtract(name = "informationElementName")
    val name = ie.name
    @IPFIXExtract(name = "informationElementDescription")
    val desc = ie.description
  }

  /** A class for the Record used to describe a Template.
    * @param tmpl: Expected to be the templateDescriptionTemplate
    *     defined in the ExportStream class.
    * @param tid: The ID of the template being described.
    * @param name: The name for the template being described.
    * @param desc: The description for the template being described.
    */
  private final case class TemplateDescriptionRecord (
    tmpl: Template,
    tid:  Int,
    name: String,
    desc: String)
      extends ExportRecord(tmpl)
  {
    @IPFIXExtract(name = "templateId")
    val templateId = tid.toShort
    @IPFIXExtract(name = "templateName")
    val templateName = name
    @IPFIXExtract(name = "templateDescription")
    val templateDescription = desc
  }

}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
