// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix


/**
 * A FieldExtractor may be used to extract data from a [[Record]].  It may
 * be used directly via the [[#extractFrom]] method, or more commonly
 * may be used as an operand to {@link Record#apply(FieldExtractor)}.
 *
 * Extractors are meant to be more than a simple index into a record.
 * Rather, they can retrieve any data based on a record, based on that
 * record's template and values.  Common uses of extractors are to
 * extract data from a particular field regardless of the actual template
 * being used, or to generate a computed value based on fields in a
 * record, such as returning a string value for a field that is encoded as
 * an integer.
 *
 * The type parameter `T` is a convenience feature.  Implementations
 * of `FieldExtractor` normally have to use {@link Record#apply(int)}
 * under the hood, which returns `Any`.  As such, a cast will be
 * made at some point, and a `ClassCastException` may be thrown if
 * care is not taken.  It is suggested that implementers of objects that
 * implement `FieldExtractor` should include a constructor that takes
 * the class type as an argument, and do type checking based on the
 * record field type in order to throw an error during extractor construction
 * rather than during use.
 *
 * @tparam T the type the extractor should return
 * @see Record
 */
trait FieldExtractor[T] {
  /**
    * Returns the value of the field referenced by the extractor from
    * from the record as an Option.  The value may be `None` if this
    * extractor does not reference a field in this record.
    *
    * @param rec The record from which to extract the value.
    * @return  The value referenced from the record or `None`.
    */
  def extractFrom(rec: Record): Option[T]
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
