// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix

/**
  * A FieldSpec is used to locate an [[InfoElement]] within an existing
  * [[Template]].
  *
  * @param name The optional name of the element.
  * @param ident The optional Identifier of the element.
  * @param nth The occurrence of element to return; the first occurrence is 1.
  *
  * @see [[FieldSpec$ The companion object]] for more details
  *
  * @see [[IEFieldSpecifier]] if you are attempting to include an
  * [[InfoElement Information Element]] in new a [[Template]].
  */
final class FieldSpec private (
  val name:  Option[String],
  val ident: Option[Identifier],
  val nth:   Int)
{
  if (nth < 1) {
    throw new IPFIXRuntimeException(s"Illegal nth parameter: ${nth}")
  }

  /**
    * Attempts to find an [[InfoElement]] that matches this field specifier in
    * the given [[Template]] and returns its position as an [[scala.Option
    * Option]].
    */
  def findIn(template: Template): Option[Int] = {
    val pos = template.indexOf(this)
    if (-1 == pos) {
      None
    } else {
      Option(pos)
    }
  }

  override def toString(): String = {
    val sb: StringBuilder = new StringBuilder("FieldSpec(")
    name foreach { n => sb.append(s"${n}, ") }
    ident foreach { id => sb.append(s"${id}, ") }
    sb.append(s"${nth})")
    sb.mkString
  }

  override def hashCode: Int =
    41 * (
      41 * (
        nth
      ) + ident.hashCode
    ) + name.hashCode

  //  According to Scala book, no need to define canEqual() since this
  //  is a final class

  override def equals(other: Any): Boolean =
    other match {
      case that: FieldSpec =>
        name == that.name &&
          ident == that.ident &&
          nth == that.nth
      case _ => false
    }

}

/**
  * A [[FieldSpec]] factory.
  */
object FieldSpec {
  /**
    * A FieldSpec factory that finds the first [[InfoElement]] in a
    * [[Template]] whose ident matches the given value.
    */
  def apply(ident: Identifier): FieldSpec =
    new FieldSpec(None, Option(ident), 1)

  /**
    * A FieldSpec factory that finds the `nth` occurrence of the
    * [[InfoElement]] in a [[Template]] whose ident matches the given value.
    *
    * @throws IPFIXRuntimeException if `nth` is less than 1.
    */
  def apply(ident: Identifier, nth: Int): FieldSpec =
    new FieldSpec(None, Option(ident), nth)

  /**
    * A FieldSpec factory that finds the first [[InfoElement]] in a
    * [[Template]] whose name matches the given value.
    */
  def apply(name: String): FieldSpec =
    new FieldSpec(Option(name), None, 1)

  /**
    * A FieldSpec factory that finds the `nth` occurrence of the
    * [[InfoElement]] in a [[Template]] whose name matches the given value.
    *
    * @throws IPFIXRuntimeException if `nth` is less than 1.
    */
  def apply(name: String, nth: Int): FieldSpec =
    new FieldSpec(Option(name), None, nth)

  /**
    * A FieldSpec factory that finds the `nth` occurrence of an
    * [[InfoElement]] in a [[Template]] where either the element's name is
    * `name` or the element's ident is `ident`.
    *
    * @throws IPFIXRuntimeException if `nth` is less than 1.
    */
  def apply(name: String, ident: Identifier, nth: Int): FieldSpec =
    new FieldSpec(Option(name), Option(ident), nth)
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
