// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix

import java.nio.BufferUnderflowException
import java.nio.ByteBuffer

/**
  * A single IPFIX Information Element specification for inclusion of
  * an [[InfoElement Information Element]] when constructing a
  * [[Template]].
  *
  * @param ident The [[Identifier]] for the element
  * @param length The octet length of the element
  *
  * @see [[IEFieldSpecifier$ The companion object]] for more details
  *
  * @see [[FieldSpec]] if you are attempting to locate an
  * [[InfoElement Information Element]] within an existing
  * [[Template]].
  *
  * @throws IllegalFieldSpecifierException if `length` is negative
  * or larger than [[VARLEN]].
  */
final case class IEFieldSpecifier private (
  val ident: Identifier,
  val length: Int)
{
  if (length < 0 || length > VARLEN) {
    throw new IllegalFieldSpecifierException
  }

  override def toString(): String = (
    s"IEFieldSpecifier(${ident}, length=" +
      (if (length == VARLEN) { "varlen" } else { length }) + ")")
}


/**
  * An [[IEFieldSpecifier]] factory.
  */
object IEFieldSpecifier {
  /*
   * Since this is a case class, there is an apply method:
   *
   * def apply(ident: Identifier, length: Int): IEFieldSpecifier
   */

  /**
    * Creates a field specifier using the [[Identifier]] of the
    * specified `infoElement` and the specified `length`.
    *
    * @param infoElement The InfoElement to use
    * @param length The octet length of the element
    *
    * @throws IllegalFieldSpecifierException if `length` is negative
    * or larger than [[VARLEN]].
    */
  def apply(infoElement: InfoElement, length: Int): IEFieldSpecifier =
    new IEFieldSpecifier(infoElement.ident, length)

  /**
    * Creates a field specifier using an InfoElement and a length
    * given as a Tuple2.
    *
    * @param pair A Tuple2 of the InfoElement and its length
    *
    * @throws IllegalFieldSpecifierException if `length` is negative
    * or larger than [[VARLEN]].
    */
  def apply(pair: (InfoElement, Int)): IEFieldSpecifier =
    new IEFieldSpecifier(pair._1.ident, pair._2)

  /**
    * Creates a field specifier by using the [[Identifier]] of the
    * [[InfoElement Information Element]] found by searching
    * `infoModel` for the element whose name is `name` and the
    * specified `length`.
    *
    * @param infoModel The InfoModel to search
    * @param name The name of the InfoElement to find in `infoModel`
    * @param length The octet length of the element
    *
    * @throws IllegalFieldSpecifierException if `length` is negative
    * or larger than [[VARLEN]].
    * @throws java.util.NoSuchElementException if `name` is not known
    * to `infoModel`.
    */
  def apply(infoModel: InfoModel, name: String, length: Int): IEFieldSpecifier =
    infoModel.get(name) match {
      case None =>
        throw new NoSuchElementException(s"No such element '$name' in model")
      case Some(infoElement) =>
        new IEFieldSpecifier(infoElement.ident, length)
    }

  /**
    * Creates a field specifier by reading the element's identifier
    * and length from a buffer containing two 16-bit values and an
    * optional 32-bit Private Enterprise Number.
    */
  def fromBuffer(buffer: ByteBuffer): IEFieldSpecifier = {
    //println("IEFieldSpecifier::apply(b).  size of buffer is " + b.remaining()
    //  + " and current position is " + b.position())
    val pos = buffer.position()
    try {
      val id: Int = 0xffff & buffer.getShort()
      val len: Int = 0xffff & buffer.getShort()
      if ((id & 0x8000) == 0) {
        new IEFieldSpecifier(Identifier(id), len)
      } else {
        val pen: Long = 0xffffffffL & buffer.getInt()
        new IEFieldSpecifier(Identifier((id & 0x7fff), pen), len)
      }
    } catch {
      case ex: BufferUnderflowException => {
        buffer.position(pos)
        throw ex
      }
    }
  }
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
