// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix


/**
  * An Information Element Identifier provides a way to identify an
  * [[InfoElement Information Element]] of an [[InfoModel Information
  * Model]].  The Identifier is also used when creating a
  * [[IEFieldSpecifier Field Specifier]].
  *
  * An Identifier consists of two numbers: an Element ID and an
  * optional Private Enterprise Number (PEN).  An absent PEN is
  * represented as a zero value.  The valid range of the Element ID is
  * 1 to 32767 (0x7fff) inclusive.  The valid range of Enterprise ID
  * is 1 to 4294967295 (0xffffffff).
  *
  * @param elementId The numeric identifier for the element.
  * @param enterpriseId The PEN for the element.
  *
  * @throws IllegalInfoElementAttributeException when both the
  * elementId and enterpriseId are 0, when the elementId is negative
  * or larger than 32767 (0x7fff), or when the enterpriseId is not
  * representable as an unsigned 32-bit number.
  */
final case class Identifier(
  val elementId: Int,
  val enterpriseId: Long = 0L)
    extends Ordered[Identifier]
{
  validate()

  override def compare(that: Identifier): Int = {
    if (enterpriseId == that.enterpriseId) {
      elementId - that.elementId
    } else if (enterpriseId < that.enterpriseId) {
      -1
    } else {
      1
    }
  }

  override def toString(): String = {
    if (enterpriseId != 0L) {
      s"Ident(${enterpriseId}/${elementId})"
    } else {
      s"Ident(${elementId})"
    }
  }

  private[this] def validate(): Unit = {
    if (elementId < 0 || elementId > 0x7fff) {
      throw new IllegalInfoElementAttributeException(
        "Illegal elementId: " + elementId)
    }
    if (enterpriseId < 0L || enterpriseId > 0xffffffffL) {
      throw new IllegalInfoElementAttributeException(
        "Illegal enterpriseId: " + enterpriseId)
    }
  }

  /*
   *  private[this] def readObject(in: java.io.ObjectInputStream): Unit = {
   *    in.defaultReadObject()
   *    validate()
   *  }
   *
   *  private[this] def writeObject(out: java.io.ObjectOutputStream): Unit = {
   *    out.defaultWriteObject()
   *  }
   */
}


/**  A factory for [[Identifier]] instances. */
object Identifier
{
  /** Returns an Identifier having the given `elementId` and `enterpriseId`.
    *
    * @param elementId The numeric identifier for the element.
    * @param enterpriseId The PEN for the element.  The PEN is 0 when this is
    * [[scala.None$ None]].
    * @since 1.3.1
    */
  def apply(elementId: Int, enterpriseId: Option[Long]): Identifier =
    new Identifier(elementId, enterpriseId.getOrElse(0L))
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
